#!/bin/bash
# client-setup.sh

# Color definitions
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

# 기본 설정
API_SERVER=${API_SERVER:-"http://localhost:5500"}
AIRUN_HOME="$HOME/.airun"
VENV_PATH="$HOME/.airun_venv"
CONFIG_FILE="$AIRUN_HOME/airun.conf"

echo -e "${GREEN}AI.RUN Client Environment Setup${NC}"

# API 서버 연결 확인 및 설정 가져오기
check_api_server() {
    echo -e "${GREEN}API 서버 연결을 확인합니다...${NC}"
    
    # API 서버 설정 가져오기
    CONFIG_RESPONSE=$(curl -s "${API_SERVER}/api/v1/client/config")
    
    if [ $? -ne 0 ]; then
        echo -e "${RED}API 서버에 연결할 수 없습니다.${NC}"
        echo -e "${YELLOW}API_SERVER 환경변수를 설정하여 서버 주소를 지정할 수 있습니다:${NC}"
        echo -e "export API_SERVER=http://your-server:port"
        exit 1
    fi
    
    # JSON 파싱 (jq가 없는 경우를 위한 대체 방법)
    SMTP_HOST=$(echo "$CONFIG_RESPONSE" | python3 -c "import sys, json; print(json.load(sys.stdin)['data']['smtp']['host'])")
    SMTP_PORT=$(echo "$CONFIG_RESPONSE" | python3 -c "import sys, json; print(json.load(sys.stdin)['data']['smtp']['port'])")
    UTILS_VERSION=$(echo "$CONFIG_RESPONSE" | python3 -c "import sys, json; print(json.load(sys.stdin)['data']['utils_version'])")
    
    echo -e "${GREEN}API 서버 연결 성공${NC}"
}

# 시스템 확인
check_system() {
    echo -e "${GREEN}시스템 환경을 확인합니다...${NC}"
    
    # OS 확인
    if [[ "$OSTYPE" == "darwin"* ]]; then
        OS_TYPE="macos"
    elif [[ "$OSTYPE" == "msys" || "$OSTYPE" == "cygwin" ]]; then
        OS_TYPE="windows"
        echo -e "${RED}Windows 환경은 아직 지원되지 않습니다.${NC}"
        exit 1
    else
        OS_TYPE="linux"
    fi
    
    # 아키텍처 확인
    ARCH=$(uname -m)
    if [[ "$ARCH" != "x86_64" && "$ARCH" != "amd64" ]]; then
        echo -e "${RED}지원되지 않는 아키텍처입니다: $ARCH${NC}"
        echo -e "${RED}현재 x86_64/amd64 아키텍처만 지원됩니다.${NC}"
        exit 1
    fi
    
    echo -e "${GREEN}✓ 시스템 환경 확인 완료${NC}"
}

# Python 버전 체크
check_python() {
    echo -e "${GREEN}Python 환경을 확인합니다...${NC}"
    
    if ! command -v python3 &> /dev/null; then
        echo -e "${RED}Python3가 설치되어 있지 않습니다. Python3를 먼저 설치해 주세요.${NC}"
        exit 1
    fi

    # API 서버에서 받은 Python 버전 정보 사용
    PYTHON_VERSION=$(python3 -c "import sys; print('.'.join(map(str, sys.version_info[:3])))")
    MIN_VERSION=$(echo "$CONFIG_RESPONSE" | python3 -c "import sys, json; print(json.load(sys.stdin)['data']['python']['min_version'])")
    RECOMMENDED_VERSION=$(echo "$CONFIG_RESPONSE" | python3 -c "import sys, json; print(json.load(sys.stdin)['data']['python']['recommended_version'])")
    
    # 버전 비교 함수
    version_compare() {
        echo "$@" | awk -F. '{ printf("%d%03d%03d\n", $1,$2,$3); }'
    }
    
    if [ $(version_compare $PYTHON_VERSION) -lt $(version_compare $MIN_VERSION) ]; then
        echo -e "${RED}Python 버전이 너무 낮습니다.${NC}"
        echo -e "${RED}최소 버전: $MIN_VERSION${NC}"
        echo -e "${RED}현재 버전: $PYTHON_VERSION${NC}"
        echo -e "${YELLOW}권장 버전: $RECOMMENDED_VERSION${NC}"
        exit 1
    fi
    
    if [ $(version_compare $PYTHON_VERSION) -lt $(version_compare $RECOMMENDED_VERSION) ]; then
        echo -e "${YELLOW}경고: 현재 Python 버전($PYTHON_VERSION)이 권장 버전($RECOMMENDED_VERSION)보다 낮습니다.${NC}"
    fi

    echo -e "${GREEN}✓ Python 버전 확인 완료: $PYTHON_VERSION${NC}"
}

# 디렉토리 구조 생성
create_directories() {
    echo -e "${GREEN}디렉토리 구조를 생성합니다...${NC}"
    mkdir -p "$AIRUN_HOME"
    mkdir -p "$AIRUN_HOME/cache"
    mkdir -p "$AIRUN_HOME/logs"
    echo -e "${GREEN}✓ 디렉토리 생성 완료${NC}"
}

# 가상환경 설정
setup_venv() {
    echo -e "${GREEN}Python 가상환경을 설정합니다...${NC}"
    if [ -d "$VENV_PATH" ]; then
        echo -e "${YELLOW}기존 가상환경을 제거하고 새로 생성합니다...${NC}"
        rm -rf "$VENV_PATH"
    fi
    
    python3 -m venv "$VENV_PATH"
    source "$VENV_PATH/bin/activate"
    python -m pip install --upgrade pip
    echo -e "${GREEN}✓ 가상환경 설정 완료${NC}"
}

# 필수 패키지 설치
install_packages() {
    echo -e "${GREEN}필수 패키지를 설치합니다...${NC}"
    pip install \
        numpy==1.26.4 \
        pandas \
        matplotlib \
        requests \
        pillow \
        reportlab \
        aiohttp \
        sqlalchemy \
        fpdf \
        cairosvg

    if [ $? -ne 0 ]; then
        echo -e "${RED}패키지 설치에 실패했습니다.${NC}"
        exit 1
    fi
    echo -e "${GREEN}✓ 패키지 설치 완료${NC}"
}

# 실행 스크립트 생성
create_runner() {
    echo "실행 스크립트 생성 중..."
    
    # 운영체제 확인
    if [[ "$OSTYPE" == "msys" ]] || [[ "$OSTYPE" == "win32" ]] || [[ "$OSTYPE" == "cygwin" ]]; then
        # Windows 환경
        cat > "${AIRUN_HOME}/run.bat" << 'EOL'
@echo off
set AIRUN_HOME=%USERPROFILE%\.airun
set VENV_PATH=%USERPROFILE%\.airun_venv

:: 가상 환경 활성화
call %VENV_PATH%\Scripts\activate.bat

:: 환경 변수 로드
if exist "%AIRUN_HOME%\airun.conf.bat" (
    call %AIRUN_HOME%\airun.conf.bat
)

:: Python 스크립트 실행
python %*

:: 가상 환경 비활성화
call %VENV_PATH%\Scripts\deactivate.bat
EOL
        chmod +x "${AIRUN_HOME}/run.bat"
        echo "Windows 실행 스크립트가 생성되었습니다: ${AIRUN_HOME}/run.bat"
        
        # Windows 환경변수 설정 파일 생성
        cat > "${AIRUN_HOME}/airun.conf.bat" << EOL
@echo off
set SMTP_SERVER=${SMTP_SERVER}
set SMTP_PORT=${SMTP_PORT}
set SMTP_USERNAME=${SMTP_USERNAME}
set SMTP_PASSWORD=${SMTP_PASSWORD}
set API_SERVER=${API_SERVER}
EOL
    else
        # Unix 환경 (Linux/macOS)
        cat > "${AIRUN_HOME}/run.sh" << 'EOL'
#!/bin/bash
AIRUN_HOME=~/.airun
VENV_PATH=~/.airun_venv

# 가상 환경 활성화
source ${VENV_PATH}/bin/activate

# 환경 변수 로드
source ${AIRUN_HOME}/airun.conf

# Python 스크립트 실행
python "$@"

# 가상 환경 비활성화
deactivate
EOL
        chmod +x "${AIRUN_HOME}/run.sh"
        echo "Unix 실행 스크립트가 생성되었습니다: ${AIRUN_HOME}/run.sh"
        
        # Unix 환경변수 설정 파일 생성
        cat > "${AIRUN_HOME}/airun.conf" << EOL
export SMTP_SERVER="${SMTP_SERVER}"
export SMTP_PORT="${SMTP_PORT}"
export SMTP_USERNAME="${SMTP_USERNAME}"
export SMTP_PASSWORD="${SMTP_PASSWORD}"
export API_SERVER="${API_SERVER}"
EOL
    fi
}

# utils 패키지 설치 부분 수정
install_utils() {
    echo -e "${GREEN}utils 패키지를 설치합니다...${NC}"
    
    # Python 버전 정보 가져오기
    PYTHON_VERSION=$(python -c "import sys; print(f'{sys.version_info.major}{sys.version_info.minor}')")
    
    # 시스템별 파일명 구성
    if [[ "$OSTYPE" == "darwin"* ]]; then
        PLATFORM="macos"
        UTILS_FILENAME="utils.cpython-${PYTHON_VERSION}-darwin.so"
    elif [[ "$OSTYPE" == "msys" || "$OSTYPE" == "cygwin" ]]; then
        PLATFORM="windows"
        UTILS_FILENAME="utils.cp${PYTHON_VERSION}-win_amd64.pyd"
    else
        PLATFORM="linux"
        UTILS_FILENAME="utils.cpython-${PYTHON_VERSION}-x86_64-linux-gnu.so"
    fi
    
    # site-packages 경로 확인
    SITE_PACKAGES=$(python -c "import site; print(site.getsitepackages()[0])")
    
    # utils 패키지 디렉토리 생성
    UTILS_DIR="$SITE_PACKAGES/utils"
    mkdir -p "$UTILS_DIR"
    
    # 컴파일된 utils 파일 다운로드
    echo -e "${GREEN}컴파일된 utils 파일을 다운로드합니다... ($UTILS_FILENAME)${NC}"
    
    UTILS_URL="${API_SERVER}/api/v1/client/utils/${PLATFORM}/${PYTHON_VERSION}/${UTILS_FILENAME}"
    echo -e "${YELLOW}다운로드 URL: ${UTILS_URL}${NC}"
    
    # 파일 다운로드
    curl -o "$UTILS_DIR/$UTILS_FILENAME" "$UTILS_URL"
    
    if [ $? -ne 0 ]; then
        echo -e "${RED}utils 파일 다운로드에 실패했습니다.${NC}"
        exit 1
    fi
    
    # __init__.py 생성
    cat > "$UTILS_DIR/__init__.py" << EOF
import os
import importlib.util

# 현재 디렉토리의 .so 또는 .pyd 파일 찾기
module_dir = os.path.dirname(__file__)
for file in os.listdir(module_dir):
    if file.startswith('utils.') and (file.endswith('.so') or file.endswith('.pyd')):
        # 동적으로 모듈 로드
        spec = importlib.util.spec_from_file_location("utils", os.path.join(module_dir, file))
        module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(module)
        # 모든 속성을 현재 네임스페이스로 가져오기
        for attr in dir(module):
            if not attr.startswith('_'):
                globals()[attr] = getattr(module, attr)
        break

__version__ = "${UTILS_VERSION}"
EOF
    
    # user_tools.py 설치
    echo -e "${GREEN}user_tools.py를 설치합니다...${NC}"
    curl -o "$SITE_PACKAGES/user_tools.py" "${API_SERVER}/api/v1/client/user-tools"
    
    if [ $? -ne 0 ]; then
        echo -e "${RED}user_tools.py 다운로드에 실패했습니다.${NC}"
        exit 1
    fi
}

# 기본 설정 파일 생성 - API 서버에서 받아온 설정 사용
create_config() {
    echo -e "${GREEN}설정 파일을 생성합니다...${NC}"
    cat > "$CONFIG_FILE" << EOF
# AI.RUN Client Configuration

# API Server
export API_SERVER="${API_SERVER}"

# SMTP Settings
export SMTP_HOST="${SMTP_HOST}"
export SMTP_PORT="${SMTP_PORT}"
export SMTP_USERNAME=""
export SMTP_PASSWORD=""
export USE_SMTP="NO"
export SMTP_SECURE="YES"

# Python Environment
export PYTHON_VENV_PATH="$VENV_PATH"
export PYTHONPATH="$SITE_PACKAGES"

# Paths
export AIRUN_HOME="$AIRUN_HOME"
export AIRUN_CACHE_DIR="$AIRUN_HOME/cache"
export AIRUN_LOGS_DIR="$AIRUN_HOME/logs"

# Features
export USE_REVIEW="NO"
export AUTO_EXECUTE="NO"
export HIDE_CODE="NO"
EOF

    chmod 600 "$CONFIG_FILE"
}

# 자동완성 설치 함수
install_completion() {
    echo -e "${GREEN}자동완성을 설치합니다...${NC}"
    
    # 현재 스크립트의 디렉토리 확인
    SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
    PROJECT_ROOT="$(cd "$SCRIPT_DIR/.." && pwd)"
    
    # completion 스크립트 존재 확인
    BASH_COMPLETION_SCRIPT="$PROJECT_ROOT/scripts/airun-completion.bash"
    ZSH_COMPLETION_SCRIPT="$PROJECT_ROOT/scripts/airun-completion.zsh"
    
    if [ ! -f "$BASH_COMPLETION_SCRIPT" ] || [ ! -f "$ZSH_COMPLETION_SCRIPT" ]; then
        echo -e "${YELLOW}자동완성 스크립트를 찾을 수 없습니다. 건너뜁니다.${NC}"
        return 0
    fi
    
    # 현재 쉘 확인
    CURRENT_SHELL=$(basename "$SHELL")
    
    # Bash completion 설치
    if command -v bash &> /dev/null; then
        COMPLETION_DIR="$HOME/.local/share/bash-completion/completions"
        mkdir -p "$COMPLETION_DIR"
        cp "$BASH_COMPLETION_SCRIPT" "$COMPLETION_DIR/airun"
        chmod 644 "$COMPLETION_DIR/airun"
        
        # .bashrc에 completion 설정 추가 (중복 방지)
        if ! grep -q "AI.RUN bash completion" "$HOME/.bashrc" 2>/dev/null; then
            echo "" >> "$HOME/.bashrc"
            echo "# AI.RUN bash completion" >> "$HOME/.bashrc"
            echo "if [ -f $COMPLETION_DIR/airun ]; then" >> "$HOME/.bashrc"
            echo "    source $COMPLETION_DIR/airun" >> "$HOME/.bashrc"
            echo "fi" >> "$HOME/.bashrc"
        fi
        echo -e "${GREEN}✓ Bash 자동완성 설치 완료${NC}"
    fi
    
    # Zsh completion 설치
    if command -v zsh &> /dev/null; then
        ZSH_COMPLETION_DIR="$HOME/.zsh/completion"
        mkdir -p "$ZSH_COMPLETION_DIR"
        cp "$ZSH_COMPLETION_SCRIPT" "$ZSH_COMPLETION_DIR/_airun"
        chmod 644 "$ZSH_COMPLETION_DIR/_airun"
        
        # .zshrc에 completion 설정 추가 (중복 방지)
        if ! grep -q "AI.RUN zsh completion" "$HOME/.zshrc" 2>/dev/null; then
            echo "" >> "$HOME/.zshrc"
            echo "# AI.RUN zsh completion" >> "$HOME/.zshrc"
            echo "autoload -Uz compinit" >> "$HOME/.zshrc"
            echo "compinit" >> "$HOME/.zshrc"
            echo "fpath=($ZSH_COMPLETION_DIR \$fpath)" >> "$HOME/.zshrc"
        fi
        echo -e "${GREEN}✓ Zsh 자동완성 설치 완료${NC}"
    fi
}

# 메인 설치 프로세스
main() {
    echo -e "${GREEN}AI.RUN 클라이언트 환경 설정을 시작합니다...${NC}"
    
    check_api_server
    check_system
    check_python
    create_directories
    setup_venv
    install_packages
    install_utils
    create_config
    create_runner
    install_completion

    echo -e "\n${GREEN}설치가 완료되었습니다!${NC}"
    echo -e "\n${YELLOW}설정 방법:${NC}"
    echo -e "1. SMTP 설정: $CONFIG_FILE 파일을 편집하여 이메일 설정을 구성하세요."
    echo -e "2. 실행 방법: $AIRUN_HOME/run.sh <python_file>"
    echo -e "\n${YELLOW}환경 변수가 적용되도록 쉘을 재시작하거나 다음 명령어를 실행하세요:${NC}"
    echo -e "source ~/.bashrc"
    echo -e "\n${YELLOW}자동완성 사용법:${NC}"
    echo -e "터미널에서 ${GREEN}airun <TAB><TAB>${NC}을 입력하여 자동완성을 사용할 수 있습니다."
    
    # 설치 결과 확인
    echo -e "\n${GREEN}설치 확인:${NC}"
    echo -e "1. 가상환경: $([ -d "$VENV_PATH" ] && echo "${GREEN}✓${NC}" || echo "${RED}✗${NC}")"
    echo -e "2. 설정 파일: $([ -f "$CONFIG_FILE" ] && echo "${GREEN}✓${NC}" || echo "${RED}✗${NC}")"
    echo -e "3. 실행 스크립트: $([ -f "$AIRUN_HOME/run.sh" ] && echo "${GREEN}✓${NC}" || echo "${RED}✗${NC}")"
    echo -e "4. 자동완성: $([ -f "$HOME/.local/share/bash-completion/completions/airun" ] && echo "${GREEN}✓${NC}" || echo "${RED}✗${NC}")"
    
    # utils 패키지 설치 확인
    if python -c "import utils" &>/dev/null; then
        echo -e "5. utils 패키지: ${GREEN}✓${NC}"
    else
        echo -e "5. utils 패키지: ${RED}✗${NC}"
    fi
    
    # user_tools.py 설치 확인
    if python -c "import user_tools" &>/dev/null; then
        echo -e "6. user_tools: ${GREEN}✓${NC}"
    else
        echo -e "6. user_tools: ${RED}✗${NC}"
    fi
}

main
