#!/bin/bash

# 색상 정의
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# 스크립트의 실제 위치 확인
SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
# 프로젝트 루트 디렉토리 (스크립트의 상위 디렉토리)
PROJECT_ROOT="$( cd "$SCRIPT_DIR/.." && pwd )"

# 프로젝트 루트 디렉토리로 이동
cd "$PROJECT_ROOT"

echo -e "${YELLOW}프로젝트 디렉토리: ${NC}$PROJECT_ROOT"

# 가상환경 활성화
echo -e "${YELLOW}가상환경 활성화 중...${NC}"
if [ ! -d "$HOME/.airun_venv" ]; then
    echo -e "${RED}Error: ~/.airun_venv 가상환경이 설치되어 있지 않습니다.${NC}"
    echo -e "${YELLOW}먼저 install.sh를 실행하여 가상환경을 설치해주세요.${NC}"
    exit 1
fi

source "$HOME/.airun_venv/bin/activate"
if [ $? -ne 0 ]; then
    echo -e "${RED}Error: 가상환경 활성화에 실패했습니다.${NC}"
    exit 1
fi

echo -e "${GREEN}✅ 가상환경이 활성화되었습니다.${NC}"
echo -e "${YELLOW}Python 경로: ${NC}$(which python)"
echo -e "${YELLOW}Python 버전: ${NC}$(python --version)"

# 현재 브랜치 확인
CURRENT_BRANCH=$(git rev-parse --abbrev-ref HEAD)
if [ "$CURRENT_BRANCH" != "dev" ]; then
    echo -e "${RED}Error: dev 브랜치에서만 실행할 수 있습니다.${NC}"
    echo -e "${YELLOW}현재 브랜치: ${NC}$CURRENT_BRANCH"
    exit 1
fi

# 커밋되지 않은 변경사항 확인
if [ -n "$(git status --porcelain)" ]; then
    echo -e "${RED}Error: 커밋되지 않은 변경사항이 있습니다. 먼저 모든 변경사항을 커밋해주세요.${NC}"
    exit 1
fi

# master와 dev 브랜치의 차이 확인
git fetch origin master dev
MASTER_HEAD=$(git rev-parse origin/master)
DEV_HEAD=$(git rev-parse origin/dev)

if [ "$MASTER_HEAD" == "$DEV_HEAD" ]; then
    echo -e "${RED}Error: master 브랜치와 dev 브랜치의 내용이 동일합니다.${NC}"
    echo -e "${YELLOW}릴리즈할 새로운 변경사항이 없습니다.${NC}"
    exit 1
fi

# 현재 버전 가져오기
CURRENT_VERSION=$(git tag --sort=-v:refname | grep '^v' | head -n1 || echo "v0.0.0")

# 버전 증가 함수
increment_version() {
    local version=$1
    local increment_type=$2
    
    # v 접두사 제거
    version=${version#v}
    
    # 버전을 . 으로 분리
    IFS='.' read -ra VERSION_PARTS <<< "$version"
    
    major=${VERSION_PARTS[0]:-0}
    minor=${VERSION_PARTS[1]:-0}
    patch=${VERSION_PARTS[2]:-0}
    
    case $increment_type in
        major)
            major=$((major + 1))
            minor=0
            patch=0
            ;;
        minor)
            minor=$((minor + 1))
            patch=0
            ;;
        patch|*)
            patch=$((patch + 1))
            ;;
    esac
    
    echo "v$major.$minor.$patch"
}

# 파라미터 파싱
AUTO_YES=false
VERSION_TYPE="patch"

for arg in "$@"; do
    case $arg in
        --yes|-yes|-y|-Y)
            AUTO_YES=true
            ;;
        major|minor|patch)
            VERSION_TYPE=$arg
            ;;
        *)
            echo -e "${RED}Error: 알 수 없는 파라미터: $arg${NC}"
            echo -e "${YELLOW}사용법: $0 [major|minor|patch] [--yes|-yes|-y|-Y]${NC}"
            exit 1
            ;;
    esac
done

# 버전 타입 확인
if [[ ! "$VERSION_TYPE" =~ ^(major|minor|patch)$ ]]; then
    echo -e "${RED}Error: 버전 타입은 major, minor, patch 중 하나여야 합니다.${NC}"
    exit 1
fi

# 새 버전 생성
NEW_VERSION=$(increment_version $CURRENT_VERSION $VERSION_TYPE)

# package.json 버전 업데이트
echo -e "\n${YELLOW}package.json 버전 업데이트 중...${NC}"
# 업데이트 전 버전 출력
echo -e "${YELLOW}업데이트 전 package.json 버전:${NC}"
grep "\"version\"" package.json

# OS 확인 후 적절한 sed 명령어 사용
if [[ "$OSTYPE" == "darwin"* ]]; then
    # macOS
    sed -i '' "s/\"version\": \".*\"/\"version\": \"${NEW_VERSION#v}\"/" package.json
else
    # Linux 및 기타
    sed -i "s/\"version\": \".*\"/\"version\": \"${NEW_VERSION#v}\"/" package.json
fi

# 업데이트 후 버전 출력
echo -e "${YELLOW}업데이트 후 package.json 버전:${NC}"
grep "\"version\"" package.json

# 변경사항 요약 표시
echo -e "\n${YELLOW}변경사항 요약:${NC}"
git --no-pager log --oneline origin/master..origin/dev

echo -e "\n${YELLOW}현재 버전: ${NC}$CURRENT_VERSION"
echo -e "${YELLOW}새로운 버전: ${NC}$NEW_VERSION"
echo -e "\n${GREEN}릴리즈 프로세스를 시작합니다...${NC}"

# 사용자 확인
if [ "$AUTO_YES" = true ]; then
    echo -e "\n${YELLOW}--yes 옵션이 설정되어 릴리즈를 자동으로 진행합니다.${NC}"
else
    read -p "계속 진행하시겠습니까? (y/N) " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        echo -e "${RED}릴리즈가 취소되었습니다.${NC}"
        exit 1
    fi
fi

# 사용자 확인 전에 package-lock.json 업데이트
echo -e "\n${YELLOW}dev 브랜치에서 npm install 실행${NC}"
npm install

# package.json과 package-lock.json 변경사항 커밋
echo -e "\n${YELLOW}package.json package-lock.json 커밋${NC}"
git add package.json package-lock.json
git commit -m "chore: update dependencies and bump version to ${NEW_VERSION}"

# 릴리즈 프로세스 실행
echo -e "\n${YELLOW}1. master 브랜치로 전환${NC}"
git checkout master

echo -e "\n${YELLOW}2. master 브랜치 업데이트${NC}"
git pull origin master

echo -e "\n${YELLOW}3. dev 브랜치 머지${NC}"
git merge dev

echo -e "\n${YELLOW}5. 새로운 태그 생성${NC}"
git tag -a $NEW_VERSION -m "Release $NEW_VERSION"

echo -e "\n${YELLOW}6. 변경사항 푸시${NC}"
git push origin master

echo -e "\n${YELLOW}7. 태그 푸시${NC}"
git push origin $NEW_VERSION

echo -e "\n${YELLOW}8. dev 브랜치로 복귀${NC}"
git checkout dev

echo -e "\n${YELLOW}9. dev 브랜치에 master의 변경사항 동기화${NC}"
git merge master
git push origin dev

# 빌드 및 배포 파일 생성
echo -e "\n${YELLOW}10. 빌드 실행${NC}"

# 가상환경의 Python 사용
echo -e "${YELLOW}가상환경 Python 사용:${NC}"
python --version

npm run build

# 원격 서버에 파일 복사
echo -e "\n${YELLOW}11. 원격 서버에 파일 복사${NC}"
REMOTE_PATH="/mirror/invesume/airun"
REMOTE_USER="ivs01"

# 실제 생성된 최신 배포 파일 찾기
LATEST_DEPLOY_FILE=$(ls -t deploy/airun-*-linux-x64.tar.gz 2>/dev/null | head -n1)

if [ -z "$LATEST_DEPLOY_FILE" ]; then
    echo -e "${RED}Error: deploy 디렉토리에서 배포 파일을 찾을 수 없습니다.${NC}"
    echo -e "${YELLOW}현재 디렉토리 내용:${NC}"
    ls -l deploy/
    exit 1
fi

echo -e "${YELLOW}배포 파일 경로 확인: ${NC}$LATEST_DEPLOY_FILE"
ls -l "$LATEST_DEPLOY_FILE" || echo -e "${RED}파일이 존재하지 않습니다.${NC}"

# 사용자에게 원격 서버 복사 여부 확인
if [ "$AUTO_YES" = true ]; then
    echo -e "\n${YELLOW}--yes 옵션이 설정되어 원격 서버 복사를 자동으로 진행합니다.${NC}"
else
    echo -e "\n${YELLOW}원격 서버에 파일을 복사하시겠습니까? (y/N)${NC}"
    read -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        echo -e "${YELLOW}원격 서버 복사를 건너뜁니다.${NC}"
        echo -e "\n${GREEN}✨ 릴리즈가 완료되었습니다! ($NEW_VERSION)${NC}"
        exit 0
    fi
fi

# SSH 연결 테스트
echo -e "${YELLOW}SSH 연결 테스트 중...${NC}"
if ! ssh -q $REMOTE_USER@repo.invesume.com "exit"; then
    echo -e "${RED}Error: SSH 연결에 실패했습니다. SSH 키와 권한을 확인해주세요.${NC}"
    exit 1
fi

# 원격 디렉토리 존재 여부 확인 및 생성
echo -e "${YELLOW}원격 디렉토리 확인 중...${NC}"
ssh -v $REMOTE_USER@repo.invesume.com "mkdir -p $REMOTE_PATH"

# SSH 키를 사용하여 파일 복사 (각 파일 별도 처리)
echo -e "${YELLOW}배포 파일 복사 시작...${NC}"
scp -v "$LATEST_DEPLOY_FILE" $REMOTE_USER@repo.invesume.com:$REMOTE_PATH/
if [ $? -eq 0 ]; then
    echo -e "${GREEN}✨ 배포 파일 복사가 완료되었습니다!${NC}"
else
    echo -e "${RED}Error: 배포 파일 복사 중 오류가 발생했습니다.${NC}"
    exit 1
fi

echo -e "${YELLOW}package.json 복사 시작...${NC}"
scp -v package.json $REMOTE_USER@repo.invesume.com:$REMOTE_PATH/
if [ $? -eq 0 ]; then
    echo -e "${GREEN}✨ package.json 복사가 완료되었습니다!${NC}"
else
    echo -e "${RED}Error: package.json 복사 중 오류가 발생했습니다.${NC}"
    exit 1
fi

echo -e "\n${GREEN}✨ 릴리즈가 완료되었습니다! ($NEW_VERSION)${NC}"
echo -e "${GREEN}✨ 모든 파일이 원격 서버에 성공적으로 복사되었습니다!${NC}"

# 가상환경 비활성화
echo -e "\n${YELLOW}가상환경 비활성화 중...${NC}"
deactivate
echo -e "${GREEN}✅ 가상환경이 비활성화되었습니다.${NC}" 