#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
스마트 Python 빌드 스크립트
변경사항이 없는 Python 파일은 건너뛰고 필요한 파일만 컴파일합니다.
"""

import os
import sys
import hashlib
import json
import subprocess
import time
import threading
from pathlib import Path

# Windows 환경에서 인코딩 문제 방지
if os.name == 'nt':  # Windows
    # 콘솔 출력 인코딩을 UTF-8로 설정
    if hasattr(sys.stdout, 'reconfigure'):
        sys.stdout.reconfigure(encoding='utf-8')
    if hasattr(sys.stderr, 'reconfigure'):
        sys.stderr.reconfigure(encoding='utf-8')
    
    # 환경 변수 설정
    os.environ['PYTHONIOENCODING'] = 'utf-8'
    
    # 안전한 출력을 위한 함수
    def safe_print(*args, **kwargs):
        try:
            print(*args, **kwargs)
        except UnicodeEncodeError:
            # 인코딩 오류 시 ASCII로 변환하여 출력
            safe_args = []
            for arg in args:
                if isinstance(arg, str):
                    safe_args.append(arg.encode('ascii', 'replace').decode('ascii'))
                else:
                    safe_args.append(arg)
            print(*safe_args, **kwargs)
else:
    # Linux/Mac에서는 일반 print 사용
    safe_print = print

class Spinner:
    """스피너 클래스 - 장시간 실행되는 작업 중에 시각적 진행 상태를 보여줍니다."""
    
    def __init__(self, message="Processing", delay=0.1):
        self.message = message
        self.delay = delay
        self.running = False
        self.spinner_thread = None
        
        # 다양한 스피너 패턴 중 하나 선택
        self.spinner_chars = ['⠋', '⠙', '⠹', '⠸', '⠼', '⠴', '⠦', '⠧', '⠇', '⠏']
        # Windows에서 Unicode 지원이 안되는 경우를 위한 폴백
        if os.name == 'nt':
            self.spinner_chars = ['|', '/', '-', '\\']
    
    def spin(self):
        """스피너 애니메이션을 실행하는 메소드"""
        idx = 0
        while self.running:
            char = self.spinner_chars[idx % len(self.spinner_chars)]
            sys.stdout.write(f'\r{char} {self.message}')
            sys.stdout.flush()
            idx += 1
            time.sleep(self.delay)
    
    def start(self):
        """스피너를 시작합니다."""
        self.running = True
        self.spinner_thread = threading.Thread(target=self.spin)
        self.spinner_thread.daemon = True
        self.spinner_thread.start()
    
    def stop(self, final_message=None):
        """스피너를 중지하고 최종 메시지를 출력합니다."""
        self.running = False
        if self.spinner_thread:
            self.spinner_thread.join()
        
        # 스피너 라인 지우기
        sys.stdout.write('\r' + ' ' * (len(self.message) + 10) + '\r')
        sys.stdout.flush()
        
        if final_message:
            safe_print(final_message)

class SmartBuilder:
    def __init__(self, hash_file=".build_hashes.json"):
        self.hash_file = hash_file
        self.project_root = Path.cwd()
        
        # 컴파일 대상 파일들
        self.target_files = [
            "utils.py",
            "plugins/websearch/web_search.py",
            "plugins/rag/rag_process.py",
            # "plugins/rag/win_rag_process.py",
            "license/license_manager.py",
            "services/report/report_generator.py"
        ]
    
    def get_file_hash(self, filepath):
        """파일의 MD5 해시를 계산합니다."""
        if not os.path.exists(filepath):
            return None
        
        hash_md5 = hashlib.md5()
        try:
            with open(filepath, "rb") as f:
                for chunk in iter(lambda: f.read(4096), b""):
                    hash_md5.update(chunk)
            return hash_md5.hexdigest()
        except Exception as e:
            safe_print(f"Warning: Could not calculate hash for {filepath}: {e}")
            return None
    
    def get_compiled_file_path(self, source_file):
        """컴파일된 파일의 경로를 반환합니다."""
        import platform
        import sysconfig
        
        base_name = os.path.splitext(source_file)[0]
        if source_file.endswith('.py'):
            if os.name == 'nt':  # Windows
                return f"{base_name}.pyd"
            else:  # Linux/Mac
                # sysconfig를 사용하여 정확한 확장자 가져오기
                ext_suffix = sysconfig.get_config_var('EXT_SUFFIX')
                if ext_suffix:
                    compiled_name = f"{base_name}{ext_suffix}"
                else:
                    # 폴백: 수동으로 구성
                    arch = platform.machine()
                    platform_name = sys.platform
                    python_version = f"{sys.version_info.major}{sys.version_info.minor}"
                    compiled_name = f"{base_name}.cpython-{python_version}-{arch}-{platform_name}-gnu.so"
                
                # 여러 가능한 경로 확인
                possible_paths = [
                    compiled_name,  # 현재 디렉토리
                    f"build/lib.linux-x86_64-cpython-{sys.version_info.major}{sys.version_info.minor}/{source_file.replace('.py', '')}{ext_suffix or '.so'}",  # build 디렉토리
                    f"build/lib.linux-x86_64-cpython-{sys.version_info.major}{sys.version_info.minor}/{compiled_name}"  # build 디렉토리 (전체 경로)
                ]
                
                # 존재하는 첫 번째 경로 반환
                for path in possible_paths:
                    if os.path.exists(path):
                        return path
                
                # build 디렉토리에서 실제 .so 파일 검색
                build_dir = f"build/lib.linux-x86_64-cpython-{sys.version_info.major}{sys.version_info.minor}"
                if os.path.exists(build_dir):
                    for root, dirs, files in os.walk(build_dir):
                        for file in files:
                            if file.startswith(os.path.basename(base_name)) and file.endswith('.so'):
                                return os.path.join(root, file)
                
                # 존재하지 않으면 기본 경로 반환
                return compiled_name
        return None
    
    def needs_rebuild(self, source_file):
        """파일이 재빌드가 필요한지 확인합니다."""
        if not os.path.exists(source_file):
            safe_print(f"Source file not found: {source_file}")
            return True
        
        # 해시 파일이 없으면 빌드 필요
        if not os.path.exists(self.hash_file):
            safe_print(f"Hash file not found, building all files")
            return True
        
        try:
            with open(self.hash_file, 'r') as f:
                hashes = json.load(f)
        except (json.JSONDecodeError, FileNotFoundError):
            safe_print(f"Invalid hash file, rebuilding all files")
            return True
        
        current_hash = self.get_file_hash(source_file)
        if current_hash is None:
            return True
        
        # 해시가 다르거나 없으면 빌드 필요
        if source_file not in hashes or hashes[source_file] != current_hash:
            return True
        
        # 컴파일된 파일이 없으면 빌드 필요
        compiled_file = self.get_compiled_file_path(source_file)
        if compiled_file and not os.path.exists(compiled_file):
            safe_print(f"Compiled file missing: {compiled_file}")
            return True
        
        return False
    
    def update_build_hash(self, source_file):
        """빌드 후 파일의 해시를 업데이트합니다."""
        try:
            if os.path.exists(self.hash_file):
                with open(self.hash_file, 'r') as f:
                    hashes = json.load(f)
            else:
                hashes = {}
            
            hashes[source_file] = self.get_file_hash(source_file)
            
            with open(self.hash_file, 'w') as f:
                json.dump(hashes, f, indent=2)
            
            safe_print(f"Updated hash for {source_file}")
        except Exception as e:
            safe_print(f"Warning: Could not update build hash for {source_file}: {e}")
    
    def clean_old_files(self):
        """이전 빌드 파일들을 정리합니다."""
        safe_print("Cleaning old compiled files...")
        
        # .so, .pyd, .c 파일들 제거
        for root, dirs, files in os.walk('.'):
            for file in files:
                if file.endswith(('.so', '.pyd', '.c')):
                    file_path = os.path.join(root, file)
                    try:
                        os.remove(file_path)
                        safe_print(f"Removed: {file_path}")
                    except Exception as e:
                        safe_print(f"Warning: Could not remove {file_path}: {e}")
            
            # __pycache__ 디렉토리 제거
            if '__pycache__' in dirs:
                cache_dir = os.path.join(root, '__pycache__')
                try:
                    import shutil
                    shutil.rmtree(cache_dir)
                    safe_print(f"Removed: {cache_dir}")
                except Exception as e:
                    safe_print(f"Warning: Could not remove {cache_dir}: {e}")
    
    def build_files(self, files_to_build):
        """지정된 파일들을 빌드합니다."""
        if not files_to_build:
            safe_print("No files need to be built.")
            return True
        
        safe_print(f"\nBuilding {len(files_to_build)} files...")
        
        # 직접 Cython 컴파일 수행
        try:
            import Cython.Build
            import setuptools.extension
            import platform
            import sysconfig
            
            safe_print("Using direct Cython compilation...")
            
            # 각 파일별로 컴파일
            for i, file_path in enumerate(files_to_build, 1):
                file_name = os.path.basename(file_path)
                spinner_message = f"[{i}/{len(files_to_build)}] Compiling {file_name}..."
                
                spinner = Spinner(spinner_message)
                spinner.start()
                
                try:
                    # 파일 경로를 모듈 이름으로 변환
                    module_name = file_path.replace('/', '.').replace('.py', '')
                    
                    # Extension 객체 생성
                    extension = setuptools.extension.Extension(module_name, [file_path])
                    
                    # 컴파일 옵션
                    compiler_directives = {
                        'language_level': "3",
                        'boundscheck': False,
                        'wraparound': False,
                        'initializedcheck': False,
                        'nonecheck': False,
                        'cdivision': True,
                    }
                    
                    # Cython 컴파일
                    compiled_extensions = Cython.Build.cythonize(
                        [extension], 
                        compiler_directives=compiler_directives,
                        build_dir='build'
                    )
                    
                    # Cython이 C 파일을 생성한 후 실제 컴파일 수행
                    if compiled_extensions:
                        # C 파일이 생성되었는지 확인 (build 디렉토리에서 찾기)
                        c_file = f"build/{file_path.replace('.py', '.c')}"
                        if os.path.exists(c_file):
                            # gcc로 컴파일
                            import sysconfig
                            
                            # 컴파일된 파일 경로
                            ext_suffix = sysconfig.get_config_var('EXT_SUFFIX')
                            if not ext_suffix:
                                python_version = f"{sys.version_info.major}{sys.version_info.minor}"
                                arch = platform.machine()
                                ext_suffix = f".cpython-{python_version}-{arch}-{sys.platform}-gnu.so"
                            
                            output_file = file_path.replace('.py', ext_suffix)
                            
                            # gcc 컴파일 명령
                            gcc_cmd = [
                                'gcc', '-shared', '-fPIC',
                                '-I' + sysconfig.get_path('include'),
                                '-I' + os.path.dirname(sys.executable) + '/include',
                                '-o', output_file,
                                c_file
                            ]
                            
                            try:
                                result = subprocess.run(gcc_cmd, capture_output=True, text=True)
                                if result.returncode == 0:
                                    safe_print(f"  Compiled {output_file}")
                                else:
                                    safe_print(f"  gcc compilation failed: {result.stderr}")
                                    return False
                            except Exception as e:
                                safe_print(f"  gcc compilation error: {e}")
                                return False
                        else:
                            safe_print(f"  Warning: C file not found at {c_file}")
                        
                        spinner.stop(f"✓ [{i}/{len(files_to_build)}] {file_name} compiled successfully")
                        # 빌드된 파일의 해시 업데이트
                        self.update_build_hash(file_path)
                    else:
                        spinner.stop(f"✗ [{i}/{len(files_to_build)}] {file_name} compilation failed")
                        return False
                        
                except Exception as e:
                    spinner.stop(f"✗ [{i}/{len(files_to_build)}] {file_name} compilation error")
                    safe_print(f"Build error for {file_path}: {e}")
                    return False
            
            safe_print(f"\n🎉 All {len(files_to_build)} files compiled successfully!")
            return True
            
        except ImportError as e:
            safe_print(f"Warning: Could not import Cython: {e}")
            safe_print("Falling back to setup.py...")
            
            # setup.py를 사용하여 빌드 실행
            spinner_message = f"Compiling {len(files_to_build)} files with setup.py..."
            spinner = Spinner(spinner_message)
            spinner.start()
            
            try:
                result = subprocess.run([
                    sys.executable, 'setup.py', 'build_ext', '--inplace'
                ], capture_output=True, text=True)
                
                if result.returncode == 0:
                    spinner.stop(f"✓ All files compiled successfully with setup.py")
                    return True
                else:
                    spinner.stop(f"✗ Compilation failed with setup.py")
                    safe_print(f"Error details:")
                    safe_print("STDOUT:", result.stdout)
                    safe_print("STDERR:", result.stderr)
                    return False
                    
            except Exception as e:
                spinner.stop(f"✗ Compilation error with setup.py")
                safe_print(f"Build error: {e}")
                return False
        
        return True
    
    def run(self, clean_first=False):
        """스마트 빌드를 실행합니다."""
        safe_print("=== Smart Python Builder ===")
        safe_print(f"Project root: {self.project_root}")
        safe_print(f"Hash file: {self.hash_file}")
        
        if clean_first:
            self.clean_old_files()
        
        # 빌드가 필요한 파일들 확인
        files_to_build = []
        for file_path in self.target_files:
            if self.needs_rebuild(file_path):
                safe_print(f"[+] {file_path} - needs rebuild")
                files_to_build.append(file_path)
            else:
                safe_print(f"[o] {file_path} - up to date")
        
        if not files_to_build:
            safe_print("\n[SUCCESS] All files are up to date! No compilation needed.")
            return True
        
        safe_print(f"\n[BUILD] Files to build: {len(files_to_build)}")
        for file_path in files_to_build:
            safe_print(f"  - {file_path}")
        
        # 빌드 실행
        success = self.build_files(files_to_build)
        
        if success:
            safe_print("\n[SUCCESS] Smart build completed successfully!")
        else:
            safe_print("\n[ERROR] Smart build failed!")
        
        return success

def main():
    import argparse
    
    parser = argparse.ArgumentParser(description="Smart Python Builder")
    parser.add_argument("--clean", action="store_true", help="Clean old files before building")
    parser.add_argument("--hash-file", default=".build_hashes.json", help="Hash file path")
    
    args = parser.parse_args()
    
    builder = SmartBuilder(hash_file=args.hash_file)
    success = builder.run(clean_first=args.clean)
    
    sys.exit(0 if success else 1)

if __name__ == "__main__":
    main()