import bcrypt from 'bcrypt';
import crypto from 'crypto';

// =============================================================================
// HAMONIZE (AIRUN) DATABASE SCHEMA v2.1
// 전체 시스템 스키마 통합 정리
// =============================================================================

const newCurrentVersion = '3.3';

// 마이그레이션 버전 관리
const createMigrationTable = async (dbManager) => {
  await dbManager.query(`
        CREATE TABLE IF NOT EXISTS schema_migrations (
            version VARCHAR(50) PRIMARY KEY,
            applied_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        );
    `);
};

const getCurrentVersion = async (dbManager) => {
  try {
    await createMigrationTable(dbManager);
    const result = await dbManager.query('SELECT version FROM schema_migrations ORDER BY version DESC LIMIT 1');
    return result.rows[0]?.version || '0';
  } catch (error) {
    console.error('마이그레이션 버전 확인 실패:', error);
    return '0';
  }
};

const recordMigration = async (dbManager, version) => {
  await dbManager.query('INSERT INTO schema_migrations (version) VALUES ($1) ON CONFLICT (version) DO NOTHING', [version]);
};

const checkTableExists = async (dbManager, tableName) => {
  const result = await dbManager.query(
    `
        SELECT EXISTS (
            SELECT FROM information_schema.tables 
            WHERE table_schema = 'public' 
            AND table_name = $1
        );
    `,
    [tableName]
  );
  return result.rows[0].exists;
};

const ensureProjectsTable = async (dbManager) => {
  console.log('projects 테이블 준비 확인 중...');
  try {
    await dbManager.query(`
          CREATE TABLE IF NOT EXISTS projects (
              id SERIAL PRIMARY KEY,
              name VARCHAR(255) NOT NULL,
              description TEXT,
              user_id VARCHAR(100) NOT NULL,
              created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
              updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
              UNIQUE(user_id, name)
          );
      `);
    console.log('projects 테이블 준비 완료');
  } catch (error) {
    console.error('projects 테이블 생성/확인 실패:', error);
    throw error;
  }
};

const createTables = async (dbManager) => {
  try {
    await dbManager.query('BEGIN');

    const currentVersion = await getCurrentVersion(dbManager);
    console.log(`현재 데이터베이스 버전: ${currentVersion}`);

    // 버전 2.2이면 이미 최신
    // if (currentVersion === '2.2') {
    if (currentVersion === newCurrentVersion) {
      console.log('데이터베이스가 이미 최신 버전(' + newCurrentVersion + ')입니다.');
      await dbManager.query('COMMIT');
      return;
    }

    console.log('데이터베이스 마이그레이션 시작 (버전 ' + newCurrentVersion + ' - PR #265 통합 및 스키마 안정성 개선)...');

    await ensureProjectsTable(dbManager);

    // =============================================================================
    // 0. 기존 테이블에 이미지 임베딩 컬럼 추가
    // =============================================================================

    // document_embeddings 테이블에 image_embedding 컬럼 추가
    const tableExists = await checkTableExists(dbManager, 'document_embeddings');
    if (tableExists) {
      console.log('document_embeddings 테이블에 image_embedding 컬럼 추가...');

      // 컬럼이 이미 존재하는지 확인
      const columnExists = await dbManager.query(`
                SELECT EXISTS (
                    SELECT 1 FROM information_schema.columns 
                    WHERE table_name = 'document_embeddings' 
                    AND column_name = 'image_embedding'
                );
            `);

      if (!columnExists.rows[0].exists) {
        // image_embedding 컬럼 추가
        await dbManager.query(`
                    ALTER TABLE document_embeddings 
                    ADD COLUMN image_embedding vector(768);
                `);
        console.log('image_embedding 컬럼 추가 완료');
      } else {
        console.log('image_embedding 컬럼이 이미 존재합니다.');
      }

      const columnExists_isEmbedding = await dbManager.query(`
                SELECT EXISTS (
                    SELECT 1 FROM information_schema.columns 
                    WHERE table_name = 'document_embeddings' 
                    AND column_name = 'is_embedding'
                );
            `);

      if (!columnExists_isEmbedding.rows[0].exists) {
        // image_embedding 컬럼 추가
        await dbManager.query(`
                    ALTER TABLE document_embeddings 
                    ADD COLUMN is_embedding boolean NOT NULL DEFAULT FALSE;
                `);
        console.log('is_embedding 컬럼 추가 완료');
      } else {
        console.log('is_embedding 컬럼이 이미 존재합니다.');
      }
    } else {
      console.log('document_embeddings 테이블이 존재하지 않습니다. 건너뜁니다.');
    }

    // =============================================================================
    // RAG 시스템 무결성 향상을 위한 chat_documents 테이블 확장 (v2.6)
    // =============================================================================

    // chat_documents 테이블에 상태 추적 및 메타데이터 컬럼 추가
    const chatDocTableExists = await checkTableExists(dbManager, 'chat_documents');
    if (chatDocTableExists) {
      console.log('chat_documents 테이블에 RAG 무결성 추적 컬럼 추가...');

      // 상태 추적 컬럼들 추가
      const statusColumns = [
        { name: 'upload_status', definition: "upload_status VARCHAR(20) DEFAULT 'uploaded' CHECK (upload_status IN ('uploading', 'uploaded', 'failed'))", description: '업로드 상태' },
        { name: 'embedding_status', definition: "embedding_status VARCHAR(20) DEFAULT 'pending' CHECK (embedding_status IN ('pending', 'processing', 'completed', 'failed'))", description: '임베딩 상태' },
        { name: 'project_name', definition: 'project_name VARCHAR(100)', description: '프로젝트명' },
        { name: 'project_id', definition: 'project_id INTEGER REFERENCES projects(id) ON DELETE SET NULL', description: '프로젝트 ID (외래키)' },
        { name: 'tags', definition: 'tags TEXT[]', description: '태그 배열' },
        { name: 'metadata', definition: "metadata JSONB DEFAULT '{}'", description: '추가 메타데이터' },
        { name: 'error_message', definition: 'error_message TEXT', description: '오류 메시지' },
        { name: 'processed_at', definition: 'processed_at TIMESTAMP', description: '처리 완료 시간' }
      ];

      for (const column of statusColumns) {
        const columnExists = await dbManager.query(`
          SELECT EXISTS (
            SELECT 1 FROM information_schema.columns 
            WHERE table_name = 'chat_documents' 
            AND column_name = '${column.name}'
          );
        `);

        if (!columnExists.rows[0].exists) {
          await dbManager.query(`
            ALTER TABLE chat_documents 
            ADD COLUMN ${column.definition};
          `);
          console.log(`${column.description} 컬럼 추가 완료: ${column.name}`);
        } else {
          console.log(`${column.description} 컬럼이 이미 존재합니다: ${column.name}`);
        }
      }

      // 인덱스 추가
      await dbManager.query(`
        CREATE INDEX IF NOT EXISTS idx_chat_documents_upload_status ON chat_documents(upload_status);
        CREATE INDEX IF NOT EXISTS idx_chat_documents_embedding_status ON chat_documents(embedding_status);
        CREATE INDEX IF NOT EXISTS idx_chat_documents_project_name ON chat_documents(project_name);
        CREATE INDEX IF NOT EXISTS idx_chat_documents_processed_at ON chat_documents(processed_at);
      `);
      console.log('chat_documents 테이블 인덱스 추가 완료');

      // 기존 레코드의 상태 업데이트 (임베딩 상태 확인 후 설정)
      await dbManager.query(`
        UPDATE chat_documents 
        SET embedding_status = CASE 
          WHEN EXISTS (
            SELECT 1 FROM document_embeddings 
            WHERE document_embeddings.filename LIKE '%' || chat_documents.filename 
            AND document_embeddings.user_id = chat_documents.user_id
            AND is_embedding = true
          ) THEN 'completed'
          ELSE 'pending'
        END,
        processed_at = CASE 
          WHEN EXISTS (
            SELECT 1 FROM document_embeddings 
            WHERE document_embeddings.filename LIKE '%' || chat_documents.filename 
            AND document_embeddings.user_id = chat_documents.user_id
            AND is_embedding = true
          ) THEN updated_at
          ELSE NULL
        END
        WHERE embedding_status IS NULL OR embedding_status = 'pending';
      `);
      console.log('기존 chat_documents 레코드의 상태 업데이트 완료');

    } else {
      console.log('chat_documents 테이블이 존재하지 않습니다. 건너뜁니다.');
    }

    // =============================================================================
    // chat_documents 테이블 UNIQUE 제약 조건 추가 (v2.7)
    // =============================================================================

    if (chatDocTableExists) {
      console.log('chat_documents 테이블에 UNIQUE 제약 조건 추가...');

      // 기존 중복 데이터 확인
      const duplicateCheck = await dbManager.query(`
        SELECT user_id, filename, COUNT(*) as count
        FROM chat_documents
        GROUP BY user_id, filename
        HAVING COUNT(*) > 1;
      `);

      if (duplicateCheck.rows.length > 0) {
        console.log(`중복 데이터 발견: ${duplicateCheck.rows.length}개 그룹`);

        // 중복 데이터 처리 (최신 것만 유지)
        for (const row of duplicateCheck.rows) {
          await dbManager.query(
            `
            DELETE FROM chat_documents
            WHERE user_id = $1 AND filename = $2
            AND id NOT IN (
              SELECT id FROM chat_documents
              WHERE user_id = $1 AND filename = $2
              ORDER BY created_at DESC
              LIMIT 1
            );
          `,
            [row.user_id, row.filename]
          );
        }
        console.log('중복 데이터 정리 완료');
      } else {
        console.log('중복 데이터가 없습니다.');
      }

      // UNIQUE 인덱스 존재 여부 확인
      const indexExists = await dbManager.query(`
        SELECT EXISTS (
          SELECT 1 FROM pg_indexes
          WHERE tablename = 'chat_documents'
          AND indexname = 'idx_chat_documents_user_filename'
        );
      `);

      if (!indexExists.rows[0].exists) {
        // UNIQUE 인덱스 생성
        await dbManager.query(`
          CREATE UNIQUE INDEX idx_chat_documents_user_filename
          ON chat_documents (user_id, filename);
        `);
        console.log('chat_documents 테이블 UNIQUE 인덱스 생성 완료');
      } else {
        console.log('chat_documents 테이블 UNIQUE 인덱스가 이미 존재합니다.');
      }
    }

    // =============================================================================
    // 1. 사용자 인증 및 권한 관리
    // =============================================================================

    // 사용자 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS users (
                id SERIAL PRIMARY KEY,
                username VARCHAR(50) UNIQUE NOT NULL,
                email VARCHAR(255) UNIQUE,
                password_hash VARCHAR(255) NOT NULL,
                name VARCHAR(100),
                role VARCHAR(50) DEFAULT 'user',
                language VARCHAR(10) DEFAULT 'ko',
                status VARCHAR(20) DEFAULT 'active',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                last_login TIMESTAMP
            );
        `);

    // 권한 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS permissions (
                id SERIAL PRIMARY KEY,
                name VARCHAR(100) UNIQUE NOT NULL,
                description TEXT
            );
        `);

    // 역할-권한 매핑 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS role_permissions (
                role VARCHAR(50),
                permission_id INTEGER REFERENCES permissions(id) ON DELETE CASCADE,
                PRIMARY KEY (role, permission_id)
            );
        `);

    // API 키 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS api_keys (
                id SERIAL PRIMARY KEY,
                user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
                key_value VARCHAR(64) UNIQUE NOT NULL,
                name VARCHAR(100),
                status VARCHAR(20) DEFAULT 'active',
                permissions JSONB DEFAULT '{}',
                allowed_ips TEXT[],
                rate_limit INTEGER DEFAULT 1000,
                last_used TIMESTAMP,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                expires_at TIMESTAMP
            );
        `);

    // 활동 로그 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS activity_logs (
                id SERIAL PRIMARY KEY,
                user_id INTEGER REFERENCES users(id) ON DELETE SET NULL,
                action VARCHAR(255),
                details JSONB DEFAULT '{}',
                ip_address VARCHAR(45),
                user_agent TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            );
        `);

    // 사용자 IP 로그 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS user_ip_logs (
                id SERIAL PRIMARY KEY,
                user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
                ip_address VARCHAR(45) NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            );
        `);

    // =============================================================================
    // 2. 세션 관리
    // =============================================================================

    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS sessions (
                id VARCHAR(64) PRIMARY KEY,
                user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
                username VARCHAR(100),
                data JSONB NOT NULL DEFAULT '{}',
                provider VARCHAR(50),
                model VARCHAR(100),
                title VARCHAR(255),
                message_count INTEGER DEFAULT 0,
                last_message TEXT,
                status VARCHAR(20) DEFAULT 'active',
                type VARCHAR(20) DEFAULT 'chat',
                created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
                expires_at TIMESTAMP WITH TIME ZONE
            );
        `);

    // =============================================================================
    // 3. RAG 및 임베딩 시스템
    // =============================================================================

    // pgvector 확장 생성
    await dbManager.query(`CREATE EXTENSION IF NOT EXISTS vector;`);

    // 문서 임베딩 테이블 (통일된 스키마)
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS document_embeddings (
                id SERIAL PRIMARY KEY,
                is_embedding boolean NOT NULL DEFAULT FALSE,
                doc_id VARCHAR(500) NOT NULL,
                filename VARCHAR(500) NOT NULL,
                chunk_index INTEGER NOT NULL,        -- 청크 인덱스 (0부터 시작)
                chunk_text TEXT NOT NULL,
                embedding vector(1024),             -- 1024차원 임베딩 벡터
                image_embedding vector(512),        -- 512차원 이미지 임베딩 벡터
                user_id VARCHAR(255),               -- 사용자별 필터링
                source TEXT,                        -- 파일 경로
                file_mtime BIGINT,                  -- 파일 수정 시간 (Unix timestamp)
                metadata JSONB DEFAULT '{}',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                
                -- 복합 유니크 제약조건
                CONSTRAINT unique_doc_chunk UNIQUE (doc_id, chunk_index)
            );
        `);

    // =============================================================================
    // 4. 보고서 시스템
    // =============================================================================

    // 보고서 생성 작업 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS report_jobs (
                id SERIAL PRIMARY KEY,
                job_id VARCHAR(255) UNIQUE NOT NULL,
                user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
                username VARCHAR(100),
                title VARCHAR(500),
                executive_summary TEXT,
                template VARCHAR(100),
                output_format VARCHAR(10) DEFAULT 'pdf',
                status VARCHAR(20) DEFAULT 'queued',
                progress INTEGER DEFAULT 0,
                output_path VARCHAR(1000),
                cache_key VARCHAR(255),
                error_message TEXT,
                metadata JSONB DEFAULT '{}',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                completed_at TIMESTAMP
            );
        `);

    // 보고서 섹션 캐시 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS report_section_cache (
                id SERIAL PRIMARY KEY,
                job_id VARCHAR(255) NOT NULL,
                section_name VARCHAR(255) NOT NULL,
                subsection_name VARCHAR(255),
                content TEXT,
                chart_data JSONB,
                table_data JSONB,
                cache_key VARCHAR(255),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                expires_at TIMESTAMP,
                FOREIGN KEY (job_id) REFERENCES report_jobs(job_id) ON DELETE CASCADE
            );
        `);

    // 사용자별 보고서 설정 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS user_report_settings (
                id SERIAL PRIMARY KEY,
                user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
                setting_key VARCHAR(255) NOT NULL,
                setting_value TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                UNIQUE(user_id, setting_key)
            );
        `);

    // =============================================================================
    // 5. 작업 스케줄러 시스템
    // =============================================================================

    // 저장된 작업 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS saved_tasks (
                id SERIAL PRIMARY KEY,
                task_id VARCHAR(255) UNIQUE NOT NULL,
                user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
                name VARCHAR(255) NOT NULL,
                description TEXT,
                prompt TEXT NOT NULL,
                python_code TEXT,
                agent_options JSONB DEFAULT '{}',
                execution_count INTEGER DEFAULT 0,
                success_count INTEGER DEFAULT 0,
                metadata JSONB DEFAULT '{}',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                last_used TIMESTAMP
            );
        `);

    // 예약 작업 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS scheduled_tasks (
                id SERIAL PRIMARY KEY,
                schedule_id VARCHAR(255) UNIQUE NOT NULL,
                user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
                task_id VARCHAR(255) NOT NULL,
                name VARCHAR(255) NOT NULL,
                schedule_type VARCHAR(20) NOT NULL CHECK (schedule_type IN ('once', 'recurring')),
                scheduled_at TIMESTAMP NOT NULL,
                recurring_pattern JSONB,
                status VARCHAR(20) DEFAULT 'active' CHECK (status IN ('active', 'inactive', 'completed', 'failed', 'running')),
                next_execution TIMESTAMP,
                last_execution TIMESTAMP,
                execution_count INTEGER DEFAULT 0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (task_id) REFERENCES saved_tasks(task_id) ON DELETE CASCADE
            );
        `);

    // 작업 실행 로그 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS task_execution_logs (
                id SERIAL PRIMARY KEY,
                execution_id VARCHAR(255) UNIQUE NOT NULL,
                task_id VARCHAR(255) NOT NULL,
                schedule_id VARCHAR(255),
                user_id INTEGER REFERENCES users(id) ON DELETE SET NULL,
                execution_type VARCHAR(20) NOT NULL CHECK (execution_type IN ('manual', 'scheduled')),
                status VARCHAR(20) NOT NULL CHECK (status IN ('pending', 'running', 'completed', 'failed', 'cancelled')),
                python_code TEXT,
                stdout TEXT,
                stderr TEXT,
                exit_code INTEGER,
                error_message TEXT,
                agent_job_id VARCHAR(255),
                execution_time_ms INTEGER,
                metadata JSONB DEFAULT '{}',
                started_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                completed_at TIMESTAMP,
                FOREIGN KEY (task_id) REFERENCES saved_tasks(task_id) ON DELETE CASCADE,
                FOREIGN KEY (schedule_id) REFERENCES scheduled_tasks(schedule_id) ON DELETE SET NULL
            );
        `);

    // =============================================================================
    // 6. 웹 포털 및 지원 시스템
    // =============================================================================

    // 기술지원 티켓 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS support_tickets (
                id SERIAL PRIMARY KEY,
                title VARCHAR(255) NOT NULL,
                question TEXT NOT NULL,
                status VARCHAR(20) NOT NULL DEFAULT 'open',
                author_name VARCHAR(255) NOT NULL,
                author_role VARCHAR(20) NOT NULL,
                attachment_image BYTEA,
                attachment_image_mimetype VARCHAR(100),
                created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
            );
        `);

    // 기술지원 티켓 답변 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS support_ticket_replies (
                id SERIAL PRIMARY KEY,
                ticket_id INTEGER NOT NULL REFERENCES support_tickets(id) ON DELETE CASCADE,
                content TEXT NOT NULL,
                author_name VARCHAR(255) NOT NULL,
                author_role VARCHAR(20) NOT NULL,
                images BYTEA[] DEFAULT '{}',
                image_mimetypes VARCHAR(100)[] DEFAULT '{}',
                created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
            );
        `);

    // 기술지원 문서 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS support_documents (
                id SERIAL PRIMARY KEY,
                filename VARCHAR(255) NOT NULL,
                filepath TEXT NOT NULL,
                filesize BIGINT,
                mimetype VARCHAR(100),
                user_id VARCHAR(255),
                created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
            );
        `);

    // 채팅 페이지 문서 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS chat_documents (
                id SERIAL PRIMARY KEY,
                filename VARCHAR(255) NOT NULL,
                filepath VARCHAR(500) NOT NULL,
                filesize BIGINT,
                mimetype VARCHAR(100),
                user_id VARCHAR(100) NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            );
        `);

    // 프로젝트 테이블 (ensureProjectsTable에서 처리)

    // 사용자 메모리 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS user_memories (
                id SERIAL PRIMARY KEY,
                user_id VARCHAR(255) NOT NULL,
                title VARCHAR(255) NOT NULL,
                content TEXT NOT NULL,
                category VARCHAR(100) DEFAULT 'general',
                importance_level INTEGER DEFAULT 1 CHECK (importance_level BETWEEN 1 AND 5),
                tags TEXT[],
                is_active BOOLEAN DEFAULT true,
                access_count INTEGER DEFAULT 0,
                created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
                last_accessed TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
            );
        `);

    // =============================================================================
    // 7. FlowAI 워크플로우 시스템
    // =============================================================================

    // 워크플로우 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS flowai_workflows (
                id VARCHAR(255) PRIMARY KEY,
                name VARCHAR(255) NOT NULL,
                description TEXT,
                nodes TEXT NOT NULL DEFAULT '[]',
                connections TEXT NOT NULL DEFAULT '[]',
                variables TEXT DEFAULT '{}',
                user_id VARCHAR(255) NOT NULL,
                is_public BOOLEAN DEFAULT false,
                tags TEXT DEFAULT '[]',
                created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
            );
        `);

    // 워크플로우 실행 기록 테이블
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS flowai_executions (
                id SERIAL PRIMARY KEY,
                workflow_id VARCHAR(255) NOT NULL REFERENCES flowai_workflows(id) ON DELETE CASCADE,
                user_id VARCHAR(255) NOT NULL,
                status VARCHAR(20) NOT NULL DEFAULT 'pending',
                results TEXT DEFAULT '[]',
                total_duration INTEGER,
                error_message TEXT,
                final_output TEXT,
                start_time TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
                end_time TIMESTAMP WITH TIME ZONE
            );
        `);

    // =============================================================================
    // 8. AI 학습 및 피드백 시스템
    // =============================================================================

    // 질의-답변 데이터셋 테이블 (LoRA 학습용)
    await dbManager.query(`
            CREATE TABLE IF NOT EXISTS qa_dataset (
                id SERIAL PRIMARY KEY,
                user_question TEXT NOT NULL,
                ai_answer TEXT NOT NULL,
                user_id VARCHAR(100) NOT NULL,
                rating INTEGER NOT NULL CHECK (rating >= 1 AND rating <= 5),
                feedback_type VARCHAR(50) DEFAULT 'thumbs',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            );
        `);

    // =============================================================================
    // 9. 지식 그래프 시스템
    // =============================================================================

    // 지식 그래프 엔티티 테이블 (기존 테이블명 유지)
    await dbManager.query(`
        CREATE TABLE IF NOT EXISTS graph_entities (
            id SERIAL PRIMARY KEY,
            name VARCHAR(255) NOT NULL,
            type VARCHAR(100) NOT NULL,
            description TEXT,
            properties JSONB DEFAULT '{}',
            source_documents TEXT[],
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(name, type)
        );
    `);

    // 지식 그래프 관계 테이블 (기존 테이블명 유지)
    await dbManager.query(`
        CREATE TABLE IF NOT EXISTS graph_relationships (
            id SERIAL PRIMARY KEY,
            source_entity_id INTEGER REFERENCES graph_entities(id) ON DELETE CASCADE,
            target_entity_id INTEGER REFERENCES graph_entities(id) ON DELETE CASCADE,
            relationship_type VARCHAR(100) NOT NULL,
            description TEXT,
            weight FLOAT DEFAULT 1.0,
            properties JSONB DEFAULT '{}',
            source_documents TEXT[],
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        );
    `);

    // 지식 그래프 커뮤니티 테이블 (기존 테이블명 유지)
    await dbManager.query(`
        CREATE TABLE IF NOT EXISTS graph_communities (
            id SERIAL PRIMARY KEY,
            name VARCHAR(255) NOT NULL,
            description TEXT,
            entities INTEGER[],
            level INTEGER DEFAULT 0,
            properties JSONB DEFAULT '{}',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        );
    `);

    // =============================================================================
    // 10. 비즈니스 공고 분석 시스템
    // =============================================================================
    // 비즈니스 공고 메인 테이블
    await dbManager.query(`
        CREATE TABLE IF NOT EXISTS business_announcements (
            id VARCHAR(255) PRIMARY KEY,                    -- 통합 ID (source_external_id)
            source VARCHAR(50) NOT NULL,                    -- BIZINFO, NTIS, PPS 등
            external_id VARCHAR(255) NOT NULL,              -- 원본 시스템의 ID
            
            -- 공고 기본 정보
            title TEXT NOT NULL,                            -- 공고 제목
            description TEXT,                               -- 공고 설명
            category VARCHAR(100),                          -- 주요 카테고리 (기술, 창업, 수출 등)
            subcategory VARCHAR(100),                       -- 세부 카테고리
            
            -- 예산 및 지원 정보
            budget BIGINT,                                  -- 예산 (원)
            support_amount BIGINT,                          -- 지원금액 (원)
            support_ratio FLOAT,                            -- 지원비율 (%)
            
            -- 기간 정보
            announcement_date TIMESTAMP,                    -- 공고일
            application_start TIMESTAMP,                    -- 신청 시작일
            application_end TIMESTAMP,                      -- 신청 마감일
            project_start TIMESTAMP,                        -- 사업 시작일
            project_end TIMESTAMP,                          -- 사업 종료일
            
            -- 자격 요건 (JSON)
            eligibility JSONB DEFAULT '{}',                 -- {company_size: [], industry: [], location: []}
            requirements JSONB DEFAULT '{}',                -- {technology: [], certification: [], personnel: []}
            
            -- 평가 정보 (JSON)
            evaluation JSONB DEFAULT '{}',                  -- {criteria: [], weights: []}
            
            -- 연락처 정보 (JSON)
            contact JSONB DEFAULT '{}',                     -- {name: "", department: "", phone: "", email: ""}
            
            -- 문서 정보 (JSON)
            documents JSONB DEFAULT '{}',                   -- {attachments: [], application_form: "", guidelines: ""}
            
            -- 상태 및 메타데이터
            status VARCHAR(20) DEFAULT 'active',            -- active, closed, cancelled
            view_count INTEGER DEFAULT 0,                   -- 조회수
            
            -- 원본 데이터
            raw_data JSONB DEFAULT '{}',                    -- 원본 API/스크래핑 데이터
            
            -- 시스템 정보
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        );
    `);

    // 기업 정보 테이블
    await dbManager.query(`
        CREATE TABLE IF NOT EXISTS companies (
            id SERIAL PRIMARY KEY,
            business_number VARCHAR(20) UNIQUE NOT NULL,    -- 사업자등록번호
            company_name VARCHAR(200) NOT NULL,             -- 회사명
            
            -- 기업 분류
            company_size VARCHAR(20),                       -- 중소기업, 중견기업, 대기업, 스타트업
            industry_code VARCHAR(10),                      -- 표준산업분류코드
            industry_name VARCHAR(100),                     -- 업종명
            
            -- 위치 정보
            address TEXT,                                   -- 주소
            region VARCHAR(50),                             -- 지역 (서울, 경기 등)
            
            -- 기업 현황
            employee_count INTEGER,                         -- 직원 수
            annual_revenue BIGINT,                          -- 연매출 (원)
            establishment_date TIMESTAMP,                   -- 설립일
            
            -- 기술 및 역량 (JSON)
            technologies JSONB DEFAULT '{}',                -- 보유 기술 목록
            certifications JSONB DEFAULT '{}',              -- 보유 인증 목록
            capabilities JSONB DEFAULT '{}',                -- 핵심 역량
            
            -- 사업 이력 (JSON)
            business_history JSONB DEFAULT '{}',            -- 과거 정부사업 참여 이력
            financial_info JSONB DEFAULT '{}',              -- 재무 정보
            
            -- 연락처
            contact_person VARCHAR(50),                     -- 담당자명
            contact_phone VARCHAR(20),                      -- 연락처
            contact_email VARCHAR(100),                     -- 이메일
            
            -- 시스템 정보
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        );
    `);

    // 기업-공고 매칭 결과 테이블
    await dbManager.query(`
        CREATE TABLE IF NOT EXISTS company_announcement_matches (
            id SERIAL PRIMARY KEY,
            company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
            announcement_id VARCHAR(255) REFERENCES business_announcements(id) ON DELETE CASCADE,
            
            -- 매칭 점수
            total_score FLOAT NOT NULL,                     -- 전체 매칭 점수 (0-100)
            category_score FLOAT,                           -- 카테고리 적합성 점수
            size_score FLOAT,                               -- 기업규모 적합성 점수
            location_score FLOAT,                           -- 지역 적합성 점수
            technology_score FLOAT,                         -- 기술 적합성 점수
            experience_score FLOAT,                         -- 경험 적합성 점수
            
            -- 매칭 상세 정보 (JSON)
            match_details JSONB DEFAULT '{}',               -- 매칭 근거 및 상세 정보
            
            -- 추천 상태
            recommendation_status VARCHAR(20) DEFAULT 'pending',  -- pending, accepted, rejected
            recommendation_reason TEXT,                     -- 추천 이유
            
            -- 시스템 정보
            matched_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        );
    `);

    // 사업 제안서 테이블
    await dbManager.query(`
        CREATE TABLE IF NOT EXISTS business_proposals (
            id SERIAL PRIMARY KEY,
            company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
            announcement_id VARCHAR(255) REFERENCES business_announcements(id) ON DELETE CASCADE,
            
            -- 제안서 기본 정보
            proposal_title VARCHAR(500) NOT NULL,           -- 제안서 제목
            proposal_summary TEXT,                          -- 제안서 요약
            
            -- 제안서 구성 (JSON)
            sections JSONB DEFAULT '{}',                    -- 제안서 섹션별 내용
            
            -- 생성 정보
            generation_method VARCHAR(50),                  -- AI, template, manual
            template_used VARCHAR(100),                     -- 사용된 템플릿
            ai_model_used VARCHAR(100),                     -- 사용된 AI 모델
            
            -- 상태 정보
            status VARCHAR(20) DEFAULT 'draft',             -- draft, final, submitted
            version INTEGER DEFAULT 1,                      -- 버전 번호
            
            -- 파일 정보
            file_path VARCHAR(500),                         -- 생성된 파일 경로
            file_format VARCHAR(10),                        -- pdf, docx, hwp
            file_size INTEGER,                              -- 파일 크기 (bytes)
            
            -- 평가 정보
            quality_score FLOAT,                            -- 제안서 품질 점수
            completeness_score FLOAT,                       -- 완성도 점수
            
            -- 시스템 정보
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        );
    `);

    // 데이터 수집 로그 테이블
    await dbManager.query(`
        CREATE TABLE IF NOT EXISTS data_collection_logs (
            id SERIAL PRIMARY KEY,
            
            -- 수집 정보
            source VARCHAR(50) NOT NULL,                    -- BIZINFO, NTIS, PPS 등
            collection_type VARCHAR(20) NOT NULL,           -- api, scraping
            
            -- 수집 결과
            total_items INTEGER DEFAULT 0,                  -- 총 수집 항목 수
            new_items INTEGER DEFAULT 0,                    -- 신규 항목 수
            updated_items INTEGER DEFAULT 0,                -- 업데이트 항목 수
            failed_items INTEGER DEFAULT 0,                 -- 실패 항목 수
            
            -- 실행 정보
            start_time TIMESTAMP NOT NULL,
            end_time TIMESTAMP,
            duration_seconds INTEGER,                       -- 실행 시간 (초)
            
            -- 상태 및 오류
            status VARCHAR(20) NOT NULL,                    -- success, partial, failed
            error_message TEXT,                             -- 오류 메시지
            
            -- 상세 로그 (JSON)
            details JSONB DEFAULT '{}',                     -- 상세 실행 로그
            
            -- 시스템 정보
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        );
    `);

    // 시스템 설정 테이블
    await dbManager.query(`
        CREATE TABLE IF NOT EXISTS system_configurations (
            id SERIAL PRIMARY KEY,
            
            -- 설정 정보
            config_key VARCHAR(100) NOT NULL UNIQUE,
            config_value TEXT,                              -- JSON 문자열로 저장
            config_type VARCHAR(20) DEFAULT 'string',       -- string, number, boolean, json
            
            -- 설명
            description TEXT,
            category VARCHAR(50),                           -- collection, matching, generation 등
            
            -- 시스템 정보
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        );
    `);

    // =============================================================================
    // 11. 컨텍스트 메모리 시스템 (v3.0)
    // =============================================================================

    // 컨텍스트 메모리 테이블 - 사용자별 장기 메모리 저장
    await dbManager.query(`
      CREATE TABLE IF NOT EXISTS context_memories (
        id SERIAL PRIMARY KEY,
        user_id VARCHAR(255) NOT NULL,
        session_id VARCHAR(255),                    -- 원본 세션 ID (선택적)

        -- 메모리 분류
        memory_type VARCHAR(50) NOT NULL CHECK (memory_type IN (
          'entity', 'fact', 'preference', 'pattern', 'skill', 'project', 'relationship'
        )),

        -- 메모리 내용
        content TEXT NOT NULL,
        summary TEXT,                               -- 요약 (검색용)
        entities JSONB DEFAULT '[]',                -- 관련 엔티티들 ['React', 'JavaScript', 'Frontend']
        keywords TEXT[],                            -- 검색 키워드들

        -- 중요도 및 신뢰도
        importance_score FLOAT DEFAULT 0.5 CHECK (importance_score >= 0 AND importance_score <= 1),
        confidence_score FLOAT DEFAULT 0.8 CHECK (confidence_score >= 0 AND confidence_score <= 1),

        -- 시간 정보
        valid_from TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        valid_until TIMESTAMP,                      -- NULL이면 무기한 유효

        -- 메타데이터
        source_context TEXT,                        -- 원본 대화 컨텍스트
        reference_count INTEGER DEFAULT 0,         -- 참조된 횟수
        last_referenced TIMESTAMP,                 -- 마지막 참조 시간

        -- 벡터 임베딩 (의미적 검색용)
        embedding vector(1024),                     -- 텍스트 임베딩

        -- 시스템 정보
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `);

    // 컨텍스트 메모리 중복 방지를 위한 UNIQUE 제약조건 추가
    await dbManager.query(`
      DO $$
      BEGIN
        IF NOT EXISTS (
          SELECT 1 FROM pg_constraint
          WHERE conname = 'unique_user_memory_content'
        ) THEN
          ALTER TABLE context_memories
          ADD CONSTRAINT unique_user_memory_content
          UNIQUE (user_id, memory_type, content);
        END IF;
      END $$;
    `);

    // 사용자 선호도 테이블 - 명시적 선호도 저장
    await dbManager.query(`
      CREATE TABLE IF NOT EXISTS user_preferences (
        id SERIAL PRIMARY KEY,
        user_id VARCHAR(255) NOT NULL,
        
        -- 선호도 분류
        category VARCHAR(100) NOT NULL,             -- 'technology', 'framework', 'language', 'tool' 등
        preference_key VARCHAR(255) NOT NULL,       -- 'frontend_framework', 'database_type' 등
        preference_value TEXT NOT NULL,             -- 'React', 'PostgreSQL' 등
        
        -- 선호도 강도
        strength FLOAT DEFAULT 0.5 CHECK (strength >= 0 AND strength <= 1),
        confidence FLOAT DEFAULT 0.8 CHECK (confidence >= 0 AND confidence <= 1),
        
        -- 학습 정보
        learned_from VARCHAR(50) DEFAULT 'conversation',  -- 'conversation', 'explicit', 'inferred'
        evidence_count INTEGER DEFAULT 1,          -- 증거 횟수
        
        -- 시간 정보
        first_observed TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        last_confirmed TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        
        -- 메타데이터
        notes TEXT,
        tags TEXT[],
        
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        
        -- 유니크 제약조건
        UNIQUE(user_id, category, preference_key)
      );
    `);

    // 컨텍스트 관계 테이블 - 메모리 간 관계 저장
    await dbManager.query(`
      CREATE TABLE IF NOT EXISTS context_relationships (
        id SERIAL PRIMARY KEY,
        user_id VARCHAR(255) NOT NULL,
        
        -- 관계 정보
        source_memory_id INTEGER REFERENCES context_memories(id) ON DELETE CASCADE,
        target_memory_id INTEGER REFERENCES context_memories(id) ON DELETE CASCADE,
        
        -- 관계 타입 및 강도
        relationship_type VARCHAR(50) NOT NULL,     -- 'related_to', 'depends_on', 'conflicts_with' 등
        strength FLOAT DEFAULT 0.5 CHECK (strength >= 0 AND strength <= 1),
        
        -- 메타데이터
        description TEXT,
        evidence TEXT,                              -- 관계의 근거
        
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `);

    // 컨텍스트 세션 요약 테이블 - 세션별 중요 정보 요약
    await dbManager.query(`
      CREATE TABLE IF NOT EXISTS context_session_summaries (
        id SERIAL PRIMARY KEY,
        session_id VARCHAR(255) UNIQUE NOT NULL,
        user_id VARCHAR(255) NOT NULL,
        
        -- 요약 정보
        title VARCHAR(255),
        summary TEXT,
        key_topics TEXT[],                          -- 주요 주제들
        mentioned_entities TEXT[],                  -- 언급된 엔티티들
        
        -- 세션 메타데이터
        message_count INTEGER DEFAULT 0,
        session_type VARCHAR(50),
        provider VARCHAR(50),
        model VARCHAR(100),
        
        -- 중요도 및 통계
        importance_score FLOAT DEFAULT 0.5,
        engagement_score FLOAT DEFAULT 0.5,        -- 사용자 참여도
        
        -- 시간 정보
        session_start TIMESTAMP,
        session_end TIMESTAMP,
        duration_minutes INTEGER,
        
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `);

    console.log('컨텍스트 메모리 시스템 테이블 생성 완료');

    // =============================================================================
    // 12. 사용자별 시스템프롬프트 컬럼 추가 (v3.1)
    // =============================================================================

    // users 테이블에 system_prompt 컬럼 추가
    const usersTableExists = await checkTableExists(dbManager, 'users');
    if (usersTableExists) {
      console.log('users 테이블에 system_prompt 컬럼 추가...');

      // system_prompt 컬럼이 이미 존재하는지 확인
      const systemPromptColumnExists = await dbManager.query(`
        SELECT EXISTS (
          SELECT 1 FROM information_schema.columns
          WHERE table_name = 'users'
          AND column_name = 'system_prompt'
        );
      `);

      if (!systemPromptColumnExists.rows[0].exists) {
        // system_prompt 컬럼 추가
        await dbManager.query(`
          ALTER TABLE users
          ADD COLUMN system_prompt TEXT;
        `);
        console.log('사용자별 시스템프롬프트 컬럼 추가 완료');
      } else {
        console.log('system_prompt 컬럼이 이미 존재합니다.');
      }
    } else {
      console.log('users 테이블이 존재하지 않습니다. 건너뜁니다.');
    }

    console.log('사용자별 시스템프롬프트 시스템 완료');

    // =============================================================================
    // 13. 웹훅 시스템 (v3.3)
    // =============================================================================

    // 웹훅 등록 테이블
    await dbManager.query(`
      CREATE TABLE IF NOT EXISTS webhooks (
        id SERIAL PRIMARY KEY,
        user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
        url VARCHAR(500) NOT NULL,
        events TEXT[] NOT NULL,
        secret VARCHAR(255),
        is_active BOOLEAN DEFAULT true,
        retry_count INTEGER DEFAULT 3,
        timeout_seconds INTEGER DEFAULT 30,
        headers JSONB DEFAULT '{}',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `);

    // 웹훅 전송 로그 테이블
    await dbManager.query(`
      CREATE TABLE IF NOT EXISTS webhook_logs (
        id SERIAL PRIMARY KEY,
        webhook_id INTEGER REFERENCES webhooks(id) ON DELETE CASCADE,
        event_type VARCHAR(100) NOT NULL,
        payload JSONB,
        response_status INTEGER,
        response_body TEXT,
        response_headers JSONB,
        attempts INTEGER DEFAULT 1,
        delivered_at TIMESTAMP,
        error_message TEXT,
        execution_time_ms INTEGER,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `);

    // 웹훅 인덱스 생성
    await dbManager.query(`
      CREATE INDEX IF NOT EXISTS idx_webhooks_user_id ON webhooks(user_id);
      CREATE INDEX IF NOT EXISTS idx_webhooks_active ON webhooks(is_active);
      CREATE INDEX IF NOT EXISTS idx_webhooks_events ON webhooks USING GIN(events);
      
      CREATE INDEX IF NOT EXISTS idx_webhook_logs_webhook_id ON webhook_logs(webhook_id);
      CREATE INDEX IF NOT EXISTS idx_webhook_logs_event_type ON webhook_logs(event_type);
      CREATE INDEX IF NOT EXISTS idx_webhook_logs_created_at ON webhook_logs(created_at);
      CREATE INDEX IF NOT EXISTS idx_webhook_logs_status ON webhook_logs(response_status);
    `);

    console.log('웹훅 시스템 테이블 생성 완료');

    // =============================================================================
    // 13. 건강검진 배치 처리 시스템 (v2.9 - 기존 유지)
    // =============================================================================

    // 건강검진 배치 작업 테이블
    await dbManager.query(`
      CREATE TABLE IF NOT EXISTS health_batch_jobs (
        id VARCHAR(255) PRIMARY KEY,
        user_id VARCHAR(255) NOT NULL,
        filename VARCHAR(255) NOT NULL,
        status VARCHAR(20) DEFAULT 'pending' CHECK (status IN ('pending', 'processing', 'completed', 'failed')),
        total_records INTEGER DEFAULT 0,
        processed_records INTEGER DEFAULT 0,
        successful_records INTEGER DEFAULT 0,
        failed_records INTEGER DEFAULT 0,
        progress FLOAT DEFAULT 0,
        current_patient VARCHAR(255),
        error_message TEXT,
        metadata JSONB DEFAULT '{}',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        completed_at TIMESTAMP
      );
    `);

    // 건강검진 배치 환자 데이터 테이블
    await dbManager.query(`
      CREATE TABLE IF NOT EXISTS health_batch_patients (
        id SERIAL PRIMARY KEY,
        job_id VARCHAR(255) NOT NULL REFERENCES health_batch_jobs(id) ON DELETE CASCADE,
        patient_name VARCHAR(255) NOT NULL,
        row_number INTEGER NOT NULL,
        
        -- 원시 건강 데이터 (동적 컬럼 지원을 위해 JSONB 사용)
        raw_data JSONB DEFAULT '{}',
        
        -- AI 분석 결과 컬럼들
        blood_pressure_score INTEGER DEFAULT 3,    -- 혈압 해당 (1: 해당, 3: 비해당)
        cholesterol_score INTEGER DEFAULT 3,       -- 지질 해당 (1: 해당, 3: 비해당)  
        hba1c_received INTEGER DEFAULT 3,          -- HbA1c 수진 유무 (1: 수진, 3: 미수진)
        blood_sugar_score INTEGER DEFAULT 3,       -- 혈당 해당 (1: 해당, 3: 비해당)
        obesity_score INTEGER DEFAULT 3,           -- 비만 해당 (1: 해당, 3: 비해당)
        urine_protein_score INTEGER DEFAULT 1,     -- 요단백 (1~9)
        past_history_treatment TEXT,               -- 기왕력 (병명)
        current_history_treatment TEXT,            -- 현병력 (병명)
        final_judgment INTEGER DEFAULT 3,          -- 최종 판정 (1:고위험, 2:중위험, 3:저위험, 4:대상외)
        
        -- 분석 상태
        analysis_status VARCHAR(20) DEFAULT 'pending' CHECK (analysis_status IN ('pending', 'processing', 'completed', 'failed')),
        analysis_error TEXT,
        analyzed_at TIMESTAMP,
        
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `);

    // 인덱스 생성
    await dbManager.query(`
      CREATE INDEX IF NOT EXISTS idx_health_batch_jobs_user_id ON health_batch_jobs(user_id);
      CREATE INDEX IF NOT EXISTS idx_health_batch_jobs_status ON health_batch_jobs(status);
      CREATE INDEX IF NOT EXISTS idx_health_batch_jobs_created_at ON health_batch_jobs(created_at);
      
      CREATE INDEX IF NOT EXISTS idx_health_batch_patients_job_id ON health_batch_patients(job_id);
      CREATE INDEX IF NOT EXISTS idx_health_batch_patients_analysis_status ON health_batch_patients(analysis_status);
      CREATE INDEX IF NOT EXISTS idx_health_batch_patients_patient_name ON health_batch_patients(patient_name);
    `);

    console.log('건강검진 배치 처리 테이블 생성 완료');

    // =============================================================================
    // 14. 인덱스 생성 (성능 최적화)
    // =============================================================================

    console.log('인덱스 생성 중...');

    // 사용자 및 인증 관련 인덱스
    await dbManager.query(`
            CREATE INDEX IF NOT EXISTS idx_users_username ON users(username);
            CREATE INDEX IF NOT EXISTS idx_users_email ON users(email);
            CREATE INDEX IF NOT EXISTS idx_users_role ON users(role);
            CREATE INDEX IF NOT EXISTS idx_users_status ON users(status);
            CREATE INDEX IF NOT EXISTS idx_api_keys_key_value ON api_keys(key_value);
            CREATE INDEX IF NOT EXISTS idx_api_keys_user_id ON api_keys(user_id);
            CREATE INDEX IF NOT EXISTS idx_activity_logs_user_id ON activity_logs(user_id);
            CREATE INDEX IF NOT EXISTS idx_activity_logs_created_at ON activity_logs(created_at);
            CREATE INDEX IF NOT EXISTS idx_user_ip_logs_user_id ON user_ip_logs(user_id);
        `);

    // 세션 관련 인덱스
    await dbManager.query(`
            CREATE INDEX IF NOT EXISTS idx_sessions_user_id ON sessions(user_id);
            CREATE INDEX IF NOT EXISTS idx_sessions_username ON sessions(username);
            CREATE INDEX IF NOT EXISTS idx_sessions_created_at ON sessions(created_at);
            CREATE INDEX IF NOT EXISTS idx_sessions_updated_at ON sessions(updated_at);
            CREATE INDEX IF NOT EXISTS idx_sessions_type ON sessions(type);
            CREATE INDEX IF NOT EXISTS idx_sessions_status ON sessions(status);
        `);

    // RAG 및 임베딩 관련 인덱스
    await dbManager.query(`
            CREATE INDEX IF NOT EXISTS idx_document_embeddings_doc_id ON document_embeddings(doc_id);
            CREATE INDEX IF NOT EXISTS idx_document_embeddings_filename ON document_embeddings(filename);
            CREATE INDEX IF NOT EXISTS idx_document_embeddings_chunk_index ON document_embeddings(chunk_index);
            CREATE INDEX IF NOT EXISTS idx_document_embeddings_user_id ON document_embeddings(user_id);
            CREATE INDEX IF NOT EXISTS idx_document_embeddings_source ON document_embeddings(source);
            CREATE INDEX IF NOT EXISTS idx_document_embeddings_created_at ON document_embeddings(created_at);
            CREATE INDEX IF NOT EXISTS idx_document_embeddings_has_image ON document_embeddings((image_embedding IS NOT NULL));
            CREATE INDEX IF NOT EXISTS idx_document_embeddings_is_embedding ON document_embeddings(is_embedding);
            
        `);

    // 벡터 유사도 검색을 위한 HNSW 인덱스 (데이터가 많을 때 효과적)
    await dbManager.query(`
            CREATE INDEX IF NOT EXISTS idx_document_embeddings_vector_cosine 
            ON document_embeddings USING ivfflat (embedding vector_cosine_ops) 
            WITH (lists = 100);
        `);

    // 이미지 임베딩 벡터 검색을 위한 인덱스
    await dbManager.query(`
            CREATE INDEX IF NOT EXISTS idx_document_embeddings_image_vector_cosine 
            ON document_embeddings USING ivfflat (image_embedding vector_cosine_ops) 
            WITH (lists = 100);
        `);

    // 보고서 관련 인덱스
    await dbManager.query(`
            CREATE INDEX IF NOT EXISTS idx_report_jobs_user_id ON report_jobs(user_id);
            CREATE INDEX IF NOT EXISTS idx_report_jobs_username ON report_jobs(username);
            CREATE INDEX IF NOT EXISTS idx_report_jobs_status ON report_jobs(status);
            CREATE INDEX IF NOT EXISTS idx_report_jobs_created_at ON report_jobs(created_at);
            CREATE INDEX IF NOT EXISTS idx_report_section_cache_job_id ON report_section_cache(job_id);
            CREATE INDEX IF NOT EXISTS idx_report_section_cache_expires_at ON report_section_cache(expires_at);
            CREATE INDEX IF NOT EXISTS idx_user_report_settings_user_id ON user_report_settings(user_id);
        `);

    // 작업 스케줄러 관련 인덱스
    await dbManager.query(`
            CREATE INDEX IF NOT EXISTS idx_saved_tasks_user_id ON saved_tasks(user_id);
            CREATE INDEX IF NOT EXISTS idx_saved_tasks_task_id ON saved_tasks(task_id);
            CREATE INDEX IF NOT EXISTS idx_saved_tasks_last_used ON saved_tasks(last_used);
            CREATE INDEX IF NOT EXISTS idx_scheduled_tasks_user_id ON scheduled_tasks(user_id);
            CREATE INDEX IF NOT EXISTS idx_scheduled_tasks_task_id ON scheduled_tasks(task_id);
            CREATE INDEX IF NOT EXISTS idx_scheduled_tasks_status ON scheduled_tasks(status);
            CREATE INDEX IF NOT EXISTS idx_scheduled_tasks_next_execution ON scheduled_tasks(next_execution);
            CREATE INDEX IF NOT EXISTS idx_task_execution_logs_task_id ON task_execution_logs(task_id);
            CREATE INDEX IF NOT EXISTS idx_task_execution_logs_schedule_id ON task_execution_logs(schedule_id);
            CREATE INDEX IF NOT EXISTS idx_task_execution_logs_user_id ON task_execution_logs(user_id);
            CREATE INDEX IF NOT EXISTS idx_task_execution_logs_status ON task_execution_logs(status);
            CREATE INDEX IF NOT EXISTS idx_task_execution_logs_started_at ON task_execution_logs(started_at);
        `);

    // 웹 포털 관련 인덱스
    await dbManager.query(`
            CREATE INDEX IF NOT EXISTS idx_support_tickets_status ON support_tickets(status);
            CREATE INDEX IF NOT EXISTS idx_support_tickets_created_at ON support_tickets(created_at);
            CREATE INDEX IF NOT EXISTS idx_support_ticket_replies_ticket_id ON support_ticket_replies(ticket_id);
            CREATE INDEX IF NOT EXISTS idx_support_documents_user_id ON support_documents(user_id);
            CREATE INDEX IF NOT EXISTS idx_support_documents_created_at ON support_documents(created_at);
            CREATE INDEX IF NOT EXISTS idx_chat_documents_user_id ON chat_documents(user_id);
            CREATE INDEX IF NOT EXISTS idx_chat_documents_created_at ON chat_documents(created_at);
            CREATE INDEX IF NOT EXISTS idx_user_memories_user_id ON user_memories(user_id);
            CREATE INDEX IF NOT EXISTS idx_user_memories_category ON user_memories(category);
            CREATE INDEX IF NOT EXISTS idx_user_memories_importance ON user_memories(importance_level);
            CREATE INDEX IF NOT EXISTS idx_user_memories_active ON user_memories(is_active);
        `);

    // FlowAI 관련 인덱스
    await dbManager.query(`
            CREATE INDEX IF NOT EXISTS idx_flowai_workflows_user_id ON flowai_workflows(user_id);
            CREATE INDEX IF NOT EXISTS idx_flowai_workflows_public ON flowai_workflows(is_public);
            CREATE INDEX IF NOT EXISTS idx_flowai_workflows_updated ON flowai_workflows(updated_at);
            CREATE INDEX IF NOT EXISTS idx_flowai_executions_workflow ON flowai_executions(workflow_id);
            CREATE INDEX IF NOT EXISTS idx_flowai_executions_user ON flowai_executions(user_id);
            CREATE INDEX IF NOT EXISTS idx_flowai_executions_status ON flowai_executions(status);
        `);

    // AI 피드백 관련 인덱스
    await dbManager.query(`
            CREATE INDEX IF NOT EXISTS idx_qa_dataset_rating ON qa_dataset(rating);
            CREATE INDEX IF NOT EXISTS idx_qa_dataset_user_id ON qa_dataset(user_id);
            CREATE INDEX IF NOT EXISTS idx_qa_dataset_created_at ON qa_dataset(created_at);
        `);

    // 지식 그래프 시스템 인덱스 (기존 테이블명)
    await dbManager.query(`
            -- 지식 그래프 엔티티 인덱스
            CREATE INDEX IF NOT EXISTS idx_graph_entities_name ON graph_entities(name);
            CREATE INDEX IF NOT EXISTS idx_graph_entities_type ON graph_entities(type);
            CREATE INDEX IF NOT EXISTS idx_graph_entities_created_at ON graph_entities(created_at);

            -- 지식 그래프 관계 인덱스
            CREATE INDEX IF NOT EXISTS idx_graph_relationships_source ON graph_relationships(source_entity_id);
            CREATE INDEX IF NOT EXISTS idx_graph_relationships_target ON graph_relationships(target_entity_id);
            CREATE INDEX IF NOT EXISTS idx_graph_relationships_type ON graph_relationships(relationship_type);
            CREATE INDEX IF NOT EXISTS idx_graph_relationships_weight ON graph_relationships(weight);

            -- 지식 그래프 커뮤니티 인덱스
            CREATE INDEX IF NOT EXISTS idx_graph_communities_name ON graph_communities(name);
            CREATE INDEX IF NOT EXISTS idx_graph_communities_level ON graph_communities(level);
        `);

    // 비즈니스 공고 시스템 인덱스
    await dbManager.query(`
            -- 비즈니스 공고 인덱스
            CREATE INDEX IF NOT EXISTS idx_announcement_source_external ON business_announcements(source, external_id);
            CREATE INDEX IF NOT EXISTS idx_announcement_category ON business_announcements(category);
            CREATE INDEX IF NOT EXISTS idx_announcement_deadline ON business_announcements(application_end);
            CREATE INDEX IF NOT EXISTS idx_announcement_status ON business_announcements(status);
            CREATE INDEX IF NOT EXISTS idx_announcement_created_at ON business_announcements(created_at);
            
            -- 기업 인덱스
            CREATE INDEX IF NOT EXISTS idx_company_business_number ON companies(business_number);
            CREATE INDEX IF NOT EXISTS idx_company_size_industry ON companies(company_size, industry_code);
            CREATE INDEX IF NOT EXISTS idx_company_region ON companies(region);
            CREATE INDEX IF NOT EXISTS idx_company_created_at ON companies(created_at);
            
            -- 매칭 인덱스
            CREATE INDEX IF NOT EXISTS idx_match_company_score ON company_announcement_matches(company_id, total_score);
            CREATE INDEX IF NOT EXISTS idx_match_announcement_score ON company_announcement_matches(announcement_id, total_score);
            CREATE INDEX IF NOT EXISTS idx_match_status ON company_announcement_matches(recommendation_status);
            CREATE INDEX IF NOT EXISTS idx_match_matched_at ON company_announcement_matches(matched_at);
            
            -- 제안서 인덱스
            CREATE INDEX IF NOT EXISTS idx_proposal_company_status ON business_proposals(company_id, status);
            CREATE INDEX IF NOT EXISTS idx_proposal_announcement ON business_proposals(announcement_id);
            CREATE INDEX IF NOT EXISTS idx_proposal_created_at ON business_proposals(created_at);
            CREATE INDEX IF NOT EXISTS idx_proposal_quality_score ON business_proposals(quality_score);
            
            -- 수집 로그 인덱스
            CREATE INDEX IF NOT EXISTS idx_collection_log_source_time ON data_collection_logs(source, start_time);
            CREATE INDEX IF NOT EXISTS idx_collection_log_status ON data_collection_logs(status);
            CREATE INDEX IF NOT EXISTS idx_collection_log_created_at ON data_collection_logs(created_at);
            
            -- 시스템 설정 인덱스
            CREATE INDEX IF NOT EXISTS idx_system_config_key ON system_configurations(config_key);
            CREATE INDEX IF NOT EXISTS idx_system_config_category ON system_configurations(category);
        `);

    // 컨텍스트 메모리 시스템 인덱스
    await dbManager.query(`
            -- 컨텍스트 메모리 인덱스
            CREATE INDEX IF NOT EXISTS idx_context_memories_user_id ON context_memories(user_id);
            CREATE INDEX IF NOT EXISTS idx_context_memories_type ON context_memories(memory_type);
            CREATE INDEX IF NOT EXISTS idx_context_memories_importance ON context_memories(importance_score);
            CREATE INDEX IF NOT EXISTS idx_context_memories_valid_period ON context_memories(valid_from, valid_until);
            CREATE INDEX IF NOT EXISTS idx_context_memories_keywords ON context_memories USING GIN(keywords);
            CREATE INDEX IF NOT EXISTS idx_context_memories_entities ON context_memories USING GIN(entities);
            CREATE INDEX IF NOT EXISTS idx_context_memories_created_at ON context_memories(created_at);
            CREATE INDEX IF NOT EXISTS idx_context_memories_reference_count ON context_memories(reference_count);
            
            -- 사용자 선호도 인덱스
            CREATE INDEX IF NOT EXISTS idx_user_preferences_user_id ON user_preferences(user_id);
            CREATE INDEX IF NOT EXISTS idx_user_preferences_category ON user_preferences(category);
            CREATE INDEX IF NOT EXISTS idx_user_preferences_strength ON user_preferences(strength);
            CREATE INDEX IF NOT EXISTS idx_user_preferences_last_confirmed ON user_preferences(last_confirmed);
            
            -- 컨텍스트 관계 인덱스
            CREATE INDEX IF NOT EXISTS idx_context_relationships_user_id ON context_relationships(user_id);
            CREATE INDEX IF NOT EXISTS idx_context_relationships_source ON context_relationships(source_memory_id);
            CREATE INDEX IF NOT EXISTS idx_context_relationships_target ON context_relationships(target_memory_id);
            CREATE INDEX IF NOT EXISTS idx_context_relationships_type ON context_relationships(relationship_type);
            
            -- 세션 요약 인덱스
            CREATE INDEX IF NOT EXISTS idx_context_session_summaries_user_id ON context_session_summaries(user_id);
            CREATE INDEX IF NOT EXISTS idx_context_session_summaries_session_id ON context_session_summaries(session_id);
            CREATE INDEX IF NOT EXISTS idx_context_session_summaries_importance ON context_session_summaries(importance_score);
            CREATE INDEX IF NOT EXISTS idx_context_session_summaries_topics ON context_session_summaries USING GIN(key_topics);
            CREATE INDEX IF NOT EXISTS idx_context_session_summaries_entities ON context_session_summaries USING GIN(mentioned_entities);
        `);

    // =============================================================================
    // 15. 기본 데이터 삽입
    // =============================================================================

    console.log('기본 데이터 확인 및 삽입...');

    // 기본 권한 데이터 삽입
    await dbManager.query(`
            INSERT INTO permissions (name, description) VALUES
                ('read:report', '보고서 읽기 권한'),
                ('write:report', '보고서 작성 권한'),
                ('delete:report', '보고서 삭제 권한'),
                ('share:report', '보고서 공유 권한'),
                ('manage:users', '사용자 관리 권한'),
                ('manage:api_keys', 'API 키 관리 권한'),
                ('admin', '관리자 권한')
            ON CONFLICT (name) DO NOTHING;
        `);

    // 기본 역할-권한 매핑
    await dbManager.query(`
            INSERT INTO role_permissions (role, permission_id)
            SELECT 'admin', id FROM permissions
            ON CONFLICT DO NOTHING;
        `);

    await dbManager.query(`
            INSERT INTO role_permissions (role, permission_id)
            SELECT 'manager', id FROM permissions 
            WHERE name IN ('read:report', 'write:report', 'share:report', 'manage:api_keys')
            ON CONFLICT DO NOTHING;
        `);

    await dbManager.query(`
            INSERT INTO role_permissions (role, permission_id)
            SELECT 'user', id FROM permissions 
            WHERE name IN ('read:report', 'write:report')
            ON CONFLICT DO NOTHING;
        `);

    // 기본 사용자 생성 (존재하지 않을 경우만)
    const existingUsers = await dbManager.query('SELECT COUNT(*) as count FROM users');
    if (parseInt(existingUsers.rows[0].count) === 0) {
      // 관리자 계정
      const adminPassword = await bcrypt.hash('admin1234', 10);
      const adminResult = await dbManager.query(
        `
                INSERT INTO users (username, email, password_hash, name, role, status)
                VALUES ('admin', 'admin@airun.local', $1, 'Admin User', 'admin', 'active')
                RETURNING id;
            `,
        [adminPassword]
      );

      // 데모 사용자 계정
      const demoPassword = await bcrypt.hash('demo', 10);
      const demoResult = await dbManager.query(
        `
                INSERT INTO users (username, email, password_hash, name, role, status)
                VALUES ('demo', 'demo@airun.local', $1, 'Demo User', 'user', 'active')
                RETURNING id;
            `,
        [demoPassword]
      );

      // 기본 API 키 생성
      if (adminResult.rows.length > 0) {
        const adminId = adminResult.rows[0].id;
        const randomKey = crypto.randomBytes(16).toString('hex');
        const keyValue = `airun_${adminId}_${randomKey}`;

        await dbManager.query(
          `
                    INSERT INTO api_keys (
                        user_id, name, key_value, permissions, rate_limit, status
                    ) VALUES ($1, $2, $3, $4, $5, $6)
                `,
          [
            adminId,
            '기본 API 키',
            keyValue,
            {
              reports: { read: true, write: true, delete: true, share: true },
              users: { read: true, write: true, manage: true }
            },
            1000,
            'active'
          ]
        );
      }

      console.log('기본 사용자 생성 완료');
      console.log('관리자 계정: admin / admin1234');
      console.log('데모 계정: demo / demo');
    }

    // 비즈니스 공고 시스템 기본 설정 데이터 삽입
    await dbManager.query(`
        INSERT INTO system_configurations (config_key, config_value, config_type, description, category) VALUES
            ('data_collection_schedule', '{"interval_hours": 6, "enabled": true}', 'json', '데이터 수집 스케줄 설정', 'collection'),
            ('matching_threshold', '70.0', 'number', '기업-공고 매칭 최소 임계점', 'matching'),
            ('proposal_template_path', 'templates/business_proposal.docx', 'string', '제안서 템플릿 파일 경로', 'generation'),
            ('ai_model_config', '{"provider": "openai", "model": "gpt-4", "temperature": 0.7}', 'json', 'AI 모델 설정', 'generation'),
            ('notification_settings', '{"email_enabled": false, "webhook_enabled": false}', 'json', '알림 설정', 'notification')
        ON CONFLICT (config_key) DO NOTHING;
    `);

    console.log('비즈니스 공고 시스템 기본 설정 완료');

    // =============================================================================
    // PostgreSQL 타임아웃 설정 마이그레이션 (v2.8)
    // =============================================================================

    // 마이그레이션 버전 기록
    await recordMigration(dbManager, newCurrentVersion);
    await dbManager.query('COMMIT');

    if (currentVersion < '2.8') {
      console.log('PostgreSQL 타임아웃 설정 마이그레이션 시작...');

      try {
        // PostgreSQL 타임아웃 설정 업데이트
        await dbManager.query(`ALTER SYSTEM SET statement_timeout = '300s';`);
        await dbManager.query(`ALTER SYSTEM SET idle_in_transaction_session_timeout = '600s';`);

        // 설정 다시 로드
        await dbManager.query(`SELECT pg_reload_conf();`);

        console.log('PostgreSQL 타임아웃 설정 완료:');
        console.log('- statement_timeout: 300s (5분)');
        console.log('- idle_in_transaction_session_timeout: 600s (10분)');
      } catch (timeoutError) {
        console.warn('PostgreSQL 타임아웃 설정 실패:', timeoutError.message);
        console.warn('이 설정은 수동으로 적용해야 할 수 있습니다.');
      }
    }
    console.log('데이터베이스 마이그레이션 완료 (버전 ' + newCurrentVersion + ' - PR #265 통합 및 스키마 안정성 개선)');
  } catch (error) {
    console.error('데이터베이스 마이그레이션 중 오류 발생:', error);
    await dbManager.query('ROLLBACK');
    throw error;
  }
};

export { createTables };
