import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import { dirname } from 'path';
import { getVarVal, setVarVal } from '../../configuration.js';
import { getAppConfigPath } from '../core/index.js';
import dbManager from '../../services/database/index.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

/**
 * 프롬프트 파일의 경로를 가져옵니다.
 */
export async function getPromptPath() {
    const configDir = await getAppConfigPath();
    return path.join(configDir, 'prompt.txt');
}

/**
 * 원본 프롬프트 내용을 읽어옵니다.
 */
export async function readOriginalPrompt() {
    const templatePath = path.join(__dirname, '../../prompts', 'prompt.template');
    return await fs.promises.readFile(templatePath, 'utf8');
}

/**
 * 현재 프롬프트 내용을 읽어옵니다.
 */
export async function readPrompt() {
    try {
        const promptPath = await getPromptPath();
        
        // 프롬프트 파일이 없으면 템플릿에서 복사
        if (!fs.existsSync(promptPath)) {
            await resetPrompt();
        }
        
        return await fs.promises.readFile(promptPath, 'utf8');
    } catch (error) {
        console.error('Error reading prompt:', error);
        throw new Error('Failed to read system prompt');
    }
}

/**
 * 프롬프트를 초기화합니다.
 */
export async function resetPrompt(mode = 'code') {
    try {
        // 현재 선택된 LLM 확인
        const useLLM = await getVarVal('USE_LLM');
        
        // USE_RAG와 USE_WEB_SEARCH를 no로 설정
        await setVarVal('USE_RAG', 'no', true);
        await setVarVal('USE_WEB_SEARCH', 'no', true);
        
        // 각 LLM별 템플릿 파일 경로
        let templatePath;
        
        if (mode === 'code' && useLLM) {
            // code 모드일 경우 prompt.template.code.프로바이더명 형식의 파일 사용
            const codeTemplatePath = path.join(__dirname, '../../prompts', `prompt.template.code.${useLLM.toLowerCase()}`);
            if (fs.existsSync(codeTemplatePath)) {
                templatePath = codeTemplatePath;
                // console.debug(`Using ${templatePath}`);
            } else {
                // 프로바이더별 코드 템플릿이 없으면 기본 코드 템플릿 사용
                templatePath = path.join(__dirname, '../../prompts', 'prompt.template.code');
            }
        } else {
            // 기본 템플릿 경로
            templatePath = path.join(__dirname, '../../prompts', `prompt.template${useLLM ? '.' + useLLM.toLowerCase() : ''}`);
        }
        
        // 템플릿 파일이 존재하는지 확인
        let templateContent;
        if (fs.existsSync(templatePath)) {
            // 선택된 템플릿 사용
            templateContent = await fs.promises.readFile(templatePath, 'utf8');
        } else {
            // 기본 템플릿 사용
            templateContent = await readOriginalPrompt();
        }
        
        // 설정 디렉토리 확인 및 생성
        const configDir = await getAppConfigPath();
        if (!fs.existsSync(configDir)) {
            await fs.promises.mkdir(configDir, { recursive: true });
        }
        
        // prompt.txt 파일에 템플릿 내용 저장
        const promptPath = await getPromptPath();
        await fs.promises.writeFile(promptPath, templateContent, 'utf8');
        
        // 저장이 성공적으로 이루어졌는지 확인
        if (!fs.existsSync(promptPath)) {
            throw new Error('Failed to verify saved prompt file');
        }
        await setVarVal('AIMODE', mode, true);
        return true;
    } catch (error) {
        console.error('Error resetting prompt:', error);
        throw new Error('Failed to reset system prompt');
    }
}

/**
 * RAG 프롬프트를 초기화합니다.
 */
export async function resetRagPrompt() {
    try {
        // AUTO_EXECUTE와 USE_REVIEW를 no로 설정
        await setVarVal('AUTO_EXECUTE', 'no', true);
        await setVarVal('USE_REVIEW', 'no', true);

        const useLLM = await getVarVal('USE_LLM');
        const mode = await getVarVal('AIMODE');
        
        let templatePath;
        // 프로바이더별 chat 모드 템플릿 파일 경로 설정
        if (useLLM && mode === 'chat') {
            // prompt.template.chat.프로바이더명 형식의 파일 사용
            const chatTemplatePath = path.join(__dirname, '../../prompts', `prompt.template.chat.${useLLM.toLowerCase()}`);
            if (fs.existsSync(chatTemplatePath)) {
                templatePath = chatTemplatePath;
                // console.debug(`Using ${templatePath}`);
            } else {
                // 프로바이더별 템플릿이 없으면 기본 chat 템플릿 사용
                templatePath = path.join(__dirname, '../../prompts', 'prompt.template.chat');
            }
        } else {
            // 기본 chat 템플릿 사용
            templatePath = path.join(__dirname, '../../prompts', 'prompt.template.chat');
        }
        
        // 템플릿 파일이 존재하는지 확인
        let templateContent;
        if (fs.existsSync(templatePath)) {
            templateContent = await fs.promises.readFile(templatePath, 'utf8');
        } else {
            // RAG 템플릿이 없으면 기본 템플릿 사용
            templateContent = await readOriginalPrompt();
        }
        
        // 설정 디렉토리 확인 및 생성
        const configDir = await getAppConfigPath();
        if (!fs.existsSync(configDir)) {
            await fs.promises.mkdir(configDir, { recursive: true });
        }
        
        // prompt.txt 파일에 템플릿 내용 저장
        const promptPath = await getPromptPath();
        await fs.promises.writeFile(promptPath, templateContent, 'utf8');
        
        // 저장이 성공적으로 이루어졌는지 확인
        if (!fs.existsSync(promptPath)) {
            throw new Error('Failed to verify saved prompt file');
        }
        await setVarVal('AIMODE', 'chat', true);
        return true;
    } catch (error) {
        console.error('Error resetting RAG prompt:', error);
        throw new Error('Failed to reset RAG prompt');
    }
}

/**
 * RAG 프롬프트를 읽어옵니다.
 */
export async function readRagPrompt() {
    try {
        const promptPath = await getPromptPath();
        
        // 프롬프트 파일이 없으면 템플릿에서 복사
        if (!fs.existsSync(promptPath)) {
            await resetRagPrompt();
        }
        
        return await fs.promises.readFile(promptPath, 'utf8');
    } catch (error) {
        console.error('Error reading prompt:', error);
        throw new Error('Failed to read RAG prompt');
    }
}

/**
 * 프롬프트를 저장합니다.
 */
export async function savePrompt({ content }) {
    try {
        if (!content) {
            throw new Error('Content is required for saving prompt');
        }

        const promptPath = await getPromptPath();

        // 설정 디렉토리가 없으면 생성
        const configDir = await getAppConfigPath();
        if (!fs.existsSync(configDir)) {
            await fs.promises.mkdir(configDir, { recursive: true });
        }

        // 파일 저장
        await fs.promises.writeFile(promptPath, content, 'utf8');

        // 파일이 실제로 저장되었는지 확인
        if (!fs.existsSync(promptPath)) {
            throw new Error('Failed to verify saved prompt file');
        }

        return true;
    } catch (error) {
        console.error('Error saving prompt:', error);
        throw new Error(`Failed to save system prompt: ${error.message}`);
    }
}

/**
 * 사용자별 시스템프롬프트를 데이터베이스에서 읽어옵니다.
 */
export async function readUserSystemPrompt(userId) {
    if (!userId) {
        throw new Error('User ID is required');
    }

    try {
        await dbManager.initialize();
        const result = await dbManager.query(
            'SELECT system_prompt FROM users WHERE id = $1',
            [userId]
        );

        if (result.rows.length > 0) {
            return result.rows[0].system_prompt;
        }

        return null;
    } catch (error) {
        console.error('Error reading user system prompt:', error);
        throw new Error(`Failed to read user system prompt: ${error.message}`);
    }
}

/**
 * 사용자별 시스템프롬프트를 데이터베이스에 저장합니다.
 */
export async function saveUserSystemPrompt(userId, content) {
    if (!userId) {
        throw new Error('User ID is required');
    }

    if (typeof content !== 'string') {
        throw new Error('Content must be a string');
    }

    try {
        await dbManager.initialize();
        await dbManager.query(
            'UPDATE users SET system_prompt = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
            [content, userId]
        );

        return true;
    } catch (error) {
        console.error('Error saving user system prompt:', error);
        throw new Error(`Failed to save user system prompt: ${error.message}`);
    }
}

/**
 * 사용자별 RAG 프롬프트를 읽어옵니다. (사용자 설정이 있으면 우선 사용, 없으면 기본 템플릿)
 */
export async function readUserRagPrompt(userId) {
    try {
        console.log(`[DEBUG-PROMPT] readUserRagPrompt 호출됨 - userId: ${userId}`);

        // 먼저 사용자별 시스템프롬프트 확인
        if (userId) {
            console.log(`[DEBUG-PROMPT] 사용자 ${userId}의 프롬프트 조회 시도`);
            const userPrompt = await readUserSystemPrompt(userId);
            if (userPrompt) {
                console.log(`[DEBUG-PROMPT] ✅ 사용자 ${userId}의 커스텀 프롬프트 발견: ${userPrompt.substring(0, 50)}...`);
                return userPrompt;
            } else {
                console.log(`[DEBUG-PROMPT] ❌ 사용자 ${userId}의 커스텀 프롬프트 없음 - 기본 템플릿 사용`);
            }
        }

        // 사용자 설정이 없으면 기본 RAG 프롬프트 사용
        console.log(`[DEBUG-PROMPT] 기본 RAG 프롬프트 로드 시도`);
        const defaultPrompt = await readRagPrompt();
        console.log(`[DEBUG-PROMPT] 기본 RAG 프롬프트 로드됨: ${defaultPrompt.substring(0, 50)}...`);
        return defaultPrompt;
    } catch (error) {
        console.error('Error reading user RAG prompt:', error);
        // 오류가 발생해도 기본 프롬프트라도 반환하도록 시도
        try {
            return await readRagPrompt();
        } catch (fallbackError) {
            console.error('Error reading fallback RAG prompt:', fallbackError);
            throw new Error('Failed to read any system prompt');
        }
    }
} 