import crypto from 'crypto';
import axios from 'axios';

class WebhookService {
    constructor(db) {
        this.db = db;
        this.retryDelays = [1000, 3000, 9000]; // 1초, 3초, 9초 (지수 백오프)
    }

    async registerWebhook(userId, url, events, secret = null, options = {}) {
        const query = `
            INSERT INTO webhooks (user_id, url, events, secret, is_active, retry_count, timeout_seconds, headers)
            VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
            RETURNING *
        `;

        const values = [
            userId,
            url,
            events,
            secret,
            options.isActive !== false,
            options.retryCount || 3,
            options.timeoutSeconds || 30,
            JSON.stringify(options.headers || {})
        ];

        try {
            const result = await this.db.query(query, values);
            return { success: true, webhook: result.rows[0] };
        } catch (error) {
            console.error('웹훅 등록 실패:', error);
            return { success: false, error: error.message };
        }
    }

    async getWebhooks(userId) {
        const query = `
            SELECT * FROM webhooks
            WHERE user_id = $1 AND is_active = true
            ORDER BY created_at DESC
        `;

        try {
            const result = await this.db.query(query, [userId]);
            return { success: true, webhooks: result.rows };
        } catch (error) {
            console.error('웹훅 조회 실패:', error);
            return { success: false, error: error.message };
        }
    }

    async updateWebhook(webhookId, userId, updates) {
        const allowedFields = ['url', 'events', 'secret', 'is_active', 'retry_count', 'timeout_seconds', 'headers'];
        const setClause = [];
        const values = [];
        let valueIndex = 1;

        for (const [field, value] of Object.entries(updates)) {
            if (allowedFields.includes(field)) {
                setClause.push(`${field} = $${valueIndex}`);
                values.push(field === 'headers' ? JSON.stringify(value) : value);
                valueIndex++;
            }
        }

        if (setClause.length === 0) {
            return { success: false, error: '업데이트할 필드가 없습니다' };
        }

        setClause.push(`updated_at = CURRENT_TIMESTAMP`);
        values.push(webhookId, userId);

        const query = `
            UPDATE webhooks
            SET ${setClause.join(', ')}
            WHERE id = $${valueIndex} AND user_id = $${valueIndex + 1}
            RETURNING *
        `;

        try {
            const result = await this.db.query(query, values);
            if (result.rows.length === 0) {
                return { success: false, error: '웹훅을 찾을 수 없습니다' };
            }
            return { success: true, webhook: result.rows[0] };
        } catch (error) {
            console.error('웹훅 업데이트 실패:', error);
            return { success: false, error: error.message };
        }
    }

    async deleteWebhook(webhookId, userId) {
        const query = `
            UPDATE webhooks
            SET is_active = false, updated_at = CURRENT_TIMESTAMP
            WHERE id = $1 AND user_id = $2
            RETURNING *
        `;

        try {
            const result = await this.db.query(query, [webhookId, userId]);
            if (result.rows.length === 0) {
                return { success: false, error: '웹훅을 찾을 수 없습니다' };
            }
            return { success: true, message: '웹훅이 비활성화되었습니다' };
        } catch (error) {
            console.error('웹훅 삭제 실패:', error);
            return { success: false, error: error.message };
        }
    }

    generateSignature(payload, secret) {
        if (!secret) return null;
        return 'sha256=' + crypto
            .createHmac('sha256', secret)
            .update(JSON.stringify(payload))
            .digest('hex');
    }

    async triggerWebhook(userId, eventType, payload) {
        const query = `
            SELECT * FROM webhooks
            WHERE user_id = $1 AND is_active = true
            AND $2 = ANY(events)
        `;

        try {
            const result = await this.db.query(query, [userId, eventType]);
            const webhooks = result.rows;

            if (webhooks.length === 0) {
                console.log(`사용자 ${userId}에 대한 ${eventType} 이벤트 웹훅이 없습니다`);
                return;
            }

            for (const webhook of webhooks) {
                await this.deliverWebhook(webhook, eventType, payload);
            }
        } catch (error) {
            console.error('웹훅 트리거 실패:', error);
        }
    }

    async deliverWebhook(webhook, eventType, payload, attempt = 1) {
        const startTime = Date.now();
        const webhookPayload = {
            event: eventType,
            timestamp: new Date().toISOString(),
            data: payload
        };

        const headers = {
            'Content-Type': 'application/json',
            'User-Agent': 'Hamonize-Webhook/1.0',
            'X-Webhook-Event': eventType,
            'X-Webhook-Delivery': crypto.randomUUID(),
            ...JSON.parse(webhook.headers || '{}')
        };

        if (webhook.secret) {
            headers['X-Webhook-Signature'] = this.generateSignature(webhookPayload, webhook.secret);
        }

        try {
            const response = await axios({
                method: 'POST',
                url: webhook.url,
                data: webhookPayload,
                headers,
                timeout: webhook.timeout_seconds * 1000,
                validateStatus: (status) => status < 500
            });

            const executionTime = Date.now() - startTime;
            const isSuccess = response.status >= 200 && response.status < 300;

            await this.logWebhookDelivery(webhook.id, eventType, webhookPayload, {
                status: response.status,
                body: response.data,
                headers: response.headers,
                attempts: attempt,
                delivered: isSuccess,
                executionTime
            });

            if (!isSuccess && attempt < webhook.retry_count) {
                const delay = this.retryDelays[Math.min(attempt - 1, this.retryDelays.length - 1)];
                setTimeout(() => {
                    this.deliverWebhook(webhook, eventType, payload, attempt + 1);
                }, delay);
            }

            return isSuccess;

        } catch (error) {
            const executionTime = Date.now() - startTime;

            await this.logWebhookDelivery(webhook.id, eventType, webhookPayload, {
                status: 0,
                body: null,
                headers: null,
                attempts: attempt,
                delivered: false,
                executionTime,
                error: error.message
            });

            if (attempt < webhook.retry_count) {
                const delay = this.retryDelays[Math.min(attempt - 1, this.retryDelays.length - 1)];
                setTimeout(() => {
                    this.deliverWebhook(webhook, eventType, payload, attempt + 1);
                }, delay);
            }

            return false;
        }
    }

    async logWebhookDelivery(webhookId, eventType, payload, result) {
        const query = `
            INSERT INTO webhook_logs
            (webhook_id, event_type, payload, response_status, response_body, response_headers,
             attempts, delivered_at, error_message, execution_time_ms)
            VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
        `;

        const values = [
            webhookId,
            eventType,
            JSON.stringify(payload),
            result.status || null,
            result.body && typeof result.body === 'object' ? JSON.stringify(result.body) : (result.body || null),
            result.headers ? JSON.stringify(result.headers) : null,
            result.attempts,
            result.delivered ? new Date() : null,
            result.error || null,
            result.executionTime
        ];

        try {
            await this.db.query(query, values);
        } catch (error) {
            console.error('웹훅 로그 저장 실패:', error);
        }
    }

    async getWebhookLogs(webhookId, userId, limit = 50) {
        const query = `
            SELECT wl.* FROM webhook_logs wl
            JOIN webhooks w ON wl.webhook_id = w.id
            WHERE w.id = $1 AND w.user_id = $2
            ORDER BY wl.created_at DESC
            LIMIT $3
        `;

        try {
            const result = await this.db.query(query, [webhookId, userId, limit]);
            return { success: true, logs: result.rows };
        } catch (error) {
            console.error('웹훅 로그 조회 실패:', error);
            return { success: false, error: error.message };
        }
    }

    async testWebhook(webhookId, userId) {
        const query = `
            SELECT * FROM webhooks
            WHERE id = $1 AND user_id = $2 AND is_active = true
        `;

        try {
            const result = await this.db.query(query, [webhookId, userId]);
            if (result.rows.length === 0) {
                return { success: false, error: '웹훅을 찾을 수 없습니다' };
            }

            const webhook = result.rows[0];
            const testPayload = {
                message: '웹훅 테스트 메시지입니다',
                userId: userId,
                timestamp: new Date().toISOString()
            };

            const success = await this.deliverWebhook(webhook, 'webhook.test', testPayload);

            return {
                success: true,
                delivered: success,
                message: success ? '웹훅 테스트가 성공했습니다' : '웹훅 전송에 실패했습니다'
            };
        } catch (error) {
            console.error('웹훅 테스트 실패:', error);
            return { success: false, error: error.message };
        }
    }
}

export default WebhookService;