import apiClient, { ApiResponse } from './apiClient';

// 보고서 옵션 인터페이스
export interface ReportOptions {
  format?: 'pdf' | 'hwpx' | 'docx';
  template?: string;
  filepath?: string;
}

// 보고서 응답 인터페이스
export interface ReportResponse {
  success: boolean;
  file_path: string;
  format: string;
  template: string;
  message: string;
  timestamp: string;
}

// 보고서 작업 상태 인터페이스
export interface ReportJobStatus {
  jobId: string;
  state: string;
  success?: boolean;
  file_path?: string;
  format?: string;
  template?: string;
  message: string;
  error?: string;
  timestamp: string;
}

// 보고서 템플릿 인터페이스
export interface ReportTemplate {
  id: string;
  name: string;
  description: string;
  format: string[];
  sections: string[];
  example?: string;
}

// 문서 서비스 클래스
class DocumentService {
  // 보고서 생성 요청 (비동기)
  async generateReport(prompt: string, options: ReportOptions = {}): Promise<ApiResponse<{ jobId: string }>> {
    return apiClient.post('/report', {
      prompt,
      ...options
    });
  }

  // 보고서 작업 상태 확인
  async checkReportStatus(jobId: string): Promise<ApiResponse<ReportJobStatus>> {
    return apiClient.get(`/report/status/${jobId}`);
  }

  // 보고서 템플릿 목록 조회
  async getReportTemplates(): Promise<ApiResponse<ReportTemplate[]>> {
    return apiClient.get('/report/templates');
  }

  // 보고서 템플릿 상세 조회
  async getReportTemplateDetails(templateName: string): Promise<ApiResponse<ReportTemplate>> {
    return apiClient.get(`/report/templates/${templateName}`);
  }

  // 보고서 템플릿 추가
  async addReportTemplate(templateName: string, templateData: any): Promise<ApiResponse<any>> {
    return apiClient.post(`/report/templates/${templateName}`, templateData);
  }

  // 보고서 템플릿 삭제
  async deleteReportTemplate(templateName: string): Promise<ApiResponse<any>> {
    return apiClient.delete(`/template/${templateName}`);
  }

  // 파일 업로드를 통한 템플릿 추가 (FormData 사용)
  async uploadTemplateFile(file: File, templateName: string): Promise<ApiResponse<any>> {
    const formData = new FormData();
    formData.append('file', file);
    formData.append('templateName', templateName);

    return apiClient.request('/template/upload', {
      method: 'POST',
      headers: {
        // Content-Type은 자동으로 설정되므로 여기서 지정하지 않음
      },
      body: formData,
    });
  }
}

// 문서 서비스 인스턴스 생성
const documentService = new DocumentService();

export default documentService; 