'use client';

import { useState, useEffect, useRef } from 'react';
import { useRouter } from 'next/navigation';
import externalApiClient, { ApiClient, internalApiClient } from '@/lib/apiClient';
import { CodeViewer } from '@/components/CodeViewer';

interface AgentSession {
  id: string;
  title: string;
  provider: string;
  model: string;
  parameters: any;
  history: any[];
  updatedAt: string;
  userId: number;
  type: 'agent' | 'scheduled_task';
}

interface ScheduledTaskHistory {
  id: string;
  taskId: string;
  scheduleId: string;
  taskName: string;
  scheduleName: string;
  prompt: string;
  status: string;
  startedAt: string;
  completedAt: string;
  executionTimeMs: number;
  pythonCode: string;
  stdout: string;
  stderr: string;
  exitCode: number;
  errorMessage: string;
  metadata: any;
  type: 'scheduled_task';
}

type HistoryItem = AgentSession | ScheduledTaskHistory;

// 타입 가드 함수들
const isAgentSession = (item: HistoryItem): item is AgentSession => {
  return item.type === 'agent' || (item.type === 'scheduled_task' && !('taskId' in item));
};

const isScheduledTaskHistory = (item: HistoryItem): item is ScheduledTaskHistory => {
  return item.type === 'scheduled_task' && 'taskId' in item;
};

interface AgentJob {
  id: string;
  state: 'pending' | 'waiting' | 'active' | 'completed' | 'failed';
  jobId?: string;
  timestamp?: string;
  processedOn?: string;
  createdAt?: string;
  queue?: string;
  name?: string;
  progress?: any;
  currentProgress?: any;
  result?: any;
  error?: string;
  data?: {
    execution_result?: {
      code: number;
      stdout: string;
      stderr: string;
      python_code: string;
    };
    python_code?: string;
    prompt?: string;
    result?: any;
    error?: string;
    type?: string;
  };
}

interface AgentRequest {
  prompt: string;
  options: {
    model?: string;
    provider?: string;
    temperature?: number;
    rag?: boolean;
    web?: boolean;
    ragSearchScope?: 'personal' | 'shared' | 'all';
  };
}

interface ProviderData {
  name: string;
  requiresApiKey: boolean;
  apiKeyConfigured: boolean;
  isDynamic: boolean;
  isAvailable: boolean;
  models: Model[];
}

interface Model {
  id: string;
  name: string;
  size?: number;
}

interface AvailableProvider {
  key: string;
  name: string;
  requiresApiKey: boolean;
  apiKeyConfigured: boolean;
  isDynamic: boolean;
  models?: Model[];
}

interface SavedTask {
  id: string;
  name: string;
  prompt: string;
  code?: string;
  createdAt: string;
  lastUsed: string;
  executionCount: number;
  metadata?: {
    lastExecutionTime: string;
    successRate: string;
  };
}

interface ScheduledTask {
  id: string;
  name: string;
  taskId: string;
  taskName: string;
  scheduleType: 'once' | 'recurring';
  scheduledAt: string;
  recurringPattern?: {
    type: 'daily' | 'weekly' | 'monthly' | 'cron';
    interval: number;
    daysOfWeek?: number[];
    dayOfMonth?: number;
    cronExpression?: string;
  };
  status: 'active' | 'inactive' | 'completed';
  nextExecution?: string;
  lastExecution?: string;
  executionCount: number;
  createdAt: string;
  updatedAt: string;
}

interface ScheduleFormData {
  taskId: string;
  name: string;
  scheduleType: 'once' | 'recurring';
  scheduledAt: string;
  recurringPattern?: {
    type: 'daily' | 'weekly' | 'monthly' | 'cron';
    interval: number;
    daysOfWeek?: number[];
    dayOfMonth?: number;
    cronExpression?: string;
  };
}

interface TaskExecutionResult {
  success: boolean;
  code?: string;
  output?: string;
  error?: string;
}

export default function AgentPage() {
  const router = useRouter();
  const [isAuthenticated, setIsAuthenticated] = useState(false);
  const [authLoading, setAuthLoading] = useState(true);
  const [sessions, setSessions] = useState<AgentSession[]>([]);
  const [scheduledTaskHistory, setScheduledTaskHistory] = useState<ScheduledTaskHistory[]>([]);
  const [unifiedHistory, setUnifiedHistory] = useState<HistoryItem[]>([]);
  const [activeJobs, setActiveJobs] = useState<AgentJob[]>([]);
  const [savedTasks, setSavedTasks] = useState<SavedTask[]>([]);
  const [currentSessionJobIds, setCurrentSessionJobIds] = useState<string[]>([]); // 현재 세션에서 실행된 작업 ID들
  const currentSessionJobIdsRef = useRef<string[]>([]); // 최신 값을 참조하기 위한 ref
  const [isLoading, setIsLoading] = useState(true);
  const [isExecuting, setIsExecuting] = useState<string | null>(null);
  const [isExecutingTask, setIsExecutingTask] = useState<string | null>(null);
  const [error, setError] = useState<string | null>(null);
  const [prompt, setPrompt] = useState('');
  const [executionProgress, setExecutionProgress] = useState<{
    stage: string;
    message: string;
    progress: number;
    startTime: number;
  } | null>(null);
  const [agentOptions, setAgentOptions] = useState({
    model: '',  // 빈 문자열로 시작하여 프로바이더 로드 후 자동 설정
    provider: '',  // 빈 문자열로 시작하여 프로바이더 로드 후 자동 설정
    temperature: 0.7,
    rag: false,
    web: false,
    ragSearchScope: 'personal' as 'personal' | 'shared' | 'all'
  });
  const [showSaveTaskModal, setShowSaveTaskModal] = useState(false);
  const [saveTaskName, setSaveTaskName] = useState('');

// 예약 작업 관련 타입들
interface SavedTask {
  id: string;
  name: string;
  prompt: string;
  code?: string;
  createdAt: string;
  lastUsed: string;
  executionCount: number;
  metadata?: {
    lastExecutionTime: string;
    successRate: string;
  };
}

interface ScheduledTask {
  id: string;
  name: string;
  taskId: string;
  taskName: string;
  scheduleType: 'once' | 'recurring';
  scheduledAt: string;
  recurringPattern?: {
    type: 'daily' | 'weekly' | 'monthly' | 'cron';
    interval: number;
    daysOfWeek?: number[];
    dayOfMonth?: number;
    cronExpression?: string;
  };
  status: 'active' | 'inactive' | 'completed';
  nextExecution?: string;
  lastExecution?: string;
  executionCount: number;
  createdAt: string;
  updatedAt: string;
}

interface ScheduleFormData {
  taskId: string;
  name: string;
  scheduleType: 'once' | 'recurring';
  scheduledAt: string;
  recurringPattern?: {
    type: 'daily' | 'weekly' | 'monthly' | 'cron';
    interval: number;
    daysOfWeek?: number[];
    dayOfMonth?: number;
    cronExpression?: string;
  };
}
  const [lastAgentResult, setLastAgentResult] = useState<any>(null);
  const [selectedSession, setSelectedSession] = useState<AgentSession | null>(null);
  const [selectedHistoryItem, setSelectedHistoryItem] = useState<HistoryItem | null>(null);
  
  // 프로바이더 관련 상태
  const [providers, setProviders] = useState<Record<string, ProviderData>>({});
  const [availableProviders, setAvailableProviders] = useState<AvailableProvider[]>([]);
  const [isLoadingProviders, setIsLoadingProviders] = useState(true);

  // 예약 작업 관련 상태
  const [scheduledTasks, setScheduledTasks] = useState<ScheduledTask[]>([]);
  const [showScheduleModal, setShowScheduleModal] = useState(false);
  const [scheduleForm, setScheduleForm] = useState<ScheduleFormData>({
    taskId: '',
    name: '',
    scheduleType: 'once',
    scheduledAt: '',
    recurringPattern: undefined
  });
  const [isLoadingSchedules, setIsLoadingSchedules] = useState(true);
  const [isCreatingSchedule, setIsCreatingSchedule] = useState(false);
  
  // 현재 세션에서 실행한 작업 결과 관리
  const [currentSessionTaskResults, setCurrentSessionTaskResults] = useState<{[taskId: string]: {output: string, status: 'success' | 'error', timestamp: string}}>({});

  // 폼 유효성 검사 함수
  const isFormValid = (): boolean => {
    if (!scheduleForm.taskId || !scheduleForm.name) return false;
    
    if (scheduleForm.scheduleType === 'once') {
      return !!scheduleForm.scheduledAt;
    }
    
    if (scheduleForm.scheduleType === 'recurring') {
      if (scheduleForm.recurringPattern?.type === 'cron') {
        // Cron 설정: 유효한 cron 표현식만 있으면 됨 (scheduledAt 선택사항)
        return !!(scheduleForm.recurringPattern?.cronExpression && 
                  validateCronExpression(scheduleForm.recurringPattern.cronExpression));
      } else {
        // 일반 반복: scheduledAt 필수
        return !!scheduleForm.scheduledAt;
      }
    }
    
    return false;
  };

  // Cron 표현식 유효성 검사 함수 (브라우저 호환)
  const validateCronExpression = (expression: string): boolean => {
    if (!expression || typeof expression !== 'string') return false;
    
    const parts = expression.trim().split(/\s+/);
    if (parts.length !== 5) return false;
    
    // 각 필드별 유효성 검사
    const validators = [
      (val: string) => validateCronField(val, 0, 59),   // 분 (0-59)
      (val: string) => validateCronField(val, 0, 23),   // 시 (0-23)
      (val: string) => validateCronField(val, 1, 31),   // 일 (1-31)
      (val: string) => validateCronField(val, 1, 12),   // 월 (1-12)
      (val: string) => validateCronField(val, 0, 6),    // 요일 (0-6)
    ];
    
    return parts.every((part, index) => validators[index](part));
  };

  // Cron 필드 유효성 검사 헬퍼 함수
  const validateCronField = (field: string, min: number, max: number): boolean => {
    if (field === '*') return true;
    
    // 콤마로 구분된 여러 값 (예: 1,3,5)
    if (field.includes(',')) {
      return field.split(',').every(part => validateCronField(part.trim(), min, max));
    }
    
    // 범위 (예: 1-5)
    if (field.includes('-')) {
      const [start, end] = field.split('-').map(s => parseInt(s.trim()));
      return !isNaN(start) && !isNaN(end) && start >= min && end <= max && start <= end;
    }
    
    // 스텝 값 (예: */5, 2-10/2)
    if (field.includes('/')) {
      const [range, step] = field.split('/');
      const stepNum = parseInt(step);
      if (isNaN(stepNum) || stepNum <= 0) return false;
      
      if (range === '*') return stepNum <= max;
      return validateCronField(range, min, max);
    }
    
    // 단일 숫자
    const num = parseInt(field);
    return !isNaN(num) && num >= min && num <= max;
  };

  // Cron 표현식 설명 생성 함수
  const getCronDescription = (expression: string): string => {
    const parts = expression.split(' ');
    if (parts.length !== 5) return '잘못된 형식';
    
    const [minute, hour, day, month, weekday] = parts;
    
    // 간단한 패턴 해석
    if (expression === '*/5 * * * *') return '5분마다';
    if (expression === '*/10 * * * *') return '10분마다';
    if (expression === '*/30 * * * *') return '30분마다';
    if (expression === '0 */1 * * *') return '1시간마다';
    if (expression === '0 */2 * * *') return '2시간마다';
    if (expression === '0 9 * * *') return '매일 오전 9시';
    if (expression === '0 9 * * 1-5') return '평일 오전 9시';
    if (expression === '0 0 * * *') return '매일 자정';
    if (expression === '0 0 1 * *') return '매월 1일 자정';
    if (expression === '0 0 * * 0') return '매주 일요일 자정';
    if (expression === '0 0 * * 1') return '매주 월요일 자정';
    
    // 일반적인 패턴 해석
    let desc = '';
    if (minute.startsWith('*/')) {
      desc = `${minute.substring(2)}분마다`;
    } else if (minute === '0') {
      if (hour.startsWith('*/')) {
        desc = `${hour.substring(2)}시간마다`;
      } else if (hour !== '*') {
        desc = `매일 ${hour}시`;
      }
    }
    
    return desc || '사용자 정의 패턴';
  };

  // 인증 상태 확인 (다른 페이지들과 동일한 방식)
  useEffect(() => {
    const checkAuth = () => {
      const token = localStorage.getItem('userToken');
      const username = localStorage.getItem('username');
      const userId = localStorage.getItem('userId');
      const apiKey = localStorage.getItem('apiKey');
      
      // anonymous 사용자 정보가 있으면 제거
      if (username === 'anonymous' || userId === 'anonymous') {
        localStorage.removeItem('userToken');
        localStorage.removeItem('username');
        localStorage.removeItem('userId');
        localStorage.removeItem('userEmail');
        localStorage.removeItem('userRole');
        localStorage.removeItem('apiKey');
        console.log('Anonymous 사용자 정보 제거됨');
        // 로그인 페이지로 리디렉션
        router.push('/login?redirect=' + encodeURIComponent('/agent'));
        return;
      }
      
      if (token && username && userId && apiKey && username !== 'anonymous') {
        // console.log('에이전트 페이지 인증 완료:', username);
        setIsAuthenticated(true);
        setAuthLoading(false);
      } else {
        // 인증 정보가 없으면 로그인 페이지로 리디렉션
        // console.log('에이전트 페이지 인증 정보 없음, 로그인 페이지로 리디렉션');
        router.push('/login?redirect=' + encodeURIComponent('/agent'));
        return;
      }
    };

    checkAuth();
  }, []);

  // 인증된 사용자만 데이터 로드
  useEffect(() => {
    if (!isAuthenticated) return;
    
    // 페이지 로드 시 실행 상태 초기화
    setCurrentSessionJobIds([]);
    setExecutionProgress(null);
    setLastAgentResult(null);
    setIsExecuting(null);
    setIsExecutingTask(null);
    setError(null);
    setCurrentSessionTaskResults({}); // 현재 세션 작업 결과 초기화
    
    loadSessions();
    loadCurrentSessionJobs([]);  // 빈 배열로 명시적 호출
    loadSavedTasks();
    loadScheduledTasks();
    loadScheduledTaskHistory();
    loadProviders();
  }, [isAuthenticated]); // isAuthenticated가 true가 되면 데이터 로드
  
  // currentSessionJobIds 변경 시에만 작업 목록 업데이트 및 주기적 업데이트 설정
  useEffect(() => {
    // ref 업데이트
    currentSessionJobIdsRef.current = currentSessionJobIds;
    
    if (currentSessionJobIds.length > 0) {
      loadCurrentSessionJobs(currentSessionJobIds);
      
      // 작업이 있을 때만 주기적 업데이트 시작
      const interval = setInterval(() => {
        if (currentSessionJobIdsRef.current.length > 0) {
          loadCurrentSessionJobs(currentSessionJobIdsRef.current);
        }
      }, 5000);

      return () => clearInterval(interval);
    }
  }, [currentSessionJobIds]);

  // 실행 진행률 실시간 업데이트
  useEffect(() => {
    if (!executionProgress) return;

    const interval = setInterval(() => {
      setExecutionProgress(prev => prev ? { ...prev } : null);
    }, 1000);

    return () => clearInterval(interval);
  }, [executionProgress]);

  // 세션과 예약 작업 히스토리를 통합하여 정렬
  useEffect(() => {
    const combinedHistory: HistoryItem[] = [
      ...sessions,
      ...scheduledTaskHistory
    ];

    // 최신 순으로 정렬 (updatedAt 또는 completedAt 기준)
    combinedHistory.sort((a, b) => {
      const dateA = isAgentSession(a) 
        ? new Date(a.updatedAt) 
        : new Date(a.completedAt || a.startedAt);
      const dateB = isAgentSession(b) 
        ? new Date(b.updatedAt) 
        : new Date(b.completedAt || b.startedAt);
      return dateB.getTime() - dateA.getTime();
    });

    setUnifiedHistory(combinedHistory);
  }, [sessions, scheduledTaskHistory]);

  const loadSessions = async () => {
    try {
      // API 서버의 DB에서 agent 타입 세션 조회
      const response = await externalApiClient.getSessions('agent');
      
      if (response.success && response.data) {
        // 세션 데이터가 객체 형태이므로 배열로 변환
        let sessionsArray = [];
        if (Array.isArray(response.data)) {
          sessionsArray = response.data;
        } else if (typeof response.data === 'object') {
          // 객체를 배열로 변환
          sessionsArray = Object.values(response.data);
        }
        
        const agentSessions = sessionsArray.map((session: any): AgentSession => {
          // 안전하게 content를 문자열로 변환하는 함수
          const safeStringify = (content: any): string => {
            if (typeof content === 'string') return content;
            if (content === null || content === undefined) return '';
            try {
              return typeof content === 'object' ? JSON.stringify(content) : String(content);
            } catch {
              return String(content);
            }
          };

          // 사용자 메시지와 어시스턴트 메시지 필터링
          const userMessages = (session.history ?? []).filter((msg: any) => {
            return msg && msg.role === 'user' && msg.content;
          });
          
          const assistantMessages = (session.history ?? []).filter((msg: any) => {
            return msg && msg.role === 'assistant' && msg.content;
          });
          
          // 제목은 세션 타입에 따라 다르게 생성
          const title = session.id.startsWith('scheduled-') 
            ? (() => {
                // 예약 작업 세션인 경우
                const scheduleName = session.parameters?.scheduleName || '예약 작업';
                const timestamp = session.updated_at || session.created_at;
                if (timestamp) {
                  const date = new Date(timestamp);
                  return `📅 ${scheduleName} (${date.toLocaleString('ko-KR').slice(0, 16)})`;
                }
                return `📅 ${scheduleName}`;
              })()
            : userMessages.length > 0 
              ? (() => {
                  const content = safeStringify(userMessages[0].content);
                  // 시스템 프롬프트나 긴 내용은 간단히 처리
                  if (content.length > 1000 || content.includes('다음 문서들의 내용을 바탕으로') || content.includes('검색 결과:')) {
                    return '🤖 에이전트 작업'; // 시스템 프롬프트인 경우 기본 제목
                  }
                  return `🤖 ${content.slice(0, 30) + (content.length > 30 ? '...' : '')}`;
                })()
              : assistantMessages.length > 0 
                ? (() => {
                    const content = safeStringify(assistantMessages[0].content);
                    return `🤖 ${content.slice(0, 30) + (content.length > 30 ? '...' : '')}`;
                  })()
                : (() => {
                    // 히스토리가 없는 경우 세션 ID에서 시간 정보 추출하여 표시
                    const sessionId = session.id;
                    const timestamp = session.updated_at || session.created_at;
                    if (timestamp) {
                      const date = new Date(timestamp);
                      return `🤖 에이전트 작업 (${date.toLocaleString('ko-KR').slice(0, 16)})`;
                    }
                    return `🤖 에이전트 작업 (${sessionId.slice(-8)})`;
                  })();

          return {
            id: session.id,
            title: title,
            provider: session.provider || 'ollama',
            model: session.model || 'hamonize:latest',
            parameters: session.parameters || {},
            history: session.history ?? [],
            updatedAt: session.updated_at || new Date().toISOString(),
            userId: session.userId || 1,
            type: session.id.startsWith('scheduled-') ? 'scheduled_task' : 'agent'
          };
        });
        
        setSessions(agentSessions);
      } else {
        console.error('세션 로드 실패:', response.error);
      }
    } catch (error) {
      console.error('세션 로드 중 오류:', error);
    }
  };

  const loadCurrentSessionJobs = async (jobIds: string[] = currentSessionJobIds) => {
    try {
      // console.log('loadCurrentSessionJobs 호출됨, jobIds:', jobIds);
      
      // 현재 세션에서 실행된 작업이 없으면 빈 배열 반환
      if (!jobIds || jobIds.length === 0) {
        // console.log('작업 ID 없음, 빈 배열로 설정');
        setActiveJobs([]);
        setIsLoading(false);
        return;
      }

      // console.log('현재 세션 작업 ID들:', jobIds);
      
      const queueApiClient = new ApiClient('/api');
      
      // 각 작업 ID에 대해 개별 상태 조회
      const enhancedJobs = await Promise.all(
        jobIds.map(async (jobId) => {
          try {
            // console.log(`작업 ${jobId} 상세 정보 조회 시도...`);
            const jobDetailResponse = await queueApiClient.get(`/agent/status/${jobId}`);
            // console.log(`작업 ${jobId} 상세 조회 응답:`, jobDetailResponse);
            
            if (jobDetailResponse.success && jobDetailResponse.data) {
              const jobData = jobDetailResponse.data;
              
              // 다양한 결과 구조 처리
              let finalExecutionResult = null;
              
              // 1. execution_result 객체가 있는 경우
              if (jobData.execution_result) {
                finalExecutionResult = jobData.execution_result;
              }
              // 2. result 안에 execution_result가 있는 경우
              else if (jobData.result?.execution_result) {
                finalExecutionResult = jobData.result.execution_result;
              }
              // 3. 직접 code, output 필드가 있는 경우 (현재 상황)
              else if (jobData.code || jobData.output) {
                finalExecutionResult = {
                  python_code: jobData.code || jobData.python_code,
                  stdout: jobData.output || jobData.stdout || '',
                  stderr: jobData.error || jobData.stderr || '',
                  code: jobData.success ? 0 : (jobData.error ? 1 : 0)
                };
              }
              // 4. result 안에 code, output 필드가 있는 경우
              else if (jobData.result?.code || jobData.result?.output) {
                finalExecutionResult = {
                  python_code: jobData.result.code || jobData.result.python_code,
                  stdout: jobData.result.output || jobData.result.stdout || '',
                  stderr: jobData.result.error || jobData.result.stderr || '',
                  code: jobData.result.success ? 0 : (jobData.result.error ? 1 : 0)
                };
              }
              
              const enhancedJob: AgentJob = {
                id: jobId,
                jobId: jobId,
                state: jobData.state || 'pending',
                timestamp: jobData.timestamp || new Date().toISOString(),
                data: {
                  execution_result: finalExecutionResult,
                  result: jobData.result,
                  prompt: jobData.prompt || jobData.data?.prompt,
                  error: jobData.error
                }
              };
              
              // console.log(`향상된 작업 ${jobId}:`, enhancedJob);
              return enhancedJob;
            }
          } catch (error) {
            console.warn(`작업 ${jobId} 상세 정보 로드 실패:`, error);
            // 오류 발생 시에도 기본 정보는 반환
            return {
              id: jobId,
              jobId: jobId,
              state: 'failed' as const,
              timestamp: new Date().toISOString(),
              data: {
                error: '작업 상태 조회 실패'
              }
            } as AgentJob;
          }
        })
      );
      
      // null 값과 undefined 값 제거
      const validJobs = enhancedJobs.filter((job): job is AgentJob => job !== null && job !== undefined);
      
      console.log('최종 현재 세션 작업 목록:', validJobs);
      
      // 상태별로 정렬: 실행 중인 작업 먼저, 그 다음 완료된 작업
      const sortedJobs = validJobs.sort((a, b) => {
        const stateOrder: Record<string, number> = { 'active': 0, 'pending': 1, 'waiting': 2, 'completed': 3, 'failed': 4 };
        return (stateOrder[a.state as string] || 5) - (stateOrder[b.state as string] || 5);
      });
      
      setActiveJobs(sortedJobs);
    } catch (error) {
      console.error('현재 세션 작업 로드 중 오류:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const loadProviders = async () => {
    try {
      setIsLoadingProviders(true);
      // console.log('[AGENT] 프로바이더 로드 시작...');
      // console.log('[AGENT] 현재 시간:', new Date().toISOString());
      
      // serverConfig에서 API URL 가져오기
      const { getApiServerUrl } = await import('@/config/serverConfig');
      const apiUrl = getApiServerUrl();
      // console.log('[AGENT] 환경변수에서 가져온 API URL:', apiUrl);
      
      // 새로운 ApiClient 인스턴스를 동적으로 생성
      const dynamicApiClient = new (externalApiClient.constructor as any)(apiUrl);
      // console.log('[AGENT] 동적 생성된 ApiClient 기본 URL:', (dynamicApiClient as any).baseUrl);
      // console.log('[AGENT] 동적 ApiClient로 호출할 URL:', (dynamicApiClient as any).baseUrl + '/providers/available');
      
      const availableResponse = await dynamicApiClient.getAvailableProviders();
      
      // console.log('[AGENT] 사용 가능한 프로바이더 응답:', availableResponse);
      
      if (availableResponse.success && availableResponse.data) {
        // console.log('[AGENT] 사용 가능한 프로바이더 로드 성공, 모델 정보 로드 중...');
        setAvailableProviders(availableResponse.data);
        
        // 각 프로바이더의 모델 목록 로드
        const providersData: any = {};
        for (const provider of availableResponse.data) {
          try {
            // console.log(`[AGENT] ${provider.key} 프로바이더의 모델 목록 로드 중...`);
            const modelsResponse = await dynamicApiClient.getProviderModels(provider.key);
            if (modelsResponse.success && modelsResponse.data) {
              // console.log(`[AGENT] ${provider.key} 모델 로드 성공:`, modelsResponse.data.length, '개');
              providersData[provider.key] = {
                name: provider.name,
                requiresApiKey: provider.requiresApiKey,
                apiKeyConfigured: provider.apiKeyConfigured !== false,
                isDynamic: provider.isDynamic,
                isAvailable: true,
                models: modelsResponse.data
              };
            }
          } catch (modelError) {
            console.warn(`[AGENT] ${provider.key} 모델 목록 로드 실패:`, modelError);
            // 모델 목록 로드 실패해도 프로바이더는 표시
            providersData[provider.key] = {
              name: provider.name,
              requiresApiKey: provider.requiresApiKey,
              apiKeyConfigured: provider.apiKeyConfigured !== false,
              isDynamic: provider.isDynamic,
              isAvailable: true,
              models: []
            };
          }
        }
        
        // console.log('[AGENT] 포맷된 프로바이더:', providersData);
        setProviders(providersData);
        
        // 현재 선택된 프로바이더가 유효한지 확인하고, 필요시 기본값으로 설정
        if (availableResponse.data.length > 0) {
          const currentProvider = agentOptions.provider;
          const isCurrentProviderValid = availableResponse.data.some((p: any) => p.key === currentProvider);
          
          if (!isCurrentProviderValid) {
            // 첫 번째 사용 가능한 프로바이더로 설정
            const firstProvider = availableResponse.data[0];
            const firstModel = providersData[firstProvider.key]?.models?.length > 0 
              ? providersData[firstProvider.key].models[0].id 
              : 'hamonize:latest';
            
            // console.log('[AGENT] 유효하지 않은 프로바이더, 변경:', currentProvider, '->', firstProvider.key);
            setAgentOptions(prev => ({
              ...prev,
              provider: firstProvider.key,
              model: firstModel
            }));
          } else {
            // 현재 프로바이더의 모델이 유효한지 확인
            const currentProviderData = providersData[currentProvider];
            if (currentProviderData && currentProviderData.models && currentProviderData.models.length > 0) {
              const currentModel = agentOptions.model;
              const isCurrentModelValid = currentProviderData.models.some((m: any) => m.id === currentModel);
              
              if (!isCurrentModelValid) {
                // 첫 번째 사용 가능한 모델로 설정
                // console.log('[AGENT] 유효하지 않은 모델, 변경:', currentModel, '->', currentProviderData.models[0].id);
                setAgentOptions(prev => ({
                  ...prev,
                  model: currentProviderData.models[0].id
                }));
              }
            }
          }
        }
        // console.log('[AGENT] 프로바이더 로드 완료');
      } else {
        console.error('[AGENT] 사용 가능한 프로바이더 응답 실패:', availableResponse);
        throw new Error(`사용 가능한 프로바이더 API 응답 실패: ${availableResponse.error?.message || '알 수 없는 오류'}`);
      }
    } catch (error: any) {
      console.error('[AGENT] 프로바이더 정보 로드 실패:', error);
      console.error('[AGENT] 오류 상세:', {
        message: error.message,
        stack: error.stack,
        name: error.name
      });
      
      // 네트워크 오류인지 확인
      if (error.message?.includes('fetch') || error.message?.includes('NetworkError') || error.message?.includes('NETWORK_ERROR')) {
        console.error('[AGENT] 네트워크 연결 오류 - API 서버에 연결할 수 없습니다');
        setError('API 서버에 연결할 수 없습니다. 네트워크 연결을 확인해주세요.');
      } else if (error.message?.includes('401') || error.message?.includes('403')) {
        console.error('[AGENT] 인증 오류 - API 키 또는 토큰이 유효하지 않습니다');
        setError('인증에 실패했습니다. 로그인 정보를 확인해주세요.');
      } else {
        console.error('[AGENT] 일반 오류:', error.message);
        setError(`프로바이더 정보 로드 실패: ${error.message}`);
      }
      
      // 오류 발생 시 기본 프로바이더 정보 설정
      // console.log('[AGENT] 기본 프로바이더 설정 중...');
      const defaultModel = {
        id: 'hamonize:latest',
        name: 'hamonize:latest',
        size: 0
      };
      
      const defaultProviders = {
        ollama: {
          name: 'Ollama',
          isAvailable: true,
          requiresApiKey: false,
          apiKeyConfigured: true,
          isDynamic: true,
          models: [defaultModel]
        }
      };
      setProviders(defaultProviders);
      setAvailableProviders([
        { key: 'ollama', name: 'Ollama', requiresApiKey: false, apiKeyConfigured: true, isDynamic: true, models: [defaultModel] }
      ]);
      // console.log('[AGENT] 기본 프로바이더 설정 완료');
    } finally {
      setIsLoadingProviders(false);
    }
  };

  const handleProviderChange = async (newProvider: string) => {
    // console.log(`[AGENT] 프로바이더 변경: ${newProvider}`);
    setAgentOptions(prev => ({ ...prev, provider: newProvider }));
    
    // 해당 프로바이더의 모델 목록이 비어있거나 없으면 실시간으로 다시 로드
    const providerData = providers[newProvider];
    if (!providerData || !providerData.models || providerData.models.length === 0) {
      // console.log(`[AGENT] 프로바이더 ${newProvider}의 모델 목록이 없어서 실시간 로드를 시도합니다.`);
      
      try {
        // 동적으로 새로운 ApiClient 인스턴스 생성
        const { getApiServerUrl } = await import('@/config/serverConfig');
        const apiUrl = getApiServerUrl();
        const dynamicApiClient = new (externalApiClient.constructor as any)(apiUrl);
        const modelsResponse = await dynamicApiClient.getProviderModels(newProvider);
        if (modelsResponse.success && modelsResponse.data && modelsResponse.data.length > 0) {
          // console.log(`[AGENT] 프로바이더 ${newProvider} 모델 실시간 로드 성공:`, modelsResponse.data.length, '개');
          
          // 프로바이더 정보 업데이트
          setProviders((prev) => ({
            ...prev,
            [newProvider]: {
              ...prev[newProvider],
              models: modelsResponse.data
            }
          }));
          
          // 첫 번째 모델로 자동 설정
          const newModel = modelsResponse.data[0].id;
          setAgentOptions(prev => ({ 
            ...prev, 
            provider: newProvider,
            model: newModel
          }));
          
          // console.log(`[AGENT] 프로바이더 ${newProvider} 모델 ${newModel}로 설정 완료`);
          return;
        } else {
          console.warn(`[AGENT] 프로바이더 ${newProvider}의 모델 목록 실시간 로드 실패:`, modelsResponse);
          
          // Ollama인 경우 기본 모델로 fallback
          if (newProvider === 'ollama') {
            const defaultModel = {
              id: 'hamonize:latest',
              name: 'hamonize:latest',
              size: 0
            };
            
            setProviders((prev) => ({
              ...prev,
              [newProvider]: {
                ...prev[newProvider],
                models: [defaultModel]
              }
            }));
            
            setAgentOptions(prev => ({ 
              ...prev, 
              provider: newProvider,
              model: 'hamonize:latest'
            }));
            
            console.log(`[AGENT] 프로바이더 ${newProvider} 실시간 로드 실패, 기본 모델로 fallback 완료`);
            return;
          }
        }
      } catch (error) {
        console.error(`[AGENT] 프로바이더 ${newProvider}의 모델 목록 실시간 로드 중 오류:`, error);
        
        // Ollama인 경우 기본 모델로 fallback
        if (newProvider === 'ollama') {
          const defaultModel = {
            id: 'hamonize:latest',
            name: 'hamonize:latest',
            size: 0
          };
          
          setProviders((prev) => ({
            ...prev,
            [newProvider]: {
              ...prev[newProvider],
              models: [defaultModel]
            }
          }));
          
          setAgentOptions(prev => ({ 
            ...prev, 
            provider: newProvider,
            model: 'hamonize:latest'
          }));
          
          console.log(`[AGENT] 프로바이더 ${newProvider} 오류 발생, 기본 모델로 fallback 완료`);
          return;
        }
      }
    }
    
    // 기존 로직: 프로바이더에 모델이 있는 경우
    if (providerData && providerData.models && providerData.models.length > 0) {
      const newModel = providerData.models[0].id;
      setAgentOptions(prev => ({ 
        ...prev, 
        provider: newProvider,
        model: newModel
      }));
      
      console.log(`[AGENT] 프로바이더 변경: ${newProvider}, 모델: ${newModel}`);
    } else {
      console.warn(`[AGENT] 프로바이더 ${newProvider}의 모델 목록이 없습니다:`, providerData);
      
      // Ollama 프로바이더인 경우 기본 모델 설정
      if (newProvider === 'ollama') {
        const defaultModel = 'hamonize:latest';
        
        setAgentOptions(prev => ({ 
          ...prev, 
          provider: newProvider,
          model: defaultModel
        }));
        
        console.log(`[AGENT] 프로바이더 ${newProvider}에 기본 모델 ${defaultModel} 설정 완료`);
      } else {
        // 다른 프로바이더인 경우 빈 모델로 설정
        setAgentOptions(prev => ({ 
          ...prev, 
          provider: newProvider,
          model: '' // 모델을 빈 문자열로 설정
        }));
      }
    }
  };

  const executeAgent = async (agentRequest: AgentRequest) => {
    try {
      setIsExecuting('executing');
      setError(null);
      setExecutionProgress({
        stage: 'preparing',
        message: '에이전트 실행 준비 중...',
        progress: 10,
        startTime: Date.now()
      });

      // 로그인한 사용자 정보 가져오기
      const userId = localStorage.getItem('userId');
      const username = localStorage.getItem('username');

      setExecutionProgress({
        stage: 'sending',
        message: '에이전트 요청 전송 중...',
        progress: 20,
        startTime: Date.now()
      });

      // 동적으로 새로운 ApiClient 인스턴스 생성  
      const { getApiServerUrl } = await import('@/config/serverConfig');
      const apiUrl = getApiServerUrl();
      const dynamicApiClient = new (externalApiClient.constructor as any)(apiUrl);
      
      const response = await dynamicApiClient.post('/agent', {
        prompt: agentRequest.prompt,
        type: 'agent',  // API 문서에서 필수 필드로 요구
        userId: userId,
        username: username,
        options: agentRequest.options
      });

      // console.log('에이전트 실행 응답:', response);

      if (response.success && (response as any).jobId) {
        const jobId = (response as any).jobId;
        // console.log('생성된 작업 ID:', jobId);
        
        // 현재 세션 작업 ID 목록에 추가
        setCurrentSessionJobIds(prev => [...prev, jobId]);
        
        setExecutionProgress({
          stage: 'processing',
          message: `AI 에이전트 처리 중... (작업 ID: ${jobId})`,
          progress: 30,
          startTime: Date.now()
        });

        // 즉시 작업 목록 새로고침하여 새 작업 표시
        await loadCurrentSessionJobs([...currentSessionJobIds, jobId]);

        const result = await pollJobStatus(jobId);
        
        // 성공한 경우 작업 저장 옵션 제공
        if (result) {
          setExecutionProgress({
            stage: 'completed',
            message: '에이전트 실행 완료!',
            progress: 100,
            startTime: Date.now()
          });

          // 작업 목록 즉시 새로고침
          await loadCurrentSessionJobs();

          setTimeout(() => {
            setExecutionProgress(null);
          }, 5000);

          setLastAgentResult({
            prompt: agentRequest.prompt,
            code: result.code || result.execution_result?.python_code,
            result: result,
            execution_result: result.execution_result
          });
        }
      } else {
        throw new Error(response.error?.message || '에이전트 실행 요청 실패');
      }
    } catch (err) {
      setError(err instanceof Error ? err.message : '에이전트 실행 중 오류가 발생했습니다.');
      setExecutionProgress({
        stage: 'error',
        message: '에이전트 실행 실패',
        progress: 0,
        startTime: Date.now()
      });
      setTimeout(() => {
        setExecutionProgress(null);
      }, 3000);
    } finally {
      setIsExecuting(null);
    }
  };

  const pollJobStatus = async (jobId: string): Promise<any> => {
    const maxAttempts = 60;
    let attempts = 0;

    const checkStatus = async (): Promise<any> => {
      try {
        // 에이전트 상태는 프록시를 통해 접근
        const statusApiClient = new ApiClient('/api');
        // console.log(`작업 상태 확인 중... (시도 ${attempts + 1}/${maxAttempts}) 작업 ID: ${jobId}`);
        const response = await statusApiClient.get(`/agent/status/${jobId}`);
        
        // console.log(`작업 ${jobId} 상태 응답:`, response);
        
        if (response.success && response.data) {
          const { state, result, execution_result, code, output, error } = response.data;
          console.log(`작업 ${jobId} 상태: ${state}, 결과:`, result, '실행결과:', execution_result, '코드:', code, '출력:', output);

          // 진행률 업데이트
          if (state === 'active' || state === 'processing') {
            const progress = Math.min(40 + (attempts * 2), 90);
            setExecutionProgress({
              stage: 'processing',
              message: `AI 에이전트 처리 중... (${attempts + 1}/${maxAttempts})`,
              progress,
              startTime: Date.now()
            });
          }

          // 작업 목록 주기적으로 새로고침
          if (attempts % 3 === 0) {
            await loadCurrentSessionJobs();
          }

          if (state === 'completed') {
            // console.log('작업 완료됨, 결과:', result, '실행결과:', execution_result);
            await loadSessions();
            await loadCurrentSessionJobs();
            
            // 다양한 결과 구조 처리
            let finalExecutionResult = null;
            
            // 1. execution_result 객체가 있는 경우
            if (execution_result) {
              finalExecutionResult = execution_result;
            }
            // 2. result 안에 execution_result가 있는 경우
            else if (result?.execution_result) {
              finalExecutionResult = result.execution_result;
            }
            // 3. 직접 code, output 필드가 있는 경우 (현재 상황)
            else if (code || output) {
              finalExecutionResult = {
                python_code: code || result?.code || result?.python_code,
                stdout: output || result?.output || result?.stdout || '',
                stderr: error || result?.error || result?.stderr || '',
                code: response.data.success ? 0 : (error ? 1 : 0)
              };
            }
            // 4. result 안에 code, output 필드가 있는 경우
            else if (result?.code || result?.output) {
              finalExecutionResult = {
                python_code: result.code || result.python_code,
                stdout: result.output || result.stdout || '',
                stderr: result.error || result.stderr || '',
                code: result.success ? 0 : (result.error ? 1 : 0)
              };
            }
            
            const completeResult = {
              ...result,
              execution_result: finalExecutionResult,
              code: finalExecutionResult?.python_code || result?.code || result?.python_code
            };
            
            // console.log('최종 완료된 결과:', completeResult);
            return completeResult;
          } else if (state === 'failed') {
            console.error('작업 실패:', response.data.error);
            throw new Error(response.data.error || '에이전트 실행 실패');
          } else if (attempts < maxAttempts) {
            attempts++;
            return new Promise((resolve) => {
              setTimeout(async () => {
                resolve(await checkStatus());
              }, 2000);
            });
          } else {
            throw new Error('실행 시간 초과');
          }
        } else {
          console.warn(`작업 ${jobId} 상태 조회 실패:`, response.error);
        }
        return null;
      } catch (error) {
        console.error('상태 확인 실패:', error);
        setError('에이전트 실행 상태 확인 중 오류가 발생했습니다.');
        return null;
      }
    };

    return await checkStatus();
  };

  const deleteSession = async (sessionId: string) => {
    try {
      // API 서버의 DB에서 세션 삭제
      const response = await externalApiClient.deleteSession(sessionId);
      
      if (response.success) {
        // 선택된 세션이 삭제되면 선택 해제
        if (selectedSession?.id === sessionId) {
          setSelectedSession(null);
        }
        await loadSessions();
      } else {
        setError(response.error?.message || '세션 삭제 실패');
      }
    } catch (error) {
      console.error('세션 삭제 중 오류:', error);
      setError('세션 삭제 중 오류가 발생했습니다.');
    }
  };

  const deleteScheduledTaskHistory = async (historyId: string) => {
    try {
      const historyApiClient = new ApiClient('/api');
      const response = await historyApiClient.delete(`/schedules/history/${historyId}`);
      
      if (response.success) {
        // 선택된 히스토리가 삭제되면 선택 해제
        if (selectedHistoryItem?.id === historyId) {
          setSelectedHistoryItem(null);
        }
        await loadScheduledTaskHistory();
      } else {
        setError(response.error?.message || '예약 작업 히스토리 삭제 실패');
      }
    } catch (error) {
      console.error('예약 작업 히스토리 삭제 중 오류:', error);
      setError('예약 작업 히스토리 삭제 중 오류가 발생했습니다.');
    }
  };

  const handleSessionClick = (session: AgentSession) => {
    setSelectedSession(session);
    setSelectedHistoryItem(session);
  };

  const handleHistoryItemClick = (item: HistoryItem) => {
    setSelectedHistoryItem(item);
    if (item.type === 'agent') {
      setSelectedSession(item as AgentSession);
    } else {
      setSelectedSession(null);
    }
  };

  const loadSavedTasks = async () => {
    try {
      // console.log('저장된 작업 로드 시작...');
      const taskApiClient = new ApiClient('/api');
      const response = await taskApiClient.get('/tasks');
      
      // console.log('저장된 작업 로드 응답:', response);
      
      if (response.success && response.data) {
        const tasks = response.data.tasks || [];
        // console.log('로드된 작업 수:', tasks.length);
        setSavedTasks(tasks);
      } else {
        console.error('저장된 작업 로드 실패:', response.error);
      }
    } catch (error) {
      console.error('저장된 작업 로드 중 오류:', error);
    }
  };

  const loadScheduledTasks = async () => {
    try {
      // console.log('예약된 작업 로드 시작...');
      setIsLoadingSchedules(true);
      const scheduleApiClient = new ApiClient('/api');
      const response = await scheduleApiClient.get('/schedules');
      
      // console.log('예약된 작업 로드 응답:', response);
      
      if (response.success && response.data) {
        const schedules = response.data.schedules || [];
        // console.log('로드된 예약 작업 수:', schedules.length);
        setScheduledTasks(schedules);
        setError(null); // 성공 시 에러 상태 초기화
      } else {
        console.error('예약된 작업 로드 실패:', response.error);
        setError('예약된 작업을 불러올 수 없습니다: ' + (response.error?.message || '알 수 없는 오류'));
        setScheduledTasks([]); // 실패 시 빈 배열로 설정
      }
    } catch (error) {
      console.error('예약된 작업 로드 중 오류:', error);
      const errorMessage = error instanceof Error ? error.message : '알 수 없는 오류';
      setError('예약된 작업 로드 중 네트워크 오류가 발생했습니다: ' + errorMessage);
      setScheduledTasks([]); // 오류 시 빈 배열로 설정
    } finally {
      setIsLoadingSchedules(false);
    }
  };

  const loadScheduledTaskHistory = async () => {
    try {
      // console.log('예약 작업 히스토리 로드 시작...');
      const historyApiClient = new ApiClient('/api');
      const response = await historyApiClient.get('/schedules/history');
      
      // console.log('예약 작업 히스토리 로드 응답:', response);
      
      if (response.success && response.data) {
        const history = response.data.history || [];
        // console.log('로드된 예약 작업 히스토리 수:', history.length);
        setScheduledTaskHistory(history);
      } else {
        console.error('예약 작업 히스토리 로드 실패:', response.error);
        setScheduledTaskHistory([]); // 실패 시 빈 배열로 설정
      }
    } catch (error) {
      console.error('예약 작업 히스토리 로드 중 오류:', error);
      setScheduledTaskHistory([]); // 오류 시 빈 배열로 설정
    }
  };

  const createSchedule = async (scheduleData: ScheduleFormData) => {
    try {
      setIsCreatingSchedule(true);
      const scheduleApiClient = new ApiClient('/api');
      const response = await scheduleApiClient.post('/schedules', scheduleData);
      
      if (response.success) {
        await loadScheduledTasks();
        setShowScheduleModal(false);
        setScheduleForm({
          taskId: '',
          name: '',
          scheduleType: 'once',
          scheduledAt: '',
          recurringPattern: undefined
        });
        setError(null);
      } else {
        setError(response.error?.message || '예약 작업 생성 실패');
      }
    } catch (error) {
      console.error('예약 작업 생성 중 오류:', error);
      setError('예약 작업 생성 중 오류가 발생했습니다.');
    } finally {
      setIsCreatingSchedule(false);
    }
  };

  const deleteSchedule = async (scheduleId: string) => {
    try {
      const scheduleApiClient = new ApiClient('/api');
      const response = await scheduleApiClient.delete(`/schedules/${scheduleId}`);
      
      if (response.success) {
        await loadScheduledTasks();
      } else {
        setError(response.error?.message || '예약 작업 삭제 실패');
      }
    } catch (error) {
      console.error('예약 작업 삭제 중 오류:', error);
      setError('예약 작업 삭제 중 오류가 발생했습니다.');
    }
  };

  const toggleScheduleStatus = async (scheduleId: string, currentStatus: string) => {
    try {
      const newStatus = currentStatus === 'active' ? 'inactive' : 'active';
      const scheduleApiClient = new ApiClient('/api');
      const response = await scheduleApiClient.put(`/schedules/${scheduleId}`, {
        status: newStatus
      });
      
      if (response.success) {
        await loadScheduledTasks();
      } else {
        setError(response.error?.message || '예약 작업 상태 변경 실패');
      }
    } catch (error) {
      console.error('예약 작업 상태 변경 중 오류:', error);
      setError('예약 작업 상태 변경 중 오류가 발생했습니다.');
    }
  };

  const executeTask = async (taskId: string) => {
    try {
      setIsExecutingTask(taskId);
      setError(null);
      console.log(`작업 ${taskId} 실행 시작...`);

      const taskApiClient = new ApiClient('/api');
      const response = await taskApiClient.post(`/tasks/${taskId}/execute`);
      
      console.log(`작업 ${taskId} 실행 응답:`, response);

      if (response.success) {
        console.log(`작업 ${taskId} 실행 성공`);
        // console.log(`전체 응답 구조:`, JSON.stringify(response, null, 2));
        
        // 다양한 응답 구조 처리
        let executionResult = null;
        let outputText = '';
        let isSuccess = false;
        
        // 1. response.data.data.result가 있는 경우 (이중 감싸진 응답)
        if (response.data?.data?.result) {
          executionResult = response.data.data.result;
          outputText = executionResult.output || executionResult.stdout || '실행 완료';
          isSuccess = executionResult.success !== false && response.success;
          // console.log(`케이스 1: data.data.result에서 결과 추출:`, executionResult);
        }
        // 2. response.data.result가 있는 경우
        else if (response.data?.result) {
          executionResult = response.data.result;
          outputText = executionResult.output || executionResult.stdout || '실행 완료';
          isSuccess = executionResult.success !== false && response.success;
          // console.log(`케이스 2: data.result에서 결과 추출:`, executionResult);
        }
        // 3. response.data.task.metadata에 결과가 있는 경우
        else if (response.data?.task?.metadata?.lastOutput) {
          outputText = response.data.task.metadata.lastOutput;
          isSuccess = response.data.task.metadata.lastExecutionStatus === 'success';
          // console.log(`케이스 3: task.metadata에서 결과 추출:`, outputText);
        }
        // 4. response.result가 직접 있는 경우
        else if ((response as any).result) {
          executionResult = (response as any).result;
          outputText = executionResult.output || executionResult.stdout || '실행 완료';
          isSuccess = executionResult.success !== false && response.success;
          // console.log(`케이스 4: response.result에서 결과 추출:`, executionResult);
        }
        
        // 실행 결과를 현재 세션 결과에 저장
        if (outputText) {
          setCurrentSessionTaskResults(prev => ({
            ...prev,
            [taskId]: {
              output: outputText,
              status: isSuccess ? 'success' : 'error',
              timestamp: new Date().toISOString()
            }
          }));
          
          console.log(`작업 ${taskId} 실행 결과 저장됨:`, {
            output: outputText.substring(0, 100) + (outputText.length > 100 ? '...' : ''),
            status: isSuccess ? 'success' : 'error'
          });
        } else {
          console.log(`작업 ${taskId} 실행 결과가 응답에 없음`);
        }
        
        // 실행 성공 - 작업 목록 새로고침
        await loadSavedTasks();
        setError(null);
      } else {
        console.error(`작업 ${taskId} 실행 실패:`, response.error);
        
        // 실행 실패 결과도 저장
        setCurrentSessionTaskResults(prev => ({
          ...prev,
          [taskId]: {
            output: response.error?.message || '실행 실패',
            status: 'error',
            timestamp: new Date().toISOString()
          }
        }));
        
        setError(response.error?.message || '작업 실행 실패');
      }
    } catch (error) {
      console.error('작업 실행 중 오류:', error);
      
      // 오류 결과도 저장
      setCurrentSessionTaskResults(prev => ({
        ...prev,
        [taskId]: {
          output: error instanceof Error ? error.message : '작업 실행 중 오류가 발생했습니다.',
          status: 'error',
          timestamp: new Date().toISOString()
        }
      }));
      
      setError('작업 실행 중 오류가 발생했습니다.');
    } finally {
      setIsExecutingTask(null);
    }
  };

  const clearTaskOutput = async (taskId: string) => {
    try {
      // 현재 세션의 실행 결과만 지우기
      setCurrentSessionTaskResults(prev => {
        const newResults = { ...prev };
        delete newResults[taskId];
        return newResults;
      });
    } catch (error) {
      console.error('실행 결과 지우기 중 오류:', error);
      setError('실행 결과 지우기 중 오류가 발생했습니다.');
    }
  };

  const clearAllTaskOutputs = async () => {
    try {
      console.log('모든 작업 실행 결과 지우기...');
      // 현재 세션의 모든 실행 결과 지우기
      setCurrentSessionTaskResults({});
      console.log('모든 실행 결과가 지워졌습니다.');
    } catch (error) {
      console.error('모든 실행 결과 지우기 중 오류:', error);
      setError('모든 실행 결과 지우기 중 오류가 발생했습니다.');
    }
  };

  const deleteTask = async (taskId: string) => {
    try {
      const taskApiClient = new ApiClient('/api');
      const response = await taskApiClient.delete(`/tasks/${taskId}`);
      
      if (response.success) {
        await loadSavedTasks();
      } else {
        setError(response.error?.message || '작업 삭제 실패');
      }
    } catch (error) {
      console.error('작업 삭제 중 오류:', error);
      setError('작업 삭제 중 오류가 발생했습니다.');
    }
  };

  const saveTaskFromResult = async (taskName: string, prompt: string, code?: string) => {
    try {
      const taskApiClient = new ApiClient('/api');
      const response = await taskApiClient.post('/tasks', {
        name: taskName,
        prompt: prompt,
        code: code
      });

      if (response.success) {
        await loadSavedTasks();
        setShowSaveTaskModal(false);
        setSaveTaskName('');
        setLastAgentResult(null);
      } else {
        setError(response.error?.message || '작업 저장 실패');
      }
    } catch (error) {
      console.error('작업 저장 중 오류:', error);
      setError('작업 저장 중 오류가 발생했습니다.');
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!prompt.trim()) {
      setError('프롬프트를 입력해주세요.');
      return;
    }

    await executeAgent({
      prompt,
      options: {
        ...agentOptions
      }
    });
  };

  const getStatusColor = (state: string) => {
    switch (state) {
      case 'completed':
        return 'text-green-500';
      case 'active':
        return 'text-blue-500';
      case 'failed':
        return 'text-red-500';
      case 'waiting':
      case 'pending':
        return 'text-yellow-500';
      default:
        return 'text-gray-500';
    }
  };

  const getStatusBadge = (state: string) => {
    const baseClasses = 'px-2 py-1 rounded-full text-xs font-medium';
    switch (state) {
      case 'completed':
        return `${baseClasses} bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200`;
      case 'active':
        return `${baseClasses} bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200`;
      case 'failed':
        return `${baseClasses} bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200`;
      case 'waiting':
      case 'pending':
        return `${baseClasses} bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200`;
      default:
        return `${baseClasses} bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-200`;
    }
  };

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleString('ko-KR');
  };

  // 인증 중 로딩 표시
  if (authLoading) {
    return (
      <div className="container mx-auto px-4 py-8">
        <div className="flex justify-center items-center h-64">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500"></div>
          <span className="ml-4 text-lg">인증 확인 중...</span>
        </div>
      </div>
    );
  }

  // 인증되지 않은 사용자는 접근 불가
  if (!isAuthenticated) {
    return (
      <div className="container mx-auto px-4 py-8">
        <div className="flex flex-col items-center justify-center h-64">
          <div className="w-full max-w-md bg-white dark:bg-gray-800 rounded-lg shadow-lg p-6">
            <h1 className="text-2xl font-semibold mb-4 text-gray-900 dark:text-white">인증 필요</h1>
            <p className="mb-6 text-gray-600 dark:text-gray-300">
              AI 에이전트를 사용하려면 로그인하거나 API 키가 필요합니다.
            </p>
            <button
              onClick={() => router.push('/login')}
              className="w-full py-3 rounded-lg font-medium bg-blue-600 text-white hover:bg-blue-700"
            >
              로그인 페이지로 이동
            </button>
          </div>
        </div>
      </div>
    );
  }

  // 인증된 사용자의 데이터 로딩 중 표시
  if (isAuthenticated && isLoading) {
    return (
      <div className="container mx-auto px-4 py-8">
        <div className="flex justify-center items-center h-64">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500"></div>
          <span className="ml-4 text-lg">데이터 로딩 중...</span>
        </div>
      </div>
    );
  }

  return (
    <div className="flex min-h-screen" style={{ backgroundColor: 'var(--body-bg)' }}>
      {/* 왼쪽 사이드바 - 작업목록 & 세션 목록 */}
      <div className="w-80 border-r border-gray-200 dark:border-gray-700 overflow-y-auto" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
        {/* 작업목록 섹션 */}
        <div className="border-b border-gray-200 dark:border-gray-700">
          <div className="p-4 border-b border-gray-200 dark:border-gray-700">
            <div className="flex justify-between items-center">
              <h3 className="text-lg font-semibold">Tasks</h3>
              <div className="flex space-x-2">
                <button
                  onClick={() => {
                    loadSavedTasks();
                    clearAllTaskOutputs();
                  }}
                  className="px-2 py-1 bg-gray-500 text-white rounded hover:bg-gray-600 text-xs"
                >
                  새로고침
                </button>
              </div>
            </div>
          </div>

          <div className="p-4">
            {savedTasks.length === 0 ? (
              <div className="text-center py-8 text-gray-500 text-sm">
                저장된 작업이 없습니다.
              </div>
            ) : (
              <div className="space-y-3">
                {savedTasks.map((task) => (
                  <div
                    key={task.id}
                    className="bg-[var(--body-bg)] border border-gray-200 dark:border-gray-700 rounded-lg p-3 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                  >
                    <div className="flex justify-between items-start mb-2">
                      <p className="text-sm font-bold truncate pr-2">{task.name}</p>
                      <div className="flex space-x-1 flex-shrink-0">
                        <button
                          onClick={() => executeTask(task.id)}
                          disabled={isExecutingTask === task.id}
                          className="px-2 py-1 bg-blue-500 text-white rounded text-xs hover:bg-blue-600 disabled:opacity-50 disabled:cursor-not-allowed"
                        >
                          {isExecutingTask === task.id ? (
                            <div className="flex items-center space-x-1">
                              <div className="animate-spin rounded-full h-3 w-3 border-b-2 border-white"></div>
                              <span>실행중</span>
                            </div>
                          ) : '실행'}
                        </button>
                        <button
                          onClick={() => deleteTask(task.id)}
                          className="px-2 py-1 bg-red-500 text-white rounded text-xs hover:bg-red-600"
                        >
                          삭제
                        </button>
                      </div>
                    </div>
                    
                    <p className="text-xs text-gray-600 dark:text-gray-400 mb-2 line-clamp-2">
                      {task.prompt}
                    </p>
                    
                    {task.code && (
                      <div className="mb-2">
                        <span className="inline-block px-2 py-1 bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200 text-xs rounded">
                          💻 코드 포함 ({task.code.split('\n').length}줄)
                        </span>
                      </div>
                    )}



                    {/* 실행 결과 미리보기 - 현재 세션에서 실행한 결과만 표시 */}
                    {currentSessionTaskResults[task.id] && (
                      <div className="mt-2 p-2 bg-blue-50 dark:bg-blue-900 rounded border border-blue-200 dark:border-blue-700">
                        <div className="flex justify-between items-center mb-1">
                          <p className="text-xs font-medium text-blue-800 dark:text-blue-200">📄 최근 실행 결과:</p>
                          <button
                            onClick={(e) => {
                              e.stopPropagation();
                              clearTaskOutput(task.id);
                            }}
                            className="text-xs text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200"
                            title="실행 결과 지우기"
                          >
                            ✕
                          </button>
                        </div>
                        <pre className="text-xs text-blue-700 dark:text-blue-300 whitespace-pre-wrap max-h-16 overflow-y-auto">
                          {currentSessionTaskResults[task.id].output.length > 100 
                            ? currentSessionTaskResults[task.id].output.substring(0, 100) + '...' 
                            : currentSessionTaskResults[task.id].output}
                        </pre>
                        <div className="mt-1">
                          <span className={`inline-block px-2 py-1 text-xs rounded ${
                            currentSessionTaskResults[task.id].status === 'success' 
                              ? 'bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200'
                              : 'bg-red-100 dark:bg-red-900 text-red-800 dark:text-red-200'
                          }`}>
                            {currentSessionTaskResults[task.id].status === 'success' ? '✅ 성공' : '❌ 실패'}
                          </span>
                        </div>
                      </div>
                    )}
                  </div>
                ))}
              </div>
            )}
          </div>
        </div>

        {/* 예약 작업 섹션 */}
        <div className="border-b border-gray-200 dark:border-gray-700">
          <div className="p-4 border-b border-gray-200 dark:border-gray-700">
            <div className="flex justify-between items-center">
              <h3 className="text-lg font-semibold">예약 실행</h3>
              <div className="flex space-x-2">
                <button
                  onClick={() => setShowScheduleModal(true)}
                  className="px-2 py-1 bg-blue-500 text-white rounded hover:bg-blue-600 text-xs"
                  title="새 예약 작업"
                >
                  + 추가
                </button>
                <button
                  onClick={loadScheduledTasks}
                  className="px-2 py-1 bg-gray-500 text-white rounded hover:bg-gray-600 text-xs"
                >
                  새로고침
                </button>
              </div>
            </div>
          </div>

          <div className="p-4">
            {isLoadingSchedules ? (
              <div className="text-center py-4 text-gray-500 text-sm">
                <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-blue-500 mx-auto mb-2"></div>
                로딩 중...
              </div>
            ) : scheduledTasks.length === 0 ? (
              <div className="text-center py-8 text-gray-500 text-sm">
                예약된 작업이 없습니다.
              </div>
            ) : (
              <div className="space-y-3">
                {scheduledTasks.map((schedule) => (
                  <div
                    key={schedule.id}
                    className="bg-[var(--body-bg)] border border-gray-200 dark:border-gray-700 rounded-lg p-3 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                  >
                    <div className="flex justify-between items-start mb-2">
                      <div className="flex-1 min-w-0">
                        <p className="text-sm font-bold truncate">{schedule.name}</p>
                        <p className="text-xs text-gray-600 dark:text-gray-400">
                          작업: {schedule.taskName}
                        </p>
                        <p className="text-xs text-gray-500">
                          스케줄: {schedule.scheduleType === 'once' ? '한 번 실행' : '반복 실행'}
                        </p>
                        {schedule.nextExecution && (
                          <p className="text-xs text-blue-600 dark:text-blue-400">
                            다음 실행: {formatDate(schedule.nextExecution)}
                          </p>
                        )}
                      </div>
                      <div className="flex space-x-1 flex-shrink-0">
                        <button
                          onClick={() => toggleScheduleStatus(schedule.id, schedule.status)}
                          className={`px-2 py-1 rounded text-xs ${
                            schedule.status === 'active'
                              ? 'bg-green-500 text-white hover:bg-green-600'
                              : 'bg-gray-500 text-white hover:bg-gray-600'
                          }`}
                        >
                          {schedule.status === 'active' ? '활성' : '비활성'}
                        </button>
                        <button
                          onClick={() => deleteSchedule(schedule.id)}
                          className="px-2 py-1 bg-red-500 text-white rounded text-xs hover:bg-red-600"
                        >
                          삭제
                        </button>
                      </div>
                    </div>
                    


                    {/* 스케줄 상세 정보 */}
                    <div className="mt-2 p-2 bg-blue-50 dark:bg-blue-900 rounded border border-blue-200 dark:border-blue-700">
                      <p className="text-xs text-blue-800 dark:text-blue-200 mb-1">📅 스케줄 정보:</p>
                      <p className="text-xs text-blue-700 dark:text-blue-300">
                        예약 시간: {formatDate(schedule.scheduledAt)}
                      </p>
                      {schedule.recurringPattern && (
                        <p className="text-xs text-blue-700 dark:text-blue-300">
                          반복: {schedule.recurringPattern.type === 'daily' ? '매일' : 
                                schedule.recurringPattern.type === 'weekly' ? '매주' : '매월'} 
                          {schedule.recurringPattern.interval > 1 && ` (${schedule.recurringPattern.interval}간격)`}
                        </p>
                      )}
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>
        </div>

        {/* 통합 히스토리 섹션 */}
        <div>
          <div className="p-4 border-b border-gray-200 dark:border-gray-700">
            <div className="flex justify-between items-center">
              <h3 className="text-lg font-semibold">History</h3>
              <button
                onClick={() => {
                  loadSessions();
                  loadScheduledTaskHistory();
                }}
                className="px-2 py-1 bg-gray-500 text-white rounded hover:bg-gray-600 text-xs"
              >
                새로고침
              </button>
            </div>
          </div>

          <div className="p-4">
            {unifiedHistory.length === 0 ? (
              <div className="text-center py-8 text-gray-500 text-sm">
                히스토리가 없습니다.
              </div>
            ) : (
              <div className="space-y-3">
                {unifiedHistory.map((item) => (
                  <div
                    key={item.id}
                    className={`bg-[var(--body-bg)] border border-gray-200 dark:border-gray-700 rounded-lg p-3 transition-colors cursor-pointer ${
                      selectedHistoryItem?.id === item.id 
                        ? 'bg-blue-50 dark:bg-blue-900 border-blue-300 dark:border-blue-600' 
                        : 'hover:bg-gray-50 dark:hover:bg-gray-700'
                    }`}
                    onClick={() => handleHistoryItemClick(item)}
                  >
                    <div className="flex justify-between items-start">
                      <div className="flex-1 min-w-0">
                        {isAgentSession(item) ? (
                          // 에이전트 세션 표시 (2줄)
                          <>
                            <div className="flex items-center space-x-2">
                              <p className="text-sm font-bold truncate">{item.title}</p>
                            </div>
                            <p className="text-xs text-gray-500">
                              {item.provider}/{item.model} • {item.history.length}개 메시지 • {formatDate(item.updatedAt)}
                            </p>
                          </>
                        ) : (
                          // 예약 작업 히스토리 표시 (2줄)
                          <>
                            <div className="flex items-center space-x-2">
                              <span className="text-green-500 text-xs">📅</span>
                              <p className="text-sm font-bold truncate">{item.taskName}</p>
                              <span className={`px-2 py-1 rounded text-xs ${
                                item.status === 'completed' ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' :
                                item.status === 'failed' ? 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200' :
                                'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200'
                              }`}>
                                {item.status === 'completed' ? '완료' : 
                                 item.status === 'failed' ? '실패' : item.status}
                              </span>
                            </div>
                            <p className="text-xs text-gray-500">
                              {item.scheduleName} • {item.executionTimeMs ? `${Math.round(item.executionTimeMs / 1000)}초` : '미상'} • {formatDate(item.completedAt || item.startedAt)}
                            </p>
                          </>
                        )}
                      </div>
                      
                      <button
                        onClick={(e) => {
                          e.stopPropagation(); // 클릭 이벤트 방지
                          if (item.type === 'agent') {
                            deleteSession(item.id);
                          } else {
                            // 스케줄 작업 히스토리 삭제
                            deleteScheduledTaskHistory(item.id);
                          }
                        }}
                        className="px-2 py-1 bg-red-500 text-white rounded text-xs hover:bg-red-600 flex-shrink-0"
                      >
                        삭제
                      </button>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>
        </div>
      </div>

      {/* 오른쪽 메인 콘텐츠 */}
      <div className="flex-1 p-6 overflow-y-auto">

        {/* 에러 메시지 */}
        {error && (
          <div className="mb-6 p-4 bg-red-100 dark:bg-red-900 text-red-700 dark:text-red-200 rounded-lg">
            {error}
            <button
              onClick={() => setError(null)}
              className="ml-4 text-sm underline"
            >
              닫기
            </button>
          </div>
        )}

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* 에이전트 실행 폼 */}
        <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
          <h3 className="text-xl font-semibold mb-4">자동화 작업 생성</h3>
          
          {/* 기능 안내 카드들 - 2x1 그리드 */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
            <button
              onClick={() => setPrompt("디스크 사용량을 출력해서 파일로 저장해주세요")}
              className="p-4 bg-gradient-to-br from-blue-50 to-blue-100 dark:from-blue-900/20 dark:to-blue-800/20 border border-blue-200 dark:border-blue-700 rounded-xl hover:shadow-lg transition-all duration-200 hover:scale-105 text-left group"
            >
              <div className="text-2xl mb-2 group-hover:scale-110 transition-transform">💾</div>
              <h4 className="font-semibold text-gray-900 dark:text-white mb-1">시스템 정보 수집</h4>
              <p className="text-sm text-gray-600 dark:text-gray-300 leading-relaxed">
                디스크 사용량, 메모리 상태 등 시스템 정보를 수집하고 파일로 저장
              </p>
            </button>
            
            <button
              onClick={() => setPrompt("현재 시간과 날씨 정보를 웹에서 검색해서 일일 보고서를 만들어주세요")}
              className="p-4 bg-gradient-to-br from-green-50 to-green-100 dark:from-green-900/20 dark:to-green-800/20 border border-green-200 dark:border-green-700 rounded-xl hover:shadow-lg transition-all duration-200 hover:scale-105 text-left group"
            >
              <div className="text-2xl mb-2 group-hover:scale-110 transition-transform">📊</div>
              <h4 className="font-semibold text-gray-900 dark:text-white mb-1">자동 보고서 생성</h4>
              <p className="text-sm text-gray-600 dark:text-gray-300 leading-relaxed">
                웹 검색으로 최신 정보를 수집하고 구조화된 보고서 자동 생성
              </p>
            </button>
          </div>
          
          <form onSubmit={handleSubmit} className="space-y-6">
            <div>
              <label className="block text-sm font-medium mb-2">프롬프트</label>
              <textarea
                value={prompt}
                onChange={(e) => setPrompt(e.target.value)}
                className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 h-32 resize-none"
                placeholder="실행할 작업을 설명해주세요..."
                required
              />
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium mb-2">프로바이더</label>
                <select
                  value={agentOptions.provider}
                  onChange={(e) => handleProviderChange(e.target.value)}
                  className="text-sm w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 chat-sidebar-list-text"
                  disabled={isLoadingProviders}
                >
                  {isLoadingProviders ? (
                    <option value="">프로바이더 로딩 중...</option>
                  ) : (
                    <>
                      <option value="">프로바이더를 선택하세요</option>
                      {availableProviders.map((provider) => (
                        <option key={provider.key} value={provider.key}>
                          {provider.name}
                        </option>
                      ))}
                    </>
                  )}
                </select>
              </div>

              <div>
                <label className="block text-sm font-medium mb-2">모델</label>
                <select
                  value={agentOptions.model}
                  onChange={(e) => setAgentOptions({...agentOptions, model: e.target.value})}
                  className="text-sm w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700"
                  disabled={isLoadingProviders || !providers[agentOptions.provider]?.models?.length}
                >
                  {!agentOptions.provider ? (
                    <option value="">먼저 프로바이더를 선택하세요</option>
                  ) : !providers[agentOptions.provider]?.models?.length ? (
                    <option value="">모델 로딩 중...</option>
                  ) : (
                    <>
                      <option value="">모델을 선택하세요</option>
                      {providers[agentOptions.provider]?.models?.map((model) => (
                        <option key={model.id} value={model.id}>
                          {model.name}
                        </option>
                      ))}
                    </>
                  )}
                </select>
              </div>
            </div>

            <div>
              <label className="block text-sm font-medium mb-2">
                Temperature : {agentOptions.temperature}
              </label>
              <div className="flex items-center space-x-4">
                <span className="text-sm text-gray-500">0</span>
                <input
                  type="range"
                  min="0"
                  max="2"
                  step="0.1"
                  value={agentOptions.temperature}
                  onChange={(e) => setAgentOptions({...agentOptions, temperature: parseFloat(e.target.value)})}
                  className="flex-1 h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer dark:bg-gray-700"
                />
                <span className="text-sm text-gray-500">2</span>
              </div>
              <div className="flex justify-between text-xs text-gray-400 mt-1">
                <span>정확한</span>
                <span>창의적</span>
              </div>
            </div>

            <div className="space-y-4">
              {/* RAG 사용 토글 */}
              <div className="flex items-center justify-between">
                <div>
                  <label className="text-sm font-medium">RAG 사용</label>
                  <p className="text-xs text-gray-500">문서 기반 검색 활성화</p>
                </div>
                <label className="relative inline-flex items-center cursor-pointer">
                  <input
                    type="checkbox"
                    checked={agentOptions.rag}
                    onChange={(e) => setAgentOptions({...agentOptions, rag: e.target.checked})}
                    className="sr-only peer"
                  />
                  <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 dark:peer-focus:ring-blue-800 rounded-full peer dark:bg-gray-700 peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all dark:border-gray-600 peer-checked:bg-blue-600"></div>
                </label>
              </div>

              {/* 웹 검색 사용 토글 */}
              <div className="flex items-center justify-between">
                <div>
                  <label className="text-sm font-medium">웹 검색 사용</label>
                  <p className="text-xs text-gray-500">실시간 웹 검색 활성화</p>
                </div>
                <label className="relative inline-flex items-center cursor-pointer">
                  <input
                    type="checkbox"
                    checked={agentOptions.web}
                    onChange={(e) => setAgentOptions({...agentOptions, web: e.target.checked})}
                    className="sr-only peer"
                  />
                  <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 dark:peer-focus:ring-blue-800 rounded-full peer dark:bg-gray-700 peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all dark:border-gray-600 peer-checked:bg-blue-600"></div>
                </label>
              </div>
            </div>



            <div className="flex space-x-2">
              <button
                type="submit"
                disabled={isExecuting === 'executing' || isLoadingProviders}
                className="flex-1 px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
              >
                {isExecuting === 'executing' ? '실행 중...' : '에이전트 실행'}
              </button>
              
              {/* 취소 버튼 */}
              {isExecuting === 'executing' && (
                <button
                  type="button"
                  onClick={() => {
                    setIsExecuting(null);
                    setExecutionProgress(null);
                    setError('사용자에 의해 취소되었습니다.');
                  }}
                  className="px-4 py-2 bg-red-500 text-white rounded-lg hover:bg-red-600 transition-colors"
                >
                  취소
                </button>
              )}
            </div>
          </form>

          {/* 마지막 실행 결과 저장 버튼 */}
          {lastAgentResult && (
            <div className="mt-4 p-4 bg-green-50 dark:bg-green-900 rounded-lg border border-green-200 dark:border-green-700">
              <p className="text-sm font-medium text-green-800 dark:text-green-200 mb-2">
                ✅ 에이전트 실행 완료
              </p>
              
              {/* 실행 결과 미리보기 */}
              {lastAgentResult.execution_result && (
                <div className="mb-3 p-3 bg-white dark:bg-gray-800 rounded border">
                  <div className="flex items-center justify-between mb-2">
                    <span className="text-xs font-medium text-gray-700 dark:text-gray-300">
                      실행 결과 (Exit Code: {lastAgentResult.execution_result.code})
                    </span>
                    {lastAgentResult.execution_result.code === 0 && (
                      <span className="text-xs text-green-600 dark:text-green-400">성공</span>
                    )}
                  </div>
                  
                  {lastAgentResult.execution_result.stdout && (
                    <div className="text-xs text-gray-600 dark:text-gray-400 mb-2">
                      <div className="font-medium mb-1">출력:</div>
                      <pre className="bg-gray-50 dark:bg-gray-900 p-2 rounded whitespace-pre-wrap max-h-32 overflow-y-auto">
                        {lastAgentResult.execution_result.stdout.length > 200 
                          ? lastAgentResult.execution_result.stdout.substring(0, 200) + '...' 
                          : lastAgentResult.execution_result.stdout}
                      </pre>
                    </div>
                  )}
                  
                  {/* 생성된 코드 미리보기 */}
                  {/* {(lastAgentResult.code || lastAgentResult.execution_result.python_code) && (
                    <div className="mb-2">
                      <CodeViewer
                        code={lastAgentResult.code || lastAgentResult.execution_result.python_code}
                        language="python"
                        title="💻 생성된 코드 미리보기"
                        className="max-h-32 overflow-y-auto"
                      />
                    </div>
                  )} */}
                </div>
              )}
              
              <p className="text-sm text-green-700 dark:text-green-300 mb-3">
                성공적으로 실행된 작업을 저장하시겠습니까?
              </p>
              <button
                onClick={() => setShowSaveTaskModal(true)}
                className="px-4 py-2 bg-green-500 text-white rounded hover:bg-green-600 text-sm"
              >
                작업 저장
              </button>
            </div>
          )}
        </div>

        {/* 작업 상태 모니터 */}
        <div className="rounded-lg shadow-md p-6" style={{ backgroundColor: 'var(--card-bg)' }}>
          <div className="flex justify-between items-center mb-4">
            <h3 className="text-xl font-semibold">작업 상태</h3>
            <button
              onClick={() => loadCurrentSessionJobs()}
              className="px-3 py-1 bg-gray-500 text-white rounded hover:bg-gray-600"
            >
              새로고침
            </button>
          </div>

          {/* 현재 실행 중인 작업의 진행률 표시 */}
          {executionProgress && (
            <div className="mb-4 p-4 bg-blue-50 dark:bg-blue-900 rounded-lg border border-blue-200 dark:border-blue-700">
              <div className="flex justify-between items-center mb-2">
                <span className="text-sm font-medium text-blue-800 dark:text-blue-200">
                  {executionProgress.message}
                </span>
                <span className="text-sm text-blue-600 dark:text-blue-400">
                                            {executionProgress.progress.toFixed(1)}%
                </span>
              </div>
              
              {/* 진행률 바 */}
              <div className="w-full bg-blue-200 dark:bg-blue-800 rounded-full h-2.5">
                <div 
                  className="bg-blue-600 dark:bg-blue-400 h-2.5 rounded-full transition-all duration-300 ease-out"
                  style={{ width: `${executionProgress.progress}%` }}
                ></div>
              </div>
              
              {/* 경과 시간 */}
              <div className="text-xs text-blue-600 dark:text-blue-400 mt-1">
                경과 시간: {Math.round((Date.now() - executionProgress.startTime) / 1000)}초
              </div>
              
              {/* 실행 단계 아이콘 */}
              <div className="flex items-center mt-2 text-xs text-blue-700 dark:text-blue-300">
                {executionProgress.stage === 'preparing' && (
                  <>
                    <div className="animate-spin rounded-full h-3 w-3 border-b-2 border-blue-600 mr-2"></div>
                    준비 중
                  </>
                )}
                {executionProgress.stage === 'sending' && (
                  <>
                    <div className="animate-pulse h-3 w-3 bg-blue-600 rounded-full mr-2"></div>
                    전송 중
                  </>
                )}
                {executionProgress.stage === 'processing' && (
                  <>
                    <div className="animate-spin rounded-full h-3 w-3 border-b-2 border-blue-600 mr-2"></div>
                    처리 중
                  </>
                )}
                {executionProgress.stage === 'completed' && (
                  <>
                    <div className="h-3 w-3 bg-green-600 rounded-full mr-2"></div>
                    완료
                  </>
                )}
                {executionProgress.stage === 'error' && (
                  <>
                    <div className="h-3 w-3 bg-red-600 rounded-full mr-2"></div>
                    오류 발생
                  </>
                )}
              </div>
            </div>
          )}

          {isLoading ? (
            <div className="text-center py-8 text-gray-500">
              <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500 mx-auto mb-2"></div>
              작업 목록을 로드하는 중...
            </div>
          ) : activeJobs.length === 0 && !executionProgress ? (
            <div className="text-center py-8 text-gray-500">
              표시할 작업이 없습니다.
            </div>
          ) : (
            <div className="space-y-4 max-h-400 overflow-y-auto">
              {/* 실행 중인 진행률이 표시되는 동안은 active, pending, waiting 상태의 작업은 숨김 */}
              {activeJobs
                .filter(job => {
                  // 실행 진행률이 표시되는 동안은 실행 중인 작업들을 숨김
                  if (executionProgress && (job.state === 'active' || job.state === 'pending' || job.state === 'waiting')) {
                    return false;
                  }
                  return true;
                })
                .map((job) => (
                <div
                  key={job.id}
                  className="border border-gray-200 dark:border-gray-700 rounded-lg p-4"
                >
                  <div className="flex justify-between items-start mb-2">
                    <div>
                      <h3 className="font-medium text-sm">작업 ID: {job.jobId || job.id}</h3>
                                              <p className="text-xs text-gray-600 dark:text-gray-400">
                          시작: {formatDate(job.timestamp || job.processedOn || job.createdAt || new Date().toISOString())}
                        </p>
                      {job.data?.prompt && (
                        <p className="text-xs text-gray-600 dark:text-gray-400 mt-1">
                          프롬프트: {job.data.prompt.length > 50 ? job.data.prompt.substring(0, 50) + '...' : job.data.prompt}
                        </p>
                      )}
                      <p className="text-xs text-blue-600 dark:text-blue-400 mt-1">
                        큐: {job.queue || job.name || 'agent'} | 작업명: {job.name || job.data?.type || 'agent'}
                      </p>
                    </div>
                    <span className={getStatusBadge(job.state)}>
                      {job.state}
                    </span>
                  </div>

                  {/* 완료된 작업의 실행 결과 표시 */}
                  {job.state === 'completed' && (
                    <div className="mt-3 space-y-2">
                      <h4 className="text-sm font-medium text-green-600 dark:text-green-400">✅ 실행 완료</h4>
                      
                      {/* 실행 결과가 있는 경우 */}
                      {job.data?.execution_result ? (
                        <div className="bg-green-50 dark:bg-green-900 rounded-lg p-3">
                          <div className="flex items-center justify-between mb-2">
                            <span className="text-xs font-medium text-green-800 dark:text-green-200">
                              실행 결과 (Exit Code: {job.data.execution_result.code})
                            </span>
                            {job.data.execution_result.code === 0 && (
                              <span className="text-xs text-green-600 dark:text-green-400">성공</span>
                            )}
                          </div>
                          
                          {/* 표준 출력 */}
                          {job.data.execution_result.stdout && (
                            <div className="mb-2">
                              <p className="text-xs font-medium text-green-800 dark:text-green-200 mb-1">📄 출력:</p>
                              <pre className="text-xs text-green-700 dark:text-green-300 bg-green-100 dark:bg-green-800 p-3 rounded whitespace-pre-wrap max-h-48 overflow-y-auto">
                                {job.data.execution_result.stdout}
                              </pre>
                            </div>
                          )}
                          
                          {/* 표준 에러 */}
                          {job.data.execution_result.stderr && (
                            <div className="mb-2">
                              <p className="text-xs font-medium text-red-800 dark:text-red-200 mb-1">❌ 에러:</p>
                              <pre className="text-xs text-red-700 dark:text-red-300 bg-red-100 dark:bg-red-800 p-3 rounded whitespace-pre-wrap max-h-48 overflow-y-auto">
                                {job.data.execution_result.stderr}
                              </pre>
                            </div>
                          )}
                          
                          {/* 생성된 코드 */}
                          {job.data.execution_result.python_code && (
                            <div>
                              <CodeViewer
                                code={job.data.execution_result.python_code}
                                language="python"
                                title="💻 생성된 코드"
                                className="max-h-300 overflow-y-auto"
                              />
                            </div>
                          )}
                        </div>
                      ) : (
                        /* 실행 결과가 없는 경우 일반 결과 표시 */
                        job.data?.result && (
                          <div className="bg-green-50 dark:bg-green-900 rounded-lg p-3">
                            <p className="text-xs font-medium text-green-800 dark:text-green-200 mb-1">결과:</p>
                            <pre className="text-xs text-green-700 dark:text-green-300 bg-green-100 dark:bg-green-800 p-3 rounded whitespace-pre-wrap max-h-48 overflow-y-auto">
                              {typeof job.data.result === 'string' ? job.data.result : JSON.stringify(job.data.result, null, 2)}
                            </pre>
                          </div>
                        )
                      )}
                    </div>
                  )}

                  {/* 실행 중인 작업의 진행 상태 표시 */}
                  {(job.state === 'active' || job.state === 'pending' || job.state === 'waiting') && (
                    <div className="mt-2 p-3 bg-blue-50 dark:bg-blue-900 rounded-lg border border-blue-200 dark:border-blue-700">
                      <div className="flex items-center justify-between mb-2">
                        <span className="text-sm font-medium text-blue-800 dark:text-blue-200">
                          {job.state === 'active' ? '실행 중' : job.state === 'pending' ? '대기 중' : '준비 중'}
                        </span>
                        <div className="flex items-center space-x-2">
                          <div className="animate-spin rounded-full h-3 w-3 border-b-2 border-blue-600"></div>
                          <span className="text-xs text-blue-600 dark:text-blue-400">
                            {job.state === 'active' ? '처리 중' : '대기 중'}
                          </span>
                        </div>
                      </div>
                      
                      {/* 진행률 바 */}
                      <div className="w-full bg-blue-200 dark:bg-blue-800 rounded-full h-2 mb-2">
                        <div 
                          className="bg-blue-600 dark:bg-blue-400 h-2 rounded-full transition-all duration-300 ease-out animate-pulse"
                          style={{ width: job.state === 'active' ? '60%' : '20%' }}
                        ></div>
                      </div>
                      
                      <div className="text-xs text-blue-600 dark:text-blue-400">
                        {job.state === 'active' ? 'AI 에이전트가 코드를 생성하고 실행하고 있습니다...' : 
                         job.state === 'pending' ? '작업 대기열에서 처리를 기다리고 있습니다...' : 
                         '작업 실행 준비 중입니다...'}
                      </div>
                    </div>
                  )}

                  {/* 기존 진행 상태 표시 (상세 정보) */}
                  {job.progress && (
                    <div className="mt-2 p-2 bg-gray-50 dark:bg-gray-900 rounded text-xs">
                      <p className="text-gray-800 dark:text-gray-200 font-medium mb-1">상세 정보:</p>
                      <pre className="whitespace-pre-wrap text-xs text-gray-700 dark:text-gray-300">
                        {JSON.stringify(job.progress, null, 2)}
                      </pre>
                    </div>
                  )}

                  {/* 실패한 작업의 오류 정보 표시 */}
                  {job.error && (
                    <div className="mt-2 p-2 bg-red-50 dark:bg-red-900 rounded text-xs text-red-700 dark:text-red-200">
                      <p className="font-medium mb-1">❌ 실행 실패:</p>
                      <pre className="whitespace-pre-wrap">{job.error}</pre>
                    </div>
                  )}
                </div>
              ))}
            </div>
          )}
        </div>

      </div>

        {/* 선택된 히스토리 항목 상세 정보 */}
        <div>
          {selectedHistoryItem && (
            <div className="rounded-lg shadow-md p-6 mt-6" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="flex justify-between items-center mb-4">
                <h3 className="text-xl font-semibold">
                  {isAgentSession(selectedHistoryItem) ? 
                    `에이전트 세션: ${selectedHistoryItem.title}` : 
                    `예약 작업: ${selectedHistoryItem.taskName}`
                  }
                </h3>
                <button
                  onClick={() => {
                    setSelectedHistoryItem(null);
                    setSelectedSession(null);
                  }}
                  className="px-3 py-1 bg-gray-500 text-white rounded hover:bg-gray-600 text-sm"
                >
                  닫기
                </button>
              </div>
              
              {isAgentSession(selectedHistoryItem) ? (
                // 에이전트 세션 상세 정보
                <>
                  <div className="mb-4 text-sm text-gray-600 dark:text-gray-400">
                    <p>프로바이더: {selectedHistoryItem.provider} | 모델: {selectedHistoryItem.model}</p>
                    <p>메시지 수: {selectedHistoryItem.history.length}개 | 업데이트: {formatDate(selectedHistoryItem.updatedAt)}</p>
                  </div>

                  <div className="space-y-4 max-h-400 overflow-y-auto">
                    {selectedHistoryItem.history.length === 0 ? (
                      <div className="text-center py-8 text-gray-500">
                        대화 기록이 없습니다.
                      </div>
                    ) : (
                      selectedHistoryItem.history.map((message, index) => (
                        <div
                          key={index}
                          className={`p-4 rounded-lg ${
                            message.role === 'user'
                              ? 'bg-blue-50 dark:bg-blue-900 border-l-4 border-blue-400'
                              : message.role === 'assistant'
                              ? 'bg-green-50 dark:bg-green-900 border-l-4 border-green-400'
                              : 'bg-gray-50 dark:bg-gray-700 border-l-4 border-gray-400'
                          }`}
                        >
                          <div className="flex justify-between items-start mb-2">
                            <div className="flex items-center space-x-2">
                              <span className={`px-2 py-1 rounded text-xs font-medium ${
                                message.role === 'user'
                                  ? 'bg-blue-100 text-blue-800 dark:bg-blue-800 dark:text-blue-200'
                                  : message.role === 'assistant'
                                  ? 'bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-200'
                                  : 'bg-gray-100 text-gray-800 dark:bg-gray-600 dark:text-gray-200'
                              }`}>
                                {message.role === 'user' ? '👤 사용자' : 
                                message.role === 'assistant' ? '🤖 AI' : '⚙️ 시스템'}
                              </span>
                            </div>
                            {message.timestamp && (
                              <span className="text-xs text-gray-500">
                                {formatDate(message.timestamp)}
                              </span>
                            )}
                          </div>
                          
                          <div className="text-sm text-gray-800 dark:text-gray-200">
                            {message.role === 'assistant' && 
                            (message.content?.includes('```') || 
                              message.content?.includes('def ') || 
                              message.content?.includes('import ')) ? (
                              <CodeViewer
                                code={message.content}
                                language="python"
                                title="AI 응답"
                                className="max-h-300 overflow-y-auto"
                              />
                            ) : (
                              <pre className="whitespace-pre-wrap font-sans">
                                {message.content}
                              </pre>
                            )}
                          </div>
                        </div>
                      ))
                    )}
                  </div>
                </>
              ) : (
                // 예약 작업 히스토리 상세 정보
                <>
                  <div className="mb-4 text-sm text-gray-600 dark:text-gray-400">
                    <p>예약 이름: {selectedHistoryItem.scheduleName}</p>
                    <p>작업 ID: {selectedHistoryItem.taskId} | 예약 ID: {selectedHistoryItem.scheduleId}</p>
                    <p>실행 시작: {formatDate(selectedHistoryItem.startedAt)}</p>
                    {selectedHistoryItem.completedAt && (
                      <p>실행 완료: {formatDate(selectedHistoryItem.completedAt)}</p>
                    )}
                    <p>실행 시간: {selectedHistoryItem.executionTimeMs ? `${Math.round(selectedHistoryItem.executionTimeMs / 1000)}초` : '미상'}</p>
                    <p>종료 코드: {selectedHistoryItem.exitCode}</p>
                  </div>

                  <div className="space-y-4 max-h-400 overflow-y-auto">
                    {/* 원본 프롬프트 */}
                    {selectedHistoryItem.prompt && (
                      <div className="p-4 bg-blue-50 dark:bg-blue-900 border-l-4 border-blue-400 rounded-lg">
                        <div className="flex items-center space-x-2 mb-2">
                          <span className="px-2 py-1 bg-blue-100 text-blue-800 dark:bg-blue-800 dark:text-blue-200 rounded text-xs font-medium">
                            📝 원본 프롬프트
                          </span>
                        </div>
                        <pre className="text-sm text-blue-800 dark:text-blue-200 whitespace-pre-wrap font-sans">
                          {selectedHistoryItem.prompt}
                        </pre>
                      </div>
                    )}

                    {/* 생성된 코드 */}
                    {selectedHistoryItem.pythonCode && (
                      <div className="p-4 bg-gray-50 dark:bg-gray-700 border-l-4 border-gray-400 rounded-lg">
                        <div className="flex items-center space-x-2 mb-2">
                          <span className="px-2 py-1 bg-gray-100 text-gray-800 dark:bg-gray-600 dark:text-gray-200 rounded text-xs font-medium">
                            💻 실행된 코드
                          </span>
                        </div>
                        <CodeViewer
                          code={selectedHistoryItem.pythonCode}
                          language="python"
                          title="실행된 Python 코드"
                          className="max-h-300 overflow-y-auto"
                        />
                      </div>
                    )}

                    {/* 실행 결과 */}
                    {selectedHistoryItem.stdout && (
                      <div className="p-4 bg-green-50 dark:bg-green-900 border-l-4 border-green-400 rounded-lg">
                        <div className="flex items-center space-x-2 mb-2">
                          <span className="px-2 py-1 bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-200 rounded text-xs font-medium">
                            ✅ 실행 출력
                          </span>
                        </div>
                        <pre className="text-sm text-green-800 dark:text-green-200 whitespace-pre-wrap font-mono bg-green-100 dark:bg-green-800 p-3 rounded">
                          {selectedHistoryItem.stdout}
                        </pre>
                      </div>
                    )}

                    {/* 에러 메시지 */}
                    {selectedHistoryItem.stderr && (
                      <div className="p-4 bg-red-50 dark:bg-red-900 border-l-4 border-red-400 rounded-lg">
                        <div className="flex items-center space-x-2 mb-2">
                          <span className="px-2 py-1 bg-red-100 text-red-800 dark:bg-red-800 dark:text-red-200 rounded text-xs font-medium">
                            ❌ 실행 에러
                          </span>
                        </div>
                        <pre className="text-sm text-red-800 dark:text-red-200 whitespace-pre-wrap font-mono bg-red-100 dark:bg-red-800 p-3 rounded">
                          {selectedHistoryItem.stderr}
                        </pre>
                      </div>
                    )}

                    {/* 에러 메시지 (별도) */}
                    {selectedHistoryItem.errorMessage && (
                      <div className="p-4 bg-red-50 dark:bg-red-900 border-l-4 border-red-400 rounded-lg">
                        <div className="flex items-center space-x-2 mb-2">
                          <span className="px-2 py-1 bg-red-100 text-red-800 dark:bg-red-800 dark:text-red-200 rounded text-xs font-medium">
                            ⚠️ 에러 메시지
                          </span>
                        </div>
                        <pre className="text-sm text-red-800 dark:text-red-200 whitespace-pre-wrap font-sans">
                          {selectedHistoryItem.errorMessage}
                        </pre>
                      </div>
                    )}
                  </div>
                </>
              )}
            </div>
          )}
        </div>


      </div>

      {/* 작업 저장 모달 */}
      {showSaveTaskModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="rounded-lg p-6 max-w-md w-full mx-4" style={{ backgroundColor: 'var(--card-bg)' }}>
            <h3 className="text-lg font-semibold mb-4">작업 저장</h3>
            
            <div className="mb-4">
              <label className="block text-sm font-medium mb-2">작업 이름</label>
              <input
                type="text"
                value={saveTaskName}
                onChange={(e) => setSaveTaskName(e.target.value)}
                className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700"
                placeholder="저장할 작업의 이름을 입력하세요"
                autoFocus
              />
            </div>

            {lastAgentResult && (
              <div className="mb-4">
                <label className="block text-sm font-medium mb-2">프롬프트</label>
                <textarea
                  value={lastAgentResult.prompt}
                  readOnly
                  className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 bg-gray-50 dark:bg-gray-900 text-sm h-20 resize-none"
                />
              </div>
            )}

            <div className="flex justify-end space-x-3">
              <button
                onClick={() => {
                  setShowSaveTaskModal(false);
                  setSaveTaskName('');
                }}
                className="px-4 py-2 bg-gray-500 text-white rounded hover:bg-gray-600"
              >
                취소
              </button>
              <button
                onClick={() => {
                  if (saveTaskName.trim() && lastAgentResult) {
                    saveTaskFromResult(
                      saveTaskName.trim(),
                      lastAgentResult.prompt,
                      lastAgentResult.code
                    );
                  }
                }}
                disabled={!saveTaskName.trim()}
                className="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                저장
              </button>
            </div>
          </div>
        </div>
      )}

      {/* 예약 작업 생성 모달 */}
      {showScheduleModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="rounded-lg p-6 max-w-lg w-full mx-4 max-h-[90vh] overflow-y-auto" style={{ backgroundColor: 'var(--card-bg)' }}>
            <h3 className="text-lg font-semibold mb-4">예약 작업 생성</h3>
            
            <div className="space-y-4">
              {/* 작업 선택 */}
              <div>
                <label className="block text-sm font-medium mb-2">실행할 작업</label>
                <select
                  value={scheduleForm.taskId}
                  onChange={(e) => setScheduleForm({...scheduleForm, taskId: e.target.value})}
                  className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700"
                  required
                >
                  <option value="">작업을 선택하세요</option>
                  {savedTasks.map((task) => (
                    <option key={task.id} value={task.id}>
                      {task.name}
                    </option>
                  ))}
                </select>
              </div>

              {/* 예약 작업 이름 */}
              <div>
                <label className="block text-sm font-medium mb-2">예약 작업 이름</label>
                <input
                  type="text"
                  value={scheduleForm.name}
                  onChange={(e) => setScheduleForm({...scheduleForm, name: e.target.value})}
                  className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700"
                  placeholder="예약 작업의 이름을 입력하세요"
                  required
                />
              </div>

              {/* 스케줄 타입 */}
              <div>
                <label className="block text-sm font-medium mb-2">스케줄 타입</label>
                <div className="flex space-x-4">
                  <label className="flex items-center">
                    <input
                      type="radio"
                      value="once"
                      checked={scheduleForm.scheduleType === 'once'}
                      onChange={(e) => setScheduleForm({...scheduleForm, scheduleType: e.target.value as 'once' | 'recurring'})}
                      className="mr-2"
                    />
                    한 번 실행
                  </label>
                  <label className="flex items-center">
                    <input
                      type="radio"
                      value="recurring"
                      checked={scheduleForm.scheduleType === 'recurring'}
                      onChange={(e) => setScheduleForm({...scheduleForm, scheduleType: e.target.value as 'once' | 'recurring'})}
                      className="mr-2"
                    />
                    반복 실행
                  </label>
                </div>
              </div>

              {/* 실행 날짜/시간 */}
              <div>
                <label className="block text-sm font-medium mb-2">
                  {scheduleForm.scheduleType === 'once' ? '실행 날짜/시간' : '시작 날짜/시간'}
                  {scheduleForm.scheduleType === 'recurring' && 
                   scheduleForm.recurringPattern?.type === 'cron' && 
                   <span className="text-gray-500 font-normal"> (선택사항)</span>}
                </label>
                <input
                  type="datetime-local"
                  value={scheduleForm.scheduledAt}
                  onChange={(e) => setScheduleForm({...scheduleForm, scheduledAt: e.target.value})}
                  className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 dark:text-white dark:color-scheme-dark"
                  required={scheduleForm.scheduleType === 'once' || 
                           (scheduleForm.scheduleType === 'recurring' && 
                            scheduleForm.recurringPattern?.type !== 'cron')}
                />
                {scheduleForm.scheduleType === 'recurring' && 
                 scheduleForm.recurringPattern?.type === 'cron' && (
                  <p className="text-xs text-gray-500 mt-1">
                    Cron 표현식이 실행 시간을 정의하므로 시작 날짜는 선택사항입니다.
                  </p>
                )}
              </div>

              {/* 반복 패턴 (반복 실행 선택 시) */}
              {scheduleForm.scheduleType === 'recurring' && (
                <div className="space-y-3 p-4 bg-blue-50 dark:bg-blue-900 rounded-lg">
                  <h4 className="text-sm font-medium text-blue-800 dark:text-blue-200">반복 설정</h4>
                  
                  {/* 반복 타입 */}
                  <div>
                    <label className="block text-sm font-medium mb-2">반복 주기</label>
                    <select
                      value={scheduleForm.recurringPattern?.type || 'daily'}
                      onChange={(e) => setScheduleForm({
                        ...scheduleForm,
                        recurringPattern: {
                          ...scheduleForm.recurringPattern,
                          type: e.target.value as 'daily' | 'weekly' | 'monthly' | 'cron',
                          interval: scheduleForm.recurringPattern?.interval || 1,
                          cronExpression: e.target.value === 'cron' ? '0 9 * * *' : undefined
                        }
                      })}
                      className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700"
                    >
                      <option value="daily">매일</option>
                      <option value="weekly">매주</option>
                      <option value="monthly">매월</option>
                      <option value="cron">고급 설정 (Cron)</option>
                    </select>
                  </div>

                  {/* 반복 간격 (Cron 설정이 아닐 때만 표시) */}
                  {scheduleForm.recurringPattern?.type !== 'cron' && (
                    <div>
                      <label className="block text-sm font-medium mb-2">반복 간격</label>
                      <input
                        type="number"
                        min="1"
                        max="100"
                        value={scheduleForm.recurringPattern?.interval || 1}
                        onChange={(e) => setScheduleForm({
                          ...scheduleForm,
                          recurringPattern: {
                            ...scheduleForm.recurringPattern,
                            type: scheduleForm.recurringPattern?.type || 'daily',
                            interval: parseInt(e.target.value) || 1
                          }
                        })}
                        className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700"
                      />
                      <p className="text-xs text-gray-500 mt-1">
                        {scheduleForm.recurringPattern?.type === 'daily' && '일 마다'}
                        {scheduleForm.recurringPattern?.type === 'weekly' && '주 마다'}
                        {scheduleForm.recurringPattern?.type === 'monthly' && '개월 마다'}
                      </p>
                    </div>
                  )}

                  {/* 요일 선택 (주간 반복 시) */}
                  {scheduleForm.recurringPattern?.type === 'weekly' && (
                    <div>
                      <label className="block text-sm font-medium mb-2">실행 요일</label>
                      <div className="grid grid-cols-7 gap-2">
                        {['일', '월', '화', '수', '목', '금', '토'].map((day, index) => (
                          <label key={index} className="flex flex-col items-center">
                            <input
                              type="checkbox"
                              checked={scheduleForm.recurringPattern?.daysOfWeek?.includes(index) || false}
                              onChange={(e) => {
                                const currentDays = scheduleForm.recurringPattern?.daysOfWeek || [];
                                const newDays = e.target.checked
                                  ? [...currentDays, index]
                                  : currentDays.filter(d => d !== index);
                                setScheduleForm({
                                  ...scheduleForm,
                                  recurringPattern: {
                                    ...scheduleForm.recurringPattern,
                                    type: 'weekly',
                                    interval: scheduleForm.recurringPattern?.interval || 1,
                                    daysOfWeek: newDays
                                  }
                                });
                              }}
                              className="mb-1"
                            />
                            <span className="text-xs">{day}</span>
                          </label>
                        ))}
                      </div>
                    </div>
                  )}

                  {/* 월 일자 선택 (월간 반복 시) */}
                  {scheduleForm.recurringPattern?.type === 'monthly' && (
                    <div>
                      <label className="block text-sm font-medium mb-2">매월 실행 일자</label>
                      <input
                        type="number"
                        min="1"
                        max="31"
                        value={scheduleForm.recurringPattern?.dayOfMonth || 1}
                        onChange={(e) => setScheduleForm({
                          ...scheduleForm,
                          recurringPattern: {
                            ...scheduleForm.recurringPattern,
                            type: 'monthly',
                            interval: scheduleForm.recurringPattern?.interval || 1,
                            dayOfMonth: parseInt(e.target.value) || 1
                          }
                        })}
                        className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700"
                      />
                      <p className="text-xs text-gray-500 mt-1">매월 이 날짜에 실행됩니다</p>
                    </div>
                  )}

                  {/* Cron 표현식 입력 (고급 설정 시) */}
                  {scheduleForm.recurringPattern?.type === 'cron' && (
                    <div className="space-y-3">
                      <div>
                        <label className="block text-sm font-medium mb-2">Cron 표현식</label>
                        <input
                          type="text"
                          value={scheduleForm.recurringPattern?.cronExpression || '0 9 * * *'}
                          onChange={(e) => {
                            const cronExpression = e.target.value;
                            setScheduleForm({
                              ...scheduleForm,
                              recurringPattern: {
                                ...scheduleForm.recurringPattern,
                                type: 'cron',
                                interval: 0,
                                cronExpression: cronExpression
                              }
                            });
                          }}
                          className={`w-full p-2 border rounded-lg dark:bg-gray-700 font-mono ${
                            scheduleForm.recurringPattern?.cronExpression && 
                            !validateCronExpression(scheduleForm.recurringPattern.cronExpression)
                              ? 'border-red-500 dark:border-red-400'
                              : 'border-gray-300 dark:border-gray-600'
                          }`}
                          placeholder="0 9 * * *"
                        />
                        {scheduleForm.recurringPattern?.cronExpression && 
                         !validateCronExpression(scheduleForm.recurringPattern.cronExpression) && (
                          <p className="text-xs text-red-500 mt-1">
                            유효하지 않은 cron 표현식입니다. 형식: 분 시 일 월 요일
                          </p>
                        )}
                        {scheduleForm.recurringPattern?.cronExpression && 
                         validateCronExpression(scheduleForm.recurringPattern.cronExpression) && (
                          <p className="text-xs text-green-600 dark:text-green-400 mt-1">
                            ✓ 유효한 cron 표현식입니다 ({getCronDescription(scheduleForm.recurringPattern.cronExpression)})
                          </p>
                        )}
                        <p className="text-xs text-gray-500 mt-1">
                          형식: 분 시 일 월 요일 (예: "0 9 * * *" = 매일 오전 9시)
                        </p>
                      </div>

                      {/* Cron 표현식 예시 */}
                      <div className="bg-gray-50 dark:bg-gray-800 p-3 rounded">
                        <h5 className="text-xs font-medium mb-2">자주 사용하는 패턴 예시:</h5>
                        <div className="space-y-1 text-xs">
                          <div className="flex justify-between">
                            <span className="font-mono">*/5 * * * *</span>
                            <span className="text-gray-600 dark:text-gray-400">5분마다</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="font-mono">*/10 * * * *</span>
                            <span className="text-gray-600 dark:text-gray-400">10분마다</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="font-mono">0 */2 * * *</span>
                            <span className="text-gray-600 dark:text-gray-400">2시간마다</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="font-mono">0 9 * * 1-5</span>
                            <span className="text-gray-600 dark:text-gray-400">평일 오전 9시</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="font-mono">0 0 1 * *</span>
                            <span className="text-gray-600 dark:text-gray-400">매월 1일 자정</span>
                          </div>
                        </div>
                        <button
                          type="button"
                          onClick={() => setScheduleForm({
                            ...scheduleForm,
                            recurringPattern: {
                              ...scheduleForm.recurringPattern,
                              type: 'cron',
                              interval: 0,
                              cronExpression: '*/5 * * * *'
                            }
                          })}
                          className="mt-2 text-xs text-blue-600 dark:text-blue-400 hover:underline"
                        >
                          5분마다 실행으로 설정
                        </button>
                      </div>
                    </div>
                  )}
                </div>
              )}
            </div>

            <div className="flex justify-end space-x-3 mt-6">
              <button
                onClick={() => {
                  setShowScheduleModal(false);
                  setScheduleForm({
                    taskId: '',
                    name: '',
                    scheduleType: 'once',
                    scheduledAt: '',
                    recurringPattern: undefined
                  });
                }}
                className="px-4 py-2 bg-gray-500 text-white rounded hover:bg-gray-600"
              >
                취소
              </button>
              <button
                onClick={() => {
                  if (isFormValid()) {
                    createSchedule(scheduleForm);
                  }
                }}
                disabled={!isFormValid() || isCreatingSchedule}
                className="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {isCreatingSchedule ? '생성 중...' : '예약 작업 생성'}
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
} 