import { NextRequest, NextResponse } from 'next/server';

/**
 * GET /api/a2a-ping
 * A2A 네트워크 상태 확인
 */
export async function GET(request: NextRequest): Promise<NextResponse> {
  try {
    const url = new URL(request.url);
    const action = url.searchParams.get('action');

    switch (action) {
      case 'capabilities':
        return NextResponse.json({
          status: 'ok',
          message: 'A2A 네트워크 테스트 성공',
          capabilities: [],
          agent: {
            id: 'airun-mcp-agent',
            name: 'AI MCP Agent',
            version: '1.0.0',
            protocol: 'A2A/1.0'
          },
          timestamp: new Date().toISOString()
        });

      case 'status':
        return NextResponse.json({
          isConnected: false,
          connectedAgents: 0,
          exposedTools: 0,
          health: 'testing',
          timestamp: new Date().toISOString()
        });

      default:
        return NextResponse.json({
          status: 'ok',
          message: 'A2A 엔드포인트 작동 중',
          timestamp: new Date().toISOString(),
          agent: 'airun-mcp-agent',
          protocol: 'A2A/1.0',
          enabled: true
        });
    }
  } catch (error) {
    console.error('A2A ping failed:', error);
    return NextResponse.json(
      { error: 'Service error', details: error instanceof Error ? error.message : 'Unknown error' },
      { status: 500 }
    );
  }
} 