import { NextRequest, NextResponse } from 'next/server';

import { getLLMFinetuneServerUrl } from '@/config/serverConfig';

const LLM_FINETUNE_API_BASE_URL = getLLMFinetuneServerUrl();

export async function GET() {
  try {
    console.log('[LLM API] 데이터셋 목록 요청 시작');
    console.log('[LLM API] LLM_FINETUNE_API_BASE_URL:', LLM_FINETUNE_API_BASE_URL);
    
    const response = await fetch(`${LLM_FINETUNE_API_BASE_URL}/datasets`);
    console.log('[LLM API] 응답 상태:', response.status, response.statusText);
    
    if (!response.ok) {
      console.error('[LLM API] HTTP 오류:', response.status, response.statusText);
      const errorText = await response.text();
      console.error('[LLM API] 오류 내용:', errorText);
      
      return NextResponse.json(
        { 
          success: false, 
          error: `LLM 서버 오류: ${response.status} ${response.statusText}`,
          details: errorText
        },
        { status: response.status }
      );
    }
    
    const data = await response.json();
    console.log('[LLM API] 성공적으로 데이터 받음:', data);
    
    return NextResponse.json(data);
  } catch (error) {
    console.error('[LLM API] 데이터셋 목록 호출 실패:', error);
    
    const isNetworkError = error instanceof Error && (
      error.message.includes('ECONNREFUSED') ||
      error.message.includes('ENOTFOUND') ||
      error.message.includes('ETIMEDOUT') ||
      error.message.includes('fetch failed')
    );
    
    return NextResponse.json(
      { 
        success: false, 
        error: isNetworkError 
          ? `LLM 파인튜닝 서버(${LLM_FINETUNE_API_BASE_URL})에 연결할 수 없습니다. 서버가 실행 중인지 확인하세요.`
          : '데이터셋 목록을 불러올 수 없습니다.',
        details: error instanceof Error ? error.message : String(error)
      },
      { status: isNetworkError ? 503 : 500 }
    );
  }
}

export async function POST(request: NextRequest) {
  try {
    const contentType = request.headers.get('content-type');
    
    // FormData (파일 업로드)인 경우
    if (contentType?.includes('multipart/form-data')) {
      const formData = await request.formData();
      const method = formData.get('method') as string;
      
      if (method === 'web-crawling') {
        // 웹 크롤링 방식
        const urlsFile = formData.get('urlsFile') as File;
        const name = formData.get('name') as string;
        const description = formData.get('description') as string;
        const chunkSize = formData.get('chunkSize') as string;
        const generateEmbeddings = formData.get('generateEmbeddings') === 'true';
        const generateQA = formData.get('generateQA') as string;
        const tag = formData.get('tag') as string;
        
        if (!urlsFile || !name) {
          return NextResponse.json(
            { success: false, error: 'URLs 파일과 데이터셋 이름은 필수입니다.' },
            { status: 400 }
          );
        }
        
        // LLM 서버로 전송
        const llmFormData = new FormData();
        llmFormData.append('urlsFile', urlsFile);
        llmFormData.append('name', name);
        llmFormData.append('description', description || '');
        llmFormData.append('chunkSize', chunkSize || '900');
        llmFormData.append('generateEmbeddings', generateEmbeddings.toString());
        if (generateQA) llmFormData.append('generateQA', generateQA);
        if (tag) llmFormData.append('tag', tag);
        
        const response = await fetch(`${LLM_FINETUNE_API_BASE_URL}/datasets/web-crawling`, {
          method: 'POST',
          body: llmFormData
        });
        
        const data = await response.json();
        return NextResponse.json(data);
        
      } else if (method === 'file-upload') {
        // 파일 업로드 방식
        const file = formData.get('file') as File;
        const name = formData.get('name') as string;
        const description = formData.get('description') as string;
        
        if (!file || !name) {
          return NextResponse.json(
            { success: false, error: '파일과 데이터셋 이름은 필수입니다.' },
            { status: 400 }
          );
        }
        
        const llmFormData = new FormData();
        llmFormData.append('file', file);
        llmFormData.append('name', name);
        llmFormData.append('description', description || '');
        
        const response = await fetch(`${LLM_FINETUNE_API_BASE_URL}/datasets/file-upload`, {
          method: 'POST',
          body: llmFormData
        });
        
        const data = await response.json();
        return NextResponse.json(data);
        
      } else if (method === 'document-instruction') {
        // 문서 기반 instruction 생성
        const document = formData.get('document') as File;
        const name = formData.get('name') as string;
        const description = formData.get('description') as string;
        const provider = formData.get('provider') as string;
        const model = formData.get('model') as string;
        const instructionCount = formData.get('instructionCount') as string;
        
        if (!document || !name) {
          return NextResponse.json(
            { success: false, error: '문서와 데이터셋 이름은 필수입니다.' },
            { status: 400 }
          );
        }
        
        const llmFormData = new FormData();
        llmFormData.append('document', document);
        llmFormData.append('name', name);
        llmFormData.append('description', description || '');
        llmFormData.append('provider', provider || 'ollama');
        llmFormData.append('model', model || 'hamonize:latest');
        llmFormData.append('instructionCount', instructionCount || '10');
        
        const response = await fetch(`${LLM_FINETUNE_API_BASE_URL}/datasets/document-instruction`, {
          method: 'POST',
          body: llmFormData
        });
        
        const data = await response.json();
        return NextResponse.json(data);
      }
    } else {
      // JSON 데이터인 경우
      let body;
      try {
        const text = await request.text();
        console.log('Raw request body:', text);
        
        if (!text.trim()) {
          return NextResponse.json(
            { success: false, error: '요청 본문이 비어있습니다.' },
            { status: 400 }
          );
        }
        
        body = JSON.parse(text);
      } catch (parseError) {
        console.error('JSON 파싱 오류:', parseError);
        return NextResponse.json(
          { success: false, error: '유효하지 않은 JSON 형식입니다.' },
          { status: 400 }
        );
      }
      
      const method = body.method || body.type; // type 필드도 지원

      if (method === 'web-crawl' || method === 'web-crawling') {
        // 웹 크롤링 JSON 방식
        const response = await fetch(`${LLM_FINETUNE_API_BASE_URL}/datasets/web-crawling`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            name: body.name,
            description: body.description || '',
            urls: body.url || body.urls, // url 또는 urls 필드 지원
            chunkSize: body.chunkSize || 900,
            generateQA: body.generateQA || false,
            embed: body.embed || false,
            tag: body.tag || ''
          })
        });

        const data = await response.json();
        return NextResponse.json(data);

      } else if (method === 'rag-conversion') {
        // RAG → LLM 데이터셋 변환
        const response = await fetch(`${LLM_FINETUNE_API_BASE_URL}/datasets/rag-conversion`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            ragDatasetId: body.ragDatasetId,
            name: body.name,
            description: body.description || '',
            instructionTemplate: body.instructionTemplate || '',
            provider: body.provider || 'ollama',
            model: body.model || 'hamonize:latest'
          })
        });
        
        const data = await response.json();
        return NextResponse.json(data);
      }
    }
    
    return NextResponse.json(
      { success: false, error: '지원하지 않는 요청 형식입니다.' },
      { status: 400 }
    );
    
  } catch (error) {
    console.error('LLM 데이터셋 생성 실패:', error);
    return NextResponse.json(
      { success: false, error: 'LLM 데이터셋을 생성할 수 없습니다.' },
      { status: 500 }
    );
  }
}

export async function DELETE(request: NextRequest) {
  try {
    const body = await request.json();
    const { datasetId } = body;
    
    if (!datasetId) {
      return NextResponse.json(
        { success: false, error: '데이터셋 ID가 필요합니다.' },
        { status: 400 }
      );
    }
    
    const response = await fetch(`${LLM_FINETUNE_API_BASE_URL}/datasets/${datasetId}`, {
      method: 'DELETE'
    });
    
    const result = await response.json();
    
    if (response.ok && result.success) {
      return NextResponse.json({
        success: true,
        message: 'LLM 데이터셋이 성공적으로 삭제되었습니다.'
      });
    } else {
      return NextResponse.json(
        { success: false, error: result.error || 'LLM 데이터셋 삭제에 실패했습니다.' },
        { status: response.status || 500 }
      );
    }
    
  } catch (error) {
    console.error('LLM 데이터셋 삭제 실패:', error);
    return NextResponse.json(
      { success: false, error: 'LLM 데이터셋 삭제 중 오류가 발생했습니다.' },
      { status: 500 }
    );
  }
}