import { NextRequest, NextResponse } from 'next/server';
import { createClient } from 'redis';

// Redis 클라이언트 생성
const createRedisClient = async () => {
  const { getRedisUrl } = await import('@/config/serverConfig');
  const redisUrl = getRedisUrl();
  
  const client = createClient({
    url: redisUrl
  });
  
  await client.connect();
  return client;
};

// Redis 키 패턴
const JOB_KEY_PREFIX = 'background_job:';
const JOB_INDEX_KEY = 'background_jobs:index';

// 오래된 작업 정리 함수 (완료/실패된 작업 중 24시간 이상 된 것들)
async function cleanupOldJobs() {
  const redis = await createRedisClient();
  const results = {
    cleaned: 0,
    active: 0,
    errors: [] as string[]
  };
  
  try {
    const jobIds = await redis.sMembers(JOB_INDEX_KEY);
    console.log(`[cleanupOldJobs] 총 ${jobIds.length}개 작업 확인 중...`);
    
    for (const jobId of jobIds) {
      try {
        const jobKey = `${JOB_KEY_PREFIX}${jobId}`;
        const jobData = await redis.hGetAll(jobKey);
        
        if (!jobData || !jobData.id) {
          // 데이터가 없는 잘못된 인덱스 항목 제거
          await redis.sRem(JOB_INDEX_KEY, jobId);
          results.cleaned++;
          continue;
        }
        
        const status = jobData.status;
        const updatedAt = new Date(jobData.updatedAt || jobData.createdAt);
        const now = new Date();
        const ageInHours = (now.getTime() - updatedAt.getTime()) / (1000 * 60 * 60);
        
        // 완료/실패된 작업이 24시간 이상 된 경우 정리
        if ((status === 'completed' || status === 'failed') && ageInHours > 24) {
          await redis.del(jobKey);
          await redis.sRem(JOB_INDEX_KEY, jobId);
          results.cleaned++;
          console.log(`[cleanupOldJobs] 정리됨: ${jobId} (${status}, ${Math.round(ageInHours)}시간 전)`);
        }
        // 진행 중인 작업이 2시간 이상 된 경우 실패로 처리
        else if ((status === 'pending' || status === 'running') && ageInHours > 2) {
          await redis.hSet(jobKey, {
            status: 'failed',
            progress: '0',
            message: '❌ 시스템 정리: 오래된 작업으로 실패 처리됨',
            error: 'Job too old, marked as failed by cleanup',
            updatedAt: new Date().toISOString()
          });
          
          // WebSocket 알림 발행
          await redis.publish('websocket:job-status', JSON.stringify({
            jobId,
            status: 'failed',
            message: '❌ 시스템 정리: 오래된 작업으로 실패 처리됨',
            timestamp: new Date().toISOString()
          }));
          
          results.cleaned++;
          console.log(`[cleanupOldJobs] 오래된 작업 실패 처리: ${jobId} (${Math.round(ageInHours)}시간)`);
        } else {
          results.active++;
        }
      } catch (error) {
        results.errors.push(`Job ${jobId}: ${error instanceof Error ? error.message : 'Unknown error'}`);
        console.error(`[cleanupOldJobs] 작업 처리 오류: ${jobId}`, error);
      }
    }
    
  } finally {
    await redis.quit();
  }
  
  return results;
}

// 스터크 상태 검증 및 정리 (실제 프로세스가 없는 작업들)
async function cleanupStuckJobs() {
  const redis = await createRedisClient();
  const results = {
    stuck: 0,
    validated: 0
  };
  
  try {
    const jobIds = await redis.sMembers(JOB_INDEX_KEY);
    
    for (const jobId of jobIds) {
      try {
        const jobKey = `${JOB_KEY_PREFIX}${jobId}`;
        const jobData = await redis.hGetAll(jobKey);
        
        if (!jobData || !jobData.id) continue;
        
        const status = jobData.status;
        const updatedAt = new Date(jobData.updatedAt || jobData.createdAt);
        const now = new Date();
        const ageInMinutes = (now.getTime() - updatedAt.getTime()) / (1000 * 60);
        
        // 진행 중 상태이지만 30분 이상 업데이트가 없는 경우 스터크 상태로 간주
        if ((status === 'running' || status === 'pending') && ageInMinutes > 30) {
          // 실제로는 더 정교한 검증이 필요하지만, 여기서는 간단히 처리
          await redis.hSet(jobKey, {
            status: 'failed',
            progress: '0',
            message: '❌ 시스템 정리: 30분 이상 반응 없는 작업',
            error: 'Job appears to be stuck, marked as failed',
            updatedAt: new Date().toISOString()
          });
          
          results.stuck++;
          console.log(`[cleanupStuckJobs] 스터크 작업 정리: ${jobId} (${Math.round(ageInMinutes)}분 무응답)`);
        } else {
          results.validated++;
        }
      } catch (error) {
        console.error(`[cleanupStuckJobs] 작업 검증 오류: ${jobId}`, error);
      }
    }
    
  } finally {
    await redis.quit();
  }
  
  return results;
}

// Redis job cleanup API
export async function POST(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const mode = searchParams.get('mode') || 'safe'; // 'safe', 'force', 'stuck'
    
    console.log(`[Redis Job Cleanup] 시작 (모드: ${mode})`);
    
    let results: any = {};
    
    if (mode === 'safe' || mode === 'force') {
      // 오래된 작업 정리
      results.cleanup = await cleanupOldJobs();
    }
    
    if (mode === 'stuck' || mode === 'force') {
      // 스터크 작업 정리
      results.stuck = await cleanupStuckJobs();
    }
    
    const totalCleaned = (results.cleanup?.cleaned || 0) + (results.stuck?.stuck || 0);
    const totalActive = results.cleanup?.active || 0;
    
    console.log(`[Redis Job Cleanup] 완료: ${totalCleaned}개 정리, ${totalActive}개 유지`);
    
    return NextResponse.json({ 
      success: true,
      mode,
      cleaned: totalCleaned,
      active: totalActive,
      results,
      timestamp: new Date().toISOString(),
      message: `Redis job cleanup 완료 (${mode} 모드): ${totalCleaned}개 정리, ${totalActive}개 유지`
    });
    
  } catch (error) {
    console.error('[Redis Job Cleanup] 실패:', error);
    return NextResponse.json({ 
      success: false, 
      error: error instanceof Error ? error.message : 'Unknown error',
      timestamp: new Date().toISOString()
    }, { status: 500 });
  }
}

// GET 요청으로 상태만 조회 (정리 없이)
export async function GET(request: NextRequest) {
  try {
    const redis = await createRedisClient();
    
    try {
      const jobIds = await redis.sMembers(JOB_INDEX_KEY);
      const stats = {
        total: 0,
        pending: 0,
        running: 0,
        completed: 0,
        failed: 0,
        old_completed: 0, // 24시간 이상 된 완료 작업
        old_running: 0,   // 2시간 이상 된 진행 중 작업
        stuck: 0          // 30분 이상 무응답 작업
      };
      
      for (const jobId of jobIds) {
        const jobKey = `${JOB_KEY_PREFIX}${jobId}`;
        const jobData = await redis.hGetAll(jobKey);
        
        if (!jobData || !jobData.id) continue;
        
        stats.total++;
        const status = jobData.status;
        
        if (status === 'pending') stats.pending++;
        else if (status === 'running') stats.running++;
        else if (status === 'completed') stats.completed++;
        else if (status === 'failed') stats.failed++;
        
        // 나이 계산
        const updatedAt = new Date(jobData.updatedAt || jobData.createdAt);
        const now = new Date();
        const ageInHours = (now.getTime() - updatedAt.getTime()) / (1000 * 60 * 60);
        const ageInMinutes = (now.getTime() - updatedAt.getTime()) / (1000 * 60);
        
        if ((status === 'completed' || status === 'failed') && ageInHours > 24) {
          stats.old_completed++;
        }
        
        if ((status === 'pending' || status === 'running') && ageInHours > 2) {
          stats.old_running++;
        }
        
        if ((status === 'running' || status === 'pending') && ageInMinutes > 30) {
          stats.stuck++;
        }
      }
      
      return NextResponse.json({
        success: true,
        stats,
        recommendations: {
          safe_cleanup: stats.old_completed > 0 ? `${stats.old_completed}개의 오래된 완료 작업을 정리할 수 있습니다.` : '정리할 오래된 작업이 없습니다.',
          stuck_cleanup: stats.stuck > 0 ? `${stats.stuck}개의 무응답 작업을 확인하세요.` : '무응답 작업이 없습니다.',
          old_running: stats.old_running > 0 ? `${stats.old_running}개의 오래된 진행 중 작업이 있습니다.` : null
        }
      });
      
    } finally {
      await redis.quit();
    }
    
  } catch (error) {
    console.error('[Redis Job Status] 실패:', error);
    return NextResponse.json({ 
      success: false, 
      error: error instanceof Error ? error.message : 'Unknown error'
    }, { status: 500 });
  }
}