'use client';

import { useState } from 'react';
import { useRouter } from 'next/navigation';
import Image from 'next/image';
import { externalApiClient } from '@/lib/apiClient';

export default function LoginPage() {
  const router = useRouter();
  const [username, setUsername] = useState('');
  const [password, setPassword] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const handleLogin = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!username || !password) {
      setError('아이디와 비밀번호를 모두 입력해주세요.');
      return;
    }
    
    setIsLoading(true);
    setError(null);
    
    try {
      const response = await fetch('/api/auth/login', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ username, password }),
      });
      
      const data = await response.json();
      
      if (!response.ok) {
        throw new Error(data.error?.message || '로그인에 실패했습니다.');
      }
      
      if (data.success) {
        console.log('로그인 성공 응답:', data);
        
        // 로컬 스토리지에 사용자 정보와 토큰 저장
        if (data.data.token) {
          localStorage.setItem('userToken', data.data.token);
        } else {
          console.warn('⚠️ 토큰이 응답에 포함되지 않았습니다!');
        }
        
        // API 키 저장
        if (data.data.apiKeys && Array.isArray(data.data.apiKeys) && data.data.apiKeys.length > 0) {
          const apiKey = data.data.apiKeys[0].key;
          localStorage.setItem('apiKey', apiKey);
          externalApiClient.setApiKey(apiKey);
          console.log('✅ API 키가 저장되었습니다:', apiKey.substring(0, 10) + '...');
        } else {
          console.warn('⚠️ API 키가 응답에 포함되지 않았습니다!', data.data);
          // API 키가 없는 경우 기본값 사용하지 않고 오류 표시
          setError('API 키가 없습니다. 관리자에게 문의하세요.');
          setIsLoading(false);
          return;
        }
        
        if (data.data.user) {
          localStorage.setItem('userId', data.data.user.id);
          localStorage.setItem('username', data.data.user.username);
          localStorage.setItem('userEmail', data.data.user.email || '');
          localStorage.setItem('userRole', data.data.user.role || 'user');
          
          console.log('✅ 로컬 스토리지에 저장된 사용자 정보:', {
            userId: data.data.user.id,
            username: data.data.user.username,
            email: data.data.user.email || '',
            role: data.data.user.role || 'user'
          });
        } else {
          console.warn('⚠️ 사용자 정보가 응답에 포함되지 않았습니다!');
        }
        
        // 세션 스토리지에도 정보 저장 (백업)
        sessionStorage.setItem('isLoggedIn', 'true');
        sessionStorage.setItem('username', localStorage.getItem('username') || '');
        
        // 로컬 스토리지 확인
        console.log('확인: 로컬 스토리지 데이터:', {
          token: localStorage.getItem('userToken') ? '있음' : '없음',
          apiKey: localStorage.getItem('apiKey') ? '있음' : '없음',
          userId: localStorage.getItem('userId'),
          username: localStorage.getItem('username'),
          role: localStorage.getItem('userRole')
        });
        
        // 로그인 이벤트 발생
        console.log('🔔 login 이벤트 발생');
        window.dispatchEvent(new Event('login'));
        
        // 홈 페이지로 이동
        console.log('🚀 홈 페이지로 이동');
        window.location.href = '/';
      } else {
        throw new Error('로그인 처리 중 오류가 발생했습니다.');
      }
    } catch (err) {
      console.error('Login error:', err);
      setError(err instanceof Error ? err.message : '알 수 없는 오류가 발생했습니다.');
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen flex items-center justify-center" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
      <div className="w-full max-w-md p-8 space-y-8 rounded-lg shadow-md" style={{ backgroundColor: 'var(--body-bg)', border: '1px solid var(--border-color)' }}>
        <div className="text-center">
            <h3 className="text-3xl font-bold text-gray-900 dark:text-white">로그인</h3>
        </div>
        
        {error && (
          <div className="p-4 mb-4 text-sm text-red-700 dark:text-red-300 rounded-lg" style={{ backgroundColor: 'var(--error-bg)' }}>
            {error}
          </div>
        )}
        
        <form className="mt-8 space-y-6" onSubmit={handleLogin}>
          <div>
            <label htmlFor="username" className="block text-sm font-medium text-gray-700 dark:text-gray-300">
              아이디
            </label>
            <input
              id="username"
              name="username"
              type="text"
              autoComplete="username"
              required
              value={username}
              onChange={(e) => setUsername(e.target.value)}
              className="mt-1 block w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
              disabled={isLoading}
            />
          </div>
          
          <div>
            <label htmlFor="password" className="block text-sm font-medium text-gray-700 dark:text-gray-300">
              비밀번호
            </label>
            <input
              id="password"
              name="password"
              type="password"
              autoComplete="current-password"
              required
              value={password}
              onChange={(e) => setPassword(e.target.value)}
              className="mt-1 block w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
              disabled={isLoading}
            />
          </div>
          
          <div>
            <button
              type="submit"
              disabled={isLoading}
              className={`w-full flex justify-center py-2 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white ${
                isLoading
                  ? 'bg-blue-400 cursor-not-allowed'
                  : 'bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500'
              }`}
            >
              {isLoading ? (
                <div className="flex items-center">
                  <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                  <span>로그인 중...</span>
                </div>
              ) : (
                '로그인'
              )}
            </button>
          </div>
        </form>
        
        <div className="mt-6">
          <div className="text-sm text-center text-gray-600 dark:text-gray-400">
          </div>
        </div>
      </div>
    </div>
  );
} 