import React from 'react';
import ReactMarkdown from 'react-markdown';
import remarkGfm from 'remark-gfm';
import Prism from 'prismjs';
import 'prismjs/components/prism-python';
import 'prismjs/components/prism-jsx';
import 'prismjs/components/prism-tsx';
import 'prismjs/components/prism-typescript';
import 'prismjs/components/prism-json';
import 'prismjs/components/prism-bash';
import 'prismjs/components/prism-markdown';
import 'prismjs/themes/prism-tomorrow.css';
import { useRef, useEffect } from 'react';
import { useTheme } from 'next-themes';
import remarkBreaks from 'remark-breaks';
// remarkBoldBox 임시 제거 - 플러그인 오류 방지
// TODO: remarkBoldBox 플러그인 문제 해결 후 다시 활성화

interface DocumentRendererProps {
  content: string;
  className?: string;
}

// 문서용 VSCode 스타일 코드블록 컴포넌트
function DocumentCodeBlock({ code, language = 'plaintext' }: { code: string; language?: string }) {
  const { resolvedTheme } = useTheme();
  const isDark = resolvedTheme === 'dark';
  const darkBg = '#101014';
  const codeBg = isDark ? `bg-[${darkBg}]` : 'bg-[#f5f5f5]';
  const borderColor = isDark ? 'border-gray-800' : 'border-gray-200';
  const lineNumberBg = isDark ? `bg-[${darkBg}]` : 'bg-[#f5f5f5]';
  const lineNumberText = isDark ? 'text-gray-500' : 'text-gray-400';

  // Prism 하이라이트
  const highlighted = Prism.highlight(code, Prism.languages[language] || Prism.languages.plaintext, language);
  const highlightedLines = highlighted.split('\n');
  const codeLines = code.split('\n');

  return (
    <div className={`relative group rounded-lg my-3 border ${borderColor} overflow-hidden`}>
      {/* 복사 버튼 */}
      <button
        className="absolute top-2 right-2 z-10 px-2 py-1 text-xs bg-gray-700 text-white rounded hover:bg-blue-600 transition"
        onClick={() => {
          navigator.clipboard.writeText(code);
        }}
        title="코드 복사"
      >
        복사
      </button>
      <pre className="m-0 p-0 font-mono" style={{ fontSize: '11px', lineHeight: '1.6' }}>
        {codeLines.map((line, i) => (
          <div key={i} className="flex">
            <span
              className={`select-none text-right pr-2 pl-3 ${lineNumberText} ${lineNumberBg}`}
              style={{ minWidth: '2em', background: isDark ? darkBg : '#f5f5f5', fontSize: '11px', lineHeight: '1.6' }}
            >
              {i + 1}
            </span>
            <span
              className="pl-2 pr-4 flex-1"
              style={{ background: 'transparent', display: 'block', fontSize: '11px', lineHeight: '1.6' }}
              dangerouslySetInnerHTML={{ __html: highlightedLines[i] || '' }}
            />
          </div>
        ))}
      </pre>
    </div>
  );
}

export const DocumentRenderer: React.FC<DocumentRendererProps> = ({ content, className = '' }) => {
  // 콘텐츠 안전성 검사
  const safeContent = content || '';
  
  // 출처 섹션이 있는지 확인 (안전하게)
  const hasReferencesSection = safeContent.includes('# 출처(References)') || safeContent.includes('## 출처');

  // 플러그인 배열을 안전하게 구성
  const remarkPlugins = [
    remarkGfm,
    remarkBreaks
    // remarkBoldBox 임시 제거됨
  ];

  return (
    <div className={`markdown-content text-sm ${className}`} style={{ fontFamily: 'Spoqa Han Sans Neo' }}>
      <ReactMarkdown
        remarkPlugins={remarkPlugins}
        unwrapDisallowed={true}
        components={{
          // 강조 텍스트 (**text**)
          strong: (props: any) => (
            <strong className="font-bold text-orange-600 dark:text-orange-400 bg-orange-50 dark:bg-orange-900/20 px-1 py-0.5 rounded">
              {props.children}
            </strong>
          ),
          // 기울임 텍스트 (*text*)
          em: (props: any) => (
            <span className="italic text-purple-600 dark:text-purple-400">
              {props.children}
            </span>
          ),
          // 코드 블록
          code: (props: any) => {
            const { className, children, ...restProps } = props || {};
            const inline = !className?.includes('language-');
            if (inline) {
              return (
                <code
                  className="px-1.5 py-0.5 bg-gray-100 dark:bg-gray-800 text-red-600 dark:text-red-400 rounded text-sm font-mono"
                  {...restProps}
                >
                  {children}
                </code>
              );
            }
            // language 추출
            const match = /language-(\w+)/.exec(className || '');
            return (
              <DocumentCodeBlock code={String(children || '').replace(/\n$/, '')} language={match?.[1] || 'plaintext'} />
            );
          },
          // 링크
          a: (props: any) => {
            const { href, children } = props || {};
            if (!href) return <span>{children}</span>;
            
            // 내부 링크 처리
            if (href.startsWith('#')) {
              return (
                <a
                  href={href}
                  className="text-blue-600 dark:text-blue-400 hover:underline"
                >
                  {children}
                </a>
              );
            }
            
            // 다운로드 링크 처리
            if (href.includes('/api/v1/rag/download')) {
              const childrenText = React.Children.toArray(children).join('');
              const filename = childrenText.trim();
              
              return (
                <button
                  onClick={async (e) => {
                    e.preventDefault();
                    e.stopPropagation();
                    
                    try {
                      console.log('[다운로드] 시작:', filename);
                      
                      const response = await fetch(href, {
                        method: 'GET',
                        headers: {
                          'X-API-Key': localStorage.getItem('apiKey') || 'airun_1_3d85009c98964579c622eded69997b16',
                          'Authorization': `Bearer ${localStorage.getItem('token') || ''}`
                        }
                      });
                      
                      if (!response.ok) {
                        throw new Error('파일을 찾을 수 없습니다.');
                      }
                      
                      const blob = await response.blob();
                      const url = window.URL.createObjectURL(blob);
                      const a = document.createElement('a');
                      a.href = url;
                      a.download = filename;
                      document.body.appendChild(a);
                      a.click();
                      window.URL.revokeObjectURL(url);
                      document.body.removeChild(a);
                      
                      console.log('[다운로드] 완료:', filename);
                    } catch (error) {
                      console.error('다운로드 오류:', error);
                      alert('다운로드에 실패했습니다: ' + (error instanceof Error ? error.message : String(error)));
                    }
                  }}
                  className="text-blue-600 dark:text-blue-400 hover:underline cursor-pointer"
                  title={`파일 다운로드: ${filename}`}
                >
                  {children}
                </button>
              );
            }
            
            // 출처 섹션의 링크인지 확인 (링크 텍스트나 URL로 판단)
            const linkText = String(React.Children.toArray(children || []).join('') || '').toLowerCase();
            const hrefString = String(href || '');
            const isReferenceLink = hasReferencesSection && (
              hrefString.includes('http') || 
              linkText.includes('pdf') ||
              linkText.includes('문서') ||
              linkText.includes('페이지') ||
              hrefString.includes('github') ||
              hrefString.includes('docs') ||
              linkText.length > 30 // 긴 제목의 링크는 출처일 가능성이 높음
            );
            
            if (isReferenceLink) {
              return (
                <a
                  href={href}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="text-blue-600 dark:text-blue-400 hover:underline text-sm break-words break-all"
                  style={{ wordBreak: 'break-all', overflowWrap: 'anywhere' }}
                >
                  {children}
                </a>
              );
            }
            
            return (
              <a
                href={href}
                target="_blank"
                rel="noopener noreferrer"
                className="text-blue-600 dark:text-blue-400 hover:underline break-words break-all"
                style={{ wordBreak: 'break-all', overflowWrap: 'anywhere' }}
              >
                {children}
              </a>
            );
          },
          // 리스트
          ul: (props: any) => (
            <ul className="list-disc list-inside my-2 space-y-1 ml-4">
              {props?.children}
            </ul>
          ),
          ol: (props: any) => (
            <ol className="list-decimal list-outside my-2 space-y-1 ml-10">
              {props?.children}
            </ol>
          ),
          li: (props: any) => {
            const children = props?.children;
            // 출처 섹션의 리스트 항목인지 확인 (더 정확한 감지)
            const childrenText = String(React.Children.toArray(children || []).join('') || '').toLowerCase();
            const isReferenceItem = hasReferencesSection && (
              childrenText.includes('.pdf') || 
              childrenText.includes('.md') || 
              childrenText.includes('.doc') ||
              childrenText.includes('.txt') ||
              childrenText.includes('페이지') ||
              childrenText.includes('page') ||
              childrenText.includes('http') ||
              /^\d+\.\s/.test(childrenText) // 숫자로 시작하는 리스트 (1. 2. 등)
            );
            
            if (isReferenceItem) {
              return (
                <li className="text-gray-600 dark:text-gray-400 pl-2 text-sm leading-relaxed">
                  {children}
                </li>
              );
            }
            
            return (
              <li className="text-gray-700 dark:text-gray-300 pl-2">
                {children}
              </li>
            );
          },
                     // 제목
           h1: (props: any) => {
             const children = props?.children;
             // 출처(References) 제목인지 확인
             const childrenText = String(React.Children.toArray(children || []).join('') || '').toLowerCase();
             const isReferences = childrenText.includes('출처') || childrenText.includes('references');
             
             if (isReferences) {
               return (
                 <h3 className="text-lg text-gray-900 dark:text-white my-3 flex items-center space-x-2 pt-8">
                   <svg className="w-5 h-5 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                     <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                   </svg>
                   <span>References</span>
                 </h3>
               );
             }
             
             return (
            <h1 className="text-sm text-gray-900 dark:text-white my-2">
                {children}
              </h1>
            );
           },
           h2: (props: any) => (
             <h2 className="text-sm text-gray-900 dark:text-white mt-6 my-2">
               {props?.children}
             </h2>
           ),
           h3: (props: any) => (
             <h3 className="text-sm text-gray-900 dark:text-white mt-6 my-2">
               {props?.children}
             </h3>
           ),
           h4: (props: any) => (
             <h4 className="text-sm text-gray-900 dark:text-white mt-4 my-2">
               {props?.children}
             </h4>
           ),
           h5: (props: any) => (
             <h5 className="text-sm text-gray-900 dark:text-white mt-4 my-2">
               {props?.children}
             </h5>
           ),
           h6: (props: any) => (
             <h6 className="text-sm text-gray-900 dark:text-white mt-4 my-2">
               {props?.children}
             </h6>
           ),
          // 인용문
          blockquote: (props: any) => (
            <blockquote className="border-l-4 border-gray-300 dark:border-gray-600 pl-4 my-2 italic text-gray-600 dark:text-gray-400 ml-4">
              {props?.children}
            </blockquote>
          ),
          // 구분선
          hr: () => (
            <hr className="border-gray-300 dark:border-gray-600 my-4" />
          ),

          // 표
          table: (props: any) => (
            <div className="overflow-x-auto my-4 ml-4 pl-4">
              <table className="min-w-full border border-gray-300 dark:border-gray-600 text-sm">
                {props?.children}
              </table>
            </div>
          ),
          thead: (props: any) => (
            <thead className="bg-gray-50 dark:bg-gray-800">
              {props?.children}
            </thead>
          ),
          tbody: (props: any) => (
            <tbody className="bg-white dark:bg-gray-900">
              {props?.children}
            </tbody>
          ),
          tr: (props: any) => (
            <tr className="border-b border-gray-300 dark:border-gray-600">
              {props?.children}
            </tr>
          ),
          th: (props: any) => (
            <th className="px-4 py-2 text-left font-medium text-gray-900 dark:text-white">
              {props?.children}
            </th>
          ),
          td: (props: any) => (
            <td className="px-4 py-2 text-gray-700 dark:text-gray-300">
              {props?.children}
            </td>
          ),
          // 이미지 처리 - PNG 이미지만 처리 (SVG는 제외)
          img: (props: any) => {
            const { src, alt, title } = props || {};
            if (!src) return null;
            
            // src가 문자열인지 확인
            if (typeof src !== 'string') return null;
            
            // PNG 이미지만 처리하고 SVG는 제외
            const isPngImage = src.toLowerCase().includes('.png');
            const isSvgImage = src.toLowerCase().includes('.svg');
            
            // SVG 이미지는 렌더링하지 않음 (PNG 버전이 있다고 가정)
            if (isSvgImage) {
              return null;
            }
            
            // PNG 이미지가 아닌 경우에도 일반적인 이미지 처리 (jpg, jpeg, gif 등)
            return (
              <img
                src={src}
                alt={alt || '이미지'}
                title={title || alt || '이미지'}
                className="block max-w-full h-auto ml-4 my-4 rounded-lg border border-gray-200 dark:border-gray-700 shadow-sm hover:shadow-md transition-shadow"
                onError={(e) => {
                  console.error('[이미지 로딩 오류]', {
                    src,
                    alt,
                    error: e
                  });
                  
                  const target = e.target as HTMLImageElement;
                  
                  // 한 번 더 재시도 (서버에서 해시 복구 시도)
                  if (!target.dataset.retried) {
                    target.dataset.retried = 'true';
                    console.log('[이미지 재시도]', src);
                    
                    // 1초 후 재시도
                    setTimeout(() => {
                      target.src = src + '&retry=' + Date.now();
                    }, 1000);
                    return;
                  }
                  
                  // 재시도 후에도 실패하면 오류 표시
                  target.style.display = 'none';
                  
                  // 새로운 오류 표시 요소 생성
                  const errorDiv = document.createElement('div');
                  errorDiv.className = 'flex items-center justify-center p-4 bg-gray-100 dark:bg-gray-800 rounded-lg border border-gray-300 dark:border-gray-600 my-4 ml-4';
                  errorDiv.innerHTML = `
                    <div class="text-center">
                      <svg class="w-8 h-8 mx-auto mb-2 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"></path>
                      </svg>
                      <p class="text-sm text-gray-500 dark:text-gray-400">이미지를 불러올 수 없습니다</p>
                      <p class="text-xs text-gray-400 dark:text-gray-500 mt-1">문서를 다시 생성해주세요</p>
                      <p class="text-xs text-gray-400 dark:text-gray-500 mt-1 break-all">${src}</p>
                    </div>
                  `;
                  
                  // 이미지 요소를 오류 표시 요소로 교체
                  target.parentNode?.replaceChild(errorDiv, target);
                }}
              />
            );
          },
                     // 문단 처리 - 이미지가 포함된 경우 별도 처리
           p: (props: any) => {
             const children = props?.children;
             // 문단 안에 이미지만 있는 경우 처리
             const hasOnlyImage = React.Children.count(children) === 1 && 
               React.Children.toArray(children || []).some(child => 
                 React.isValidElement(child) && child.type === 'img'
               );
             
             if (hasOnlyImage) {
               // 이미지만 있는 문단은 div로 감싸서 블록 레벨로 처리
               return <div className="my-4 ml-4">{children}</div>;
             }
             
             return (
               <p className="my-3 leading-loose ml-4 text-sm">
                 {children}
               </p>
             );
           },
        }}
      >
        {safeContent}
      </ReactMarkdown>
    </div>
  );
}; 