'use client';

import React, { useCallback, useRef } from 'react';
import ReactFlow, {
  Node,
  Edge,
  Connection,
  useNodesState,
  useEdgesState,
  Controls,
  Background,
  BackgroundVariant,
  NodeTypes,
  EdgeTypes,
  ReactFlowProvider,
  useReactFlow,
  MarkerType,
} from 'reactflow';
import 'reactflow/dist/style.css';

import type { NodeData, Connection as FlowConnection, NodeType } from '@/types/flowai';
import CustomNode from './CustomNode';
import CustomEdge from './CustomEdge';

interface FlowCanvasProps {
  nodes: NodeData[];
  connections: FlowConnection[];
  selectedNode: NodeData | null;
  onNodeSelect: (node: NodeData | null) => void;
  onNodeUpdate: (nodeId: string, updates: Partial<NodeData>) => void;
  onNodeDelete: (nodeId: string) => void;
  onNodeExecute?: (nodeId: string) => void;
  onNodeStop?: (nodeId: string) => void;
  onNodeDuplicate?: (nodeId: string) => void;
  onConnectionAdd: (source: string, target: string, sourceHandle?: string, targetHandle?: string) => void;
  onConnectionDelete: (connectionId: string) => void;
  isDragging: boolean;
  draggedNodeType: NodeType | null;
  onDrop: (type: NodeType | null, position: { x: number; y: number }) => void;
}

const nodeTypes: NodeTypes = {
  custom: CustomNode,
};

const edgeTypes: EdgeTypes = {
  custom: CustomEdge,
};

// 내부 FlowCanvas 컴포넌트 (useReactFlow 사용)
function FlowCanvasInner({
  nodes,
  connections,
  selectedNode,
  onNodeSelect,
  onNodeUpdate,
  onNodeDelete,
  onNodeExecute,
  onNodeStop,
  onNodeDuplicate,
  onConnectionAdd,
  onConnectionDelete,
  isDragging,
  draggedNodeType,
  onDrop,
}: FlowCanvasProps) {
  
  const reactFlowWrapper = useRef(null);
  const { screenToFlowPosition } = useReactFlow();
  
  // NodeData를 React Flow Node로 변환 (memoized)
  const reactFlowNodes: Node[] = React.useMemo(() => nodes.map(node => ({
    id: node.id,
    type: 'custom',
    position: {
      x: typeof node.position?.x === 'number' && !isNaN(node.position.x) ? node.position.x : 0,
      y: typeof node.position?.y === 'number' && !isNaN(node.position.y) ? node.position.y : 0,
    },
    data: {
      ...node.data,
      label: node.data.label || node.label || `${node.type} 노드`,
      nodeType: node.type,
      selected: selectedNode?.id === node.id,
      isRunning: node.data.isRunning || false,
      hasError: node.data.hasError || false,
      onSelect: () => onNodeSelect(node),
      onUpdate: (updates: any) => onNodeUpdate(node.id, updates),
      onDelete: () => onNodeDelete(node.id),
      onExecute: () => onNodeExecute?.(node.id),
      onStop: () => onNodeStop?.(node.id),
      onDuplicate: () => onNodeDuplicate?.(node.id),
    },
  })), [nodes, selectedNode, onNodeSelect, onNodeUpdate, onNodeDelete, onNodeExecute, onNodeStop, onNodeDuplicate]);

  // FlowConnection을 React Flow Edge로 변환 (memoized)
  const reactFlowEdges: Edge[] = React.useMemo(() => connections.map(conn => ({
    id: conn.id,
    source: conn.source,
    target: conn.target,
    sourceHandle: conn.sourceHandle,
    targetHandle: conn.targetHandle,
    type: 'custom',
    animated: false,
    style: {
      stroke: 'var(--neutral-500)',
      strokeWidth: 2,
    },
    markerEnd: {
      type: MarkerType.ArrowClosed,
      color: 'var(--neutral-500)',
      width: 20,
      height: 20,
    },
    data: {
      onDelete: onConnectionDelete,
    },
  })), [connections, onConnectionDelete]);

  // ReactFlow 상태 초기화
  const [rfNodes, setNodes, onNodesChange] = useNodesState([]);
  const [rfEdges, setEdges, onEdgesChange] = useEdgesState([]);
  
  // 기존 노드 위치 보존을 위한 스마트 동기화
  const prevNodePositionsRef = useRef<Map<string, { x: number; y: number }>>(new Map());
  
  React.useEffect(() => {
    const currentPositions = new Map(
      rfNodes.map(node => [node.id, node.position])
    );
    
    // 기존 노드의 현재 위치를 보존하면서 데이터 업데이트
    const updatedNodes = reactFlowNodes.map(node => {
      const currentPosition = currentPositions.get(node.id);
      return currentPosition ? { ...node, position: currentPosition } : node;
    });
    
    setNodes(updatedNodes);
    setEdges(reactFlowEdges);
  }, [reactFlowNodes, reactFlowEdges]);

  // 연결 생성
  const onConnect = useCallback(
    (params: Connection) => {
      if (params.source && params.target) {
        // 핸들 정보를 포함하여 연결 추가
        onConnectionAdd(params.source, params.target, params.sourceHandle, params.targetHandle);
      }
    },
    [onConnectionAdd]
  );

  // 노드 위치 변경 처리
  const handleNodesChange = useCallback(
    (changes: any) => {
      onNodesChange(changes);
      
      // 드래그가 완전히 끝났을 때만 부모에게 알림
      const finalPositionChanges = changes.filter((change: any) => 
        change.type === 'position' && change.position && change.dragging === false
      );
      
      finalPositionChanges.forEach((change: any) => {
        onNodeUpdate(change.id, { position: change.position });
      });

      // 노드 선택 상태 변경 처리
      const selectionChanges = changes.filter((change: any) => 
        change.type === 'select'
      );
      
      selectionChanges.forEach((change: any) => {
        if (change.selected) {
          const node = nodes.find(n => n.id === change.id);
          if (node) {
            onNodeSelect(node);
          }
        } else {
          onNodeSelect(null);
        }
      });
    },
    [onNodesChange, onNodeUpdate, onNodeSelect, nodes]
  );

  // 엣지 삭제
  const handleEdgesChange = useCallback(
    (changes: any) => {
      onEdgesChange(changes);
      
      changes.forEach((change: any) => {
        if (change.type === 'remove') {
          onConnectionDelete(change.id);
        }
      });
    },
    [onEdgesChange, onConnectionDelete]
  );

  // React Flow 공식 패턴에 따른 드래그오버
  const onDragOver = useCallback((event: React.DragEvent) => {
    event.preventDefault();
    event.dataTransfer.dropEffect = 'move';
  }, []);

  // 간단한 예제와 동일한 드롭 핸들러
  const onDropHandler = useCallback(
    (event: React.DragEvent) => {
      event.preventDefault();
      
      const nodeType = event.dataTransfer.getData('application/reactflow');
      
      if (!nodeType) {
        console.log('🔥 No node type found');
        return;
      }

      const position = screenToFlowPosition({
        x: event.clientX,
        y: event.clientY,
      });

      console.log('✅ Adding node at position:', position);
      onDrop(nodeType as NodeType, position);
    },
    [screenToFlowPosition, onDrop]
  );

  // 키보드 이벤트 처리 (Delete/Backspace 키로 노드 삭제)
  const onKeyDown = useCallback(
    (event: React.KeyboardEvent) => {
      if (event.key === 'Delete' || event.key === 'Backspace') {
        const selectedNodes = rfNodes.filter(node => node.selected);
        selectedNodes.forEach(node => {
          onNodeDelete(node.id);
        });
      }
    },
    [rfNodes, onNodeDelete]
  );

  return (
    <div 
      className="w-full h-full" 
      ref={reactFlowWrapper}
      tabIndex={0}
      onKeyDown={onKeyDown}
      style={{ outline: 'none', backgroundColor: 'var(--body-bg)' }}
    >
      <ReactFlow
        nodes={rfNodes}
        edges={rfEdges}
        onNodesChange={handleNodesChange}
        onEdgesChange={handleEdgesChange}
        onConnect={onConnect}
        onDrop={onDropHandler}
        onDragOver={onDragOver}
        nodeTypes={nodeTypes}
        edgeTypes={edgeTypes}
        fitView={false}
        fitViewOptions={{ includeHiddenNodes: false }}
        preventScrolling={false}
        panOnDrag={true}
        panOnScroll={false}
        zoomOnScroll={true}
        zoomOnPinch={true}
        zoomOnDoubleClick={false}
        attributionPosition="bottom-left"
        minZoom={0.1}
        maxZoom={2}
        defaultViewport={{ x: 0, y: 0, zoom: 1 }}
        nodesDraggable={true}
        nodesConnectable={true}
        edgesUpdatable={false}
        edgesFocusable={true}
        elementsSelectable={true}
        elementsSelectable={true}
        selectNodesOnDrag={false}
        deleteKeyCode="Delete"
        onNodesDelete={(nodesToDelete) => {
          nodesToDelete.forEach(node => onNodeDelete(node.id));
        }}
        onEdgesDelete={(edgesToDelete) => {
          edgesToDelete.forEach(edge => onConnectionDelete(edge.id));
        }}
      >
        <Controls position="top-right" />
        <Background variant={BackgroundVariant.Dots} gap={20} size={1} />
      </ReactFlow>
    </div>
  );
}

// 외부 FlowCanvas 컴포넌트 (ReactFlowProvider 제공)
export default function FlowCanvas(props: FlowCanvasProps) {
  return (
    <ReactFlowProvider>
      <FlowCanvasInner {...props} />
    </ReactFlowProvider>
  );
}