#!/bin/bash

#############################################
# AI.RUN Installation Script
# Version: 2.0
# Structured Installation with 8 Phases
#############################################

set -e  # Exit on error

# Color codes
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Installation phases
TOTAL_PHASES=8
CURRENT_PHASE=0

# Progress indicator
show_phase() {
    CURRENT_PHASE=$1
    PHASE_NAME=$2
    echo ""
    echo -e "${BLUE}================================================================${NC}"
    echo -e "${BLUE}  PHASE $CURRENT_PHASE/$TOTAL_PHASES: $PHASE_NAME${NC}"
    echo -e "${BLUE}================================================================${NC}"
    echo ""
}

# Error handler
handle_error() {
    echo -e "${RED}❌ 오류 발생: $1${NC}"
    echo -e "${RED}설치를 중단합니다.${NC}"
    exit 1
}

# Success message
show_success() {
    echo -e "${GREEN}✅ $1${NC}"
}

# Warning message
show_warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

# Info message
show_info() {
    echo -e "${BLUE}ℹ️  $1${NC}"
}

#############################################
# INITIAL SETUP
#############################################

echo ""
echo -e "${GREEN}╔═══════════════════════════════════════╗${NC}"
echo -e "${GREEN}║     AI.RUN 설치 스크립트 v2.0        ║${NC}"
echo -e "${GREEN}║     Structured Installation           ║${NC}"
echo -e "${GREEN}╚═══════════════════════════════════════╝${NC}"
echo ""

# Request sudo password upfront
echo -e "${YELLOW}설치를 위해 sudo 권한이 필요합니다.${NC}"
sudo -v || handle_error "sudo 권한 획득 실패"

# Keep sudo alive
while true; do sudo -n true; sleep 60; kill -0 "$$" || exit; done 2>/dev/null &

# Set variables
CURRENT_USER=$(whoami)
CURRENT_DIR="$(cd "$(dirname "$0")" && pwd)"
VENV_PATH="$HOME/.airun_venv"
AIRUN_DIR="$HOME/.airun"

#############################################
# PHASE 1: CLEANUP & PREPARATION
#############################################
show_phase 1 "기존 환경 정리 및 준비"

# Stop existing services
show_info "기존 서비스 중지 중..."
for service in airun-api airun-rag airun-websearch airun-report airun-biz airun-llm-finetune airun-vllm; do
    if systemctl list-unit-files 2>/dev/null | grep -q "^$service.service" || [ -f "/etc/systemd/system/$service.service" ]; then
        show_warning "$service 서비스 중지 및 제거..."
        sudo systemctl stop $service 2>/dev/null || true
        sudo systemctl disable $service 2>/dev/null || true
        sudo rm -f /etc/systemd/system/$service.service 2>/dev/null || true
    fi
done

# Remove timers
for timer in airun-biz-daily-collection; do
    if [ -f "/etc/systemd/system/$timer.timer" ]; then
        show_warning "$timer 타이머 제거..."
        sudo systemctl stop $timer.timer 2>/dev/null || true
        sudo systemctl disable $timer.timer 2>/dev/null || true
        sudo rm -f /etc/systemd/system/$timer.timer 2>/dev/null || true
        sudo rm -f /etc/systemd/system/$timer.service 2>/dev/null || true
    fi
done

sudo systemctl daemon-reload 2>/dev/null || true
show_success "systemd 서비스 정리 완료"

# Clean Redis
show_info "Redis 데이터 초기화 중..."
if command -v redis-cli &>/dev/null; then
    if redis-cli ping 2>&1 | grep -q "PONG"; then
        redis-cli FLUSHALL 2>/dev/null || true
        show_success "Redis 데이터 초기화 완료"
    else
        show_warning "Redis 서버가 실행 중이지 않음"
    fi
fi

# Clean directories (preserve PostgreSQL data)
show_info "디렉토리 정리 중..."
if [ -d "$AIRUN_DIR" ]; then
    # Backup PostgreSQL data (both possible directory names)
    if [ -d "$AIRUN_DIR/pgdata" ]; then
        show_info "PostgreSQL 데이터 백업 중 (pgdata)..."
        sudo mv "$AIRUN_DIR/pgdata" "/tmp/airun-pgdata-backup" 2>/dev/null || true
    elif [ -d "$AIRUN_DIR/postgres-data" ]; then
        show_info "PostgreSQL 데이터 백업 중 (postgres-data)..."
        sudo mv "$AIRUN_DIR/postgres-data" "/tmp/airun-pgdata-backup" 2>/dev/null || true
    fi

    # Clean other directories but keep logs and cache structure
    find "$AIRUN_DIR" -mindepth 1 -maxdepth 1 ! -name 'pgdata' ! -name 'postgres-data' -exec rm -rf {} +

    # Restore PostgreSQL data
    if [ -d "/tmp/airun-pgdata-backup" ]; then
        mkdir -p "$AIRUN_DIR"
        sudo mv "/tmp/airun-pgdata-backup" "$AIRUN_DIR/pgdata"
        sudo chown -R $USER:$USER "$AIRUN_DIR/pgdata"
        show_success "PostgreSQL 데이터 복원 완료"
    fi
else
    mkdir -p "$AIRUN_DIR"
fi

# Remove Python virtual environment
if [ -d "$VENV_PATH" ]; then
    show_info "기존 Python 가상환경 제거 중..."
    rm -rf "$VENV_PATH"
fi

# Remove node_modules and package-lock
if [ -f "package-lock.json" ]; then
    rm -f package-lock.json
fi
if [ -d "node_modules" ]; then
    show_info "node_modules 제거 중..."
    rm -rf node_modules
fi

# Clean logs
sudo rm -rf /var/log/airun 2>/dev/null || true

show_success "PHASE 1 완료: 환경 정리 완료"

#############################################
# PHASE 2: SYSTEM DEPENDENCIES
#############################################
show_phase 2 "시스템 의존성 설치"

# Detect OS
if [ -f /etc/os-release ]; then
    . /etc/os-release
    OS=$NAME
    VER=$VERSION_ID
fi

show_info "운영체제: $OS $VER"

# Install system packages
if command -v apt-get &>/dev/null; then
    show_info "APT 패키지 업데이트 중..."
    sudo apt-get update -qq

    show_info "필수 시스템 패키지 설치 중..."

    # Install packages one by one to avoid conflicts
    PACKAGES=(
        "build-essential"
        "python3-dev"
        "python3-pip"
        "python3-venv"
        "git"
        "curl"
        "wget"
        "redis-server"
        "postgresql-client"
        "libpq-dev"
        "tesseract-ocr"
        "tesseract-ocr-kor"
        "poppler-utils"
        "libmagic1"
        "ffmpeg"
        "libssl-dev"
        "libffi-dev"
        "libxml2-dev"
        "libxslt1-dev"
        "libjpeg-dev"
        "zlib1g-dev"
    )

    for package in "${PACKAGES[@]}"; do
        if ! dpkg -l | grep -q "^ii.*$package"; then
            show_info "Installing $package..."
            sudo apt-get install -y "$package" 2>/dev/null || show_warning "Could not install $package (optional)"
        fi
    done

    # Docker installation (separate handling)
    if ! command -v docker &>/dev/null; then
        show_info "Docker 설치 중..."
        # Check if docker.io can be installed
        if sudo apt-get install -y docker.io 2>/dev/null; then
            show_success "Docker installed via docker.io"
        else
            # Try Docker CE if docker.io fails
            show_info "Installing Docker CE..."
            curl -fsSL https://get.docker.com -o get-docker.sh
            sudo sh get-docker.sh
            rm get-docker.sh
        fi
    fi

    # Install docker-compose separately
    if ! command -v docker-compose &>/dev/null; then
        show_info "Docker Compose 설치 중..."
        sudo apt-get install -y docker-compose 2>/dev/null || {
            # Alternative: install via pip
            "$VENV_PATH/bin/pip" install docker-compose || show_warning "Docker Compose 설치 실패 (선택적)"
        }
    fi

    # Chrome/Chromium handling (don't fail if not available)
    show_info "Chrome/Chromium 확인 중..."
    if command -v google-chrome &>/dev/null; then
        show_success "Google Chrome이 이미 설치되어 있습니다"
    elif command -v chromium &>/dev/null; then
        show_success "Chromium이 이미 설치되어 있습니다"
    else
        # Try to install chromium (non-snap version if available)
        if sudo apt-get install -y chromium 2>/dev/null; then
            show_success "Chromium 설치 완료"
        else
            show_warning "Chrome/Chromium을 설치할 수 없습니다. Puppeteer가 자동으로 다운로드합니다."
        fi
    fi

elif command -v yum &>/dev/null; then
    show_info "YUM 패키지 설치 중..."
    sudo yum install -y \
        gcc \
        gcc-c++ \
        python3-devel \
        python3-pip \
        git \
        curl \
        wget \
        redis \
        postgresql \
        docker \
        chromium \
        tesseract \
        poppler-utils \
        file-libs \
        ffmpeg \
        || handle_error "시스템 패키지 설치 실패"
fi

show_success "시스템 패키지 설치 완료"

# Docker setup
show_info "Docker 설정 중..."
if command -v docker &>/dev/null; then
    show_success "Docker가 이미 설치되어 있습니다"
    sudo systemctl start docker 2>/dev/null || true
    sudo systemctl enable docker 2>/dev/null || true

    # 사용자를 docker 그룹에 추가
    if ! groups $USER | grep -q docker; then
        show_info "사용자를 docker 그룹에 추가 중..."
        sudo usermod -aG docker $USER 2>/dev/null || true
        show_warning "Docker 그룹 권한은 재로그인 후 적용됩니다"
    else
        show_success "사용자가 이미 docker 그룹에 속해 있습니다"
    fi
else
    show_warning "Docker가 설치되지 않았습니다. Docker를 수동으로 설치해주세요."
    show_info "Docker 설치 방법: https://docs.docker.com/engine/install/"
    handle_error "Docker가 필수 요구사항입니다"
fi
show_success "Docker 설정 완료"

# PostgreSQL Docker setup
show_info "PostgreSQL Docker 컨테이너 설정 중..."
CONTAINER_NAME="airun_postgres"

# Clean up any existing PostgreSQL containers on port 5433
show_info "기존 PostgreSQL 컨테이너 정리 중..."

# Get all containers using port 5433
EXISTING_CONTAINERS=$(sudo docker ps -a --format "{{.Names}}" --filter "publish=5433" 2>/dev/null || true)
if [ ! -z "$EXISTING_CONTAINERS" ]; then
    for container in $EXISTING_CONTAINERS; do
        show_info "기존 컨테이너 $container 제거 중..."
        sudo docker stop "$container" 2>/dev/null || true
        sudo docker rm "$container" 2>/dev/null || true
    done
fi

# Also check by name pattern
for pattern in "airun" "postgres"; do
    CONTAINERS=$(sudo docker ps -a --format "{{.Names}}" | grep -i "$pattern" | grep -v "^$CONTAINER_NAME$" || true)
    for container in $CONTAINERS; do
        if sudo docker port "$container" 2>/dev/null | grep -q "5433"; then
            show_info "컨테이너 $container 제거 중 (포트 5433 사용)..."
            sudo docker stop "$container" 2>/dev/null || true
            sudo docker rm "$container" 2>/dev/null || true
        fi
    done
done

# Check if our container exists and is running properly
CONTAINER_EXISTS=false
CONTAINER_RUNNING=false

if sudo docker ps -a --format "{{.Names}}" | grep -q "^$CONTAINER_NAME$"; then
    CONTAINER_EXISTS=true
    if sudo docker ps --format "{{.Names}}" | grep -q "^$CONTAINER_NAME$"; then
        CONTAINER_RUNNING=true
    fi
fi

# Handle container state
if [ "$CONTAINER_RUNNING" = true ]; then
    show_success "PostgreSQL 컨테이너가 이미 실행 중입니다"
elif [ "$CONTAINER_EXISTS" = true ]; then
    show_info "PostgreSQL 컨테이너 재시작 중..."
    sudo docker start "$CONTAINER_NAME" || {
        show_warning "재시작 실패. 컨테이너 재생성..."
        sudo docker rm "$CONTAINER_NAME" 2>/dev/null || true
        CONTAINER_EXISTS=false
    }
fi

# Create new container if needed
if [ "$CONTAINER_EXISTS" = false ]; then
    show_info "PostgreSQL Docker 컨테이너 생성 중..."

    # Create data directory if it doesn't exist
    mkdir -p "$AIRUN_DIR/pgdata"

    sudo docker run --name "$CONTAINER_NAME" \
        -e POSTGRES_USER=ivs \
        -e POSTGRES_PASSWORD=1234 \
        -e POSTGRES_DB=airun \
        -p 5433:5432 \
        -v "$AIRUN_DIR/pgdata:/var/lib/postgresql/data" \
        --restart always \
        -d pgvector/pgvector:pg17 \
        || handle_error "PostgreSQL Docker 실행 실패"

    show_success "PostgreSQL 컨테이너가 생성되었습니다"
fi

# Wait for PostgreSQL to be ready
show_info "PostgreSQL 시작 대기 중..."
for i in {1..30}; do
    if sudo docker exec "$CONTAINER_NAME" pg_isready -h localhost -U ivs >/dev/null 2>&1; then
        show_success "PostgreSQL 준비 완료"

        # Enable pgvector extension
        sudo docker exec "$CONTAINER_NAME" psql -U ivs -d airun -c "CREATE EXTENSION IF NOT EXISTS vector;" 2>/dev/null || true
        show_success "pgvector 확장 활성화"

        # Initialize database schema by running createTables function
        show_info "데이터베이스 스키마 초기화 중..."
        cd "$CURRENT_DIR"
        if timeout 60 node -e "
            process.env.DB_HOST = 'localhost';
            process.env.DB_PORT = '5433';
            process.env.DB_USER = 'ivs';
            process.env.DB_PASSWORD = '1234';
            process.env.DB_NAME = 'airun';

            const { createTables } = require('./services/auth/schema.js');
            (async () => {
                try {
                    await createTables();
                    console.log('데이터베이스 스키마 초기화 완료');
                    process.exit(0);
                } catch (error) {
                    console.error('데이터베이스 스키마 초기화 실패:', error.message);
                    process.exit(1);
                }
            })();
        " 2>/dev/null; then
            show_success "데이터베이스 스키마 초기화 완료"
        else
            show_warning "데이터베이스 스키마 초기화 실패 (API 서버 시작 시 자동 실행됨)"
        fi
        break
    fi

    if [ $i -eq 30 ]; then
        show_warning "PostgreSQL 시작 대기 시간 초과. 수동으로 확인이 필요할 수 있습니다."
    else
        echo -n "."
        sleep 2
    fi
done

# Redis setup
show_info "Redis 설정 중..."
sudo systemctl start redis-server || sudo systemctl start redis || true
sudo systemctl enable redis-server || sudo systemctl enable redis || true

if redis-cli ping 2>&1 | grep -q "PONG"; then
    show_success "Redis 서버 정상 작동"
else
    show_warning "Redis 서버 시작 실패 (선택적)"
fi

show_success "PHASE 2 완료: 시스템 의존성 설치 완료"

#############################################
# PHASE 3: DEVELOPMENT TOOLS
#############################################
show_phase 3 "개발 도구 설치"

# Install nvm and Node.js
show_info "Node.js 설치 중..."
if [ ! -d "$HOME/.nvm" ]; then
    curl -o- https://raw.githubusercontent.com/nvm-sh/nvm/v0.40.1/install.sh | bash
fi

export NVM_DIR="$HOME/.nvm"
[ -s "$NVM_DIR/nvm.sh" ] && \. "$NVM_DIR/nvm.sh"

nvm install 22.18.0
nvm use 22.18.0
nvm alias default 22.18.0

NODE_VERSION=$(node -v)
show_success "Node.js 설치 완료: $NODE_VERSION"

# Update npm
show_info "npm 업데이트 중..."
npm install -g npm@latest

NPM_VERSION=$(npm -v)
NPM_MAJOR=$(echo $NPM_VERSION | cut -d. -f1)

if [ "$NPM_MAJOR" -lt "11" ]; then
    npm install -g npm@11
    NPM_VERSION=$(npm -v)
fi
show_success "npm 버전: $NPM_VERSION"

# Install global npm packages
show_info "전역 npm 패키지 설치 중..."
npm install -g pm2 ws puppeteer @mermaid-js/mermaid-cli || true
show_success "전역 npm 패키지 설치 완료"

# Check Python
show_info "Python 버전 확인 중..."
if ! command -v python3 &>/dev/null; then
    handle_error "Python3가 설치되지 않았습니다"
fi

PYTHON_VERSION=$(python3 --version | awk '{print $2}')
PYTHON_MAJOR=$(echo $PYTHON_VERSION | cut -d. -f1)
PYTHON_MINOR=$(echo $PYTHON_VERSION | cut -d. -f2)

if [ "$PYTHON_MAJOR" != "3" ] || [ "$PYTHON_MINOR" -lt "10" ]; then
    handle_error "Python 3.10 이상이 필요합니다. 현재: $PYTHON_VERSION"
fi
show_success "Python 버전: $PYTHON_VERSION"

show_success "PHASE 3 완료: 개발 도구 설치 완료"

#############################################
# PHASE 4: PYTHON ENVIRONMENT
#############################################
show_phase 4 "Python 환경 설정"

# Create virtual environment
show_info "Python 가상환경 생성 중..."
python3 -m venv "$VENV_PATH" || handle_error "가상환경 생성 실패"
show_success "가상환경 생성 완료: $VENV_PATH"

# Upgrade pip
show_info "pip 업그레이드 중..."
"$VENV_PATH/bin/python" -m pip install --upgrade pip

# Install system requirements first (includes Cython)
show_info "시스템 Python 패키지 설치 중..."
if [ -f "requirement-system.txt" ]; then
    "$VENV_PATH/bin/pip" install -r requirement-system.txt \
        || show_warning "일부 시스템 패키지 설치 실패 (계속 진행)"
    show_success "시스템 Python 패키지 설치 완료"
else
    # If requirement-system.txt doesn't exist, install essential packages
    show_info "필수 Python 빌드 패키지 설치 중..."
    "$VENV_PATH/bin/pip" install Cython numpy setuptools wheel \
        || show_warning "일부 빌드 패키지 설치 실패"
fi

# Install requirements with PyTorch CUDA 12.9
show_info "Python 패키지 설치 중 (PyTorch CUDA 12.9 포함)..."
if [ -f "requirements-linux.txt" ]; then
    "$VENV_PATH/bin/pip" install -r requirements-linux.txt \
        --index-url https://download.pytorch.org/whl/cu129 \
        --extra-index-url https://pypi.org/simple \
        || handle_error "requirements 설치 실패"

    show_success "Python 패키지 설치 완료"

    # Verify PyTorch installation
    if "$VENV_PATH/bin/python" -c "import torch; print(f'PyTorch {torch.__version__} - CUDA: {torch.cuda.is_available()}')" 2>/dev/null; then
        TORCH_INFO=$("$VENV_PATH/bin/python" -c "import torch; print(f'PyTorch {torch.__version__} - CUDA: {torch.cuda.is_available()}')")
        show_success "$TORCH_INFO"
    else
        show_warning "PyTorch 설치 확인 실패"
    fi
else
    handle_error "requirements-linux.txt 파일을 찾을 수 없습니다"
fi

# python-docx는 requirements-linux.txt에 이미 포함되어 있음

# Ensure sentence-transformers is updated for PyTorch 2.8
"$VENV_PATH/bin/pip" install --upgrade sentence-transformers || true

show_success "PHASE 4 완료: Python 환경 설정 완료"

#############################################
# PHASE 5: AI MODELS DOWNLOAD
#############################################
show_phase 5 "AI 모델 다운로드"

# Create directories
mkdir -p "$AIRUN_DIR/models"
mkdir -p "$AIRUN_DIR/cache"
mkdir -p "$AIRUN_DIR/logs"
mkdir -p "$AIRUN_DIR/queues"

# Download text embedding model
show_info "[1/3] 텍스트 임베딩 모델 다운로드 중..."
MODEL_PATH="$AIRUN_DIR/models/models--nlpai-lab--KURE-v1"
if [ ! -d "$MODEL_PATH" ]; then
    show_info "Hugging Face Hub를 통해 다운로드 중..."
    "$VENV_PATH/bin/python" -c "
from huggingface_hub import snapshot_download
import os
try:
    cache_dir = os.path.expanduser('~/.airun/models')
    snapshot_download('nlpai-lab/KURE-v1', cache_dir=cache_dir, ignore_patterns=['*.bin'])
    print('✅ 텍스트 임베딩 모델 다운로드 완료')
except Exception as e:
    print(f'⚠️ 다운로드 실패: {e}')
" || show_warning "텍스트 임베딩 모델 다운로드 실패 (선택적)"
else
    show_info "텍스트 임베딩 모델이 이미 존재함"
fi

# Download semantic model
show_info "[2/3] 시맨틱 청킹 모델 다운로드 중..."
SEMANTIC_PATH="$AIRUN_DIR/models/models--snunlp--KR-SBERT-V40K-klueNLI-augSTS"
if [ ! -d "$SEMANTIC_PATH" ]; then
    show_info "Hugging Face Hub를 통해 다운로드 중..."
    "$VENV_PATH/bin/python" -c "
from huggingface_hub import snapshot_download
import os
try:
    cache_dir = os.path.expanduser('~/.airun/models')
    # Download only safetensors format to avoid torch.load issue
    snapshot_download('snunlp/KR-SBERT-V40K-klueNLI-augSTS', cache_dir=cache_dir, ignore_patterns=['*.bin'])
    print('✅ 시맨틱 청킹 모델 다운로드 완료')
except Exception as e:
    print(f'⚠️ 다운로드 실패: {e}')
" || show_warning "시맨틱 모델 다운로드 실패 (선택적)"
else
    show_info "시맨틱 모델이 이미 존재함"
fi

# Download CLIP model
show_info "[3/3] 이미지 임베딩 모델 다운로드 중..."
IMAGE_PATH="$AIRUN_DIR/models/models--Bingsu--clip-vit-base-patch32-ko"
if [ ! -d "$IMAGE_PATH" ]; then
    show_info "Hugging Face Hub를 통해 다운로드 중..."
    "$VENV_PATH/bin/python" -c "
from huggingface_hub import snapshot_download
import os
try:
    cache_dir = os.path.expanduser('~/.airun/models')
    # Download only safetensors format
    snapshot_download('Bingsu/clip-vit-base-patch32-ko', cache_dir=cache_dir, ignore_patterns=['*.bin'])
    print('✅ 이미지 임베딩 모델 다운로드 완료')
except Exception as e:
    print(f'⚠️ 다운로드 실패: {e} (선택적 모델)')
" || show_warning "이미지 모델 다운로드 실패 (선택적)"
else
    show_info "이미지 모델이 이미 존재함"
fi

show_success "PHASE 5 완료: AI 모델 다운로드 완료"

#############################################
# PHASE 6: CONFIGURATION FILES
#############################################
show_phase 6 "설정 파일 생성"

# Create .env file
show_info ".env 파일 설정 중..."
ENV_FILE="$CURRENT_DIR/.env"
ENV_EXAMPLE="$CURRENT_DIR/.env.example"

if [ ! -f "$ENV_FILE" ]; then
    if [ -f "$ENV_EXAMPLE" ]; then
        cp "$ENV_EXAMPLE" "$ENV_FILE"
    else
        touch "$ENV_FILE"
    fi
fi

# Update database settings
if grep -q "^PG_HOST=" "$ENV_FILE" 2>/dev/null; then
    sed -i "s/^PG_HOST=.*/PG_HOST=localhost/" "$ENV_FILE"
    sed -i "s/^PG_PORT=.*/PG_PORT=5433/" "$ENV_FILE"
    sed -i "s/^PG_USER=.*/PG_USER=ivs/" "$ENV_FILE"
    sed -i "s/^PG_PASSWORD=.*/PG_PASSWORD=1234/" "$ENV_FILE"
    sed -i "s/^PG_DATABASE=.*/PG_DATABASE=airun/" "$ENV_FILE"
else
    cat >> "$ENV_FILE" << EOF

# Database Configuration
PG_HOST=localhost
PG_PORT=5433
PG_USER=ivs
PG_PASSWORD=1234
PG_DATABASE=airun

# Redis Configuration
REDIS_HOST=localhost
REDIS_PORT=6379

# Log Level
LOG_LEVEL=INFO
DEBUG=false
EOF
fi
show_success ".env 파일 설정 완료"

# Create airun.conf
show_info "airun.conf 생성 중..."
AIRUN_CONF="$AIRUN_DIR/airun.conf"
cat > "$AIRUN_CONF" << EOF
# AI.RUN Configuration
export LOG_LEVEL=INFO
export DEBUG=false
export USE_RAG=true
export USE_WEB_SEARCH=true
export RAG_TOP_K=5
export RAG_MODE=fast
EOF
show_success "airun.conf 생성 완료"

# Clean npm cache and configuration
show_info "npm 캐시 정리 및 설정 중..."
rm -rf node_modules package-lock.json 2>/dev/null || true
npm cache clean --force 2>/dev/null || true

# Change npm configuration
npm config set ignore-scripts false
npm config set legacy-peer-deps true

# Install local npm packages
show_info "로컬 npm 패키지 설치 중..."
npm install --legacy-peer-deps || handle_error "npm 패키지 설치 실패"
show_success "npm 패키지 설치 완료"

# Create global airun command link
show_info "전역 airun 명령어 링크 생성 중..."

# Ensure index.js is executable BEFORE npm link
chmod +x "$CURRENT_DIR/index.js" || handle_error "index.js 실행권한 설정 실패"

# Remove existing global links if they exist
AIRUN_PATH=$(which airun 2>/dev/null || echo "")
if [ -n "$AIRUN_PATH" ]; then
    show_warning "기존 airun 링크 발견: $AIRUN_PATH"
    # Remove existing npm global links
    npm unlink airun 2>/dev/null || true
    npm uninstall -g airun 2>/dev/null || true
fi

# Create npm global link (after setting permissions)
cd "$CURRENT_DIR"
npm link || handle_error "npm link 실패"
npm link electron 2>/dev/null || show_warning "electron link 실패 (선택적)"

# Verify the link was created
LINK_PATH=$(which airun 2>/dev/null || echo "")
if [ -n "$LINK_PATH" ]; then
    if [ -L "$LINK_PATH" ]; then
        LINK_TARGET=$(readlink -f "$LINK_PATH" 2>/dev/null || readlink "$LINK_PATH")
        if [ "$LINK_TARGET" = "$CURRENT_DIR/index.js" ]; then
            show_success "전역 airun 명령어 링크 생성 완료: $LINK_PATH"

            # Ensure proper permissions for the target file
            chmod +x "$CURRENT_DIR/index.js" 2>/dev/null || true

            # Ensure the current user can access the directory
            chmod 755 "$CURRENT_DIR" 2>/dev/null || true

            # If link is in system path, note but don't require sudo
            if [[ "$LINK_PATH" == /usr/local/bin/* ]]; then
                show_info "시스템 경로에 링크됨: $LINK_PATH (관리자 권한으로 설치됨)"
            fi

        else
            show_warning "링크가 올바른 경로를 가리키지 않음 (예상: $CURRENT_DIR/index.js, 실제: $LINK_TARGET)"
        fi
    else
        show_success "전역 airun 명령어 설치 완료: $LINK_PATH"
        # Ensure binary is executable
        chmod +x "$LINK_PATH" 2>/dev/null || true
    fi

    # Test if the command is accessible
    if command -v airun >/dev/null 2>&1; then
        show_success "airun 명령어 접근 가능 확인됨"
    else
        show_warning "airun 명령어 접근 불가 - PATH 환경변수 확인 필요"

        # Add to PATH if needed
        NPM_PREFIX=$(npm config get prefix 2>/dev/null || echo "/usr/local")
        NPM_BIN_PATH="$NPM_PREFIX/bin"

        if [[ ":$PATH:" != *":$NPM_BIN_PATH:"* ]]; then
            show_info "PATH에 $NPM_BIN_PATH 추가 중..."
            echo 'export PATH="'$NPM_BIN_PATH':$PATH"' >> ~/.bashrc
            echo 'export PATH="'$NPM_BIN_PATH':$PATH"' >> ~/.zshrc 2>/dev/null || true
            show_success "PATH 설정 완료 (새 터미널에서 적용)"
        fi
    fi
else
    show_warning "airun 명령어 링크 확인 실패 (수동 확인 필요)"
fi

show_success "PHASE 6 완료: 설정 파일 생성 완료"

#############################################
# PHASE 7: SERVICE CONFIGURATION
#############################################
show_phase 7 "서비스 설정"

show_info "systemd 서비스 파일 생성 중..."

# Create log directory
sudo mkdir -p /var/log/airun
sudo chown $CURRENT_USER:$CURRENT_USER /var/log/airun
sudo chmod 755 /var/log/airun

# API Server Service
cat > /tmp/airun-api.service << EOF
[Unit]
Description=Airun API Server
After=network.target redis.service

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
WorkingDirectory=$CURRENT_DIR
Environment=NODE_ENV=production
Environment=PATH=/usr/local/bin:/usr/bin:/bin:$HOME/.nvm/versions/node/v22.18.0/bin
Environment=AIRUN_PATH=$CURRENT_DIR
Environment=HOME=$HOME
Environment=NODE_PATH=$CURRENT_DIR/node_modules
Environment=PYTHONPATH=$CURRENT_DIR
ExecStart=/usr/bin/env node api_server.js
Restart=always
RestartSec=3
StandardOutput=append:/var/log/airun/airun-api.log
StandardError=append:/var/log/airun/airun-api.error.log

[Install]
WantedBy=multi-user.target
EOF

# RAG Server Service
cat > /tmp/airun-rag.service << EOF
[Unit]
Description=AI.RUN RAG Watch Service
After=network.target

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
RuntimeDirectory=airun
RuntimeDirectoryMode=0755
WorkingDirectory=$CURRENT_DIR
Environment=PYTHONPATH=$CURRENT_DIR
Environment=PYTHONUNBUFFERED=1
Environment=HOME=$HOME
Environment=AIRUN_LOG_DIR=/var/log/airun
ExecStart=$HOME/.airun_venv/bin/python3 -u $CURRENT_DIR/plugins/rag/airun-rag.py
ExecStop=/bin/kill -TERM \$MAINPID
Restart=always
RestartSec=3
StandardOutput=append:/var/log/airun/airun-rag.log
StandardError=append:/var/log/airun/airun-rag.error.log
ProtectSystem=full
NoNewPrivileges=true
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF

# WebSearch Service
cat > /tmp/airun-websearch.service << EOF
[Unit]
Description=AI.RUN WebSearch Server (Port 5610)
After=network.target

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
WorkingDirectory=$CURRENT_DIR/services/websearch
Environment=PYTHONPATH=$CURRENT_DIR
Environment=PYTHONUNBUFFERED=1
Environment=HOME=$HOME
Environment=WEBSEARCH_SERVER_HOST=0.0.0.0
Environment=WEBSEARCH_SERVER_PORT=5610
Environment=WEBSEARCH_SERVER_WORKERS=1
ExecStart=$HOME/.airun_venv/bin/python3 -u airun-websearch-server.py
ExecStop=/bin/kill -TERM \$MAINPID
Restart=always
RestartSec=3
StandardOutput=append:/var/log/airun/airun-websearch.log
StandardError=append:/var/log/airun/airun-websearch.error.log
ProtectSystem=full
NoNewPrivileges=true
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF

# Report Service
cat > /tmp/airun-report.service << EOF
[Unit]
Description=AI.RUN Report Generation Server (Port 5620)
After=network.target airun-rag.service airun-websearch.service
Wants=airun-rag.service airun-websearch.service

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
WorkingDirectory=$CURRENT_DIR/services/report
Environment=PYTHONPATH=$CURRENT_DIR
Environment=PYTHONUNBUFFERED=1
Environment=HOME=$HOME
Environment=REPORT_SERVER_HOST=0.0.0.0
Environment=REPORT_SERVER_PORT=5620
Environment=REPORT_SERVER_WORKERS=1
ExecStart=$HOME/.airun_venv/bin/python3 -u airun-report-server.py
ExecStop=/bin/kill -TERM \$MAINPID
Restart=always
RestartSec=3
StandardOutput=append:/var/log/airun/airun-report.log
StandardError=append:/var/log/airun/airun-report.error.log
ProtectSystem=full
NoNewPrivileges=true
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF

# Business Service
cat > /tmp/airun-biz.service << EOF
[Unit]
Description=AI.RUN Business Analysis Server (Port 5640)
After=network.target
Wants=network.target

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
WorkingDirectory=$CURRENT_DIR/services/biz-announcement
Environment=PYTHONPATH=$CURRENT_DIR
Environment=PYTHONUNBUFFERED=1
Environment=HOME=$HOME
Environment=BUSINESS_SERVER_HOST=0.0.0.0
Environment=BUSINESS_SERVER_PORT=5640
Environment=BUSINESS_SERVER_WORKERS=1
ExecStart=$HOME/.airun_venv/bin/python3 -u main.py
ExecStop=/bin/kill -TERM \$MAINPID
Restart=always
RestartSec=3
StandardOutput=append:/var/log/airun/airun-biz.log
StandardError=append:/var/log/airun/airun-biz.error.log
ProtectSystem=full
NoNewPrivileges=true
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF

# LLM Fine-tuning Service
cat > /tmp/airun-llm-finetune.service << EOF
[Unit]
Description=AIRUN LLM Fine-tuning Service
Documentation=https://github.com/airun/llm-finetune
After=network.target postgresql.service redis.service
Wants=postgresql.service redis.service

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
WorkingDirectory=$CURRENT_DIR/services/llm-finetune
Environment=NODE_ENV=production
Environment=LLM_FINETUNE_PORT=5630
Environment=PATH=/usr/local/bin:/usr/bin:/bin:$HOME/.nvm/versions/node/v22.18.0/bin
Environment=HOME=$HOME
Environment=NODE_PATH=$CURRENT_DIR/node_modules:$CURRENT_DIR/services/llm-finetune/node_modules
ExecStart=/usr/bin/env node llm-finetune-server.js
Restart=always
RestartSec=10
StandardOutput=append:$HOME/.airun/logs/airun-llm-finetune.log
StandardError=append:$HOME/.airun/logs/airun-llm-finetune.error.log

[Install]
WantedBy=multi-user.target
EOF

# RQ Worker Service
cat > /tmp/airun-rq-worker.service << EOF
[Unit]
Description=AI.RUN RQ Worker Service
After=network.target redis.service

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
WorkingDirectory=$CURRENT_DIR
Environment=HOME=$HOME
Environment=NODE_PATH=$CURRENT_DIR/node_modules
Environment=PYTHONPATH=$CURRENT_DIR:$CURRENT_DIR/plugins
Environment=PYTHONUNBUFFERED=1
Environment=AIRUN_HOME=$CURRENT_DIR
Environment=AIRUN_VENV=$HOME/.airun_venv
ExecStart=/bin/bash -c 'cd $CURRENT_DIR && . $HOME/.airun_venv/bin/activate && exec rq worker image_processing'
Restart=always
RestartSec=3
StandardOutput=append:/var/log/airun/airun-rq-worker.log
StandardError=append:/var/log/airun/airun-rq-worker.error.log

[Install]
WantedBy=multi-user.target
EOF

# Install RQ if not installed
if ! "$VENV_PATH/bin/pip" show rq >/dev/null 2>&1; then
    show_info "RQ 패키지 설치 중..."
    "$VENV_PATH/bin/pip" install rq
fi

# Install service files
show_info "서비스 파일 설치 중..."
for service_file in airun-api airun-rag airun-websearch airun-report airun-biz airun-llm-finetune airun-rq-worker; do
    if [ -f "/tmp/$service_file.service" ]; then
        sudo cp "/tmp/$service_file.service" "/etc/systemd/system/"
        rm -f "/tmp/$service_file.service"
        show_success "$service_file.service 설치 완료"
    fi
done

# Reload systemd
sudo systemctl daemon-reload

# Enable services
show_info "서비스 자동 시작 설정 중..."
for service in airun-api airun-rag airun-websearch airun-report; do
    sudo systemctl enable $service 2>/dev/null || true
done

# Start services
show_info "핵심 서비스 시작 중..."
for service in airun-rag airun-websearch airun-report airun-api; do
    sudo systemctl start $service 2>/dev/null || true
    sleep 2

    if sudo systemctl is-active --quiet $service; then
        show_success "$service 시작됨"
    else
        show_warning "$service 시작 실패 (수동으로 시작 필요)"
    fi
done

show_success "PHASE 7 완료: 서비스 설정 완료"

#############################################
# PHASE 8: FINAL VERIFICATION
#############################################
show_phase 8 "최종 검증"

show_info "설치 검증 중..."

# Check services
ERRORS=0
WARNINGS=0

echo ""
echo -e "${CYAN}[시스템 구성요소 확인]${NC}"
echo ""

# 1. Check PostgreSQL
show_info "PostgreSQL 확인..."

# Docker 그룹 권한 적용을 위해 새 그룹 세션에서 실행
check_postgres() {
    # 현재 사용자가 docker 그룹에 속해있는지 확인
    if groups $USER | grep -q docker; then
        # 그룹 권한이 있으면 sudo 없이 시도
        if docker ps 2>/dev/null | grep -q airun_postgres; then
            DOCKER_ACCESS=true
        else
            DOCKER_ACCESS=false
        fi
    else
        # 그룹 권한이 없으면 sudo 사용
        if sudo docker ps | grep -q airun_postgres; then
            DOCKER_ACCESS=true
        else
            DOCKER_ACCESS=false
        fi
    fi

    if [ "$DOCKER_ACCESS" = true ]; then
        # Check connection
        if PGPASSWORD=1234 psql -h localhost -p 5433 -U ivs -d airun -c "\l" >/dev/null 2>&1; then
            show_success "PostgreSQL: 실행 중 (포트 5433)"
            # Check pgvector extension
            if PGPASSWORD=1234 psql -h localhost -p 5433 -U ivs -d airun -c "SELECT * FROM pg_extension WHERE extname='vector';" | grep -q vector; then
                show_success "pgvector 확장: 활성화됨"
            else
                show_warning "pgvector 확장: 비활성화"
                ((WARNINGS++))
            fi

            # Check if essential tables exist (schema validation)
            show_info "데이터베이스 스키마 검증..."
            if PGPASSWORD=1234 psql -h localhost -p 5433 -U ivs -d airun -c "SELECT 1 FROM information_schema.tables WHERE table_name='users';" | grep -q "1"; then
                show_success "기본 테이블: 생성됨"

                # Check for embedding_status column specifically
                if PGPASSWORD=1234 psql -h localhost -p 5433 -U ivs -d airun -c "SELECT column_name FROM information_schema.columns WHERE table_name='rag_documents' AND column_name='embedding_status';" | grep -q "embedding_status"; then
                    show_success "RAG 스키마: embedding_status 컬럼 확인됨"
                else
                    show_warning "RAG 스키마: embedding_status 컬럼 없음"
                    ((WARNINGS++))
                fi
            else
                show_warning "데이터베이스 스키마: 초기화 안됨 (API 서버 시작 시 자동 생성됨)"
                ((WARNINGS++))
            fi
        else
            show_warning "PostgreSQL: 연결 실패 (재로그인 후 확인 필요)"
            show_info "Docker 그룹 권한이 현재 세션에 적용되지 않았을 수 있습니다"
            ((WARNINGS++))
        fi
    else
        show_warning "PostgreSQL Docker: 실행 안됨"
        ((ERRORS++))
    fi
}

check_postgres

# 2. Check Redis
show_info "Redis 확인..."
if redis-cli ping 2>&1 | grep -q "PONG"; then
    show_success "Redis: 실행 중 (포트 6379)"
else
    show_warning "Redis: 실행 안됨"
    ((WARNINGS++))
fi

# 3. Check Python Environment
echo ""
echo -e "${CYAN}[Python 환경 확인]${NC}"
echo ""

show_info "Python 가상환경 확인..."
if [ -d "$VENV_PATH" ]; then
    show_success "가상환경: $VENV_PATH"

    # Check PyTorch with CUDA
    if "$VENV_PATH/bin/python" -c "import torch; print(f'PyTorch {torch.__version__}, CUDA: {torch.cuda.is_available()}')" 2>/dev/null; then
        PYTORCH_INFO=$("$VENV_PATH/bin/python" -c "import torch; print(f'PyTorch {torch.__version__}, CUDA: {torch.cuda.is_available()}')" 2>/dev/null)
        show_success "$PYTORCH_INFO"
    else
        show_warning "PyTorch: 설치 안됨"
        ((ERRORS++))
    fi

    # Check key packages
    for package in transformers sentence_transformers langchain; do
        if "$VENV_PATH/bin/python" -c "import $package" 2>/dev/null; then
            show_success "$package: 설치됨"
        else
            show_warning "$package: 설치 안됨"
            ((WARNINGS++))
        fi
    done
else
    show_warning "Python 가상환경: 없음"
    ((ERRORS++))
fi

# 4. Check AI Models
echo ""
echo -e "${CYAN}[AI 모델 확인]${NC}"
echo ""

show_info "AI 모델 디렉토리 확인..."
if [ -d "$AIRUN_DIR/models" ]; then
    # Check text embedding model
    if [ -d "$AIRUN_DIR/models/models--nlpai-lab--KURE-v1" ]; then
        show_success "텍스트 임베딩 모델: KURE-v1 ✓"
    else
        show_warning "텍스트 임베딩 모델: 없음"
        ((WARNINGS++))
    fi

    # Check semantic model
    if [ -d "$AIRUN_DIR/models/models--snunlp--KR-SBERT-V40K-klueNLI-augSTS" ]; then
        show_success "시맨틱 청킹 모델: KR-SBERT ✓"
    else
        show_warning "시맨틱 청킹 모델: 없음"
        ((WARNINGS++))
    fi

    # Check image model
    if [ -d "$AIRUN_DIR/models/models--Bingsu--clip-vit-base-patch32-ko" ]; then
        show_success "이미지 임베딩 모델: CLIP ✓"
    else
        show_warning "이미지 임베딩 모델: 없음 (선택적)"
    fi
else
    show_warning "모델 디렉토리가 없습니다"
    ((WARNINGS++))
fi

# 5. Check Services Status
echo ""
echo -e "${CYAN}[서비스 상태 확인]${NC}"
echo ""

SERVICES=(
    "airun-api:5500:API Server"
    "airun-rag:5600:RAG Server"
    "airun-websearch:5610:WebSearch Server"
    "airun-report:5620:Report Server"
)

for service_info in "${SERVICES[@]}"; do
    IFS=':' read -r service port description <<< "$service_info"

    if sudo systemctl is-active --quiet $service; then
        # Check if port is listening
        if ss -tln | grep -q ":$port "; then
            show_success "$description (포트 $port): 실행 중 ✓"
        else
            show_warning "$description: 실행 중이지만 포트 $port 미개방"
            ((WARNINGS++))
        fi
    else
        show_warning "$description: 정지됨"
        ((WARNINGS++))
    fi
done

# 6. Check Configuration Files
echo ""
echo -e "${CYAN}[설정 파일 확인]${NC}"
echo ""

if [ -f "$CURRENT_DIR/.env" ]; then
    show_success ".env 파일: 존재"
else
    show_warning ".env 파일: 없음"
    ((WARNINGS++))
fi

if [ -f "$AIRUN_DIR/airun.conf" ]; then
    show_success "airun.conf: 존재"
else
    show_warning "airun.conf: 없음"
    ((WARNINGS++))
fi

# 7. Check Node.js Environment
echo ""
echo -e "${CYAN}[Node.js 환경 확인]${NC}"
echo ""

if [ -d "node_modules" ]; then
    show_success "node_modules: 설치됨"
else
    show_warning "node_modules: 없음"
    ((ERRORS++))
fi

if command -v node >/dev/null 2>&1; then
    NODE_VER=$(node -v)
    show_success "Node.js: $NODE_VER"
else
    show_warning "Node.js: 설치 안됨"
    ((ERRORS++))
fi

if command -v npm >/dev/null 2>&1; then
    NPM_VER=$(npm -v)
    show_success "npm: v$NPM_VER"
else
    show_warning "npm: 설치 안됨"
    ((ERRORS++))
fi

#############################################
# INSTALLATION COMPLETE
#############################################

echo ""
echo -e "${BLUE}═══════════════════════════════════════════════════════════${NC}"
echo ""

# Final Summary
if [ $ERRORS -eq 0 ] && [ $WARNINGS -eq 0 ]; then
    echo -e "${GREEN}╔════════════════════════════════════════════════════╗${NC}"
    echo -e "${GREEN}║         🎉 설치가 완전히 성공했습니다! 🎉          ║${NC}"
    echo -e "${GREEN}╚════════════════════════════════════════════════════╝${NC}"
elif [ $ERRORS -eq 0 ]; then
    echo -e "${YELLOW}╔════════════════════════════════════════════════════╗${NC}"
    echo -e "${YELLOW}║    ⚠️  설치 완료 (경고 $WARNINGS개)  ⚠️            ║${NC}"
    echo -e "${YELLOW}╚════════════════════════════════════════════════════╝${NC}"
else
    echo -e "${RED}╔════════════════════════════════════════════════════╗${NC}"
    echo -e "${RED}║    ❌ 설치 중 오류 발생 (오류 $ERRORS개)  ❌       ║${NC}"
    echo -e "${RED}╚════════════════════════════════════════════════════╝${NC}"
fi

echo ""
echo -e "${CYAN}═══ 설치 요약 ═══${NC}"
echo ""
echo -e "설치 디렉토리: ${GREEN}$CURRENT_DIR${NC}"
echo -e "Python 가상환경: ${GREEN}$VENV_PATH${NC}"
echo -e "설정 디렉토리: ${GREEN}$AIRUN_DIR${NC}"
echo ""

echo -e "${CYAN}═══ 접속 정보 ═══${NC}"
echo ""
echo -e "웹 인터페이스: ${GREEN}http://localhost:5700${NC}"
echo -e "API 서버: ${GREEN}http://localhost:5500${NC}"
echo -e "RAG 서버: ${GREEN}http://localhost:5600${NC}"
echo -e "웹검색 서버: ${GREEN}http://localhost:5610${NC}"
echo -e "보고서 서버: ${GREEN}http://localhost:5620${NC}"
echo ""

echo -e "${CYAN}═══ 사용 방법 ═══${NC}"
echo ""
echo -e "${YELLOW}터미널에서 실행:${NC}"
echo -e "  ${GREEN}node index.js${NC}         # 대화형 CLI"
echo -e "  ${GREEN}airun${NC}                 # 명령줄 도구"
echo ""
echo -e "${YELLOW}GUI 모드:${NC}"
echo -e "  ${GREEN}airun -g${NC}              # GUI 인터페이스"
echo ""
echo -e "${YELLOW}웹 인터페이스:${NC}"
echo -e "  ${GREEN}npm run dev${NC}           # 개발 서버 시작"
echo -e "  브라우저에서 ${GREEN}http://localhost:5700${NC} 접속"
echo ""

echo -e "${CYAN}═══ 서비스 관리 ═══${NC}"
echo ""
echo -e "${YELLOW}서비스 상태 확인:${NC}"
echo -e "  ${GREEN}sudo systemctl status airun-api${NC}"
echo -e "  ${GREEN}sudo systemctl status airun-rag${NC}"
echo ""
echo -e "${YELLOW}서비스 재시작:${NC}"
echo -e "  ${GREEN}sudo systemctl restart airun-api${NC}"
echo -e "  ${GREEN}sudo systemctl restart airun-rag${NC}"
echo ""
echo -e "${YELLOW}로그 확인:${NC}"
echo -e "  ${GREEN}tail -f ~/.airun/logs/airun-api.log${NC}"
echo -e "  ${GREEN}sudo journalctl -u airun-api -f${NC}"
echo ""

echo -e "${CYAN}═══ 설정 파일 ═══${NC}"
echo ""
echo -e "환경 변수: ${GREEN}$CURRENT_DIR/.env${NC}"
echo -e "시스템 설정: ${GREEN}$AIRUN_DIR/airun.conf${NC}"
echo ""

if [ $ERRORS -gt 0 ] || [ $WARNINGS -gt 0 ]; then
    echo -e "${CYAN}═══ 문제 해결 ═══${NC}"
    echo ""

    if [ $ERRORS -gt 0 ]; then
        echo -e "${RED}다음 오류를 해결해야 합니다:${NC}"
        echo -e "1. 위의 오류 메시지를 확인하세요"
        echo -e "2. 필요한 경우 로그를 확인하세요: ${GREEN}~/.airun/logs/${NC}"
        echo -e "3. 시스템 요구사항을 확인하세요"
        echo ""
    fi

    if [ $WARNINGS -gt 0 ]; then
        echo -e "${YELLOW}다음 경고사항을 확인하세요:${NC}"
        echo -e "• 일부 선택적 기능이 비활성화될 수 있습니다"
        echo -e "• 서비스가 자동으로 시작되지 않을 수 있습니다"
        echo -e "• 수동으로 서비스를 시작해야 할 수 있습니다"
        echo ""
    fi
fi

echo -e "${BLUE}═══════════════════════════════════════════════════════════${NC}"
echo ""

# Save installation log
LOG_FILE="$AIRUN_DIR/logs/install_$(date +%Y%m%d_%H%M%S).log"
echo "Installation completed at $(date)" > "$LOG_FILE"
echo "Errors: $ERRORS, Warnings: $WARNINGS" >> "$LOG_FILE"

echo -e "${CYAN}설치 로그가 저장되었습니다: ${GREEN}$LOG_FILE${NC}"
echo ""

# Final message
if [ $ERRORS -eq 0 ]; then
    echo -e "${GREEN}AI.RUN이 성공적으로 설치되었습니다! 🚀${NC}"
    echo -e "${GREEN}이제 위의 명령어를 사용하여 AI.RUN을 시작할 수 있습니다.${NC}"
else
    echo -e "${YELLOW}설치가 완료되었지만 일부 문제가 있습니다.${NC}"
    echo -e "${YELLOW}위의 문제 해결 가이드를 참조하여 수정해주세요.${NC}"
fi

echo ""