import fs from 'fs';
import path from 'path';
import os from 'os';
import { fileURLToPath } from 'url';
import singleton from '../singleton.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// 가격 데이터 캐시
let priceCache = null;
let lastPriceCacheUpdate = null;
const PRICE_CACHE_DURATION = 60 * 60 * 1000; // 1시간

// 사용량 데이터 캐시
let usageCache = null;
let lastUsageUpdate = null;
const USAGE_CACHE_DURATION = 5 * 60 * 1000; // 5분

function getAppDataPath() {
    switch (process.platform) {
        case 'win32':
            return path.join(process.env.APPDATA, 'airun');
        case 'darwin':
            return path.join(os.homedir(), 'Library', 'Application Support', 'airun');
        default: // linux and others
            return path.join(os.homedir(), '.airun');
    }
}

async function loadPriceData() {
    try {
        // 캐시가 유효한 경우
        if (priceCache && lastPriceCacheUpdate && 
            (Date.now() - lastPriceCacheUpdate) < PRICE_CACHE_DURATION) {
            return priceCache;
        }

        const priceFilePath = path.join(__dirname, 'price.json');
        const fileContent = await fs.promises.readFile(priceFilePath, 'utf8');
        const priceData = JSON.parse(fileContent);

        // 캐시 업데이트
        priceCache = priceData;
        lastPriceCacheUpdate = Date.now();

        return priceData;
    } catch (error) {
        console.error('Error loading price data:', error);
        return null;
    }
}

function calculateCost(provider, model, promptTokens, completionTokens) {
    if (!priceCache) return 0;

    const providerPrices = priceCache[provider];
    if (!providerPrices) return 0;

    const modelPrices = providerPrices[model];
    if (!modelPrices) return 0;

    if (typeof modelPrices === 'number') {
        // Anthropic 스타일 단일 가격
        return (promptTokens + completionTokens) * modelPrices / 1000;
    } else {
        // OpenAI 스타일 분리 가격
        const promptCost = (promptTokens * modelPrices.prompt) / 1000;
        const completionCost = (completionTokens * modelPrices.completion) / 1000;
        return promptCost + completionCost;
    }
}

export async function saveUsageData(usageData) {
    try {
        // provider 필드 검증
        if (!usageData.provider) {
            console.error('Provider is required in usage data');
            return false;
        }

        const appDataDir = getAppDataPath();
        const usageFilePath = path.join(appDataDir, 'usage.json');
        const archiveDir = path.join(appDataDir, 'archives');
        
        // 앱 데이터 디렉토리와 아카이브 디렉토리가 없으면 생성
        if (!fs.existsSync(appDataDir)) {
            await fs.promises.mkdir(appDataDir, { recursive: true });
        }
        if (!fs.existsSync(archiveDir)) {
            await fs.promises.mkdir(archiveDir, { recursive: true });
        }

        // 기존 데이터 읽기 (캐시 활용)
        let existingData = [];
        if (usageCache) {
            existingData = usageCache;
        } else if (fs.existsSync(usageFilePath)) {
            try {
                const fileContent = await fs.promises.readFile(usageFilePath, 'utf8');
                if (fileContent.trim()) {  // 파일이 비어있지 않은 경우에만 파싱
                    existingData = JSON.parse(fileContent);
                }
            } catch (readError) {
                console.error('Error reading usage file:', readError);
                // 파일 읽기 실패 시 빈 배열로 시작
                existingData = [];
            }
        }

        // messages 데이터 최적화
        let optimizedMessages = null;
        if (usageData.messages && Array.isArray(usageData.messages)) {
            const lastUserMessage = [...usageData.messages].reverse().find(msg => msg.role === 'user');
            if (lastUserMessage) {
                optimizedMessages = [
                    usageData.messages[0],
                    lastUserMessage
                ];
            }
        }

        // 비용 계산
        const prices = await loadPriceData();
        const cost = calculateCost(
            usageData.provider,
            usageData.model,
            usageData.promptTokens,
            usageData.completionTokens
        );

        const sessionId = singleton.sessionId || `session_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
        if (!singleton.sessionId) {
            singleton.sessionId = sessionId;
        }

        const enhancedUsageData = {
            ...usageData,
            sessionId,
            cost,
            timestamp: new Date().toISOString(),
            messages: optimizedMessages,
            prompt: usageData.messages?.[usageData.messages.length - 1]?.content || '',
            totalTokens: usageData.promptTokens + usageData.completionTokens,
            requestType: usageData.messages?.[usageData.messages.length - 1]?.role || 'user',
            success: true,
            duration: usageData.duration || 0,
            // 사용자 정보 추가
            userId: usageData.userId || null,
            username: usageData.username || null,
        };

        // 데이터 아카이브 처리
        const maxRecords = 1000;
        if (existingData.length >= maxRecords) {
            // 아카이브할 데이터 추출 (가장 오래된 500개)
            const dataToArchive = existingData.slice(0, 500);
            existingData = existingData.slice(500);

            // 월별로 아카이브 파일 생성
            const oldestDate = new Date(dataToArchive[0].timestamp);
            const archiveFileName = `usage_${oldestDate.getFullYear()}_${String(oldestDate.getMonth() + 1).padStart(2, '0')}.json`;
            const archiveFilePath = path.join(archiveDir, archiveFileName);

            // 기존 아카이브 파일이 있으면 병합
            let archiveData = [];
            if (fs.existsSync(archiveFilePath)) {
                try {
                    const archiveContent = await fs.promises.readFile(archiveFilePath, 'utf8');
                    if (archiveContent.trim()) {  // 파일이 비어있지 않은 경우에만 파싱
                        archiveData = JSON.parse(archiveContent);
                    }
                } catch (archiveReadError) {
                    console.error('Error reading archive file:', archiveReadError);
                    // 아카이브 파일 읽기 실패 시 빈 배열로 시작
                    archiveData = [];
                }
            }
            archiveData.push(...dataToArchive);

            // 아카이브 파일 저장
            try {
                await fs.promises.writeFile(archiveFilePath, JSON.stringify(archiveData, null, 2));
            } catch (archiveWriteError) {
                console.error('Error writing archive file:', archiveWriteError);
            }
        }

        // 새 데이터 추가
        existingData.push(enhancedUsageData);
        
        // 현재 사용량 파일 저장
        try {
            await fs.promises.writeFile(usageFilePath, JSON.stringify(existingData, null, 2));
            
            // 캐시 업데이트
            usageCache = existingData;
            lastUsageUpdate = Date.now();
            
            return true;
        } catch (writeError) {
            console.error('Error writing usage file:', writeError);
            return false;
        }
    } catch (error) {
        console.error('Error saving usage data:', error);
        return false;
    }
}

export async function getUsageData() {
    try {
        // 캐시가 유효한 경우
        if (usageCache && lastUsageUpdate && 
            (Date.now() - lastUsageUpdate) < USAGE_CACHE_DURATION) {
            return usageCache;
        }

        const appDataDir = getAppDataPath();
        const usageFilePath = path.join(appDataDir, 'usage.json');
        
        if (!fs.existsSync(usageFilePath)) {
            usageCache = [];
            lastUsageUpdate = Date.now();
            return [];
        }

        const fileContent = await fs.promises.readFile(usageFilePath, 'utf8');
        const data = JSON.parse(fileContent);
        
        // 캐시 업데이트
        usageCache = data;
        lastUsageUpdate = Date.now();
        
        return data;
    } catch (error) {
        console.error('Error reading usage data:', error);
        return [];
    }
}

export function invalidateCache() {
    usageCache = null;
    lastUsageUpdate = null;
    priceCache = null;
    lastPriceCacheUpdate = null;
}

export async function getSessionStats(sessionId) {
    const usageData = await getUsageData();
    const sessionData = usageData.filter(data => data.sessionId === sessionId);
    
    return {
        totalCost: sessionData.reduce((sum, data) => sum + data.cost, 0),
        totalTokens: sessionData.reduce((sum, data) => sum + data.totalTokens, 0),
        totalRequests: sessionData.length,
        promptTokens: sessionData.reduce((sum, data) => sum + data.promptTokens, 0),
        completionTokens: sessionData.reduce((sum, data) => sum + data.completionTokens, 0),
        averageResponseTime: sessionData.reduce((sum, data) => sum + data.duration, 0) / sessionData.length,
        startTime: sessionData[0]?.timestamp,
        endTime: sessionData[sessionData.length - 1]?.timestamp
    };
}

export async function getUsageStats(startDate, endDate) {
    const usageData = await getUsageData();
    const filteredData = usageData.filter(data => {
        const timestamp = new Date(data.timestamp);
        return timestamp >= startDate && timestamp <= endDate;
    });

    const stats = {
        totalCost: 0,
        totalTokens: 0,
        totalRequests: filteredData.length,
        byProvider: {},
        byModel: {},
        bySession: {}
    };

    filteredData.forEach(data => {
        // 전체 통계
        stats.totalCost += data.cost;
        stats.totalTokens += data.totalTokens;

        // 프로바이더별 통계
        if (!stats.byProvider[data.provider]) {
            stats.byProvider[data.provider] = {
                cost: 0,
                tokens: 0,
                requests: 0
            };
        }
        stats.byProvider[data.provider].cost += data.cost;
        stats.byProvider[data.provider].tokens += data.totalTokens;
        stats.byProvider[data.provider].requests++;

        // 모델별 통계
        if (!stats.byModel[data.model]) {
            stats.byModel[data.model] = {
                cost: 0,
                tokens: 0,
                requests: 0
            };
        }
        stats.byModel[data.model].cost += data.cost;
        stats.byModel[data.model].tokens += data.totalTokens;
        stats.byModel[data.model].requests++;

        // 세션별 통계
        if (!stats.bySession[data.sessionId]) {
            stats.bySession[data.sessionId] = {
                cost: 0,
                tokens: 0,
                requests: 0,
                startTime: data.timestamp
            };
        }
        stats.bySession[data.sessionId].cost += data.cost;
        stats.bySession[data.sessionId].tokens += data.totalTokens;
        stats.bySession[data.sessionId].requests++;
        stats.bySession[data.sessionId].endTime = data.timestamp;
    });

    return stats;
}

// 아카이브된 데이터를 포함한 전체 사용량 데이터 조회
export async function getAllUsageData() {
    try {
        const appDataDir = getAppDataPath();
        const archiveDir = path.join(appDataDir, 'archives');
        let allData = [];

        // 아카이브 디렉토리가 있으면 모든 아카이브 파일 읽기
        if (fs.existsSync(archiveDir)) {
            const archiveFiles = await fs.promises.readdir(archiveDir);
            for (const file of archiveFiles) {
                if (file.startsWith('usage_') && file.endsWith('.json')) {
                    const archiveContent = await fs.promises.readFile(path.join(archiveDir, file), 'utf8');
                    const archiveData = JSON.parse(archiveContent);
                    allData.push(...archiveData);
                }
            }
        }

        // 현재 사용량 데이터 추가
        const currentData = await getUsageData();
        allData.push(...currentData);

        // 날짜순으로 정렬
        allData.sort((a, b) => new Date(a.timestamp) - new Date(b.timestamp));

        return allData;
    } catch (error) {
        console.error('Error reading all usage data:', error);
        return [];
    }
}

export { loadPriceData }; 