#!/bin/bash

# AI Tool System - Quick Start Script
# 새로운 도구를 빠르게 생성하고 테스트할 수 있는 스크립트

set -e

# 색상 정의
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 함수들
print_header() {
    echo -e "${BLUE}===============================================${NC}"
    echo -e "${BLUE}         AI Tool System - Quick Start          ${NC}"
    echo -e "${BLUE}===============================================${NC}"
    echo ""
}

print_success() {
    echo -e "${GREEN}✅ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

print_error() {
    echo -e "${RED}❌ $1${NC}"
}

print_info() {
    echo -e "${BLUE}ℹ️  $1${NC}"
}

# 도구 템플릿 생성
create_tool_template() {
    local tool_name=$1
    local tool_file=$2
    local category=${3:-"utility"}
    local description=${4:-"새로운 커스텀 도구입니다."}
    
    cat > "$tool_file" << EOF
// $tool_name 도구
// 생성 시간: $(date)

const ${tool_name}Tool = {
    name: '${tool_name}',
    category: '${category}',
    description: '${description}',
    version: '1.0.0',
    parameters: {
        input: { 
            type: 'string', 
            required: true, 
            description: '입력 값' 
        },
        options: {
            type: 'object',
            description: '추가 옵션들',
            default: {}
        }
    },
    handler: async (args, context) => {
        const { input, options = {} } = args;
        const { userId, job, toolName } = context;
        
        console.log(\`[\${toolName}] 실행 시작: \${input}\`);
        
        try {
            // 진행 상황 업데이트 (선택사항)
            if (job && job.progress) {
                await job.progress(30, '작업 진행 중...');
            }
            
            // TODO: 여기에 실제 도구 로직을 구현하세요
            await new Promise(resolve => setTimeout(resolve, 1000)); // 예시 지연
            
            const result = {
                processed_input: input.toUpperCase(), // 예시 처리
                timestamp: new Date().toISOString(),
                user_id: userId,
                options_used: options
            };
            
            if (job && job.progress) {
                await job.progress(100, '작업 완료');
            }
            
            return {
                success: true,
                data: result,
                message: \`\${tool_name} 도구 실행 완료\`
            };
        } catch (error) {
            console.error(\`[\${toolName}] 오류:, error\`);
            return {
                success: false,
                error: error.message,
                message: \`\${tool_name} 도구 실행 실패: \${error.message}\`
            };
        }
    }
};

// 추가 도구들을 여기에 정의할 수 있습니다
// const anotherTool = { ... };

export default [${tool_name}Tool];
EOF
}

# CLI 사용법 출력
show_usage() {
    echo "사용법: $0 [명령어] [옵션]"
    echo ""
    echo "명령어:"
    echo "  create <도구명>           - 새 도구 생성"
    echo "  test <도구명>             - 도구 테스트"
    echo "  list                     - 등록된 도구 목록"
    echo "  info <도구명>             - 도구 정보 확인"
    echo "  reload                   - 도구 다시 로드"
    echo "  help                     - 이 도움말 표시"
    echo ""
    echo "예제:"
    echo "  $0 create my_calculator"
    echo "  $0 test my_calculator"
    echo "  $0 list"
    echo ""
}

# 도구 생성
create_tool() {
    local tool_name=$1
    
    if [ -z "$tool_name" ]; then
        print_error "도구 이름을 입력해주세요."
        echo "예: $0 create my_tool"
        return 1
    fi
    
    # 도구 이름 검증 (소문자, 숫자, 언더스코어만 허용)
    if [[ ! $tool_name =~ ^[a-z0-9_]+$ ]]; then
        print_error "도구 이름은 소문자, 숫자, 언더스코어만 사용 가능합니다."
        return 1
    fi
    
    print_info "새 도구 '$tool_name' 생성 중..."
    
    # 파일 경로 설정
    local tools_dir="tools/custom-tools"
    local tool_file="$tools_dir/${tool_name}.js"
    
    # 디렉토리 생성 (없으면)
    mkdir -p "$tools_dir"
    
    # 파일이 이미 존재하는지 확인
    if [ -f "$tool_file" ]; then
        print_warning "파일이 이미 존재합니다: $tool_file"
        read -p "덮어쓰시겠습니까? (y/N): " confirm
        if [[ ! $confirm =~ ^[Yy]$ ]]; then
            print_info "취소됨"
            return 0
        fi
    fi
    
    # 사용자 입력 받기
    echo ""
    read -p "카테고리 (기본값: utility): " category
    category=${category:-"utility"}
    
    read -p "도구 설명: " description
    description=${description:-"새로운 커스텀 도구입니다."}
    
    # 템플릿 생성
    create_tool_template "$tool_name" "$tool_file" "$category" "$description"
    
    print_success "도구 파일 생성됨: $tool_file"
    print_info "다음 단계:"
    echo "  1. $tool_file 파일을 편집하여 도구 로직을 구현하세요"
    echo "  2. $0 test $tool_name 명령으로 테스트하세요"
    echo "  3. 시스템을 재시작하여 도구를 활성화하세요"
    echo ""
}

# 도구 테스트
test_tool() {
    local tool_name=$1
    
    if [ -z "$tool_name" ]; then
        print_error "테스트할 도구 이름을 입력해주세요."
        return 1
    fi
    
    print_info "도구 '$tool_name' 테스트 중..."
    
    # CLI 도구로 테스트 실행
    if [ -f "tools/cli-tools.js" ]; then
        node tools/cli-tools.js test "$tool_name"
    else
        print_error "CLI 도구가 없습니다: tools/cli-tools.js"
        return 1
    fi
}

# 도구 목록
list_tools() {
    print_info "등록된 도구 목록 조회 중..."
    
    if [ -f "tools/cli-tools.js" ]; then
        node tools/cli-tools.js list
    else
        print_error "CLI 도구가 없습니다: tools/cli-tools.js"
        return 1
    fi
}

# 도구 정보
tool_info() {
    local tool_name=$1
    
    if [ -z "$tool_name" ]; then
        print_error "정보를 확인할 도구 이름을 입력해주세요."
        return 1
    fi
    
    print_info "도구 '$tool_name' 정보 조회 중..."
    
    if [ -f "tools/cli-tools.js" ]; then
        node tools/cli-tools.js info "$tool_name"
    else
        print_error "CLI 도구가 없습니다: tools/cli-tools.js"
        return 1
    fi
}

# 도구 다시 로드
reload_tools() {
    print_info "도구 다시 로드 중..."
    
    if [ -f "tools/cli-tools.js" ]; then
        node tools/cli-tools.js reload
    else
        print_error "CLI 도구가 없습니다: tools/cli-tools.js"
        return 1
    fi
}

# 메인 로직
main() {
    print_header
    
    case "${1:-help}" in
        "create")
            create_tool "$2"
            ;;
        "test")
            test_tool "$2"
            ;;
        "list")
            list_tools
            ;;
        "info")
            tool_info "$2"
            ;;
        "reload")
            reload_tools
            ;;
        "help"|"--help"|"-h"|"")
            show_usage
            ;;
        *)
            print_error "알 수 없는 명령어: $1"
            echo ""
            show_usage
            exit 1
            ;;
    esac
}

# 스크립트 실행
main "$@" 