import { NextRequest, NextResponse } from "next/server";
import { promises as fs } from "fs";
import path from "path";

// 기본 MCP 설정 (app.py와 동일)
const DEFAULT_MCP_CONFIG = {
  "hamonize": {
    "command": "node",
    "args": ["./mcp_tools/hamonize/dist/index.js"],
    "transport": "stdio"
  },
  "postgres": {
    "transport": "stdio",
    "command": "node",
    "args": ["./mcp_tools/mcp-postgres-server/build/index.js"]
  }
};

// MCP 설정 파일 경로
const getMCPConfigPath = () => {
  // 웹 프로젝트 디렉터리의 mcp_config.json 사용 (다른 엔드포인트와 통일)
  return path.join(process.cwd(), 'mcp_config.json');
};

// MCP 설정 로드 함수
const loadMCPConfig = async () => {
  try {
    const configPath = getMCPConfigPath();
    const configData = await fs.readFile(configPath, 'utf8');
    const config = JSON.parse(configData);
    
    // 설정 파일의 내용을 그대로 반환 (자동 추가 제거)
    return config;
  } catch (error) {
    console.log('MCP 설정 파일을 찾을 수 없습니다. 빈 설정을 사용합니다.');
    return {};
  }
};

// MCP 설정 저장 함수
const saveMCPConfig = async (config: any) => {
  try {
    const configPath = getMCPConfigPath();
    
    // 프로젝트 루트 디렉터리에 저장 (디렉터리 생성 불필요)
    await fs.writeFile(configPath, JSON.stringify(config, null, 2), 'utf8');
    return true;
  } catch (error) {
    console.error('MCP 설정 파일 저장 중 오류:', error);
    return false;
  }
};

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const isOriginal = searchParams.get('original') === 'true';
    
    if (isOriginal) {
      // 원본 설정 파일을 기본 설정 파일로 복사
      try {
        const originalConfigPath = path.join(process.cwd(), 'mcp_config.orig.json');
        const configPath = getMCPConfigPath();
        
        // 원본 파일이 존재하는지 확인
        await fs.access(originalConfigPath);
        
        // 원본 파일을 기본 파일로 복사
        await fs.copyFile(originalConfigPath, configPath);
        
        // 복사된 설정 로드
        const config = await loadMCPConfig();
        
        return NextResponse.json({
          success: true,
          config,
          message: `초기 설정으로 복원되었습니다. (${Object.keys(config).length}개의 MCP 서버)`
        });
      } catch (error) {
        console.error('초기 설정 복원 실패:', error);
        return NextResponse.json({
          success: false,
          error: '초기 설정 파일(mcp_config.orig.json)을 찾을 수 없습니다.',
          config: {}
        }, { status: 404 });
      }
    } else {
      // 기본 설정 파일 로드
      const config = await loadMCPConfig();
      
      return NextResponse.json({
        success: true,
        config,
        message: `MCP 설정을 성공적으로 로드했습니다. (${Object.keys(config).length}개의 MCP 서버)`
      });
    }
  } catch (error) {
    console.error('MCP 설정 로드 중 오류:', error);
    return NextResponse.json(
      { 
        success: false, 
        error: 'MCP 설정 로드 중 오류 발생',
        config: { ...DEFAULT_MCP_CONFIG }
      },
      { status: 500 }
    );
  }
}

export async function POST(request: NextRequest) {
  try {
    const { config } = await request.json();
    
    if (!config || typeof config !== 'object') {
      return NextResponse.json(
        { success: false, error: '유효하지 않은 설정 데이터입니다.' },
        { status: 400 }
      );
    }
    
    // 설정을 그대로 사용 (자동 추가 제거)
    
    const success = await saveMCPConfig(config);
    
    if (success) {
      return NextResponse.json({
        success: true,
        message: 'MCP 설정이 성공적으로 저장되었습니다.',
        config
      });
    } else {
      return NextResponse.json(
        { success: false, error: 'MCP 설정 저장에 실패했습니다.' },
        { status: 500 }
      );
    }
  } catch (error) {
    console.error('MCP 설정 저장 중 오류:', error);
    return NextResponse.json(
      { success: false, error: 'MCP 설정 저장 중 오류 발생' },
      { status: 500 }
    );
  }
} 