'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { useRouter } from 'next/navigation';

interface Document {
  id: string;
  name: string;
  description: string;
  format: 'PDF' | 'HWPX' | 'DOCX' | 'PPTX';
  status: 'completed' | 'processing' | 'failed';
  createdAt: string;
  updatedAt: string;
  fileSize?: number;
  downloadUrl?: string;
}

export default function DocumentListPage() {
  const router = useRouter();
  const [documents, setDocuments] = useState<Document[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [formatFilter, setFormatFilter] = useState<string>('');
  const [statusFilter, setStatusFilter] = useState<string>('');
  
  // 문서 데이터 로드
  useEffect(() => {
    const fetchDocuments = async () => {
      try {
        // API 호출 시뮬레이션
        await new Promise(resolve => setTimeout(resolve, 1000));
        
        // 샘플 데이터
        const sampleDocuments: Document[] = [
          {
            id: '1',
            name: '분기별 마케팅 보고서',
            description: '2023년 2분기 마케팅 캠페인 성과 분석 및 향후 전략 제안',
            format: 'PDF',
            status: 'completed',
            createdAt: '2023-06-15T14:30:00Z',
            updatedAt: '2023-06-15T14:35:22Z',
            fileSize: 2540000,
            downloadUrl: '/api/documents/1/download'
          },
          {
            id: '2',
            name: '제품 사용 설명서',
            description: '신규 출시 제품의 기능 및 사용법 설명',
            format: 'HWPX',
            status: 'completed',
            createdAt: '2023-06-10T09:45:00Z',
            updatedAt: '2023-06-10T09:48:15Z',
            fileSize: 1820000,
            downloadUrl: '/api/documents/2/download'
          },
          {
            id: '3',
            name: '신규 서비스 제안서',
            description: '고객 만족도 향상을 위한 신규 서비스 제안 및 도입 계획',
            format: 'PPTX',
            status: 'completed',
            createdAt: '2023-06-05T16:20:00Z',
            updatedAt: '2023-06-05T16:25:18Z',
            fileSize: 3240000,
            downloadUrl: '/api/documents/3/download'
          },
          {
            id: '4',
            name: '월간 재무 보고서',
            description: '6월 월간 재무 현황 및 분석',
            format: 'DOCX',
            status: 'processing',
            createdAt: '2023-06-20T10:15:00Z',
            updatedAt: '2023-06-20T10:15:00Z'
          },
          {
            id: '5',
            name: '고객 조사 결과 분석',
            description: '제품 만족도 고객 조사 결과 분석 및 개선 방안',
            format: 'PDF',
            status: 'failed',
            createdAt: '2023-06-18T13:40:00Z',
            updatedAt: '2023-06-18T13:42:35Z'
          }
        ];
        
        setDocuments(sampleDocuments);
      } catch (error) {
        console.error('문서 로드 오류:', error);
      } finally {
        setLoading(false);
      }
    };
    
    fetchDocuments();
  }, []);

  // 검색 및 필터링된 문서 목록
  const filteredDocuments = documents.filter(doc => {
    const matchesSearch = searchTerm === '' || 
      doc.name.toLowerCase().includes(searchTerm.toLowerCase()) || 
      doc.description.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchesFormat = formatFilter === '' || doc.format === formatFilter;
    const matchesStatus = statusFilter === '' || doc.status === statusFilter;
    
    return matchesSearch && matchesFormat && matchesStatus;
  });

  // 포맷별 문서 수 계산
  const formatCounts = documents.reduce((acc, doc) => {
    acc[doc.format] = (acc[doc.format] || 0) + 1;
    return acc;
  }, {} as Record<string, number>);
  
  // 상태별 문서 수 계산
  const statusCounts = documents.reduce((acc, doc) => {
    acc[doc.status] = (acc[doc.status] || 0) + 1;
    return acc;
  }, {} as Record<string, number>);

  // 날짜 포맷팅 함수
  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleDateString('ko-KR', {
      year: 'numeric',
      month: 'long',
      day: 'numeric'
    });
  };

  // 파일 크기 포맷팅 함수
  const formatFileSize = (bytes?: number) => {
    if (!bytes) return '미정';
    
    if (bytes < 1024) return bytes + ' B';
    if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(1) + ' KB';
    return (bytes / (1024 * 1024)).toFixed(1) + ' MB';
  };

  // 포맷별 아이콘
  const getFormatIcon = (format: Document['format']) => {
    switch (format) {
      case 'PDF':
        return '📄';
      case 'HWPX':
        return '📝';
      case 'DOCX':
        return '📃';
      case 'PPTX':
        return '📊';
      default:
        return '📄';
    }
  };

  // 상태별 배지 스타일
  const getStatusBadgeStyle = (status: Document['status']) => {
    switch (status) {
      case 'completed':
        return 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300';
      case 'processing':
        return 'bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300';
      case 'failed':
        return 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300';
      default:
        return 'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300';
    }
  };

  // 상태 한글 표시
  const getStatusText = (status: Document['status']) => {
    switch (status) {
      case 'completed':
        return '완료';
      case 'processing':
        return '처리 중';
      case 'failed':
        return '실패';
      default:
        return '알 수 없음';
    }
  };

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="flex justify-between items-center mb-8">
        <h1 className="text-3xl font-bold">문서 목록</h1>
        <Link 
          href="/document/new"
          className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors"
        >
          새 문서 작성
        </Link>
      </div>
      
      <div className="grid grid-cols-1 lg:grid-cols-4 gap-6">
        {/* 필터 사이드바 */}
        <div className="lg:col-span-1">
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow p-4 mb-4">
            <h2 className="text-lg font-semibold mb-4">검색</h2>
            <input
              type="text"
              className="w-full px-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white mb-4"
              placeholder="문서 검색..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
            />
            
            <h2 className="text-lg font-semibold mb-4">문서 형식</h2>
            <div className="space-y-2">
              <div className="flex items-center">
                <input
                  id="all-formats"
                  type="radio"
                  name="format-filter"
                  checked={formatFilter === ''}
                  onChange={() => setFormatFilter('')}
                  className="h-4 w-4 text-indigo-600 focus:ring-indigo-500 border-gray-300"
                />
                <label htmlFor="all-formats" className="ml-2 block text-sm text-gray-700 dark:text-gray-300">
                  모든 형식 ({documents.length})
                </label>
              </div>
              
              {Object.entries(formatCounts).map(([format, count]) => (
                <div key={format} className="flex items-center">
                  <input
                    id={`format-${format}`}
                    type="radio"
                    name="format-filter"
                    checked={formatFilter === format}
                    onChange={() => setFormatFilter(format)}
                    className="h-4 w-4 text-indigo-600 focus:ring-indigo-500 border-gray-300"
                  />
                  <label htmlFor={`format-${format}`} className="ml-2 block text-sm text-gray-700 dark:text-gray-300">
                    {format} ({count})
                  </label>
                </div>
              ))}
            </div>
            
            <h2 className="text-lg font-semibold my-4">문서 상태</h2>
            <div className="space-y-2">
              <div className="flex items-center">
                <input
                  id="all-statuses"
                  type="radio"
                  name="status-filter"
                  checked={statusFilter === ''}
                  onChange={() => setStatusFilter('')}
                  className="h-4 w-4 text-indigo-600 focus:ring-indigo-500 border-gray-300"
                />
                <label htmlFor="all-statuses" className="ml-2 block text-sm text-gray-700 dark:text-gray-300">
                  모든 상태 ({documents.length})
                </label>
              </div>
              
              {Object.entries(statusCounts).map(([status, count]) => (
                <div key={status} className="flex items-center">
                  <input
                    id={`status-${status}`}
                    type="radio"
                    name="status-filter"
                    checked={statusFilter === status}
                    onChange={() => setStatusFilter(status)}
                    className="h-4 w-4 text-indigo-600 focus:ring-indigo-500 border-gray-300"
                  />
                  <label htmlFor={`status-${status}`} className="ml-2 block text-sm text-gray-700 dark:text-gray-300">
                    {getStatusText(status as Document['status'])} ({count})
                  </label>
                </div>
              ))}
            </div>
          </div>
        </div>
        
        {/* 문서 목록 */}
        <div className="lg:col-span-3">
          {loading ? (
            <div className="flex justify-center py-12">
              <div className="animate-spin rounded-full h-10 w-10 border-t-2 border-b-2 border-blue-500"></div>
            </div>
          ) : filteredDocuments.length > 0 ? (
            <div className="bg-white dark:bg-gray-800 rounded-lg shadow overflow-hidden">
              <ul className="divide-y divide-gray-200 dark:divide-gray-700">
                {filteredDocuments.map((doc) => (
                  <li key={doc.id} className="p-4 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors">
                    <div className="flex justify-between items-start">
                      <div className="flex items-start space-x-4">
                        <div className="flex-shrink-0 text-3xl">{getFormatIcon(doc.format)}</div>
                        <div>
                          <Link href={`/document/${doc.id}`}>
                            <h3 className="text-lg font-medium text-blue-600 dark:text-blue-400 hover:underline">{doc.name}</h3>
                          </Link>
                          <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">{doc.description}</p>
                          <div className="mt-2 flex flex-wrap gap-2">
                            <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 dark:bg-gray-700 text-gray-800 dark:text-gray-200">
                              {doc.format}
                            </span>
                            <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getStatusBadgeStyle(doc.status)}`}>
                              {getStatusText(doc.status)}
                            </span>
                          </div>
                          <div className="mt-2 text-xs text-gray-500 dark:text-gray-400 space-x-4">
                            <span>생성일: {formatDate(doc.createdAt)}</span>
                            <span>크기: {formatFileSize(doc.fileSize)}</span>
                          </div>
                        </div>
                      </div>
                      <div className="flex space-x-2">
                        {doc.status === 'completed' && (
                          <>
                            <Link 
                              href={`/document/${doc.id}`}
                              className="px-2 py-1 text-sm text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300"
                            >
                              보기
                            </Link>
                            <a 
                              href={doc.downloadUrl}
                              className="px-2 py-1 text-sm text-green-600 hover:text-green-800 dark:text-green-400 dark:hover:text-green-300"
                            >
                              다운로드
                            </a>
                          </>
                        )}
                        {doc.status === 'processing' && (
                          <span className="px-2 py-1 text-sm text-gray-500 dark:text-gray-400">
                            처리 중...
                          </span>
                        )}
                        {doc.status === 'failed' && (
                          <button
                            className="px-2 py-1 text-sm text-orange-600 hover:text-orange-800 dark:text-orange-400 dark:hover:text-orange-300"
                            onClick={() => alert('문서 재생성은 현재 데모에서 지원되지 않습니다.')}
                          >
                            재시도
                          </button>
                        )}
                      </div>
                    </div>
                  </li>
                ))}
              </ul>
            </div>
          ) : (
            <div className="bg-white dark:bg-gray-800 rounded-lg shadow p-8 text-center">
              <p className="text-gray-500 dark:text-gray-400">
                {searchTerm || formatFilter || statusFilter ? '검색 조건에 맞는 문서가 없습니다.' : '문서가 없습니다. 새 문서를 작성해보세요.'}
              </p>
              {(searchTerm || formatFilter || statusFilter) && (
                <button
                  onClick={() => {
                    setSearchTerm('');
                    setFormatFilter('');
                    setStatusFilter('');
                  }}
                  className="mt-4 px-4 py-2 bg-gray-200 dark:bg-gray-700 rounded-md text-sm"
                >
                  필터 초기화
                </button>
              )}
            </div>
          )}
        </div>
      </div>
    </div>
  );
} 