import { login } from './auth.js';

export async function loginPage(chatMessages, translations, reqAPI, getText, showCustomDialog) {
    // 입력 컨테이너 숨기기
    const inputContainer = document.querySelector('.input-container');
    inputContainer.style.display = 'none';

    // 페이지 기본 구조 생성
    chatMessages.innerHTML = `
        <div class="login-container">
            <div class="login-box">
                <div class="login-header">
                    <h2>${getText('auth', 'title')}</h2>
                    <p>${getText('auth', 'description')}</p>
                </div>
                
                <div class="login-form">
                    <div class="form-group">
                        <label for="username">${getText('auth', 'username') || '아이디'}</label>
                        <input type="text" id="username" class="form-input" 
                            placeholder="${getText('auth', 'usernamePlaceholder') || '아이디를 입력하세요'}">
                    </div>
                    
                    <div class="form-group">
                        <label for="password">${getText('auth', 'password')}</label>
                        <input type="password" id="password" class="form-input" 
                            placeholder="${getText('auth', 'passwordPlaceholder')}">
                    </div>

                    <!-- 오류 메시지 영역 추가 -->
                    <div id="error-message" class="error-message" style="display: none;"></div>

                    <div class="form-actions">
                        <button id="loginButton" class="btn-login">${getText('auth', 'login')}</button>
                    </div>
                </div>
            </div>
        </div>

        <style>
            .login-container {
                display: flex;
                justify-content: center;
                align-items: center;
                min-height: 100vh;
                padding: 20px;
                background: var(--bg-color);
            }

            .login-box {
                background: var(--component-bg);
                border-radius: 8px;
                padding: 40px;
                width: 100%;
                max-width: 400px;
                box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            }

            .login-header {
                text-align: center;
                margin-bottom: 30px;
            }

            .login-header h2 {
                margin: 0 0 10px 0;
                color: var(--text-color);
                font-size: 24px;
            }

            .login-header p {
                margin: 0;
                color: var(--text-secondary);
                font-size: 14px;
            }

            .login-form {
                display: flex;
                flex-direction: column;
                gap: 20px;
            }

            .form-group {
                display: flex;
                flex-direction: column;
                gap: 8px;
            }

            .form-group label {
                color: var(--text-color);
                font-size: 14px;
                font-weight: 500;
            }

            .form-input {
                padding: 10px 12px;
                border: 1px solid var(--border-color);
                border-radius: 4px;
                background: var(--input-bg);
                color: var(--text-color);
                font-size: 14px;
                transition: border-color 0.2s;
            }

            .form-input:focus {
                border-color: var(--primary-color);
                outline: none;
            }

            .form-actions {
                margin-top: 10px;
                display: flex;
                justify-content: flex-end;
            }

            .btn-login {
                background: var(--accent-color);
                color: white;
                border: none;
                border-radius: 4px;
                padding: 10px 24px;
                font-size: 14px;
                font-weight: 500;
                cursor: pointer;
                transition: all 0.2s ease;
                position: relative;
                overflow: hidden;
            }

            .btn-login:hover {
                transform: translateY(-1px);
                box-shadow: 0 2px 8px rgba(0, 0, 0, 0.15);
                filter: brightness(1.1);
            }

            .btn-login:active {
                transform: translateY(1px);
                box-shadow: none;
            }

            .error-message {
                color: var(--error-color);
                font-size: 14px;
                text-align: center;
                padding: 8px;
                margin-top: 10px;
                border-radius: 4px;
                background-color: var(--error-bg-color, rgba(255, 0, 0, 0.1));
                display: none;
            }

            .error-message.show {
                display: block;
            }

            /* 입력 필드 오류 스타일 */
            .form-input.error {
                border-color: var(--error-color);
            }
        </style>
    `;

    // 오류 메시지 표시 함수
    function showError(message) {
        const errorDiv = document.getElementById('error-message');
        errorDiv.textContent = message;
        errorDiv.style.display = 'block';
        
        // 입력 필드에 오류 스타일 적용
        document.querySelectorAll('.form-input').forEach(input => {
            input.classList.add('error');
        });
    }

    // 오류 메시지 숨기기 함수
    function hideError() {
        const errorDiv = document.getElementById('error-message');
        errorDiv.style.display = 'none';
        
        // 입력 필드의 오류 스타일 제거
        document.querySelectorAll('.form-input').forEach(input => {
            input.classList.remove('error');
        });
    }

    // 입력 필드 이벤트: 입력 시작하면 오류 메시지 숨기기
    document.querySelectorAll('.form-input').forEach(input => {
        input.addEventListener('input', hideError);
    });

    // 로그인 버튼 이벤트
    document.getElementById('loginButton').addEventListener('click', async () => {
        const username = document.getElementById('username').value;
        const password = document.getElementById('password').value;

        console.log('[DEBUG] 로그인 시도:', { username });

        if (!username || !password) {
            showError(getText('login', 'emptyFields'));
            return;
        }

        try {
            console.log('[DEBUG] 로그인 요청 전송');
            const result = await login(reqAPI, username, password);
            console.log('[DEBUG] 로그인 결과:', result);

            if (result.success && result.user) {
                console.log('[DEBUG] 로그인 성공:', result.user);
                hideError();
                await reqAPI('errnotify', getText('auth', 'loginSuccess'));

                // 사용자 역할에 따라 페이지 이동
                if (result.user.role === 'admin') {
                    console.log('[DEBUG] 관리자 권한 확인됨, 사용자 페이지로 이동');
                    // 잠시 대기하여 토큰이 저장되도록 함
                    await new Promise(resolve => setTimeout(resolve, 100));
                    
                    // 토큰이 제대로 저장되었는지 확인
                    const token = localStorage.getItem('userToken');
                    console.log('[DEBUG] 저장된 토큰 확인:', token ? '토큰 있음' : '토큰 없음');
                    
                    if (token) {
                        window.navigateToPage('user');
                    } else {
                        showError(getText('auth', 'tokenSaveError'));
                    }
                } else {
                    console.log('[DEBUG] 일반 사용자 권한:', result.user.role);
                    showError(getText('auth', 'adminRequired'));
                }
            } else {
                console.error('[DEBUG] 로그인 실패:', result.error);
                showError(result.error);
            }
        } catch (error) {
            console.error('[DEBUG] 로그인 에러:', error);
            // error.message가 객체인 경우를 처리
            let errorMessage = error?.message;
            if (typeof errorMessage === 'object') {
                errorMessage = JSON.stringify(errorMessage);
            }
            showError(errorMessage || getText('login', 'unknownError'));
        }
    });

    // 엔터 키 이벤트 처리
    const inputs = document.querySelectorAll('#username, #password');
    inputs.forEach(input => {
        input.addEventListener('keypress', async (e) => {
            if (e.key === 'Enter') {
                e.preventDefault();
                document.getElementById('loginButton').click();
            }
        });
    });
}