import fs from 'fs';
import path from 'path';
import os from 'os';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

async function setupPuppeteer() {
    try {
        console.log('Starting puppeteer setup...');
        const isGlobalInstall = process.env.npm_config_global === 'true';
        console.log('Is global install:', isGlobalInstall);
        
        // 로컬 설치인 경우에도 설정 진행
        console.log('Configuring puppeteer...');
        
        // .airun 디렉토리 생성
        const airunDir = path.join(os.homedir(), '.airun');
        console.log('Creating .airun directory at:', airunDir);
        if (!fs.existsSync(airunDir)) {
            fs.mkdirSync(airunDir, { recursive: true });
        }
        
        // config.json 파일 경로
        const configPath = path.join(airunDir, 'config.json');
        console.log('Config path:', configPath);
        
        // 기존 설정 읽기 또는 새로 생성
        let config = {};
        if (fs.existsSync(configPath)) {
            console.log('Reading existing config...');
            config = JSON.parse(fs.readFileSync(configPath, 'utf8'));
        }
        
        // Chrome 경로 설정
        let chromePath;
        switch (process.platform) {
            case 'win32':
                chromePath = 'C:\\Program Files\\Google\\Chrome\\Application\\chrome.exe';
                break;
            case 'darwin':
                chromePath = '/Applications/Google Chrome.app/Contents/MacOS/Google Chrome';
                break;
            default: // linux
                chromePath = '/usr/bin/google-chrome';
                break;
        }
        
        console.log('Setting Chrome path to:', chromePath);
        
        // 설정 업데이트
        config.CHROME_PATH = chromePath;
        
        // 설정 저장
        fs.writeFileSync(configPath, JSON.stringify(config, null, 2));
        
        console.log('Puppeteer configuration completed successfully');
    } catch (error) {
        console.error('Error during puppeteer setup:', error);
    }
}

setupPuppeteer().catch(error => {
    console.error('Fatal error during setup:', error);
    process.exit(1);
});
