// 커스텀 도구 예제들
// 이 파일을 참고하여 새로운 도구를 만들 수 있습니다.

// 예제 1: 간단한 계산기 도구
const calculatorTool = {
    name: 'calculator',
    category: 'utility',
    description: '기본적인 수학 계산을 수행합니다.',
    version: '1.0.0',
    parameters: {
        operation: { 
            type: 'string', 
            required: true, 
            description: '수행할 연산 (add, subtract, multiply, divide)',
            enum: ['add', 'subtract', 'multiply', 'divide']
        },
        a: { type: 'number', required: true, description: '첫 번째 숫자' },
        b: { type: 'number', required: true, description: '두 번째 숫자' }
    },
    handler: async (args, context) => {
        const { operation, a, b } = args;
        const { userId, toolName } = context;
        
        console.log(`[${toolName}] 계산 수행: ${a} ${operation} ${b} (사용자: ${userId})`);
        
        let result;
        switch (operation) {
            case 'add':
                result = a + b;
                break;
            case 'subtract':
                result = a - b;
                break;
            case 'multiply':
                result = a * b;
                break;
            case 'divide':
                if (b === 0) {
                    return {
                        success: false,
                        error: '0으로 나눌 수 없습니다.',
                        message: '계산 실패: 0으로 나누기'
                    };
                }
                result = a / b;
                break;
            default:
                return {
                    success: false,
                    error: `지원하지 않는 연산: ${operation}`,
                    message: '계산 실패: 잘못된 연산'
                };
        }
        
        return {
            success: true,
            data: {
                operation,
                operands: [a, b],
                result,
                calculation: `${a} ${operation} ${b} = ${result}`
            },
            message: `계산 완료: ${a} ${operation} ${b} = ${result}`
        };
    }
};

// 예제 2: 날씨 정보 도구 (모의)
const weatherTool = {
    name: 'get_weather',
    category: 'information',
    description: '특정 지역의 날씨 정보를 가져옵니다.',
    version: '1.1.0',
    parameters: {
        location: { type: 'string', required: true, description: '지역명 (예: 서울, 부산)' },
        format: { type: 'string', default: 'summary', description: '응답 형식 (summary, detailed)' }
    },
    handler: async (args, context) => {
        const { location, format = 'summary' } = args;
        const { userId, job } = context;
        
        console.log(`[Weather] 날씨 정보 요청: ${location} (형식: ${format})`);
        
        // 진행 상황 업데이트
        if (job && job.progress) {
            await job.progress(30, `${location} 날씨 정보 조회 중...`);
        }
        
        // 실제로는 날씨 API를 호출해야 하지만, 여기서는 모의 데이터 사용
        await new Promise(resolve => setTimeout(resolve, 1000)); // API 호출 시뮬레이션
        
        const mockWeatherData = {
            location,
            temperature: Math.floor(Math.random() * 30) + 5, // 5-35도
            humidity: Math.floor(Math.random() * 50) + 30, // 30-80%
            condition: ['맑음', '흐림', '비', '눈'][Math.floor(Math.random() * 4)],
            wind_speed: Math.floor(Math.random() * 20) + 5, // 5-25 km/h
            timestamp: new Date().toISOString()
        };
        
        if (job && job.progress) {
            await job.progress(100, '날씨 정보 조회 완료');
        }
        
        const responseData = format === 'detailed' ? {
            ...mockWeatherData,
            forecast: ['내일은 맑을 예정입니다.', '주말에 비가 올 수 있습니다.']
        } : {
            location: mockWeatherData.location,
            temperature: mockWeatherData.temperature,
            condition: mockWeatherData.condition
        };
        
        return {
            success: true,
            data: responseData,
            message: `${location} 날씨 정보: ${mockWeatherData.temperature}°C, ${mockWeatherData.condition}`
        };
    }
};

// 예제 3: 파일 시스템 도구
const fileSystemTool = {
    name: 'list_files',
    category: 'filesystem',
    description: '디렉토리의 파일 목록을 조회합니다.',
    version: '1.0.0',
    parameters: {
        path: { type: 'string', default: '.', description: '조회할 디렉토리 경로' },
        show_hidden: { type: 'boolean', default: false, description: '숨김 파일 표시 여부' },
        limit: { type: 'number', default: 50, description: '최대 파일 개수' }
    },
    handler: async (args, context) => {
        const { path = '.', show_hidden = false, limit = 50 } = args;
        const { userId } = context;
        
        console.log(`[FileSystem] 파일 목록 조회: ${path} (사용자: ${userId})`);
        
        try {
            const fs = await import('fs');
            const pathModule = await import('path');
            
            // 보안을 위해 경로 검증
            const safePath = pathModule.resolve(path);
            if (!safePath.includes(process.cwd())) {
                return {
                    success: false,
                    error: '접근 권한이 없는 경로입니다.',
                    message: '파일 목록 조회 실패: 권한 없음'
                };
            }
            
            const files = fs.readdirSync(safePath);
            let filteredFiles = show_hidden ? files : files.filter(f => !f.startsWith('.'));
            
            if (filteredFiles.length > limit) {
                filteredFiles = filteredFiles.slice(0, limit);
            }
            
            const fileDetails = filteredFiles.map(file => {
                const filePath = pathModule.join(safePath, file);
                const stats = fs.statSync(filePath);
                return {
                    name: file,
                    type: stats.isDirectory() ? 'directory' : 'file',
                    size: stats.size,
                    modified: stats.mtime.toISOString()
                };
            });
            
            return {
                success: true,
                data: {
                    path: safePath,
                    files: fileDetails,
                    total_count: files.length,
                    shown_count: filteredFiles.length
                },
                message: `파일 목록 조회 완료: ${filteredFiles.length}개 항목`
            };
        } catch (error) {
            return {
                success: false,
                error: error.message,
                message: `파일 목록 조회 실패: ${error.message}`
            };
        }
    }
};

// 예제 4: 데이터 변환 도구
const dataTransformTool = {
    name: 'transform_data',
    category: 'data',
    description: 'JSON 데이터를 다양한 형식으로 변환합니다.',
    version: '2.0.0',
    parameters: {
        data: { type: 'object', required: true, description: '변환할 JSON 데이터' },
        format: { 
            type: 'string', 
            required: true, 
            description: '출력 형식 (csv, xml, yaml)',
            enum: ['csv', 'xml', 'yaml']
        },
        include_headers: { type: 'boolean', default: true, description: 'CSV 헤더 포함 여부 (CSV만 해당)' }
    },
    handler: async (args, context) => {
        const { data, format, include_headers = true } = args;
        const { toolName, version } = context;
        
        console.log(`[${toolName}] v${version} 데이터 변환: ${format} 형식으로`);
        
        try {
            let result;
            
            switch (format) {
                case 'csv':
                    if (Array.isArray(data) && data.length > 0) {
                        const headers = Object.keys(data[0]);
                        let csv = include_headers ? headers.join(',') + '\n' : '';
                        csv += data.map(row => 
                            headers.map(header => `"${row[header] || ''}"`).join(',')
                        ).join('\n');
                        result = csv;
                    } else {
                        throw new Error('CSV 변환을 위해서는 객체 배열이 필요합니다.');
                    }
                    break;
                    
                case 'xml':
                    result = jsonToXml(data);
                    break;
                    
                case 'yaml':
                    // 간단한 YAML 변환 (실제로는 yaml 라이브러리 사용 권장)
                    result = jsonToYaml(data);
                    break;
                    
                default:
                    throw new Error(`지원하지 않는 형식: ${format}`);
            }
            
            return {
                success: true,
                data: {
                    original_format: 'json',
                    target_format: format,
                    result,
                    size: result.length
                },
                message: `데이터 변환 완료: JSON → ${format.toUpperCase()}`
            };
        } catch (error) {
            return {
                success: false,
                error: error.message,
                message: `데이터 변환 실패: ${error.message}`
            };
        }
    }
};

// 유틸리티 함수들
function jsonToXml(obj, rootName = 'root') {
    if (typeof obj !== 'object' || obj === null) {
        return `<${rootName}>${obj}</${rootName}>`;
    }
    
    let xml = `<${rootName}>`;
    for (const [key, value] of Object.entries(obj)) {
        if (Array.isArray(value)) {
            value.forEach(item => {
                xml += jsonToXml(item, key);
            });
        } else if (typeof value === 'object' && value !== null) {
            xml += jsonToXml(value, key);
        } else {
            xml += `<${key}>${value}</${key}>`;
        }
    }
    xml += `</${rootName}>`;
    return xml;
}

function jsonToYaml(obj, indent = 0) {
    const spaces = '  '.repeat(indent);
    let yaml = '';
    
    if (Array.isArray(obj)) {
        obj.forEach(item => {
            yaml += `${spaces}- ${typeof item === 'object' ? '\n' + jsonToYaml(item, indent + 1) : item}\n`;
        });
    } else if (typeof obj === 'object' && obj !== null) {
        Object.entries(obj).forEach(([key, value]) => {
            if (typeof value === 'object' && value !== null) {
                yaml += `${spaces}${key}:\n${jsonToYaml(value, indent + 1)}`;
            } else {
                yaml += `${spaces}${key}: ${value}\n`;
            }
        });
    }
    
    return yaml;
}

// 도구들을 배열로 export - AIRUN 시스템에서 사용하지 않도록 모두 비활성화
const exampleTools = [
    // calculatorTool, // 비활성화
    // weatherTool, // 비활성화
    // fileSystemTool, // 비활성화
    // dataTransformTool // 비활성화
];

export default exampleTools;
export {
    // calculatorTool, // 비활성화
    // weatherTool, // 비활성화
    // fileSystemTool, // 비활성화
    // dataTransformTool // 비활성화
}; 