"use client";

import React, { useState, useEffect, useRef, useCallback } from "react";
import { getRagServerUrl, getApiServerUrl } from "@/config/serverConfig";
import { externalApiClient, setConfig } from "@/lib/apiClient";
import { useAuth } from "@/hooks/useAuth";
import { CustomMarkdownRenderer } from "@/components/CustomMarkdownRenderer";
import { ChevronDownIcon, ChevronRightIcon } from "@heroicons/react/24/outline";
import { XMarkIcon } from "@heroicons/react/24/solid";
import FileManager, { IFileManagerRef } from "@nolesh/react-file-manager";
import "@nolesh/react-file-manager/dist/styles.css";
import { Tree, NodeRendererProps } from "react-arborist";
import { BackgroundJobStatus } from "@/components/BackgroundJobStatus";
import type { BackgroundJob } from "@/types/support";
import ExcelPreprocessModal from "@/components/rag/ExcelPreprocessModal";
import PdfPreprocessModal from "@/components/rag/PdfPreprocessModal";
import GraphRAGModal from "@/components/GraphRAGModal";
import GraphRAGFilePreviewModal from "@/components/GraphRAGFilePreviewModal";

// GraphRAG 활성화 상태를 전역적으로 관리 (FileTreeNode에서 참조하기 위함)
let globalGraphRAGEnabled = false;

// @nolesh/react-file-manager를 위한 파일 데이터 인터페이스
interface IRemoteFileData {
  fileName: string;
  fileSize: number;
  fileType?: string;
  uid?: string;
  description?: string;
  previewData?: {
    src?: string;
  };
  readOnly?: boolean;
  disabled?: boolean;
  path?: string;
  userId?: string;
}

// RagFile 타입(chat/page.tsx와 동일)
interface RagFile {
  name: string;
  size: number;
  modified?: string;
  type?: string;
  status: "uploading" | "processing" | "completed" | "error";
  progress?: number;
  error?: string;
}

// 타입 정의
interface RagSettingsPanelProps {
  onChange: (settings: Record<string, any>) => void;
  settings: Record<string, any>;
}

// 백그라운드 작업 타입
interface BackgroundJobType {
  id: string;
  type: "file_upload" | "ticket_sync";
  status: "pending" | "running" | "completed" | "failed";
  progress: number;
  message: string;
  filename: string;
  createdAt: string;
  updatedAt: string;
  userId: string;
  error?: string;
  metadata?: {
    [key: string]: unknown;
  };
}

interface RagFileUploadAndProgressProps {
  onUpload: (file: File, targetUserId?: string) => Promise<void>;
  progress: any;
  collection: any;
}

interface RagChunkPreviewAndChatProps {
  chunks: any[];
  onChat: (message: string) => void;
  chatHistory: any[];
}

interface RagSidebarProps {
  settings: Record<string, any>;
  onSettingsChange: (settings: Record<string, any>) => void;
  collection: any;
  onSelectDoc: (doc: any) => void;
  selectedDoc: string | null;
  selectedFolder?: any;
  setSelectedFolder?: (folder: any) => void;
  onFolderSelect?: (folder: any) => void;
  chunks: any[];
  currentChunkIndex: number;
  setCurrentChunkIndex: (idx: number) => void;
  onPrevChunk: () => void;
  onNextChunk: () => void;
  user?: any;
  onJobComplete?: (job: BackgroundJob) => void;
  refreshTrigger?: number;
  onFileUpload?: (file: File, targetUserId?: string) => Promise<void>;
  onShowExcelModal?: () => void;
  onShowPdfModal?: () => void;
  onGraphRAGView?: (doc: any) => void;
  onFullSystemInit?: () => void;
  onDocumentInit?: (docName: string, userId: string) => void;
  onFolderInit?: (userId: string) => void;
}

interface RagMainPanelProps {
  progress: any;
  chunks: any[];
  selectedDoc: string | null;
  currentChunkIndex: number;
  setCurrentChunkIndex: (idx: number) => void;
  onPrevChunk: () => void;
  onNextChunk: () => void;
  showGraphRAGModal?: boolean;
  setShowGraphRAGModal?: (show: boolean) => void;
  setCurrentGraphQuery?: (query: string) => void;
  getInitButtonText?: () => string;
  isInitializing?: boolean;
  setIsInitializing?: (value: boolean) => void;
  documentStats: {
    total: number;
    completed: number;
    processing: number;
  };
}

interface RagChatPanelProps {
  chatHistory: any[];
  onChat: (message: string, searchScope?: string) => void;
  selectedChunk: any;
  provider: string;
  model: string;
  temperature: number;
  providerOptions: { value: string; label: string }[];
  modelOptions: { value: string; label: string }[];
  onProviderChange: (provider: string) => void;
  onModelChange: (model: string) => void;
  onTemperatureChange: (temperature: number) => void;
  ragSearching?: boolean;
  aiResponding?: boolean;
}

interface RagChunksResponse {
  success: boolean;
  chunks: any[];
}

// --- RAG 설정 메타/키 목록 ---
const ragConfigKeys = [
  "RAG_DOCS_PATH",
  "RAG_MODE",
  "RAG_CHUNKING_STRATEGY",
  "RAG_EMBEDDING_MODEL",
  "RAG_PDF_BACKEND",
  "RAG_FORCE_PDF_CONVERSION",
  "RAG_PROCESS_TEXT_ONLY",
  "RAG_DETECT_COLUMN_TYPE",
  "RAG_SEMANTIC_CHUNKER",
  "RAG_LARGE_CONTEXT_CHUNKING",
  "RAG_SENTENCE_TRANSFORMER_MODEL",
  "RAG_CHUNK_OVERLAP",
  "RAG_CHUNK_SIZE",
  "RAG_FORCE_CPU",
  "RAG_GENERATE_IMAGE_EMBEDDINGS",
  "USE_ENHANCED_EMBEDDING",
  "ENHANCED_EMBEDDING_STRATEGY",
  // Graph RAG 설정 (기존 설정 상속)
  "RAG_ENABLE_GRAPHRAG",
  "RAG_SEARCH_MODE",
  "RAG_GRAPHRAG_ENTITY_TYPES",
  "RAG_GRAPHRAG_MAX_GLEANINGS",
  "RAG_GRAPHRAG_COMMUNITY_DETECTION",
  // 검색 품질 설정
  "RAG_TOP_K",
  "RAG_KEYWORD_PRECISION_THRESHOLD",
  "RAG_SEMANTIC_RELEVANCE_THRESHOLD",
  "RAG_CONTENT_QUALITY_THRESHOLD",
  "RAG_RECENCY_THRESHOLD",
  // 검색 점수 가중치 설정
  "RAG_KEYWORD_PRECISION_WEIGHT",
  "RAG_SEMANTIC_RELEVANCE_WEIGHT",
  "RAG_CONTENT_QUALITY_WEIGHT",
  "RAG_RECENCY_WEIGHT"
];

// RAG 설정 그룹 정의 - 문서처리와 검색으로 분리
const ragSettingGroups = {
  // 🔄 문서 처리 설정 그룹
  documentProcessing: {
    title: "🔄 문서 처리 설정 (Document Processing)",
    description: "문서 업로드 시 처리 방식과 임베딩 생성 관련 설정",
    icon: "🔄",
    keys: [
      "RAG_DOCS_PATH",
      "RAG_MODE",
      "RAG_PROCESS_TEXT_ONLY",
      "RAG_GENERATE_IMAGE_EMBEDDINGS",
      "RAG_CHUNKING_STRATEGY",
      "USE_ENHANCED_EMBEDDING",
      "ENHANCED_EMBEDDING_STRATEGY",
      "RAG_CHUNK_SIZE",
      "RAG_CHUNK_OVERLAP",
      "RAG_SEMANTIC_CHUNKER",
      "RAG_LARGE_CONTEXT_CHUNKING",
      "RAG_PDF_BACKEND",
      "RAG_FORCE_PDF_CONVERSION",
      "RAG_DETECT_COLUMN_TYPE",
      "RAG_EMBEDDING_MODEL",
      "RAG_FORCE_CPU",
      "RAG_ENABLE_GRAPHRAG",
      "RAG_GRAPHRAG_ENTITY_TYPES",
      "RAG_GRAPHRAG_MAX_GLEANINGS",
      "RAG_GRAPHRAG_COMMUNITY_DETECTION"
    ]
  },
  // 🔍 문서 검색 설정 그룹
  documentSearch: {
    title: "🔍 문서 검색 설정 (Document Search)",
    description: "사용자 질문 시 문서 검색 방식과 품질 관련 설정",
    icon: "🔍",
    keys: ["RAG_SEARCH_MODE", "RAG_TOP_K", "RAG_KEYWORD_PRECISION_THRESHOLD", "RAG_SEMANTIC_RELEVANCE_THRESHOLD", "RAG_CONTENT_QUALITY_THRESHOLD", "RAG_RECENCY_THRESHOLD", "RAG_KEYWORD_PRECISION_WEIGHT", "RAG_SEMANTIC_RELEVANCE_WEIGHT", "RAG_CONTENT_QUALITY_WEIGHT", "RAG_RECENCY_WEIGHT"]
  }
};

// 메타데이터에 step 정보 추가
const ragConfigMeta: Record<
  string,
  {
    label: string;
    type: "string" | "number" | "boolean" | "select";
    step?: number;
    min?: number;
    max?: number;
    options?: string[];
  }
> = {
  // 임베딩/청크 설정
  RAG_DOCS_PATH: { label: "문서 저장 경로", type: "string" },
  RAG_MODE: { label: "처리 모드", type: "string" },
  RAG_CHUNKING_STRATEGY: { label: "청크 전략", type: "string" },
  RAG_PROCESS_TEXT_ONLY: { label: "텍스트 우선", type: "boolean" },
  RAG_PDF_BACKEND: { label: "PDF 처리 엔진", type: "string" },
  RAG_FORCE_PDF_CONVERSION: { label: "강제 PDF 변환", type: "boolean" },
  RAG_CHUNK_SIZE: { label: "청크 크기", type: "number", step: 100, min: 100, max: 10000 },
  RAG_CHUNK_OVERLAP: { label: "청크 중첩", type: "number", step: 10, min: 0, max: 1000 },

  RAG_SEMANTIC_CHUNKER: { label: "시맨틱 청킹", type: "boolean" },
  RAG_LARGE_CONTEXT_CHUNKING: { label: "파일 전체 청크 반환", type: "boolean" },
  RAG_EMBEDDING_MODEL: { label: "임베딩 모델", type: "string" },
  RAG_DETECT_COLUMN_TYPE: { label: "컬럼 타입 감지", type: "boolean" },

  RAG_FORCE_CPU: { label: "CPU 강제 사용", type: "boolean" },
  USE_ENHANCED_EMBEDDING: { label: "예상 질문 AI 생성", type: "boolean" },
  ENHANCED_EMBEDDING_STRATEGY: { label: "예상 질문 검색 방식", type: "select", options: ["original", "questions", "hybrid"] },
  RAG_GENERATE_IMAGE_EMBEDDINGS: { label: "이미지 임베딩 생성", type: "boolean" },

  // Graph RAG 설정 추가 (기본 RAG 설정을 fallback으로 사용)
  RAG_ENABLE_GRAPHRAG: { label: "Graph RAG 활성화", type: "boolean" },
  RAG_SEARCH_MODE: { label: "검색 모드", type: "select", options: ["vector", "graph", "hybrid"] },
  RAG_GRAPHRAG_ENTITY_TYPES: { label: "엔티티 타입", type: "string" },
  RAG_GRAPHRAG_MAX_GLEANINGS: { label: "최대 정보 수집 횟수", type: "number", step: 1, min: 1, max: 5 },
  RAG_GRAPHRAG_COMMUNITY_DETECTION: { label: "커뮤니티 감지", type: "boolean" },

  // 검색 품질 임계값 설정 (0~1)
  RAG_KEYWORD_PRECISION_THRESHOLD: { label: "키워드 정확도 임계값", type: "number", step: 0.05, min: 0, max: 1 },
  RAG_SEMANTIC_RELEVANCE_THRESHOLD: { label: "의미 관련성 임계값", type: "number", step: 0.05, min: 0, max: 1 },
  RAG_CONTENT_QUALITY_THRESHOLD: { label: "내용 품질 임계값", type: "number", step: 0.05, min: 0, max: 1 },
  RAG_RECENCY_THRESHOLD: { label: "최신성 임계값", type: "number", step: 0.05, min: 0, max: 1 },

  // 검색 범위 설정
  RAG_TOP_K: { label: "검색 결과 개수", type: "number", step: 1, min: 1, max: 100 },

  // 검색 점수 가중치 설정 (합계 1.0 권장)
  RAG_KEYWORD_PRECISION_WEIGHT: { label: "키워드 정확도 가중치", type: "number", step: 0.05, min: 0, max: 1 },
  RAG_SEMANTIC_RELEVANCE_WEIGHT: { label: "의미 관련성 가중치", type: "number", step: 0.05, min: 0, max: 1 },
  RAG_CONTENT_QUALITY_WEIGHT: { label: "내용 품질 가중치", type: "number", step: 0.05, min: 0, max: 1 },
  RAG_RECENCY_WEIGHT: { label: "최신성 가중치", type: "number", step: 0.05, min: 0, max: 1 }
};

function parseBool(val: string | undefined): boolean {
  return /^(yes|true|1)$/i.test(String(val).trim());
}

// --- Embedding/Retriever 그룹 분리 ---
const embeddingKeys = ["RAG_DOCS_PATH", "RAG_MODE", "RAG_CHUNKING_STRATEGY", "RAG_PROCESS_TEXT_ONLY", "RAG_GENERATE_IMAGE_EMBEDDINGS", "USE_ENHANCED_EMBEDDING", "ENHANCED_EMBEDDING_STRATEGY", "RAG_CHUNK_SIZE", "RAG_CHUNK_OVERLAP", "RAG_SEMANTIC_CHUNKER", "RAG_PDF_BACKEND", "RAG_FORCE_PDF_CONVERSION", "RAG_DETECT_COLUMN_TYPE", "RAG_LARGE_CONTEXT_CHUNKING", "RAG_FORCE_CPU", "RAG_EMBEDDING_MODEL"];
const retrieverKeys = ["RAG_TOP_K"];

// --- RagSettingsPanel 구현 ---
function RagSettingsPanel({ onChange, settings }: RagSettingsPanelProps) {
  const [edited, setEdited] = useState<Record<string, string>>({});
  const [isSaving, setIsSaving] = useState(false);
  const [modal, setModal] = useState<{ type: "success" | "error"; title: string; message: string } | null>(null);
  const [openGroups, setOpenGroups] = useState<Record<string, boolean>>(() => Object.keys(ragSettingGroups).reduce((acc, key) => ({ ...acc, [key]: true }), {}));
  const [showResetConfirm, setShowResetConfirm] = useState(false);

  const toggleGroup = (groupKey: string) => {
    setOpenGroups((prev) => ({ ...prev, [groupKey]: !prev[groupKey] }));
  };

  useEffect(() => {
    // settings가 실제로 로드된 경우에만 edited 상태를 동기화
    console.log("📝 RagSettingsPanel settings 변경 감지:", settings);

    // settings가 비어있으면 아직 로딩 중이므로 동기화하지 않음
    if (!settings || Object.keys(settings).length === 0) {
      console.log("📝 RagSettingsPanel: settings가 비어있어 동기화 건너뜀");
      return;
    }

    setEdited((prevEdited) => {
      // 새로운 settings로 업데이트하되, 사용자가 편집 중인 값은 보존
      const updated = { ...settings };

      // 사용자가 편집 중인 값이 있다면 우선적으로 적용
      Object.keys(prevEdited).forEach((key) => {
        if (prevEdited[key] !== undefined && prevEdited[key] !== null && prevEdited[key] !== "") {
          // 설정값과 다른 경우에만 편집 중인 값을 유지 (사용자가 변경한 것)
          if (String(prevEdited[key]) !== String(settings[key])) {
            updated[key] = prevEdited[key];
          }
        }
      });

      console.log("📝 RagSettingsPanel edited 상태 업데이트:", { prev: prevEdited, settings, updated });
      return updated;
    });
  }, [settings]);

  const handleChange = (key: string, value: string) => {
    setEdited((prev) => ({ ...prev, [key]: value }));

    // RAG 모드 변경 시 관련 옵션들을 자동으로 설정
    if (key === "RAG_MODE") {
      const autoSettings: Record<string, string> = {};

      if (value === "fast") {
        // Fast 모드: 청크전략: default, 텍스트 우선: 활성화, 이미지 임베딩 생성: 비활성화, 예상질문 AI생성: 비활성화
        autoSettings["RAG_CHUNKING_STRATEGY"] = "default";
        autoSettings["RAG_PROCESS_TEXT_ONLY"] = "yes";
        autoSettings["RAG_GENERATE_IMAGE_EMBEDDINGS"] = "no";
        autoSettings["USE_ENHANCED_EMBEDDING"] = "no";
      } else if (value === "normal") {
        // Normal 모드: 청크전략: page, 텍스트 우선: 활성화, 이미지 임베딩 생성: 활성화, 예상질문 AI생성: 비활성화
        autoSettings["RAG_CHUNKING_STRATEGY"] = "page";
        autoSettings["RAG_PROCESS_TEXT_ONLY"] = "yes";
        autoSettings["RAG_GENERATE_IMAGE_EMBEDDINGS"] = "yes";
        autoSettings["USE_ENHANCED_EMBEDDING"] = "no";
      } else if (value === "rich") {
        // Rich 모드: 청크전략: default, 텍스트 우선: 비활성화, 이미지 임베딩 생성: 활성화, 예상질문 AI생성: 활성화
        autoSettings["RAG_CHUNKING_STRATEGY"] = "default";
        autoSettings["RAG_PROCESS_TEXT_ONLY"] = "no";
        autoSettings["RAG_GENERATE_IMAGE_EMBEDDINGS"] = "yes";
        autoSettings["USE_ENHANCED_EMBEDDING"] = "yes";
      }

      // 자동 설정된 값들을 edited 상태에 추가
      setEdited((prev) => ({ ...prev, ...autoSettings }));

      // 사용자에게 자동 설정 알림
      const autoSettingsList = Object.entries(autoSettings)
        .map(([k, v]) => `${ragConfigMeta[k]?.label || k}: ${v === "yes" ? "활성화" : v === "no" ? "비활성화" : v}`)
        .join(", ");

      console.log(`🔄 RAG 모드 ${value}로 변경됨. 자동 설정: ${autoSettingsList}`);
    }
  };

  const handleSave = async () => {
    setIsSaving(true);
    setModal(null);
    try {
      const changed: Record<string, any> = {};
      for (const key of Object.keys(edited)) {
        const newValue = edited[key];
        const currentValue = settings[key];
        if (String(currentValue) !== String(newValue) && newValue !== undefined && newValue !== "") {
          const meta = ragConfigMeta[key];
          if (meta?.type === "boolean") {
            changed[key] = parseBool(newValue) ? "yes" : "no";
          } else if (meta?.type === "number") {
            changed[key] = Number(newValue);
          } else {
            changed[key] = newValue;
          }
        }
      }

      if (Object.keys(changed).length === 0) {
        setModal({ type: "success", title: "알림", message: "변경된 설정이 없습니다." });
        return;
      }

      // 순차적으로 설정 저장 (동시성 문제 방지)
      const failedKeys: string[] = [];

      for (const [key, value] of Object.entries(changed)) {
        try {
          await setConfig(key, value);
          console.log(`✅ RAG 설정 ${key} 저장 성공:`, value);
        } catch (error) {
          console.error(`❌ RAG 설정 ${key} 저장 실패:`, error);
          failedKeys.push(key);
        }
        // 각 설정 저장 후 잠시 대기
        await new Promise((resolve) => setTimeout(resolve, 50));
      }

      if (failedKeys.length > 0) {
        setModal({ type: "error", title: "오류", message: `다음 설정 저장 실패: ${failedKeys.join(", ")}` });
        return;
      }

      // 기존 설정과 변경사항만 병합 (전체 재로드 대신)
      const updatedSettings = { ...settings, ...changed };
      onChange(updatedSettings);
      // edited 상태도 기존 값들을 보존하면서 업데이트
      setEdited((prev) => ({ ...prev, ...changed }));

      setModal({ type: "success", title: "성공", message: `설정이 저장되었습니다. (${Object.keys(changed).length}개 항목)` });
    } catch (e) {
      console.error("설정 저장 오류:", e);
      setModal({ type: "error", title: "오류", message: "설정 저장에 실패했습니다." });
    } finally {
      setIsSaving(false);
    }
  };

  const handleResetDefaults = async () => {
    setShowResetConfirm(false);
    try {
      // RAG 설정 페이지에서 실제로 표시되는 키들만 추출
      const ragSettingKeys = Object.values(ragSettingGroups).flatMap((group) => group.keys);

      const res = await externalApiClient.get("/config/defaults");
      if (res.success && res.data) {
        const allDefaults = res.data;

        // 페이지에 표시되는 RAG 키들의 기본값만 추출
        const ragDefaults = ragSettingKeys.reduce((acc, key) => {
          if (allDefaults.hasOwnProperty(key)) {
            acc[key] = String(allDefaults[key]);
          }
          return acc;
        }, {} as Record<string, string>);

        // 기본값을 찾지 못한 키들에 대한 fallback 기본값 (4-Score System)
        const fallbackDefaults: Record<string, string> = {
          RAG_KEYWORD_PRECISION_THRESHOLD: "0.1",
          RAG_SEMANTIC_RELEVANCE_THRESHOLD: "0.15",
          RAG_CONTENT_QUALITY_THRESHOLD: "0.3",
          RAG_RECENCY_THRESHOLD: "0.0",
          RAG_TOP_K: "5"
        };

        // 4-Score 가중치 기본값 추가 (균형적인 단일 세트)
        const weightDefaults: Record<string, string> = {
          RAG_KEYWORD_PRECISION_WEIGHT: "0.35", // 키워드 정확도
          RAG_SEMANTIC_RELEVANCE_WEIGHT: "0.35", // 의미 관련성
          RAG_CONTENT_QUALITY_WEIGHT: "0.2", // 내용 품질
          RAG_RECENCY_WEIGHT: "0.1" // 최신성
        };

        // 기본값이 없는 키들에 fallback 적용 (페이지에 표시되는 키들만)
        ragSettingKeys.forEach((key) => {
          if (!ragDefaults[key]) {
            if (fallbackDefaults[key]) {
              ragDefaults[key] = fallbackDefaults[key];
            } else if (weightDefaults[key]) {
              ragDefaults[key] = weightDefaults[key];
            }
          }
        });

        // 1. 로컬 상태 업데이트 (RAG 설정만, 기존 설정 보존)
        setEdited((prev) => ({ ...prev, ...ragDefaults }));
        // 기존 설정에서 RAG 설정만 업데이트 (다른 설정은 유지)
        const updatedSettings = { ...settings, ...ragDefaults };
        onChange(updatedSettings);

        // 2. 서버에 개별 키만 저장 (RAG 설정만, 순차 저장)
        const failedKeys: string[] = [];

        for (const [key, value] of Object.entries(ragDefaults)) {
          try {
            await setConfig(key, value);
            console.log(`✅ RAG 기본값 ${key} 저장 성공:`, value);
          } catch (error) {
            console.error(`❌ RAG 기본값 ${key} 저장 실패:`, error);
            failedKeys.push(key);
          }
          // 각 설정 저장 후 잠시 대기
          await new Promise((resolve) => setTimeout(resolve, 50));
        }

        if (failedKeys.length > 0) {
          setModal({ type: "error", title: "오류", message: `다음 설정 초기화 실패: ${failedKeys.join(", ")}` });
          return;
        }

        setModal({ type: "success", title: "성공", message: `RAG 설정을 기본값으로 초기화하고 저장했습니다! (${Object.keys(ragDefaults).length}개 설정)` });
      } else {
        setModal({ type: "error", title: "오류", message: `기본 설정 로드 실패: ${res.error?.message || "알 수 없는 오류"}` });
      }
    } catch (e: any) {
      console.error("기본 설정 리셋 오류:", e);
      setModal({ type: "error", title: "오류", message: `기본 설정 초기화 중 오류: ${e.message}` });
    }
  };

  // RAG 모드에 따라 자동 설정되는 옵션인지 확인
  const isAutoConfigured = (key: string): boolean => {
    const currentRagMode = (edited["RAG_MODE"] || settings["RAG_MODE"] || "fast") as "fast" | "normal" | "rich";
    const autoConfiguredKeys: Record<"fast" | "normal" | "rich", string[]> = {
      fast: ["RAG_CHUNKING_STRATEGY", "RAG_PROCESS_TEXT_ONLY", "RAG_GENERATE_IMAGE_EMBEDDINGS", "USE_ENHANCED_EMBEDDING"],
      normal: ["RAG_CHUNKING_STRATEGY", "RAG_PROCESS_TEXT_ONLY", "RAG_GENERATE_IMAGE_EMBEDDINGS", "USE_ENHANCED_EMBEDDING"],
      rich: ["RAG_CHUNKING_STRATEGY", "RAG_PROCESS_TEXT_ONLY", "RAG_GENERATE_IMAGE_EMBEDDINGS", "USE_ENHANCED_EMBEDDING"]
    };
    return autoConfiguredKeys[currentRagMode]?.includes(key) || false;
  };

  // 입력폼 렌더
  const renderInput = (key: string, value: string, meta: { label: string; type: string }) => {
    const isAuto = isAutoConfigured(key);

    // 디버깅: 실제 전달되는 value 확인
    if (["RAG_MODE", "RAG_SEARCH_MODE", "RAG_CHUNKING_STRATEGY", "RAG_PROCESS_TEXT_ONLY", "RAG_GENERATE_IMAGE_EMBEDDINGS", "USE_ENHANCED_EMBEDDING"].includes(key)) {
      console.log(`🔍 renderInput ${key}:`, { value, type: typeof value, isAuto, meta });
    }

    switch (meta.type) {
      case "boolean":
        return (
          <div className="relative">
            <select
              value={parseBool(value) ? "true" : "false"}
              onChange={(e) => handleChange(key, e.target.value === "true" ? "yes" : "no")}
              className={`w-full border rounded px-1 py-1 transition text-sm ${isAuto ? "border-blue-300 dark:border-blue-600 bg-blue-50 dark:bg-blue-800/60 text-blue-800 dark:text-blue-200" : "border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100"}`}
            >
              <option value="true">활성화</option>
              <option value="false">비활성화</option>
            </select>
            {isAuto && <div className="absolute -top-2 -right-2 bg-blue-500 text-white text-xs px-1.5 py-0.5 rounded-full">자동</div>}
          </div>
        );
      case "number":
        // 0.01 단위로 조정할 키
        const floatStepKeys = ["RAG_SIMILARITY_THRESHOLD"];
        // 최소값 0, 최대값 1 (임계값류), 나머지는 1~10000 등 자유
        const isFloat = floatStepKeys.includes(key);
        const groupMeta = Object.values(ragSettingGroups).find((g) => g.keys.includes(key)) as any;
        const keyMeta = ragConfigMeta[key];
        const step = keyMeta?.step || groupMeta?.step || (isFloat ? 0.01 : 1);
        const min = keyMeta?.min ?? groupMeta?.min ?? (isFloat ? 0 : 1);
        const max = keyMeta?.max ?? groupMeta?.max ?? (isFloat ? 1 : undefined);

        return <input type="number" value={value ?? ""} onChange={(e) => handleChange(key, e.target.value)} step={step} min={min} max={max} className="w-full border border-gray-300 dark:border-gray-600 rounded px-1 py-1 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 transition text-sm" />;
      case "string":
        if (key === "RAG_PDF_BACKEND") {
          const options = ["pypdf2", "pdfplumber", "docling", "pymupdf", "unstructured"];
          return (
            <select value={value ?? ""} onChange={(e) => handleChange(key, e.target.value)} className="w-full border border-gray-300 dark:border-gray-600 rounded px-1 py-1 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 transition text-sm">
              {options.map((opt) => (
                <option key={opt} value={opt}>
                  {opt}
                </option>
              ))}
            </select>
          );
        }
        if (key === "RAG_MODE") {
          const options = ["fast", "normal", "rich"];
          return (
            <select value={value ?? ""} onChange={(e) => handleChange(key, e.target.value)} className="w-full border border-gray-300 dark:border-gray-600 rounded px-1 py-1 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 transition text-sm">
              {options.map((opt) => (
                <option key={opt} value={opt}>
                  {opt}
                </option>
              ))}
            </select>
          );
        }
        if (key === "RAG_CHUNKING_STRATEGY") {
          const options = ["default", "semantic", "page"];
          return (
            <div className="relative">
              <select value={value ?? ""} onChange={(e) => handleChange(key, e.target.value)} className={`w-full border rounded px-1 py-1 transition text-sm ${isAuto ? "border-blue-300 dark:border-blue-600 bg-blue-50 dark:bg-blue-800/60 text-blue-800 dark:text-blue-200" : "border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100"}`}>
                {options.map((opt) => (
                  <option className="px-1 py-1" key={opt} value={opt}>
                    {opt}
                  </option>
                ))}
              </select>
              {isAuto && <div className="absolute -top-2 -right-2 bg-blue-500 text-white text-xs px-1.5 py-0.5 rounded-full">자동</div>}
            </div>
          );
        }
        return <input type="text" value={value ?? ""} onChange={(e) => handleChange(key, e.target.value)} className="w-full border border-gray-300 dark:border-gray-600 rounded px-1 py-1 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 transition text-sm" />;
      case "select":
        const selectMeta = ragConfigMeta[key];
        const options = selectMeta?.options || [];
        return (
          <div className="relative">
            <select value={value ?? ""} onChange={(e) => handleChange(key, e.target.value)} className={`w-full border rounded px-1 py-1 transition text-sm ${isAuto ? "border-blue-300 dark:border-blue-600 bg-blue-50 dark:bg-blue-800/60 text-blue-800 dark:text-blue-200" : "border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-800 text-gray-900 dark:text-blue-200"}`}>
              {options.map((opt) => (
                <option key={opt} value={opt}>
                  {opt}
                </option>
              ))}
            </select>
            {isAuto && <div className="absolute -top-2 -right-2 bg-blue-500 text-white text-xs px-1 py-1 rounded-full">자동</div>}
          </div>
        );
      default:
        return null;
    }
  };

  return (
    <div className="p-2">
      {/* RAG 모드 자동 설정 안내 */}
      {edited["RAG_MODE"] && (
        <div className="mb-4 p-3 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-700 rounded-lg">
          <div className="flex items-start gap-2">
            <div className="text-blue-600 dark:text-blue-400 mt-0.5">
              <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
            </div>
            <div className="text-sm">
              <p className="font-medium text-blue-800 dark:text-blue-200 mb-1">
                RAG 모드: <span className="font-bold">{edited["RAG_MODE"].toUpperCase()}</span>
              </p>
              <p className="text-blue-700 dark:text-blue-300 text-sm">
                {edited["RAG_MODE"] === "fast" && "빠른 텍스트 처리 모드"}
                {edited["RAG_MODE"] === "normal" && "본문과 페이지 정보를 포함"}
                {edited["RAG_MODE"] === "rich" && "AI 이미지 설명과 예상질문 자동생성을 포함."}
              </p>
            </div>
          </div>
        </div>
      )}

      {/* 각 설정 그룹 렌더링 */}
      {Object.entries(ragSettingGroups).map(([groupKey, group]) => {
        const weightSum = ["scoreWeights"].includes(groupKey) ? group.keys.reduce((sum, key) => sum + (Number(edited[key]) || 0), 0) : null;

        const isWeightGroup = (group: any): group is { totalSum: number } => "totalSum" in group && typeof group.totalSum === "number";

        return (
          <div key={groupKey} className="mb-6 border rounded-lg overflow-hidden">
            <button onClick={() => toggleGroup(groupKey)} className={`submenu-header ${openGroups[groupKey] ? "expanded" : ""}`}>
              <div className="flex-1">
                <h4>{group.title}</h4>
                <p>{group.description}</p>
              </div>
              <svg className={`toggle-arrow ${openGroups[groupKey] ? "rotate-180" : ""}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
              </svg>
            </button>

            {openGroups[groupKey] && (
              <div className="p-4 space-y-3">
                {/* 그룹의 입력 필드들 */}
                {group.keys.map((key) => {
                  const meta = ragConfigMeta[key];
                  const value = edited[key] !== undefined ? edited[key] : settings[key];
                  const isWeightGroup = ["mainWeights", "subWeights"].includes(groupKey);

                  return (
                    <div key={key} className="flex flex-col">
                      <div className="flex items-center gap-2">
                        <label className="text-sm min-w-[100px]">{meta?.label}</label>
                        <div className="flex-1 text-sm">{renderInput(key, value, meta)}</div>
                      </div>
                    </div>
                  );
                })}
              </div>
            )}
          </div>
        );
      })}

      {/* 저장 버튼 */}
      <div className="mt-6 flex gap-2">
        <button onClick={handleSave} disabled={isSaving} className="flex-1 px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg disabled:opacity-50 transition-colors duration-200 font-medium shadow-lg hover:shadow-xl transition-all duration-200 transform hover:-translate-y-0.5">
          {isSaving ? "저장 중..." : "설정 저장"}
        </button>
        <button onClick={() => setShowResetConfirm(true)} disabled={isSaving} className="flex-1 px-4 py-2 bg-gray-500 hover:bg-gray-600 dark:bg-gray-600 dark:hover:bg-gray-700 text-white rounded-lg disabled:opacity-50 transition-colors duration-200 font-medium shadow-lg hover:shadow-xl transition-all duration-200 transform hover:-translate-y-0.5">
          기본값으로 초기화
        </button>
      </div>

      {/* 결과 모달 (기존 코드 유지) */}
      {modal && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/30">
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow-lg p-6 min-w-[280px] max-w-xs flex flex-col items-center">
            <div className={`text-lg font-bold mb-2 ${modal.type === "success" ? "text-green-600" : "text-red-500"}`}>{modal.title}</div>
            <div className="mb-4 text-center text-base text-gray-700 dark:text-gray-200">
              <div>{modal.message}</div>
            </div>
            <button
              className="mt-2 px-4 py-2 rounded-lg transition-colors duration-200 font-medium text-white"
              style={{
                backgroundColor: "var(--neutral-600)"
              }}
              onMouseEnter={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = "var(--neutral-700)")}
              onMouseLeave={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = "var(--neutral-600)")}
              onClick={() => setModal(null)}
              autoFocus
            >
              닫기
            </button>
          </div>
        </div>
      )}

      {/* Reset confirmation modal */}
      {showResetConfirm && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/30">
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow-lg p-6 min-w-[280px] max-w-xs flex flex-col items-center">
            <h3 className="text-lg font-bold mb-4">설정 초기화</h3>
            <div className="mb-4 text-center text-base text-gray-700 dark:text-gray-200">RAG 설정을 기본값으로 되돌리시겠습니까? 변경사항은 저장 버튼을 눌러야 적용됩니다.</div>
            <div className="flex w-full gap-2">
              <button className="flex-1 mt-2 px-4 py-2 bg-gray-500 hover:bg-gray-600 dark:bg-gray-600 dark:hover:bg-gray-700 text-white rounded-lg transition-colors duration-200 font-medium" onClick={() => setShowResetConfirm(false)}>
                취소
              </button>
              <button className="flex-1 mt-2 px-4 py-2 bg-red-600 hover:bg-red-700 text-white rounded-lg transition-colors duration-200 font-medium" onClick={handleResetDefaults} autoFocus>
                초기화
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}

// 문서 업로드 및 진행상황 컴포넌트(더미)
function RagFileUploadAndProgress({ onUpload, progress, collection }: RagFileUploadAndProgressProps) {
  const [dragActive, setDragActive] = useState(false);
  const [uploadingFiles, setUploadingFiles] = useState<string[]>([]);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const { user } = useAuth();

  // 폴링 로직 제거 - 사이드바에서 임베딩 상태로 대체

  const handleDrag = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (e.type === "dragenter" || e.type === "dragover") {
      setDragActive(true);
    } else if (e.type === "dragleave") {
      setDragActive(false);
    }
  };

  const handleDrop = async (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setDragActive(false);

    const files = Array.from(e.dataTransfer.files);
    await processFiles(files);
  };

  const handleFileChange = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = Array.from(e.target.files || []);
    await processFiles(files);

    // 파일 입력 초기화 (같은 파일을 다시 선택할 수 있도록)
    if (fileInputRef.current) {
      fileInputRef.current.value = "";
    }
  };

  const processFiles = async (files: File[]) => {
    if (!user?.username) return;

    for (const file of files) {
      if (uploadingFiles.includes(file.name)) continue;

      setUploadingFiles((prev) => [...prev, file.name]);

      try {
        await onUpload(file);
      } catch (error) {
        console.error(`파일 ${file.name} 업로드 실패:`, error);
      } finally {
        setUploadingFiles((prev) => prev.filter((name) => name !== file.name));
      }
    }
  };

  // 백그라운드 작업 관련 함수들 제거 - 사이드바에서 상태 관리

  return (
    <div className="p-4 rounded-lg shadow mb-4" style={{ backgroundColor: "var(--card-bg)" }}>
      <h3 className="text-base font-bold mb-4">문서 업로드 및 컬렉션 진행상황</h3>

      {/* 파일 업로드 영역 */}
      <div className={`border-2 border-dashed rounded-lg p-6 text-center transition-colors ${dragActive ? "border-blue-500 bg-blue-50 dark:bg-blue-900/20" : "border-gray-300 dark:border-gray-600"}`} onDragEnter={handleDrag} onDragLeave={handleDrag} onDragOver={handleDrag} onDrop={handleDrop}>
        <input ref={fileInputRef} type="file" multiple accept=".pdf,.doc,.docx,.txt,.md" onChange={handleFileChange} className="hidden" />

        <div className="space-y-2">
          <svg className="mx-auto h-12 w-12 text-gray-400" stroke="currentColor" fill="none" viewBox="0 0 48 48">
            <path d="M28 8H12a4 4 0 00-4 4v20m32-12v8m0 0v8a4 4 0 01-4 4H12a4 4 0 01-4-4v-4m32-4l-3.172-3.172a4 4 0 00-5.656 0L28 28M8 32l9.172-9.172a4 4 0 015.656 0L28 28m0 0l4 4m4-24h8m-4-4v8m-12 4h.02" strokeWidth={2} strokeLinecap="round" strokeLinejoin="round" />
          </svg>
          <div>
            <button onClick={() => fileInputRef.current?.click()} className="text-blue-600 hover:text-blue-500 font-medium">
              파일 선택
            </button>
            <span className="text-gray-500"> 또는 여기로 드래그</span>
          </div>
          <p className="text-base text-gray-500">PDF, DOC, DOCX, TXT, MD 파일만 지원</p>
        </div>
      </div>

      {/* 업로딩 중인 파일들 */}
      {uploadingFiles.length > 0 && (
        <div className="mt-4 space-y-2">
          <h4 className="text-base font-medium text-gray-700 dark:text-gray-300">업로드 중</h4>
          {uploadingFiles.map((filename) => (
            <div key={filename} className="flex items-center space-x-2 p-2 bg-blue-50 dark:bg-blue-900/20 rounded">
              <div className="animate-spin h-4 w-4 border-2 border-blue-500 border-t-transparent rounded-full"></div>
              <span className="text-base text-blue-700 dark:text-blue-300">{filename}</span>
            </div>
          ))}
        </div>
      )}

      {/* 백그라운드 작업 상태 표시 제거 - 사이드바 임베딩 상태로 대체 */}
    </div>
  );
}

// 청크 미리보기 + RAG 채팅 컴포넌트(더미)
function RagChunkPreviewAndChat({ chunks, onChat, chatHistory }: RagChunkPreviewAndChatProps) {
  return (
    <div className="p-4 rounded-lg shadow mb-4" style={{ backgroundColor: "var(--card-bg)" }}>
      <h3 className="text-xl font-semibold mb-4 text-gray-900 dark:text-gray-100">청크 미리보기 & RAG 채팅</h3>
      {/* TODO: 문서별 청크 리스트, 청크 상세, 채팅 UI */}
      <div className="text-gray-500 mb-2">청크 미리보기/채팅 UI (추후 구현)</div>
    </div>
  );
}

// HTML 테이블 생성 함수
function createHtmlTable(tableRows: string[]): string {
  if (tableRows.length === 0) return "";

  const parseRow = (row: string): string[] => {
    return row
      .split("|")
      .slice(1, -1)
      .map((cell) => cell.trim());
  };

  const rows = tableRows.map(parseRow);
  const headerRow = rows[0];
  const dataRows = rows.slice(1);

  let html = '<table style="border-collapse: collapse; width: 100%; margin: 10px 0; font-family: inherit; font-size: inherit;">';

  // 헤더
  if (headerRow) {
    html += "<thead><tr>";
    headerRow.forEach((cell) => {
      html += `<th style="border: 1px solid currentColor; padding: 8px 12px; font-weight: 600; text-align: left;">${cell.replace(/</g, "&lt;").replace(/>/g, "&gt;")}</th>`;
    });
    html += "</tr></thead>";
  }

  // 데이터 행
  if (dataRows.length > 0) {
    html += "<tbody>";
    dataRows.forEach((row, index) => {
      html += "<tr>";
      row.forEach((cell) => {
        html += `<td style="border: 1px solid currentColor; padding: 8px 12px;">${cell.replace(/</g, "&lt;").replace(/>/g, "&gt;")}</td>`;
      });
      html += "</tr>";
    });
    html += "</tbody>";
  }

  html += "</table>";
  return html;
}

// getFileIcon, formatFileSize 함수(chat/page.tsx 참고)
function getFileIcon(fileName: string, status: string) {
  const extension = fileName.split(".").pop()?.toLowerCase() || "";
  const iconClass = "w-4 h-4 flex-shrink-0";
  if (status === "uploading") {
    return (
      <div className="w-4 h-4 flex-shrink-0">
        <div className="animate-spin rounded-full h-4 w-4 border-2 border-blue-500 border-t-transparent"></div>
      </div>
    );
  }
  if (status === "processing") {
    return (
      <div className="w-4 h-4 flex-shrink-0">
        <div className="animate-pulse rounded-full h-4 w-4 bg-yellow-500"></div>
      </div>
    );
  }
  if (status === "error") {
    return (
      <svg className={`${iconClass} text-red-500`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
      </svg>
    );
  }
  switch (extension) {
    case "pdf":
      return (
        <svg className={`${iconClass} text-red-600`} fill="currentColor" viewBox="0 0 24 24">
          <path d="M8.267 14.68c-.184 0-.308.018-.372.036v1.178c.076.018.171.023.302.023.479 0 .774-.242.774-.651 0-.366-.254-.586-.704-.586zm3.487.012c-.2 0-.33.018-.407.036v2.61c.077.018.201.018.313.018.817.006 1.349-.444 1.349-1.396.006-.83-.479-1.268-1.255-1.268z" />
          <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM9.498 16.19c-.309.29-.765.42-1.296.42a2.23 2.23 0 0 1-.308-.018v.426H7v-3.936A7.558 7.558 0 0 1 8.219 14c.557 0 .953.106 1.22.319.254.202.426.533.426.923-.001.392-.131.723-.367.948zm3.807 1.355c-.42.349-1.059.515-1.84.515-.468 0-.799-.03-1.024-.06v-3.917A7.947 7.947 0 0 1 11.66 14c.757 0 1.249.136 1.633.426.415.308.675.799.675 1.504 0 .763-.279 1.29-.663 1.615zM17 14.77h-1.532v.911H16.9v.734h-1.432v1.604h-.906V14.03H17v.74z" />
        </svg>
      );
    case "doc":
    case "docx":
      return (
        <svg className={`${iconClass} text-blue-600`} fill="currentColor" viewBox="0 0 24 24">
          <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM16 18H8v-2h8v2zm0-4H8v-2h8v2zm-3-5V3.5L18.5 9H13z" />
        </svg>
      );
    case "xls":
    case "xlsx":
      return (
        <svg className={`${iconClass} text-green-600`} fill="currentColor" viewBox="0 0 24 24">
          <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM12 18H8v-2h4v2zm4-4H8V8h8v2zm-3-5V3.5L18.5 9H13z" />
        </svg>
      );
    case "ppt":
    case "pptx":
      return (
        <svg className={`${iconClass} text-orange-600`} fill="currentColor" viewBox="0 0 24 24">
          <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM11 15H9v3H7v-7h4c1.1 0 2 .9 2 2s-.9 2-2 2zm2-6V3.5L18.5 9H13z" />
          <path d="M9 11v2h2c.6 0 1-.4 1-1s-.4-1-1-1H9z" />
        </svg>
      );
    case "txt":
    case "md":
      return (
        <svg className={`${iconClass} text-gray-600`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
        </svg>
      );
    default:
      return (
        <svg className={`${iconClass} text-gray-400`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
        </svg>
      );
  }
}
function formatFileSize(bytes: number) {
  if (bytes < 1024) return `${bytes}B`;
  if (bytes < 1024 * 1024) return `${(bytes / 1024).toFixed(1)}KB`;
  if (bytes < 1024 * 1024 * 1024) return `${(bytes / 1024 / 1024).toFixed(1)}MB`;
  return `${(bytes / 1024 / 1024 / 1024).toFixed(1)}GB`;
}

// --- 왼쪽: 사이드바(설정/컬렉션/문서) ---
function RagSidebar({ settings, onSettingsChange, collection, onSelectDoc, selectedDoc, selectedFolder, setSelectedFolder, onFolderSelect, chunks, currentChunkIndex, setCurrentChunkIndex, onPrevChunk, onNextChunk, user, onJobComplete, refreshTrigger, onFileUpload, onShowExcelModal, onShowPdfModal, onGraphRAGView, onFullSystemInit, onDocumentInit, onFolderInit }: RagSidebarProps) {
  // 폴딩 상태
  const [showRagBrowser, setShowRagBrowser] = useState(true);
  const [showPreProcessing, setShowPreProcessing] = useState(false);
  const [showRagSettings, setShowRagSettings] = useState(false);
  const [showSmartTune, setShowSmartTune] = useState(false);

  // 관리자 권한 확인
  const [isAdmin, setIsAdmin] = useState(false);

  // 드래그 앤 드롭 상태
  const [isDragOver, setIsDragOver] = useState<boolean>(false);

  // 파일 입력 참조
  const fileInputRef = useRef<HTMLInputElement>(null);

  // 폴더 선택 핸들러 추가
  const handleFolderSelect = (folder: any) => {
    console.log("[RagSidebar] 폴더 선택됨:", {
      folderName: folder.name,
      folderId: folder.id,
      userId: folder.userId,
      isShared: folder.isShared,
      path: folder.path,
      fullFolder: folder
    });
    onFolderSelect?.(folder);
  };

  // 관리자 권한 확인
  useEffect(() => {
    const username = localStorage.getItem("username") || "";
    const userRole = localStorage.getItem("userRole") || "";
    const userToken = localStorage.getItem("userToken") || "";
    const adminStatus = ["admin", "manager"].includes(userRole.toLowerCase());
    // console.log('[RagSidebar 관리자 권한 확인]', {
    //   username,
    //   userRole,
    //   adminStatus
    // });
    setIsAdmin(adminStatus);
  }, []);

  // 드래그 앤 드롭 이벤트 핸들러
  const handleDragEnter = (e: React.DragEvent) => {
    // 파일이 드래그되고 있는지 확인
    if (e.dataTransfer.types.includes("Files")) {
      e.preventDefault();
      e.stopPropagation();
      setIsDragOver(true);
    }
  };

  const handleDragLeave = (e: React.DragEvent) => {
    // 파일이 드래그되고 있는지 확인
    if (e.dataTransfer.types.includes("Files")) {
      e.preventDefault();
      e.stopPropagation();
      if (!e.currentTarget.contains(e.relatedTarget as Node)) {
        setIsDragOver(false);
      }
    }
  };

  const handleDragOver = (e: React.DragEvent) => {
    // 파일이 드래그되고 있는지 확인
    if (e.dataTransfer.types.includes("Files")) {
      e.preventDefault();
      e.stopPropagation();
    }
  };

  const handleDrop = async (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOver(false);

    console.log("드롭 이벤트 처리 시작...");
    console.log("dataTransfer.files 길이:", e.dataTransfer.files?.length);

    // 직접 dataTransfer.files 사용 (가장 안정적인 방법)
    const allFiles = Array.from(e.dataTransfer.files);
    console.log(
      "드롭된 파일들:",
      allFiles.map((f) => `${f.name} (${f.size} bytes, ${f.type})`)
    );

    if (allFiles.length === 0) {
      console.log("드롭된 파일이 없습니다.");
      return;
    }

    // 지원되는 파일만 필터링
    const supportedExtensions = ["pdf", "txt", "md", "doc", "docx", "hwp", "hwpx", "ppt", "pptx", "xls", "xlsx"];

    const files = allFiles.filter((file) => {
      const fileExt = file.name.split(".").pop()?.toLowerCase();
      const hasExtension = file.name.includes(".") && !file.name.endsWith(".");
      const isNotHidden = !file.name.startsWith(".");
      const isSupportedExtension = fileExt && supportedExtensions.includes(fileExt);
      const isValidFile = hasExtension && isNotHidden && isSupportedExtension;

      console.log(`파일 ${file.name}: 확장자=${fileExt}, 지원여부=${isSupportedExtension}, 결과=${isValidFile ? "INCLUDE" : "EXCLUDE"}`);

      return isValidFile;
    });

    console.log(
      "필터링 후 최종 파일 목록:",
      files.map((f) => f.name)
    );

    if (files.length === 0) {
      console.log("업로드할 수 있는 파일이 없습니다.");
      return;
    }

    console.log(`총 ${files.length}개 파일을 업로드합니다.`);

    // 각 파일을 순차적으로 업로드
    for (const file of files) {
      try {
        if (onFileUpload) {
          // 선택된 폴더가 있으면 해당 폴더의 userId로 업로드, 없으면 현재 사용자로 업로드
          const targetUserId = selectedFolder?.userId || user?.username || "admin";
          console.log("[RagSidebar] 드래그 앤 드롭 파일 업로드:", {
            fileName: file.name,
            selectedFolder: selectedFolder,
            selectedFolderUserId: selectedFolder?.userId,
            targetUserId: targetUserId,
            currentUser: user?.username,
            fallbackUsed: !selectedFolder?.userId
          });
          await onFileUpload(file, targetUserId);
        }
      } catch (error) {
        console.error(`파일 업로드 실패: ${file.name}`, error);
      }
    }
  };

  // 메뉴 토글 함수들
  const toggleRagBrowser = () => {
    setShowRagBrowser(!showRagBrowser);
    // 다른 메뉴들이 열려있다면 자동으로 접기
    if (!showRagBrowser) {
      setShowPreProcessing(false);
      setShowRagSettings(false);
      setShowSmartTune(false);
    }
  };

  const togglePreProcessing = () => {
    setShowPreProcessing(!showPreProcessing);
    // 다른 메뉴들이 열려있다면 자동으로 접기
    if (!showPreProcessing) {
      setShowRagBrowser(false);
      setShowRagSettings(false);
      setShowSmartTune(false);
    }
  };

  const toggleRagSettings = () => {
    setShowRagSettings(!showRagSettings);
    // 다른 메뉴들이 열려있다면 자동으로 접기
    if (!showRagSettings) {
      setShowRagBrowser(false);
      setShowPreProcessing(false);
      setShowSmartTune(false);
    }
  };

  const toggleSmartTune = () => {
    setShowSmartTune(!showSmartTune);
    // 다른 메뉴들이 열려있다면 자동으로 접기
    if (!showSmartTune) {
      setShowRagBrowser(false);
      setShowPreProcessing(false);
      setShowRagSettings(false);
    }
  };

  return (
    <div className="flex-shrink-0 h-full flex flex-col overflow-hidden" style={{ backgroundColor: "var(--sidebar-bg)" }}>
      {/* 스크롤 가능한 컨테이너 */}
      <div className="flex-1 overflow-y-auto">
        {/* 내 문서 섹션 */}
        <div className="border-b border-gray-200 dark:border-gray-700">
          <div className="p-3">
            <div className="flex items-center justify-between mb-2">
              <button onClick={toggleRagBrowser} className={`menu-header ${showRagBrowser ? "expanded" : ""}`}>
                <svg className="menu-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                </svg>
                <h3>{isAdmin ? "전체 문서 관리" : "내 문서"}</h3>
                <svg className={`toggle-arrow ${showRagBrowser ? "rotate-180" : ""}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                </svg>
              </button>
            </div>
            <div className={`transition-all duration-300 ease-in-out overflow-hidden ${showRagBrowser ? "max-h-[800px] opacity-100" : "max-h-0 opacity-0"}`}>
              <div className="space-y-3">
                {/* 드래그 앤 드롭 파일 업로드 영역 */}
                <div data-no-dnd="true" data-drag-over={isDragOver} onDragEnter={handleDragEnter} onDragLeave={handleDragLeave} onDragOver={handleDragOver} onDrop={handleDrop} onClick={() => fileInputRef.current?.click()} className="drag-drop-zone rounded-lg p-4 text-center transition-colors cursor-pointer mb-3">
                  <svg className="w-8 h-8 mx-auto mb-2 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12" />
                  </svg>
                  <p className="text-sm text-gray-600 dark:text-gray-400 mb-1">{isDragOver ? "여기에 파일을 놓으세요" : "업로드"}</p>

                  {/* 숨겨진 파일 입력 */}
                  <input
                    ref={fileInputRef}
                    type="file"
                    multiple
                    accept=""
                    onChange={async (e) => {
                      const files = Array.from(e.target.files || []);
                      if (files.length > 0 && onFileUpload) {
                        for (const file of files) {
                          try {
                            // 선택된 폴더가 있으면 해당 폴더의 userId로 업로드, 없으면 현재 사용자로 업로드
                            const targetUserId = selectedFolder?.userId || user?.username || "admin";
                            console.log("[RagSidebar] 파일 업로드:", {
                              fileName: file.name,
                              selectedFolder: selectedFolder,
                              selectedFolderUserId: selectedFolder?.userId,
                              targetUserId: targetUserId,
                              currentUser: user?.username,
                              fallbackUsed: !selectedFolder?.userId
                            });
                            await onFileUpload(file, targetUserId);
                          } catch (error) {
                            console.error(`파일 업로드 실패: ${file.name}`, error);
                          }
                        }
                      }
                      // 파일 선택 후 input 초기화
                      e.target.value = "";
                    }}
                    className="hidden"
                  />
                </div>

                {/* 백그라운드 작업 상태 제거 - 사이드바 파일 아이콘으로 대체 */}

                {/* 선택된 폴더 정보 표시 */}
                {selectedFolder && (
                  <div className="mb-3 p-2 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-2">
                        <svg className="w-4 h-4 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2H5a2 2 0 00-2-2z" />
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2H5a2 2 0 00-2-2z" />
                        </svg>
                        <span className="text-sm font-medium text-blue-800 dark:text-blue-200">{selectedFolder.name}</span>
                      </div>
                      <button onClick={() => setSelectedFolder?.(null)} className="text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-200">
                        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                        </svg>
                      </button>
                    </div>
                  </div>
                )}

                {/* 새로운 파일 매니저 */}
                <div className="h-[500px] border rounded-lg" style={{ borderColor: "var(--card-border)" }}>
                  <RagFileTreeView onSelectDoc={onSelectDoc} selectedDoc={selectedDoc} onFileUpload={onFileUpload} refreshTrigger={refreshTrigger} onFolderSelect={handleFolderSelect} onGraphRAGView={onGraphRAGView} onFullSystemInit={onFullSystemInit} onDocumentInit={onDocumentInit} onFolderInit={onFolderInit} />
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Pre Processing 섹션 */}
        <div className="border-b border-gray-200 dark:border-gray-700">
          <div className="p-3">
            <div className="flex items-center justify-between mb-2">
              <button onClick={togglePreProcessing} className={`menu-header ${showPreProcessing ? "expanded" : ""}`}>
                <svg className="menu-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 3v2m6-2v2M9 19v2m6-2v2M5 9H3m2 6H3m18-6h-2m2 6h-2M7 19h10a2 2 0 002-2V7a2 2 0 00-2-2H7a2 2 0 00-2 2v10a2 2 0 002 2zM9 9h6v6H9V9z" />
                </svg>
                <h3>문서 전처리</h3>
                <svg className={`toggle-arrow ${showPreProcessing ? "rotate-180" : ""}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                </svg>
              </button>
            </div>

            {showPreProcessing && (
              <div className="mt-4 space-y-3">
                <div className="text-sm text-gray-600 dark:text-gray-400 mb-3">문서를 업로드하기 전에 전처리하여 RAG 성능을 향상시킵니다.</div>

                <div className="grid grid-cols-2 gap-3">
                  <button onClick={() => onShowExcelModal?.()} className="flex flex-col items-center p-4 bg-green-300 dark:bg-green-800 hover:bg-green-400 dark:hover:bg-green-700 rounded-lg transition-colors border border-green-400 dark:border-green-600">
                    <svg className="w-8 h-8 mb-2 text-green-600 dark:text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 17V7m0 10a2 2 0 01-2 2H5a2 2 0 01-2-2V7a2 2 0 012-2h2a2 2 0 012 2m0 10a2 2 0 002 2h2a2 2 0 002-2M9 7a2 2 0 012-2h2a2 2 0 012 2m0 10V7m0 10a2 2 0 002 2h2a2 2 0 002-2V7a2 2 0 00-2-2h-2a2 2 0 00-2 2" />
                    </svg>
                    <span className="text-sm font-medium">Excel</span>
                  </button>

                  <button onClick={() => onShowPdfModal?.()} className="flex flex-col items-center p-4 bg-red-300 dark:bg-red-800 hover:bg-red-400 dark:hover:bg-red-700 rounded-lg transition-colors border border-red-400 dark:border-red-600">
                    <svg className="w-8 h-8 mb-2 text-red-600 dark:text-red-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 21h10a2 2 0 002-2V9.414a1 1 0 00-.293-.707l-5.414-5.414A1 1 0 0012.586 3H7a2 2 0 00-2 2v14a2 2 0 002 2z" />
                    </svg>
                    <span className="text-sm font-medium">PDF</span>
                  </button>
                </div>

                <div className="text-xs text-gray-500 dark:text-gray-400 mt-3">
                  • Excel: 시트 선택, 헤더 정규화, 열/행 필터링
                  <br />• PDF: 회전 감지, 2-up 분할, 여백 크롭
                </div>
              </div>
            )}
          </div>
        </div>

        {/* Smart Tune 섹션 - 쉬운 설정 */}
        <div className="border-b border-gray-200 dark:border-gray-700">
          <div className="p-3">
            <div className="flex items-center justify-between mb-2">
              <button onClick={toggleSmartTune} className={`menu-header ${showSmartTune ? "expanded" : ""}`}>
                <svg className="menu-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 10V3L4 14h7v7l9-11h-7z" />
                </svg>
                <h3>Smart Tune</h3>
                <svg className={`toggle-arrow ${showSmartTune ? "rotate-180" : ""}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                </svg>
              </button>
            </div>
            {showSmartTune && <RagOptimizationPanel settings={settings} onSettingsChange={onSettingsChange} user={user} />}
          </div>
        </div>

        {/* Advanced Settings 섹션 - 고급 설정 */}
        <div className="border-b border-gray-200 dark:border-gray-700">
          <div className="p-3">
            <div className="flex items-center justify-between mb-2">
              <button onClick={toggleRagSettings} className={`menu-header ${showRagSettings ? "expanded" : ""}`}>
                <svg className="menu-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"
                  />
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                </svg>
                <h3>RAG 상세 설정</h3>
                <svg className={`toggle-arrow ${showRagSettings ? "rotate-180" : ""}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                </svg>
              </button>
            </div>
            {showRagSettings && (
              <div className="mt-4">
                <RagSettingsPanel key={`${settings["RAG_MODE"]}-${settings["RAG_SEARCH_MODE"]}-${JSON.stringify(settings).length}`} settings={settings} onChange={onSettingsChange} />
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}

// --- Progress Tracker 단계 정의 ---
const RAG_STEPS = [
  { key: "queue", label: "로딩" },
  { key: "extract", label: "내용 추출" },
  { key: "preprocess", label: "전처리" },
  { key: "split", label: "청크 분할" },
  { key: "metadata", label: "정보 생성" },
  { key: "embedding", label: "임베딩" },
  { key: "save", label: "저장" },
  { key: "done", label: "완료" },
  { key: "error", label: "오류" }
];

// 로그 파싱은 더 이상 사용하지 않음 (서버에서 직접 상태 제공)

// --- ProgressTracker 컴포넌트 ---
function ProgressTracker({ docStatuses, large = false }: { docStatuses: RagDocStatus[]; large?: boolean }) {
  const [ragProgress, setRagProgress] = useState<{
    total: number;
    completed: number;
    processing: number;
    failed: number;
    progress_percentage: number;
    remaining: number;
    total_documents: number;
  } | null>(null);
  const [ragDocuments, setRagDocuments] = useState<Array<{ filename: string; step: string; step_label: string }>>([]);

  // RAG 서버에서 실시간 진행 상태 가져오기
  useEffect(() => {
    const fetchRagProgress = async () => {
      try {
        const response = await fetch(`${getRagServerUrl()}/rag/progress`);
        if (response.ok) {
          const data = await response.json();
          // API 응답 구조에 따라 적절히 처리
          if (data.queue_status) {
            setRagProgress(data.queue_status);
            setRagDocuments(data.documents || []);
          } else if (data.total_documents !== undefined) {
            // fallback 응답 구조 처리
            const fallbackData = {
              total: data.total_documents,
              total_documents: data.total_documents,
              completed: data.completed || 0,
              processing: data.processing || 0,
              failed: data.failed || 0,
              remaining: data.total_documents - (data.completed || 0) - (data.failed || 0),
              progress_percentage: data.progress_percentage || 0
            };
            setRagProgress(fallbackData);
            setRagDocuments([]);
          }
        }
      } catch (error) {
        console.error("RAG 진행 상태 조회 실패:", error);
      }
    };

    fetchRagProgress();
    const interval = setInterval(fetchRagProgress, 2000); // 2초마다 업데이트

    return () => clearInterval(interval);
  }, []);

  return (
    <div className="space-y-4 mt-6 w-full flex flex-col items-center">
      {docStatuses.length === 0 ? (
        <div className="text-base text-gray-400">진행 중인 백그라운드 작업이 없습니다.</div>
      ) : (
        docStatuses.map((doc) => {
          const stepOrder: RagDocStatus["step"][] = ["queue", "extract", "preprocess", "split", "metadata", "embedding", "save", "done"];
          const currentIdx = stepOrder.indexOf(doc.step);

          // 문서가 완료되었는지 확인
          const isDocCompleted = doc.step === "done";

          return (
            <div key={doc.filename} className="flex items-center gap-4 p-4 bg-gray-50 dark:bg-gray-800 rounded-xl shadow-lg w-full max-w-4xl">
              <div className="flex flex-row gap-4 flex-1 justify-center">
                {RAG_STEPS.filter((s) => s.key !== "error").map((step, idx) => {
                  const stepIdx = stepOrder.indexOf(step.key as RagDocStatus["step"]);
                  const isDone = idx < currentIdx || doc.step === "done" || (isDocCompleted && step.key !== "error");
                  const isCurrent = idx === currentIdx && doc.step !== "done" && doc.step !== "error" && !isDocCompleted;

                  return (
                    <div key={step.key} className="flex flex-col items-center min-w-[40px]">
                      <div
                        className={`
                        flex items-center justify-center rounded-full border-2 transition-all
                        w-12 h-12 text-base
                        ${isDone ? "bg-green-500 border-green-500 text-white" : isCurrent ? "bg-blue-500 border-blue-500 text-white animate-pulse" : "bg-gray-200 dark:bg-gray-700 border-gray-300 dark:border-gray-600 text-gray-400"}
                      `}
                      >
                        {isDone ? (
                          <svg className="w-6 h-6" fill="none" stroke="currentColor" strokeWidth={2} viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" d="M5 13l4 4L19 7" />
                          </svg>
                        ) : (
                          <span className="font-bold">{idx + 1}</span>
                        )}
                      </div>
                      <span className="mt-1 text-base font-normal ...">{step.label}</span>
                      {/* 임베딩 단계일 때 진행률 */}
                      {step.key === "embedding" && doc.step === "embedding" && doc.chunkTotal && !isDocCompleted ? (
                        <span className={`${large ? "text-base" : "text-base"} text-blue-500 mt-1`}>
                          {doc.chunkCurrent}/{doc.chunkTotal}
                        </span>
                      ) : null}
                    </div>
                  );
                })}
                {/* 오류 단계도 동일하게 min-w, w/h, text-base 등으로 조정 */}
              </div>
            </div>
          );
        })
      )}
    </div>
  );
}

// RagDocStatus 타입 추가
type RagDocStatus = {
  filename: string;
  step: "queue" | "extract" | "preprocess" | "split" | "metadata" | "save" | "embedding" | "done" | "error";
  stepLabel: string;
  progress?: number;
  chunkCurrent?: number;
  chunkTotal?: number;
  errorMsg?: string;
};

// --- 가운데: 업로드/진행상황/청크 미리보기 ---
function RagMainPanel({ chunks, selectedDoc, currentChunkIndex, setCurrentChunkIndex, onPrevChunk, onNextChunk, showGraphRAGModal, setShowGraphRAGModal, setCurrentGraphQuery, isInitializing, docStatuses, progressPercent, statusMessage, initStartTime, documentStats }: any) {
  // GraphRAG 기능을 기본적으로 활성화 (별도 설정 확인 불필요)
  const [globalGraphRAGEnabled, setGlobalGraphRAGEnabled] = useState(true);

  return (
    <div className="flex-1 flex flex-col min-w-0 h-full" style={{ backgroundColor: "var(--body-bg)" }}>
      <div className="p-6 flex-1 overflow-y-auto">
        {/* 문서 관리 섹션 */}
        <div className="grid grid-cols-2 gap-4 mb-6">
          {/* 좌측: 문서 전체 통계 정보 */}
          <div className="bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg p-4 shadow-sm">
            <h3 className="text-gray-800 dark:text-gray-200 mb-3 flex items-center">📊 문서 통계</h3>
            <div className="space-y-3">
              <div className="flex justify-between items-center">
                <span className="text-sm text-gray-600 dark:text-gray-400">전체 문서</span>
                <span className="text-sm font-medium text-gray-900 dark:text-gray-100">{documentStats.total}개</span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm text-gray-600 dark:text-gray-400">임베딩 완료</span>
                <span className="text-sm font-medium text-green-600 dark:text-green-400">{documentStats.completed}개</span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm text-gray-600 dark:text-gray-400">임베딩 중</span>
                <span className="text-sm font-medium text-blue-600 dark:text-blue-400">{documentStats.processing}개</span>
              </div>

              {/* 프로그래스바 */}
              <div className="pt-2 border-t border-gray-200 dark:border-gray-600">
                <div className="flex justify-between items-center mb-2">
                  <span className="text-xs text-gray-500 dark:text-gray-400">진행률</span>
                  <span className="text-xs font-medium text-gray-700 dark:text-gray-300">{documentStats.total > 0 ? Math.round((documentStats.completed / documentStats.total) * 100) : 0}%</span>
                </div>
                <div className="w-full bg-gray-200 dark:bg-gray-600 rounded-full h-2">
                  <div
                    className="bg-gradient-to-r from-green-500 to-green-600 h-2 rounded-full transition-all duration-300 ease-in-out"
                    style={{
                      width: `${documentStats.total > 0 ? Math.round((documentStats.completed / documentStats.total) * 100) : 0}%`
                    }}
                  ></div>
                </div>
                {documentStats.processing > 0 && (
                  <div className="text-xs text-blue-600 dark:text-blue-400 mt-1 flex items-center">
                    <div className="animate-pulse w-2 h-2 bg-blue-500 rounded-full mr-1"></div>
                    {documentStats.processing}개 문서 임베딩 중...
                  </div>
                )}
              </div>
            </div>
          </div>

          {/* 우측: RAG 기능 안내 */}
          <div className="bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg p-4 shadow-sm">
            <h3 className="text-gray-800 dark:text-gray-200 mb-3 flex items-center">ℹ️ 안내</h3>
            <div className="space-y-2 text-sm text-gray-600 dark:text-gray-400">
              <div className="flex items-start space-x-2">
                <span className="text-blue-500">•</span>
                <span>문서를 선택하여 청크별 내용을 미리 볼 수 있습니다</span>
              </div>
              <div className="flex items-start space-x-2">
                <span className="text-purple-500">•</span>
                <span>지식그래프 버튼으로 문서 간 연관성을 확인하세요</span>
              </div>
              <div className="flex items-start space-x-2">
                <span className="text-green-500">•</span>
                <span>Smart Tune 기능으로 RAG 성능을 자동 최적화할 수 있습니다</span>
              </div>
              <div className="flex items-start space-x-2">
                <span className="text-orange-500">•</span>
                <span>문서 초기화는 사이드바에서 진행할 수 있습니다</span>
              </div>
            </div>
          </div>
        </div>

        <div className="bg-gray-50 dark:bg-gray-800 p-3 rounded-lg mb-4">
          <h4 className="text-base font-semibold text-gray-800 dark:text-gray-200 flex items-center">📄 청크 미리보기</h4>
        </div>
        {/* 청크 미리보기 UI */}
        {!selectedDoc ? (
          <div className="text-base text-gray-500 dark:text-gray-400 mb-2">왼쪽 사이드바에서 문서를 선택하세요.</div>
        ) : chunks.length === 0 ? (
          <div className="text-gray-400 mb-6">청크 데이터가 없습니다.</div>
        ) : (
          <div className="mb-6">
            <div className="flex items-center justify-between mb-2 pb-4">
              <div className="text-base font-semibold text-gray-700 dark:text-gray-200 truncate max-w-[70%]" title={selectedDoc}>
                {selectedDoc}
              </div>
              <div className="flex items-center gap-3">
                <div className="text-base text-gray-500 dark:text-gray-400 flex items-center gap-1">
                  <input
                    type="number"
                    min={1}
                    max={chunks.length}
                    value={currentChunkIndex + 1}
                    onChange={(e) => {
                      let val = Number(e.target.value);
                      if (isNaN(val)) return;
                      if (val < 1) val = 1;
                      if (val > chunks.length) val = chunks.length;
                      setCurrentChunkIndex(val - 1);
                    }}
                    onKeyDown={(e) => {
                      if (e.key === "Enter") {
                        let val = Number((e.target as HTMLInputElement).value);
                        if (isNaN(val)) return;
                        if (val < 1) val = 1;
                        if (val > chunks.length) val = chunks.length;
                        setCurrentChunkIndex(val - 1);
                      }
                    }}
                    className="w-14 px-1 py-0.5 border border-gray-300 dark:border-gray-600 rounded text-center text-base bg-white dark:bg-gray-800 focus:outline-none focus:ring-2 focus:ring-blue-400"
                    style={{ width: "3.5rem" }}
                    aria-label="청크 번호 직접 입력"
                  />
                  <span>/ {chunks.length}</span>
                </div>

                {/* GraphRAG 지식그래프 버튼 */}
                {globalGraphRAGEnabled && chunks.length > 0 && (
                  <button
                    onClick={() => {
                      const currentChunk = chunks[currentChunkIndex];
                      if (currentChunk) {
                        // 현재 청크의 내용을 직접 전달 (더 확실한 방법)
                        const queryData = {
                          type: "chunk_content",
                          chunk_content: currentChunk.content || "",
                          chunk_metadata: currentChunk.metadata || {},
                          document_name: selectedDoc?.split("/").pop() || "unknown",
                          chunk_index: currentChunkIndex,
                          user_id: "admin"
                        };
                        console.log("청크 내용을 GraphRAG로 전달:", queryData);
                        setCurrentGraphQuery(JSON.stringify(queryData));
                        setShowGraphRAGModal(true);
                      }
                    }}
                    className="flex items-center gap-1 px-3 py-1 text-sm rounded-lg transition-colors text-purple-600 dark:text-purple-400 hover:text-purple-700 dark:hover:text-purple-300 hover:bg-purple-50 dark:hover:bg-purple-900/20 border border-purple-300 dark:border-purple-600"
                    title="현재 청크의 지식 그래프 보기"
                  >
                    🕸️ 지식그래프
                  </button>
                )}

                {/* 이전/다음 버튼을 여기로 이동 */}
                <div className="flex gap-2">
                  <button onClick={onPrevChunk} disabled={currentChunkIndex === 0} className="px-3 py-1 text-base bg-blue-500 text-white rounded hover:bg-blue-600 disabled:opacity-50 disabled:cursor-not-allowed shadow-md">
                    이전
                  </button>
                  <button onClick={onNextChunk} disabled={currentChunkIndex === chunks.length - 1} className="px-3 py-1 text-base bg-blue-500 text-white rounded hover:bg-blue-600 disabled:opacity-50 disabled:cursor-not-allowed shadow-md">
                    다음
                  </button>
                </div>
              </div>
            </div>
            {/* 청크 메타데이터 표시 - 섹션별로 명확하게 구분 */}
            {(() => {
              const currentChunk = chunks[currentChunkIndex];
              if (!currentChunk) return null;

              // 메타데이터에서 enhanced_embedding 정보 추출
              let extractedQuestions: string[] = [];
              let extractedKeywords: string[] = [];
              let extractedTopics: string[] = [];

              try {
                const metadata = currentChunk.metadata || {};

                // enhanced_embedding 정보 추출
                if (metadata.enhanced_embedding) {
                  const enhanced = metadata.enhanced_embedding;
                  if (enhanced.questions && Array.isArray(enhanced.questions)) {
                    extractedQuestions = enhanced.questions;
                  }
                  if (enhanced.keywords && Array.isArray(enhanced.keywords)) {
                    extractedKeywords = enhanced.keywords;
                  }
                  if (enhanced.topics && Array.isArray(enhanced.topics)) {
                    extractedTopics = enhanced.topics;
                  }
                }

                // 기존 방식으로도 시도 (하위 호환성)
                if (extractedQuestions.length === 0 && metadata.extracted_questions) {
                  if (typeof metadata.extracted_questions === "string") {
                    extractedQuestions = JSON.parse(metadata.extracted_questions);
                  } else if (Array.isArray(metadata.extracted_questions)) {
                    extractedQuestions = metadata.extracted_questions;
                  }
                }

                if (extractedKeywords.length === 0 && metadata.extracted_keywords) {
                  if (typeof metadata.extracted_keywords === "string") {
                    extractedKeywords = JSON.parse(metadata.extracted_keywords);
                  } else if (Array.isArray(metadata.extracted_keywords)) {
                    extractedKeywords = metadata.extracted_keywords;
                  }
                }

                if (extractedTopics.length === 0 && metadata.extracted_topics) {
                  if (typeof metadata.extracted_topics === "string") {
                    extractedTopics = JSON.parse(metadata.extracted_topics);
                  } else if (Array.isArray(metadata.extracted_topics)) {
                    extractedTopics = metadata.extracted_topics;
                  }
                }
              } catch (e) {
                console.warn("메타데이터 파싱 오류:", e);
              }

              // 기본 메타데이터 정보
              const metadata = currentChunk.metadata || {};
              const pageNumber = currentChunk.page_number || metadata.page_number;
              const modificationDate = currentChunk.modification_date || metadata.modification_date;
              const source = currentChunk.source || metadata.source;
              const chunkId = currentChunk.id || metadata.id;
              const hasBasicInfo = pageNumber || modificationDate || source || chunkId;
              const hasAiMetadata = extractedQuestions.length > 0 || extractedKeywords.length > 0 || extractedTopics.length > 0;

              // 기본 정보나 AI 메타데이터가 하나라도 있으면 표시
              if (hasBasicInfo || hasAiMetadata) {
                return (
                  <div className="mb-6">
                    {/* 청크 메타데이터 */}
                    <div className="border border-gray-200 dark:border-gray-600 rounded-lg p-4 shadow-sm" style={{ backgroundColor: "var(--card-bg)" }}>
                      <h5 className="text-base text-gray-800 dark:text-gray-200 mb-4 pb-2 border-b-2 border-blue-200 dark:border-blue-700">📋 메타데이터</h5>

                      {/* 청크 기본 정보 */}
                      {hasBasicInfo && (
                        <div className="mb-4">
                          <div className="grid grid-cols-2 gap-1 text-sm">
                            {pageNumber && (
                              <div className="flex items-center gap-2">
                                <span className="text-gray-600 dark:text-gray-400">📄 페이지:</span>
                                <span className="font-medium text-gray-700 dark:text-gray-300">{pageNumber}</span>
                              </div>
                            )}
                            {modificationDate && (
                              <div className="flex items-center gap-2">
                                <span className="text-gray-600 dark:text-gray-400">📅 수정일:</span>
                                <span className="font-medium text-gray-700 dark:text-gray-300">{new Date(modificationDate).toLocaleString("ko-KR")}</span>
                              </div>
                            )}
                            {source && (
                              <div className="flex items-center gap-2">
                                <span className="text-gray-600 dark:text-gray-400">📁 파일:</span>
                                <span className="font-medium text-gray-700 dark:text-gray-300 truncate max-w-[120px]" title={source}>
                                  {source.split("/").pop()}
                                </span>
                              </div>
                            )}
                            {chunkId && (
                              <div className="flex items-center gap-2">
                                <span className="text-gray-600 dark:text-gray-400">🆔 청크ID:</span>
                                <span className="font-medium text-gray-700 dark:text-gray-300 font-mono text-base">{String(chunkId).substring(0, 8)}...</span>
                              </div>
                            )}
                            {currentChunk.content && (
                              <div className="flex items-center gap-2">
                                <span className="text-gray-600 dark:text-gray-400">📏 길이:</span>
                                <span className="font-medium text-gray-700 dark:text-gray-300">{currentChunk.content.length.toLocaleString()}자</span>
                              </div>
                            )}
                          </div>
                        </div>
                      )}

                      {/* AI 분석 결과 */}
                      {hasAiMetadata && (
                        <div>
                          {/* 예상질문 */}
                          {extractedQuestions.length > 0 && (
                            <div className="mb-4">
                              <h6 className="text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2 flex items-center gap-2">💡 예상질문 ({extractedQuestions.length}개)</h6>
                              <div className="space-y-1">
                                {extractedQuestions.slice(0, 5).map((question, idx) => (
                                  <div key={idx} className="text-sm text-gray-700 dark:text-gray-300 pl-3 ">
                                    Q. {question}
                                  </div>
                                ))}
                                {extractedQuestions.length > 5 && <div className="text-sm text-blue-600 dark:text-blue-400 pl-3 font-medium">... 외 {extractedQuestions.length - 5}개</div>}
                              </div>
                            </div>
                          )}

                          {/* 키워드 */}
                          {extractedKeywords.length > 0 && (
                            <div className="mb-4">
                              <h6 className="text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2 flex items-center gap-2">🏷️ 키워드 ({extractedKeywords.length}개)</h6>
                              <div className="flex flex-wrap gap-2 pl-3">
                                {extractedKeywords.slice(0, 10).map((keyword, idx) => (
                                  <span key={idx} className="inline-block px-3 py-1 text-sm bg-blue-100 dark:bg-blue-800 text-blue-800 dark:text-blue-200 rounded-full border border-blue-200 dark:border-blue-700 shadow-sm">
                                    {keyword}
                                  </span>
                                ))}
                                {extractedKeywords.length > 10 && <span className="inline-block px-3 py-1 text-sm bg-gray-100 dark:bg-gray-700 text-gray-600 dark:text-gray-400 rounded-full border border-gray-200 dark:border-gray-600">+{extractedKeywords.length - 10}개</span>}
                              </div>
                            </div>
                          )}

                          {/* 토픽 */}
                          {extractedTopics.length > 0 && (
                            <div>
                              <h6 className="text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2 flex items-center gap-2">📚 주제 ({extractedTopics.length}개)</h6>
                              <div className="flex flex-wrap gap-2 pl-3">
                                {extractedTopics.map((topic, idx) => (
                                  <span key={idx} className="inline-block px-3 py-1 text-sm bg-green-100 dark:bg-green-800 text-green-800 dark:text-green-200 rounded-full border border-green-200 dark:border-green-700 shadow-sm">
                                    {topic}
                                  </span>
                                ))}
                              </div>
                            </div>
                          )}
                        </div>
                      )}
                    </div>
                  </div>
                );
              }

              return null;
            })()}

            {/* 청크 내용 섹션 */}
            <div className="border border-gray-200 dark:border-gray-600 rounded-lg p-4 shadow-sm mb-4" style={{ backgroundColor: "var(--card-bg)" }}>
              <h5 className="text-base font-semibold text-gray-800 dark:text-gray-200 mb-3">📄 청크 내용</h5>
              <div
                className="
                  chunk-preview
                  bg-white dark:bg-gray-900 border border-gray-300 dark:border-gray-600 rounded-lg p-4
                  min-h-[400px] overflow-auto
                  text-gray-900 dark:text-gray-100 text-sm
                  relative
                "
                style={{
                  resize: "both",
                  minWidth: "300px",
                  minHeight: "400px",
                  fontFamily: 'ui-sans-serif, system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, "Noto Sans", sans-serif',
                  lineHeight: "1.6",
                  whiteSpace: "pre-wrap",
                  wordBreak: "break-word",
                  overflowWrap: "break-word"
                }}
              >
                {/* 마크다운 테이블을 HTML 테이블로 변환하여 렌더링 */}
                <div
                  dangerouslySetInnerHTML={{
                    __html: (() => {
                      const content = chunks[currentChunkIndex]?.content || "(내용 없음)";

                      // 마크다운 테이블 패턴 감지 및 HTML 테이블로 변환
                      const lines = content.split("\n");
                      let inTable = false;
                      let tableRows: string[] = [];
                      let result: string[] = [];

                      for (let i = 0; i < lines.length; i++) {
                        const line = lines[i];

                        // 테이블 행 감지 (| 로 시작하고 끝나는 행)
                        if (line.trim().match(/^\|.*\|$/)) {
                          if (!inTable) {
                            inTable = true;
                            tableRows = [];
                          }
                          tableRows.push(line);
                        } else if (inTable && line.trim().match(/^[\|\-\s:]+$/)) {
                          // 테이블 구분선 (|---|---|)
                          continue;
                        } else {
                          // 테이블 끝
                          if (inTable) {
                            // HTML 테이블 생성
                            const tableHtml = createHtmlTable(tableRows);
                            result.push(tableHtml);
                            inTable = false;
                            tableRows = [];
                          }
                          result.push(line.replace(/</g, "&lt;").replace(/>/g, "&gt;"));
                        }
                      }

                      // 마지막에 테이블이 있었다면 처리
                      if (inTable && tableRows.length > 0) {
                        const tableHtml = createHtmlTable(tableRows);
                        result.push(tableHtml);
                      }

                      return result.join("\n");
                    })()
                  }}
                  style={{
                    whiteSpace: "pre-line",
                    fontFamily: "inherit",
                    fontSize: "inherit",
                    lineHeight: "inherit"
                  }}
                />
              </div>
            </div>

            {/* 상세 메타데이터 섹션 (접을 수 있음) */}
            {chunks[currentChunkIndex]?.metadata && (
              <div className="border border-gray-200 dark:border-gray-600 rounded-lg p-4 shadow-sm mb-4" style={{ backgroundColor: "var(--card-bg)" }}>
                <details className="text-base">
                  <summary className="cursor-pointer text-gray-700 dark:text-gray-300 hover:text-gray-900 dark:hover:text-gray-100 px-3 py-2 transition-colors duration-200">📊 상세 정보 보기</summary>
                  <div className="mt-3 space-y-1 text-sm pl-3 text-gray-600 dark:text-gray-400">
                    {/* 중요하지 않은 메타데이터는 필터링하여 표시 */}
                    {Object.entries(chunks[currentChunkIndex].metadata || {})
                      .filter(([key, value]) => {
                        // 빈 값이나 의미없는 값 제외
                        if (value === null || value === undefined || value === "") return false;
                        if (typeof value === "object" && Object.keys(value).length === 0) return false;
                        return true;
                      })
                      .map(([key, value]) => {
                        // 메타데이터 필드별 설명 추가
                        const getFieldDescription = (fieldKey: string) => {
                          const descriptions: Record<string, string> = {
                            db_id: "데이터베이스 ID",
                            doc_id: "문서 ID",
                            filename: "파일명",
                            chunk_index: "청크 인덱스",
                            created_at: "생성 시간",
                            has_embedding: "임베딩 생성 여부",
                            embedding_dimensions: "연결된 임베딩 벡터의 총 차원 (원본+질문+키워드 등 여러 임베딩 연결)",
                            user_id: "사용자 ID (all=전체 공유)",
                            timestamp: "타임스탬프",
                            matches: "검색 매칭 결과 (exact_matches: 정확한 일치, partial_matches: 부분 일치)",
                            sub_scores: "하위 점수 (길이, 내용, 키워드 품질, 구조, 부분 일치, 강한 키워드 일치) - 모든 점수는 0~1 범위",
                            thresholds: "임계값 설정 (관련성, 유사도, 정확한 일치, 키워드 일치, 강한 키워드 일치)",
                            main_scores: "주요 점수 (날짜, 관련성, 유사도, 정확한 일치)",
                            sub_weights: "하위 가중치 (길이, 내용, 구조, 부분 일치, 키워드 품질)",
                            main_weights: "주요 가중치 (날짜, 관련성, 유사도, 정확한 일치)",
                            processed_at: "처리 시간 (Unix timestamp)",
                            total_chunks: "전체 청크 수",
                            filter_settings: "필터 설정 (임계값들)",
                            enhanced_embedding: "AI 강화 임베딩 (토픽, 키워드, 예상질문)"
                          };
                          return descriptions[fieldKey] || fieldKey;
                        };

                        return (
                          <div key={key} className="flex items-start gap-2">
                            <span className="text-sm text-gray-800 dark:text-gray-200 min-w-[140px]" title={getFieldDescription(key)}>
                              {key}:
                            </span>
                            <span className="break-all text-sm text-gray-600 dark:text-gray-400">{typeof value === "object" ? JSON.stringify(value, null, 2) : typeof value === "number" ? (key === "chunk_index" || key === "page_number" || key === "total_chunks" || key === "db_id" ? Math.round(value) : Number(value).toFixed(3)) : String(value)}</span>
                          </div>
                        );
                      })}
                  </div>
                </details>
              </div>
            )}

            {/* 우측 하단 작은 이전/다음 버튼 */}
            <div className="flex justify-end mt-4 relative">
              <div className="flex gap-1">
                <button onClick={onPrevChunk} disabled={currentChunkIndex === 0} className="px-2 py-1 text-base bg-blue-500 text-white rounded hover:bg-blue-600 disabled:opacity-50 disabled:cursor-not-allowed shadow-md">
                  이전
                </button>
                <button onClick={onNextChunk} disabled={currentChunkIndex === chunks.length - 1} className="px-2 py-1 text-base bg-blue-500 text-white rounded hover:bg-blue-600 disabled:opacity-50 disabled:cursor-not-allowed shadow-md">
                  다음
                </button>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}

// 다운로드 링크를 절대 경로로 변환하는 함수
const convertDownloadLinksToAbsolute = (content: string): string => {
  const apiServerUrl = getApiServerUrl();
  const baseServerUrl = apiServerUrl.replace("/api/v1", "");

  // 🔗 다운로드: /uploads/filename 패턴을 찾아서 절대 경로로 변환
  // 파일명에 공백이 있을 수 있으므로 줄바꿈이 아닌 모든 문자를 매치하되, 다음 줄까지 포함
  const downloadLinkPattern = /🔗 다운로드: (\/uploads\/[^\n]+?)(?=\n|$)/g;

  return content.replace(downloadLinkPattern, (match, relativePath) => {
    // 파일명에서 불필요한 공백 제거
    const cleanPath = relativePath.trim();

    // URL 인코딩 처리 - 파일명 부분만 인코딩
    const pathParts = cleanPath.split("/");
    const filename = pathParts.pop() || "document";
    const encodedFilename = encodeURIComponent(filename);
    const encodedPath = pathParts.join("/") + "/" + encodedFilename;

    const absoluteUrl = `${baseServerUrl}${encodedPath}`;
    return `🔗 다운로드: [${filename}](${absoluteUrl})`;
  });
};

// --- 오른쪽: 대화하기 ---
function RagChatPanel({
  chatHistory,
  onChat,
  selectedChunk,
  provider,
  model,
  temperature,
  providerOptions,
  modelOptions,
  onProviderChange,
  onModelChange,
  onTemperatureChange,
  onNewChat,
  isLoading,
  ragSearching,
  aiResponding
}: RagChatPanelProps & {
  onNewChat: () => void;
  isLoading: boolean;
  ragSearching?: boolean;
  aiResponding?: boolean;
}) {
  // 채팅 입력 상태
  const [input, setInput] = useState("");
  const inputRef = useRef<HTMLTextAreaElement>(null);
  const scrollRef = useRef<HTMLDivElement>(null);
  const [thinkingDots, setThinkingDots] = useState("");

  // 검색 범위 상태
  const [searchScope, setSearchScope] = useState("personal");
  // Think... 애니메이션
  useEffect(() => {
    if (!isLoading) {
      setThinkingDots("");
      return;
    }
    let i = 0;
    const interval = setInterval(() => {
      setThinkingDots(".".repeat((i % 3) + 1));
      i++;
    }, 500);
    return () => clearInterval(interval);
  }, [isLoading]);

  // 채팅 전송 핸들러
  const handleSend = async () => {
    if (!input.trim()) return;
    await onChat(input, searchScope);
    setInput("");
    inputRef.current?.focus();
  };

  // 엔터+Shift: 줄바꿈, 엔터만: 전송
  const handleKeyDown = (e: React.KeyboardEvent<HTMLTextAreaElement>) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      handleSend();
    }
  };

  // 새 메시지 오면 스크롤
  useEffect(() => {
    if (scrollRef.current) {
      scrollRef.current.scrollTop = scrollRef.current.scrollHeight;
    }
  }, [chatHistory]);

  return (
    <div className="flex h-full">
      <div className="w-full h-full flex flex-col" style={{ backgroundColor: "var(--body-bg)" }}>
        {/* 대화하기 제목 + 새 대화 버튼만 */}
        <div className="px-4 py-3 pt-6 pr-6 border-b flex items-center gap-2 min-w-0" style={{ borderColor: "var(--card-border)", backgroundColor: "var(--body-bg)" }}>
          <h3 className="font-semibold flex-1 min-w-0 truncate text-gray-900 dark:text-gray-100">AI 응답 확인</h3>
          <button className="px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg text-base whitespace-nowrap flex-shrink-0 font-semibold shadow-lg hover:shadow-xl transition-all duration-200 transform hover:-translate-y-0.5" onClick={onNewChat} disabled={isLoading}>
            새 대화
          </button>
        </div>
        {/* 대화 내역 (상단) */}
        <div className="flex-1 flex flex-col p-2 pt-10 overflow-y-auto gap-2 max-h-[600px] min-h-0" ref={scrollRef}>
          {chatHistory.length === 0 ? (
            <div className="text-base text-gray-500 dark:text-gray-400 mb-2 text-left pl-2">
              아래 입력창에서 문서에 대한 질문을 입력하세요.
              <br />
              <br />
              특정 키워드를 찾고 싶은 경우 <span className="text-blue-500 dark:text-blue-400 font-semibold">" 또는 ' 로 묶어서</span> 물어보세요.
              <br />
              예) <span className="text-blue-500 dark:text-blue-400 font-semibold">"프로젝트 A의 진행 상황"</span>
              <br />
              <br />
              <span className="text-blue-500 dark:text-blue-400 font-semibold">[] 또는 ()</span> 특수문자가 있는 키워드의 경우 특수문자를 포함하세요.
              <br />
              예) <span className="text-blue-500 dark:text-blue-400 font-semibold">[설비 1]</span> 또는 <span className="text-blue-500 dark:text-blue-400 font-semibold">(품목 1)</span>
            </div>
          ) : (
            chatHistory.map((msg, i) =>
              msg.role === "rag" ? (
                <div key={i} className="mb-6 w-full">
                  {/* 문서 검색 결과 헤더 - 투명한 배경 */}
                  <div className="flex items-center gap-3 mb-4 p-3 bg-transparent border-b-2 border-blue-300 dark:border-blue-600">
                    <div className="text-base text-blue-700 dark:text-blue-300 font-bold flex items-center gap-2">🔍 문서 검색 결과</div>
                    <div className="text-sm text-gray-600 dark:text-gray-400">{msg.timestamp && new Date(msg.timestamp).toLocaleTimeString()}</div>
                    {/* 검색 범위 표시 - 투명한 배경 */}
                    <div className="text-sm px-3 py-1 rounded-full bg-blue-100/80 dark:bg-blue-800/80 text-blue-700 dark:text-blue-300 font-medium border border-blue-300 dark:border-blue-600">
                      {msg.searchScope === "personal" && "📁 개인 문서"}
                      {msg.searchScope === "shared" && "🔗 공용 문서"}
                      {msg.searchScope === "all" && "🌐 전체 검색"}
                    </div>
                  </div>

                  {/* 문서 검색 결과 내용 */}
                  {msg.error ? (
                    <div className="bg-red-50 dark:bg-red-900/30 border-2 border-red-300 dark:border-red-600 rounded-b-lg p-4 text-base text-red-800 dark:text-red-200 shadow-md">❌ {msg.error}</div>
                  ) : (msg.content || []).length === 0 ? (
                    <div className="bg-yellow-50 dark:bg-yellow-900/30 border-2 border-yellow-300 dark:border-yellow-600 rounded-b-lg p-4 text-base text-yellow-800 dark:text-yellow-200 shadow-md">⚠️ 관련 문서를 찾을 수 없습니다.</div>
                  ) : (
                    <div className="space-y-4 p-4 border-2 rounded-b-lg shadow-lg" style={{ backgroundColor: "var(--card-bg)", borderColor: "var(--accent-blue)" }}>
                      {(msg.content || []).map((chunk: any, idx: number) => (
                        <div key={idx} className="bg-gradient-to-br from-blue-50 to-blue-100 dark:from-blue-900/40 dark:to-blue-800/40 border-2 border-blue-300 dark:border-blue-600 rounded-xl p-4 shadow-md hover:shadow-lg transition-shadow duration-200">
                          <div className="flex items-start justify-between mb-3">
                            <div className="font-bold text-blue-900 dark:text-blue-100 text-sm break-all max-w-[80%] flex items-center gap-2">📄 {chunk.source ? chunk.source.split("/").pop() : "알 수 없는 문서"}</div>
                            <div className="text-sm text-blue-700 dark:text-blue-300 ml-2 flex-shrink-0 bg-blue-200 dark:bg-blue-700 px-2 py-1 rounded-full font-bold">#{idx + 1}</div>
                          </div>
                          <div className="text-gray-800 dark:text-gray-100 text-sm whitespace-pre-line break-all mb-4 p-3 bg-white/80 dark:bg-gray-700/80 rounded-lg border border-blue-200 dark:border-blue-600 shadow-inner">{chunk.content}</div>
                          <details className="text-sm" open>
                            <summary className="cursor-pointer text-blue-700 dark:text-blue-300 hover:text-blue-900 dark:hover:text-blue-100 font-medium bg-blue-100 dark:bg-blue-800 px-3 py-2 rounded-lg hover:bg-blue-200 dark:hover:bg-blue-700 transition-colors duration-200">📊 상세 점수 보기</summary>
                            <div className="mt-3 p-3 bg-white/90 dark:bg-gray-800/90 rounded-lg text-gray-700 dark:text-gray-200 border border-blue-200 dark:border-blue-600">
                              <div className="grid grid-cols-1 gap-3">
                                <div>
                                  <strong className="text-blue-800 dark:text-blue-200">4-Score RAG 점수:</strong>
                                  <br />
                                  키워드 정확도: {chunk.core_scores?.keyword_precision?.toFixed(3) || "0.000"}
                                  <br />
                                  의미 관련성: {chunk.core_scores?.semantic_relevance?.toFixed(3) || "0.000"}
                                  <br />
                                  내용 품질: {chunk.core_scores?.content_quality?.toFixed(3) || "0.000"}
                                  <br />
                                  최신성: {chunk.core_scores?.recency?.toFixed(3) || "0.000"}
                                </div>
                              </div>
                              {(chunk.matches?.exact_matches?.length > 0 || chunk.matches?.partial_matches?.length > 0) && (
                                <div className="mt-3 pt-3 border-t border-blue-300 dark:border-blue-600">
                                  {chunk.matches?.exact_matches?.length > 0 && (
                                    <div>
                                      <strong>정확 매칭:</strong> {chunk.matches.exact_matches.map((m: any) => m.query).join(", ")}
                                    </div>
                                  )}
                                  {chunk.matches?.partial_matches?.length > 0 && (
                                    <div>
                                      <strong>부분 매칭:</strong> {chunk.matches.partial_matches.map((m: any) => m.query).join(", ")}
                                    </div>
                                  )}
                                </div>
                              )}
                            </div>
                          </details>
                        </div>
                      ))}

                      {/* Top으로 이동하는 앵커 */}
                      <div className="flex justify-end mt-4 mb-2">
                        <button
                          onClick={() => {
                            if (scrollRef.current) {
                              scrollRef.current.scrollTop = 0;
                            }
                          }}
                          className="px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white text-sm rounded-lg shadow-md hover:shadow-lg transition-all duration-200 transform hover:-translate-y-1 font-medium"
                          title="맨 위로 이동"
                        >
                          ↑ Top
                        </button>
                      </div>
                    </div>
                  )}
                </div>
              ) : (
                <div key={i} className={`flex ${msg.role === "user" ? "justify-end" : "justify-start"} mb-4`}>
                  <div
                    className={`rounded-2xl px-4 py-3 max-w-[95%] whitespace-pre-wrap break-all text-base shadow-md
                      ${msg.role === "user" ? "bg-gradient-to-r from-blue-500 to-blue-600 text-white rounded-br-md" : "bg-gradient-to-r from-gray-100 to-gray-200 dark:from-gray-700 dark:to-gray-800 text-gray-900 dark:text-gray-100 rounded-bl-md border border-gray-300 dark:border-gray-600"}`}
                  >
                    {msg.role === "assistant" ? <CustomMarkdownRenderer content={convertDownloadLinksToAbsolute(msg.content)} /> : <div className="break-all text-white">{msg.content}</div>}
                    {msg.role === "assistant" && msg.elapsed && <div className="text-xs text-gray-500 dark:text-gray-400 mt-2 text-right">응답 시간: {msg.elapsed}초</div>}
                  </div>
                </div>
              )
            )
          )}
          {ragSearching && (
            <div className="flex justify-start mb-4">
              <div className="rounded-2xl px-6 py-3 bg-gradient-to-r from-blue-500 to-blue-600 text-white text-base shadow-lg animate-pulse border-2 border-blue-300 dark:border-blue-500">🔍 문서 검색 중{thinkingDots}</div>
            </div>
          )}
          {aiResponding && (
            <div className="flex justify-start mb-4">
              <div className="rounded-2xl px-6 py-3 bg-gradient-to-r from-purple-500 to-purple-600 text-white text-base shadow-lg animate-pulse border-2 border-purple-300 dark:border-purple-500">🤖 AI 응답 생성 중{thinkingDots}</div>
            </div>
          )}
        </div>
        {/* Provider/Model/Temperature 설정 UI (입력창 바로 위) */}
        <div className="flex flex-row gap-2 px-4 py-2 pt-10 items-end border-t" style={{ borderColor: "var(--card-border)", backgroundColor: "var(--body-bg)" }}>
          <div className="flex items-center gap-1">
            <span className="text-sm text-gray-500 dark:text-gray-400">Provider</span>
            <select className="text-sm h-9 border border-gray-300 dark:border-gray-600 rounded px-3 py-2 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 text-base" value={provider} onChange={(e) => onProviderChange(e.target.value)} disabled={providerOptions.length === 0}>
              {providerOptions.length === 0 ? (
                <option>로딩 중...</option>
              ) : (
                providerOptions.map((opt) => (
                  <option key={opt.value} value={opt.value}>
                    {opt.label}
                  </option>
                ))
              )}
            </select>
          </div>
          <div className="flex items-center gap-1">
            <span className="text-sm text-gray-500 dark:text-gray-400">Model</span>
            <select className="text-sm h-9 border border-gray-300 dark:border-gray-600 rounded px-3 py-2 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 text-base" value={model} onChange={(e) => onModelChange(e.target.value)} disabled={modelOptions.length === 0}>
              {modelOptions.length === 0 ? (
                <option>로딩 중...</option>
              ) : (
                modelOptions.map((opt) => (
                  <option key={opt.value} value={opt.value}>
                    {opt.label}
                  </option>
                ))
              )}
            </select>
          </div>
        </div>
        <div className="px-4 mt-2 mb-1">
          <div className="flex items-center gap-2 min-w-[100px]">
            <span className="text-sm text-gray-500 dark:text-gray-400">Temperature: {temperature}</span>
            <input type="range" min={0} max={2} step={0.1} value={temperature} onChange={(e) => onTemperatureChange(Number(e.target.value))} className="h-2 bg-gray-200 dark:bg-gray-600 rounded-lg appearance-none cursor-pointer slider" style={{ width: "150px" }} />
          </div>
        </div>

        {/* 검색 범위 설정 */}
        <div className="px-4 mt-2 mb-1">
          <div className="flex items-center gap-2">
            <span className="text-sm text-gray-500 dark:text-gray-400">검색 범위:</span>
            <select className="text-sm h-9 border border-gray-300 dark:border-gray-600 rounded px-3 py-2 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 text-base" value={searchScope} onChange={(e) => setSearchScope(e.target.value)}>
              <option value="personal">자기문서 (개인)</option>
              <option value="shared">공용문서 (공유)</option>
              <option value="all">전체검색 (모든 문서)</option>
            </select>
            <span className="text-sm text-gray-400 dark:text-gray-500">
              {searchScope === "personal" && "개인 업로드 문서만 검색"}
              {searchScope === "shared" && "공용 문서만 검색"}
              {searchScope === "all" && "모든 문서 검색"}
            </span>
          </div>
        </div>
        <div className="border-t px-4 py-3 flex items-end flex-shrink-0" style={{ borderColor: "var(--card-border)", backgroundColor: "var(--body-bg)", minHeight: "80px" }}>
          <textarea
            ref={inputRef}
            className="flex-1 text-base resize-none rounded border border-gray-300 dark:border-gray-600 px-3 py-2 text-gray-900 dark:text-gray-100 focus:outline-none focus:ring-2 focus:ring-blue-400 min-h-[48px] max-h-32 transition disabled:bg-gray-100 dark:disabled:bg-gray-700"
            style={{ backgroundColor: "var(--sidebar-bg)" }}
            placeholder="질문을 입력하세요"
            value={input}
            onChange={(e) => setInput(e.target.value)}
            onKeyDown={handleKeyDown}
            disabled={isLoading}
            rows={7}
          />
        </div>
      </div>
    </div>
  );
}

// --- 새로 추가: 성능 최적화 관련 인터페이스 ---
interface TestQuery {
  id: string;
  query: string;
  category: string;
  expectedResults?: number;
  description?: string;
}

interface OptimizationResult {
  config: Record<string, number>;
  score: number;
  metrics: {
    precision: number;
    recall: number;
    relevance: number;
    speed: number;
  };
  testResults: Array<{
    query: string;
    resultCount: number;
    avgScore: number;
    topScore: number;
  }>;
  message?: string;
}

interface RagOptimizationPanelProps {
  settings: Record<string, any>;
  onSettingsChange: (settings: Record<string, any>) => void;
  user?: any;
}

// --- 기본 테스트 쿼리 세트 ---
const DEFAULT_TEST_QUERIES: TestQuery[] = [
  {
    id: "1",
    query: "AI 설치 방법",
    category: "user_need",
    expectedResults: 1,
    description: "사용자가 실제로 알고 싶어하는 정보 (설치)"
  },
  {
    id: "2",
    query: "문서 업로드 후 검색이 안되는 문제",
    category: "user_need",
    expectedResults: 1,
    description: "사용자가 실제로 겪는 문제 해결"
  },
  {
    id: "3",
    query: "RAG 검색 결과가 부정확할 때 해결 방법",
    category: "user_need",
    expectedResults: 1,
    description: "사용자가 실제로 필요한 해결책"
  },
  {
    id: "4",
    query: "API 사용법과 예제",
    category: "user_need",
    expectedResults: 1,
    description: "사용자가 실제로 찾는 실용적 정보"
  },
  {
    id: "5",
    query: "에러 메시지 해결 방법",
    category: "user_need",
    expectedResults: 1,
    description: "사용자가 실제로 필요한 문제 해결"
  },
  {
    id: "6",
    query: "성능 향상을 위한 설정 최적화",
    category: "user_need",
    expectedResults: 1,
    description: "사용자가 실제로 원하는 개선 방법"
  },
  {
    id: "7",
    query: "긴 질문을 하겠습니다. AI에서 대용량 PDF 문서를 업로드했는데 검색이 느려지는 현상을 경험하고 있습니다. 어떻게 해결할 수 있을까요?",
    category: "user_need",
    expectedResults: 1,
    description: "사용자가 실제로 경험하는 복잡한 문제"
  },
  {
    id: "8",
    query: "화성 여행 계획서 작성법",
    category: "out_of_scope",
    expectedResults: 0,
    description: "시스템 범위 밖의 질의 (적절한 필터링 확인)"
  }
];

// --- 4-Score System 최적화 대상 파라미터 정의 ---
const OPTIMIZATION_PARAMS = {
  RAG_KEYWORD_PRECISION_THRESHOLD: { min: 0.05, max: 0.25, step: 0.05 }, // 키워드 정확도 임계값
  RAG_SEMANTIC_RELEVANCE_THRESHOLD: { min: 0.1, max: 0.3, step: 0.05 }, // 의미 관련성 임계값
  RAG_CONTENT_QUALITY_THRESHOLD: { min: 0.2, max: 0.5, step: 0.05 }, // 내용 품질 임계값
  RAG_RECENCY_THRESHOLD: { min: 0.0, max: 0.2, step: 0.05 } // 최신성 임계값
};

// 4-Score System 설정키의 사용자 친화적 한국어 이름 매핑
const OPTIMIZATION_PARAM_LABELS = {
  RAG_KEYWORD_PRECISION_THRESHOLD: "키워드 정확도 임계값",
  RAG_SEMANTIC_RELEVANCE_THRESHOLD: "의미 관련성 임계값",
  RAG_CONTENT_QUALITY_THRESHOLD: "내용 품질 임계값",
  RAG_RECENCY_THRESHOLD: "최신성 임계값"
};

// --- RagOptimizationPanel 구현 ---
function RagOptimizationPanel({ settings, onSettingsChange, user }: RagOptimizationPanelProps) {
  const [testQueries, setTestQueries] = useState<TestQuery[]>(DEFAULT_TEST_QUERIES);
  const [isOptimizing, setIsOptimizing] = useState(false);
  const [optimizationProgress, setOptimizationProgress] = useState(0);
  const [currentConfig, setCurrentConfig] = useState<string>("");
  const [optimizationResults, setOptimizationResults] = useState<OptimizationResult[]>([]);
  const [bestResult, setBestResult] = useState<OptimizationResult | null>(null);
  const [isTestingCurrent, setIsTestingCurrent] = useState(false);
  const [currentTestResults, setCurrentTestResults] = useState<any>(null);
  const [showAddQuery, setShowAddQuery] = useState(false);
  const [newQuery, setNewQuery] = useState({ query: "", category: "general", description: "" });
  const [isGeneratingQueries, setIsGeneratingQueries] = useState(false);
  const [modal, setModal] = useState<{ type: "success" | "error"; title: string; message: string } | null>(null);

  // 의미 있는 키워드 추출 함수 (개선된 버전)
  const extractMeaningfulKeywords = (text: string): string[] => {
    if (!text) return [];

    // 1. 불용어 목록 (의미 없는 단어들)
    const stopWords = new Set([
      "the",
      "and",
      "or",
      "but",
      "in",
      "on",
      "at",
      "to",
      "for",
      "of",
      "with",
      "by",
      "from",
      "is",
      "are",
      "was",
      "were",
      "be",
      "been",
      "have",
      "has",
      "had",
      "do",
      "does",
      "did",
      "will",
      "would",
      "could",
      "should",
      "may",
      "might",
      "this",
      "that",
      "these",
      "those",
      "a",
      "an",
      "as",
      "if",
      "then",
      "than",
      "so",
      "very",
      "can",
      "just",
      "now",
      "번호",
      "아이디",
      "식별자",
      "값",
      "내용",
      "항목",
      "요소",
      "구성",
      "이것",
      "그것",
      "저것",
      "것이",
      "것을",
      "것은",
      "최종",
      "버전",
      "파일",
      "문서",
      "페이지",
      "제목",
      "섹션",
      "항목",
      "내용",
      "설명",
      "참고",
      "주의",
      "그림",
      "표"
    ]);

    // 2. 파일명 특수처리 (확장자 제거, 특수문자 분리)
    let processedText = text;

    // 확장자 제거
    processedText = processedText.replace(/\.\w+$/, "");

    // 특수문자로 단어 분리 (언더스코어, 하이픈, 괄호 등)
    processedText = processedText.replace(/[_\-\(\)\[\]{}\.]/g, " ");

    // 숫자와 한글/영문 사이에 공백 추가 (예: "240610강진품애" → "240610 강진품애")
    processedText = processedText.replace(/(\d)([가-힣a-zA-Z])/g, "$1 $2");
    processedText = processedText.replace(/([가-힣a-zA-Z])(\d)/g, "$1 $2");

    // 3. 다양한 패턴으로 단어 추출
    const patterns = [
      /[가-힣]{2,}/g, // 한글 단어 (2글자 이상)
      /[A-Za-z]{3,}/g, // 영문 단어 (3글자 이상)
      /\b\w+\b/g // 일반 단어
    ];

    const allWords = new Set<string>();
    patterns.forEach((pattern) => {
      const matches = processedText.match(pattern) || [];
      matches.forEach((word) => allWords.add(word));
    });

    // 4. 키워드 빈도 계산
    const wordFreq = new Map<string, number>();
    Array.from(allWords).forEach((word) => {
      const normalized = word.toLowerCase();
      if (!stopWords.has(normalized) && !stopWords.has(word)) {
        wordFreq.set(word, (wordFreq.get(word) || 0) + 1);
      }
    });

    // 5. 의미 있는 키워드 필터링 및 점수 계산
    const scoredKeywords = Array.from(wordFreq.entries())
      .filter(([word, freq]) => {
        // 숫자만 있는 단어 제외
        if (/^\d+$/.test(word)) return false;

        // 연속된 숫자 제외 (예: 061, 430, 3073)
        if (/^\d{3,}$/.test(word)) return false;

        // 반복 문자 제외 (예: aaaa, bbbb)
        if (/^(.)\1{3,}$/.test(word)) return false;

        // 너무 긴 단어 제외 (15글자 이상)
        if (word.length > 15) return false;

        // 너무 짧은 단어 제외 (한글 1글자, 영문 2글자 이하)
        if ((/^[가-힣]+$/.test(word) && word.length < 2) || (/^[A-Za-z]+$/.test(word) && word.length < 3)) {
          return false;
        }

        return true;
      })
      .map(([word, freq]) => {
        let score = freq; // 기본 빈도 점수

        // 한글 키워드에 높은 보너스 점수
        if (/^[가-힣]+$/.test(word)) {
          score += 5; // 한글 키워드 우선
        }

        // 의미 있는 키워드 보너스
        if (/^(정책|가이드|계획|방법|관리|운영|시스템|서비스|사업|프로그램|지원|현황|분석|보고서|매뉴얼|절차|규정|품질|개발|설계|구현|테스트|배포|운영|유지보수)$/i.test(word)) {
          score += 3;
        }

        // AI.RUN 관련 키워드에 보너스 점수
        if (/^(AI|RUN|chat|admin|api|config|setting|user|system|server|database|model|llm|rag|embedding|vector|search|retrieval)$/i.test(word)) {
          score += 2;
        }

        // 길이에 따른 보너스 (적당한 길이가 좋음)
        if (word.length >= 3 && word.length <= 8) {
          score += 1;
        }

        return { word, score };
      })
      .sort((a, b) => b.score - a.score); // 점수 높은 순으로 정렬

    // 6. 중복 제거 및 다양성 확보
    const uniqueKeywords = new Set<string>();
    const finalKeywords: string[] = [];

    for (const { word } of scoredKeywords) {
      const normalized = word.toLowerCase();
      if (!uniqueKeywords.has(normalized)) {
        uniqueKeywords.add(normalized);
        finalKeywords.push(word);
        if (finalKeywords.length >= 10) break;
      }
    }

    // console.log(`키워드 추출 결과: "${text}" → [${finalKeywords.join(', ')}]`);
    return finalKeywords;
  };

  // 실제 문서 기반 테스트 쿼리 자동 생성 (RAG 검색 없이 빠르게)
  const generateDocumentBasedQueries = async () => {
    setIsGeneratingQueries(true);
    try {
      // RAG 서버 URL 가져오기
      const ragServerUrl = getRagServerUrl();

      // RAG 서버의 전용 키워드 추출 엔드포인트 호출
      const response = await fetch(`${ragServerUrl}/extract-keywords`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          user_id: user?.username || "admin",
          max_chunks: 20,
          max_keywords: 6
        })
      });

      if (!response.ok) {
        throw new Error("키워드 추출 API 호출 실패");
      }

      const data = await response.json();

      if (data.success && data.keywords && data.keywords.length > 0) {
        // 서버에서 받은 키워드를 테스트 쿼리로 변환
        const documentBasedQueries: TestQuery[] = data.keywords.map((item: any, index: number) => ({
          id: `keyword_${Date.now()}_${index}`,
          query: item.keyword,
          category: "user_need",
          expectedResults: 1,
          description: item.description || "추출된 키워드"
        }));

        // 필터링 테스트용 쿼리 추가
        documentBasedQueries.push({
          id: `filter_${Date.now()}`,
          query: "화성 여행 계획서 작성법",
          category: "out_of_scope",
          expectedResults: 0,
          description: "시스템 범위 밖의 질의 (적절한 필터링 확인)"
        });

        // 기존 키워드를 완전히 교체
        setTestQueries(documentBasedQueries);
        console.log(`키워드 추출 완료: ${data.keywords.length}개 키워드, ${data.total_chunks_processed}개 청크 처리됨`);
      } else {
        // 키워드 추출 실패시 기본값 사용
        throw new Error("키워드 추출 결과 없음");
      }
    } catch (error) {
      console.error("키워드 생성 실패:", error);

      // 오류 발생시 기본 쿼리 사용
      setTestQueries([
        {
          id: `default_1_${Date.now()}`,
          query: "정책 가이드",
          category: "user_need",
          expectedResults: 1,
          description: "기본 정책 관련 질의"
        },
        {
          id: `default_2_${Date.now()}`,
          query: "사업 계획",
          category: "user_need",
          expectedResults: 1,
          description: "사업 관련 질의"
        },
        {
          id: `default_3_${Date.now()}`,
          query: "운영 방법",
          category: "user_need",
          expectedResults: 1,
          description: "운영 관련 질의"
        },
        {
          id: `filter_default_${Date.now()}`,
          query: "화성 여행 계획서",
          category: "out_of_scope",
          expectedResults: 0,
          description: "필터링 테스트"
        }
      ]);
    } finally {
      setIsGeneratingQueries(false);
    }
  };

  // 현재 설정으로 성능 테스트
  const testCurrentSettings = async () => {
    setIsTestingCurrent(true);
    try {
      // 배치 검색을 사용하여 모든 쿼리를 한 번에 처리
      const ragServerUrl = getRagServerUrl();
      const batchResponse = await fetch(`${ragServerUrl}/batch-search`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          keywords: testQueries.map((q) => q.query),
          user_id: user?.username || "admin",
          settings: {
            ...settings,
            top_k: 5 // 빠른 테스트를 위해 5개로 제한
          }
        })
      });

      if (batchResponse.ok) {
        const batchData = await batchResponse.json();

        if (batchData.success) {
          // 배치 결과를 기존 형식으로 변환
          const testResults: Array<{
            query: string;
            resultCount: number;
            avgScore: number;
            topScore: number;
          }> = [];
          let totalScore = 0;
          let totalPrecision = 0;
          let totalRecall = 0;
          let totalRelevance = 0;

          for (let i = 0; i < testQueries.length; i++) {
            const testQuery = testQueries[i];
            const result = batchData.results?.[i];

            if (result) {
              // 새로운 batch-search 응답 형식 사용
              const resultCount = result.numResults || 0;
              const avgScore = (result.avgScore || 0) / 100; // 100점 만점을 0-1로 변환

              // 성능 지표 계산
              let score = 0;
              let precision = 0;
              let recall = 0;

              if (testQuery.category === "out_of_scope") {
                score = resultCount === 0 ? 0.8 : 0.2;
                precision = resultCount === 0 ? 1 : 0.1;
                recall = 1;
              } else {
                if (resultCount === 0) {
                  score = 0.0;
                  precision = 0.0;
                  recall = 0.0;
                } else {
                  score = Math.min(1.0, 0.8 + Math.min(0.2, resultCount * 0.05) + avgScore * 0.1);
                  precision = Math.max(0.7, avgScore);
                  recall = Math.min(1, resultCount / (testQuery.expectedResults || 1));
                }
              }

              totalScore += score;
              totalPrecision += precision;
              totalRecall += recall;
              totalRelevance += avgScore;

              testResults.push({
                query: testQuery.query,
                resultCount,
                avgScore,
                topScore: avgScore
              });
            }
          }

          // 최종 결과 계산
          const finalResults = {
            config: settings,
            score: totalScore / testQueries.length,
            metrics: {
              precision: totalPrecision / testQueries.length,
              recall: totalRecall / testQueries.length,
              relevance: totalRelevance / testQueries.length,
              speed: 100 // 배치 처리는 빠름
            },
            testResults
          };

          setCurrentTestResults(finalResults);
          console.log("배치 테스트 완료:", batchData.successful_queries, "/", batchData.total_queries);
          return;
        }
      }

      // 배치 검색 실패 시 에러 처리
      console.error("배치 검색 API 호출 실패");
      setModal({ type: "error", title: "오류", message: "성능 테스트 서버 오류가 발생했습니다. 잠시 후 다시 시도해주세요." });
    } catch (error) {
      console.error("현재 설정 성능 테스트 실패:", error);
    } finally {
      setIsTestingCurrent(false);
    }
  };

  // 자동 최적화 실행 (개선된 버전 - 스마트 샘플링)
  const runOptimization = async () => {
    setIsOptimizing(true);
    setBestResult(null);
    setOptimizationResults([]);

    try {
      // RAG 서버의 자동 최적화 엔드포인트 호출
      const ragServerUrl = getRagServerUrl();
      const response = await fetch(`${ragServerUrl}/auto-optimize`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          user_id: user?.username || "admin",
          test_queries: testQueries.map((q) => q.query)
        })
      });

      if (!response.ok) {
        throw new Error("최적화 API 호출 실패");
      }

      const data = await response.json();

      if (data.success && data.best_config) {
        // 최적 결과 설정 - 서버에서 받은 실제 메트릭 사용
        // 서버에서 이미 백분율로 제공 (예: 86.57 = 86.57%)
        setBestResult({
          config: data.best_config,
          score: (data.best_score || 0) / 100, // 백분율을 0-1 범위로 변환
          metrics: {
            precision: (data.precision_score || data.best_score || 0) / 100, // 백분율을 0-1 범위로 변환
            recall: (data.relevance_score || data.best_score || 0) / 100, // relevance_score를 recall로 사용
            relevance: (data.relevance_score || data.best_score || 0) / 100,
            speed: data.metrics?.speed || 1500 // ms 단위
          },
          testResults: data.test_results || [],
          message: data.message // 서버 메시지
        });

        console.log("최적화 완료:", data);
      } else {
        throw new Error(data.error || "최적화 실패");
      }
    } catch (error) {
      console.error("자동 최적화 실패:", error);
      // 서버 오류 시 사용자에게 알림
      setModal({ type: "error", title: "오류", message: "자동 최적화 서버 오류가 발생했습니다. 잠시 후 다시 시도해주세요." });
    } finally {
      setIsOptimizing(false);
    }
  };

  // 4-Score System: 가중치 정규화가 더 이상 필요하지 않음 (임계값만 사용)

  // 최적 설정 적용 (안전한 방식으로 개선)
  const applyBestSettings = async () => {
    if (!bestResult) return;

    try {
      // 자동최적화 대상 파라미터만 필터링하고 값 검증
      const optimizationKeys = Object.keys(OPTIMIZATION_PARAMS);
      const validSettings: Record<string, string> = {};

      for (const key of optimizationKeys) {
        const value = bestResult.config[key];
        if (value !== undefined && value !== null) {
          // 숫자값을 소수점 3자리로 반올림하여 문자열로 변환
          const numValue = typeof value === "number" ? value : Number(value);
          validSettings[key] = (Math.round(numValue * 1000) / 1000).toString();
        }
      }

      console.log("적용할 설정값:", validSettings);
      console.log("현재 설정값 (참고용):", settings);

      // 순차적으로 설정 저장 (동시성 문제 방지)
      const failedKeys: string[] = [];

      for (const [key, value] of Object.entries(validSettings)) {
        try {
          await setConfig(key, value);
          console.log(`✅ ${key} 저장 성공:`, value);
        } catch (error) {
          console.error(`❌ ${key} 저장 실패:`, error);
          failedKeys.push(key);
        }
        // 각 설정 저장 후 잠시 대기 (서버 부하 방지)
        await new Promise((resolve) => setTimeout(resolve, 100));
      }

      if (failedKeys.length > 0) {
        throw new Error(`다음 설정 저장 실패: ${failedKeys.join(", ")}`);
      }

      // 기존 설정을 보존하면서 새 설정만 업데이트 (안전한 병합)
      const updatedSettings = { ...settings };
      Object.entries(validSettings).forEach(([key, value]) => {
        updatedSettings[key] = value;
      });

      // 로컬 상태 업데이트
      onSettingsChange(updatedSettings);

      console.log("✅ 모든 설정 적용 완료. 업데이트된 설정:", updatedSettings);
      setModal({ type: "success", title: "성공", message: "최적 설정이 성공적으로 적용되었습니다!" });
    } catch (error) {
      console.error("설정 적용 실패:", error);
      setModal({ type: "error", title: "오류", message: `설정 적용에 실패했습니다: ${error instanceof Error ? error.message : "알 수 없는 오류"}` });
    }
  };

  // 테스트 쿼리 추가 (쉼표로 구분된 여러 키워드 지원)
  const addTestQuery = () => {
    if (!newQuery.query.trim()) return;

    // 쉼표로 구분하여 키워드 배열로 변환
    const keywords = newQuery.query
      .split(",")
      .map((k) => k.trim())
      .filter((k) => k.length > 0);

    if (keywords.length === 0) return;

    // 기존 키워드와 중복 체크
    const existingQueries = new Set(testQueries.map((q) => q.query.toLowerCase()));
    const newKeywords = keywords.filter((keyword) => !existingQueries.has(keyword.toLowerCase()));
    const duplicateKeywords = keywords.filter((keyword) => existingQueries.has(keyword.toLowerCase()));

    if (duplicateKeywords.length > 0) {
      setModal({ type: "error", title: "중복 키워드", message: `다음 키워드는 이미 존재합니다: ${duplicateKeywords.join(", ")}` });
    }

    if (newKeywords.length === 0) {
      if (duplicateKeywords.length > 0) return;
    }

    // 새 키워드들을 TestQuery 객체로 변환
    const newQueries: TestQuery[] = newKeywords.map((keyword, index) => ({
      id: `${Date.now()}_${index}`,
      query: keyword,
      category: "user_need",
      description: "수동 추가된 키워드",
      expectedResults: 1
    }));

    setTestQueries((prev) => [...prev, ...newQueries]);
    setNewQuery({ query: "", category: "general", description: "" });
    setShowAddQuery(false);

    if (newKeywords.length > 0) {
      console.log(`${newKeywords.length}개의 새 키워드가 추가되었습니다:`, newKeywords);
    }
  };

  // 테스트 쿼리 삭제
  const removeTestQuery = (id: string) => {
    setTestQueries((prev) => prev.filter((q) => q.id !== id));
  };

  return (
    <div className="p-4 shadow mb-4" style={{ backgroundColor: "var(--sidebar-bg)" }}>
      {/* 🚀 쉬운 설정 (Smart Tune) */}
      <div className="mb-6 p-4 bg-gradient-to-r from-blue-50 to-green-50 dark:from-blue-900/20 dark:to-green-900/20 border-2 border-blue-200 dark:border-blue-700 rounded-lg">
        <h3 className="text-lg font-bold text-blue-800 dark:text-blue-200 mb-4 flex items-center gap-2">RAG 설정</h3>

        <div className="grid grid-cols-1 gap-4">
          {/* 문서 처리 모드 */}
          <div>
            <label className="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">📄 문서 처리 모드</label>
            <select
              value={settings["RAG_MODE"] || "normal"}
              onChange={(e) => {
                const newValue = e.target.value;

                // RAG 모드 변경에 따른 자동 설정
                const autoSettings: Record<string, string> = {};

                if (newValue === "fast") {
                  // Fast 모드: 청크전략: default, 텍스트 우선: 활성화, 이미지 임베딩 생성: 비활성화, 예상질문 AI생성: 비활성화
                  autoSettings["RAG_CHUNKING_STRATEGY"] = "default";
                  autoSettings["RAG_PROCESS_TEXT_ONLY"] = "yes";
                  autoSettings["RAG_GENERATE_IMAGE_EMBEDDINGS"] = "no";
                  autoSettings["USE_ENHANCED_EMBEDDING"] = "no";
                  autoSettings["RAG_ENABLE_GRAPHRAG"] = "no";
                } else if (newValue === "normal") {
                  // Normal 모드: 청크전략: page, 텍스트 우선: 활성화, 이미지 임베딩 생성: 활성화, 예상질문 AI생성: 비활성화
                  autoSettings["RAG_CHUNKING_STRATEGY"] = "page";
                  autoSettings["RAG_PROCESS_TEXT_ONLY"] = "yes";
                  autoSettings["RAG_GENERATE_IMAGE_EMBEDDINGS"] = "yes";
                  autoSettings["USE_ENHANCED_EMBEDDING"] = "no";
                  autoSettings["RAG_ENABLE_GRAPHRAG"] = "no";
                } else if (newValue === "rich") {
                  // Rich 모드: 청크전략: default, 텍스트 우선: 비활성화, 이미지 임베딩 생성: 활성화, 예상질문 AI생성: 활성화
                  autoSettings["RAG_CHUNKING_STRATEGY"] = "default";
                  autoSettings["RAG_PROCESS_TEXT_ONLY"] = "no";
                  autoSettings["RAG_GENERATE_IMAGE_EMBEDDINGS"] = "yes";
                  autoSettings["USE_ENHANCED_EMBEDDING"] = "yes";
                  autoSettings["RAG_ENABLE_GRAPHRAG"] = "yes";
                }

                // RAG_MODE와 자동 설정된 값들을 모두 적용
                const updatedSettings = {
                  ...settings,
                  RAG_MODE: newValue,
                  ...autoSettings
                };

                onSettingsChange(updatedSettings);

                // 자동 설정 알림
                const autoSettingsList = Object.entries(autoSettings)
                  .map(([k, v]) => `${ragConfigMeta[k]?.label || k}: ${v === "yes" ? "활성화" : v === "no" ? "비활성화" : v}`)
                  .join(", ");

                console.log(`🔄 Smart Tune RAG 모드 ${newValue}로 변경됨. 자동 설정: ${autoSettingsList}`);
              }}
              className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100"
            >
              <option value="fast">빠름 - 텍스트만 처리</option>
              <option value="normal">표준 - 텍스트 + 페이지 정보</option>
              <option value="rich">고급 - AI 이미지 설명 + 예상질문 + 지식그래프</option>
            </select>
          </div>

          {/* 문서 검색 모드 */}
          <div>
            <label className="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">🔍 문서 검색 모드</label>
            <select
              value={settings["RAG_SEARCH_MODE"] || "vector"}
              onChange={(e) => {
                const newValue = e.target.value;
                // RAG_SEARCH_MODE만 변경하고 다른 설정은 유지
                onSettingsChange({ ...settings, RAG_SEARCH_MODE: newValue });
              }}
              className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100"
            >
              <option value="vector">벡터 검색 - 유사도/키워드/예상질문</option>
              <option value="hybrid">하이브리드 - 벡터 + 관계 검색</option>
            </select>
          </div>

          {/* Smart Tune 저장 버튼 */}
          <div className="mt-4 flex justify-end">
            <button
              onClick={async () => {
                try {
                  // 현재 Smart Tune 설정값들과 RAG 모드에 따른 자동 설정값들을 서버에 저장
                  const ragMode = settings["RAG_MODE"] || "normal";
                  const searchMode = settings["RAG_SEARCH_MODE"] || "vector";

                  // RAG 모드에 따른 자동 설정값들 계산
                  let autoSettings: Record<string, string> = {};
                  if (ragMode === "fast") {
                    autoSettings = {
                      RAG_CHUNKING_STRATEGY: "default",
                      RAG_PROCESS_TEXT_ONLY: "yes",
                      RAG_GENERATE_IMAGE_EMBEDDINGS: "no",
                      USE_ENHANCED_EMBEDDING: "no",
                      RAG_ENABLE_GRAPHRAG: "no"
                    };
                  } else if (ragMode === "normal") {
                    autoSettings = {
                      RAG_CHUNKING_STRATEGY: "page",
                      RAG_PROCESS_TEXT_ONLY: "yes",
                      RAG_GENERATE_IMAGE_EMBEDDINGS: "yes",
                      USE_ENHANCED_EMBEDDING: "no",
                      RAG_ENABLE_GRAPHRAG: "no"
                    };
                  } else if (ragMode === "rich") {
                    autoSettings = {
                      RAG_CHUNKING_STRATEGY: "default",
                      RAG_PROCESS_TEXT_ONLY: "no",
                      RAG_GENERATE_IMAGE_EMBEDDINGS: "yes",
                      USE_ENHANCED_EMBEDDING: "yes",
                      RAG_ENABLE_GRAPHRAG: "yes"
                    };
                  }

                  const smartTuneSettings: Record<string, string> = {
                    RAG_MODE: ragMode,
                    RAG_SEARCH_MODE: searchMode,
                    ...autoSettings
                  };

                  const failedKeys: string[] = [];
                  const successfulSettings: Record<string, any> = {};

                  for (const [key, value] of Object.entries(smartTuneSettings)) {
                    try {
                      await setConfig(key, value);
                      console.log(`✅ Smart Tune 설정 ${key} 저장 성공:`, value);
                      successfulSettings[key] = value;
                    } catch (error) {
                      console.error(`❌ Smart Tune 설정 ${key} 저장 실패:`, error);
                      failedKeys.push(key);
                    }
                    await new Promise((resolve) => setTimeout(resolve, 50));
                  }

                  // 성공한 설정들은 로컬 상태에 반영
                  if (Object.keys(successfulSettings).length > 0) {
                    const updatedSettings = { ...settings, ...successfulSettings };
                    console.log("🔄 Smart Tune 상태 업데이트:", {
                      before: settings,
                      successful: successfulSettings,
                      after: updatedSettings
                    });
                    onSettingsChange(updatedSettings);
                  }

                  if (failedKeys.length > 0) {
                    setModal({ type: "error", title: "오류", message: `일부 설정 저장 실패: ${failedKeys.join(", ")}` });
                  } else {
                    setModal({ type: "success", title: "성공", message: "설정이 저장되었습니다!" });
                  }
                } catch (error) {
                  console.error("Smart Tune 설정 저장 오류:", error);
                  setModal({ type: "error", title: "오류", message: "설정 저장 중 오류가 발생했습니다." });
                }
              }}
              className="px-6 py-2 bg-green-600 hover:bg-green-700 text-white rounded-lg font-medium shadow-lg hover:shadow-xl transition-all duration-200 transform hover:-translate-y-0.5"
            >
              💾 설정 저장
            </button>
          </div>
        </div>
      </div>

      {/* 고급 최적화 기능 */}
      <div className="mb-6">
        <h4 className="text-base font-semibold text-gray-700 dark:text-gray-300 mb-3 border-b border-gray-200 dark:border-gray-600 pb-2">⚙️ 자동 최적화</h4>
      </div>

      {/* 테스트 쿼리 관리 */}
      <div className="mb-6">
        <div className="mb-3">
          <h4 className="text-base font-semibold text-gray-900 dark:text-gray-100">키워드 관리</h4>
        </div>

        <div className="grid grid-cols-2 gap-3 mb-3">
          <button onClick={generateDocumentBasedQueries} disabled={isGeneratingQueries} className="w-full px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg disabled:opacity-50 text-base font-medium shadow-lg hover:shadow-xl transition-all duration-200 transform hover:-translate-y-0.5">
            {isGeneratingQueries ? "생성 중..." : "📄 키워드 생성"}
          </button>
          <button onClick={() => setShowAddQuery(true)} className="w-full px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg disabled:opacity-50 text-base font-medium shadow-lg hover:shadow-xl transition-all duration-200 transform hover:-translate-y-0.5">
            ⚪ 키워드 추가
          </button>
        </div>

        <div className="flex flex-wrap gap-2">
          {testQueries.map((query) => (
            <span key={query.id} className="inline-flex items-center gap-1 px-3 py-1 bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-200 rounded-full text-sm border border-blue-200 dark:border-blue-700">
              {query.query}
              <button onClick={() => removeTestQuery(query.id)} className="ml-1 text-blue-600 dark:text-blue-300 hover:text-red-500 dark:hover:text-red-400 transition-colors duration-200">
                ×
              </button>
            </span>
          ))}
        </div>
      </div>

      {/* 쿼리 추가 모달 */}
      <div className={`fixed inset-0 z-50 flex items-center justify-center transition-all duration-300 ease-in-out ${showAddQuery ? "opacity-100 pointer-events-auto" : "opacity-0 pointer-events-none"}`} style={{ backgroundColor: "rgba(0, 0, 0, 0.3)" }}>
        <div className={`rounded-lg shadow-lg p-6 min-w-[400px] max-w-[500px] w-full transition-all duration-300 ease-in-out transform ${showAddQuery ? "scale-100 translate-y-0" : "scale-95 translate-y-4"}`} style={{ backgroundColor: "var(--card-bg)" }}>
          <h3 className="text-lg font-bold mb-4">키워드 추가</h3>

          {/* 현재 키워드 목록 */}
          <div className="mb-4">
            <label className="block text-sm font-medium mb-2" style={{ color: "var(--text-primary)" }}>
              현재 키워드
            </label>
            <div
              className="flex flex-wrap gap-2 p-3 rounded-lg min-h-[40px] border"
              style={{
                backgroundColor: "var(--body-hover)",
                borderColor: "var(--card-border)"
              }}
            >
              {testQueries.length > 0 ? (
                testQueries.map((query) => (
                  <span
                    key={query.id}
                    className="inline-block px-2 py-1 rounded text-xs break-words max-w-full"
                    style={{
                      backgroundColor: "var(--primary-100)",
                      color: "var(--primary-800)"
                    }}
                  >
                    {query.query}
                  </span>
                ))
              ) : (
                <span className="text-sm" style={{ color: "var(--text-muted)" }}>
                  키워드가 없습니다
                </span>
              )}
            </div>
          </div>

          <div className="space-y-4">
            <div>
              <label className="block text-base font-medium mb-2" style={{ color: "var(--text-primary)" }}>
                새 키워드 추가
              </label>
              <textarea
                value={newQuery.query}
                onChange={(e) => setNewQuery((prev) => ({ ...prev, query: e.target.value }))}
                className="w-full border rounded-lg px-3 py-2 h-20 resize-none focus:outline-none focus:ring-2"
                style={{
                  borderColor: "var(--input-border)",
                  backgroundColor: "var(--body-bg)",
                  color: "var(--text-primary)"
                }}
                placeholder="키워드를 쉼표(,)로 구분해서 입력하세요. 예: 인공지능, 머신러닝, 딥러닝"
              />
              <p className="text-xs mt-1" style={{ color: "var(--text-muted)" }}>
                쉼표(,)로 구분하여 여러 키워드를 한 번에 추가할 수 있습니다
              </p>
            </div>
          </div>

          <div className="flex justify-end gap-2 mt-6">
            <button onClick={() => setShowAddQuery(false)} className="px-4 py-2 bg-gray-300 dark:bg-gray-600 rounded hover:bg-gray-400 dark:hover:bg-gray-500">
              취소
            </button>
            <button onClick={addTestQuery} className="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600">
              추가
            </button>
          </div>
        </div>
      </div>

      {/* 모달창 */}
      <div className={`fixed inset-0 z-50 flex items-center justify-center transition-all duration-300 ease-in-out ${modal ? "opacity-100 pointer-events-auto" : "opacity-0 pointer-events-none"}`} style={{ backgroundColor: "rgba(0, 0, 0, 0.3)" }}>
        <div
          className={`rounded-lg shadow-lg p-6 min-w-[280px] max-w-xs flex flex-col items-center transition-all duration-300 ease-in-out transform ${modal ? "scale-100 translate-y-0" : "scale-95 translate-y-4"}`}
          style={{
            backgroundColor: "var(--card-bg)",
            boxShadow: "var(--card-shadow)"
          }}
        >
          <div className={`text-lg font-bold mb-2 ${modal?.type === "success" ? "text-green-600" : "text-red-500"}`}>{modal?.title}</div>
          <div className="mb-4 text-center text-base" style={{ color: "var(--text-secondary)" }}>
            <div>{modal?.message}</div>
          </div>
          <button
            className="mt-2 px-4 py-2 rounded transition-all duration-200 text-white"
            style={{
              backgroundColor: "var(--neutral-600)"
            }}
            onMouseEnter={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = "var(--neutral-700)")}
            onMouseLeave={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = "var(--neutral-600)")}
            onClick={() => setModal(null)}
            autoFocus
          >
            닫기
          </button>
        </div>
      </div>

      {/* 현재 설정 테스트 */}
      <div className="mb-6 p-4 bg-blue-50 dark:bg-blue-900/30 rounded-lg">
        <div className="flex items-center justify-between mb-3">
          <h4 className="font-semibold text-blue-800 dark:text-blue-200">현재 설정 성능 테스트</h4>
          <button onClick={testCurrentSettings} disabled={isTestingCurrent} className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 disabled:opacity-50 shadow-lg hover:shadow-xl transition-all duration-200 transform hover:-translate-y-0.5">
            {isTestingCurrent ? "테스트 중..." : "성능 테스트"}
          </button>
        </div>

        {currentTestResults && (
          <div className="space-y-4">
            {/* 기본 성능 메트릭 */}
            <div className="grid grid-cols-4 gap-2 text-center">
              <div className="bg-white dark:bg-gray-200 p-2 rounded">
                <div className="text-sm font-bold text-blue-600">{(currentTestResults.score * 100).toFixed(1)}</div>
                <div className="text-sm text-gray-500">총점</div>
              </div>
              <div className="bg-white dark:bg-gray-200 p-2 rounded">
                <div className="text-sm font-bold text-green-600">{(currentTestResults.metrics.precision * 100).toFixed(1)}%</div>
                <div className="text-sm text-gray-500">정확도</div>
              </div>
              <div className="bg-white dark:bg-gray-200 p-2 rounded">
                <div className="text-sm font-bold text-orange-600">{(currentTestResults.metrics.recall * 100).toFixed(1)}%</div>
                <div className="text-sm text-gray-500">재현율</div>
              </div>
              <div className="bg-white dark:bg-gray-200 p-2 rounded">
                <div className="text-sm font-bold text-purple-600">{(currentTestResults.metrics.speed / 1000).toFixed(2)}s</div>
                <div className="text-sm text-gray-500">응답속도</div>
              </div>
            </div>
          </div>
        )}
      </div>

      {/* 자동 최적화 실행 */}
      <div className="mb-6 p-4 bg-green-50 dark:bg-green-900/30 rounded-lg">
        <div className="flex items-center justify-between mb-3">
          <h4 className="font-semibold text-green-800 dark:text-green-200">자동 최적화</h4>
          <button onClick={runOptimization} disabled={isOptimizing} className="px-4 py-2 bg-green-600 text-white rounded hover:bg-green-700 disabled:opacity-50 shadow-lg hover:shadow-xl transition-all duration-200 transform hover:-translate-y-0.5">
            {isOptimizing ? "최적화 중..." : "최적화 실행"}
          </button>
        </div>

        {isOptimizing && (
          <div className="mb-3">
            <div className="flex justify-between text-sm mb-2">
              <span className="text-gray-600 dark:text-gray-400">최적화 진행 중...</span>
              <span className="text-gray-500">{currentConfig}</span>
            </div>

            {/* 인디터미네이트 프로그레스바 (좌우 왕복 애니메이션) */}
            <div className="relative h-2 bg-gray-200 dark:bg-gray-700 rounded-full overflow-hidden">
              <div className="absolute inset-0">
                <div className="h-full w-1/3 bg-gradient-to-r from-transparent via-green-500 to-transparent rounded-full animate-shimmer"></div>
              </div>
            </div>

            <style jsx>{`
              @keyframes shimmer {
                0% {
                  transform: translateX(-100%);
                }
                100% {
                  transform: translateX(400%);
                }
              }
              .animate-shimmer {
                animation: shimmer 2s linear infinite;
              }
            `}</style>
          </div>
        )}

        {bestResult && (
          <div className="mt-4">
            <div className="flex items-center justify-between mb-3">
              <h5 className="text-sm font-semibold text-green-700 dark:text-green-300">{bestResult.message || "최적 설정 발견!"}</h5>
            </div>

            <div className="grid grid-cols-4 gap-2 text-center mb-4">
              <div className="bg-white dark:bg-green-50 p-2 rounded">
                <div className="text-sm font-bold text-green-600">{(bestResult.score * 100).toFixed(1)}</div>
                <div className="text-sm text-gray-500">최고점</div>
              </div>
              <div className="bg-white dark:bg-green-50 p-2 rounded">
                <div className="text-sm font-bold text-blue-600">{(bestResult.metrics.precision * 100).toFixed(1)}%</div>
                <div className="text-sm text-gray-500">정확도</div>
              </div>
              <div className="bg-white dark:bg-green-50 p-2 rounded">
                <div className="text-sm font-bold text-orange-600">{(bestResult.metrics.recall * 100).toFixed(1)}%</div>
                <div className="text-sm text-gray-500">재현율</div>
              </div>
              <div className="bg-white dark:bg-green-50 p-2 rounded">
                <div className="text-sm font-bold text-purple-600">{(bestResult.metrics.speed / 1000).toFixed(2)}s</div>
                <div className="text-sm text-gray-500">응답속도</div>
              </div>
            </div>

            <details className="p-3 rounded" open={!!bestResult}>
              <summary className="cursor-pointer font-bold text-base mb-2">최적 설정값 보기</summary>
              <div className="space-y-3">
                {/* 4-Score System 최적화 결과 */}
                <div>
                  <div className="grid grid-cols-1 gap-1 text-base">
                    {["RAG_KEYWORD_PRECISION_THRESHOLD", "RAG_SEMANTIC_RELEVANCE_THRESHOLD", "RAG_CONTENT_QUALITY_THRESHOLD", "RAG_RECENCY_THRESHOLD"].map(
                      (key) =>
                        bestResult.config[key] !== undefined && (
                          <div key={key} className="flex justify-between text-sm">
                            <span className="text-gray-600 dark:text-gray-400">{OPTIMIZATION_PARAM_LABELS[key as keyof typeof OPTIMIZATION_PARAM_LABELS]}:</span>
                            <span className="font-mono text-blue-600 dark:text-orange-400">{Number(bestResult.config[key]).toFixed(2)}</span>
                          </div>
                        )
                    )}
                  </div>
                  <div className="flex justify-center pt-2">
                    <button onClick={applyBestSettings} className="w-full px-6 py-3 bg-green-600 text-white rounded hover:bg-green-700 shadow-lg hover:shadow-xl transition-all duration-200 transform hover:-translate-y-0.5 text-base font-medium">
                      설정 적용
                    </button>
                  </div>
                </div>
              </div>
            </details>
          </div>
        )}
      </div>
    </div>
  );
}

// 새로운 인터페이스 추가
interface TreeNode {
  id: string;
  name: string;
  type: "folder" | "file";
  children?: TreeNode[];
  size?: number;
  modified?: string;
  status?: "uploading" | "processing" | "completed" | "error";
  progress?: number;
  error?: string;
  userId?: string;
  isShared?: boolean;
  path?: string;
}

interface RagFileTreeViewProps {
  onSelectDoc: (doc: any) => void;
  selectedDoc: string | null;
  onFileUpload?: (file: File, targetUserId?: string) => Promise<void>;
  refreshTrigger?: number;
  onFolderSelect?: (folder: any) => void;
  onGraphRAGView?: (doc: any) => void;
  settings?: Record<string, any>;
  onFullSystemInit?: () => void;
  onDocumentInit?: (docName: string, userId: string) => void;
  onFolderInit?: (userId: string) => void;
}

// TreeNode를 IRemoteFileData로 변환하는 함수
function convertTreeDataToRemoteFiles(treeNodes: TreeNode[]): IRemoteFileData[] {
  const files: IRemoteFileData[] = [];

  treeNodes.forEach((node) => {
    if (node.type === "folder" && node.children) {
      // 폴더 내의 파일들을 처리
      node.children.forEach((child) => {
        if (child.type === "file") {
          files.push({
            fileName: child.name,
            fileSize: child.size || 0,
            fileType: child.name.split(".").pop() || "",
            uid: `${node.userId}/${child.name}`,
            description: `User: ${node.name}`,
            path: `/${node.userId}/${child.name}`,
            userId: node.userId,
            readOnly: false,
            disabled: false,
            previewData: {
              src: undefined
            }
          });
        }
      });
    }
  });

  return files;
}

// 파일 유형별 이모지 아이콘 함수
function getFileTypeIcon(fileName: string): string {
  const ext = fileName.split(".").pop()?.toLowerCase() || "";
  const icons: Record<string, string> = {
    pdf: "📕",
    doc: "📄",
    docx: "📄",
    xls: "📊",
    xlsx: "📊",
    ppt: "📈",
    pptx: "📈",
    txt: "📝",
    md: "📋",
    hwp: "📃",
    hwpx: "📃",
    jpg: "🖼️",
    jpeg: "🖼️",
    png: "🖼️",
    gif: "🖼️",
    bmp: "🖼️",
    svg: "🖼️",
    mp4: "🎬",
    avi: "🎬",
    mov: "🎬",
    wmv: "🎬",
    mp3: "🎵",
    wav: "🎵",
    flac: "🎵",
    zip: "📦",
    rar: "📦",
    "7z": "📦",
    tar: "📦",
    gz: "📦",
    json: "⚙️",
    xml: "⚙️",
    csv: "⚙️"
  };
  return icons[ext] || "📄";
}

function RagFileTreeView({ onSelectDoc, selectedDoc, onFileUpload, refreshTrigger, onFolderSelect, onGraphRAGView, settings, onFullSystemInit, onDocumentInit, onFolderInit }: RagFileTreeViewProps) {
  const [treeData, setTreeData] = useState<TreeNode[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [isAdmin, setIsAdmin] = useState(false);

  const [selectedFolder, setSelectedFolder] = useState<TreeNode | null>(null);
  const [expandedFolders, setExpandedFolders] = useState<Set<string>>(new Set());
  const [uploading, setUploading] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [uploadingFileName, setUploadingFileName] = useState<string>("");
  const [treeHeight, setTreeHeight] = useState(400);
  const [deleteModal, setDeleteModal] = useState<{ fileName: string; userId: string } | null>(null);
  const [folderDeleteModal, setFolderDeleteModal] = useState<{ folderName: string; userId: string; isShared: boolean } | null>(null);
  const [folderDeleteSuccessModal, setFolderDeleteSuccessModal] = useState<{ show: boolean; message: string }>({ show: false, message: "" });
  const [modal, setModal] = useState<{ type: "success" | "error"; title: string; message: string } | null>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const treeContainerRef = useRef<HTMLDivElement>(null);
  const [username, setUsername] = useState<string>("");
  const [embeddingStatuses, setEmbeddingStatuses] = useState<{ [filename: string]: { step: string; step_label: string; isProcessing: boolean } }>({}); // ← 로그인 아이디 보관(폴더 미선택시  파일 업로드시 사용, 로그인한 정보로 업로드)

  // RAG 서버에서 임베딩 상태 가져오기 (모든 사용자)
  const fetchEmbeddingStatuses = async () => {
    try {
      const response = await fetch(`${getRagServerUrl()}/rag/admin/all-statuses`);
      if (response.ok) {
        const data = await response.json();
        const statusMap: { [filename: string]: { step: string; step_label: string; isProcessing: boolean } } = {};

        if (data.documents && typeof data.documents === "object") {
          // 새로운 API 응답 구조에 맞게 처리
          Object.entries(data.documents).forEach(([filename, status]: [string, any]) => {
            statusMap[filename] = {
              step: status.step,
              step_label: status.step_label,
              isProcessing: status.isProcessing
            };
          });
        }

        setEmbeddingStatuses(statusMap);
      }
    } catch (error) {
      console.error("임베딩 상태 조회 실패:", error);
    }
  };

  // 관리자 권한 확인
  useEffect(() => {
    const username = localStorage.getItem("username") || "";
    const userRole = localStorage.getItem("userRole") || "";
    const userToken = localStorage.getItem("userToken") || "";
    const adminStatus = ["admin", "manager"].includes(userRole.toLowerCase());

    // 로그인 아이디 보관(폴더 미선택시  파일 업로드시 사용, 로그인한 정보로 업로드)
    const u = (typeof window !== "undefined" && localStorage.getItem("username")) || "";
    const role = (typeof window !== "undefined" && localStorage.getItem("userRole")) || "";
    setUsername(u.trim());
    setIsAdmin(["admin", "manager"].includes(role.toLowerCase()));

    // console.log('[RagFileTreeView 관리자 권한 확인]', {
    //   username,
    //   userRole,
    //   userToken: userToken ? userToken.substring(0, 10) + '...' : 'none',
    //   adminStatus
    // });
    setIsAdmin(adminStatus);
  }, []);

  // 임베딩 상태를 주기적으로 업데이트
  useEffect(() => {
    if (!isAdmin) return;

    fetchEmbeddingStatuses(); // 초기 로드
    const interval = setInterval(fetchEmbeddingStatuses, 3000); // 3초마다 업데이트

    return () => clearInterval(interval);
  }, [isAdmin]);

  // 커스텀 이벤트 리스너 등록
  useEffect(() => {
    const handleFileDeleteRequestEvent = (e: CustomEvent) => {
      const { fileName, userId } = e.detail;
      setDeleteModal({ fileName, userId });
    };

    const handleFolderDeleteRequestEvent = (e: CustomEvent) => {
      const { folderName, userId, isShared } = e.detail;
      setFolderDeleteModal({ folderName, userId, isShared });
    };

    const handleGraphRAGAnalyzeRequestEvent = (e: CustomEvent) => {
      const { document } = e.detail;
      if (onGraphRAGView) {
        onGraphRAGView(document);
      }
    };

    window.addEventListener("fileDeleteRequest", handleFileDeleteRequestEvent as EventListener);
    window.addEventListener("folderDeleteRequest", handleFolderDeleteRequestEvent as EventListener);
    window.addEventListener("graphragAnalyzeRequest", handleGraphRAGAnalyzeRequestEvent as EventListener);

    return () => {
      window.removeEventListener("fileDeleteRequest", handleFileDeleteRequestEvent as EventListener);
      window.removeEventListener("folderDeleteRequest", handleFolderDeleteRequestEvent as EventListener);
      window.removeEventListener("graphragAnalyzeRequest", handleGraphRAGAnalyzeRequestEvent as EventListener);
    };
  }, [onGraphRAGView]);

  const loadTreeData = async () => {
    // console.log('[loadTreeData] 호출됨 - isAdmin:', isAdmin);
    if (!isAdmin) {
      // console.log('[loadTreeData] 관리자가 아니므로 종료');
      return;
    }

    try {
      setLoading(true);
      setError(null);

      // console.log('[loadTreeData] RAG 파일 목록 가져오기 시작');
      // 모든 사용자의 RAG 파일 목록 가져오기
      const response = await externalApiClient.getRagAllUsers();
      // console.log('[loadTreeData] API 응답:', response);
      // console.log('[loadTreeData] API 응답 데이터 구조:', {
      //   success: response.success,
      //   hasData: !!response.data,
      //   users: response.data?.users,
      //   userCount: response.data?.userCount,
      //   ragDocsPath: response.data?.ragDocsPath
      // });
      if (response.success && response.data) {
        const treeNodes: TreeNode[] = [];

        // 공용 저장소 폴더 (항상 표시)
        const sharedUser = response.data.users?.find((u: any) => u.userId === "shared");
        const sharedFiles = sharedUser?.files || [];

        // shared 폴더가 없어도 빈 폴더로 표시
        treeNodes.push({
          id: "shared",
          name: "공용 저장소",
          type: "folder",
          userId: "shared",
          isShared: true,
          path: "/shared",
          children: sharedFiles.map((file: any) => {
            const embeddingStatus = embeddingStatuses[file.name];
            return {
              id: `shared_${file.name}`,
              name: file.name,
              type: "file" as const,
              size: file.size,
              modified: file.modified,
              status: "completed" as const,
              userId: "shared",
              path: `/shared/${file.name}`,
              embeddingStatus: embeddingStatus
            };
          })
        });

        // 고객센터 폴더 (항상 표시)
        const supportUser = response.data.users?.find((u: any) => u.userId === "support");
        const supportFiles = supportUser?.files || [];

        // support 폴더가 없어도 빈 폴더로 표시
        treeNodes.push({
          id: "support",
          name: "고객센터",
          type: "folder",
          userId: "support",
          isShared: true,
          path: "/support",
          children: supportFiles.map((file: any) => {
            const embeddingStatus = embeddingStatuses[file.name];
            return {
              id: `support_${file.name}`,
              name: file.name,
              type: "file" as const,
              size: file.size,
              modified: file.modified,
              status: "completed" as const,
              userId: "support",
              path: `/support/${file.name}`,
              embeddingStatus: embeddingStatus
            };
          })
        });

        // 사용자 폴더들
        const allUsers = response.data.users || [];
        // console.log('[loadTreeData] 전체 사용자 목록:', allUsers);

        const users = allUsers.filter((u: any) => u.userId !== "shared" && u.userId !== "support");
        // console.log('[loadTreeData] 필터링된 사용자 목록 (shared 제외):', users);

        // demo 사용자가 있는지 특별히 확인
        const demoUser = allUsers.find((u: any) => u.userId === "demo");
        // console.log('[loadTreeData] demo 사용자 존재 여부:', !!demoUser);
        if (demoUser) {
          // console.log('[loadTreeData] demo 사용자 데이터:', demoUser);
        }

        users.forEach((user: any) => {
          const userFiles = user.files || [];
          // console.log(`[loadTreeData] 사용자 ${user.userId}의 파일 수:`, userFiles.length);

          // 파일이 없어도 사용자 폴더는 표시 (관리자가 업로드할 수 있도록)
          treeNodes.push({
            id: user.userId,
            name: `User: ${user.userId}`,
            type: "folder",
            userId: user.userId,
            path: `/users/${user.userId}`,
            children: userFiles.map((file: any) => {
              const embeddingStatus = embeddingStatuses[file.name];
              return {
                id: `${user.userId}_${file.name}`,
                name: file.name,
                type: "file" as const,
                size: file.size,
                modified: file.modified,
                status: "completed" as const,
                userId: user.userId,
                path: `/users/${user.userId}/${file.name}`,
                embeddingStatus: embeddingStatus
              };
            })
          });
        });

        setTreeData(treeNodes);

        // 문서가 있는 폴더를 자동으로 펼침
        const foldersWithDocs = new Set<string>();
        treeNodes.forEach((node) => {
          if (node.type === "folder" && node.children && node.children.length > 0) {
            // 해당 폴더에 파일이 있는지 확인
            const hasFiles = node.children.some((child) => child.type === "file");
            if (hasFiles) {
              foldersWithDocs.add(node.name || node.id);
            }
          }
        });
        setExpandedFolders(foldersWithDocs);

        // 초기 선택 없이 사용자가 직접 폴더를 선택하도록 함
        // console.log('[RagFileTreeView] 트리 데이터 로드 완료, 폴더 수:', treeNodes.length);
      } else {
        const errorMsg = response.error ? (typeof response.error === "string" ? response.error : (response.error as any)?.message || "Failed to load tree data") : "Failed to load tree data";
        throw new Error(errorMsg);
      }
    } catch (error) {
      console.error("Error loading tree data:", error);
      setError(error instanceof Error ? error.message : "Failed to load tree data");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (isAdmin) {
      loadTreeData();
    }
  }, [isAdmin]);

  // refreshTrigger가 변경되면 트리 데이터 새로고침
  useEffect(() => {
    if (isAdmin && refreshTrigger && refreshTrigger > 0) {
      loadTreeData();
    }
  }, [isAdmin, refreshTrigger]);

  // 컨테이너 높이 측정
  useEffect(() => {
    const updateTreeHeight = () => {
      if (treeContainerRef.current) {
        const containerHeight = treeContainerRef.current.clientHeight;
        setTreeHeight(containerHeight);
      }
    };

    updateTreeHeight();

    const resizeObserver = new ResizeObserver(updateTreeHeight);
    if (treeContainerRef.current) {
      resizeObserver.observe(treeContainerRef.current);
    }

    return () => {
      resizeObserver.disconnect();
    };
  }, [treeData]);

  // 파일 선택 처리
  const handleFileSelect = (fileName: string) => {
    onSelectDoc(fileName);
  };

  // RAG chat 문서 삭제 (chat_documents DB 및 파일 제거)
  const deleteChatDocument = async (filename: string, username: string) => {
    if (username === "shared") {
      // shared 사용자의 경우 두 테이블 모두에서 삭제 시도
      let success = false;
      let lastError = null;

      // 1. support_documents에서 삭제 시도
      try {
        const supportRes = await fetch("/api/support/documents", {
          method: "DELETE",
          headers: {
            "Content-Type": "application/json"
          },
          body: JSON.stringify({ filename, id: username })
        });

        if (supportRes.ok) {
          const result = await supportRes.json();
          if (result.success) {
            success = true;
          }
        } else if (supportRes.status !== 404) {
          // 404가 아닌 다른 에러만 기록
          lastError = new Error(`Support documents delete failed: ${supportRes.status}`);
        }
      } catch (error) {
        console.warn("support_documents에서 삭제 실패:", error);
        lastError = error;
      }

      // 2. chat_documents에서 삭제 시도
      try {
        const chatRes = await fetch("/api/chat/documents", {
          method: "DELETE",
          headers: {
            "Content-Type": "application/json"
          },
          body: JSON.stringify({ filename, id: username })
        });

        if (chatRes.ok) {
          const result = await chatRes.json();
          if (result.success) {
            success = true;
          }
        } else if (chatRes.status !== 404) {
          // 404가 아닌 다른 에러만 기록
          lastError = new Error(`Chat documents delete failed: ${chatRes.status}`);
        }
      } catch (error) {
        console.warn("chat_documents에서 삭제 실패:", error);
        lastError = error;
      }

      if (!success) {
        const errorMessage = lastError instanceof Error ? lastError.message : "Document not found in any table.";
        throw new Error(errorMessage);
      }

      return { success: true };
    } else {
      // 일반 사용자의 경우 기존 로직
      const res = await fetch("/api/chat/documents", {
        method: "DELETE",
        headers: {
          "Content-Type": "application/json"
        },
        body: JSON.stringify({ filename, id: username })
      });

      const result = await res.json();
      if (!res.ok || !result.success) {
        throw new Error(result.error || "문서 삭제 실패");
      }
      return result;
    }
  };
  // 파일 삭제 처리
  const handleFileDelete = async (fileName: string, userId: string) => {
    try {
      setDeleteModal(null); // 모달 닫기

      console.log("파일 삭제 시작:", { fileName, userId });

      // 1. 모든 관련 테이블에서 문서 정보 삭제
      let deleteResults = {
        chatDocuments: false,
        ragFiles: false,
        embeddings: false
      };

      // 1-1. chat_documents에서 삭제
      try {
        await deleteChatDocument(fileName, userId);
        deleteResults.chatDocuments = true;
        console.log("chat_documents에서 문서 정보 삭제 완료:", fileName);
      } catch (dbError) {
        if (dbError instanceof Error && dbError.message.includes("Document not found")) {
          console.log("chat_documents에서 이미 삭제된 문서:", fileName);
          deleteResults.chatDocuments = true; // 이미 삭제된 것으로 간주
        } else {
          console.warn("chat_documents 삭제 중 오류:", dbError);
        }
      }

      // 2. RAG 파일 및 임베딩 데이터 삭제 (외부 API)
      try {
        const response = await externalApiClient.deleteRagFile(fileName, userId);
        if (response.success) {
          deleteResults.ragFiles = true;
          deleteResults.embeddings = true;
          console.log("RAG 파일 및 임베딩 데이터 삭제 완료:", fileName);
        } else {
          console.error("RAG 파일 삭제 실패:", response.error || "Unknown error");
        }
      } catch (ragError) {
        console.error("RAG 파일 삭제 중 오류:", ragError);
      }

      // 3. documents_embeddings에서 직접 삭제 시도 (백업 방법)
      if (!deleteResults.embeddings) {
        try {
          const embeddingResponse = await externalApiClient.post("/rag/delete", {
            file_paths: [fileName],
            user_id: userId
          });

          if (embeddingResponse.success) {
            deleteResults.embeddings = true;
            console.log("documents_embeddings에서 직접 삭제 완료:", fileName);
          }
        } catch (embeddingError) {
          console.warn("documents_embeddings 직접 삭제 중 오류:", embeddingError);
        }
      }

      // 4. 삭제 결과 요약 및 사용자 피드백
      const successCount = Object.values(deleteResults).filter(Boolean).length;
      const totalCount = Object.keys(deleteResults).length;

      if (successCount >= 2) {
        // 최소 2개 이상 성공해야 정상으로 간주
        console.log("파일 삭제 완료:", { fileName, deleteResults });
        await loadTreeData(); // 데이터 새로고침

        // 파일 삭제 완료 이벤트 발생 - 미리보기 모달 자동 닫기용
        const fileDeletedEvent = new CustomEvent("fileDeleted", {
          detail: { fileName: fileName }
        });
        window.dispatchEvent(fileDeletedEvent);
      } else {
        console.error("파일 삭제 실패 - 대부분의 작업이 실패:", { fileName, deleteResults });
        setModal({ type: "error", title: "파일 삭제 실패", message: `파일 삭제에 실패했습니다. 성공: ${successCount}/${totalCount}` });
        return;
      }
    } catch (error) {
      console.error("Error deleting file:", error);
      setModal({ type: "error", title: "오류", message: "파일 삭제 중 오류가 발생했습니다." });
    }
  };

  // 삭제 확인 모달 처리
  const handleConfirmDelete = () => {
    if (deleteModal) {
      handleFileDelete(deleteModal.fileName, deleteModal.userId);
    }
  };

  const handleCancelDelete = () => {
    setDeleteModal(null);
  };

  const handleFolderDelete = async (folderName: string, userId: string, isShared: boolean) => {
    try {
      setUploading(true);

      console.log("폴더 삭제 시작:", { folderName, userId, isShared });

      let deleteResults = {
        embeddings: false,
        ragFolders: false,
        chatDocuments: false
      };

      // 1. 폴더 내 모든 문서의 임베딩 데이터 삭제
      try {
        const embeddingDeleteResponse = await externalApiClient.post("/rag/delete", {
          file_paths: [], // 빈 배열로 전달하면 사용자별 전체 삭제
          user_id: userId
        });

        if (embeddingDeleteResponse.success) {
          deleteResults.embeddings = true;
          console.log("폴더 내 document_embeddings에서 청크 데이터 삭제 완료:", userId);
        } else {
          console.error("폴더 내 document_embeddings 삭제 실패:", embeddingDeleteResponse.error || "Unknown error");
        }
      } catch (embeddingError) {
        console.warn("폴더 내 document_embeddings 삭제 중 오류:", embeddingError);
      }

      // 2. RAG 폴더 삭제 (API 서버)
      try {
        const response = await fetch("/api/support/rag-folders", {
          method: "DELETE",
          headers: {
            "Content-Type": "application/json"
          },
          body: JSON.stringify({
            userId: userId,
            isShared: isShared
          })
        });

        if (response.ok) {
          const result = await response.json();
          deleteResults.ragFolders = true;
          console.log("RAG 폴더 삭제 완료:", result);
        } else {
          const errorData = await response.json().catch(() => ({ message: "알 수 없는 오류" }));
          throw new Error(errorData.message || "폴더 삭제 실패");
        }
      } catch (ragError) {
        console.error("RAG 폴더 삭제 중 오류:", ragError);
      }

      // 3. chat_documents에서 해당 사용자의 모든 문서 삭제
      try {
        const chatDeleteResponse = await fetch("/api/chat/documents", {
          method: "DELETE",
          headers: {
            "Content-Type": "application/json"
          },
          body: JSON.stringify({
            filename: "*", // 모든 파일 삭제를 위한 와일드카드
            id: userId
          })
        });

        if (chatDeleteResponse.ok) {
          const result = await chatDeleteResponse.json();
          if (result.success) {
            deleteResults.chatDocuments = true;
            console.log("chat_documents에서 사용자 문서 삭제 완료:", userId);
          }
        } else if (chatDeleteResponse.status === 404) {
          console.log("chat_documents에서 사용자 문서를 찾을 수 없음:", userId);
          deleteResults.chatDocuments = true; // 404는 정상적인 상황
        } else {
          console.warn("chat_documents 삭제 중 오류:", chatDeleteResponse.status);
        }
      } catch (chatError) {
        console.warn("chat_documents 삭제 중 오류:", chatError);
      }

      // 4. 삭제 결과 요약 및 사용자 피드백
      const successCount = Object.values(deleteResults).filter(Boolean).length;
      const totalCount = Object.keys(deleteResults).length;

      if (successCount >= 2) {
        // 최소 2개 이상 성공해야 정상으로 간주
        console.log("폴더 삭제 완료:", { folderName, userId, deleteResults });

        // 성공 모달 표시
        setFolderDeleteSuccessModal({
          show: true,
          message: `${folderName} 폴더가 성공적으로 삭제되었습니다.`
        });

        // 트리 데이터 새로고침
        await loadTreeData();
      } else {
        console.error("폴더 삭제 실패 - 대부분의 작업이 실패:", { folderName, userId, deleteResults });
        setModal({ type: "error", title: "폴더 삭제 실패", message: "폴더 삭제에 실패했습니다." });
        return;
      }
    } catch (error) {
      console.error("폴더 삭제 실패:", error);
      setModal({ type: "error", title: "폴더 삭제 실패", message: `폴더 삭제 실패: ${error instanceof Error ? error.message : "알 수 없는 오류"}` });
    } finally {
      setUploading(false);
    }
  };

  const handleConfirmFolderDelete = () => {
    if (folderDeleteModal) {
      handleFolderDelete(folderDeleteModal.folderName, folderDeleteModal.userId, folderDeleteModal.isShared);
      setFolderDeleteModal(null);
    }
  };

  const handleCancelFolderDelete = () => {
    setFolderDeleteModal(null);
  };

  const handleCloseFolderDeleteSuccess = () => {
    setFolderDeleteSuccessModal({ show: false, message: "" });
  };

  // 파일 업로드 처리 (백그라운드 작업으로 변경)
  const handleFileUpload = async (files: FileList | null) => {
    if (!files || files.length === 0) return;
    // const targetUserId = selectedFolder?.userId || 'admin';
    // 폴더 미선택시 admin 이 아닌 접속한 사용자 정보로 셋팅
    const targetUserId = selectedFolder?.userId ? selectedFolder.userId : username || "admin";
    const totalFiles = files.length;

    console.log("[RagFileTreeView] 파일 업로드 시작:", {
      selectedFolder: selectedFolder,
      targetUserId: targetUserId,
      filesCount: totalFiles
    });

    setUploading(true);
    setUploadProgress(0);

    for (let i = 0; i < files.length; i++) {
      const file = files[i];
      setUploadingFileName(file.name);

      try {
        // 파일별 진행률 업데이트 (시작)
        const fileStartProgress = (i / totalFiles) * 100;
        setUploadProgress(fileStartProgress);

        // 백그라운드 업로드 함수 사용
        if (onFileUpload) {
          await onFileUpload(file, targetUserId);
        } else {
          // 폴백: 기존 방식으로 업로드
          const response = await externalApiClient.uploadFile(file, targetUserId);
          if (!response.success) {
            console.error("파일 업로드 실패:", response.error || "Unknown error");
          }
        }

        // 파일별 진행률 업데이트 (완료)
        const fileEndProgress = ((i + 1) / totalFiles) * 100;
        setUploadProgress(fileEndProgress);
      } catch (error) {
        console.error("Error uploading file:", error);
        console.error("파일 업로드 중 오류가 발생했습니다.");
      }
    }

    // 업로드 완료
    setUploading(false);
    setUploadProgress(0);
    setUploadingFileName("");

    // 백그라운드 업로드를 사용하지 않는 경우에만 직접 새로고침
    if (!onFileUpload) {
      loadTreeData();
    }
  };

  // 폴더 선택 처리
  const handleFolderSelect = (node: any) => {
    // node가 undefined이거나 null인 경우 처리
    if (!node) {
      console.warn("[RagFileTreeView] handleFolderSelect: node is undefined or null");
      return;
    }

    // node가 직접 데이터 객체인 경우 (data 속성이 없는 경우)
    const item = node.data || node;

    if (item && item.type === "folder") {
      console.log("[RagFileTreeView] 폴더 선택됨:", {
        folderName: item.name,
        folderId: item.id,
        userId: item.userId,
        isShared: item.isShared,
        path: item.path,
        fullItem: item
      });
      setSelectedFolder(item);

      // 상위 컴포넌트에 폴더 선택 알림
      if (onFolderSelect) {
        console.log("[RagFileTreeView] 상위 컴포넌트에 폴더 선택 알림:", item);
        onFolderSelect(item);
      }
    }
  };

  // 폴더 확장/축소 토글 (더블클릭 전용)
  const handleFolderToggle = (node: any) => {
    const item = node.data || node;

    if (item && item.type === "folder") {
      const folderId = item.name || item.id;
      if (folderId) {
        setExpandedFolders((prev) => {
          const newExpanded = new Set(prev);
          if (newExpanded.has(folderId)) {
            newExpanded.delete(folderId);
          } else {
            newExpanded.add(folderId);
          }
          return newExpanded;
        });
      }
    }
  };

  // 파일 업로드 버튼 클릭
  const handleUploadClick = () => {
    // fileInputRef.current?.click();
    if (fileInputRef.current) {
      fileInputRef.current.value = ""; // 파일 선택 전 value 초기화
      fileInputRef.current.click();
    }
  };

  if (!isAdmin) {
    return (
      <div className="h-full flex items-center justify-center">
        <div className="text-center">
          <div className="text-gray-500 mb-2">🔒</div>
          <div className="text-gray-500">관리자 권한이 필요합니다</div>
        </div>
      </div>
    );
  }

  return (
    <div className="flex flex-col h-full">
      {/* 헤더: 파일 탐색기 제목 + 전체 초기화 버튼 */}
      <div className="flex items-center justify-between p-2 border-b border-gray-200 dark:border-gray-700">
        <span className="text-base font-bold text-gray-800 dark:text-gray-200">📁 문서 저장소</span>
        <button onClick={() => onFullSystemInit?.()} className="px-4 py-2 text-sm font-medium bg-red-600 hover:bg-red-700 text-white rounded-md transition-colors duration-200 flex items-center gap-1.5" title="전체 시스템 초기화">
          🗑️ 전체 초기화
        </button>
      </div>

      {/* 트리 뷰 영역 */}
      <div className="flex-1 overflow-y-auto min-h-0 relative">
        {loading && (
          <div className="px-4 py-6 text-center">
            <div className="text-gray-500">로딩 중...</div>
          </div>
        )}

        {error && (
          <div className="px-4 py-4">
            <div className="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
              <p>오류: {error}</p>
              <button onClick={loadTreeData} className="mt-2 px-4 py-2 bg-red-500 text-white rounded hover:bg-red-600">
                다시 시도
              </button>
            </div>
          </div>
        )}

        {!loading && !error && treeData.length === 0 && <div className="px-4 py-6 text-center text-gray-500">파일이 없습니다. 파일을 업로드하세요.</div>}

        {!loading && !error && treeData.length > 0 && (
          <div ref={treeContainerRef} data-no-dnd="true" className="px-4 pb-4 relative">
            <div className="tree-container">
              {/* 완전한 파일 트리 구조 - 삭제 아이콘 포함 */}
              {treeData.map((node) => (
                <div key={node.id} className="tree-item group">
                  <div
                    className={`tree-row flex items-center gap-2 px-2 py-1 cursor-pointer hover:bg-gray-100 dark:hover:bg-gray-700 ${selectedDoc === node.name ? "bg-blue-100 dark:bg-blue-900" : ""}`}
                    onClick={(e) => {
                      // 버튼, SVG, 폴딩 화살표, 폴더 아이콘이 클릭된 경우 이벤트 무시
                      const target = e.target as HTMLElement;
                      if (target instanceof HTMLButtonElement || target instanceof SVGElement || target.closest("button") || target.classList.contains("folder-toggle") || target.classList.contains("folder-icon") || target.tagName?.toLowerCase() === "path") {
                        console.log("🚫 클릭 이벤트 무시:", target.tagName, target.className);
                        return;
                      }

                      if (node.type === "file") {
                        handleFileSelect(node.name);
                      } else if (node.type === "folder") {
                        handleFolderSelect(node);
                      }
                    }}
                    onDoubleClick={() => {
                      if (node.type === "folder") {
                        handleFolderToggle(node);
                      }
                    }}
                  >
                    <span className="flex-shrink-0 flex items-center gap-1 text-base">
                      {node.type === "folder" && (
                        <span
                          className={`folder-toggle text-gray-500 transition-transform duration-200 cursor-pointer ${expandedFolders.has(node.name || node.id) ? "rotate-90" : ""}`}
                          onClick={(e) => {
                            e.stopPropagation();
                            handleFolderToggle(node);
                          }}
                        >
                          &gt;
                        </span>
                      )}

                      {/* 임베딩 상태 아이콘 (파일에만 표시, 처리 중/오류 상태만) */}
                      {node.type === "file" && (
                        <span className="flex-shrink-0 w-4 h-4 flex items-center justify-center mr-1">
                          {(() => {
                            const status = embeddingStatuses[node.name];

                            // 상태 정보가 없거나 완료된 경우 아이콘 표시하지 않음
                            if (!status || status.step === "done") {
                              return null;
                            }

                            if (status.isProcessing) {
                              // 처리 중
                              return <div className="w-3 h-3 rounded-full bg-orange-400 animate-pulse" title={`임베딩 진행 중: ${status.step_label}`} />;
                            } else if (status.step === "error" || status.step === "failed") {
                              // 오류
                              return <div className="w-3 h-3 rounded-full bg-red-500" title="임베딩 실패" />;
                            } else {
                              // 대기 중
                              return <div className="w-3 h-3 rounded-full bg-gray-400" title="임베딩 대기 중" />;
                            }
                          })()}
                        </span>
                      )}

                      <span
                        className={node.type === "folder" ? "folder-icon cursor-pointer" : ""}
                        onClick={(e) => {
                          if (node.type === "folder") {
                            e.stopPropagation();
                            handleFolderToggle(node);
                          }
                        }}
                      >
                        {node.type === "folder" ? (expandedFolders.has(node.name || node.id) ? "📂" : "📁") : getFileTypeIcon(node.name)}
                      </span>
                    </span>

                    <span className="flex-1 text-sm truncate pr-2" title={node.name}>
                      {node.name}
                    </span>
                    {node.size && <span className="flex-shrink-0 text-sm text-gray-500 mr-1">{formatFileSize(node.size)}</span>}

                    {/* 초기화 버튼 */}
                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        e.preventDefault();
                        if (node.type === "folder") {
                          onFolderInit?.(node.userId || "admin");
                        } else {
                          onDocumentInit?.(node.name, node.userId || "admin");
                        }
                      }}
                      className="flex-shrink-0 w-5 h-5 text-gray-400 hover:text-blue-500 dark:hover:text-blue-400 opacity-0 group-hover:opacity-100 transition-opacity"
                      title={node.type === "folder" ? "폴더 초기화" : "문서 초기화"}
                    >
                      <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                      </svg>
                    </button>

                    {/* 삭제 버튼 */}
                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        if (node.type === "folder") {
                          // 공용저장소는 삭제 불가
                          if (node.userId === "shared") {
                            setModal(null);
                            setModal({ type: "success", title: "경고", message: "공용저장소는 삭제할 수 없습니다" });
                            console.warn("공용저장소는 삭제할 수 없습니다.");
                            return;
                          }
                          // 폴더 삭제 요청
                          const event = new CustomEvent("folderDeleteRequest", {
                            detail: {
                              folderName: node.name,
                              userId: node.userId || "admin",
                              isShared: node.userId === "shared"
                            }
                          });
                          window.dispatchEvent(event);
                        } else {
                          // 파일 삭제 요청
                          const event = new CustomEvent("fileDeleteRequest", {
                            detail: { fileName: node.name, userId: node.userId || "admin" }
                          });
                          window.dispatchEvent(event);
                        }
                      }}
                      className="flex-shrink-0 w-5 h-5 text-gray-400 hover:text-red-500 dark:hover:text-red-400 opacity-0 group-hover:opacity-100 transition-opacity"
                      title={node.type === "folder" ? "폴더 삭제" : "파일 삭제"}
                    >
                      <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1-1H9a1 1 0 00-1 1v3M4 7h16" />
                      </svg>
                    </button>
                  </div>
                  {node.children && node.children.length > 0 && expandedFolders.has(node.name || node.id) && (
                    <div className="ml-4">
                      {node.children.map((child) => (
                        <div key={child.id} className="tree-item group">
                          <div
                            className={`tree-row flex items-center gap-2 px-2 py-1 cursor-pointer hover:bg-gray-100 dark:hover:bg-gray-700 ${selectedDoc === child.name ? "bg-blue-100 dark:bg-blue-900" : ""}`}
                            onClick={(e) => {
                              // 버튼, SVG, 폴딩 화살표, 폴더 아이콘이 클릭된 경우 이벤트 무시
                              const target = e.target as HTMLElement;
                              if (target instanceof HTMLButtonElement || target instanceof SVGElement || target.closest("button") || target.classList.contains("folder-toggle") || target.classList.contains("folder-icon")) {
                                return;
                              }

                              if (child.type === "file") {
                                handleFileSelect(child.name);
                              } else if (child.type === "folder") {
                                handleFolderSelect(child);
                              }
                            }}
                            onDoubleClick={() => {
                              if (child.type === "folder") {
                                handleFolderToggle(child);
                              }
                            }}
                          >
                            <span className="flex-shrink-0 flex items-center gap-1 text-base">
                              {child.type === "folder" && (
                                <span
                                  className={`folder-toggle text-gray-500 transition-transform duration-200 cursor-pointer ${expandedFolders.has(child.name || child.id) ? "rotate-90" : ""}`}
                                  onClick={(e) => {
                                    e.stopPropagation();
                                    handleFolderToggle(child);
                                  }}
                                >
                                  &gt;
                                </span>
                              )}

                              {/* 임베딩 상태 아이콘 (파일에만 표시, 처리 중/오류 상태만) */}
                              {child.type === "file" && (
                                <span className="flex-shrink-0 w-4 h-4 flex items-center justify-center mr-1">
                                  {(() => {
                                    const status = embeddingStatuses[child.name];

                                    // 상태 정보가 없거나 완료된 경우 아이콘 표시하지 않음
                                    if (!status || status.step === "done") {
                                      return null;
                                    }

                                    if (status.isProcessing) {
                                      // 처리 중
                                      return <div className="w-3 h-3 rounded-full bg-orange-400 animate-pulse" title={`임베딩 진행 중: ${status.step_label}`} />;
                                    } else if (status.step === "error" || status.step === "failed") {
                                      // 오류
                                      return <div className="w-3 h-3 rounded-full bg-red-500" title="임베딩 실패" />;
                                    } else {
                                      // 대기 중
                                      return <div className="w-3 h-3 rounded-full bg-gray-400" title="임베딩 대기 중" />;
                                    }
                                  })()}
                                </span>
                              )}

                              <span
                                className={child.type === "folder" ? "folder-icon cursor-pointer" : ""}
                                onClick={(e) => {
                                  if (child.type === "folder") {
                                    e.stopPropagation();
                                    handleFolderToggle(child);
                                  }
                                }}
                              >
                                {child.type === "folder" ? (expandedFolders.has(child.name || child.id) ? "📂" : "📁") : getFileTypeIcon(child.name)}
                              </span>
                            </span>

                            <span className="flex-1 text-sm truncate pr-2" title={child.name}>
                              {child.name}
                            </span>
                            {child.size && <span className="flex-shrink-0 text-sm text-gray-500 mr-1">{formatFileSize(child.size)}</span>}

                            {/* 초기화 버튼 */}
                            <button
                              onClick={(e) => {
                                e.stopPropagation();
                                e.preventDefault();
                                if (child.type === "folder") {
                                  onFolderInit?.(child.userId || "admin");
                                } else {
                                  onDocumentInit?.(child.name, child.userId || "admin");
                                }
                              }}
                              className="flex-shrink-0 w-5 h-5 text-gray-400 hover:text-blue-500 dark:hover:text-blue-400 opacity-0 group-hover:opacity-100 transition-opacity mr-1"
                              title={child.type === "folder" ? "폴더 초기화" : "문서 초기화"}
                            >
                              <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                              </svg>
                            </button>

                            {/* 삭제 버튼 */}
                            <button
                              onClick={(e) => {
                                e.stopPropagation();
                                if (child.type === "folder") {
                                  // 공용저장소는 삭제 불가
                                  if (child.userId === "shared") {
                                    console.warn("공용저장소는 삭제할 수 없습니다.");
                                    return;
                                  }
                                  // 폴더 삭제 요청
                                  const event = new CustomEvent("folderDeleteRequest", {
                                    detail: {
                                      folderName: child.name,
                                      userId: child.userId || "admin",
                                      isShared: child.userId === "shared"
                                    }
                                  });
                                  window.dispatchEvent(event);
                                } else {
                                  // 파일 삭제 요청
                                  const event = new CustomEvent("fileDeleteRequest", {
                                    detail: { fileName: child.name, userId: child.userId || "admin" }
                                  });
                                  window.dispatchEvent(event);
                                }
                              }}
                              className="flex-shrink-0 w-5 h-5 text-gray-400 hover:text-red-500 dark:hover:text-red-400 opacity-0 group-hover:opacity-100 transition-opacity"
                              title={child.type === "folder" ? "폴더 삭제" : "파일 삭제"}
                            >
                              <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1-1H9a1 1 0 00-1 1v3M4 7h16" />
                              </svg>
                            </button>
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
              ))}
            </div>
          </div>
        )}
      </div>

      {/* 삭제 확인 모달 */}
      {deleteModal && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black bg-opacity-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow-lg p-6 min-w-[320px] max-w-md">
            <div className="text-lg font-bold mb-4 text-gray-900 dark:text-white">파일 삭제 확인</div>
            <div className="mb-6 text-gray-700 dark:text-gray-300">
              <p className="mb-2">다음 파일을 삭제하시겠습니까?</p>
              <p className="font-semibold text-red-600 dark:text-red-400 break-all">{deleteModal.fileName}</p>
              <p className="text-base text-gray-500 mt-2">이 작업은 되돌릴 수 없습니다.</p>
            </div>
            <div className="flex gap-3 justify-end">
              <button onClick={handleCancelDelete} className="px-4 py-2 text-gray-700 dark:text-gray-300 bg-gray-200 dark:bg-gray-600 rounded hover:bg-gray-300 dark:hover:bg-gray-500 transition-colors">
                취소
              </button>
              <button onClick={handleConfirmDelete} className="px-4 py-2 text-white bg-red-600 rounded hover:bg-red-700 transition-colors">
                삭제
              </button>
            </div>
          </div>
        </div>
      )}

      {/* 폴더 삭제 확인 모달 */}
      {folderDeleteModal && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black bg-opacity-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow-lg p-6 min-w-[400px] max-w-md">
            <div className="text-lg font-bold mb-4 text-gray-900 dark:text-white">폴더 삭제 확인</div>
            <div className="mb-6 text-gray-700 dark:text-gray-300">
              <p className="mb-2">다음 폴더를 삭제하시겠습니까?</p>
              <p className="font-semibold text-orange-600 dark:text-orange-400 break-all">{folderDeleteModal.folderName}</p>
              <p className="text-base text-gray-500 mt-2">
                사용자 ID: <span className="font-mono">{folderDeleteModal.userId}</span>
              </p>
              <div className="mt-4 p-3 bg-yellow-50 dark:bg-yellow-900 border border-yellow-200 dark:border-yellow-700 rounded">
                <p className="text-base text-yellow-800 dark:text-yellow-200">
                  ⚠️ <strong>경고:</strong> 이 작업은 폴더와 폴더 내의 모든 파일을 삭제합니다.
                </p>
                <p className="text-base text-yellow-800 dark:text-yellow-200 mt-1">삭제된 데이터는 복구할 수 없습니다.</p>
              </div>
            </div>
            <div className="flex gap-3 justify-end">
              <button onClick={handleCancelFolderDelete} className="px-4 py-2 text-gray-700 dark:text-gray-300 bg-gray-200 dark:bg-gray-600 rounded hover:bg-gray-300 dark:hover:bg-gray-500 transition-colors">
                취소
              </button>
              <button onClick={handleConfirmFolderDelete} className="px-4 py-2 text-white bg-orange-600 rounded hover:bg-orange-700 transition-colors">
                폴더 삭제
              </button>
            </div>
          </div>
        </div>
      )}

      {/* 일반 안내 모달 */}
      {modal && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black bg-opacity-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow-lg p-6 min-w-[320px] max-w-sm">
            <div className={`text-lg font-bold mb-4 ${modal.type === "success" ? "text-green-600" : "text-red-500"}`}>{modal.title}</div>
            <div className="text-gray-700 dark:text-gray-300 mb-6">{modal.message}</div>
            <div className="flex justify-end">
              <button onClick={() => setModal(null)} className="px-4 py-2 bg-gray-200 dark:bg-gray-600 text-gray-800 dark:text-gray-200 rounded hover:bg-gray-300 dark:hover:bg-gray-500 transition-colors">
                닫기
              </button>
            </div>
          </div>
        </div>
      )}

      {/* 폴더 삭제 성공 모달 */}
      {folderDeleteSuccessModal.show && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black bg-opacity-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow-lg p-6 min-w-[400px] max-w-md">
            <div className="text-lg font-bold mb-4 text-gray-900 dark:text-white flex items-center">
              <span className="text-green-500 mr-2 text-2xl">✓</span>
              삭제 완료
            </div>
            <div className="mb-6 text-gray-700 dark:text-gray-300">
              <p className="text-center">{folderDeleteSuccessModal.message}</p>
            </div>
            <div className="flex justify-center">
              <button onClick={handleCloseFolderDeleteSuccess} className="px-6 py-2 text-white bg-green-600 rounded hover:bg-green-700 transition-colors">
                확인
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}

export default function AdminRagPage() {
  const { user, isLoading: isAuthLoading, isAuthenticated, isInitialized } = useAuth();
  // 상태: 설정, 컬렉션, 업로드 진행, 청크, 채팅 등(추후 구현)
  const [settings, setSettings] = useState<Record<string, any>>({});
  const [collection, setCollection] = useState<any>(null);
  const [progress, setProgress] = useState<any>(null);
  const [chunks, setChunks] = useState<any[]>([]);
  const [selectedDoc, setSelectedDoc] = useState<string | null>(null);
  const [selectedFolder, setSelectedFolder] = useState<any>(null);
  const [isInitializing, setIsInitializing] = useState(false);
  const [progressPercent, setProgressPercent] = useState(0);
  const [statusMessage, setStatusMessage] = useState("");
  const [error, setError] = useState<string | null>(null);
  const [result, setResult] = useState<any>(null);
  const [initStartTime, setInitStartTime] = useState<number | null>(null);
  const [polling, setPolling] = useState(false);
  const [logs, setLogs] = useState<string[]>([]);
  const [isLogPaused, setIsLogPaused] = useState(false);
  const [isLogOpen, setIsLogOpen] = useState(false);
  const [docStatuses, setDocStatuses] = useState<any[]>([]);
  const [allFilenames, setAllFilenames] = useState<string[]>([]);
  const [documentStats, setDocumentStats] = useState({
    total: 0,
    completed: 0,
    processing: 0
  });
  const logBoxRef = useRef<HTMLDivElement>(null);
  const pollIntervalRef = useRef<NodeJS.Timeout | null>(null);

  // 초기화 트리거 상태
  const [initTrigger, setInitTrigger] = useState<{
    type: "full" | "document" | "folder" | null;
    docName?: string;
    userId?: string;
  }>({ type: null });

  // 컬렉션 초기화 핸들러
  const handleInitCollection = useCallback(async (isFullSystem = false, docName: string | null = null, folder: any = null) => {
    setIsInitializing(true);
    setProgressPercent(10);
    const username = localStorage.getItem("username") || "all";

    // 디버깅: 전달받은 파라미터 로깅
    console.log("🔧 handleInitCollection 호출됨:", {
      isFullSystem,
      docName,
      folder,
      username
    });

    // 선택된 항목에 따른 초기화 메시지 및 로직 결정
    let initMessage = "";
    let initPayload = {};

    if (isFullSystem) {
      initMessage = "전체 시스템 초기화 요청 중...";
      initPayload = {};
    }
    // 1. 선택된 문서가 있는 경우 - 해당 문서만 초기화(삭제)
    else if (docName) {
      initMessage = `선택된 문서 '${docName}' 초기화 요청 중...`;
      initPayload = {
        user_id: folder?.userId || username,
        document_name: docName
      };
    }
    // 2. 선택된 폴더가 있는 경우 - 해당 폴더(사용자) 초기화
    else if (folder && folder.userId) {
      initMessage = `선택된 폴더 사용자 '${folder.userId}' 문서 초기화 요청 중...`;
      initPayload = { user_id: folder.userId };
      console.log("📁 [폴더 초기화] 요청 정보:", {
        folder: folder,
        userId: folder.userId,
        initPayload: initPayload
      });
    }
    // 3. 선택된 항목이 없는 경우 - 내 문서 초기화
    else {
      initMessage = `사용자 '${username}' 문서 초기화 요청 중...`;
      initPayload = { user_id: username };
    }

    setStatusMessage(initMessage);
    setError(null);
    setResult(null);
    setInitStartTime(Date.now());

    // 폴더 초기화 시 selectedDoc을 초기화하여 이전 선택된 파일이 남아있지 않도록 함
    if (!docName && (folder?.userId || !isFullSystem)) {
      console.log("📁 폴더 초기화 시작 - selectedDoc 초기화");
      setSelectedDoc(null);
      // 폴더 초기화 상태임을 표시 (첫 번째 폴링 전까지 이전 파일이 표시되지 않도록)
      setDocStatuses([]);
    }

    // 초기화 시작 시 처리 중 상태로 docStatuses 설정
    if (docName) {
      setDocStatuses([
        {
          filename: docName,
          step: "processing",
          stepLabel: "처리 중",
          progress: 0.1,
          timestamp: new Date().toISOString()
        }
      ]);
    } else {
      setDocStatuses([]);
    }
    console.log("🚀 초기화 상태 설정:", { initStartTime: Date.now(), initMessage, docName });

    // 디버깅: 최종 전송될 데이터 로깅
    console.log("📤 API 전송 데이터:", {
      isFullSystem,
      initPayload,
      endpoint: isFullSystem ? "/rag/init (전체)" : "/rag/init (선택적)",
      message: initMessage
    });

    try {
      let res;
      if (isFullSystem) {
        // 전체 시스템 초기화
        res = await externalApiClient.post("/rag/init");
      } else {
        // 선택된 항목에 따른 초기화
        res = await externalApiClient.post("/rag/init", initPayload);
      }
      const data = res.data || res;
      console.log("📥 [백엔드 응답]", data);

      // 초기화할 파일이 없는 경우 처리
      if (data?.message?.includes("초기화할 파일이 없습니다") || data?.message?.includes("파일이 없습니다") || data?.message?.includes("처리할 문서가 없습니다")) {
        setProgressPercent(0);
        setStatusMessage("");
        setError(data.message || "초기화할 파일이 없습니다.");
        setIsInitializing(false);
        setSelectedDoc(null); // 선택된 파일 초기화
        console.log("📭 초기화할 파일이 없음 - selectedDoc을 null로 설정");
        return;
      }

      if (data?.status === "success" || data?.message?.includes("시작")) {
        setStatusMessage("초기화 대기 중...");

        // API 성공 시 즉시 처리 중 상태 설정 (폴링 전에 시각화 표시)
        if (docName) {
          setDocStatuses([
            {
              filename: docName,
              step: "processing",
              stepLabel: "처리 중",
              progress: 0.2,
              timestamp: new Date().toISOString()
            }
          ]);
        }

        setPolling(true);
        // 백엔드가 작업을 시작할 시간을 주기 위해 3초 후에 폴링 시작
        setTimeout(() => {
          console.log("⏰ 지연된 폴링 시작 (3초 후)");
          pollInitStatus();
        }, 3000);
      } else {
        setProgressPercent(0);
        setStatusMessage("");
        setError(data?.message || "초기화 실패");
        setIsInitializing(false);
      }
    } catch (e) {
      setProgressPercent(0);
      setStatusMessage("");
      setError("초기화 중 네트워크 오류");
      setIsInitializing(false);
    }
  }, []);

  // 상태 폴링 함수
  const pollInitStatus = () => {
    let tries = 0;
    const maxTries = 600; // 20분으로 증가 (600 * 2초)
    let consecutiveErrors = 0;
    const maxConsecutiveErrors = 5;

    if (pollIntervalRef.current) clearInterval(pollIntervalRef.current);
    pollIntervalRef.current = setInterval(async () => {
      tries++;
      try {
        const res = await externalApiClient.get("/rag/init-status");
        const data = res.data || res;
        const qs = data.queue_status || {};
        const totalTime = data.total_time?.total_time?.formatted || null;

        // 에러 카운터 리셋
        consecutiveErrors = 0;

        // --- progress_detail.documents가 있으면 우선 사용 (이미지 파일 제외) ---
        let docStatusesToUse: any[] = [];
        let allFilenamesToUse: string[] = [];
        if (data.progress_detail && Array.isArray(data.progress_detail.documents)) {
          // 이미지 파일과 임시 파일 제외
          docStatusesToUse = (data.progress_detail.documents ?? []).filter((doc: any) => {
            const filename = doc.filename || "";
            return !/\.(jpg|jpeg|png|gif|bmp|svg|webp)$/i.test(filename) && !/^[a-f0-9]+_page_\d+_\d+\./i.test(filename);
          });
          allFilenamesToUse = docStatusesToUse.map((d: any) => d.filename);

          // 폴더 초기화 디버깅을 위한 상세 로그
          console.log("🔍 [폴링] 받은 원본 데이터:", {
            status: data.status,
            user_id: data.user_id,
            queue_status: qs,
            total_documents: data.progress_detail.documents.length,
            filtered_documents: docStatusesToUse.length,
            message: data.message,
            tries: tries,
            elapsed: `${((Date.now() - (initStartTime || Date.now())) / 1000).toFixed(1)}초`
          });
          console.log("🔍 [폴링] data.user_id 확인:", data.user_id, "typeof:", typeof data.user_id);
          console.log(
            "📄 [폴링] 필터링된 문서 목록:",
            docStatusesToUse.map((doc) => ({
              filename: doc.filename,
              status: doc.status || doc.step,
              user_id: doc.user_id
            }))
          );
        }

        // 폴더 초기화 시 현재 처리 중인 파일을 메인 영역에 표시
        // 현재 초기화 중인 사용자의 문서들만 처리 (다른 사용자의 완료된 문서 제외)
        if (docStatusesToUse.length > 0) {
          // 현재 초기화 중인 사용자 ID가 있으면 해당 사용자의 문서만 필터링
          const currentUserId = data.user_id;
          if (currentUserId && currentUserId !== "all") {
            const userDocuments = docStatusesToUse.filter((doc: any) => {
              // 문서에 user_id가 있으면 해당 사용자의 문서인지 확인
              // 문서에 user_id가 없으면 현재 진행 중인 문서로 간주
              return !doc.user_id || doc.user_id === currentUserId;
            });

            console.log(`🎯 [필터링] 사용자 '${currentUserId}' 문서만 필터링: ${userDocuments.length}/${docStatusesToUse.length}`);
            docStatusesToUse = userDocuments;
            allFilenamesToUse = allFilenamesToUse.filter((filename) => docStatusesToUse.some((doc: any) => doc.filename === filename));
          }
        }

        // 필터링된 결과를 문서 통계에 반영
        console.log("📡 [폴링] 필터링된 docStatuses 업데이트:", docStatusesToUse);
        setDocStatuses(docStatusesToUse);
        setAllFilenames(allFilenamesToUse);

        if (docStatusesToUse.length > 0) {
          // 현재 진행 중인 초기화 작업이 있는지 확인
          const hasActiveProcessing = qs.processing > 0;
          const hasRemainingDocs = qs.remaining > 0;
          const hasActiveStatus = data.status && data.status !== "completed" && data.status !== "success";
          const hasProcessingDocs = docStatusesToUse.some((doc: any) => {
            const status = doc.status || doc.step;
            return status === "processing" || status === "extract" || status === "embed" || status === "chunk";
          });

          const hasActiveInit = hasActiveProcessing || hasRemainingDocs || hasActiveStatus || hasProcessingDocs;

          console.log("🔍 [진행상태 확인]", {
            hasActiveProcessing,
            hasRemainingDocs,
            hasActiveStatus,
            hasProcessingDocs,
            hasActiveInit,
            qs: qs,
            dataStatus: data.status
          });

          // 진행 중인 초기화가 없고 모든 문서가 완료 상태라면 이전 완료된 문서들이므로 선택하지 않음
          const allCompleted = docStatusesToUse.every((doc: any) => {
            const status = doc.status || doc.step;
            return status === "completed" || status === "success" || status === "done";
          });

          if (!hasActiveInit && allCompleted) {
            console.log("⚠️ 진행 중인 초기화 없음 - 이전 완료된 문서들만 존재");
            console.log("🔍 [상세] hasActiveInit:", hasActiveInit, "allCompleted:", allCompleted);
            console.log("🔍 [상세] qs:", qs);
            console.log("🔍 [상세] data.status:", data.status);

            // 폴링 초기(10회 이내)에 이런 상황이면서 초기화 시작 후 30초가 지났으면 파일이 없다고 판단
            const elapsedSeconds = (Date.now() - (initStartTime || Date.now())) / 1000;
            if (tries <= 10 && elapsedSeconds > 30) {
              setPolling(false);
              if (pollIntervalRef.current) clearInterval(pollIntervalRef.current);
              setIsInitializing(false);
              setProgressPercent(0);
              setStatusMessage("");
              setError("선택한 폴더에 초기화할 파일이 없습니다.");
              setSelectedDoc(null);
              console.log("🛑 초기화 중단 - 처리할 파일 없음 (30초 경과)");
              return;
            }

            // 초기화 시작 후 30초 이내라면 대기 상태 유지
            if (elapsedSeconds <= 30) {
              setStatusMessage("처리 대기 중... (RAG 서버에서 작업 준비 중)");
              setSelectedDoc(null);
              console.log("⏳ 대기 중...", `${elapsedSeconds.toFixed(1)}초 경과`);
              return;
            }

            // selectedDoc을 null로 설정하여 이전 파일이 표시되지 않도록 함
            if (selectedDoc !== null) {
              console.log("🔄 selectedDoc을 null로 변경:", selectedDoc, "->", null);
              setSelectedDoc(null);
            }
            return; // selectedDoc 변경하지 않음
          }

          // 자동 선택 로직 제거 - 사용자가 직접 파일을 선택하도록 함
          console.log("📊 파일 상태 업데이트 완료 - 자동 선택 없음");

          // 디버깅을 위한 전체 상태 로깅
          console.log(
            "📊 전체 문서 상태:",
            docStatusesToUse.map((doc) => ({
              filename: doc.filename,
              status: doc.status || doc.step,
              progress: doc.progress
            }))
          );
        }

        // 완료 조건을 더 유연하게 확인
        const isCompleted =
          (data.status === "success" && qs.remaining === 0) ||
          (data.message && data.message.includes("모든 문서 처리가 완료")) ||
          (data.message && data.message.includes("초기화 완료")) ||
          (data.message && data.message.includes("문서 초기화 완료")) ||
          (qs.remaining === 0 && qs.total_documents > 0 && qs.completed === qs.total_documents) ||
          (qs.total_documents > 0 && qs.completed === qs.total_documents && qs.processing === 0);

        if (isCompleted) {
          setProgressPercent(100);
          // 사용자별 초기화인지 전체 초기화인지 구분하여 메시지 표시
          const isUserSpecific = data.user_id && data.user_id !== "all";
          const completionMessage = isUserSpecific ? `사용자 '${data.user_id}' 문서 초기화 완료!` : `전체 시스템 초기화 완료!`;
          setStatusMessage(`${completionMessage}${totalTime ? ` (총 소요시간: ${totalTime})` : ""}`);
          setResult(data.message || "초기화 성공");

          // 완료 시 모든 문서 상태를 강제로 완료로 설정
          setDocStatuses((prev) =>
            prev.map((doc) => ({
              ...doc,
              step: "done" as const,
              stepLabel: "완료",
              progress: 1
            }))
          );

          setIsInitializing(false);
          setPolling(false);
          // 초기화 완료 후 selectedDoc을 null로 유지하여 자동 선택 방지
          setSelectedDoc(null);
          console.log("🎯 초기화 완료 - selectedDoc을 null로 설정");
          if (pollIntervalRef.current) clearInterval(pollIntervalRef.current);
          // 초기화 완료 후 문서 통계 새로고침
          setTimeout(() => loadDocumentStats(), 1000);
        } else if (qs.total_documents) {
          setProgressPercent(qs.progress_percentage || 50);
          const elapsed = tries * 2; // 경과 시간 (초)
          const minutes = Math.floor(elapsed / 60);
          const seconds = elapsed % 60;
          setStatusMessage(`진행 중... (${qs.completed || 0}/${qs.total_documents} 문서 완료, ${qs.progress_percentage || 0}%)` + (totalTime ? `, 총 경과: ${totalTime}` : "") + `, 모니터링: ${minutes}분 ${seconds}초`);
        } else {
          const elapsed = tries * 2;
          const minutes = Math.floor(elapsed / 60);
          const seconds = elapsed % 60;
          setStatusMessage(`초기화 대기 중... (${minutes}분 ${seconds}초)`);
        }
      } catch (e) {
        consecutiveErrors++;
        if (consecutiveErrors >= maxConsecutiveErrors) {
          setError("상태 확인 중 연속 오류 발생");
          setPolling(false);
          setIsInitializing(false);
          if (pollIntervalRef.current) clearInterval(pollIntervalRef.current);
          return;
        }
      }

      // 최대 시도 횟수 초과
      if (tries >= maxTries) {
        setError("초기화 시간 초과");
        setPolling(false);
        setIsInitializing(false);
        if (pollIntervalRef.current) clearInterval(pollIntervalRef.current);
      }
    }, 2000);
  };

  // 초기화 트리거 효과
  useEffect(() => {
    if (initTrigger.type === null) return;

    if (initTrigger.type === "full") {
      handleInitCollection(true, null, null);
    } else if (initTrigger.type === "document" && initTrigger.docName && initTrigger.userId) {
      const folder = { userId: initTrigger.userId };
      setSelectedDoc(initTrigger.docName);
      // Don't set selectedFolder for file initialization to avoid showing folder selection UI
      handleInitCollection(false, initTrigger.docName, folder);
    } else if (initTrigger.type === "folder" && initTrigger.userId) {
      const folder = { userId: initTrigger.userId };
      // Don't set selectedFolder for folder initialization to avoid showing folder selection UI
      setSelectedDoc(null);
      handleInitCollection(false, null, folder);
    }

    // 트리거 초기화
    setInitTrigger({ type: null });
  }, [initTrigger, handleInitCollection]);
  const [currentChunkIndex, setCurrentChunkIndex] = useState(0);
  const [chatHistory, setChatHistory] = useState<any[]>([]);
  const [selectedChunk, setSelectedChunk] = useState<any>(null);
  const [provider, setProvider] = useState<string>("");
  const [model, setModel] = useState<string>("");
  const [temperature, setTemperature] = useState<number>(0.7);
  const [providerOptions, setProviderOptions] = useState<{ value: string; label: string; models: { value: string; label: string }[] }[]>([]);
  const [modelOptions, setModelOptions] = useState<{ value: string; label: string }[]>([]);
  const [currentSessionId, setCurrentSessionId] = useState<string | null>(null);
  const [currentMessages, setCurrentMessages] = useState<any[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const [ragSearching, setRagSearching] = useState(false);
  const [aiResponding, setAiResponding] = useState(false);
  const [rightPanelWidth, setRightPanelWidth] = useState(400);
  const [isResizing, setIsResizing] = useState(false);
  const [windowWidth, setWindowWidth] = useState(1400); // 기본값 설정
  const leftPanelWidth = 340; // 왼쪽 사이드바 너비
  const [modal, setModal] = useState<{ type: "success" | "error"; title: string; message: string } | null>(null);

  // PDF 변환 모달 상태

  // 전처리 모달 상태
  const [showExcelModal, setShowExcelModal] = useState(false);
  const [showPdfModal, setShowPdfModal] = useState(false);
  const [showSmartTuneModal, setShowSmartTuneModal] = useState(false);

  // GraphRAG 모달 상태
  const [showGraphRAGModal, setShowGraphRAGModal] = useState(false);
  const [currentGraphQuery, setCurrentGraphQuery] = useState<string>("");

  // GraphRAG 파일 미리보기 모달 상태
  const [showGraphRAGFileModal, setShowGraphRAGFileModal] = useState(false);
  const [selectedGraphRAGDoc, setSelectedGraphRAGDoc] = useState<any>(null);
  const [graphRAGChunks, setGraphRAGChunks] = useState<any[]>([]);

  // 백그라운드 작업 관련 상태
  const [refreshTrigger, setRefreshTrigger] = useState(0);

  // 백그라운드 작업 완료 시 콜백
  const handleJobComplete = async (job: BackgroundJob) => {
    console.log("!! 백그라운드 작업 완료:", job);

    if (job.type === "file_upload") {
      // 파일 트리 새로고침 트리거 (setRefreshTrigger를 통해 RagFileTreeView 리로드)
      setRefreshTrigger((prev) => prev + 1);
      // 작업 완료 모달 제거 - 백그라운드 모니터링에서 완료 상태 확인 가능
    }
  };

  // 전처리 핸들러
  const handleExcelProcess = async (options: any) => {
    console.log("Excel 전처리 시작:", options);
    try {
      const formData = new FormData();
      formData.append("file", options.file);
      formData.append("options", JSON.stringify(options));

      const response = await fetch("/api/preprocess/excel", {
        method: "POST",
        body: formData
      });

      if (response.ok) {
        const result = await response.json();
        console.log("Excel 전처리 완료:", result);

        // 여러 파일 업로드 처리
        if (result.data?.processedFiles && result.data.processedFiles.length > 0) {
          const uploadPromises = result.data.processedFiles.map(async (processedFile: any) => {
            const fileBuffer = Buffer.from(processedFile.content, "base64");
            const file = new File([fileBuffer], processedFile.name, { type: "text/markdown" });
            return await handleFileUploadSimple(file);
          });

          await Promise.all(uploadPromises);

          // 성공 메시지
          const fileCount = result.data.processedFiles.length;
          const splitMessage = result.data.splitEnabled ? `데이터가 ${fileCount}개 파일로 분할되어` : "";

          setModal({
            type: "success",
            title: "전처리 완료",
            message: `${splitMessage} ${fileCount}개 파일이 업로드되었습니다.`
          });
        } else if (result.data?.processedFile) {
          // 하위 호환성: 단일 파일 처리
          await handleFileUploadSimple(new File([Buffer.from(result.data.processedFile.content, "base64")], result.data.fileName, { type: "text/markdown" }));
          setModal({
            type: "success",
            title: "전처리 완료",
            message: "전처리 후 업로드되었습니다."
          });
        }

        // 문서 목록 새로고침 트리거
        setRefreshTrigger((prev) => prev + 1);
      } else {
        throw new Error("전처리 실패");
      }
    } catch (error) {
      console.error("Excel 전처리 실패:", error);
      setModal({
        type: "error",
        title: "전처리 실패",
        message: "전처리 중 오류가 발생했습니다."
      });
    }
  };

  const handlePdfProcess = async (options: any) => {
    console.log("PDF 전처리 시작:", options);
    try {
      const formData = new FormData();
      formData.append("file", options.file);
      formData.append("options", JSON.stringify(options));

      const response = await fetch("/api/preprocess/pdf", {
        method: "POST",
        body: formData
      });

      if (response.ok) {
        const result = await response.json();
        console.log("PDF 전처리 완료:", result);
        // 완료 후 파일 업로드 로직 호출
        if (result.data?.processedFile) {
          await handleFileUploadSimple(new File([Buffer.from(result.data.processedFile.content, "base64")], result.data.fileName, { type: "application/pdf" }));
        }
        setModal({
          type: "success",
          title: "전처리 완료",
          message: `전처리 후 업로드 되었습니다.`
        });
      } else {
        throw new Error("전처리 실패");
      }
    } catch (error) {
      console.error("PDF 전처리 실패:", error);
      setModal({
        type: "error",
        title: "전처리 실패",
        message: "전처리 중 오류가 발생했습니다."
      });
    }
  };

  // GraphRAG 분석 처리 함수
  const handleGraphRAGProcess = async (documentId: string): Promise<void> => {
    try {
      const response = await fetch("/api/v1/rag/graphrag/process", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          "X-API-Key": localStorage.getItem("apiKey") || ""
        },
        body: JSON.stringify({
          document_id: documentId,
          user_id: user?.username || "admin"
        })
      });

      if (!response.ok) {
        if (response.status === 404) {
          throw new Error("GraphRAG API가 아직 활성화되지 않았습니다. API 서버를 확인해주세요.");
        }
        throw new Error(`GraphRAG 처리 실패: ${response.status}`);
      }

      const result = await response.json();
      if (!result.success) {
        throw new Error(result.error?.message || "GraphRAG 처리 실패");
      }

      setModal({
        type: "success",
        title: "GraphRAG 분석 완료",
        message: "GraphRAG 분석이 성공적으로 완료되었습니다."
      });
    } catch (error) {
      console.error("GraphRAG 처리 오류:", error);
      setModal({
        type: "error",
        title: "GraphRAG 분석 실패",
        message: error instanceof Error ? error.message : "GraphRAG 분석 중 오류가 발생했습니다."
      });
    }
  };

  // 문서 선택 시 GraphRAG 파일 미리보기 모달 열기
  const handleDocumentGraphRAGView = async (doc: any) => {
    setSelectedGraphRAGDoc(doc);

    try {
      // 문서의 청크 정보 가져오기
      const response = await fetch(`/api/v1/rag/documents/${doc.id}/chunks`, {
        headers: {
          "X-API-Key": localStorage.getItem("apiKey") || ""
        }
      });

      if (response.ok) {
        const result = await response.json();
        setGraphRAGChunks(result.data || []);
      } else {
        setGraphRAGChunks([]);
      }
    } catch (error) {
      console.error("청크 정보 로드 실패:", error);
      setGraphRAGChunks([]);
    }

    setShowGraphRAGFileModal(true);
  };

  // 간단한 파일 업로드 함수 (chat/page.tsx와 동일한 방식 - 백그라운드 작업으로 처리)
  const sleep = (ms: number) => new Promise((r) => setTimeout(r, ms));
  async function fileToBase64(file: File): Promise<string> {
    return new Promise<string>((resolve, reject) => {
      const reader = new FileReader();
      reader.onload = () => {
        try {
          const result = reader.result as string; // data:<mime>;base64,<...>
          const base64 = result.split(",")[1] ?? "";
          resolve(base64);
        } catch (e) {
          reject(e);
        }
      };
      reader.onerror = reject;
      reader.readAsDataURL(file);
    });
  }

  const handleFileUploadSimple = async (file: File, targetUserId?: string): Promise<void> => {
    if (!user?.username) {
      setModal({
        type: "error",
        title: "오류",
        message: "파일 업로드를 위해 로그인이 필요합니다."
      });
      return;
    }

    const uploadUserId = targetUserId || user.username;
    const fileName = file.name;

    try {
      console.log(`파일 "${fileName}" 백그라운드 업로드 시작...`);

      // 1) 파일 → Base64
      const base64Content = await fileToBase64(file);

      // 2) 백그라운드 작업 생성
      const response = await fetch("/api/support/background-jobs", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({
          type: "file_upload",
          userId: uploadUserId,
          filename: fileName,
          fileData: {
            name: fileName,
            size: file.size,
            type: file.type,
            content: base64Content
          },
          metadata: {
            targetFolder: "admin-rag",
            uploadedBy: user.username
          }
        })
      });

      // 3) 응답 바디는 한 번만 파싱
      let payload: any = null;
      try {
        payload = await response.json();
      } catch {
        // 빈 바디일 수도 있으니 무시
        payload = null;
      }

      // 4) 1초 대기(서버 처리 여유, pdf 임베딩시 파일 용량에 따라 시간이 더 걸릴 수 있음)
      await sleep(1000);

      if (!response.ok) {
        const msg = payload?.error?.message || `백그라운드 작업 생성 실패 (${response.status})`;
        throw new Error(msg);
      }

      console.log(`파일 "${fileName}" 백그라운드 작업 생성 완료:`, payload);

      // 5) 성공 시 한 번만 목록 새로고침
      setRefreshTrigger((prev) => prev + 1);
    } catch (error) {
      console.error(`파일 ${fileName} 업로드 실패:`, error);
      setModal({
        type: "error",
        title: "업로드 실패",
        message: `파일 ${fileName} 업로드 중 오류가 발생했습니다: ` + (error instanceof Error ? error.message : "알 수 없는 오류")
      });
    } finally {
      console.log(`파일 "${fileName}" 백그라운드 업로드 처리 완료.`);
      // 필요하면 여기에서 추가 정리 작업만 수행 (중복 새로고침 방지 위해 목록 갱신은 성공 분기에서만)
    }
  };

  // 크기 조정 핸들러
  const handleMouseDown = (e: React.MouseEvent) => {
    e.preventDefault();
    setIsResizing(true);
  };

  const handleMouseMove = (e: MouseEvent) => {
    if (!isResizing) return;

    const minCenterWidth = 500; // 가운데 패널 최소 너비 (청크 내용 보기용)
    const minRightWidth = 420; // 오른쪽 패널 최소 너비 (새 대화 버튼 보장)

    // 마우스 위치에서 오른쪽 패널 너비 계산
    const newRightWidth = windowWidth - e.clientX;

    // 최소/최대 너비 제한
    const clampedRightWidth = Math.max(minRightWidth, Math.min(windowWidth - leftPanelWidth - minCenterWidth, newRightWidth));

    setRightPanelWidth(clampedRightWidth);
  };

  const handleMouseUp = () => {
    setIsResizing(false);
  };

  // 윈도우 크기 추적
  useEffect(() => {
    const updateWindowWidth = () => {
      setWindowWidth(window.innerWidth);
    };

    updateWindowWidth(); // 초기값 설정
    window.addEventListener("resize", updateWindowWidth);

    return () => {
      window.removeEventListener("resize", updateWindowWidth);
    };
  }, []);

  // 마우스 이벤트 리스너 등록
  useEffect(() => {
    if (isResizing) {
      document.addEventListener("mousemove", handleMouseMove);
      document.addEventListener("mouseup", handleMouseUp);

      // 드래그 중 텍스트 선택 방지
      document.body.style.userSelect = "none";
      document.body.style.cursor = "col-resize";

      return () => {
        document.removeEventListener("mousemove", handleMouseMove);
        document.removeEventListener("mouseup", handleMouseUp);
        document.body.style.userSelect = "";
        document.body.style.cursor = "";
      };
    }
  }, [isResizing, handleMouseMove, handleMouseUp]);

  // provider/model 정보 불러오기 (최초 1회)
  useEffect(() => {
    const loadProviderConfig = async () => {
      try {
        const res = await externalApiClient.getCurrentProviderConfig();
        if (res.success && res.data) {
          setProvider(res.data.provider || "openai");
          setModel(res.data.model || "gpt-4");
          if (typeof res.data.temperature === "number") setTemperature(res.data.temperature);
        }
      } catch {}
    };
    loadProviderConfig();
  }, []);

  // provider/model 목록 불러오기 (동적)
  useEffect(() => {
    const fetchProviders = async () => {
      try {
        const res = await externalApiClient.get("/providers");
        // 실제 응답 구조에 맞게 파싱
        if (res.success && res.data && typeof res.data === "object") {
          const options = Object.entries(res.data).map(([key, p]: [string, any]) => ({
            value: key,
            label: p.name || key,
            models: p.models ? p.models.map((m: any) => ({ value: m.id || m, label: m.name || m })) : p.defaultModel ? [{ value: p.defaultModel, label: p.defaultModel }] : []
          }));
          setProviderOptions(options);
        }
      } catch (e) {
        console.error(e);
      }
    };
    fetchProviders();
  }, []);

  // provider 변경 시 model 목록 동기화
  useEffect(() => {
    const found = providerOptions.find((p) => p.value === provider);
    setModelOptions(found?.models || []);
    if (found && found.models.length > 0 && !found.models.find((m) => m.value === model)) {
      setModel(found.models[0].value);
    }
  }, [provider, providerOptions]);

  // 마운트 시 RAG 설정값 불러오기
  useEffect(() => {
    const loadRagSettings = async () => {
      try {
        console.log("🔄 RAG 설정 로딩 시작...");
        const res = await externalApiClient.get("/config");
        console.log("📡 /config API 응답:", res);

        if (res.success && res.data && res.data.configs) {
          const loadedSettings = res.data.configs;
          console.log("📦 로드된 설정:", loadedSettings);

          // fallback 기본값 정의 (4-Score System)
          const fallbackDefaults: Record<string, string> = {
            RAG_KEYWORD_PRECISION_THRESHOLD: "0.1",
            RAG_SEMANTIC_RELEVANCE_THRESHOLD: "0.15",
            RAG_CONTENT_QUALITY_THRESHOLD: "0.3",
            RAG_RECENCY_THRESHOLD: "0.0",
            RAG_TOP_K: "5"
          };

          // 페이지에 표시되는 RAG 키들 추출
          const ragSettingKeys = Object.values(ragSettingGroups).flatMap((group) => group.keys);
          console.log("🔑 RAG 설정 키들:", ragSettingKeys);

          // 로드된 설정값이 없는 경우 fallback 적용
          const settingsWithFallback = { ...loadedSettings };
          ragSettingKeys.forEach((key) => {
            if ((!settingsWithFallback[key] || settingsWithFallback[key] === "" || settingsWithFallback[key] === null) && fallbackDefaults[key]) {
              settingsWithFallback[key] = fallbackDefaults[key];
            }
          });

          console.log("✅ 최종 설정 (fallback 적용 후):", settingsWithFallback);
          setSettings(settingsWithFallback);

          // GraphRAG 활성화 상태를 전역 변수에 저장
          globalGraphRAGEnabled = settingsWithFallback["RAG_ENABLE_GRAPHRAG"] === "yes" || settingsWithFallback["RAG_ENABLE_GRAPHRAG"] === true;
        } else {
          console.warn("⚠️ API 응답이 올바르지 않음:", res);
        }
      } catch (e) {
        console.error("❌ 설정 로드 중 오류", e);
      }
    };
    loadRagSettings();
  }, []);

  // settings 변경 시 GraphRAG 전역 변수 업데이트
  useEffect(() => {
    console.log("🔄 메인 settings 상태 변경됨:", settings);
    console.log("📊 settings 객체 크기:", Object.keys(settings).length);
    globalGraphRAGEnabled = settings["RAG_ENABLE_GRAPHRAG"] === "yes" || settings["RAG_ENABLE_GRAPHRAG"] === true;
  }, [settings]);

  // 마운트 시 실패한 백그라운드 작업 자동 정리
  useEffect(() => {
    const cleanupFailedJobs = async () => {
      if (!user?.username) return;

      try {
        // 현재 사용자의 모든 백그라운드 작업 조회
        const response = await fetch(`/api/support/background-jobs?userId=${user.username}`);

        if (response.ok) {
          const result = await response.json();

          if (result.success) {
            const allJobs = result.data.jobs || [];

            // 모든 실패한 파일 업로드 작업 선별
            const failedJobs = allJobs.filter((job: any) => job.status === "failed" && job.type === "file_upload");

            // 실패한 작업 정리
            if (failedJobs.length > 0) {
              console.log(`[AdminRagPage] 실패한 작업 ${failedJobs.length}개 정리 중...`);

              // 실패한 작업들을 일괄 삭제
              for (const job of failedJobs) {
                try {
                  const deleteResponse = await fetch(`/api/support/background-jobs?jobId=${job.id}`, {
                    method: "DELETE"
                  });

                  if (!deleteResponse.ok) {
                    console.warn(`[AdminRagPage] 작업 삭제 실패: ${job.filename}`);
                  }
                } catch (deleteError) {
                  console.error(`[AdminRagPage] 작업 삭제 중 오류: ${job.filename}`, deleteError);
                }
              }

              console.log(`[AdminRagPage] 실패한 작업 정리 완료: ${failedJobs.length}개`);
            }

            // 완료된 작업 중 오래된 것들 정리 (7일 이상)
            const now = Date.now();
            const oldCompletedJobs = allJobs.filter(
              (job: any) => job.status === "completed" && job.type === "file_upload" && now - new Date(job.updatedAt).getTime() > 7 * 24 * 60 * 60 * 1000 // 7일 이상
            );

            if (oldCompletedJobs.length > 0) {
              console.log(`[AdminRagPage] 오래된 완료 작업 ${oldCompletedJobs.length}개 정리 중...`);

              // 오래된 완료 작업들을 일괄 삭제
              for (const job of oldCompletedJobs) {
                try {
                  const deleteResponse = await fetch(`/api/support/background-jobs?jobId=${job.id}`, {
                    method: "DELETE"
                  });

                  if (!deleteResponse.ok) {
                    console.warn(`[AdminRagPage] 오래된 완료 작업 삭제 실패: ${job.filename}`);
                  }
                } catch (deleteError) {
                  console.error(`[AdminRagPage] 오래된 완료 작업 삭제 중 오류: ${job.filename}`, deleteError);
                }
              }

              console.log(`[AdminRagPage] 오래된 완료 작업 정리 완료: ${oldCompletedJobs.length}개`);
            }
          }
        }
      } catch (error) {
        console.error("[AdminRagPage] 백그라운드 작업 정리 중 오류:", error);
      }
    };

    // 존재하지 않는 문서에 대한 Redis 작업 정리
    const cleanupOrphanedJobs = async () => {
      if (!user?.username) return;

      try {
        console.log("[cleanupOrphanedJobs] 고아 작업 정리 시작");

        // 백그라운드 작업 API에서 모든 작업 로드
        const response = await fetch(`/api/support/background-jobs?userId=${user.username}`);
        if (!response.ok) {
          console.log("[cleanupOrphanedJobs] 작업 로드 실패");
          return;
        }

        const result = await response.json();
        if (!result.success) {
          console.log("[cleanupOrphanedJobs] 작업 데이터 로드 실패");
          return;
        }

        const allJobs = result.data.jobs || [];
        if (allJobs.length === 0) {
          console.log("[cleanupOrphanedJobs] 정리할 작업이 없음");
          return;
        }

        let cleanedCount = 0;

        for (const job of allJobs) {
          if (job.type === "file_upload" && job.filename) {
            try {
              // 파일이 실제로 존재하는지 확인
              const apiKey = localStorage.getItem("apiKey");
              const authHeaders: Record<string, string> = {
                "Content-Type": "application/json"
              };
              if (apiKey) {
                // API 서버는 X-API-Key 헤더를 요구함
                authHeaders["X-API-Key"] = apiKey;
              }
              const fileResponse = await fetch(`/api/v1/rag/files?userId=${job.userId}`, {
                headers: authHeaders
              });
              if (fileResponse.ok) {
                const fileData = await fileResponse.json();
                if (fileData.success) {
                  // 파일 트리에서 해당 파일 찾기
                  const findFileInTree = (nodes: any[]): boolean => {
                    for (const node of nodes) {
                      if (node.type === "file" && node.name === job.filename) {
                        return true;
                      }
                      if (node.children && findFileInTree(node.children)) {
                        return true;
                      }
                    }
                    return false;
                  };

                  const fileExists = findFileInTree(fileData.files || []);

                  if (!fileExists) {
                    console.log(`[cleanupOrphanedJobs] 존재하지 않는 파일 작업 정리: ${job.filename}`);

                    // Redis에서 작업 제거
                    const deleteResponse = await fetch(`/api/support/background-jobs?jobId=${job.id}`, {
                      method: "DELETE"
                    });

                    if (deleteResponse.ok) {
                      cleanedCount++;
                    } else {
                      console.warn(`[cleanupOrphanedJobs] 작업 제거 실패: ${job.filename}`);
                    }
                  }
                }
              }
            } catch (error) {
              console.error(`[cleanupOrphanedJobs] 파일 확인 중 오류 (${job.filename}):`, error);

              // 오류 발생 시에도 작업 정리 (안전을 위해)
              try {
                const deleteResponse = await fetch(`/api/support/background-jobs?jobId=${job.id}`, {
                  method: "DELETE"
                });
                if (deleteResponse.ok) {
                  cleanedCount++;
                }
              } catch (deleteError) {
                console.error(`[cleanupOrphanedJobs] 작업 제거 중 오류 (${job.id}):`, deleteError);
              }
            }
          }
        }

        if (cleanedCount > 0) {
          console.log(`[cleanupOrphanedJobs] ${cleanedCount}개의 고아 작업 정리 완료`);
          // 작업 목록 새로고침
          setRefreshTrigger((prev: number) => prev + 1);
        } else {
          console.log("[cleanupOrphanedJobs] 정리할 고아 작업이 없음");
        }
      } catch (error) {
        console.error("[cleanupOrphanedJobs] 고아 작업 정리 중 오류:", error);
      }
    };

    // 페이지 로딩 시 한 번만 실행
    const initializeCleanup = async () => {
      await cleanupFailedJobs();
      await cleanupOrphanedJobs(); // 고아 작업 정리 추가
    };

    initializeCleanup();
  }, [user?.username]);

  // 문서 선택 시 청크 불러오기
  const handleSelectDoc = async (docName: string) => {
    setSelectedDoc(docName);
    setChunks([]);
    setCurrentChunkIndex(0);
    try {
      const res = (await externalApiClient.get(`/rag/chunks?filename=${encodeURIComponent(docName)}&user_id=${user?.id || ""}`)) as RagChunksResponse;
      if (res.success && Array.isArray(res.chunks)) {
        setChunks(res.chunks);
      } else {
        setChunks([]);
      }
    } catch {
      setChunks([]);
    }
  };

  const handlePrevChunk = () => {
    setCurrentChunkIndex((idx) => Math.max(0, idx - 1));
  };
  const handleNextChunk = () => {
    setCurrentChunkIndex((idx) => (chunks.length ? Math.min(chunks.length - 1, idx + 1) : 0));
  };

  // 마운트 시 가장 최근 세션 불러오기
  useEffect(() => {
    const loadLastSession = async () => {
      // API 서버의 DB에서 support 타입 세션 조회
      const res = await externalApiClient.getSessions("support");
      if (res.success && Array.isArray(res.data) && res.data.length > 0) {
        const lastSession = res.data[0];
        setCurrentSessionId(lastSession.id);
        setCurrentMessages(lastSession.history || []);
      } else {
        // support 세션이 없으면 새로 시작
        handleNewChat();
      }
    };
    loadLastSession();
  }, []);

  // 새 대화 시작
  const handleNewChat = async () => {
    // 인증 토큰 가져오기
    const apiKey = localStorage.getItem("apiKey");
    const authHeaders: Record<string, string> = {
      "Content-Type": "application/json"
    };
    if (apiKey) {
      authHeaders["Authorization"] = `Bearer ${apiKey}`;
    }

    // RAG 캐시 삭제도 제거됨 - 새 대화와 RAG 캐시는 논리적으로 무관

    // 웹검색 캐시 삭제 제거됨 - 30분 TTL로 충분하며, 성능상 이점이 더 큼

    // support- 접두사가 붙은 새로운 세션 ID 생성
    const newSessionId = `support-${Date.now()}-${Math.random().toString(36).substring(2, 9)}`;
    setCurrentSessionId(newSessionId);
    setCurrentMessages([]);
  };

  // 메시지 전송 핸들러 (세션 ID 포함)
  const handleChat = async (message: string, searchScope: string = "personal") => {
    setCurrentMessages((prev) => [...prev, { role: "user", content: message }]);
    setIsLoading(true);
    const startTime = performance.now();

    // API 키 확인
    const apiKey = localStorage.getItem("apiKey");
    if (!apiKey) {
      console.error("API 키가 없습니다. 로그인을 다시 해주세요.");
      setCurrentMessages((prev) => [
        ...prev,
        {
          role: "assistant",
          content: "API 키가 없습니다. 로그인을 다시 해주세요.",
          error: true
        }
      ]);
      setIsLoading(false);
      return;
    }

    // 사용자 인증 상태 확인
    if (!isAuthenticated || !user) {
      console.error("사용자가 인증되지 않았습니다:", { isAuthenticated, user, isInitialized });
      setCurrentMessages((prev) => [
        ...prev,
        {
          role: "assistant",
          content: "사용자 인증이 필요합니다. 로그인을 다시 해주세요.",
          error: true
        }
      ]);
      setIsLoading(false);
      return;
    }

    // 사용자 username 확인
    if (!user.username || user.username.trim() === "") {
      console.error("사용자 이름이 없습니다:", { user });
      setCurrentMessages((prev) => [
        ...prev,
        {
          role: "assistant",
          content: "사용자 정보가 불완전합니다. 로그인을 다시 해주세요.",
          error: true
        }
      ]);
      setIsLoading(false);
      return;
    }

    // 사용자 정보 디버그 로그
    console.log("사용자 정보:", { user, id: user?.id, username: user?.username });
    console.log("API 키:", apiKey.substring(0, 10) + "...");
    console.log("인증 상태:", { isAuthenticated, isInitialized });

    // username 문자열을 일관되게 사용 - 검증 및 정리
    let usernameForRag = user.username;
    let numericUserId = 1; // Chat API용 기본값

    // Chat API용 숫자 ID 준비
    if (user?.id) {
      const parsed = parseInt(user.id, 10);
      numericUserId = isNaN(parsed) ? 1 : parsed;
    }

    // username 유효성 검증 및 정리 로직
    if (!usernameForRag || usernameForRag.trim() === "" || usernameForRag === "undefined" || usernameForRag === "null") {
      console.warn("유효하지 않은 사용자명입니다. 기본값으로 설정:", user);
      usernameForRag = user?.id ? `user_${user.id}` : "anonymous";
    }

    // 순수 숫자만으로 된 username을 읽기 좋은 형태로 변경
    if (/^\d+$/.test(usernameForRag.trim())) {
      console.warn("숫자로만 구성된 사용자명을 식별 가능한 형태로 변경:", usernameForRag);
      usernameForRag = `user_${usernameForRag}`;
    }

    // 공백 및 특수문자 정리
    usernameForRag = usernameForRag.trim().replace(/[^\w\-_]/g, "_");

    console.log("최종 사용자 정보:", {
      usernameForRag, // RAG API용 (문자열)
      numericUserId, // Chat API용 (숫자)
      originalData: { id: user?.id, username: user?.username },
      processedUsername: usernameForRag
    });

    // support- 접두사가 붙은 세션 ID 생성 (없는 경우에만)
    let sessionId = currentSessionId;
    if (!sessionId) {
      sessionId = `support-${Date.now()}-${Math.random().toString(36).substring(2, 9)}`;
      setCurrentSessionId(sessionId);
    }

    try {
      // 1. 먼저 RAG 검색 결과 요청 및 표시
      setRagSearching(true);
      console.log("RAG 검색 요청 시작:", { message, userId: usernameForRag });

      try {
        // 검색 범위에 따른 API 호출 준비
        const ragRequest: any = {
          query: message,
          numericUserId: numericUserId, // 세션 관리용 정수형 ID
          ragSearchScope: searchScope // 검색 범위 설정 (API 서버에서 요구하는 키)
        };

        // 검색 범위는 ragSearchScope로, 실제 사용자 ID는 별도로 전달
        if (searchScope === "personal" || searchScope === "all") {
          ragRequest.actual_user_id = usernameForRag; // 실제 사용자 ID 전달
        }
        // user_id는 더 이상 검색 범위 표시용으로 사용하지 않음

        const ragRes = await externalApiClient.post("/rag/search/sync", ragRequest);

        console.log("RAG 검색 결과:", ragRes);

        // RAG 검색 결과 파싱 및 표시
        let ragResults = [];
        if (ragRes?.success && ragRes?.data) {
          const { documents, metadatas, distances } = ragRes.data;

          // 응답 구조를 파싱하여 표준 형태로 변환
          if (Array.isArray(documents) && documents.length > 0) {
            ragResults = documents[0].map((content: string, idx: number) => {
              const metadata = metadatas?.[0]?.[idx] || {};
              return {
                content: content,
                source: metadata.source || "알 수 없는 문서",
                // 4-Score RAG System
                core_scores: {
                  keyword_precision: metadata.core_scores?.keyword_precision ?? 0,
                  semantic_relevance: metadata.core_scores?.semantic_relevance ?? 0,
                  content_quality: metadata.core_scores?.content_quality ?? 0,
                  recency: metadata.core_scores?.recency ?? 0
                },
                matches: {
                  exact_matches: metadata.matches?.exact_matches || [],
                  partial_matches: metadata.matches?.partial_matches || []
                }
              };
            });
          }
        }

        setCurrentMessages((prev) => [
          ...prev,
          {
            role: "rag",
            content: ragResults,
            timestamp: new Date().toISOString(),
            searchScope: searchScope
          }
        ]);
      } catch (ragError) {
        console.warn("RAG 검색 실패:", ragError);
        // RAG 검색 실패시에도 빈 결과 표시
        setCurrentMessages((prev) => [
          ...prev,
          {
            role: "rag",
            content: [],
            timestamp: new Date().toISOString(),
            error: "RAG 검색 중 오류가 발생했습니다.",
            searchScope: searchScope
          }
        ]);
      } finally {
        setRagSearching(false);
      }

      // 2. AI 응답 요청
      setAiResponding(true);
      console.log("AI 응답 요청 시작:", { message, sessionId, userId: numericUserId });

      const res = await externalApiClient.post("/chat/sync", {
        prompt: message,
        options: {
          rag: true,
          provider,
          model,
          temperature,
          userId: numericUserId, // 숫자 ID 사용 (API 서버에서 요구)
          sessionId,
          type: "support", // sessionType 대신 type 사용
          ragSearchScope: searchScope // RAG 검색 범위 전달
        }
      });

      console.log("AI 응답 받음:", res);

      // === [수정] 응답 파싱 간소화 ===
      let aiMsg = "";

      // 응답 구조 확인 및 파싱
      if (res?.data?.success && res?.data?.data) {
        const responseData = res.data.data;

        // 직접 response 필드 확인
        if (responseData.response) {
          aiMsg = responseData.response;
        }
        // history에서 마지막 assistant 메시지 확인
        else if (Array.isArray(responseData.history)) {
          const lastAssistant = [...responseData.history].reverse().find((msg) => msg.role === "assistant" && msg.content);
          aiMsg = lastAssistant?.content || "";
        }
        // data 내부에 또 다른 data가 있는 경우
        else if (responseData.data) {
          if (responseData.data.response) {
            aiMsg = responseData.data.response;
          } else if (Array.isArray(responseData.data.history)) {
            const lastAssistant = [...responseData.data.history].reverse().find((msg) => msg.role === "assistant" && msg.content);
            aiMsg = lastAssistant?.content || "";
          }
        }
      }

      // 텍스트 정리
      if (aiMsg && typeof aiMsg === "string") {
        aiMsg = aiMsg.replace(/<br\s*\/?>/gi, "\n").replace(/\n{2,}/g, "\n");
      }

      const elapsed = ((performance.now() - startTime) / 1000).toFixed(2);

      // 세션 ID 업데이트
      if (res?.data?.data?.sessionId) {
        setCurrentSessionId(res.data.data.sessionId);
      }

      // AI 응답 메시지 추가
      setCurrentMessages((prev) => [
        ...prev,
        {
          role: "assistant",
          content: aiMsg || "AI 응답을 불러올 수 없습니다. 응답 구조를 확인해주세요.",
          elapsed
        }
      ]);
    } catch (error) {
      console.error("AI 응답 요청 중 오류:", error);
      const elapsed = ((performance.now() - startTime) / 1000).toFixed(2);
      setCurrentMessages((prev) => [
        ...prev,
        {
          role: "assistant",
          content: `AI 응답 중 오류가 발생했습니다: ${error instanceof Error ? error.message : "알 수 없는 오류"}`,
          elapsed
        }
      ]);
    } finally {
      setAiResponding(false);
      setIsLoading(false);
    }
  };

  // API 키 확인 및 설정 함수 추가
  const checkAndSetApiKey = () => {
    const currentApiKey = localStorage.getItem("apiKey");
    // console.log('현재 API 키:', currentApiKey);

    // 테스트용 API 키가 없다면 설정
    if (!currentApiKey || currentApiKey.length < 10) {
      const testApiKey = "airun_1_3d85009c98964579c622eded69997b16"; // 테스트에서 동작한 API 키
      localStorage.setItem("apiKey", testApiKey);
      externalApiClient.setApiKey(testApiKey);
      // console.log('테스트 API 키 설정됨:', testApiKey.substring(0, 10) + '...');
      return testApiKey;
    }

    return currentApiKey;
  };

  // 컴포넌트 마운트 시 API 키 확인 및 상태 정보 자동 정리
  useEffect(() => {
    checkAndSetApiKey();

    // 페이지 로딩 시 자동으로 상태 정보 정리 (진행 중인 작업은 보존)
    const autoCleanup = async () => {
      try {
        // 1. RAG 서버 상태 정리 (Redis의 rag:status:* 키들 정리)
        const response = await fetch(`${getRagServerUrl()}/rag/cleanup-status`, {
          method: "POST"
        });

        if (response.ok) {
          console.log("RAG 서버 상태 정보가 정리되었습니다.");
        } else {
          console.warn("RAG 서버 상태 정리 실패");
        }

        // 2. 백그라운드 작업 정보는 안전하게 정리 (완료된 작업만)
        const cleanupResponse = await fetch("/api/support/background-jobs/cleanup-safe", {
          method: "POST"
        });

        if (cleanupResponse.ok) {
          console.log("백그라운드 작업 정보가 안전하게 정리되었습니다.");
        } else {
          console.warn("백그라운드 작업 정보 정리 실패");
        }
      } catch (error) {
        console.warn("상태 정보 자동 정리 중 오류 발생:", error);
      }
    };

    autoCleanup();

    // 문서 통계 로드
    loadDocumentStats();

    // 주기적으로 문서 통계 새로고침 (3초마다)
    const statsInterval = setInterval(() => {
      loadDocumentStats();
    }, 3000);

    return () => {
      clearInterval(statsInterval);
    };
  }, []);

  // PostgreSQL chat_documents 기반 문서 통계 가져오기 (관리자용)
  const loadDocumentStats = async () => {
    try {
      const apiServerUrl = getApiServerUrl();
      const response = await fetch(`${apiServerUrl}/rag/admin/document-stats`, {
        headers: {
          "X-API-Key": localStorage.getItem("apiKey") || "",
          "X-Username": localStorage.getItem("username") || "admin",
          "X-User-Role": localStorage.getItem("userRole") || "admin"
        }
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      const result = await response.json();

      if (result.success && result.data) {
        const { total, completed, processing, userStats } = result.data;
        setDocumentStats({
          total: total || 0,
          completed: completed || 0,
          processing: processing || 0
        });
        console.log('📊 문서 통계 업데이트 (PostgreSQL 기반):', { total, completed, processing });
        console.log('👥 사용자별 통계:', userStats);
      } else {
        console.warn("문서 통계 응답이 예상과 다릅니다:", result);
        setDocumentStats({ total: 0, completed: 0, processing: 0 });
      }
    } catch (error) {
      console.error("문서 통계 로드 실패:", error);
      setDocumentStats({ total: 0, completed: 0, processing: 0 });
    }
  };

  // 선택된 항목에 따른 버튼 텍스트 결정
  const getInitButtonText = () => {
    if (isInitializing) return "초기화 중...";

    // 1. 선택된 문서가 있는 경우 - 해당 문서만 초기화(삭제)
    if (selectedDoc) {
      return `'${selectedDoc}' 초기화`;
    }
    // 2. 선택된 폴더가 있는 경우 - 해당 폴더(사용자) 초기화
    else if (selectedFolder && selectedFolder.userId) {
      return `'${selectedFolder.userId}' 초기화`;
    }
    // 3. 선택된 항목이 없는 경우
    else {
      return "내 문서 초기화";
    }
  };

  if (isAuthLoading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500"></div>
      </div>
    );
  }
  if (!isAuthenticated) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="text-red-500">인증되지 않은 사용자입니다.</div>
      </div>
    );
  }

  // 가운데 패널 너비 계산 - 가로 스크롤 방지
  const centerPanelWidth = Math.max(300, windowWidth - leftPanelWidth - rightPanelWidth - 4); // 최소 300px 보장

  return (
    <div className="h-screen flex overflow-x-hidden" style={{ backgroundColor: "var(--body-bg)" }}>
      {/* 왼쪽: 사이드바 (설정 + 파일 매니저) */}
      <div
        style={{
          width: `${leftPanelWidth}px`,
          backgroundColor: "var(--card-bg)",
          borderRight: "1px solid var(--card-border)"
        }}
        className="min-h-0 h-full border-r flex-shrink-0"
      >
        <RagSidebar
          settings={settings}
          onSettingsChange={setSettings}
          collection={collection}
          onSelectDoc={handleSelectDoc}
          selectedDoc={selectedDoc}
          selectedFolder={selectedFolder}
          setSelectedFolder={setSelectedFolder}
          onFolderSelect={setSelectedFolder}
          chunks={chunks}
          currentChunkIndex={currentChunkIndex}
          setCurrentChunkIndex={setCurrentChunkIndex}
          onPrevChunk={handlePrevChunk}
          onNextChunk={handleNextChunk}
          user={user}
          onJobComplete={handleJobComplete}
          refreshTrigger={refreshTrigger}
          onFileUpload={handleFileUploadSimple}
          onShowExcelModal={() => setShowExcelModal(true)}
          onShowPdfModal={() => setShowPdfModal(true)}
          onGraphRAGView={handleDocumentGraphRAGView}
          onFullSystemInit={() => {
            setInitTrigger({ type: "full" });
            // 초기화 시작 즉시 및 완료 후 통계 새로고침
            loadDocumentStats();
            setTimeout(() => loadDocumentStats(), 5000);
            setTimeout(() => loadDocumentStats(), 15000);
          }}
          onDocumentInit={(docName, userId) => {
            setInitTrigger({ type: "document", docName, userId });
            // 문서 초기화 시작 즉시 및 완료 후 통계 새로고침
            loadDocumentStats();
            setTimeout(() => loadDocumentStats(), 3000);
            setTimeout(() => loadDocumentStats(), 10000);
          }}
          onFolderInit={(userId) => {
            setInitTrigger({ type: "folder", userId });
            // 폴더 초기화 시작 즉시 및 완료 후 통계 새로고침
            loadDocumentStats();
            setTimeout(() => loadDocumentStats(), 5000);
            setTimeout(() => loadDocumentStats(), 15000);
          }}
        />
      </div>

      {/* 가운데: 메인 패널 (업로드 + 청크 미리보기) */}
      <div
        style={{
          width: `${centerPanelWidth}px`,
          minWidth: "300px",
          backgroundColor: "var(--card-bg)"
        }}
        className="h-full overflow-y-auto flex-shrink-0"
      >
        <RagMainPanel
          chunks={chunks}
          selectedDoc={selectedDoc}
          currentChunkIndex={currentChunkIndex}
          setCurrentChunkIndex={setCurrentChunkIndex}
          onPrevChunk={handlePrevChunk}
          onNextChunk={handleNextChunk}
          showGraphRAGModal={showGraphRAGModal}
          setShowGraphRAGModal={setShowGraphRAGModal}
          setCurrentGraphQuery={setCurrentGraphQuery}
          isInitializing={isInitializing}
          docStatuses={docStatuses}
          progressPercent={progressPercent}
          statusMessage={statusMessage}
          initStartTime={initStartTime}
          documentStats={documentStats}
        />
      </div>

      {/* 드래그 가능한 구분선 */}
      <div
        className="w-1 h-full cursor-col-resize transition-colors"
        onMouseDown={handleMouseDown}
        style={{
          minWidth: "4px",
          userSelect: "none",
          WebkitUserSelect: "none",
          MozUserSelect: "none",
          msUserSelect: "none",
          backgroundColor: isResizing ? "#3b82f6" : "var(--card-border)"
        }}
      />

      {/* 오른쪽: 채팅 패널 (크기 조정 가능) */}
      <div
        style={{
          width: `${rightPanelWidth}px`,
          minWidth: "300px",
          backgroundColor: "var(--card-bg)"
        }}
        className="h-full overflow-y-auto flex-shrink-0"
      >
        <RagChatPanel
          chatHistory={currentMessages}
          onChat={handleChat}
          selectedChunk={selectedChunk}
          provider={provider}
          model={model}
          temperature={temperature}
          providerOptions={providerOptions}
          modelOptions={modelOptions}
          onProviderChange={setProvider}
          onModelChange={setModel}
          onTemperatureChange={setTemperature}
          onNewChat={handleNewChat}
          isLoading={isLoading}
          ragSearching={ragSearching}
          aiResponding={aiResponding}
        />
      </div>

      {/* 모달창 */}
      <div className={`fixed inset-0 z-50 flex items-center justify-center transition-all duration-300 ease-in-out ${modal ? "opacity-100 pointer-events-auto" : "opacity-0 pointer-events-none"}`} style={{ backgroundColor: "rgba(0, 0, 0, 0.3)" }}>
        <div
          className={`rounded-lg shadow-lg p-6 min-w-[280px] max-w-xs flex flex-col items-center transition-all duration-300 ease-in-out transform ${modal ? "scale-100 translate-y-0" : "scale-95 translate-y-4"}`}
          style={{
            backgroundColor: "var(--card-bg)",
            boxShadow: "var(--card-shadow)"
          }}
        >
          <div className={`text-lg font-bold mb-2 ${modal?.type === "success" ? "text-green-600" : "text-red-500"}`}>{modal?.title}</div>
          <div className="mb-4 text-center text-base" style={{ color: "var(--text-secondary)" }}>
            <div>{modal?.message}</div>
          </div>
          <button
            className="mt-2 px-4 py-2 rounded transition-all duration-200 text-white"
            style={{
              backgroundColor: "var(--neutral-600)"
            }}
            onMouseEnter={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = "var(--neutral-700)")}
            onMouseLeave={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = "var(--neutral-600)")}
            onClick={() => setModal(null)}
            autoFocus
          >
            닫기
          </button>
        </div>
      </div>

      {/* Excel 전처리 모달 */}
      <ExcelPreprocessModal isOpen={showExcelModal} onClose={() => setShowExcelModal(false)} onProcess={handleExcelProcess} />

      {/* PDF 전처리 모달 */}
      <PdfPreprocessModal isOpen={showPdfModal} onClose={() => setShowPdfModal(false)} onProcess={handlePdfProcess} />

      {/* GraphRAG 모달 */}
      <GraphRAGModal isOpen={showGraphRAGModal} onClose={() => setShowGraphRAGModal(false)} query={currentGraphQuery} userId={user?.username || ""} />

      {/* GraphRAG 파일 미리보기 모달 */}
      <GraphRAGFilePreviewModal isOpen={showGraphRAGFileModal} onClose={() => setShowGraphRAGFileModal(false)} selectedDoc={selectedGraphRAGDoc} chunks={graphRAGChunks} onProcessGraphRAG={handleGraphRAGProcess} />

      {/* Smart Tune 자동 실행 모달 */}
    </div>
  );
}
