import { NextRequest, NextResponse } from 'next/server';
import { getNextjsServerUrl } from '@/config/serverConfig';

// A2A 관련 타입 정의
interface A2AMessage {
  id: string;
  from: string;
  to?: string;
  type: string;
  method?: string;
  params?: any;
  result?: any;
  error?: any;
  timestamp?: string;
}

interface A2AAgent {
  id: string;
  name: string;
  capabilities: string[];
  endpoint: string;
  protocol: string;
  trustLevel: 'high' | 'medium' | 'low';
  status?: 'connected' | 'disconnected' | 'error';
  lastSeen?: Date;
}

// 글로벌 에이전트 레지스트리 (실제로는 데이터베이스나 Redis 사용)
let registeredAgents: A2AAgent[] = [];

/**
 * GET /api/a2a
 * A2A 네트워크 상태 확인 및 기능 조회
 */
export async function GET(request: NextRequest): Promise<NextResponse> {
  try {
    const url = new URL(request.url);
    const action = url.searchParams.get('action');

    switch (action) {
      case 'capabilities':
        // MCP 도구 목록을 A2A 능력으로 노출
        try {
          const mcpResponse = await fetch(`${getNextjsServerUrl()}/api/mcp/tools`);
          const mcpData = await mcpResponse.json();
          
          const capabilities = [];
          if (mcpData.success && mcpData.data) {
            for (const server of mcpData.data.servers) {
              for (const tool of server.tools) {
                capabilities.push({
                  name: `mcp.${tool.name}`,
                  description: tool.description,
                  inputSchema: tool.inputSchema || {},
                  serverName: server.serverName
                });
              }
            }
          }

          return NextResponse.json({
            status: 'ok',
            message: 'A2A 네트워크 활성화됨',
            capabilities,
            exposedTools: capabilities.length,
            agent: {
              id: 'airun-mcp-agent',
              name: 'AI MCP Agent',
              version: '1.0.0',
              protocol: 'A2A/1.0'
            },
            timestamp: new Date().toISOString()
          });
        } catch (mcpError) {
          console.error('MCP 도구 로드 실패:', mcpError);
          return NextResponse.json({
            status: 'ok',
            message: 'A2A 네트워크 활성화됨 (MCP 도구 로드 실패)',
            capabilities: [],
            exposedTools: 0,
            agent: {
              id: 'airun-mcp-agent',
              name: 'AI MCP Agent',
              version: '1.0.0',
              protocol: 'A2A/1.0'
            },
            timestamp: new Date().toISOString()
          });
        }

      case 'status':
        return NextResponse.json({
          isConnected: true,
          connectedAgents: registeredAgents.filter(a => a.status === 'connected').length,
          exposedTools: 0, // capabilities 조회에서 실제 값 계산
          health: 'good',
          networkHealth: 'good',
          timestamp: new Date().toISOString()
        });

      case 'agents':
        return NextResponse.json({
          agents: registeredAgents,
          count: registeredAgents.length,
          timestamp: new Date().toISOString()
        });

      default:
        // 기본 ping 응답
        return NextResponse.json({
          status: 'ok',
          message: 'A2A 네트워크 정상 작동',
          protocol: 'A2A/1.0',
          timestamp: new Date().toISOString()
        });
    }
  } catch (error) {
    console.error('A2A GET 요청 처리 실패:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

/**
 * POST /api/a2a
 * A2A 메시지 처리 (외부 에이전트로부터의 요청)
 */
export async function POST(request: NextRequest): Promise<NextResponse> {
  try {
    const body = await request.json();
    const message: A2AMessage = body;

    // 메시지 유효성 검증
    if (!message.id || !message.from || !message.type) {
      return NextResponse.json(
        { error: 'Invalid A2A message format' },
        { status: 400 }
      );
    }

    // A2A 메시지 타입별 처리
    switch (message.type) {
      case 'tool_call':
        if (message.method && message.method.startsWith('mcp.')) {
          // MCP 도구 호출 처리
          const toolName = message.method.substring(4); // 'mcp.' 제거
          
          try {
            // MCP API를 통해 도구 호출
            const mcpResponse = await fetch(`${getNextjsServerUrl()}/api/mcp/call-tool`, {
              method: 'POST',
              headers: { 'Content-Type': 'application/json' },
              body: JSON.stringify({
                toolName,
                args: message.params || {}
              })
            });

            const mcpResult = await mcpResponse.json();
            
            return NextResponse.json({
              id: `response-${message.id}`,
              type: 'tool_result',
              result: mcpResult,
              timestamp: new Date().toISOString()
            });
          } catch (toolError) {
            console.error(`MCP 도구 호출 실패 (${toolName}):`, toolError);
            return NextResponse.json({
              id: `response-${message.id}`,
              type: 'tool_error',
              error: `Tool execution failed: ${toolError}`,
              timestamp: new Date().toISOString()
            });
          }
        }
        break;

      case 'ping':
        return NextResponse.json({
          id: `response-${message.id}`,
          type: 'pong',
          result: {
            status: 'ok',
            agent: 'airun-mcp-agent',
            timestamp: new Date().toISOString()
          }
        });

      default:
        return NextResponse.json(
          { error: `Unsupported message type: ${message.type}` },
          { status: 400 }
        );
    }

    return NextResponse.json({ status: 'processed' });
  } catch (error) {
    console.error('A2A message processing failed:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

/**
 * PUT /api/a2a
 * A2A 에이전트 등록
 */
export async function PUT(request: NextRequest): Promise<NextResponse> {
  try {
    const body = await request.json();
    const agent: A2AAgent = body;

    // 에이전트 정보 유효성 검증
    if (!agent.id || !agent.name || !agent.endpoint) {
      return NextResponse.json(
        { error: 'Missing required agent fields: id, name, endpoint' },
        { status: 400 }
      );
    }

    // 연결 테스트
    try {
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 10000);

      const pingResponse = await fetch(`${agent.endpoint}/a2a`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-A2A-Protocol': 'A2A/1.0'
        },
        body: JSON.stringify({
          id: `ping-${Date.now()}`,
          from: 'airun-mcp-agent',
          to: agent.id,
          type: 'ping',
          timestamp: new Date().toISOString()
        }),
        signal: controller.signal
      });

      clearTimeout(timeoutId);

      if (pingResponse.ok) {
        // 에이전트 등록 또는 업데이트
        const existingIndex = registeredAgents.findIndex(a => a.id === agent.id);
        const updatedAgent = {
          ...agent,
          status: 'connected' as const,
          lastSeen: new Date()
        };

        if (existingIndex >= 0) {
          registeredAgents[existingIndex] = updatedAgent;
        } else {
          registeredAgents.push(updatedAgent);
        }

        return NextResponse.json({
          status: 'registered',
          agent: agent.id,
          connectionTest: 'passed',
          timestamp: new Date().toISOString()
        });
      } else {
        throw new Error(`Connection test failed: ${pingResponse.status}`);
      }
    } catch (connectionError) {
      console.error('에이전트 연결 테스트 실패:', connectionError);
      return NextResponse.json(
        { error: `Agent connection test failed: ${connectionError}` },
        { status: 400 }
      );
    }
  } catch (error) {
    console.error('A2A agent registration failed:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
} 