import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '../middlewares';
import { getApiServerUrl } from '@/config/serverConfig';

const API_SERVER = getApiServerUrl();

// GET 메서드: 모든 세션 반환 (type 필터링 제거)
export async function GET(request: NextRequest) {
  try {
    const userToken = request.headers.get('Authorization')?.split(' ')[1];
    const apiKey = request.headers.get('X-API-Key');
    
    if (!userToken || !apiKey) {
      return NextResponse.json({ 
        error: '인증 정보가 누락되었습니다. API 키와 토큰이 모두 필요합니다.' 
      }, { status: 401 });
    }

    // API 서버 URL 구성
          const apiServerUrl = getApiServerUrl().replace('/api/v1', '');
    const apiUrl = `${apiServerUrl}/api/v1/sessions`;
    
    // API 서버에서 세션 목록 조회
    const response = await fetch(apiUrl, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${userToken}`,
        'X-API-Key': apiKey,
        'accept': 'application/json'
      },
      cache: 'no-store'
    });

    if (!response.ok) {
      const errorText = await response.text();
      return NextResponse.json(
        { error: `세션 목록을 가져오는데 실패했습니다. (${response.status} ${response.statusText})` },
        { status: response.status }
      );
    }

    const apiData = await response.json();
    let sessions: any[] = [];
    if (Array.isArray(apiData.data)) {
      sessions = apiData.data.map((session: any) => ({
        id: session.id,
        title: session.title || '새 대화',
        firstUserMessage:
          session.firstUserMessage ||
          (Array.isArray(session.history)
            ? (session.history.find((msg: any) => msg.role === 'user' && msg.content && typeof msg.content === 'string')?.content)
            : null) ||
          '새 대화',
        lastMessage: session.lastMessage || session.title || '새 대화',
        updatedAt: session.updatedAt || session.createdAt || new Date().toISOString(),
        messageCount: (session.history || []).length,
        provider: session.provider,
        model: session.model,
        type: session.type,
        history: session.history || []
      }));
    } else if (typeof apiData.data === 'object' && apiData.data !== null) {
      sessions = Object.entries(apiData.data)
        .map(([id, session]: [string, any]) => ({
          id,
          title: session.title || '새 대화',
          firstUserMessage:
            session.firstUserMessage ||
            (Array.isArray(session.history)
              ? (session.history.find((msg: any) => msg.role === 'user' && msg.content && typeof msg.content === 'string')?.content)
              : null) ||
            '새 대화',
          lastMessage: session.lastMessage || session.title || '새 대화',
          updatedAt: session.updatedAt || session.createdAt || new Date().toISOString(),
          messageCount: (session.history || []).length,
          provider: session.provider,
          model: session.model,
          type: session.type,
          history: session.history || []
        }));
    }

    // 최신 순으로 정렬
    sessions.sort((a, b) => new Date(b.updatedAt).getTime() - new Date(a.updatedAt).getTime());

    return NextResponse.json({
      success: true,
      data: sessions
    });

  } catch (error) {
    return NextResponse.json(
      { error: '세션 목록을 가져오는 중 오류가 발생했습니다.' },
      { status: 500 }
    );
  }
}

export async function POST(req: NextRequest) {
  return withAnyAuth(req, async (req, user, authType) => {
    try {
      const body = await req.json();
      const { initialMessage, useRag, useWebSearch, uploadedFiles } = body;

      // 입력 유효성 검사
      if (!initialMessage || typeof initialMessage !== 'string') {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '메시지가 필요합니다.'
            }
          }, 
          { status: 400 }
        );
      }

      // 토큰 또는 API 키 설정
      const headers: Record<string, string> = {
        'Content-Type': 'application/json'
      };

      if (authType === 'token') {
        const token = req.cookies.get('auth_token')?.value;
        if (token) {
          headers['Authorization'] = `Bearer ${token}`;
        } else {
          return NextResponse.json(
            {
              success: false,
              error: {
                message: '인증 토큰이 필요합니다.'
              }
            }, 
            { status: 401 }
          );
        }
      } else if (authType === 'apikey') {
        const apiKey = req.headers.get('x-api-key');
        if (apiKey) {
          headers['X-API-Key'] = apiKey;
        } else {
          return NextResponse.json(
            {
              success: false,
              error: {
                message: 'API 키가 필요합니다.'
              }
            }, 
            { status: 401 }
          );
        }
      } else {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '인증이 필요합니다.'
            }
          }, 
          { status: 401 }
        );
      }

      // API 서버에 채팅 세션 생성 요청
      console.log('API 서버에 요청 전송:', {
        url: `${API_SERVER}/code`,
        method: 'POST',
        headers: {
          ...headers,
          'Authorization': headers['Authorization'] ? '[인증 토큰 포함]' : '없음',
          'X-API-Key': headers['X-API-Key'] ? '[API 키 포함]' : '없음'
        },
        body: {
          prompt: initialMessage,
          options: {
            rag: !!useRag,
            web: !!useWebSearch,
            first: true,
            files: uploadedFiles || []
          }
        }
      });

      const apiResponse = await fetch(`${API_SERVER}/code`, {
        method: 'POST',
        headers,
        body: JSON.stringify({
          prompt: initialMessage,
          options: {
            rag: !!useRag,
            web: !!useWebSearch,
            first: true,
            files: uploadedFiles || [],
            userId: user?.id // 사용자 ID 추가
          }
        })
      });

      if (!apiResponse.ok) {
        const errorData = await apiResponse.json();
        console.error('API 서버 응답 오류:', {
          status: apiResponse.status,
          statusText: apiResponse.statusText,
          errorData
        });
        return NextResponse.json(
          {
            success: false,
            error: {
              message: errorData.error?.message || '세션 생성에 실패했습니다.'
            }
          },
          { status: apiResponse.status }
        );
      }

      const apiData = await apiResponse.json();
      console.log('API 서버 응답 성공:', apiData);
      
      // 모의 데이터 추가 (실제 API 서버가 없는 경우)
      if (!apiData.data?.sessionId) {
        // 임의의 세션 ID 생성
        const mockSessionId = `mock-session-${Date.now()}`;
        
        return NextResponse.json({
          success: true,
          data: {
            id: mockSessionId
          }
        });
      }
      
      return NextResponse.json({
        success: true,
        data: {
          id: apiData.data?.sessionId
        }
      });
    } catch (error) {
      console.error('Error creating code session:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            message: error instanceof Error ? error.message : '세션 생성 중 오류가 발생했습니다.'
          }
        },
        { status: 500 }
      );
    }
  });
} 