import { NextRequest, NextResponse } from 'next/server';
import { pg } from '../db';
import jwt from 'jsonwebtoken';

interface UserMemory {
  id: number;
  user_id: string;
  title: string;
  content: string;
  category: string;
  importance_level: number;
  tags: string[];
  created_at: string;
  updated_at: string;
  last_accessed: string;
  access_count: number;
  is_active: boolean;
}

// 토큰에서 사용자 정보 추출
function getUserFromToken(request: NextRequest) {
  const token = request.cookies.get('auth_token')?.value;
  if (!token) return null;
  
  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET || 'fallback-secret') as any;
    return decoded;
  } catch (error) {
    return null;
  }
}

// GET - 사용자 메모리 조회
export async function GET(request: NextRequest) {
  try {
    const user = getUserFromToken(request);
    if (!user) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const { searchParams } = new URL(request.url);
    const category = searchParams.get('category');
    const limit = parseInt(searchParams.get('limit') || '50');
    const offset = parseInt(searchParams.get('offset') || '0');

    let query = `
      SELECT * FROM user_memories 
      WHERE user_id = $1 AND is_active = true
    `;
    const params: any[] = [user.username];

    if (category && category !== 'all') {
      query += ` AND category = $${params.length + 1}`;
      params.push(category);
    }

    query += ` ORDER BY importance_level DESC, last_accessed DESC LIMIT $${params.length + 1} OFFSET $${params.length + 2}`;
    params.push(limit, offset);

    const result = await pg.query(query, params);

    // 접근 횟수 업데이트 (조회할 때마다)
    if (result.rows.length > 0) {
      const memoryIds = result.rows.map(row => row.id);
      await pg.query(`
        UPDATE user_memories 
        SET last_accessed = CURRENT_TIMESTAMP, access_count = access_count + 1
        WHERE id = ANY($1)
      `, [memoryIds]);
    }

    return NextResponse.json({
      success: true,
      memories: result.rows,
      total: result.rows.length
    });

  } catch (error) {
    console.error('메모리 조회 오류:', error);
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 });
  }
}

// POST - 새 메모리 생성
export async function POST(request: NextRequest) {
  try {
    const user = getUserFromToken(request);
    if (!user) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const body = await request.json();
    const { title, content, category = 'general', importance_level = 1, tags = [] } = body;

    if (!title || !content) {
      return NextResponse.json({ error: 'Title and content are required' }, { status: 400 });
    }

    const result = await pg.query(`
      INSERT INTO user_memories (user_id, title, content, category, importance_level, tags)
      VALUES ($1, $2, $3, $4, $5, $6)
      RETURNING *
    `, [user.username, title, content, category, importance_level, tags]);

    return NextResponse.json({
      success: true,
      memory: result.rows[0]
    });

  } catch (error) {
    console.error('메모리 생성 오류:', error);
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 });
  }
}

// PUT - 메모리 업데이트
export async function PUT(request: NextRequest) {
  try {
    const user = getUserFromToken(request);
    if (!user) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const body = await request.json();
    const { id, title, content, category, importance_level, tags } = body;

    if (!id) {
      return NextResponse.json({ error: 'Memory ID is required' }, { status: 400 });
    }

    const result = await pg.query(`
      UPDATE user_memories 
      SET title = $1, content = $2, category = $3, importance_level = $4, tags = $5, updated_at = CURRENT_TIMESTAMP
      WHERE id = $6 AND user_id = $7
      RETURNING *
    `, [title, content, category, importance_level, tags, id, user.username]);

    if (result.rows.length === 0) {
      return NextResponse.json({ error: 'Memory not found or unauthorized' }, { status: 404 });
    }

    return NextResponse.json({
      success: true,
      memory: result.rows[0]
    });

  } catch (error) {
    console.error('메모리 업데이트 오류:', error);
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 });
  }
}

// DELETE - 메모리 삭제 (소프트 삭제)
export async function DELETE(request: NextRequest) {
  try {
    const user = getUserFromToken(request);
    if (!user) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const { searchParams } = new URL(request.url);
    const id = searchParams.get('id');

    if (!id) {
      return NextResponse.json({ error: 'Memory ID is required' }, { status: 400 });
    }

    const result = await pg.query(`
      UPDATE user_memories 
      SET is_active = false, updated_at = CURRENT_TIMESTAMP
      WHERE id = $1 AND user_id = $2
      RETURNING id
    `, [id, user.username]);

    if (result.rows.length === 0) {
      return NextResponse.json({ error: 'Memory not found or unauthorized' }, { status: 404 });
    }

    return NextResponse.json({
      success: true,
      message: 'Memory deleted successfully'
    });

  } catch (error) {
    console.error('메모리 삭제 오류:', error);
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 });
  }
} 