import { NextRequest, NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';
import os from 'os';

interface LogEntry {
  timestamp: string;
  level: string;
  message: string;
  source: string;
  details?: any;
  userId?: string;
  lineNumber?: number;
  fileName?: string;
}

interface LogFile {
  name: string;
  path: string;
  size: number;
  lastModified: string;
  lines: number;
}

function getAppDataPath() {
  switch (process.platform) {
    case 'win32':
      return path.join(process.env.APPDATA || '', 'airun');
    case 'darwin':
      return path.join(os.homedir(), 'Library', 'Application Support', 'airun');
    default:
      return path.join(os.homedir(), '.airun');
  }
}

async function getLogFiles(): Promise<LogFile[]> {
  const appDataDir = getAppDataPath();
  const logDir = path.join(appDataDir, 'logs');
  
  if (!fs.existsSync(logDir)) {
    return [];
  }

  const files = await fs.promises.readdir(logDir);
  const logFiles: LogFile[] = [];

  for (const file of files) {
    // 특정 파일들 제외
    if (file.includes('report_generator') || file === 'license.log') {
      continue;
    }
    
    // 로그 파일만 필터링 (.log 확장자 또는 .json)
    if (file.endsWith('.log') || file.endsWith('.json')) {
      try {
        const filePath = path.join(logDir, file);
        const stats = await fs.promises.stat(filePath);
        
        // 파일 크기가 너무 큰 경우 스킵 (100MB 이상)
        if (stats.size > 100 * 1024 * 1024) {
          continue;
        }

        // 라인 수 계산 (텍스트 파일인 경우)
        let lines = 0;
        if (file.endsWith('.log')) {
          try {
            const content = await fs.promises.readFile(filePath, 'utf8');
            lines = content.split('\n').length;
          } catch {
            lines = 0;
          }
        }

        logFiles.push({
          name: file,
          path: filePath,
          size: stats.size,
          lastModified: stats.mtime.toISOString(),
          lines
        });
      } catch (error) {
        console.error(`Error reading log file ${file}:`, error);
      }
    }
  }

  // 최근 수정된 순으로 정렬
  return logFiles.sort((a, b) => new Date(b.lastModified).getTime() - new Date(a.lastModified).getTime());
}

async function parseLogFile(filePath: string, fileName: string, fromEnd: number = 0, pageSize: number = 100): Promise<{ logs: LogEntry[], totalLines: number, hasMore: boolean }> {
  try {
    if (!fs.existsSync(filePath)) {
      return { logs: [], totalLines: 0, hasMore: false };
    }

    const fileContent = await fs.promises.readFile(filePath, 'utf8');
    const lines = fileContent.split('\n');
    const totalLines = lines.length;
    
    // 역방향 페이지네이션 (파일 끝에서부터)
    const endIndex = totalLines - fromEnd;
    const startIndex = Math.max(0, endIndex - pageSize);
    const pageLines = lines.slice(startIndex, endIndex);

    const logEntries: LogEntry[] = [];

    for (let i = 0; i < pageLines.length; i++) {
      const line = pageLines[i];
      
      // 원본 라인을 그대로 저장 (가공 없음)
      logEntries.push({
        timestamp: new Date().toISOString(),
        level: 'INFO',
        message: line,
        source: fileName,
        lineNumber: startIndex + i + 1,
        fileName
      });
    }

    return {
      logs: logEntries,
      totalLines,
      hasMore: startIndex > 0
    };
  } catch (error) {
    console.error(`Error parsing log file ${fileName}:`, error);
    return { logs: [], totalLines: 0, hasMore: false };
  }
}

// GET 요청 핸들러
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const level = searchParams.get('level') || 'all';
    const fileName = searchParams.get('file');
    const page = parseInt(searchParams.get('page') || '1');
    const pageSize = parseInt(searchParams.get('pageSize') || '50');
    const action = searchParams.get('action');

    // 로그 파일 목록 요청
    if (action === 'list') {
      const logFiles = await getLogFiles();
      return NextResponse.json({ 
        success: true, 
        data: logFiles 
      });
    }

    // 특정 파일의 로그 내용 요청
    if (fileName) {
      const appDataDir = getAppDataPath();
      const logDir = path.join(appDataDir, 'logs');
      const filePath = path.join(logDir, fileName);
      
      // 보안 검사: 디렉토리 트래버설 방지
      if (!filePath.startsWith(logDir)) {
        return NextResponse.json({ 
          success: false, 
          error: { message: '잘못된 파일 경로입니다.' } 
        }, { status: 400 });
      }

      const fromEnd = parseInt(searchParams.get('fromEnd') || '0');
      const result = await parseLogFile(filePath, fileName, fromEnd, pageSize);
      
      // 레벨 필터링
      const filteredLogs = level === 'all' 
        ? result.logs 
        : result.logs.filter(log => log.level.toLowerCase() === level.toLowerCase());

      return NextResponse.json({ 
        success: true, 
        data: filteredLogs,
        pagination: {
          fromEnd,
          pageSize,
          hasMore: result.hasMore,
          totalLines: result.totalLines
        }
      });
    }

    // 기본: 모든 로그 파일에서 최신 로그 가져오기
    const logFiles = await getLogFiles();
    const allLogs: LogEntry[] = [];

    // 최근 5개 파일에서만 로그 가져오기 (성능 최적화)
    const recentFiles = logFiles.slice(0, 5);
    
    for (const file of recentFiles) {
      const result = await parseLogFile(file.path, file.name, 0, 20);
      allLogs.push(...result.logs);
    }

    // 타임스탬프 기준으로 정렬
    allLogs.sort((a, b) => new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime());

    // 레벨 필터링
    const filteredLogs = level === 'all' 
      ? allLogs 
      : allLogs.filter(log => log.level.toLowerCase() === level.toLowerCase());

    return NextResponse.json({ 
      success: true, 
      data: filteredLogs.slice(0, 100), // 최대 100개만 반환
      logFiles: logFiles.map(f => ({ name: f.name, size: f.size, lastModified: f.lastModified, lines: f.lines }))
    });

  } catch (error) {
    console.error('로그 조회 중 오류:', error);
    return NextResponse.json({ 
      success: false, 
      error: { message: '로그를 불러오는 중 오류가 발생했습니다.' } 
    }, { status: 500 });
  }
}

// DELETE 요청 핸들러 (로그 삭제)
export async function DELETE(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const fileName = searchParams.get('file');

    if (fileName) {
      // 특정 파일 삭제
      const appDataDir = getAppDataPath();
      const logDir = path.join(appDataDir, 'logs');
      const filePath = path.join(logDir, fileName);
      
      // 보안 검사
      if (!filePath.startsWith(logDir)) {
        return NextResponse.json({ 
          success: false, 
          error: { message: '잘못된 파일 경로입니다.' } 
        }, { status: 400 });
      }

      if (fs.existsSync(filePath)) {
        await fs.promises.unlink(filePath);
      }
    } else {
      // 모든 로그 파일 삭제
      const logFiles = await getLogFiles();
      for (const file of logFiles) {
        try {
          await fs.promises.unlink(file.path);
        } catch (error) {
          console.error(`Error deleting ${file.name}:`, error);
        }
      }
    }

    return NextResponse.json({ success: true });
  } catch (error) {
    console.error('로그 삭제 중 오류:', error);
    return NextResponse.json({ 
      success: false, 
      error: { message: '로그 삭제 중 오류가 발생했습니다.' } 
    }, { status: 500 });
  }
} 