import { NextResponse, NextRequest } from 'next/server';
import fs from 'fs';
import path from 'path';
import os from 'os';
import { getUserFromRequest } from '../../auth/utils';

// 앱 데이터 경로를 반환하는 함수 (기존 모니터링 API들과 동일한 로직)
function getAppDataPath(): string {
  switch (process.platform) {
    case 'win32':
      return path.join(process.env.APPDATA || '', 'airun');
    case 'darwin':
      return path.join(os.homedir(), 'Library', 'Application Support', 'airun');
    default:
      return path.join(os.homedir(), '.airun');
  }
}

// API 모니터 파일 경로를 반환하는 함수 (기존 api/route.ts에서 사용하는 패턴)
function getApiMonitorFilePath(): string {
  return path.join(os.homedir(), '.airun', 'logs', 'api_monitor.json');
}

// 관리자 권한 확인 함수
async function checkAdminPermission(request: NextRequest): Promise<boolean> {
  try {
    const user = await getUserFromRequest(request);
    return user?.role === 'admin';
  } catch (error) {
    console.error('권한 확인 중 오류:', error);
    return false;
  }
}

export async function DELETE(request: NextRequest) {
  try {
    // 관리자 권한 확인
    const isAdmin = await checkAdminPermission(request);
    if (!isAdmin) {
      return NextResponse.json(
        { success: false, error: { message: '관리자 권한이 필요합니다.' } },
        { status: 403 }
      );
    }

    const appDataDir = getAppDataPath();
    const apiMonitorFile = getApiMonitorFilePath();
    
    console.log('🗑️ 모니터링 데이터 초기화 시작');
    console.log('📁 앱 데이터 디렉토리:', appDataDir);
    console.log('📄 API 모니터 파일:', apiMonitorFile);
    
    // 삭제할 파일/디렉토리 목록 (실제 모니터링 API들이 사용하는 파일명들)
    const itemsToDelete = [
      // 사용량 데이터 파일 (usage API에서 사용)
      path.join(appDataDir, 'usage.json'),
      
      // 실행 로그 파일 (logs API에서 사용)
      path.join(appDataDir, 'execution_logs.json'),
      
      // 시스템 로그 파일 (system API에서 사용)
      path.join(appDataDir, 'logs.json'),
      
      // API 모니터링 데이터 파일 (api API에서 사용)
      apiMonitorFile,
      
      // 기타 로그 파일들 (존재할 수 있는 추가 파일들)
      path.join(appDataDir, 'system.log'),
      path.join(appDataDir, 'app.log'),
      path.join(appDataDir, 'error.log'),
      path.join(appDataDir, 'debug.log'),
      path.join(appDataDir, 'monitoring.json'),
      path.join(appDataDir, 'sessions.json'),
      path.join(appDataDir, 'performance.json'),
    ];

    // 아카이브 디렉토리 처리
    const archiveDir = path.join(appDataDir, 'archives');
    
    let deletedFiles = 0;
    let errors: string[] = [];

    // 먼저 존재하는 파일들 확인
    console.log('🔍 존재하는 파일들 확인:');
    const existingFiles = itemsToDelete.filter(filePath => {
      const exists = fs.existsSync(filePath);
      console.log(`${exists ? '✓' : '✗'} ${filePath}`);
      return exists;
    });
    console.log(`📊 총 ${itemsToDelete.length}개 중 ${existingFiles.length}개 파일 존재`);

    // 개별 파일 삭제
    for (const filePath of itemsToDelete) {
      try {
        if (fs.existsSync(filePath)) {
          await fs.promises.unlink(filePath);
          deletedFiles++;
          console.log(`✅ 삭제됨: ${filePath}`);
        } else {
          console.log(`⏭️ 파일 없음 (건너뜀): ${filePath}`);
        }
      } catch (error) {
        const errorMsg = `파일 삭제 실패 ${filePath}: ${error}`;
        console.error(`❌ ${errorMsg}`);
        errors.push(errorMsg);
      }
    }

    // 아카이브 디렉토리 처리
    console.log('🗂️ 아카이브 디렉토리 처리:', archiveDir);
    try {
      if (fs.existsSync(archiveDir)) {
        const archiveFiles = await fs.promises.readdir(archiveDir);
        console.log(`📁 아카이브 디렉토리에서 ${archiveFiles.length}개 파일 발견`);
        
        for (const file of archiveFiles) {
          const archiveFilePath = path.join(archiveDir, file);
          try {
            const stat = await fs.promises.stat(archiveFilePath);
            if (stat.isFile()) {
              await fs.promises.unlink(archiveFilePath);
              deletedFiles++;
              console.log(`✅ 아카이브 파일 삭제됨: ${archiveFilePath}`);
            } else {
              console.log(`⏭️ 파일이 아님 (건너뜀): ${archiveFilePath}`);
            }
          } catch (error) {
            const errorMsg = `아카이브 파일 삭제 실패 ${archiveFilePath}: ${error}`;
            console.error(`❌ ${errorMsg}`);
            errors.push(errorMsg);
          }
        }
        
        // 아카이브 디렉토리가 비어있으면 삭제
        try {
          const remainingFiles = await fs.promises.readdir(archiveDir);
          if (remainingFiles.length === 0) {
            await fs.promises.rmdir(archiveDir);
            console.log(`✅ 아카이브 디렉토리 삭제됨: ${archiveDir}`);
          } else {
            console.log(`📁 아카이브 디렉토리에 ${remainingFiles.length}개 파일 남음`);
          }
        } catch (error) {
          const errorMsg = `아카이브 디렉토리 삭제 실패 ${archiveDir}: ${error}`;
          console.error(`❌ ${errorMsg}`);
          errors.push(errorMsg);
        }
      } else {
        console.log(`⏭️ 아카이브 디렉토리 없음: ${archiveDir}`);
      }
    } catch (error) {
      const errorMsg = `아카이브 디렉토리 처리 실패: ${error}`;
      console.error(`❌ ${errorMsg}`);
      errors.push(errorMsg);
    }

    // 로그 디렉토리 내 추가 정리 (패턴 기반)
    console.log('🔍 패턴 기반 추가 파일 정리:', appDataDir);
    try {
      if (fs.existsSync(appDataDir)) {
        const allFiles = await fs.promises.readdir(appDataDir);
        console.log(`📁 메인 디렉토리에서 ${allFiles.length}개 파일 발견`);
        
        const logPatterns = [
          /^usage_\d{4}-\d{2}-\d{2}\.json$/,  // usage_YYYY-MM-DD.json (사용량 아카이브)
          /^logs_\d{4}-\d{2}-\d{2}\.json$/,   // logs_YYYY-MM-DD.json (로그 아카이브)
          /^api_monitor_\d{4}-\d{2}-\d{2}\.json$/,  // api_monitor_YYYY-MM-DD.json
          /^execution_logs_\d{4}-\d{2}-\d{2}\.json$/,  // execution_logs_YYYY-MM-DD.json
          /^system_\d{4}-\d{2}-\d{2}\.log$/,  // system_YYYY-MM-DD.log
          /.*\.tmp$/,  // 임시 파일들
          /.*\.bak$/,  // 백업 파일들
          /.*\.lock$/,  // 락 파일들
        ];
        
        let patternMatched = 0;
        for (const file of allFiles) {
          const filePath = path.join(appDataDir, file);
          try {
            const stat = await fs.promises.stat(filePath);
            if (stat.isFile()) {
              const shouldDelete = logPatterns.some(pattern => pattern.test(file));
              if (shouldDelete) {
                await fs.promises.unlink(filePath);
                deletedFiles++;
                patternMatched++;
                console.log(`✅ 패턴 매치 파일 삭제됨: ${filePath}`);
              } else {
                console.log(`⏭️ 패턴 불일치 (건너뜀): ${file}`);
              }
            } else {
              console.log(`⏭️ 디렉토리 (건너뜀): ${file}`);
            }
          } catch (error) {
            const errorMsg = `패턴 매치 파일 삭제 실패 ${filePath}: ${error}`;
            console.error(`❌ ${errorMsg}`);
            errors.push(errorMsg);
          }
        }
        console.log(`🎯 패턴 매치로 ${patternMatched}개 파일 삭제됨`);
      } else {
        console.log(`⏭️ 메인 디렉토리 없음: ${appDataDir}`);
      }
    } catch (error) {
      const errorMsg = `로그 디렉토리 패턴 정리 실패: ${error}`;
      console.error(`❌ ${errorMsg}`);
      errors.push(errorMsg);
    }

    // 결과 반환
    const hasErrors = errors.length > 0;
    const message = hasErrors 
      ? `부분적으로 완료됨: ${deletedFiles}개 파일 삭제, ${errors.length}개 오류`
      : `완료됨: ${deletedFiles}개 파일이 삭제되었습니다.`;

    console.log('📊 모니터링 데이터 초기화 결과:');
    console.log(`✅ 삭제된 파일: ${deletedFiles}개`);
    console.log(`❌ 오류 발생: ${errors.length}개`);
    if (hasErrors) {
      console.log('🔍 오류 상세:');
      errors.forEach((error, index) => {
        console.log(`  ${index + 1}. ${error}`);
      });
    }
    console.log(`🎯 전체 결과: ${hasErrors ? '부분 성공' : '완전 성공'}`);

    return NextResponse.json({
      success: !hasErrors,
      message,
      data: {
        deletedFiles,
        errors: hasErrors ? errors : undefined
      }
    });

  } catch (error) {
    console.error('모니터링 데이터 초기화 중 오류:', error);
    return NextResponse.json(
      { 
        success: false, 
        error: { 
          message: error instanceof Error ? error.message : '모니터링 데이터 초기화 중 오류가 발생했습니다.' 
        } 
      },
      { status: 500 }
    );
  }
}

// 다른 메서드는 허용하지 않음
export async function GET() {
  return NextResponse.json(
    { success: false, error: { message: 'Method not allowed' } },
    { status: 405 }
  );
}

export async function POST() {
  return NextResponse.json(
    { success: false, error: { message: 'Method not allowed' } },
    { status: 405 }
  );
}

export async function PUT() {
  return NextResponse.json(
    { success: false, error: { message: 'Method not allowed' } },
    { status: 405 }
  );
} 