'use client';

import { usePathname, useRouter } from 'next/navigation';
import { 
  BarChart3, 
  FileText, 
  Target, 
  Award,
  Building2,
  Home
} from 'lucide-react';

interface BusinessLayoutProps {
  children: React.ReactNode;
}

export default function BusinessLayout({ children }: BusinessLayoutProps) {
  const pathname = usePathname();
  const router = useRouter();

  const menuItems = [
    {
      id: 'home',
      title: '사업분석',
      description: '사업공고 분석 개요',
      icon: Home,
      path: '/business',
      color: 'text-gray-600'
    },
    {
      id: 'dashboard',
      title: '분석 대시보드',
      description: '상세한 통계와 분석 결과',
      icon: BarChart3,
      path: '/business/dashboard',
      color: 'text-orange-600'
    },
    {
      id: 'announcements',
      title: '사업공고',
      description: '최신 사업공고 정보',
      icon: FileText,
      path: '/business/announcements',
      color: 'text-blue-600'
    },
    {
      id: 'matching',
      title: '기업 매칭',
      description: 'AI 기반 적합성 분석',
      icon: Target,
      path: '/business/matching',
      color: 'text-green-600'
    },
    {
      id: 'proposals',
      title: '제안서 관리',
      description: 'AI 기반 맞춤형 제안서',
      icon: Award,
      path: '/business/proposals',
      color: 'text-purple-600'
    }
  ];

  const getCurrentPageInfo = () => {
    // 정확한 경로 매칭을 위해 가장 긴 경로부터 확인
    const sortedItems = [...menuItems].sort((a, b) => b.path.length - a.path.length);
    const currentItem = sortedItems.find(item => {
      if (item.path === '/business') {
        return pathname === '/business';
      }
      return pathname.startsWith(item.path);
    });
    
    return currentItem || {
      title: '사업분석',
      description: 'AI 기반 사업분석 플랫폼',
      icon: Building2,
      color: 'text-gray-600'
    };
  };

  const currentPage = getCurrentPageInfo();

  return (
    <div className="h-screen flex overflow-hidden" style={{backgroundColor: 'var(--body-bg)'}}>
      {/* Sidebar */}
      <div 
        className="w-64 flex-shrink-0"
        style={{backgroundColor: 'var(--sidebar-bg)', borderRight: '1px solid var(--sidebar-border)'}}
      >

        {/* Navigation Menu */}
        <nav className="flex-1 overflow-y-auto p-4">
          {/* Menu Items */}
          <div className="space-y-2">
            {menuItems.map((item) => {
              const Icon = item.icon;
              const isActive = item.path === '/business' 
                ? pathname === '/business'
                : pathname.startsWith(item.path);
              
              return (
                <div
                  key={item.id}
                  className={`p-3 mx-2 rounded-lg cursor-pointer transition-all duration-200 border ${
                    isActive
                      ? 'bg-blue-50 dark:bg-blue-900/30 border-blue-200 dark:border-blue-700'
                      : 'hover:bg-gray-50 dark:hover:bg-gray-700/50 border-transparent'
                  }`}
                  onClick={() => router.push(item.path)}
                >
                  <div className="flex items-center space-x-3">
                    <Icon className={`w-5 h-5 ${item.color}`} />
                    <div className="flex-1">
                      <h3 className="text-sm font-medium text-gray-900 dark:text-gray-100">
                        {item.title}
                      </h3>
                      <p className="text-xs text-gray-600 dark:text-gray-400 mt-1">
                        {item.description}
                      </p>
                    </div>
                    {isActive && (
                      <div className="w-2 h-2 bg-blue-600 rounded-full"></div>
                    )}
                  </div>
                </div>
              );
            })}
          </div>

          {/* Quick Stats */}
          <div className="mt-6 card p-4" style={{backgroundColor: 'var(--card-bg)', borderColor: 'var(--card-border)'}}>
            <h3 className="text-base font-semibold mb-4" style={{color: 'var(--text-primary)'}}>
              빠른 통계
            </h3>
            <div className="space-y-3">
              <div className="flex justify-between items-center">
                <span className="text-base" style={{color: 'var(--text-secondary)'}}>활성 공고</span>
                <span className="text-base font-medium text-blue-600">892건</span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-base" style={{color: 'var(--text-secondary)'}}>매칭 점수</span>
                <span className="text-base font-medium text-green-600">78.3점</span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-base" style={{color: 'var(--text-secondary)'}}>생성 제안서</span>
                <span className="text-base font-medium text-purple-600">145건</span>
              </div>
            </div>
          </div>
        </nav>
      </div>

      {/* Main Content */}
      <div className="flex-1 flex flex-col min-w-0">
        {/* Top Header */}
        <div className="h-16 flex items-center justify-between px-6" style={{backgroundColor: 'var(--header-bg)', borderBottom: '1px solid var(--card-border)'}}>
          <div className="flex items-center space-x-4">
            <currentPage.icon className={`w-6 h-6 ${currentPage.color}`} />
            <div>
              <h1 className="text-base font-semibold" style={{color: 'var(--text-primary)'}}>
                {currentPage.title}
              </h1>
              <p className="text-sm" style={{color: 'var(--text-secondary)'}}>
                {currentPage.description}
              </p>
            </div>
          </div>
          
          {/* Status Indicator */}
          <div className="flex items-center space-x-2">
            <div className="flex items-center space-x-2 text-xs" style={{color: 'var(--text-secondary)'}}>
              <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
              <span>시스템 정상</span>
            </div>
          </div>
        </div>

        {/* Page Content */}
        <div className="flex-1 overflow-auto" style={{backgroundColor: 'var(--background)'}}>
          {children}
        </div>
      </div>
    </div>
  );
}