'use client';

import { useState } from 'react';
import { useRouter } from 'next/navigation';

interface Template {
  id: string;
  name: string;
  description: string;
  formats: string[];
  icon: string;
}

export default function NewDocumentPage() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  const [selectedTemplate, setSelectedTemplate] = useState<string | null>(null);
  const [prompt, setPrompt] = useState('');
  const [format, setFormat] = useState<'PDF' | 'HWPX' | 'DOCX' | 'PPTX'>('PDF');
  const [email, setEmail] = useState('');
  const [name, setName] = useState('');

  // 문서 템플릿 목록
  const templates: Template[] = [
    {
      id: 'report',
      name: '보고서',
      description: '분석 데이터와 함께 구조화된 보고서를 작성합니다.',
      formats: ['PDF', 'HWPX', 'DOCX'],
      icon: '📊'
    },
    {
      id: 'proposal',
      name: '제안서',
      description: '아이디어나 프로젝트 제안을 위한 문서를 작성합니다.',
      formats: ['PDF', 'PPTX', 'DOCX'],
      icon: '💼'
    },
    {
      id: 'manual',
      name: '매뉴얼',
      description: '제품이나 서비스 사용법을 설명하는 문서를 작성합니다.',
      formats: ['PDF', 'HWPX', 'DOCX'],
      icon: '📖'
    },
    {
      id: 'presentation',
      name: '프레젠테이션',
      description: '발표를 위한 슬라이드 자료를 작성합니다.',
      formats: ['PPTX'],
      icon: '🎯'
    },
    {
      id: 'article',
      name: '기사/블로그',
      description: '뉴스 기사나 블로그 포스트 형식의 문서를 작성합니다.',
      formats: ['PDF', 'DOCX'],
      icon: '📰'
    },
    {
      id: 'custom',
      name: '사용자 정의',
      description: '특정 형식 없이 자유롭게 문서를 작성합니다.',
      formats: ['PDF', 'HWPX', 'DOCX', 'PPTX'],
      icon: '✏️'
    }
  ];

  // 선택한 템플릿 정보
  const selectedTemplateInfo = templates.find(t => t.id === selectedTemplate);

  // 문서 생성 요청 처리
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!selectedTemplate) {
      alert('문서 템플릿을 선택해주세요.');
      return;
    }
    
    if (!prompt) {
      alert('문서 내용을 입력해주세요.');
      return;
    }
    
    if (!email || !email.includes('@')) {
      alert('유효한 이메일 주소를 입력해주세요.');
      return;
    }
    
    try {
      setLoading(true);
      
      // API 호출 시뮬레이션
      await new Promise(resolve => setTimeout(resolve, 2000));
      
      // 성공 시 문서 목록 페이지로 이동
      router.push('/document/list');
    } catch (error) {
      console.error('문서 생성 오류:', error);
      alert('문서 생성 중 오류가 발생했습니다.');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="container mx-auto px-4 py-8">
      <h1 className="text-3xl font-bold mb-8 text-center">새 문서 작성</h1>
      
      <div className="max-w-4xl mx-auto">
        <form onSubmit={handleSubmit} className="space-y-8">
          {/* 템플릿 선택 */}
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow p-6">
            <h2 className="text-xl font-semibold mb-4">1. 문서 템플릿 선택</h2>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              {templates.map((template) => (
                <div
                  key={template.id}
                  className={`border rounded-lg p-4 cursor-pointer transition-all ${
                    selectedTemplate === template.id
                      ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20'
                      : 'border-gray-200 dark:border-gray-700 hover:border-blue-300 dark:hover:border-blue-700'
                  }`}
                  onClick={() => {
                    setSelectedTemplate(template.id);
                    
                    // 템플릿에 PPTX만 있고 현재 선택된 포맷이 PPTX가 아니면 포맷 변경
                    if (template.formats.length === 1 && template.formats[0] === 'PPTX' && format !== 'PPTX') {
                      setFormat('PPTX');
                    }
                    // 템플릿에 현재 선택된 포맷이 없으면 첫 번째 포맷으로 변경
                    else if (!template.formats.includes(format)) {
                      setFormat(template.formats[0] as any);
                    }
                  }}
                >
                  <div className="flex items-center justify-between mb-2">
                    <span className="text-2xl">{template.icon}</span>
                    <span className={`h-4 w-4 rounded-full ${
                      selectedTemplate === template.id ? 'bg-blue-500' : 'bg-gray-200 dark:bg-gray-600'
                    }`}></span>
                  </div>
                  <h3 className="font-medium mb-1">{template.name}</h3>
                  <p className="text-sm text-gray-600 dark:text-gray-400">
                    {template.description}
                  </p>
                </div>
              ))}
            </div>
          </div>
          
          {/* 문서 내용 입력 */}
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow p-6">
            <h2 className="text-xl font-semibold mb-4">2. 문서 내용 작성</h2>
            <div className="mb-4">
              <label htmlFor="prompt" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                내용 설명
              </label>
              <textarea
                id="prompt"
                rows={8}
                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                placeholder="생성하고자 하는 문서 내용을 자세히 설명해주세요..."
                value={prompt}
                onChange={(e) => setPrompt(e.target.value)}
                required
              />
            </div>
          </div>
          
          {/* 출력 형식 및 사용자 정보 */}
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow p-6">
            <h2 className="text-xl font-semibold mb-4">3. 출력 형식 및 사용자 정보</h2>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  출력 형식
                </label>
                <div className="flex flex-wrap gap-3">
                  {['PDF', 'HWPX', 'DOCX', 'PPTX'].map((fmt) => {
                    const isDisabled = selectedTemplateInfo && !selectedTemplateInfo.formats.includes(fmt);
                    return (
                      <button
                        key={fmt}
                        type="button"
                        className={`px-4 py-2 rounded-md text-sm font-medium transition-colors ${
                          format === fmt
                            ? 'bg-blue-500 text-white'
                            : isDisabled
                              ? 'bg-gray-100 text-gray-400 dark:bg-gray-700 dark:text-gray-500 cursor-not-allowed'
                              : 'bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-300 dark:hover:bg-gray-600'
                        }`}
                        onClick={() => !isDisabled && setFormat(fmt as any)}
                        disabled={isDisabled}
                      >
                        {fmt}
                      </button>
                    );
                  })}
                </div>
                {selectedTemplateInfo && (
                  <p className="mt-2 text-sm text-gray-500 dark:text-gray-400">
                    {selectedTemplateInfo.name} 템플릿은 {selectedTemplateInfo.formats.join(', ')} 형식을 지원합니다.
                  </p>
                )}
              </div>
              
              <div>
                <div className="mb-4">
                  <label htmlFor="email" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                    이메일 주소
                  </label>
                  <input
                    type="email"
                    id="email"
                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                    placeholder="문서를 받을 이메일 주소"
                    value={email}
                    onChange={(e) => setEmail(e.target.value)}
                    required
                  />
                </div>
                
                <div>
                  <label htmlFor="name" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                    이름
                  </label>
                  <input
                    type="text"
                    id="name"
                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white"
                    placeholder="이름 (선택사항)"
                    value={name}
                    onChange={(e) => setName(e.target.value)}
                  />
                </div>
              </div>
            </div>
          </div>
          
          {/* 제출 버튼 */}
          <div className="flex justify-end">
            <button
              type="button"
              onClick={() => router.back()}
              className="mr-4 px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-700"
            >
              취소
            </button>
            <button
              type="submit"
              disabled={loading || !selectedTemplate || !prompt || !email}
              className="px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {loading ? (
                <span className="flex items-center">
                  <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                    <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                    <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                  </svg>
                  문서 생성 중...
                </span>
              ) : (
                '문서 생성'
              )}
            </button>
          </div>
          
          {/* 알림 */}
          <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-md p-4">
            <div className="flex">
              <div className="flex-shrink-0">
                <svg className="h-5 w-5 text-blue-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                  <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clipRule="evenodd" />
                </svg>
              </div>
              <div className="ml-3 flex-1 md:flex md:justify-between">
                <p className="text-sm text-blue-700 dark:text-blue-300">
                  문서 생성이 완료되면 입력하신 이메일로 문서가 전송됩니다.
                </p>
              </div>
            </div>
          </div>
        </form>
      </div>
    </div>
  );
} 