"""
FLOP 기반 라이선스 관리 시스템

기존 라이선스 시스템을 확장하여 FLOP(Floating Point Operations) 단위로
AI 연산량을 측정하고 라이선스 비용을 산정하는 시스템입니다.
"""

import os
import sys
import json
import logging
from typing import Dict, Any, Optional, Tuple
from datetime import datetime, timedelta
from pathlib import Path
from dataclasses import asdict

# 현재 디렉토리를 Python 경로에 추가
current_dir = os.path.dirname(os.path.abspath(__file__))
parent_dir = os.path.dirname(current_dir)
if parent_dir not in sys.path:
    sys.path.append(parent_dir)

# 기존 라이선스 매니저와 FLOP 트래커 import
from license.license_manager import LicenseManager
from services.flop_tracker import FLOPUsageTracker as FLOPTracker, format_flops

class FLOPLicenseManager(LicenseManager):
    """FLOP 기반 라이선스 관리 클래스"""
    
    def __init__(self):
        super().__init__()
        self.flop_tracker = FLOPTracker()
        
        # FLOP 기반 라이선스 설정 파일
        self.flop_config_file = os.path.join(
            os.path.dirname(self.license_file), 
            'flop_license_config.json'
        )
        
        # FLOP 라이선스 설정 로드
        self.flop_config = self._load_flop_config()
        
        self.log_message('info', "FLOP 라이선스 매니저 초기화 완료")
    
    def log_message(self, level: str, message: str, data=None):
        """부모 클래스의 logger 메서드를 래핑합니다."""
        return super().logger(level, message, data)
    
    def _load_flop_config(self) -> Dict[str, Any]:
        """FLOP 라이선스 설정 로드"""
        try:
            if os.path.exists(self.flop_config_file):
                with open(self.flop_config_file, 'r', encoding='utf-8') as f:
                    return json.load(f)
            else:
                # 기본 설정 생성
                default_config = self._create_default_flop_config()
                self._save_flop_config(default_config)
                return default_config
        except Exception as e:
            self.log_message('error', f"FLOP 설정 로드 실패: {e}")
            return self._create_default_flop_config()
    
    def _create_default_flop_config(self) -> Dict[str, Any]:
        """기본 FLOP 라이선스 설정 생성"""
        return {
            'pricing_model': 'tiered',  # 'tiered' 또는 'pay_per_use'
            'billing_cycle': 'monthly',  # 'monthly', 'daily', 'yearly'
            'currency': 'USD',
            'tiers': {
                'basic': {
                    'name': '베이직',
                    'monthly_gflops_limit': 1000000,  # 1M GFLOPS/월
                    'cost_per_gflop': 0.001,          # $0.001/GFLOP
                    'overage_cost': 0.002,            # 초과 시 $0.002/GFLOP
                    'included_features': ['basic_support', 'standard_models']
                },
                'standard': {
                    'name': '스탠다드',
                    'monthly_gflops_limit': 10000000,  # 10M GFLOPS/월
                    'cost_per_gflop': 0.0008,          # $0.0008/GFLOP
                    'overage_cost': 0.0015,            # 초과 시 $0.0015/GFLOP
                    'included_features': ['priority_support', 'all_models', 'analytics']
                },
                'premium': {
                    'name': '프리미엄',
                    'monthly_gflops_limit': 100000000,  # 100M GFLOPS/월
                    'cost_per_gflop': 0.0005,           # $0.0005/GFLOP
                    'overage_cost': 0.001,              # 초과 시 $0.001/GFLOP
                    'included_features': ['24x7_support', 'custom_models', 'advanced_analytics', 'priority_queue']
                },
                'enterprise': {
                    'name': '엔터프라이즈',
                    'monthly_gflops_limit': 0,          # 무제한
                    'cost_per_gflop': 0.0003,           # $0.0003/GFLOP
                    'overage_cost': 0,                  # 초과 비용 없음
                    'included_features': ['dedicated_support', 'custom_deployment', 'white_label', 'sla_guarantee']
                }
            },
            'exchange_rates': {
                'KRW': 1300.0,  # 1 USD = 1300 KRW (예시)
                'EUR': 0.85,    # 1 USD = 0.85 EUR (예시)
                'JPY': 110.0    # 1 USD = 110 JPY (예시)
            },
            'created_at': datetime.now().isoformat(),
            'updated_at': datetime.now().isoformat()
        }
    
    def _save_flop_config(self, config: Dict[str, Any]):
        """FLOP 라이선스 설정 저장"""
        try:
            config['updated_at'] = datetime.now().isoformat()
            with open(self.flop_config_file, 'w', encoding='utf-8') as f:
                json.dump(config, f, ensure_ascii=False, indent=2)
        except Exception as e:
            self.log_message('error', f"FLOP 설정 저장 실패: {e}")
    
    def track_ai_usage(self, provider: str, model: str, prompt_tokens: int, 
                      completion_tokens: int, duration_ms: float = 0.0,
                      session_id: Optional[str] = None, user_id: Optional[str] = None,
                      username: Optional[str] = None, feature: Optional[str] = None) -> Optional[Dict[str, Any]]:
        """
        AI 사용량을 FLOP 단위로 추적
        
        Returns:
            추적된 사용량 정보 및 비용 계산 결과
        """
        try:
            # FLOP 사용량 추적
            usage = self.flop_tracker.track_usage(
                provider=provider,
                model=model,
                prompt_tokens=prompt_tokens,
                completion_tokens=completion_tokens,
                duration_ms=duration_ms,
                session_id=session_id,
                user_id=user_id,
                username=username,
                feature=feature
            )
            
            if not usage:
                return None
            
            # 현재 라이선스 정보 조회
            license_info = self.read_license()
            if not license_info:
                self.log_message('warning', "라이선스 정보가 없음")
                return None
            
            # 비용 계산
            cost_info = self._calculate_usage_cost(usage, license_info)
            
            # 라이선스 한도 확인
            limit_check = self._check_usage_limits(usage, license_info)
            
            result = {
                'usage': usage.to_dict(),
                'cost': cost_info,
                'limits': limit_check,
                'formatted_flops': format_flops(usage.estimated_flops)
            }
            
            self.log_message('debug', f"AI 사용량 추적 완료: {format_flops(usage.estimated_flops)}")
            return result
            
        except Exception as e:
            self.log_message('error', f"AI 사용량 추적 실패: {e}")
            return None
    
    def _calculate_usage_cost(self, usage, license_info: Dict[str, Any]) -> Dict[str, Any]:
        """사용량 기반 비용 계산"""
        try:
            # 라이선스에서 플랜 정보 가져오기
            plan_name = license_info.get('apiPlan', 'basic')
            tier_config = self.flop_config['tiers'].get(plan_name, self.flop_config['tiers']['basic'])
            
            # GFLOP 단위로 변환
            gflops_used = usage.estimated_flops / 1e9
            
            # 기본 비용 계산
            base_cost = gflops_used * tier_config['cost_per_gflop']
            
            # 월간 사용량 조회
            monthly_stats = self._get_monthly_flop_usage()
            monthly_gflops = monthly_stats['total_flops'] / 1e9
            
            # 한도 초과 여부 확인
            monthly_limit = tier_config['monthly_gflops_limit']
            overage_cost = 0.0
            
            if monthly_limit > 0 and monthly_gflops > monthly_limit:
                # 초과 사용량 계산
                overage_gflops = monthly_gflops - monthly_limit
                overage_cost = overage_gflops * tier_config['overage_cost']
            
            total_cost = base_cost + overage_cost
            
            # 환율 적용 (KRW)
            krw_rate = self.flop_config['exchange_rates'].get('KRW', 1300.0)
            cost_krw = total_cost * krw_rate
            
            return {
                'gflops_used': gflops_used,
                'base_cost_usd': base_cost,
                'overage_cost_usd': overage_cost,
                'total_cost_usd': total_cost,
                'total_cost_krw': cost_krw,
                'cost_per_gflop': tier_config['cost_per_gflop'],
                'currency': self.flop_config['currency'],
                'exchange_rate_krw': krw_rate
            }
            
        except Exception as e:
            self.log_message('error', f"비용 계산 실패: {e}")
            return {
                'gflops_used': 0.0,
                'base_cost_usd': 0.0,
                'overage_cost_usd': 0.0,
                'total_cost_usd': 0.0,
                'total_cost_krw': 0.0,
                'error': str(e)
            }
    
    def _check_usage_limits(self, usage, license_info: Dict[str, Any]) -> Dict[str, Any]:
        """사용량 한도 확인"""
        try:
            plan_name = license_info.get('apiPlan', 'basic')
            tier_config = self.flop_config['tiers'].get(plan_name, self.flop_config['tiers']['basic'])
            
            # 월간 사용량 조회
            monthly_stats = self._get_monthly_flop_usage()
            monthly_gflops = monthly_stats['total_flops'] / 1e9
            
            # 한도 정보
            monthly_limit = tier_config['monthly_gflops_limit']
            
            # 사용률 계산
            if monthly_limit > 0:
                usage_percentage = (monthly_gflops / monthly_limit) * 100
                remaining_gflops = max(0, monthly_limit - monthly_gflops)
            else:
                usage_percentage = 0  # 무제한 플랜
                remaining_gflops = float('inf')
            
            # 경고 레벨 설정
            warning_level = 'normal'
            if monthly_limit > 0:
                if usage_percentage >= 95:
                    warning_level = 'critical'
                elif usage_percentage >= 80:
                    warning_level = 'warning'
                elif usage_percentage >= 60:
                    warning_level = 'caution'
            
            return {
                'plan': plan_name,
                'monthly_limit_gflops': monthly_limit,
                'monthly_used_gflops': monthly_gflops,
                'remaining_gflops': remaining_gflops,
                'usage_percentage': usage_percentage,
                'warning_level': warning_level,
                'is_unlimited': monthly_limit == 0,
                'is_over_limit': monthly_limit > 0 and monthly_gflops > monthly_limit
            }
            
        except Exception as e:
            self.log_message('error', f"사용량 한도 확인 실패: {e}")
            return {
                'error': str(e),
                'warning_level': 'error'
            }
    
    def _get_monthly_flop_usage(self) -> Dict[str, Any]:
        """월간 FLOP 사용량 조회"""
        try:
            # 현재 월의 시작일과 종료일 계산
            now = datetime.now()
            start_of_month = now.replace(day=1, hour=0, minute=0, second=0, microsecond=0)
            
            # 다음 달 첫째 날 계산
            if now.month == 12:
                next_month = now.replace(year=now.year + 1, month=1, day=1)
            else:
                next_month = now.replace(month=now.month + 1, day=1)
            
            # 월간 통계 조회
            monthly_stats = self.flop_tracker.get_usage_stats(
                start_date=start_of_month.strftime('%Y-%m-%d'),
                end_date=next_month.strftime('%Y-%m-%d')
            )
            
            return monthly_stats
            
        except Exception as e:
            self.log_message('error', f"월간 사용량 조회 실패: {e}")
            return {
                'total_flops': 0.0,
                'total_tokens': 0,
                'total_requests': 0
            }
    
    def get_usage_report(self, period: str = 'monthly', 
                        start_date: Optional[str] = None,
                        end_date: Optional[str] = None) -> Dict[str, Any]:
        """사용량 리포트 생성"""
        try:
            # 기간 설정
            if period == 'monthly' and not start_date:
                now = datetime.now()
                start_date = now.replace(day=1).strftime('%Y-%m-%d')
            elif period == 'daily' and not start_date:
                start_date = datetime.now().strftime('%Y-%m-%d')
            
            # 통계 조회
            stats = self.flop_tracker.get_usage_stats(start_date, end_date)
            
            # 라이선스 정보 조회
            license_info = self.read_license()
            plan_name = license_info.get('apiPlan', 'basic') if license_info else 'basic'
            tier_config = self.flop_config['tiers'].get(plan_name, self.flop_config['tiers']['basic'])
            
            # 비용 계산
            total_gflops = stats['total_flops'] / 1e9
            total_cost = self._calculate_period_cost(total_gflops, tier_config)
            
            # 리포트 생성
            report = {
                'period': period,
                'start_date': start_date,
                'end_date': end_date,
                'license_plan': plan_name,
                'summary': {
                    'total_flops': stats['total_flops'],
                    'total_gflops': total_gflops,
                    'formatted_flops': format_flops(stats['total_flops']),
                    'total_tokens': stats['total_tokens'],
                    'total_requests': stats['total_requests'],
                    'total_cost_usd': total_cost['total_cost_usd'],
                    'total_cost_krw': total_cost['total_cost_krw']
                },
                'by_provider': {},
                'by_model': {},
                'by_user': stats.get('by_user', {}),
                'daily_breakdown': stats.get('daily_stats', {}),
                'tier_info': tier_config,
                'generated_at': datetime.now().isoformat()
            }
            
            # 프로바이더별 상세 정보
            for provider, provider_stats in stats.get('by_provider', {}).items():
                provider_gflops = provider_stats['flops'] / 1e9
                provider_cost = self._calculate_period_cost(provider_gflops, tier_config)
                
                report['by_provider'][provider] = {
                    'flops': provider_stats['flops'],
                    'gflops': provider_gflops,
                    'formatted_flops': format_flops(provider_stats['flops']),
                    'tokens': provider_stats['tokens'],
                    'requests': provider_stats['requests'],
                    'cost_usd': provider_cost['total_cost_usd'],
                    'cost_krw': provider_cost['total_cost_krw']
                }
            
            # 모델별 상세 정보
            for model, model_stats in stats.get('by_model', {}).items():
                model_gflops = model_stats['flops'] / 1e9
                model_cost = self._calculate_period_cost(model_gflops, tier_config)
                
                report['by_model'][model] = {
                    'flops': model_stats['flops'],
                    'gflops': model_gflops,
                    'formatted_flops': format_flops(model_stats['flops']),
                    'tokens': model_stats['tokens'],
                    'requests': model_stats['requests'],
                    'cost_usd': model_cost['total_cost_usd'],
                    'cost_krw': model_cost['total_cost_krw']
                }
            
            return report
            
        except Exception as e:
            self.log_message('error', f"사용량 리포트 생성 실패: {e}")
            return {
                'error': str(e),
                'generated_at': datetime.now().isoformat()
            }
    
    def _calculate_period_cost(self, gflops: float, tier_config: Dict[str, Any]) -> Dict[str, Any]:
        """기간별 비용 계산"""
        base_cost = gflops * tier_config['cost_per_gflop']
        krw_rate = self.flop_config['exchange_rates'].get('KRW', 1300.0)
        
        return {
            'total_cost_usd': base_cost,
            'total_cost_krw': base_cost * krw_rate
        }
    
    def update_pricing_config(self, new_config: Dict[str, Any]) -> bool:
        """가격 정책 업데이트"""
        try:
            # 기존 설정과 병합
            self.flop_config.update(new_config)
            
            # 설정 저장
            self._save_flop_config(self.flop_config)
            
            self.log_message('info', "FLOP 가격 정책 업데이트 완료")
            return True
            
        except Exception as e:
            self.log_message('error', f"가격 정책 업데이트 실패: {e}")
            return False
    
    def get_pricing_info(self) -> Dict[str, Any]:
        """현재 가격 정책 정보 반환"""
        return {
            'tiers': self.flop_config['tiers'],
            'pricing_model': self.flop_config['pricing_model'],
            'billing_cycle': self.flop_config['billing_cycle'],
            'currency': self.flop_config['currency'],
            'exchange_rates': self.flop_config['exchange_rates']
        }
    
    def estimate_monthly_cost(self, estimated_gflops: float, plan: str = 'basic') -> Dict[str, Any]:
        """월간 예상 비용 계산"""
        try:
            tier_config = self.flop_config['tiers'].get(plan, self.flop_config['tiers']['basic'])
            
            # 기본 비용 계산
            base_cost = estimated_gflops * tier_config['cost_per_gflop']
            
            # 한도 초과 비용 계산
            monthly_limit = tier_config['monthly_gflops_limit']
            overage_cost = 0.0
            
            if monthly_limit > 0 and estimated_gflops > monthly_limit:
                overage_gflops = estimated_gflops - monthly_limit
                overage_cost = overage_gflops * tier_config['overage_cost']
            
            total_cost = base_cost + overage_cost
            krw_rate = self.flop_config['exchange_rates'].get('KRW', 1300.0)
            
            return {
                'plan': plan,
                'estimated_gflops': estimated_gflops,
                'monthly_limit': monthly_limit,
                'base_cost_usd': base_cost,
                'overage_cost_usd': overage_cost,
                'total_cost_usd': total_cost,
                'total_cost_krw': total_cost * krw_rate,
                'is_over_limit': monthly_limit > 0 and estimated_gflops > monthly_limit,
                'savings_vs_basic': self._calculate_savings_vs_basic(estimated_gflops, plan)
            }
            
        except Exception as e:
            self.log_message('error', f"월간 비용 추정 실패: {e}")
            return {'error': str(e)}
    
    def _calculate_savings_vs_basic(self, gflops: float, plan: str) -> float:
        """베이직 플랜 대비 절약 금액 계산"""
        try:
            if plan == 'basic':
                return 0.0
            
            # 베이직 플랜 비용 계산
            basic_config = self.flop_config['tiers']['basic']
            basic_cost = gflops * basic_config['cost_per_gflop']
            
            # 현재 플랜 비용 계산
            current_estimate = self.estimate_monthly_cost(gflops, plan)
            current_cost = current_estimate.get('total_cost_usd', 0)
            
            return basic_cost - current_cost
            
        except Exception as e:
            self.log_message('error', f"절약 금액 계산 실패: {e}")
            return 0.0

    def _read_existing_license(self) -> Dict[str, Any]:
        """기존 라이선스 시스템에서 라이선스 정보 읽기"""
        try:
            # 실제 라이선스 매니저를 사용해서 라이선스 정보 읽기
            import sys
            import os
            
            # 라이선스 매니저 모듈 경로 추가
            license_dir = os.path.dirname(__file__)
            if license_dir not in sys.path:
                sys.path.insert(0, license_dir)
                
            try:
                # 라이선스 매니저 모듈 import
                import license_manager
                
                # 실제 라이선스 정보 읽기
                license_data = license_manager.license_manager.read_license()
                
                if license_data:
                    self.log_message('debug', f"실제 라이선스 정보 로드: {license_data.get('company', 'Unknown')}, 플랜: {license_data.get('apiPlan', 'Unknown')}")
                    
                    # FLOP 라이선스에 필요한 형태로 변환
                    return {
                        'valid': True,
                        'apiPlan': license_data.get('apiPlan', 'unlimited'),  # 실제 라이선스 플랜 사용
                        'apiLimit': license_data.get('apiLimit', 100000),
                        'apiUsage': license_data.get('apiUsage', {'used': 0}),
                        'resetDate': license_data.get('resetDate'),
                        'expiresAt': license_data.get('expiresAt'),
                        'company': license_data.get('company', 'Licensed User'),
                        'user': license_data.get('user', 'Licensed User')
                    }
                else:
                    self.log_message('debug', "라이선스 파일을 찾을 수 없음")
                    
            except ImportError as e:
                self.log_message('error', f"라이선스 매니저 모듈 import 실패: {e}")
            except Exception as e:
                self.log_message('error', f"라이선스 매니저 실행 실패: {e}")
            
            # 라이선스 파일이 없거나 읽기 실패 시 Trial 모드
            license_data = {
                'valid': False,
                'apiPlan': 'basic',
                'apiLimit': 1000,
                'apiUsage': {'used': 0},
                'resetDate': None,
                'expiresAt': None,
                'company': 'Trial User',
                'user': 'Trial User'
            }
            
            self.log_message('debug', f"라이선스 정보 로드 완료: {license_data}")
            return license_data
            
        except Exception as e:
            self.log_message('error', f"기존 라이선스 정보 읽기 실패: {e}")
            # 오류 발생 시 기본 Trial 정보 반환
            return {
                'valid': False,
                'apiPlan': 'basic',
                'apiLimit': 1000,
                'apiUsage': {'used': 0},
                'resetDate': None,
                'expiresAt': None,
                'company': 'Trial User',
                'user': 'Trial User'
            }

    def check_license_limits(self, user_id: Optional[str] = None, username: Optional[str] = None) -> bool:
        """FLOP 라이선스 한도 체크"""
        try:
            current_usage = self.get_current_usage(user_id, username)
            license_info = self.get_license_status(user_id, username)
            
            if not license_info.get('active', False):
                return False
                
            limit = license_info.get('flop_limit', float('inf'))
            used = current_usage.get('total_flops', 0)
            
            return used < limit
            
        except Exception as e:
            self.log_message('error', f"라이선스 한도 체크 실패: {e}")
            return True  # 체크 실패 시 허용
    
    def get_usage_summary(self, user_id: Optional[str] = None, username: Optional[str] = None, period: str = 'all') -> Dict[str, Any]:
        """사용량 요약 조회"""
        try:
            # 기간별 필터링
            from datetime import datetime, timedelta
            
            now = datetime.now()
            if period == 'today':
                start_date = now.replace(hour=0, minute=0, second=0, microsecond=0)
            elif period == 'week':
                start_date = now - timedelta(days=7)
            elif period == 'month':
                start_date = now - timedelta(days=30)
            else:
                start_date = None
            
            # 사용량 데이터 조회
            usage_data = []
            for usage in self.flop_tracker.usage_cache:
                if user_id and usage.user_id != user_id:
                    continue
                if username and usage.username != username:
                    continue
                    
                usage_date = datetime.fromisoformat(usage.timestamp.replace('Z', '+00:00'))
                if start_date and usage_date < start_date:
                    continue
                    
                usage_data.append(usage)
            
            # 통계 계산
            total_flops = sum(u.estimated_flops for u in usage_data)
            total_tokens = sum(u.total_tokens for u in usage_data)
            total_requests = len(usage_data)
            
            # 모델별 통계
            model_stats = {}
            for usage in usage_data:
                model_key = f"{usage.provider}/{usage.model}"
                if model_key not in model_stats:
                    model_stats[model_key] = {
                        'flops': 0,
                        'tokens': 0,
                        'requests': 0,
                        'avg_duration': 0
                    }
                
                model_stats[model_key]['flops'] += usage.estimated_flops
                model_stats[model_key]['tokens'] += usage.total_tokens
                model_stats[model_key]['requests'] += 1
                model_stats[model_key]['avg_duration'] += usage.duration_ms
            
            # 평균 계산
            for stats in model_stats.values():
                if stats['requests'] > 0:
                    stats['avg_duration'] /= stats['requests']
            
            return {
                'period': period,
                'total_flops': total_flops,
                'total_tokens': total_tokens,
                'total_requests': total_requests,
                'model_stats': model_stats,
                'usage_history': [asdict(u) for u in usage_data[-100:]]  # 최근 100개 
            }
            
        except Exception as e:
            self.log_message('error', f"사용량 요약 조회 실패: {e}")
            return {'error': str(e)}
    
    def get_license_status(self, user_id: Optional[str] = None, username: Optional[str] = None) -> Dict[str, Any]:
        """라이선스 상태 조회 (기존 라이선스 시스템과 연동)"""
        try:
            # 기존 라이선스 매니저에서 라이선스 정보 직접 읽기
            license_info = self._read_existing_license()
            
            # 사용자별 사용량 조회
            current_usage = self.get_current_usage(user_id, username)
            
            # API 플랜에 따른 FLOP 한도 매핑
            flop_limits = {
                'basic': 1e12,        # 1 TFLOPS
                'standard': 10e12,    # 10 TFLOPS  
                'premium': 100e12,    # 100 TFLOPS
                'unlimited': float('inf')  # 무제한
            }
            
            plan = license_info.get('apiPlan', 'basic')
            flop_limit = flop_limits.get(plan, flop_limits['basic'])
            
            # JSON 직렬화를 위해 Infinity 값을 문자열로 변환
            flop_remaining = max(0, flop_limit - current_usage.get('total_flops', 0)) if flop_limit != float('inf') else "unlimited"
            flop_limit_json = "unlimited" if flop_limit == float('inf') else flop_limit
            
            return {
                'active': license_info.get('valid', False),
                'plan': plan,
                'flop_limit': flop_limit_json,
                'flop_used': current_usage.get('total_flops', 0),
                'flop_remaining': flop_remaining,
                'reset_date': license_info.get('resetDate'),
                'expires_at': license_info.get('expiresAt'),
                'api_limit': license_info.get('apiLimit', 1000),
                'api_used': license_info.get('apiUsage', {}).get('used', 0),
                'company': license_info.get('company', 'Unknown'),
                'user_id': user_id,
                'username': username
            }
            
        except Exception as e:
            self.log_message('error', f"라이선스 상태 조회 실패: {e}")
            return {'active': False, 'error': str(e)}
    
    def get_current_usage(self, user_id: Optional[str] = None, username: Optional[str] = None) -> Dict[str, Any]:
        """현재 사용량 조회"""
        try:
            # 현재 월의 사용량 계산
            from datetime import datetime
            
            now = datetime.now()
            start_of_month = now.replace(day=1, hour=0, minute=0, second=0, microsecond=0)
            
            total_flops = 0
            total_tokens = 0
            request_count = 0
            
            for usage in self.flop_tracker.usage_cache:
                if user_id and usage.user_id != user_id:
                    continue
                if username and usage.username != username:
                    continue
                    
                # 타임존 정보를 제거하고 비교
                usage_timestamp = usage.timestamp.replace('Z', '').replace('+00:00', '')
                usage_date = datetime.fromisoformat(usage_timestamp)
                if usage_date >= start_of_month:
                    total_flops += usage.estimated_flops
                    total_tokens += usage.total_tokens
                    request_count += 1
            
            return {
                'total_flops': total_flops,
                'total_tokens': total_tokens,
                'request_count': request_count,
                'period': 'current_month'
            }
            
        except Exception as e:
            self.log_message('error', f"현재 사용량 조회 실패: {e}")
            return {'total_flops': 0, 'total_tokens': 0, 'request_count': 0}

# 싱글톤 인스턴스
flop_license_manager = FLOPLicenseManager()

if __name__ == "__main__":
    # 테스트 코드
    logging.basicConfig(level=logging.INFO)
    
    manager = FLOPLicenseManager()
    
    # 테스트 사용량 추적
    result = manager.track_ai_usage(
        provider="openai",
        model="gpt-4",
        prompt_tokens=1000,
        completion_tokens=500,
        duration_ms=2500.0,
        session_id="test_session",
        user_id="test_user",
        username="테스트사용자"
    )
    
    if result:
        print("=== 사용량 추적 결과 ===")
        print(f"FLOP 사용량: {result['formatted_flops']}")
        print(f"비용 (USD): ${result['cost']['total_cost_usd']:.6f}")
        print(f"비용 (KRW): ₩{result['cost']['total_cost_krw']:.2f}")
        print(f"사용률: {result['limits']['usage_percentage']:.1f}%")
    
    # 월간 비용 추정
    estimate = manager.estimate_monthly_cost(1000000, 'standard')  # 1M GFLOPS
    print(f"\n=== 월간 비용 추정 (1M GFLOPS, Standard 플랜) ===")
    print(f"총 비용 (USD): ${estimate['total_cost_usd']:.2f}")
    print(f"총 비용 (KRW): ₩{estimate['total_cost_krw']:,.0f}")
    
    # 사용량 리포트
    report = manager.get_usage_report('monthly')
    print(f"\n=== 월간 사용량 리포트 ===")
    print(f"총 FLOP 사용량: {report['summary']['formatted_flops']}")
    print(f"총 비용: ${report['summary']['total_cost_usd']:.6f}") 