#!/bin/bash

# RAG 기능에 필요한 Python 패키지들을 설치하는 스크립트

# 색상 정의
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# 로그 함수
log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# 시스템 패키지 설치
install_system_packages() {
    log_info "시스템 패키지 설치를 시작합니다..."
    
    # OS 확인
    if [ -f /etc/debian_version ]; then
        # Debian/Ubuntu
        log_info "Debian/Ubuntu 시스템이 감지되었습니다."
        sudo apt update
        sudo apt install -y \
            poppler-utils \
            build-essential \
            python3-dev
    elif [ -f /etc/redhat-release ]; then
        # RHEL/CentOS
        log_info "RHEL/CentOS 시스템이 감지되었습니다."
        sudo yum install -y \
            poppler-utils \
            gcc \
            gcc-c++ \
            python3-devel
    else
        log_warn "지원되지 않는 Linux 배포판입니다. 시스템 패키지는 수동으로 설치해야 할 수 있습니다."
    fi
}

# Python 패키지 설치
install_python_packages() {
    log_info "RAG 관련 Python 패키지 설치를 시작합니다..."
    
    # 가상환경 경로
    VENV_PATH="$HOME/.airun_venv"
    
    # 가상환경이 있는지 확인
    if [ ! -d "$VENV_PATH" ]; then
        log_error "AI.RUN 가상환경($VENV_PATH)을 찾을 수 없습니다."
        log_info "먼저 'airun' 명령어를 한 번 실행하여 가상환경을 생성해주세요."
        exit 1
    fi
    
    # 가상환경 활성화
    log_info "가상환경 활성화 중..."
    source "$VENV_PATH/bin/activate"
    if [ $? -ne 0 ]; then
        log_error "가상환경 활성화 실패"
        exit 1
    fi

    # RAG 관련 Python 패키지 설치
    log_info "RAG 관련 패키지 설치 중..."
    pip install --no-cache-dir \
        # chromadb \  # Replaced with PostgreSQL
        langchain \
        langchain-huggingface \
        sentence-transformers \
        torch \
        transformers \
        pandas \
        numpy \
        matplotlib \
        PyPDF2 \
        python-docx \
        python-pptx \
        openpyxl \
        pyhwp \
        olefile \
        beautifulsoup4 \
        lxml \
        selenium \
        webdriver_manager \
        cairosvg \
        Pillow \
        trafilatura \
        langdetect \
        requests \
        fpdf \
        reportlab

    if [ $? -eq 0 ]; then
        log_info "RAG 패키지 설치 완료"
    else
        log_error "RAG 패키지 설치 실패"
        exit 1
    fi

    # transformers 캐시 디렉토리 권한 설정
    TRANSFORMERS_CACHE_DIR="/usr/local/lib/node_modules/airun/node_modules/@xenova/transformers/.cache"
    if [ -d "/usr/local/lib/node_modules/airun" ]; then
        log_info "전역 설치 감지됨, transformers 캐시 디렉토리 권한 설정 중..."
        sudo mkdir -p "$TRANSFORMERS_CACHE_DIR"
        sudo chown -R $(whoami):$(whoami) "$TRANSFORMERS_CACHE_DIR"
        sudo chmod -R 755 "$TRANSFORMERS_CACHE_DIR"
        sudo chmod -R u+w "$TRANSFORMERS_CACHE_DIR"
        log_info "캐시 디렉토리 권한 설정 완료"
    fi

    # 가상환경 비활성화
    deactivate
}

# 메인 실행
main() {
    log_info "RAG 시스템 설정 시작"
    install_system_packages
    install_python_packages
    log_info "RAG 시스템 설정 완료"
    log_info "이제 'airun rag' 명령어로 RAG 기능을 활성화할 수 있습니다."
}

# 스크립트 실행
main 