import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import { dirname } from 'path';
import { getVarVal } from '../../configuration.js';
import { isElectron } from '../../commons.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

export let translations = {};
let electronApp;

// Electron 환경에서만 app import
if (isElectron()) {
    const pkg = await import('electron');
    electronApp = pkg.app;
}

/**
 * 특정 언어의 번역 파일을 로드합니다.
 */
export async function loadTranslation(arg) {
    try {
        const langFile = path.join(__dirname, '../../lang', `${arg.lang}.json`);
        if (fs.existsSync(langFile)) {
            const content = fs.readFileSync(langFile, 'utf8');
            return content;
        } else {
            return null;
        }
    } catch (error) {
        console.error('Error loading translation:', error);
        return null;
    }
}

/**
 * 초기 번역을 로드합니다.
 */
export async function loadTranslations() {
    try {
        // 현재 설정된 언어 코드 가져오기
        const language = await getVarVal('LANGUAGE') || 'en';
        
        // 언어 파일 경로 설정
        const langFile = path.join(__dirname, '../../lang', `${language.toLowerCase()}.json`);
        
        // 해당 언어 파일이 없으면 영어로 폴백
        if (!fs.existsSync(langFile)) {
            console.warn(`Language file not found for ${language}, falling back to English`);
            const fallbackFile = path.join(__dirname, '../../lang', 'en.json');
            if (!fs.existsSync(fallbackFile)) {
                throw new Error('Language files not found');
            }
            translations = JSON.parse(await fs.promises.readFile(fallbackFile, 'utf8'));
        } else {
            translations = JSON.parse(await fs.promises.readFile(langFile, 'utf8'));
        }
        
        return translations;
    } catch (error) {
        console.error('Error loading translations:', error);
        translations = {};
        return translations;
    }
}

/**
 * 시스템의 로케일을 가져옵니다.
 */
export async function getSystemLocale() {
    try {
        let locale = 'en';

        // Electron 환경에서는 app.getLocale() 사용
        if (isElectron() && electronApp) {
            const electronLocale = electronApp.getLocale();
            if (electronLocale) {
                return electronLocale.split('-')[0].toLowerCase();
            }
        }

        // 환경 변수에서 로케일 확인
        const envLang = process.env.LANG;
        if (envLang) {
            const langMatch = envLang.match(/^([a-z]{2})/i);
            if (langMatch) {
                return langMatch[1].toLowerCase();
            }
        }
        
        return locale;
    } catch (error) {
        return 'en';
    }
} 