import { PythonBridge } from './PythonBridge.js';
import path from 'path';
import { fileURLToPath } from 'url';
import fs from 'fs/promises';
import chalk from 'chalk';
import os from 'os';
import crypto from 'crypto';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// 환경 감지
const isElectron = () => {
    return typeof process !== 'undefined' && process.versions && process.versions.electron;
};

// HTTP 클라이언트 (airun-report-server.py와 통신)
class ReportServiceClient {
    constructor() {
        this.baseUrl = `http://localhost:${process.env.REPORT_SERVER_PORT || 5620}`;  // airun-report-server.py 포트
        this.isAvailable = false;
        this.checkAvailability();
    }

    async checkAvailability() {
        try {
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), 3000); // 3초 타임아웃

            const response = await fetch(`${this.baseUrl}/health`, {
                signal: controller.signal,
                headers: {
                    'Accept': 'application/json'
                }
            }).finally(() => clearTimeout(timeoutId));

            if (!response.ok) {
                this.isAvailable = false;
                return false;
            }

            const health = await response.json();
            this.isAvailable = health.status === 'healthy';
            
            // Report Server 연결 확인 로그 제거 (사용자 요청에 따라)
            
            return this.isAvailable;

        } catch (error) {
            this.isAvailable = false;
            return false;
        }
    }

    async generateReport(data) {
        try {
            if (!this.isAvailable) {
                throw new Error('Report service not available');
            }

            // 요청 데이터 준비 (서버 API 스키마에 맞춤)
            const requestData = {
                prompt: data.executive_summary || data.prompt,  // 서버에서 기대하는 필드명
                format: data.format || 'pdf',  // 서버에서 기대하는 필드명
                template: data.template,
                use_cache: data.use_cache !== false,
                project_hash: data.project_hash,
                job_id: data.job_id,
                username: data.username || 'default',  // 서버에서 기대하는 필드 추가
                
                // 부분 수정 모드 관련 필드들
                modification_mode: data.modification_mode || 'full',
                existing_document_path: data.existing_document_path,
                existing_document_content: data.existing_document_content,
                sections_to_modify: data.sections_to_modify || [],
                preserve_formatting: data.preserve_formatting !== false
            };

            // 보고서 생성 요청
            const response = await fetch(`${this.baseUrl}/generate`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(requestData)
            });

            if (!response.ok) {
                throw new Error(`Report generation failed: ${response.status}`);
            }

            const result = await response.json();
            if (!result.success) {
                throw new Error(result.message || 'Unknown error');
            }

            return result;
        } catch (error) {
            console.log('Report service error:', error.message);
            throw error;
        }
    }

    async pollJobStatus(jobId, progressCallback) {
        try {
            const pollInterval = 1000; // 1초마다 체크 (더 빠른 응답)
            const maxPolls = 600; // 10분 최대 대기
            let pollCount = 0;

            return new Promise((resolve, reject) => {
                const poll = async () => {
                    try {
                        if (pollCount >= maxPolls) {
                            reject(new Error('작업 시간이 초과되었습니다.'));
                            return;
                        }

                        const response = await fetch(`${this.baseUrl}/status/${jobId}`, {
                            headers: { 'Accept': 'application/json' }
                        });

                        if (!response.ok) {
                            reject(new Error(`Status check failed: ${response.status}`));
                            return;
                        }

                        const status = await response.json();
                        
                        // 상태 로그 출력
                        console.log(`📊 [${new Date().toLocaleTimeString()}] 작업 상태: ${status.status}, 진행률: ${status.progress}%`);
                        
                        // 진행률 콜백 호출
                        if (progressCallback && status.progress !== undefined) {
                            progressCallback(`처리 중... (${status.progress}%)`, status.progress);
                        }

                        if (status.status === 'completed') {
                            // 완료된 경우 최종 결과 조회
                            resolve({
                                success: true,
                                result: {
                                    file_path: status.file_path,
                                    statistics: status.statistics,
                                    job_id: jobId
                                }
                            });
                        } else if (status.status === 'failed') {
                            reject(new Error(status.error || '작업이 실패했습니다.'));
                        } else if (status.status === 'cancelled') {
                            reject(new Error('작업이 취소되었습니다.'));
                        } else {
                            // 계속 대기
                            pollCount++;
                            setTimeout(poll, pollInterval);
                        }
                    } catch (error) {
                        reject(error);
                    }
                };

                poll();
            });
        } catch (error) {
            throw error;
        }
    }
}

class ReportManager {
    constructor(options = {}) {
        this.pythonBridge = options.pythonBridge || new PythonBridge(options);
        this.outputDir = options.outputDir || path.join(process.cwd(), 'output');
        this.templatesDir = options.templatesDir || path.join(__dirname, 'templates');
        this.initialized = false;
        this.debug = options.debug || false;
        
        // HTTP 클라이언트 추가
        this.reportServiceClient = new ReportServiceClient();
        
        // 템플릿 디렉토리 초기화
        this.initTemplatesDir();
    }

    /**
     * 템플릿 디렉토리 경로를 초기화합니다.
     * @returns {string} 템플릿 디렉토리의 절대 경로
     */
    initTemplatesDir() {
        // ReportManager.js 파일의 위치를 기준으로 템플릿 디렉토리 경로 설정
        const templatesDir = path.join(__dirname, 'templates');
        
        // 디버그 로그 (비동기 작업은 하지 않음)
        // if (this.debug) {
        //     console.log(chalk.blue('[ReportManager] 템플릿 디렉토리 경로:'), {
        //         __filename,
        //         __dirname,
        //         templatesDir
        //     });
        // }
        
        return templatesDir;
    }

    /**
     * 템플릿 디렉토리 경로를 반환합니다.
     * @returns {string} 템플릿 디렉토리의 절대 경로
     */
    getTemplatesDir() {
        return this.templatesDir;
    }

    log(message) {
        if (this.debug) {
            console.log(chalk.blue(`[ReportManager] ${message}`));
        }
    }

    async initialize() {
        if (!this.initialized) {
            try {
                // 템플릿 디렉토리 존재 여부 확인
                try {
                    await fs.access(this.templatesDir);
                } catch (error) {
                    this.log(`템플릿 디렉토리 생성: ${this.templatesDir}`);
                    await fs.mkdir(this.templatesDir, { recursive: true });
                }

                // 출력 디렉토리 생성
                await fs.mkdir(this.outputDir, { recursive: true });
                
                // Python 브릿지 초기화 (폴백용)
                if (this.pythonBridge) {
                    try {
                        await this.pythonBridge.initialize();
                        // this.log('Python Bridge 초기화 완료 (폴백용)');
                    } catch (error) {
                        // this.log('Python Bridge 초기화 실패 - HTTP 서비스만 사용 가능');
                    }
                }
                
                this.initialized = true;
                
                // 디버그 로그
                // this.log('초기화 완료');
                // this.log(`- 템플릿 디렉토리: ${this.templatesDir}`);
                // this.log(`- 출력 디렉토리: ${this.outputDir}`);
                
                return true;
            } catch (error) {
                console.error(chalk.red('Report manager initialization failed:'), error);
                throw new Error(`초기화 실패: ${error.message}`);
            }
        }
        return true;
    }

    async generateBusinessPlan(data) {
        try {
            if (!this.initialized) {
                await this.initialize();
            }

            // project_hash, job_id, log_file: 전달받은 값이 있으면 사용, 없으면 새로 생성
            let hash = data.project_hash;
            let jobId = data.job_id;
            let logFilePath = data.log_file;

            if (!hash || !jobId || !logFilePath) {
                const promptText = data.executive_summary || data.prompt || '';
                hash = hash || crypto.createHash('md5').update(promptText).digest('hex').slice(0, 8);
                const timestamp = new Date().toISOString().replace(/[-:TZ.]/g, '').slice(0, 15);
                jobId = jobId || `${hash}_${timestamp}`;
                logFilePath = logFilePath || path.join(os.homedir(), '.airun', 'logs', `report_generator_${jobId}.log`);
            }

            // 데이터에 생성된 값들 추가
            data.project_hash = hash;
            data.job_id = jobId;
            data.log_file = logFilePath;

            // progressCallback 함수 설정
            const progressCallback = data.progressCallback || (() => {});

            // 1. HTTP 서비스 우선 시도
            await this.reportServiceClient.checkAvailability();
            
            if (this.reportServiceClient.isAvailable) {
                console.log(chalk.green('🚀 Report Server 사용 (빠른 처리)'));
                
                try {
                    // HTTP 서비스로 보고서 생성
                    const httpResult = await this.reportServiceClient.generateReport(data);
                    
                    // 실시간 상태 polling
                    const finalResult = await this.reportServiceClient.pollJobStatus(
                        httpResult.job_id, 
                        progressCallback
                    );
                    
                    // 기존 반환값에 project_hash, job_id, log_file 추가
                    return {
                        ...finalResult.result,
                        project_hash: hash,
                        job_id: jobId,
                        log_file: logFilePath,
                    };
                    
                } catch (httpError) {
                    console.log(chalk.yellow('⚠️ Report Server 처리 실패, Python Bridge로 폴백'));
                    console.log(chalk.yellow(`   오류: ${httpError.message}`));
                    console.log(chalk.yellow(`   스택 트레이스: ${httpError.stack}`));
                    // HTTP 실패 시 폴백으로 계속 진행
                }
            } else {
                console.log(chalk.yellow('⚠️ Report Server 불가, Python Bridge 사용'));
            }

            // 2. 폴백: 기존 Python Bridge 방식
            if (!this.pythonBridge) {
                throw new Error('Report Server 불가 상태이고 Python Bridge도 사용할 수 없습니다.');
            }

            console.log(chalk.blue('🔧 Python Bridge 사용 (폴백 모드)'));
            
            // 개발/배포 모드 감지
            const reportGeneratorPath = path.join(__dirname, 'report_generator.py');
            const isDevelopmentMode = await fs.access(reportGeneratorPath).then(() => true).catch(() => false);
            
            if (isDevelopmentMode) {
                // console.log(chalk.yellow('🔧 개발 모드: 수정된 Python 소스 파일을 사용합니다.'));
            } else {
                // console.log(chalk.blue('📦 배포 모드: 컴파일된 모듈을 사용합니다.'));
            }

            // 보고서 포맷 정의
            const reportFormat = data.format || 'pdf';

            // 출력 파일 경로 설정
            const outputPath = path.join(
                this.outputDir, 
                `business_plan_${Date.now()}.${reportFormat}`
            );
            
            // 입력 데이터 검증
            if (!data.executive_summary) {
                throw new Error('사업 계획 내용이 필요합니다.');
            }

            // Python 스크립트에서 출력되는 진행 상황을 캡처하기 위한 이벤트 리스너 설정
            const originalConsoleLog = console.log;
            let lastMessage = '';  // 마지막 메시지를 저장하는 변수
            let lastProgressMessage = ''; // 마지막으로 전달된 진행 메시지

            console.log = function(message) {
                // 원본 메시지 출력 복원
                originalConsoleLog.apply(console, arguments);
                
                if (typeof message === 'string') {
                    // Python 버전 정보는 처리하지 않음
                    if (message.includes('[DEBUG] Python 버전:')) {
                        return;
                    }

                    // ANSI 이스케이프 시퀀스와 로그 레벨 제거한 메시지 생성
                    let cleanMessage = message
                        .replace(/\u001b\[\d+m/g, '')
                        .replace(/\[(DEBUG|INFO|WARN|ERROR)\]\s*/g, '')
                        .replace(/\[JobID: [^\]]+\]\s*/g, '');
                    
                    // 이전 메시지와 동일한 경우 progressCallback 호출 무시
                    if (cleanMessage === lastMessage || cleanMessage === lastProgressMessage) {
                        return;
                    }
                    lastMessage = cleanMessage;

                    // Python 스크립트의 진행 상황 메시지 감지 및 전달
                    if (cleanMessage.includes('🚀') || cleanMessage.includes('📊') || 
                        cleanMessage.includes('✅') || cleanMessage.includes('📝') || 
                        cleanMessage.includes('📑') || cleanMessage.includes('❌') ||
                        cleanMessage.includes('💾') || cleanMessage.includes('⚠️') ||
                        cleanMessage.includes('🔍')) {
                        lastProgressMessage = cleanMessage;
                        progressCallback(cleanMessage, null);
                    }
                    // 섹션 작성 중 메시지 (예: "📑 5. 기대 효과 작성 중...")
                    else if (cleanMessage.includes('작성 중')) {
                        // 백분율 추출 (있는 경우)
                        const percentMatch = cleanMessage.match(/\((\d+\.\d+)%\)/);
                        // 백분율이 있으면 그대로 사용, 없으면 null
                        const percent = percentMatch ? parseFloat(percentMatch[1]) : null;
                        lastProgressMessage = cleanMessage;
                        progressCallback(cleanMessage, percent);
                    }
                    // 진행 상태 메시지
                    else if (cleanMessage.includes('생성 중') || cleanMessage.includes('분석 중') || 
                             cleanMessage.includes('처리 중') || cleanMessage.includes('진행 중') || 
                             cleanMessage.includes('로드 중') || cleanMessage.includes('초기화 중') || 
                             cleanMessage.includes('완료') || cleanMessage.includes('시작합니다') || 
                             cleanMessage.includes('사용합니다')) {
                        lastProgressMessage = cleanMessage;
                        progressCallback(cleanMessage, null);
                    }
                    // 섹션 및 하위 섹션 구조 메시지
                    else if (cleanMessage.includes('└') || cleanMessage.includes('├') || cleanMessage.includes('│')) {
                        lastProgressMessage = cleanMessage;
                        progressCallback(cleanMessage, null);
                    }
                    // 백분율이 포함된 메시지
                    else if (cleanMessage.includes('%') && (
                             cleanMessage.includes('(') || cleanMessage.includes('진행') || 
                             cleanMessage.includes('완료') || cleanMessage.includes('처리'))) {
                        // 백분율 추출
                        const percentMatch = cleanMessage.match(/(\d+(\.\d+)?)%/);
                        // 백분율이 있으면 그대로 사용, 없으면 null
                        const percent = percentMatch ? parseFloat(percentMatch[1]) : null;
                        lastProgressMessage = cleanMessage;
                        progressCallback(cleanMessage, percent);
                    }
                    // 섹션 번호가 포함된 메시지 (예: "1. 사업 개요", "2.1 시장 분석")
                    else if (/\d+\.\s+.+/.test(cleanMessage) || /\d+\.\d+\s+.+/.test(cleanMessage)) {
                        lastProgressMessage = cleanMessage;
                        progressCallback(cleanMessage, null);
                    }
                }
            };

            // Python 스크립트에 전달할 데이터 준비
            const inputData = {
                ...data,
                format: reportFormat,
                output_path: undefined,
                project_hash: hash,
                job_id: jobId,
            };

            // Python 스크립트 실행
            try {
                const result = await this.pythonBridge.executePythonScript(
                    'generate_bizplan.py',
                    [JSON.stringify(inputData)]
                );

                // 원래 console.log 복원
                console.log = originalConsoleLog;

                if (!result.success) {
                    throw new Error(result.error || '사업계획서 생성 실패');
                }

                // 기존 반환값에 project_hash, job_id, log_file 추가
                return {
                    ...result.result,
                    project_hash: hash,
                    job_id: jobId,
                    log_file: logFilePath,
                };
            } catch (pythonError) {
                // 원래 console.log 복원
                console.log = originalConsoleLog;
                
                // Python 오류 메시지 개선
                console.error(chalk.red('Python 스크립트 실행 오류:'), pythonError.message);
                
                // 오류 메시지 그대로 전달
                throw pythonError; // 원래 오류 객체를 그대로 전달
            }

        } catch (error) {
            console.error(chalk.red('Business plan generation failed:'), error.message);
            // 오류 메시지 그대로 전달
            throw error;
        }
    }

    /**
     * 템플릿 파일의 절대 경로를 반환합니다.
     * @param {string} templateName - 템플릿 이름
     * @returns {string} 템플릿 파일의 절대 경로
     */
    getTemplatePath(templateName) {
        return path.join(this.templatesDir, `${templateName}_templates.json`);
    }

    /**
     * 템플릿 파일의 내용을 읽고 분석합니다.
     * @param {string} templatePath - 템플릿 파일 경로
     * @returns {Object} 템플릿 정보
     */
    async analyzeTemplate(templatePath) {
        try {
            const content = await fs.readFile(templatePath, 'utf8');
            const templateData = JSON.parse(content);
            
            // 템플릿 메타데이터 추출
            const name = templateData.name;
            const description = templateData.description;
            const category = templateData.category;
            
            // 메타데이터 키들 제외하고 실제 섹션들만 추출
            const { name: _, description: __, category: ___, ...sections } = templateData;
            const topLevelSections = Object.keys(sections);
            
            // 모든 하위 섹션의 특성을 분석
            let features = {
                hasCharts: false,
                hasTables: false,
                hasFlows: false,
                hasRag: false,
                hasWeb: false,
                hasCompetitorAnalysis: false,
                hasDiagrams: false
            };
            
            let sectionTypes = new Set();
            let totalPrompts = 0;

            // 재귀적으로 모든 섹션을 분석하는 함수
            const analyzeSectionFeatures = (section) => {
                if (typeof section === 'object' && section !== null) {
                    // 프롬프트를 포함하는 섹션인 경우
                    if (section.prompt) {
                        totalPrompts++;
                        features.hasCharts = features.hasCharts || section.requires_chart === true;
                        features.hasTables = features.hasTables || section.requires_table === true;
                        features.hasFlows = features.hasFlows || section.requires_flow === true;
                        features.hasRag = features.hasRag || section.requires_rag === true;
                        features.hasWeb = features.hasWeb || section.requires_web === true;
                        features.hasCompetitorAnalysis = features.hasCompetitorAnalysis || section.requires_competitor_analysis === true;
                        features.hasDiagrams = features.hasDiagrams || section.requires_diagram === true;

                        if (section.section_type) {
                            sectionTypes.add(section.section_type);
                        }
                    }
                    
                    // 하위 섹션에 대해 재귀적으로 분석
                    Object.values(section).forEach(subsection => {
                        if (typeof subsection === 'object' && subsection !== null) {
                            analyzeSectionFeatures(subsection);
                        }
                    });
                }
            };

            analyzeSectionFeatures(sections);

            // 템플릿 파일명에서 이름 추출 (백업용)
            const templateName = path.basename(templatePath, '_templates.json');
            
            return {
                name: name || templateName,
                description: description || '설명이 없습니다',
                category: category || 'default',
                type: name || templateName, // 호환성을 위해 type 필드도 유지
                structure: {
                    topLevelSections,
                    totalPrompts,
                    sectionTypes: Array.from(sectionTypes)
                },
                features,
                sections,
                metadata: {
                    totalSections: topLevelSections.length,
                    hasCharts: features.hasCharts,
                    hasTables: features.hasTables,
                    hasFlows: features.hasFlows,
                    hasRag: features.hasRag,
                    hasWeb: features.hasWeb,
                    hasCompetitorAnalysis: features.hasCompetitorAnalysis,
                    hasDiagrams: features.hasDiagrams,
                    sectionTypes: Array.from(sectionTypes)
                }
            };
        } catch (error) {
            this.log(`템플릿 분석 실패 (${path.basename(templatePath)}): ${error.message}`);
            return {
                error: `템플릿 파일 읽기 실패: ${error.message}`,
                path: templatePath
            };
        }
    }

    /**
     * 사용 가능한 모든 템플릿 목록과 정보를 반환합니다.
     * @returns {Promise<Object>} 템플릿 목록 및 정보
     */
    async getAvailableTemplates() {
        try {
            const templatesDir = this.getTemplatesDir();
            const files = await fs.readdir(templatesDir);
            
            const templates = await Promise.all(
                files
                    .filter(file => file.endsWith('_templates.json'))
                    .map(async file => {
                        const templateName = file.replace('_templates.json', '');
                        const templatePath = path.join(templatesDir, file);
                        
                        const templateInfo = await this.analyzeTemplate(templatePath);
                        
                        return {
                            name: templateName,
                            path: templatePath,
                            description: templateInfo.description,
                            error: templateInfo.error,
                            metadata: templateInfo.metadata,
                            sections: templateInfo.sections ? Object.keys(templateInfo.sections).length : 0,
                            lastModified: (await fs.stat(templatePath)).mtime
                        };
                    })
            );

            return {
                templates,
                default: 'simple',
                count: templates.length,
                templatesDir,
                timestamp: new Date().toISOString()
            };
        } catch (error) {
            this.log(`템플릿 목록 조회 실패: ${error.message}`);
            throw new Error(`템플릿 목록을 가져오는데 실패했습니다: ${error.message}`);
        }
    }

    /**
     * 특정 템플릿의 상세 정보를 반환합니다.
     * @param {string} templateName - 템플릿 이름
     * @returns {Promise<Object>} 템플릿 상세 정보
     */
    async getTemplateDetails(templateName) {
        try {
            const templatesDir = this.getTemplatesDir();
            const templatePath = path.join(templatesDir, `${templateName}_templates.json`);
            
            if (!await fs.access(templatePath).then(() => true).catch(() => false)) {
                throw new Error(`템플릿 '${templateName}'을 찾을 수 없습니다.`);
            }

            const templateInfo = await this.analyzeTemplate(templatePath);
            
            return {
                name: templateName,
                path: templatePath,
                ...templateInfo,
                lastModified: (await fs.stat(templatePath)).mtime,
                timestamp: new Date().toISOString()
            };
        } catch (error) {
            this.log(`템플릿 상세 정보 조회 실패 (${templateName}): ${error.message}`);
            throw error;
        }
    }

    /**
     * 템플릿 파일의 유효성을 검사합니다.
     * @param {Object} templateData - 검증할 템플릿 데이터
     * @returns {Object} 검증 결과
     */
    validateTemplateStructure(templateData) {
        try {
            const validationResult = {
                isValid: true,
                errors: []
            };

            // JSON 객체 검증
            if (!templateData || typeof templateData !== 'object' || Array.isArray(templateData)) {
                validationResult.isValid = false;
                validationResult.errors.push('유효한 JSON 객체 형식이 아닙니다.');
                return validationResult;
            }

            // 섹션 존재 여부 검증
            if (Object.keys(templateData).length === 0) {
                validationResult.isValid = false;
                validationResult.errors.push('템플릿에 섹션이 없습니다.');
                return validationResult;
            }

            // 각 섹션의 프롬프트 존재 여부 검증
            Object.entries(templateData).forEach(([sectionKey, section]) => {
                if (sectionKey === 'name') return; // name 필드는 검증에서 제외
                if (typeof section === 'object' && section !== null) {
                    Object.entries(section).forEach(([subSectionKey, subSection]) => {
                        if (!subSection?.prompt) {
                            validationResult.errors.push(`섹션 '${sectionKey}.${subSectionKey}'에 프롬프트가 없습니다.`);
                        }
                    });
                } else {
                    validationResult.errors.push(`섹션 '${sectionKey}'가 올바른 형식이 아닙니다.`);
                }
            });

            validationResult.isValid = validationResult.errors.length === 0;
            return validationResult;
        } catch (error) {
            return {
                isValid: false,
                errors: [`템플릿 검증 중 오류 발생: ${error.message}`]
            };
        }
    }

    /**
     * 새 템플릿을 추가합니다.
     * @param {Object} templateData - 템플릿 데이터
     * @param {string} templateNameFromFile - 파일명에서 추출한 템플릿 이름(옵션)
     * @returns {Promise<Object>} 추가 결과
     */
    async addTemplate(templateData, templateNameFromFile = null) {
        try {
            // name 필드가 없으면 파일명에서 추출한 이름을 사용 (메모리상에서만)
            const templateName = templateData.name || templateNameFromFile;
            if (!templateName || !templateName.match(/^[a-zA-Z0-9_-]+$/)) {
                throw new Error('템플릿 데이터에 유효한 name 필드가 필요합니다. (영문자, 숫자, 언더스코어, 하이픈만 허용)');
            }
            // 템플릿 구조 검증
            const validationResult = this.validateTemplateStructure(templateData);
            if (!validationResult.isValid) {
                return {
                    success: false,
                    errors: validationResult.errors,
                    warnings: []
                };
            }
            // 파일로 저장할 때 name 필드 제거
            const dataToSave = { ...templateData };
            delete dataToSave.name;
            const templatePath = this.getTemplatePath(templateName);
            try {
                await fs.access(templatePath);
                throw new Error('이미 존재하는 템플릿 이름입니다.');
            } catch (error) {
                if (error.message !== '이미 존재하는 템플릿 이름입니다.') {
                    // 파일이 없는 경우 계속 진행
                }
            }
            await fs.writeFile(
                templatePath,
                JSON.stringify(dataToSave, null, 4),
                'utf8'
            );
            // 새로 추가된 템플릿 분석
            const templateInfo = await this.analyzeTemplate(templatePath);
            return {
                success: true,
                template: templateInfo,
                message: '템플릿이 성공적으로 추가되었습니다.'
            };
        } catch (error) {
            this.log(`템플릿 추가 실패 (${templateData.name || '이름없음'}): ${error.message}`);
            throw error;
        }
    }

    /**
     * 파일에서 템플릿을 추가합니다.
     * @param {string} filePath - 템플릿 파일 경로
     * @param {string} templateName - 템플릿 이름 (옵션)
     * @returns {Promise<Object>} 추가 결과
     */
    async addTemplateFromFile(filePath, templateName) {
        try {
            this.log(`파일에서 템플릿 추가 시도: ${filePath}`);
            
            // 파일 존재 여부 확인
            try {
                await fs.access(filePath);
            } catch (error) {
                throw new Error(`템플릿 파일을 찾을 수 없습니다: ${error.message}`);
            }
            
            // 파일 내용 읽기
            let fileContent;
            try {
                fileContent = await fs.readFile(filePath, 'utf8');
                if (!fileContent || fileContent.trim().length === 0) {
                    throw new Error('파일이 비어있습니다.');
                }
            } catch (error) {
                throw new Error(`템플릿 파일 읽기 실패: ${error.message}`);
            }
            
            // JSON 파싱
            let templateData;
            try {
                templateData = JSON.parse(fileContent);
                if (!templateData || Object.keys(templateData).length === 0) {
                    throw new Error('템플릿 데이터가 비어있습니다.');
                }
            } catch (error) {
                throw new Error(`유효하지 않은 JSON 형식: ${error.message}`);
            }
            
            // 파일명에서 템플릿 이름 추출 (제공되지 않은 경우)
            if (!templateName) {
                templateName = path.basename(filePath, '.json');
                // _templates 접미사가 있으면 제거
                templateName = templateName.replace('_templates', '');
            }
            
            // 템플릿 추가 (templateData, templateName)
            return await this.addTemplate(templateData, templateName);
        } catch (error) {
            this.log(`파일에서 템플릿 추가 실패: ${error.message}`);
            throw error;
        }
    }

    /**
     * 템플릿을 삭제합니다.
     * @param {string} templateName - 삭제할 템플릿 이름
     * @returns {Promise<Object>} 삭제 결과
     */
    async deleteTemplate(templateName) {
        try {
            // 템플릿 이름 검증
            if (!templateName || typeof templateName !== 'string') {
                throw new Error('유효한 템플릿 이름이 제공되어야 합니다.');
            }

            // 보호된 기본 템플릿 목록
            const protectedTemplates = ['simple', 'bizplan', 'proposal', 'rnd', 'marketing', 'thesis'];
            if (protectedTemplates.includes(templateName)) {
                throw new Error('기본 템플릿은 삭제할 수 없습니다.');
            }

            // 템플릿 파일 경로 생성
            const templatePath = this.getTemplatePath(templateName);

            // 템플릿 존재 여부 확인
            try {
                await fs.access(templatePath);
            } catch (error) {
                throw new Error(`템플릿 '${templateName}'을 찾을 수 없습니다.`);
            }

            // 삭제 전 템플릿 정보 저장
            const templateInfo = await this.analyzeTemplate(templatePath);

            // 템플릿 파일 삭제
            await fs.unlink(templatePath);

            return {
                success: true,
                deletedTemplate: {
                    name: templateName,
                    path: templatePath,
                    info: templateInfo
                },
                message: `템플릿 '${templateName}'이 성공적으로 삭제되었습니다.`
            };
        } catch (error) {
            this.log(`템플릿 삭제 실패 (${templateName}): ${error.message}`);
            throw error;
        }
    }
}

export default ReportManager; 