# cython: language_level=3
# cython: boundscheck=False
# cython: wraparound=True
# cython: cdivision=True
# cython: nonecheck=False
# -*- coding: utf-8 -*-

"""
문서 작성 도구
utils 패키지의 기능을 활용하여 문서 생성을 지원하는 모듈
"""

# 표준 라이브러리
import os
import sys
import json
import time
import re
import traceback
import hashlib
from typing import Dict, List, Union, Optional, Tuple, TypedDict, Literal
from dataclasses import dataclass
from datetime import datetime, timedelta
from collections import Counter
from concurrent.futures import ThreadPoolExecutor
from pathlib import Path
import shutil
import logging
from functools import lru_cache
import csv
import unicodedata
import subprocess
import signal

# 서드파티 라이브러리
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import matplotlib
matplotlib.use('Agg')
# import chromadb  # Replaced with PostgreSQL
import aiohttp
import asyncio
import requests
from openai import OpenAI
import openai

# 현재 스크립트의 디렉토리를 Python 경로에 추가
current_dir = os.path.dirname(os.path.abspath(__file__))
if current_dir not in sys.path:
    sys.path.insert(0, current_dir)

# 프로젝트 루트 디렉토리를 Python 경로에 추가
root_dir = os.path.dirname(os.path.dirname(current_dir))
if root_dir not in sys.path:
    sys.path.insert(0, root_dir)

# .env 파일 로드 (프로젝트 루트에서)
def load_env_file():
    """프로젝트 루트의 .env 파일을 로드"""
    try:
        env_file = os.path.join(root_dir, ".env")
        if os.path.exists(env_file):
            with open(env_file, 'r', encoding='utf-8') as f:
                for line in f:
                    line = line.strip()
                    if line and not line.startswith('#') and '=' in line:
                        key, value = line.split('=', 1)
                        os.environ[key.strip()] = value.strip()
            print(f"[DEBUG] report_generator.py: .env 파일 로드됨: {env_file}")
        else:
            print(f"[DEBUG] report_generator.py: .env 파일 없음: {env_file}")
    except Exception as e:
        print(f"[DEBUG] report_generator.py: .env 로드 실패: {e}")

# .env 파일 로드 실행
load_env_file()
    
# rag_process 모듈 임포트
from plugins.rag.rag_process import DocumentProcessor, get_rag_settings

# utils 패키지
from utils import (
    HWPDocument,    
    PDFDocument,
    DOCXDocument,
    PPTXDocument,
    create_matplotlib,
    search_google,
    search_naver,
    search_daum,
    summarize_content,
    load_config
)

# 외부 서비스 클라이언트 기본 클래스
class BaseServiceClient:
    """서비스 클라이언트 기본 클래스"""
    
    def __init__(self, service_name: str, port: str, port_env_var: str, base_url: str = None, logger=None):
        self.service_name = service_name
        if base_url is None:
            base_url = f"http://localhost:{os.getenv(port_env_var, port)}"
        self.base_url = base_url
        self.is_available = False
        self.logger = logger or logging.getLogger(__name__)
        self.check_availability()
    
    def check_availability(self):
        """서비스 가용성 확인"""
        try:
            response = requests.get(f"{self.base_url}/health", timeout=2)
            self.is_available = response.status_code == 200
            if self.is_available:
                self.logger.info(f"✅ {self.service_name} 연결 확인됨")
            else:
                self.logger.warning(f"⚠️ {self.service_name} 응답 불량")
        except Exception as e:
            self.is_available = False
            self.logger.warning(f"⚠️ {self.service_name} 연결 실패: {str(e)}")
    
    def _ensure_available(self) -> bool:
        """서비스 가용성 확인 및 재시도"""
        if not self.is_available:
            self.check_availability()
        return self.is_available
    
    def _handle_request_error(self, operation: str, status_code: int = None, error: str = None) -> List[Dict]:
        """공통 에러 처리"""
        if status_code:
            self.logger.error(f"{operation} 실패: HTTP {status_code}")
        if error:
            self.logger.error(f"{operation} 실패: {error}")
        return []
    
    @staticmethod
    def safe_makedirs(path: str) -> bool:
        """안전한 디렉토리 생성"""
        try:
            os.makedirs(path, exist_ok=True)
            return True
        except Exception as e:
            logging.getLogger(__name__).error(f"디렉토리 생성 실패: {path} - {str(e)}")
            return False
    
    @staticmethod
    def safe_json_load(file_path: str, logger=None) -> Optional[Dict]:
        """안전한 JSON 파일 로드"""
        try:
            with open(file_path, 'r', encoding='utf-8') as f:
                return json.load(f)
        except Exception as e:
            if logger:
                logger.error(f"JSON 파일 로드 실패: {file_path} - {str(e)}")
            return None
    
    @staticmethod
    def safe_json_dump(data: Dict, file_path: str, logger=None) -> bool:
        """안전한 JSON 파일 저장"""
        try:
            with open(file_path, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
            return True
        except Exception as e:
            if logger:
                logger.error(f"JSON 파일 저장 실패: {file_path} - {str(e)}")
            return False

class RAGServiceClient(BaseServiceClient):
    """RAG 서비스 클라이언트 (5600포트)"""
    
    def __init__(self, base_url: str = None, logger=None):
        super().__init__("RAG 서비스 (5600포트)", "5600", "RAG_SERVER_PORT", base_url, logger)
    
    async def search(self, query: str, max_results: int = 5, user_id: str = None, rag_search_scope: str = 'personal') -> List[Dict]:
        """문서 검색 수행"""
        if not self._ensure_available():
            return []
        
        try:
            # 1. 임베딩 생성
            embed_response = requests.post(
                f"{self.base_url}/embed",
                json={"text": query},
                timeout=10
            )
            
            if not embed_response.ok:
                return self._handle_request_error("임베딩 생성", embed_response.status_code)
            
            embed_result = embed_response.json()
            if not embed_result.get('success'):
                return self._handle_request_error("임베딩 생성", error=embed_result.get('error'))
            
            # 2. 문서 검색
            search_payload = {
                "embedding": embed_result['embedding'],
                "query": query,
                "user_id": user_id,
                "ragSearchScope": rag_search_scope
            }
            
            search_response = requests.post(
                f"{self.base_url}/search",
                json=search_payload,
                timeout=15
            )
            
            if not search_response.ok:
                return self._handle_request_error("문서 검색", search_response.status_code)
            
            search_result = search_response.json()
            results = search_result.get('results', [])
            
            # 결과 포맷팅
            formatted_results = []
            for result in results[:max_results]:
                formatted_results.append({
                    'source': result.get('metadata', {}).get('filename', 'Unknown'),
                    'content': result.get('content', ''),
                    'score': result.get('score', 0.0)
                })
            
            self.logger.debug(f"📚 RAG 검색 완료: {len(formatted_results)}개 결과")
            return formatted_results
            
        except Exception as e:
            self.logger.error(f"RAG 검색 중 오류: {str(e)}")
            return []

class WebSearchServiceClient(BaseServiceClient):
    """웹검색 서비스 클라이언트 (5610포트)"""
    
    def __init__(self, base_url: str = None, logger=None):
        super().__init__("웹검색 서비스 (5610포트)", "5610", "WEB_SEARCH_PORT", base_url, logger)
    
    async def search(self, query: str, max_results: int = 5) -> List[Dict]:
        """웹 검색 수행"""
        if not self._ensure_available():
            return []
        
        try:
            search_payload = {
                "query": query,
                "max_results": max_results,
                "engine": "auto",
                "use_cache": True
            }
            
            search_response = requests.post(
                f"{self.base_url}/search",
                json=search_payload,
                timeout=30
            )
            
            if not search_response.ok:
                return self._handle_request_error("웹 검색", search_response.status_code)
            
            search_result = search_response.json()
            if not search_result.get('success'):
                return self._handle_request_error("웹 검색", error=search_result.get('error'))
            
            results = search_result.get('results', [])
            
            # 결과 포맷팅
            formatted_results = []
            for result in results[:max_results]:
                formatted_results.append({
                    'title': result.get('title', ''),
                    'snippet': result.get('description', ''),
                    'url': result.get('url', ''),
                    'engine': result.get('engine', 'auto')
                })
            
            self.logger.debug(f"🌐 웹검색 완료: {len(formatted_results)}개 결과")
            return formatted_results
            
        except Exception as e:
            self.logger.error(f"웹 검색 중 오류: {str(e)}")
            return []
        
# 전역 로거 설정 함수
def _setup_logger(job_id: str = None, project_hash: str = None, log_file: str = None):
    """로거 초기화 - log_file이 명시적으로 주어지면 해당 경로로 로그 파일을 생성합니다."""
    import logging
    import os

    # 환경변수에서 로그 레벨 가져오기
    def get_log_level_from_env():
        log_level_str = os.getenv('LOG_LEVEL', 'INFO').upper()
        level_mapping = {
            'DEBUG': logging.DEBUG,
            'INFO': logging.INFO,
            'WARNING': logging.WARNING,
            'ERROR': logging.ERROR,
            'CRITICAL': logging.CRITICAL
        }
        return level_mapping.get(log_level_str, logging.INFO)
    
    log_level = get_log_level_from_env()

    # log_id: project_hash > job_id > 'unknown'
    log_id = project_hash or job_id or 'unknown'
    logger_name = f'report_generator_{log_id}'
    logger = logging.getLogger(logger_name)
    logger.setLevel(log_level)
    
    # 기존 핸들러 제거
    if logger.hasHandlers():
        logger.handlers.clear()
    
    # log_file이 명시적으로 주어지면 해당 경로 사용, 아니면 사용자별 캐시 경로에 저장
    if log_file:
        log_path = log_file
    else:
        log_filename = f"report_generator_{log_id}.log"
        # 사용자별 캐시 경로에 로그 저장 (project_hash가 있으면 해당 경로 사용)
        if project_hash:
            # 사용자별 캐시 경로 패턴: ~/.airun/cache/report/{username}/{project_hash}/logs/
            cache_base = os.path.expanduser('~/.airun/cache/report')
            # 사용자명을 찾기 위해 캐시 디렉토리 스캔
            user_dir = None
            if os.path.exists(cache_base):
                for username in os.listdir(cache_base):
                    user_cache_dir = os.path.join(cache_base, username)
                    if os.path.isdir(user_cache_dir):
                        project_dir = os.path.join(user_cache_dir, project_hash)
                        if os.path.exists(project_dir):
                            user_dir = user_cache_dir
                            break
            
            if user_dir:
                log_dir = os.path.join(user_dir, project_hash, 'logs')
            else:
                # 사용자별 캐시 디렉토리를 찾지 못한 경우 기본 경로 사용
                log_dir = os.path.expanduser('~/.airun/logs')
        else:
            # project_hash가 없으면 기본 경로 사용
            log_dir = os.path.expanduser('~/.airun/logs')
        
        os.makedirs(log_dir, exist_ok=True)
        log_path = os.path.join(log_dir, log_filename)
    
    fh = logging.FileHandler(log_path, encoding='utf-8')
    fh.setLevel(log_level)
    
    # 콘솔 핸들러 설정
    ch = logging.StreamHandler()
    ch.setLevel(log_level)
    
    # job_prefix = f'[JobID: {log_id}] ' if log_id else ''
    # file_formatter = logging.Formatter(f'%(asctime)s - {job_prefix}%(name)s - %(levelname)s - %(message)s')
    file_formatter = logging.Formatter(f'%(asctime)s - [JobID: {log_id}] - %(levelname)s - %(message)s')
   
    class CustomFormatter(logging.Formatter):
        def __init__(self, job_id=None):
            super().__init__()
            self.job_id = job_id
            self.level_prefixes = {
                logging.DEBUG: '[DEBUG]',
                logging.INFO: '[INFO]',
                logging.WARNING: '[WARNING]',
                logging.ERROR: '[ERROR]',
                logging.CRITICAL: '[CRITICAL]'
            }
        
        def format(self, record):
            prefix = self.level_prefixes.get(record.levelno, '')
            job_prefix = f'[JobID: {self.job_id}] ' if self.job_id else ''
            if prefix and not str(record.msg).startswith(prefix):
                record.msg = f"{job_prefix}{prefix} {record.msg}"
            return super().format(record)
    
    ch.setFormatter(CustomFormatter(log_id))
    fh.setFormatter(file_formatter)
    
    logger.addHandler(fh)
    logger.addHandler(ch)
    
    return logger

# 기본 로거 설정
logger = logging.getLogger('report_generator')

@dataclass
class SectionConfig:
    """섹션 설정 정보"""
    title: str
    prompt: str
    requires_chart: bool = False
    requires_flow: bool = False
    requires_table: bool = False
    requires_rag: bool = False
    requires_web: bool = False
    requires_hide: bool = False
    requires_competitor_analysis: bool = False
    requires_diagram: bool = False
    section_type: str = "default"  # 섹션 타입 추가 (default, swot, timeline, organization 등)
    search_keywords: List[str] = None
    response_format: str = "text"
    table_schema: Dict = None  # 테이블 스키마 정의

    @classmethod
    def from_template(cls, title: str, template_data: Union[str, Dict]) -> 'SectionConfig':
        """템플릿 데이터로부터 섹션 설정 생성"""
        if isinstance(template_data, str):
            return cls(title=title, prompt=template_data)
        else:
            # 섹션 타입 결정
            section_type = "default"
            # if "SWOT" in title or "swot" in title.lower():
            #     section_type = "swot"
            # elif any(keyword in title.lower() for keyword in ["조직", "추진체계"]):
            #     section_type = "organization"
            # elif any(keyword in title.lower() for keyword in ["일정", "계획", "마일스톤"]):
            #     section_type = "timeline"
            
            return cls(
                title=title,
                prompt=template_data.get("prompt", ""),
                requires_chart=template_data.get("requires_chart", False),
                requires_flow=template_data.get("requires_flow", False),
                requires_table=template_data.get("requires_table", False),
                requires_rag=template_data.get("requires_rag", False),
                requires_web=template_data.get("requires_web", False),
                requires_hide=template_data.get("requires_hide", False),
                requires_competitor_analysis=template_data.get("requires_competitor_analysis", False),
                requires_diagram=template_data.get("requires_diagram", False),
                section_type=template_data.get("section_type", section_type),
                search_keywords=template_data.get("search_keywords", []),
                response_format=template_data.get("response_format", "text"),
                table_schema=template_data.get("table_schema", None)
            )

    def __post_init__(self):
        if self.search_keywords is None:
            self.search_keywords = []
    
    def to_dict(self) -> Dict:
        """SectionConfig 객체를 딕셔너리로 변환 (JSON 직렬화용)"""
        return {
            'title': self.title,
            'prompt': self.prompt,
            'requires_chart': self.requires_chart,
            'requires_flow': self.requires_flow,
            'requires_table': self.requires_table,
            'requires_rag': self.requires_rag,
            'requires_web': self.requires_web,
            'requires_hide': self.requires_hide,
            'requires_competitor_analysis': self.requires_competitor_analysis,
            'requires_diagram': self.requires_diagram,
            'section_type': self.section_type,
            'search_keywords': self.search_keywords,
            'response_format': self.response_format,
            'table_schema': self.table_schema
        }
    
    def __str__(self) -> str:
        """문자열 변환 시 prompt 반환"""
        try:
            if self.prompt is not None:
                # 명시적으로 문자열로 변환하여 unicode 오류 방지
                if isinstance(self.prompt, str):
                    return self.prompt
                else:
                    return str(self.prompt)
            return ""
        except Exception:
            # 변환 실패 시 빈 문자열 반환
            return ""
    
    def __unicode__(self) -> str:
        """유니코드 변환을 위한 메서드"""
        return self.__str__()
    
    def __repr__(self) -> str:
        """객체 표현 문자열"""
        return f"SectionConfig(title='{self.title}', prompt_length={len(self.prompt) if self.prompt else 0})"
    
    def get_prompt_safe(self) -> str:
        """안전한 프롬프트 반환 메서드"""
        try:
            if self.prompt is not None:
                if isinstance(self.prompt, str):
                    return self.prompt
                else:
                    return str(self.prompt)
            return ""
        except Exception:
            return ""


class DocumentParser:
    """기존 문서를 파싱하여 섹션별로 분리하는 클래스"""
    
    def __init__(self, logger=None):
        self.logger = logger or logging.getLogger(__name__)
    
    def parse_document(self, file_path: str) -> Dict[str, str]:
        """문서 파일을 섹션별로 파싱"""
        try:
            if not os.path.exists(file_path):
                raise FileNotFoundError(f"문서 파일을 찾을 수 없습니다: {file_path}")
            
            file_ext = os.path.splitext(file_path)[1].lower()
            
            if file_ext == '.pdf':
                return self._parse_pdf(file_path)
            elif file_ext == '.hwp':
                return self._parse_hwp(file_path)
            elif file_ext in ['.docx', '.doc']:
                return self._parse_docx(file_path)
            elif file_ext == '.txt':
                return self._parse_text(file_path)
            else:
                raise ValueError(f"지원하지 않는 파일 형식입니다: {file_ext}")
                
        except Exception as e:
            self.logger.error(f"문서 파싱 실패: {str(e)}")
            raise
    
    def parse_text_content(self, content: str) -> Dict[str, str]:
        """텍스트 내용을 섹션별로 파싱"""
        try:
            sections = {}
            current_section = None
            current_content = []
            
            lines = content.split('\n')
            
            for line in lines:
                line = line.strip()
                
                # 섹션 헤더 패턴 매칭 (예: "1. 사업 개요", "## 시장 분석", "가. 기술 개발" 등)
                section_match = self._detect_section_header(line)
                
                if section_match:
                    # 이전 섹션 저장
                    if current_section and current_content:
                        sections[current_section] = '\n'.join(current_content).strip()
                    
                    # 새 섹션 시작
                    current_section = section_match
                    current_content = []
                else:
                    # 내용 추가
                    if line:  # 빈 줄이 아닌 경우에만
                        current_content.append(line)
            
            # 마지막 섹션 저장
            if current_section and current_content:
                sections[current_section] = '\n'.join(current_content).strip()
            
            self.logger.debug(f"파싱된 섹션 수: {len(sections)}")
            return sections
            
        except Exception as e:
            self.logger.error(f"텍스트 내용 파싱 실패: {str(e)}")
            raise
    
    def _detect_section_header(self, line: str) -> Optional[str]:
        """섹션 헤더를 감지하고 표준화된 섹션명 반환"""
        import re
        
        # 다양한 섹션 헤더 패턴들
        patterns = [
            # 숫자로 시작하는 패턴 (1. 사업개요, 1-1. 개요)
            r'^(\d+\.?\s*.*?)(?:\s*\.{3,}|\s*-{3,}|\s*={3,})?$',
            # 한글 번호 패턴 (가. 개요, 나. 기술분석)
            r'^([가-힣]\.?\s*.*?)(?:\s*\.{3,}|\s*-{3,}|\s*={3,})?$',
            # 마크다운 헤더 패턴 (## 제목, ### 소제목)
            r'^(#{1,6}\s*.*?)(?:\s*\.{3,}|\s*-{3,}|\s*={3,})?$',
            # 로마숫자 패턴 (I. 개요, II. 분석)
            r'^([IVX]+\.?\s*.*?)(?:\s*\.{3,}|\s*-{3,}|\s*={3,})?$',
            # 괄호 번호 패턴 ((1) 개요, (가) 분석)
            r'^(\([^)]+\)\s*.*?)(?:\s*\.{3,}|\s*-{3,}|\s*={3,})?$'
        ]
        
        for pattern in patterns:
            match = re.match(pattern, line.strip())
            if match:
                header = match.group(1).strip()
                # 점이나 기호 정리
                header = re.sub(r'\s*[\.]+\s*$', '', header)
                return header
        
        return None
    
    def _parse_pdf(self, file_path: str) -> Dict[str, str]:
        """PDF 파일 파싱"""
        try:
            # PDF 텍스트 추출 (pdfplumber 또는 PyPDF2 사용)
            import PyPDF2
            
            with open(file_path, 'rb') as file:
                pdf_reader = PyPDF2.PdfReader(file)
                text = ""
                for page in pdf_reader.pages:
                    text += page.extract_text() + "\n"
            
            return self.parse_text_content(text)
            
        except ImportError:
            self.logger.warning("PyPDF2가 설치되지 않음. pip install PyPDF2를 실행하세요.")
            raise
        except Exception as e:
            self.logger.error(f"PDF 파싱 실패: {str(e)}")
            raise
    
    def _parse_hwp(self, file_path: str) -> Dict[str, str]:
        """HWP 파일 파싱"""
        try:
            # HWP 파일 읽기 (기존 utils의 HWPDocument 활용)
            hwp_doc = HWPDocument()
            # HWP에서 텍스트 추출 로직 필요
            # 현재는 단순히 빈 딕셔너리 반환 (구현 필요)
            self.logger.warning("HWP 파싱 기능은 아직 구현 중입니다.")
            return {}
            
        except Exception as e:
            self.logger.error(f"HWP 파싱 실패: {str(e)}")
            raise
    
    def _parse_docx(self, file_path: str) -> Dict[str, str]:
        """DOCX 파일 파싱"""
        try:
            # python-docx 사용
            from docx import Document
            
            doc = Document(file_path)
            text = ""
            for paragraph in doc.paragraphs:
                text += paragraph.text + "\n"
            
            return self.parse_text_content(text)
            
        except ImportError:
            self.logger.warning("python-docx가 설치되지 않음. pip install python-docx를 실행하세요.")
            raise
        except Exception as e:
            self.logger.error(f"DOCX 파싱 실패: {str(e)}")
            raise
    
    def _parse_text(self, file_path: str) -> Dict[str, str]:
        """텍스트 파일 파싱"""
        try:
            with open(file_path, 'r', encoding='utf-8') as file:
                content = file.read()
            return self.parse_text_content(content)
            
        except Exception as e:
            self.logger.error(f"텍스트 파일 파싱 실패: {str(e)}")
            raise


# 설정 관련 상수를 클래스로 분리
class BizPlanConfig:
    """사업계획서 설정"""
    # 토큰 제한
    TOKEN_LIMIT = 128000
    OUTPUT_LIMIT = 16384
    RESERVE_TOKENS = OUTPUT_LIMIT + 4000
    SUMMARY_THRESHOLD = TOKEN_LIMIT - RESERVE_TOKENS
    
    # 캐시 설정
    CACHE_EXPIRY = 24 * 60 * 60  # 24시간
    
    # 차트 레이블 매핑
    CHART_LABEL_MAPPING = {
        "Market Size": "시장 규모",
        "Sales": "매출",
        "Operating Profit": "영업이익",
        "Amount (Billion KRW)": "금액 (억원)",
        "Market Size Forecast": "시장 규모 전망",
        "Sales and Operating Profit Forecast": "연도별 매출 및 영업이익 전망"
    }


class BizPlanError(Exception):
    """사업계획서 관련 예외"""
    pass

class ChartGenerationError(BizPlanError):
    """차트 생성 실패"""
    pass

class DocumentGenerationError(BizPlanError):
    """문서 생성 실패"""
    pass

class CacheManager:
    """캐시 관리 클래스"""
    def __init__(self, cache_dir: str, expiry: int = 24*60*60, logger=None):
        self.logger = logger or logging.getLogger("report_generator_default")
        self.cache_dir = cache_dir
        self.expiry = expiry
        
    def get_cache_key(self, data: str) -> str:
        """캐시 키 생성"""
        return hashlib.md5(data.encode('utf-8')).hexdigest()
    
    def is_cache_valid(self, cache_path: str) -> bool:
        """캐시 유효성 검사"""
        if not os.path.exists(cache_path):
            return False
        
        modified_time = datetime.fromtimestamp(os.path.getmtime(cache_path))
        return datetime.now() - modified_time < timedelta(seconds=self.expiry)
    
    def get_cached_data(self, cache_key: str) -> Optional[Dict]:
        """캐시된 데이터 조회"""
        cache_path = os.path.join(self.cache_dir, f"{cache_key}.json")
        if not self.is_cache_valid(cache_path):
            return None
            
        try:
            with open(cache_path, 'r', encoding='utf-8') as f:
                return json.load(f)
        except Exception as e:
            self.logger.error(f"캐시 데이터 로드 실패: {e}")
            return None


import os
import sys
import json
import time
import hashlib
import logging
import traceback
from typing import List, Dict, Optional
from concurrent.futures import ThreadPoolExecutor

# 웹 검색 모듈 임포트 (컴파일된 버전)
try:
    import plugins.websearch.web_search as web_search
    parallel_search_unix = web_search.parallel_search_unix
    parallel_search_windows = web_search.parallel_search_windows
except ImportError:
    # 현재 디렉토리의 상위 디렉토리를 Python 경로에 추가
    root_dir = os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
    if root_dir not in sys.path:
        sys.path.append(root_dir)
    import plugins.websearch.web_search as web_search
    parallel_search_unix = web_search.parallel_search_unix
    parallel_search_windows = web_search.parallel_search_windows

# Windows 환경 확인
IS_WINDOWS = os.name == 'nt'

class WebSearchService:
    """웹 검색 서비스"""
    def __init__(self, logger=None):
        self.logger = logger or logging.getLogger("report_generator_default")
        # 캐시 디렉토리 설정
        self.cache_dir = os.path.join(os.path.expanduser('~'), '.airun', 'cache', 'web_search')
        BaseServiceClient.safe_makedirs(self.cache_dir)

    def _get_cache_key(self, query: str) -> str:
        """캐시 키 생성"""
        return hashlib.md5(query.encode('utf-8')).hexdigest()

    def _validate_and_format_result(self, result: Dict) -> Optional[Dict]:
        """검색 결과 검증 및 포맷팅"""
        try:
            # 필수 필드 검증
            if not isinstance(result, dict):
                return None
                
            formatted_result = {
                'type': 'web',
                'title': result.get('title', '제목 없음'),
                'url': result.get('url', ''),
                'description': result.get('description', ''),
                'engine': result.get('engine', 'unknown')
            }
            
            # URL이 없으면 무효
            if not formatted_result['url']:
                return None
                
            return formatted_result
            
        except Exception as e:
            self.logger.error(f"검색 결과 검증 실패: {str(e)}")
            return None

    def _get_from_cache(self, cache_key: str) -> Optional[List[Dict]]:
        """캐시에서 검색 결과 가져오기"""
        cache_file = os.path.join(self.cache_dir, f"{cache_key}.json")
        if os.path.exists(cache_file):
            if time.time() - os.path.getmtime(cache_file) < 24 * 60 * 60:  # 24시간 이내
                try:
                    with open(cache_file, 'r', encoding='utf-8') as f:
                        cached_results = json.load(f)
                        self.logger.debug(f"캐시 파일 내용: {json.dumps(cached_results[:2], ensure_ascii=False, indent=2)}")  # 처음 2개 결과만 로깅
                        # 결과 검증 및 포맷팅
                        validated_results = []
                        for result in cached_results:
                            formatted_result = self._validate_and_format_result(result)
                            if formatted_result:
                                validated_results.append(formatted_result)
                        self.logger.debug(f"검증된 결과: {json.dumps(validated_results[:2], ensure_ascii=False, indent=2)}")  # 처음 2개 결과만 로깅
                        return validated_results
                except Exception as e:
                    self.logger.error(f"캐시 파일 읽기 실패: {str(e)}")
        return None

    def _save_to_cache(self, cache_key: str, results: List[Dict]):
        """검색 결과를 캐시에 저장"""
        cache_file = os.path.join(self.cache_dir, f"{cache_key}.json")
        BaseServiceClient.safe_json_dump(results, cache_file, self.logger)

    async def search(self, query: str, max_results: int = 5) -> List[Dict]:
        """웹 검색 수행"""
        try:
            # 1. 캐시 확인
            cache_key = self._get_cache_key(query)
            cached_results = self._get_from_cache(cache_key)
            if cached_results:
                self.logger.debug(f"[WebSearch] 캐시된 결과 사용: {query}")
                return cached_results[:max_results]

            # 2. 검색 수행
            self.logger.debug(f"[WebSearch] 검색 시작: {query}")
            
            # 운영체제에 따라 적절한 검색 함수 사용
            if IS_WINDOWS:
                results = await parallel_search_windows(query, max_results)
            else:
                results = await parallel_search_unix(query, max_results)
            
            if not results:
                self.logger.warning(f"[WebSearch] 검색 결과 없음: {query}")
                return []

            # 3. 결과 검증 및 포맷팅
            formatted_results = []
            for result in results:
                formatted = self._validate_and_format_result(result)
                if formatted:
                    formatted_results.append(formatted)

            # 4. 결과 캐시에 저장
            if formatted_results:
                self.logger.debug(f"[WebSearch] 검색 결과 캐시 저장: {len(formatted_results)}개")
                self._save_to_cache(cache_key, formatted_results)

            return formatted_results[:max_results]

        except Exception as e:
            self.logger.error(f"[WebSearch] 웹 검색 실패: {str(e)}")
            traceback.print_exc()
            return []
class BizPlanDocument:
    """사업계획서 문서 생성 클래스"""
    def __init__(self, output_format: str = 'pdf', logger=None):
        """
        초기화
        Args:
            output_format (str): 출력 형식 ('pdf' 또는 'hwpx' 또는 'docx' 또는 'pptx')
        """
        self.logger = logger or logging.getLogger("report_generator_default")
        self.output_format = output_format
        self.doc = None  # 명시적으로 초기화
        self.output_dir = None  # 출력 디렉토리 초기화
        self.executive_summary = None  # executive_summary 추가
        self.templates = None  # templates 추가
        self.section_order = {}  # section_order 초기화
        self.subsection_order = {}  # subsection_order 초기화
        self.usage_tracker = UsageTracker()  # 사용량 추적기 초기화
        self.enable_pagebreak = True  # 페이지 넘김 설정 기본값
        self._initialize_document()
        
        # 차트 레이블 매핑 초기화
        self.CHART_LABEL_MAPPING = {
            "Market Size": "시장 규모",
            "Sales": "매출",
            "Operating Profit": "영업이익",
            "Amount (Billion KRW)": "금액 (억원)",
            "Market Size Forecast": "시장 규모 전망",
            "Sales and Operating Profit Forecast": "연도별 매출 및 영업이익 전망"
        }

        # 캐시 디렉토리 구조 정의
        self.cache_structure = {
            'business_info': 'business_info.json',
            'sections': 'sections.json',
            'charts': 'charts',
            'tables': 'tables',
            'images': 'images',
            'competitors': 'competitors.json',
            'sequence': 'sequence.json'
        }

        # 기본 캐시 디렉토리 설정
        self.base_cache_dir = os.path.join(os.path.expanduser('~'), '.airun', 'cache', 'report')
        BaseServiceClient.safe_makedirs(self.base_cache_dir)

        # 템플릿 설정 로드
        self.templates = self._load_section_templates()
        
        # 섹션 순서는 템플릿에서 로드
        raw_templates = self.load_prompt_templates()
        self.section_order = raw_templates.get('_metadata', {}).get('section_order', {})
        self.subsection_order = {
            section: data.get('_order', {})
            for section, data in raw_templates.items()
            if isinstance(data, dict) and '_order' in data
        }

        self.executive_summary = None  # 추가

    def extract_pagebreak_setting(self, executive_summary: str) -> bool:
        """executive_summary에서 페이지 넘김 설정 추출"""
        if not executive_summary:
            return True  # 기본값은 페이지 넘김 활성화
        
        # 페이지 넘김 설정 확인
        pagebreak_match = re.search(r'^#\s*pagebreak:\s*(false|true)', executive_summary, re.MULTILINE | re.IGNORECASE)
        if pagebreak_match:
            setting = pagebreak_match.group(1).lower()
            return setting != 'false'  # 'false'가 아니면 True 반환
        
        return True 

    def _identify_template_type(self, executive_summary: str) -> str:
        """사용자 요청사항을 분석하여 적절한 템플릿 타입을 반환"""
        try:
            # 헤더에서 템플릿 정의 확인
            template_match = re.match(r'^#\s*template:\s*(\w+)', executive_summary)
            if template_match:
                template_type = template_match.group(1).lower()
                self.logger.debug(f"헤더에서 템플릿 타입 발견: {template_type}")
                return template_type

            # 키워드 기반 템플릿 매칭
            keywords = {
                'bizplan': [
                    ('사업계획서', 2), ('기술개발', 1.5), ('R&D', 1), ('연구개발', 1), ('사업화', 1.5),
                    ('시장분석', 1), ('사업전략', 1.5), ('기술성', 1), ('사업성', 1.5)
                ],
                'startup': [
                    ('창업 계획서', 2), ('설립 예정', 2), ('초기 투자금', 2),
                    ('대표자', 2), ('팀', 2), ('창업', 2), ('스타트업', 2),
                    ('벤처', 1), ('신규사업', 1), ('사업모델', 1.5),
                    ('투자유치', 1.5), ('시드머니', 1), ('엔젤투자', 1), ('시리즈', 1)
                ],
                'thesis': [
                    ('논문', 2), ('학술', 2), ('이론', 2),
                    ('연구', 0.5), ('분석', 0.5), ('검증', 0.5),
                    ('실험', 1), ('조사', 0.5), ('통계', 1)
                ],
                'marketing': [
                    ('마케팅 전략', 2), ('브랜드 전략', 2), 
                    ('마케팅', 1.5), ('브랜드', 1), ('광고', 1), ('프로모션', 1),
                    ('판매', 1), ('고객', 0.5), ('시장조사', 1), ('타겟', 1), ('포지셔닝', 1.5)
                ],
                'proposal': [
                    ('제안서', 2), ('프로젝트 계획', 2),
                    ('기획', 1), ('제안', 1.5), ('프로젝트', 1), ('실증', 1), ('시범', 1),
                    ('조성', 1), ('구축', 1), ('도입', 1), ('추진', 1)
                ]
            }

            # 요약문에서 각 템플릿 타입별 키워드 매칭 점수 계산
            scores = {}
            for template_type, template_keywords in keywords.items():
                score = 0
                for keyword, weight in template_keywords:
                    if keyword in executive_summary:
                        score += weight
                scores[template_type] = score

            # 가장 높은 점수의 템플릿 타입 반환
            best_match = max(scores.items(), key=lambda x: x[1])
            
            self.logger.debug(f"템플릿 매칭 결과:")
            for template_type, score in scores.items():
                self.logger.debug(f"- {template_type}: {score}점")
            self.logger.debug(f"선택된 템플릿: {best_match[0]} ({best_match[1]}점)")

            return best_match[0] if best_match[1] > 0 else 'simple'

        except Exception as e:
            self.logger.error(f"템플릿 타입 식별 실패: {str(e)}")
            return 'simple'  # 기본값

    def load_prompt_templates(self, executive_summary=None, template=None):
        """프롬프트 템플릿 파일 로드"""
        try:
            # 현재 스크립트의 디렉토리를 기준으로 템플릿 디렉토리 경로 설정
            current_dir = os.path.dirname(os.path.abspath(__file__))
            templates_dir = os.path.join(current_dir, 'templates')
            self.logger.debug(f"템플릿 디렉토리: {templates_dir}")
            
            # template_type 변수 초기화
            template_type = 'simple'
                           
            if template:
                template_file = f"{template}_templates.json"
                template_type = template  # template_type 설정
                self.logger.debug(f"지정된 템플릿 사용: {template}")
            elif executive_summary:
                self.executive_summary = executive_summary
                self.enable_pagebreak = self.extract_pagebreak_setting(executive_summary)                
                template_type = self._identify_template_type(executive_summary)
                template_file = f"{template_type}_templates.json"
                self.logger.debug(f"자동 감지된 템플릿: {template_type}")
            else:
                template_file = "simple_templates.json"  # 기본 템플릿
                self.logger.debug("기본 템플릿 사용: simple")
                
            template_path = os.path.join(templates_dir, template_file)
            self.logger.debug(f"템플릿 파일 경로: {template_path}")
            
            if not os.path.exists(template_path):
                self.logger.warning(f"요청한 템플릿 파일을 찾을 수 없음: {template_path}")
                if template_type != 'simple':
                    self.logger.info(f"'{template_type}' 템플릿이 없어 기본 템플릿(simple)을 사용합니다.")
                    template_path = os.path.join(templates_dir, 'simple_templates.json')
                    self.logger.debug(f"기본 템플릿 사용: {template_path}")
                    
                    # 템플릿 파일 존재 여부 다시 확인
                    if not os.path.exists(template_path):
                        raise ValueError(f"기본 템플릿 파일도 찾을 수 없습니다: {template_path}")
            
            with open(template_path, 'r', encoding='utf-8') as f:
                template_data = json.load(f)
                self.logger.debug(f"템플릿 로드 완료: {os.path.basename(template_path)}")
                
                # 메타데이터 키들 제거하고 실제 섹션들만 반환
                if isinstance(template_data, dict):
                    # 메타데이터 키들을 제외한 섹션들만 추출
                    metadata_keys = {'name', 'description', 'category'}
                    templates = {k: v for k, v in template_data.items() if k not in metadata_keys}
                    self.logger.debug(f"메타데이터 제외 후 섹션 수: {len(templates)}")
                    return templates
                else:
                    return template_data
                
        except Exception as e:
            self.logger.error(f"템플릿 로드 실패: {str(e)}")
            return {}

    def _initialize_document(self):
        """문서 객체 초기화"""
        try:
            if self.output_format == 'hwpx':
                self.doc = HWPDocument()
            elif self.output_format == 'pdf':
                self.doc = PDFDocument()
            elif self.output_format == 'docx':
                self.doc = DOCXDocument()
            elif self.output_format == 'pptx':
                self.doc = PPTXDocument()
            else:
                raise ValueError(f"지원하지 않는 출력 형식입니다: {self.output_format}")
            
            
        except Exception as e:
            self.logger.error(f"문서 초기화 실패: {str(e)}")
            raise

    def _initialize_cache_directories(self, base_dir: str) -> Dict[str, str]:
        """캐시 디렉토리 초기화"""
        try:
            # 기본 캐시 디렉토리 생성
            os.makedirs(base_dir, exist_ok=True)
            
            # 캐시 파일 구조 정의
            cache_paths = {
                # 기본 필수 파일들 (prefix: 000_)
                'metadata': os.path.join(base_dir, '000_metadata.json'),  # 000_ 접두사 추가
                'business_info': os.path.join(base_dir, '000_business_info.json'),
                'competitors': os.path.join(base_dir, '000_competitors.json'),
                'sequence': os.path.join(base_dir, '000_sequence.json'),
                
                # 섹션별 파일들은 sequence 번호로 시작
                'cache_dir': base_dir
            }
            
            return cache_paths
            
        except Exception as e:
            self.logger.error(f"캐시 디렉토리 초기화 실패: {str(e)}")
            raise

    def _get_next_sequence(self, cache_paths: Dict, section_type: str) -> int:
        """새로운 파일명 규칙에 맞는 다음 시퀀스 번호 생성"""
        try:
            cache_dir = cache_paths['cache_dir']
            
            # 파일 패턴에서 시퀀스 추출
            existing_files = []
            for ext in ['.json', '.png', '.csv']:
                existing_files.extend(list(Path(cache_dir).glob(f'*-*-*-*-{section_type}-*{ext}')))
            
            max_sequence = 0
            for file in existing_files:
                try:
                    # 파일명 분해: section-subsection-name-subname-type-sequence
                    parts = file.stem.split('-')
                    if len(parts) >= 6:  # 최소 6개 부분이 있어야 함
                        section_num = int(parts[0])
                        subsection_num = int(parts[1])
                        sequence_num = int(parts[-1])
                        # 섹션과 하위섹션 번호를 고려한 시퀀스 계산
                        combined_sequence = (section_num * 100 + subsection_num) * 1000 + sequence_num
                        max_sequence = max(max_sequence, combined_sequence)
                except (ValueError, IndexError):
                    continue
            
            # 다음 시퀀스 번호는 현재 섹션/하위섹션 내에서만 증가
            next_sequence = (max_sequence % 1000) + 1
            
            self.logger.debug(f"다음 시퀀스 번호: {next_sequence} (섹션: {section_type})")
            return next_sequence
            
        except Exception as e:
            self.logger.error(f"시퀀스 생성 실패: {str(e)}")
            return 1  # 오류 시 기본값 반환

    def _get_file_patterns(self, section_name: str, subsection_name: str, section_order: str, subsection_order: str) -> Dict[str, str]:
        """파일 패턴 생성"""
        base = f"*_{section_order}_{subsection_order}_{self._sanitize_filename(section_name)}_{self._sanitize_filename(subsection_name)}"
        return {
            'content': f"{base}_content.json",
            'chart': f"{base}_chart_*.png",
            'chart_meta': f"{base}_chart_*_meta.json",
            'table_csv': f"{base}_table.csv",
            'table_json': f"{base}_table.json"
        }

    def _save_section_data(self, section_data: Dict, section_name: str, subsection_name: str, cache_paths: Dict) -> bool:
        """섹션 데이터를 캐시에 저장하고 문서 폴더로 복사"""
        try:
            # 문서 폴더 경로 설정
            doc_dir = os.path.join(self.output_dir)
            BaseServiceClient.safe_makedirs(doc_dir)

            # 1. 섹션 내용 저장
            if 'content' in section_data:
                sequence = str(self._get_next_sequence(cache_paths, f"{section_name}_{subsection_name}_content"))  # 문자열로 변환
                base_filename = self._generate_filename(section_name, subsection_name, 'content', sequence)
                content_path = os.path.join(cache_paths['cache_dir'], f"{base_filename}.json")
                
                with open(content_path, 'w', encoding='utf-8') as f:
                    content_to_save = {
                        'content': section_data['content'],
                        'section_name': section_name,
                        'subsection_name': subsection_name,
                        'created_at': datetime.now().isoformat()
                    }
                    sanitized_content = self._sanitize_for_json(content_to_save)
                    json.dump(sanitized_content, f, ensure_ascii=False, indent=2)

            # 2. 차트 생성 및 저장
            if 'chart_data' in section_data:
                sequence = str(self._get_next_sequence(cache_paths, f"{section_name}_{subsection_name}_chart"))  # 문자열로 변환
                base_filename = self._generate_filename(section_name, subsection_name, 'chart', sequence)
                
                # 차트 이미지 저장
                cache_chart_path = self.create_chart(
                    data=section_data['chart_data'],
                    chart_type=section_data['chart_data'].get('type', 'bar'),
                    output_dir=cache_paths['cache_dir'],
                    timestamp='',  # 새로운 명명규칙에서는 timestamp 제외
                    prefix=base_filename
                )
                
                if cache_chart_path and os.path.exists(cache_chart_path):
                    # 문서 폴더로 차트 이미지 복사
                    doc_chart_path = os.path.join(doc_dir, os.path.basename(cache_chart_path))
                    # shutil.copy2(cache_chart_path, doc_chart_path)
                    # self.logger.debug(f"차트 이미지 복사됨: {doc_chart_path}")
                    
                    # 차트 메타데이터 저장
                    chartdata_filename = self._generate_filename(section_name, subsection_name, 'chartdata', sequence)
                    chart_meta_path = os.path.join(cache_paths['cache_dir'], f"{chartdata_filename}.json")
                    
                    with open(chart_meta_path, 'w', encoding='utf-8') as f:
                        chart_meta_data = {
                            'cache_path': cache_chart_path,
                            'doc_path': doc_chart_path,
                            'created_at': datetime.now().isoformat(),
                            'chart_data': section_data['chart_data']
                        }
                        sanitized_chart_meta = self._sanitize_for_json(chart_meta_data)
                        json.dump(sanitized_chart_meta, f, ensure_ascii=False, indent=2)

            # 3. 테이블 저장
            if 'table' in section_data:
                sequence = str(self._get_next_sequence(cache_paths, f"{section_name}_{subsection_name}_table"))  # 문자열로 변환
                table_filename = self._generate_filename(section_name, subsection_name, 'table', sequence)
                tabledata_filename = self._generate_filename(section_name, subsection_name, 'tabledata', sequence)
                
                # CSV 파일 저장
                cache_csv_path = os.path.join(cache_paths['cache_dir'], f"{table_filename}.csv")
                with open(cache_csv_path, 'w', encoding='utf-8-sig', newline='') as f:
                    writer = csv.writer(f)
                    writer.writerow(section_data['table']['headers'])
                    writer.writerows(section_data['table']['rows'])
                
                # 문서 폴더로 CSV 파일 복사
                doc_csv_path = os.path.join(doc_dir, os.path.basename(cache_csv_path))
                # shutil.copy2(cache_csv_path, doc_csv_path)
                # self.logger.debug(f"CSV 파일 복사됨: {doc_csv_path}")
                
                # 테이블 메타데이터 저장
                table_meta_path = os.path.join(cache_paths['cache_dir'], f"{tabledata_filename}.json")
                with open(table_meta_path, 'w', encoding='utf-8') as f:
                    table_meta_data = {
                        'table': section_data['table'],
                        'cache_path': cache_csv_path,
                        'doc_path': doc_csv_path,
                        'created_at': datetime.now().isoformat()
                    }
                    sanitized_table_meta = self._sanitize_for_json(table_meta_data)
                    json.dump(sanitized_table_meta, f, ensure_ascii=False, indent=2)

            # 4. 다이어그램 저장
            if 'diagram_data' in section_data:
                sequence = str(self._get_next_sequence(cache_paths, f"{section_name}_{subsection_name}_diagram"))  # 문자열로 변환
                base_filename = self._generate_filename(section_name, subsection_name, 'diagram', sequence)
                
                # 다이어그램 이미지 저장
                cache_diagram_path = self.create_diagram(
                    data=section_data['diagram_data'],  # diagram_data는 이미 diagram 객체임
                    output_dir=cache_paths['cache_dir'],
                    timestamp='',
                    prefix=base_filename
                )
                
                if cache_diagram_path and os.path.exists(cache_diagram_path):
                    # 문서 폴더로 다이어그램 이미지와 SVG 복사
                    doc_diagram_path = os.path.join(doc_dir, os.path.basename(cache_diagram_path))
                    # shutil.copy2(cache_diagram_path, doc_diagram_path)
                    # self.logger.debug(f"다이어그램 이미지 복사됨: {doc_diagram_path}")
                    
                    # SVG 파일이 있다면 복사
                    cache_svg_path = cache_diagram_path.rsplit('.', 1)[0] + '.svg'
                    if os.path.exists(cache_svg_path):
                        doc_svg_path = os.path.join(doc_dir, os.path.basename(cache_svg_path))
                        # shutil.copy2(cache_svg_path, doc_svg_path)
                        # self.logger.debug(f"SVG 파일 복사됨: {doc_svg_path}")
                    
                    # DOT 파일이 있다면 복사
                    cache_dot_path = cache_diagram_path.rsplit('.', 1)[0] + '.dot'
                    if os.path.exists(cache_dot_path):
                        doc_dot_path = os.path.join(doc_dir, os.path.basename(cache_dot_path))
                        # shutil.copy2(cache_dot_path, doc_dot_path)
                        # self.logger.debug(f"DOT 파일 복사됨: {doc_dot_path}")
                    
                    # 다이어그램 메타데이터 저장
                    diagramdata_filename = self._generate_filename(section_name, subsection_name, 'diagramdata', sequence)
                    diagram_meta_path = os.path.join(cache_paths['cache_dir'], f"{diagramdata_filename}.json")
                    
                    with open(diagram_meta_path, 'w', encoding='utf-8') as f:
                        diagram_meta_data = {
                            'diagram_data': section_data['diagram_data'],
                            'created_at': datetime.now().isoformat(),
                            'cache_paths': {
                                'png': cache_diagram_path,
                                'svg': cache_svg_path if os.path.exists(cache_svg_path) else None,
                                'dot': cache_dot_path if os.path.exists(cache_dot_path) else None
                            },
                            'doc_paths': {
                                'png': doc_diagram_path,
                                'svg': doc_svg_path if os.path.exists(cache_svg_path) else None,
                                'dot': doc_dot_path if os.path.exists(cache_dot_path) else None
                            }
                        }
                        sanitized_diagram_meta = self._sanitize_for_json(diagram_meta_data)
                        json.dump(sanitized_diagram_meta, f, ensure_ascii=False, indent=2)

            # 5. 시퀀스 다이어그램 저장
            if 'sequence_data' in section_data:
                sequence = str(self._get_next_sequence(cache_paths, f"{section_name}_{subsection_name}_sequence"))  # 문자열로 변환
                base_filename = self._generate_filename(section_name, subsection_name, 'sequence', sequence)
                
                # 시퀀스 다이어그램 이미지 저장
                sequence_path = os.path.join(cache_paths['cache_dir'], f"{base_filename}.png")
                generated_path = self.create_mermaid_diagram(
                    mermaid_code=section_data['sequence_data']['mermaid_code'],
                    output_path=sequence_path,
                    diagram_type='sequence'  # 다이어그램 타입 명시
                )
                
                if generated_path and os.path.exists(generated_path):
                    # 문서 폴더로 시퀀스 다이어그램 이미지 복사
                    doc_sequence_path = os.path.join(doc_dir, os.path.basename(generated_path))
                    # shutil.copy2(generated_path, doc_sequence_path)
                    # self.logger.debug(f"시퀀스 다이어그램 복사됨: {doc_sequence_path}")
                    
                    # Mermaid 소스 파일이 있다면 복사
                    cache_mmd_path = generated_path.rsplit('.', 1)[0] + '.mmd'
                    if os.path.exists(cache_mmd_path):
                        doc_mmd_path = os.path.join(doc_dir, os.path.basename(cache_mmd_path))
                        # shutil.copy2(cache_mmd_path, doc_mmd_path)
                        # self.logger.debug(f"Mermaid 소스 파일 복사됨: {doc_mmd_path}")
                    
                    # 시퀀스 다이어그램 메타데이터 저장
                    sequencedata_filename = self._generate_filename(section_name, subsection_name, 'sequencedata', sequence)
                    sequence_meta_path = os.path.join(cache_paths['cache_dir'], f"{sequencedata_filename}.json")
                    
                    with open(sequence_meta_path, 'w', encoding='utf-8') as f:
                        sequence_meta_data = {
                            'sequence_data': section_data['sequence_data'],
                            'created_at': datetime.now().isoformat(),
                            'cache_paths': {
                                'png': generated_path,
                                'mmd': cache_mmd_path if os.path.exists(cache_mmd_path) else None
                            },
                            'doc_paths': {
                                'png': doc_sequence_path,
                                'mmd': doc_mmd_path if os.path.exists(cache_mmd_path) else None
                            }
                        }
                        sanitized_sequence_meta = self._sanitize_for_json(sequence_meta_data)
                        json.dump(sanitized_sequence_meta, f, ensure_ascii=False, indent=2)
                        

            # 6. 플로우차트 다이어그램 저장 (추가)
            if 'flowchart_data' in section_data:
                sequence = str(self._get_next_sequence(cache_paths, f"{section_name}_{subsection_name}_flowchart"))
                base_filename = self._generate_filename(section_name, subsection_name, 'flowchart', sequence)
                
                # 플로우차트 다이어그램 이미지 저장
                flowchart_path = os.path.join(cache_paths['cache_dir'], f"{base_filename}.png")
                generated_path = self.create_mermaid_diagram(
                    mermaid_code=section_data['flowchart_data']['mermaid_code'],
                    output_path=flowchart_path,
                    diagram_type='flowchart'  # 다이어그램 타입 명시
                )
                
                # 메타데이터 저장 및 처리...
                if generated_path and os.path.exists(generated_path):
                    # 문서 폴더로 시퀀스 다이어그램 이미지 복사
                    doc_flowchart_path = os.path.join(doc_dir, os.path.basename(generated_path))
                    # shutil.copy2(generated_path, doc_flowchart_path)
                    # self.logger.debug(f"시퀀스 다이어그램 복사됨: {doc_flowchart_path}")
                    
                    # Mermaid 소스 파일이 있다면 복사
                    cache_mmd_path = generated_path.rsplit('.', 1)[0] + '.mmd'
                    if os.path.exists(cache_mmd_path):
                        doc_mmd_path = os.path.join(doc_dir, os.path.basename(cache_mmd_path))
                        # shutil.copy2(cache_mmd_path, doc_mmd_path)
                        # self.logger.debug(f"Mermaid 소스 파일 복사됨: {doc_mmd_path}")
                    
                    # 다이어그램 메타데이터 저장
                    flowchartdata_filename = self._generate_filename(section_name, subsection_name, 'flowchartdata', sequence)
                    flowchart_meta_path = os.path.join(cache_paths['cache_dir'], f"{flowchartdata_filename}.json")
                    
                    with open(flowchart_meta_path, 'w', encoding='utf-8') as f:
                        json.dump({
                            'flowchart_data': section_data['flowchart_data'],
                            'created_at': datetime.now().isoformat(),
                            'cache_paths': {
                                'png': generated_path,
                                'mmd': cache_mmd_path if os.path.exists(cache_mmd_path) else None
                            },
                            'doc_paths': {
                                'png': doc_flowchart_path,
                                'mmd': doc_mmd_path if os.path.exists(cache_mmd_path) else None
                            }
                        }, f, ensure_ascii=False, indent=2)


            # 7. 간트 차트 저장 (추가)
            if 'gantt_data' in section_data:
                sequence = str(self._get_next_sequence(cache_paths, f"{section_name}_{subsection_name}_gantt"))
                base_filename = self._generate_filename(section_name, subsection_name, 'gantt', sequence)
                
                # 간트 차트 이미지 저장
                gantt_path = os.path.join(cache_paths['cache_dir'], f"{base_filename}.png")
                generated_path = self.create_mermaid_diagram(
                    mermaid_code=section_data['gantt_data']['mermaid_code'],
                    output_path=gantt_path,
                    diagram_type='gantt'  # 다이어그램 타입 명시
                )
                
                # 메타데이터 저장 및 처리...
                if generated_path and os.path.exists(generated_path):
                    # 문서 폴더로 시퀀스 다이어그램 이미지 복사
                    doc_gantt_path = os.path.join(doc_dir, os.path.basename(generated_path))
                    # shutil.copy2(generated_path, doc_gantt_path)
                    # self.logger.debug(f"시퀀스 다이어그램 복사됨: {doc_gantt_path}")
                    
                    # Mermaid 소스 파일이 있다면 복사
                    cache_mmd_path = generated_path.rsplit('.', 1)[0] + '.mmd'
                    if os.path.exists(cache_mmd_path):
                        doc_mmd_path = os.path.join(doc_dir, os.path.basename(cache_mmd_path))
                        # shutil.copy2(cache_mmd_path, doc_mmd_path)
                        # self.logger.debug(f"Mermaid 소스 파일 복사됨: {doc_mmd_path}")
                    
                    # 시퀀스 다이어그램 메타데이터 저장
                    ganttdata_filename = self._generate_filename(section_name, subsection_name, 'ganttdata', sequence)
                    gantt_meta_path = os.path.join(cache_paths['cache_dir'], f"{ganttdata_filename}.json")
                    
                    with open(gantt_meta_path, 'w', encoding='utf-8') as f:
                        json.dump({
                            'gantt_data': section_data['gantt_data'],
                            'created_at': datetime.now().isoformat(),
                            'cache_paths': {
                                'png': generated_path,
                                'mmd': cache_mmd_path if os.path.exists(cache_mmd_path) else None
                            },
                            'doc_paths': {
                                'png': doc_gantt_path,
                                'mmd': doc_mmd_path if os.path.exists(cache_mmd_path) else None
                            }
                        }, f, ensure_ascii=False, indent=2)
                        

            return True

        except Exception as e:
            self.logger.error(f"섹션 데이터 저장 실패: {str(e)}")
            raise

    def _load_cached_data(self, cache_dir: str) -> Dict:
        """캐시된 데이터 로드"""
        cached_data = {}
        try:
            # 1. 기본 JSON 파일들 로드
            for file_name in ['000_business_info.json', '000_competitors.json', '000_metadata.json']:
                file_path = os.path.join(cache_dir, file_name)
                if os.path.exists(file_path):
                    with open(file_path, 'r', encoding='utf-8') as f:
                        data = json.load(f)
                        key = file_name.replace('000_', '').replace('.json', '')
                        if key == 'business_info':
                            cached_data[key] = data.get('business_info', {})
                        else:
                            cached_data[key] = data

            # 2. 섹션 데이터 로드
            sections = {}
            content_files = list(Path(cache_dir).glob('*-*-*-*-content-*.json'))
            
            # 섹션 순서대로 처리하기 위한 정렬된 섹션 목록 생성
            sorted_section_orders = sorted(self.section_order.items(), key=lambda x: x[1])
            
            for section_name, section_order in sorted_section_orders:
                if section_name not in sections:
                    sections[section_name] = {}
                
                # 해당 섹션의 하위섹션 순서 가져오기
                subsection_orders = self.subsection_order.get(section_name, {})
                sorted_subsection_orders = sorted(subsection_orders.items(), key=lambda x: x[1])
                
                for subsection_name, subsection_order in sorted_subsection_orders:
                    # 파일 패턴 구성
                    file_pattern = f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}"
                    
                    # 1. content.json 파일 찾기
                    content_files = list(Path(cache_dir).glob(f"{file_pattern}-content-*.json"))
                    if content_files:
                        content_file = sorted(content_files, key=lambda x: int(x.stem.split('-')[-1]))[-1]
                        try:
                            with open(content_file, 'r', encoding='utf-8') as f:
                                content_data = json.load(f)
                                section_data = {'content': content_data.get('content', '')}
                                
                                # 2. 차트 파일 찾기
                                chart_files = list(Path(cache_dir).glob(f"{file_pattern}-chart-*.png"))
                                if chart_files:
                                    chartdata_files = list(Path(cache_dir).glob(f"{file_pattern}-chartdata-*.json"))
                                    if chartdata_files:
                                        with open(chartdata_files[0], 'r', encoding='utf-8') as f:
                                            chart_meta = json.load(f)
                                            section_data['chart_data'] = chart_meta.get('chart_data')
                                
                                # 3. 테이블 파일 찾기
                                table_files = list(Path(cache_dir).glob(f"{file_pattern}-table-*.csv"))
                                if table_files:
                                    tabledata_files = list(Path(cache_dir).glob(f"{file_pattern}-tabledata-*.json"))
                                    if tabledata_files:
                                        with open(tabledata_files[0], 'r', encoding='utf-8') as f:
                                            table_data = json.load(f)
                                            section_data['table'] = table_data.get('table')
                                
                                sections[section_name][subsection_name] = section_data
                                
                        except Exception as e:
                            self.logger.error(f"섹션 파일 처리 중 오류: {str(e)}")
                            continue

            if sections:
                cached_data['sections'] = sections
                
            # self.logger.debug(f"로드된 캐시 데이터 키: {list(cached_data.keys())}")
            # if 'sections' in cached_data:
            #     self.logger.debug(f"로드된 섹션: {list(cached_data['sections'].keys())}")
                
            return cached_data

        except Exception as e:
            self.logger.error(f"캐시 데이터 로드 실패: {str(e)}")
            return {}

    def _save_to_cache(self, data: Dict, cache_paths: Dict):
        """데이터를 캐시에 저장"""
        try:
            # 비즈니스 정보 저장
            if 'business_info' in data:
                sanitized_data = self._sanitize_for_json(data['business_info'])
                with open(os.path.join(cache_paths['business_info']), 'w', encoding='utf-8') as f:
                    json.dump(sanitized_data, f, ensure_ascii=False, indent=2)

            # 경쟁사 정보 저장
            if 'competitors' in data:
                sanitized_data = self._sanitize_for_json(data['competitors'])
                with open(os.path.join(cache_paths['competitors']), 'w', encoding='utf-8') as f:
                    json.dump(sanitized_data, f, ensure_ascii=False, indent=2)

            # 섹션 데이터 저장
            if 'sections' in data:
                sanitized_data = self._sanitize_for_json(data['sections'])
                with open(os.path.join(cache_paths['sections']), 'w', encoding='utf-8') as f:
                    json.dump(sanitized_data, f, ensure_ascii=False, indent=2)

            # 차트 이미지 저장
            if 'charts' in data:
                for chart_name, chart_data in data['charts'].items():
                    chart_path = os.path.join(cache_paths['charts'], f"{chart_name}.png")
                    plt.savefig(chart_path, dpi=300, bbox_inches='tight')
                    plt.close()

            # 테이블 데이터 저장
            if 'tables' in data:
                for table_name, table_data in data['tables'].items():
                    table_path = os.path.join(cache_paths['tables'], f"{table_name}.json")
                    sanitized_table_data = self._sanitize_for_json(table_data)
                    with open(table_path, 'w', encoding='utf-8') as f:
                        json.dump(sanitized_table_data, f, ensure_ascii=False, indent=2)

        except Exception as e:
            self.logger.error(f"캐시 저장 실패: {str(e)}")
            raise

    async def call_ai_api(self, prompt: str, require_json: bool = False, json_schema: Dict = None, system_message: str = "", max_retries: int = 3, provider: str = None, model: str = None) -> Union[str, Dict]:
        """
        AI API를 호출하여 응답을 받아오는 메소드
        
        Args:
            prompt: 프롬프트 문자열
            require_json: JSON 응답 필요 여부
            json_schema: JSON 스키마 (선택 사항)
            system_message: 시스템 메시지 (선택 사항)
            max_retries: JSON 검증 실패 시 최대 재시도 횟수
            provider: AI 프로바이더 (선택 사항, 없으면 설정파일 사용)
            model: AI 모델 (선택 사항, 없으면 설정파일 사용)
            
        Returns:
            Union[str, Dict]: AI 응답 (JSON 요청 시 Dict, 아닐 경우 str)
            
        Raises:
            Exception: API 호출 실패 시
        """
        # prompt 매개변수가 SectionConfig 객체인 경우 안전하게 처리
        if isinstance(prompt, SectionConfig):
            prompt = prompt.prompt if prompt.prompt is not None else ""
        elif prompt is None:
            prompt = ""
        
        # prompt가 여전히 문자열이 아닌 경우 강제 변환
        if not isinstance(prompt, str):
            prompt = str(prompt) if prompt is not None else ""
        
        retry_count = 0
        last_error = None
        
        while retry_count < max_retries:
            try:
                # AI 설정 가져오기 - 매개변수 → 사용자 설정 → 환경 변수 순으로 우선
                config = load_config()
                if provider is None:
                    provider = self.user_provider or config.get('USE_LLM', 'openai')
                if model is None:
                    if provider == 'ollama':
                        model = self.user_model or config.get(f'{provider.upper()}_MODEL', 'hamonize:latest')
                    else:
                        model = self.user_model or config.get(f'{provider.upper()}_MODEL', 'gpt-4o-mini')
                
                # JSON 응답이 필요한 경우 구조화 출력에 최적화된 모델 사용
                if require_json and provider == 'ollama':
                    model = 'airun-chat:latest'  # JSON 구조화 출력이 안정적인 모델로 강제 설정
                    self.logger.debug(f"JSON 응답 요청으로 모델을 {model}로 변경")
                
                # 프로바이더와 모델 정보 로깅
                # self.logger.debug(f"AI 호출: 프로바이더={provider}, 모델={model}")       
                
                # Ollama와 VLLM은 API 키가 필요 없으므로 예외 처리
                if provider not in ['ollama', 'vllm']:
                    api_key = config.get(f'{provider.upper()}_API_KEY')
                    if not api_key:
                        raise ValueError(f"{provider.upper()}_API_KEY가 설정되지 않았습니다.")
                else:
                    api_key = None  # Ollama와 VLLM은 API 키가 필요 없음

                # JSON 응답이 필요한 경우 프롬프트 수정
                if require_json:
                    prompt = f"""Please provide your response in JSON format.

{prompt}

중요: 응답은 반드시 유효한 JSON 형식이어야 합니다. 다른 텍스트 없이 JSON만 반환하세요."""

                # 메시지 구성
                messages = []
                if system_message:
                    messages.append({"role": "system", "content": system_message})
                else:
                    messages.append({"role": "system", "content": "당신은 도움이 되는 어시스턴트입니다."})
                
                # JSON 요청인 경우 시스템 메시지에 추가 지시사항 포함
                if require_json and messages[0]["role"] == "system":
                    messages[0]["content"] += "\n\n중요: 응답은 반드시 유효한 JSON 형식이어야 합니다. 다른 텍스트 없이 JSON만 반환하세요."
                
                messages.append({"role": "user", "content": prompt})
                
                # 요청 내용 로깅
                self.logger.info("=== AI 요청 내용 ===")
                for msg in messages:
                    self.logger.debug(f"[{msg['role']}]: {msg['content'][:500]}..." if len(msg['content']) > 500 else f"[{msg['role']}]: {msg['content']}")

                # 프로바이더별 API 호출
                if provider == 'openai':
                    client = OpenAI(api_key=api_key)
                    params = {
                        "model": model,
                        "messages": messages,
                        "temperature": 0.7
                    }
                    
                    # JSON 응답 형식 설정
                    if require_json:
                        try:
                            params["response_format"] = {"type": "json_object"}
                            self.logger.debug("OpenAI API에 response_format=json_object 설정 적용")
                        except Exception as e:
                            self.logger.warning(f"response_format 설정 실패: {str(e)}")
                    
                    response = await asyncio.get_event_loop().run_in_executor(
                        None,
                        lambda: client.chat.completions.create(**params)
                    )
                    result = response.choices[0].message.content
                    
                    # 응답 내용 로깅
                    self.logger.info("=== AI 응답 내용 ===")
                    self.logger.info(f"응답 길이: {len(result)} 자")
                    self.logger.info(result[:2000] + ("..." if len(result) > 2000 else ""))
                    self.logger.info("==================")

                    # 토큰 사용량 추적
                    if hasattr(response, 'usage'):
                        self.usage_tracker.track_usage(
                            provider=provider,
                            model=model,
                            prompt_tokens=response.usage.prompt_tokens,
                            completion_tokens=response.usage.completion_tokens
                        )
                    else:
                        # 토큰 수 추정
                        prompt_tokens = len(prompt.split()) * 1.3
                        completion_tokens = len(result.split()) * 1.3
                        self.usage_tracker.track_usage(
                            provider=provider,
                            model=model,
                            prompt_tokens=int(prompt_tokens),
                            completion_tokens=int(completion_tokens)
                        )

                elif provider == 'anthropic':
                    import anthropic
                    client = anthropic.Anthropic(api_key=api_key)
                    response = await asyncio.get_event_loop().run_in_executor(
                        None,
                        lambda: client.messages.create(
                            model=model,
                            messages=messages,
                            temperature=0.7,
                            max_tokens=4000
                        )
                    )
                    result = response.content[0].text

                    # 토큰 수 추정
                    prompt_tokens = len(prompt.split()) * 1.3
                    completion_tokens = len(result.split()) * 1.3
                    self.usage_tracker.track_usage(
                        provider=provider,
                        model=model,
                        prompt_tokens=int(prompt_tokens),
                        completion_tokens=int(completion_tokens)
                    )

                elif provider == 'gemini':
                    import google.generativeai as genai
                    genai.configure(api_key=api_key)
                    model_instance = genai.GenerativeModel(model)
                    
                    # JSON 응답이 필요한 경우 generation_config에 추가 설정
                    generation_config = genai.types.GenerationConfig(
                        temperature=0.7,
                        candidate_count=1
                    )
                    
                    # 프롬프트에 JSON 요구사항 강화
                    if require_json:
                        prompt = f"""Please provide your response in JSON format following the exact schema below.
The response must be a valid JSON object with no additional text.

Schema:
{json.dumps(json_schema, ensure_ascii=False, indent=2) if json_schema else "{}"}

Request:
{prompt}

Important:
1. Response must be a valid JSON object
2. Do not include any text outside the JSON
3. Use the exact field names as specified in the schema
4. All text fields must be in Korean
"""
                 
                    response = await asyncio.get_event_loop().run_in_executor(
                        None,
                        lambda: model_instance.generate_content(
                            prompt,
                            generation_config=generation_config
                        )
                    )
                    result = response.text

                    # 토큰 수 추정
                    prompt_tokens = len(prompt.split()) * 1.3
                    completion_tokens = len(result.split()) * 1.3
                    self.usage_tracker.track_usage(
                        provider=provider,
                        model=model,
                        prompt_tokens=int(prompt_tokens),
                        completion_tokens=int(completion_tokens)
                    )

                elif provider == 'groq':
                    from groq import Groq
                    client = Groq(api_key=api_key)
                    response = await asyncio.get_event_loop().run_in_executor(
                        None,
                        lambda: client.chat.completions.create(
                            model=model,
                            messages=messages,
                            temperature=0.7
                        )
                    )
                    result = response.choices[0].message.content

                    # 토큰 수 추정
                    prompt_tokens = len(prompt.split()) * 1.3
                    completion_tokens = len(result.split()) * 1.3
                    self.usage_tracker.track_usage(
                        provider=provider,
                        model=model,
                        prompt_tokens=int(prompt_tokens),
                        completion_tokens=int(completion_tokens)
                    )

                elif provider == 'ollama':
                    import requests
                    ollama_url = config.get('OLLAMA_PROXY_SERVER', 'http://127.0.0.1:11434')
                    
                    # URL 끝에 슬래시가 있는지 확인하고 정규화
                    if ollama_url.endswith('/'):
                        ollama_url = ollama_url[:-1]
                    
                    # 재시도 설정
                    max_retries = 3
                    retry_delay = 2  # 초 단위
                    
                    for retry_count in range(max_retries):
                        
                        try:
                            # API 엔드포인트 구성
                            api_endpoint = f"{ollama_url}/api/chat"
                            
                            self.logger.info(f"AI 호출: {api_endpoint}, 모델: {model}, 시도: {retry_count + 1}/{max_retries}")
                            
                            # Ollama에서는 JSON 스키마를 시스템 메시지에 직접 추가하는 대신 사용자 메시지에 추가
                            ollama_messages = messages.copy()
                            if require_json and json_schema:
                                # 마지막 사용자 메시지를 찾아서 JSON 스키마 정보 추가
                                for i in range(len(ollama_messages) - 1, -1, -1):
                                    if ollama_messages[i]["role"] == "user":
                                        ollama_messages[i]["content"] += "\n\n다음 JSON 스키마를 정확히 따라주세요:\n" + json.dumps(json_schema, ensure_ascii=False, indent=2)
                                        ollama_messages[i]["content"] += "\n\n중요: 응답은 반드시 유효한 JSON 형식이어야 합니다. 다른 텍스트 없이 JSON만 반환하세요. 모든 응답은 한국어로 하세요."
                                        break
                            
                            # 디버그 로깅 추가
                            self.logger.debug(f"=== system message ===")
                            for msg in ollama_messages:
                                if msg["role"] == "system":
                                    self.logger.debug(msg["content"])
                                    break
                            
                            self.logger.debug(f"=== user message ===")
                            for msg in ollama_messages:
                                if msg["role"] == "user":
                                    self.logger.debug(msg["content"])
                                    break

                            # 모델별 컨텍스트 길이 (고정값 사용 - 동적 조정 제거)
                            MODEL_CONTEXT_LENGTHS = {
                                "exaone3.5": 32768,
                                "qwen2.5":  32768,
                                "mistral":  32768,
                                "deepseek-r1": 131072,
                                "solar":    4096,
                                "llama3.2": 131072,
                                "gpt-oss":  131072, 
                                "hamonize":  131072,  # 고정값 - 서빙 모델과 일치
                                "default":  4096,
                            }

                            
                          
                            
                            # 모델 이름에서 기본 모델 타입 추출 함수, 등록되지 않는 모델은 Default
                            def get_base_model_name(model_name: str) -> str:
                                import re
                                name = (model_name or "").lower()
                                # ":latest" 같은 태그 제거
                                name = re.sub(r":[^/]+$", "", name)

                                known = set(MODEL_CONTEXT_LENGTHS.keys())
                                # 더 긴 문자열 우선 매칭(부분 겹침 대비)
                                for base in sorted(known, key=len, reverse=True):
                                    if base in name:
                                        return base
                                return "default"
                            
                            def pick_context_length(base_model: str,
                                                messages: list[dict],
                                                require_json: bool = False,
                                                expected_output_tokens: int = 1024) -> int:
                                # 고정된 컨텍스트 길이 사용 (동적 조정 제거)
                                return MODEL_CONTEXT_LENGTHS.get(base_model, MODEL_CONTEXT_LENGTHS["default"])

                            # 요청 데이터 구성 부분 수정
                            base_model = get_base_model_name(model)
                            # context_length2 = MODEL_CONTEXT_LENGTHS.get(base_model, MODEL_CONTEXT_LENGTHS["default"])
                            context_length = pick_context_length(
                                base_model=base_model,
                                messages=ollama_messages,
                                require_json=require_json,
                                expected_output_tokens=1024
                            )
                            

                            def _clean_code_fence(s: str) -> str:
                                s = (s or "").strip()
                                if s.startswith("```"):
                                    s = s.strip("`")
                                    if "\n" in s:
                                        first, rest = s.split("\n", 1)
                                        if first.strip().lower() == "json":
                                            s = rest
                                return s.strip()

                            def _ollama_chat_once(endpoint: str, body: dict, timeout: int = 300) -> str:
                                resp = requests.post(
                                    endpoint,
                                    json=body,
                                    headers={"Accept": "application/json"},
                                    timeout=timeout,
                                )
                                resp.raise_for_status()
                                data = resp.json()
                                # content가 비면 response도 폴백
                                return (data.get("message") or {}).get("content") or data.get("response") or ""

                            def _ollama_chat_stream(endpoint: str, body: dict, timeout=(10, 600)) -> str:
                                stream_body = dict(body); stream_body["stream"] = True
                                resp = requests.post(
                                    endpoint,
                                    json=stream_body,
                                    headers={"Accept": "application/x-ndjson"},
                                    stream=True,
                                    timeout=timeout,
                                )
                                resp.raise_for_status()
                                resp.encoding = "utf-8"

                                parts, first_lines, last_lines = [], [], []
                                for line in resp.iter_lines(decode_unicode=True):
                                    if not line:
                                        continue
                                    # 샘플 로깅 버퍼
                                    if len(first_lines) < 3:
                                        first_lines.append(line)
                                    else:
                                        if len(last_lines) >= 3:
                                            last_lines.pop(0)
                                        last_lines.append(line)

                                    try:
                                        obj = json.loads(line)
                                    except Exception:
                                        continue

                                    msg = obj.get("message") or {}
                                    piece = ""
                                    c = msg.get("content")
                                    if isinstance(c, str):
                                        piece += c
                                    r = obj.get("response")
                                    if isinstance(r, str):
                                        piece += r

                                    if piece:
                                        parts.append(piece)

                                # 디버그: 앞/뒤 샘플 라인 찍기
                                self.logger.debug(f"[ollama stream] first_lines={first_lines}")
                                self.logger.debug(f"[ollama stream] last_lines={last_lines}")

                                return "".join(parts)
                            
                            self.logger.info(f"[ollama] (model={model}, (base_model={base_model}) ctx={context_length})")
                            # self.logger.info(f"ollama_messages={ollama_messages}")

                            request_data = {
                                "model": model,
                                "messages": ollama_messages,
                                "stream": False, 
                                "options": {
                                    "temperature": 0.3 if require_json else 0.7,
                                    "num_ctx": context_length,
                                },
                            }

                            self.logger.debug("=== request data ===")
                            self.logger.debug(json.dumps(request_data, ensure_ascii=False, indent=2))

                            # 1) non-stream
                            text = await asyncio.to_thread(_ollama_chat_once, api_endpoint, request_data, 300)
                            if not text:
                                self.logger.info("[ollama gpt-oss] empty content -> fallback to stream")
                                # 2) stream fallback
                                text = await asyncio.to_thread(_ollama_chat_stream, api_endpoint, request_data, (10, 600))

                            result = text or ""
                            # 2) JSON 강제 시: 코드펜스 제거 + 검증
                            if require_json and result:
                                cleaned = _clean_code_fence(result)
                                try:
                                    json.loads(cleaned)
                                    result = cleaned
                                except Exception as e:
                                    self.logger.warning(f"[ollama gpt-oss] JSON parse failed; returning raw. err={e}")

                            if not result:
                                raise ValueError("응답이 비어있습니다")

                            self.logger.info("=== ollama AI 응답 ===")
                            self.logger.info(f"응답 길이: {len(result)} 자")
                            self.logger.debug(result[:1000] + ("..." if len(result) > 1000 else ""))
                            
                            # 토큰 수 추정
                            prompt_tokens = len(prompt.split()) * 1.3
                            completion_tokens = len(result.split()) * 1.3
                            self.usage_tracker.track_usage(
                                provider=provider,
                                model=model,
                                prompt_tokens=int(prompt_tokens),
                                completion_tokens=int(completion_tokens)
                            )
                            
                            # 성공적으로 응답을 받았으면 재시도 루프 종료
                            break
                            
                        except Exception as e:
                            self.logger.error(f"AI 호출 실패 (시도 {retry_count + 1}/{max_retries}): {str(e)}")
                            
                            # 재시도 가능한 오류인지 확인
                            retry_error = False
                            error_str = str(e).lower()
                            
                            # 네트워크 관련 오류 확인
                            if any(err in error_str for err in ['timeout', 'connection', 'network', 'reset', 'refused', 'eof']):
                                retry_error = True
                            
                            # 마지막 시도가 아니고 재시도 가능한 오류인 경우
                            if retry_count < max_retries - 1 and retry_error:
                                wait_time = retry_delay * (2 ** retry_count)  # 지수 백오프
                                self.logger.info(f"재시도 대기 중... {wait_time}초")
                                await asyncio.sleep(wait_time)
                                continue
                            else:
                                # 모든 재시도 실패 또는 재시도 불가능한 오류
                                raise

                elif provider == 'vllm':
                    import requests
                    vllm_url = config.get('VLLM_SERVER', 'http://127.0.0.1:11400')
                    
                    # URL 끝에 슬래시가 있는지 확인하고 정규화
                    if vllm_url.endswith('/'):
                        vllm_url = vllm_url[:-1]
                    
                    # OpenAI 호환 API 엔드포인트 구성
                    api_endpoint = f"{vllm_url}/v1/chat/completions"
                    
                    self.logger.debug(f"VLLM AI 호출: {api_endpoint}, 모델: {model}")
                    
                    # 요청 데이터 구성 (OpenAI 호환 형식)
                    request_data = {
                        "model": model,
                        "messages": messages,
                        "temperature": 0.3 if require_json else 0.7,
                        "max_tokens": 4000
                    }
                    
                    # JSON 응답이 필요한 경우 response_format 설정
                    if require_json:
                        request_data["response_format"] = {"type": "json_object"}
                    
                    # API 호출
                    response = await asyncio.to_thread(
                        lambda: requests.post(
                            api_endpoint,
                            json=request_data,
                            headers={"Content-Type": "application/json"},
                            timeout=90
                        )
                    )
                    
                    response.raise_for_status()
                    response_json = response.json()
                    
                    if 'choices' not in response_json or not response_json['choices']:
                        self.logger.error(f"VLLM 응답 형식 오류: {response_json}")
                        raise ValueError("VLLM 응답 형식이 예상과 다릅니다.")
                    
                    result = response_json['choices'][0]['message']['content']
                    
                    # 토큰 사용량 추적
                    if 'usage' in response_json:
                        self.usage_tracker.track_usage(
                            provider=provider,
                            model=model,
                            prompt_tokens=response_json['usage'].get('prompt_tokens', 0),
                            completion_tokens=response_json['usage'].get('completion_tokens', 0)
                        )
                    else:
                        # 토큰 수 추정
                        prompt_tokens = len(prompt.split()) * 1.3
                        completion_tokens = len(result.split()) * 1.3
                        self.usage_tracker.track_usage(
                            provider=provider,
                            model=model,
                            prompt_tokens=int(prompt_tokens),
                            completion_tokens=int(completion_tokens)
                        )

                else:
                    raise ValueError(f"지원하지 않는 프로바이더입니다: {provider}")
                
                
                # JSON 응답 처리
                if require_json:
                    try:
                        parsed_result = self._process_json_response(result, json_schema)
                        return parsed_result
                    except ValueError as e:
                        last_error = e
                        retry_count += 1
                        self.logger.warning(f"JSON 응답 검증 실패 (시도 {retry_count}/{max_retries}): {str(e)}")
                        if retry_count < max_retries:
                            # 프롬프트 강화
                            prompt = f"""이전 응답이 JSON 형식 검증에 실패했습니다. 
다음 요구사항을 정확히 따라 다시 응답해주세요:

1. 응답은 반드시 유효한 JSON 형식이어야 합니다.
2. JSON 스키마를 정확히 따라야 합니다.
3. 모든 필수 필드가 포함되어야 합니다.
4. 각 필드의 타입이 정확해야 합니다.
5. JSON 외의 다른 텍스트를 포함하지 마세요.

원본 요청:
{prompt}

JSON 스키마:
{json.dumps(json_schema, ensure_ascii=False, indent=2) if json_schema else "{}"}\n"""
                            continue
                        else:
                            self.logger.error(f"최대 재시도 횟수 초과: {str(last_error)}")
                            raise last_error
                
                return result
                
            except Exception as e:
                self.logger.error(f"AI 호출 실패: {str(e)}")
                traceback.print_exc()
                raise

    def _process_json_response(self, result: str, json_schema: Dict = None) -> Dict:
        """
        AI 응답에서 JSON을 추출하고 처리하는 공통 메소드
        
        Args:
            result: AI 응답 문자열
            json_schema: JSON 스키마 (선택 사항)
            
        Returns:
            Dict: 파싱된 JSON 객체
            
        Raises:
            ValueError: JSON 파싱 실패 시
        """
        try:
            # 응답 검증 시작
            self.logger.debug("응답 검증 시작...")
            
            # 응답이 None인 경우 처리
            if result is None:
                self.logger.error(f"응답 타입 오류: {type(result)}")
                raise ValueError("응답이 None입니다")
            
            # JSON 문자열 정제
            result_text = result.strip()
            
            # 응답이 비어있는 경우 처리
            if not result_text:
                self.logger.info("==============응답이 비어있습니다")
                raise ValueError("응답이 비어있습니다")
            
            # Ollama 모델에서 자주 출력하는 JSON 구조 설명 텍스트 제거
            # "object", "content", "string", "paragraph" 등의 텍스트 패턴 제거
            json_structure_pattern = r'^(?:object|content|string|paragraph|array|number|boolean|null|properties|items|required|type|description|format|enum|default|minimum|maximum|minLength|maxLength|pattern|multipleOf|exclusiveMinimum|exclusiveMaximum|minItems|maxItems|uniqueItems|minProperties|maxProperties|additionalProperties|patternProperties|dependencies|allOf|anyOf|oneOf|not|definitions|$schema|$ref|$id|$comment|examples|const|if|then|else|readOnly|writeOnly|xml|externalDocs|example|deprecated|discriminator|nullable|oneOf|anyOf|allOf|not|title|multipleOf|maximum|exclusiveMaximum|minimum|exclusiveMinimum|maxLength|minLength|pattern|maxItems|minItems|uniqueItems|maxProperties|minProperties|required|additionalProperties|definitions|properties|patternProperties|dependencies|enum|type|format|allOf|anyOf|oneOf|not|items|additionalItems|additionalProperties|patternProperties|dependencies|propertyNames|if|then|else|allOf|anyOf|oneOf|not|format)\s*(?:\n|$)'
            result_text = re.sub(json_structure_pattern, '', result_text, flags=re.MULTILINE)
            
            # 여러 줄의 JSON 구조 설명 텍스트 제거 (각 줄에 하나의 키워드가 있는 경우)
            lines = result_text.split('\n')
            filtered_lines = []
            json_structure_keywords = ['object', 'content', 'string', 'paragraph', 'array', 'number', 'boolean', 'null', 'properties', 'items', 'required', 'type', 'description']
            
            # JSON 시작 중괄호 이전의 줄들 중 JSON 구조 키워드만 있는 줄 제거
            json_start_found = False
            for line in lines:
                line_stripped = line.strip()
                if '{' in line:
                    json_start_found = True
                    filtered_lines.append(line)
                elif json_start_found:
                    filtered_lines.append(line)
                elif line_stripped and line_stripped not in json_structure_keywords:
                    filtered_lines.append(line)
            
            result_text = '\n'.join(filtered_lines)
            
            # JSON 시작과 끝 위치 찾기
            json_start = result_text.find('{')
            json_end = result_text.rfind('}') + 1
            
            if json_start >= 0 and json_end > json_start:
                result_text = result_text[json_start:json_end]
                self.logger.debug(f"JSON 부분 추출: {json_start}~{json_end}")
            else:
                self.logger.warning("JSON 형식 찾을 수 없음, 원본 텍스트 사용")
            
            # 불필요한 줄바꿈 제거
            result_text = re.sub(r'\n\s*\n', '\n', result_text)
            
            # 중첩된 구조에서 텍스트 값 추출
            def extract_text_values(obj):
                if isinstance(obj, dict):
                    # "내용" 키가 있는 경우
                    if "내용" in obj:
                        return obj["내용"]
                    # "content" 키가 있는 경우
                    if "content" in obj:
                        return obj["content"]
                    
                    # 중첩된 객체에서 텍스트 값 찾기
                    texts = []
                    for value in obj.values():
                        if isinstance(value, str):
                            texts.append(value)
                        elif isinstance(value, (dict, list)):
                            result = extract_text_values(value)
                            if result:
                                texts.append(result)
                    if texts:
                        return "\n".join(texts)
                elif isinstance(obj, list):
                    texts = []
                    for item in obj:
                        result = extract_text_values(item)
                        if result:
                            texts.append(result)
                    if texts:
                        return "\n".join(texts)
                elif isinstance(obj, str):
                    return obj
                return None
            
            # "내용" 키를 "content" 키로 변환
            def convert_content_key(obj):
                if isinstance(obj, dict):
                    # 최상위 레벨에서 "내용" 키가 있는 경우
                    if "내용" in obj:
                        obj["content"] = obj.pop("내용")
                    # 중첩된 객체에서 "내용" 키가 있는 경우
                    for key, value in obj.items():
                        if isinstance(value, dict) and "내용" in value:
                            value["content"] = value.pop("내용")
                        elif isinstance(value, dict):
                            convert_content_key(value)
                return obj
            
            # JSON 파싱
            try:
                parsed = json.loads(result_text)
                self.logger.debug("JSON 파싱 성공")
                
                # 리스트 형태의 응답 처리
                if isinstance(parsed, list):
                    self.logger.debug("리스트 형태의 응답을 문자열로 변환합니다")
                    parsed = {
                        "content": "\n".join(str(item) for item in parsed)
                    }

                # "내용" 키를 "content" 키로 변환
                parsed = convert_content_key(parsed)
                
                # JSON 스키마 검증
                if json_schema:
                    self.logger.debug("JSON 스키마 검증 시작...")
                    
                    # 스키마에서 필수 필드 확인
                    required_fields = []
                    if "required" in json_schema:
                        required_fields = json_schema["required"]
                    elif "properties" in json_schema:
                        for field, schema in json_schema["properties"].items():
                            if schema.get("required", False):
                                required_fields.append(field)
                    
                    # content 필드가 필수가 아닌 경우 원본 구조 유지
                    if "content" not in required_fields:
                        # 필수 필드 검증
                        missing_fields = [field for field in required_fields if field not in parsed]
                        if missing_fields:
                            self.logger.error(f"필수 필드가 누락되었습니다: {missing_fields}")
                            self.logger.error(f"현재 응답에 있는 필드: {list(parsed.keys())}")
                            raise ValueError(f"필수 필드가 누락되었습니다: {', '.join(missing_fields)}")
                        return parsed
                    
                    # content 필드가 필수이고 content 필드가 없는 경우에만 텍스트 추출
                    if "content" in required_fields and "content" not in parsed:
                        text_content = extract_text_values(parsed)
                        if text_content:
                            parsed = {"content": text_content}
                    
                    # 필수 필드 검증
                    missing_fields = [field for field in required_fields if field not in parsed]
                    if missing_fields:
                        self.logger.error(f"필수 필드가 누락되었습니다: {missing_fields}")
                        self.logger.error(f"현재 응답에 있는 필드: {list(parsed.keys())}")
                        raise ValueError(f"필수 필드가 누락되었습니다: {', '.join(missing_fields)}")
                    
                    # 타입 검증
                    properties = json_schema.get("properties", {})
                    for field, value in parsed.items():
                        if field in properties:
                            expected_type = properties[field].get("type")
                            if expected_type == "string" and not isinstance(value, str):
                                if isinstance(value, (list, tuple)):
                                    parsed[field] = '\n'.join(str(item) for item in value)
                                else:
                                    parsed[field] = str(value)
                                self.logger.debug(f"필드 '{field}'의 값을 문자열로 변환했습니다")
                            elif expected_type == "array" and not isinstance(value, list):
                                if isinstance(value, str):
                                    parsed[field] = value.split('\n')
                                else:
                                    parsed[field] = [value]
                                self.logger.debug(f"필드 '{field}'의 값을 배열로 변환했습니다")
                
                # 파싱된 JSON 로깅
                self.logger.info("=== 최종 처리된 JSON ===")
                self.logger.info(json.dumps(parsed, ensure_ascii=False, indent=2)[:2000] + ("..." if len(json.dumps(parsed, ensure_ascii=False, indent=2)) > 2000 else ""))
                self.logger.info("==================")
                
                return parsed
                
            except json.JSONDecodeError as e:
                self.logger.error(f"JSON 파싱 실패: {str(e)}")
                
                # 백틱(```) 제거 시도
                if '```json' in result_text or '```' in result_text:
                    self.logger.debug("백틱 제거 시도")
                    result_text = re.sub(r'```(?:json)?\n(.*?)\n```', r'\1', result_text, flags=re.DOTALL)
                    json_start = result_text.find('{')
                    json_end = result_text.rfind('}') + 1
                    if json_start >= 0 and json_end > json_start:
                        result_text = result_text[json_start:json_end]
                        return self._process_json_response(result_text, json_schema)
                
                # 정규식으로 JSON 객체 추출 시도
                json_pattern = r'(\{.*\})'
                matches = re.search(json_pattern, result_text, re.DOTALL)
                if matches:
                    potential_json = matches.group(1)
                    return self._process_json_response(potential_json, json_schema)
                
                raise ValueError(f"JSON 파싱 실패: {str(e)}")
                
        except Exception as e:
            self.logger.error(f"JSON 응답 처리 실패: {str(e)}")
            self.logger.error(f"원본 응답: {result[:500]}")
            raise ValueError(f"JSON 응답 처리 실패: {str(e)}")

    def create_document(self, sections: Dict, business_info: Dict = None, cache_paths: Dict = None) -> None:
        """문서 생성"""
        try:            
            
            if not hasattr(self, 'doc') or self.doc is None:
                self.logger.debug("문서 객체 초기화 필요")
                self._initialize_document()
            
            # 서비스/제품명 찾기
            service_name = self._get_service_name(sections, business_info)
            title = f"{service_name if service_name else '사업계획서'}"
            
            # executive_summary 가져오기
            self.executive_summary = None
            
            # 1. business_info에서 확인
            if business_info and 'executive_summary' in business_info:
                self.executive_summary = business_info['executive_summary']
            
            # 2. 캐시된 메타데이터에서 확인
            elif cache_paths and 'metadata' in cache_paths:
                try:
                    with open(cache_paths['metadata'], 'r', encoding='utf-8') as f:
                        metadata = json.load(f)
                        if 'executive_summary' in metadata:
                            self.executive_summary = metadata['executive_summary']
                except Exception as e:
                    self.logger.warning(f"메타데이터 파일 읽기 실패: {str(e)}")
            
            # 문서 유형별 처리
            if isinstance(self.doc, HWPDocument):
                self.logger.debug("HWP 문서 생성 시작")
                self._create_hwp_document(title, sections, cache_paths)
            elif isinstance(self.doc, PDFDocument):
                self.logger.debug("PDF 문서 생성 시작")
                self._create_pdf_document(title, sections, cache_paths)
            elif isinstance(self.doc, DOCXDocument):
                self.logger.debug("DOCX 문서 생성 시작")
                self._create_docx_document(title, sections, cache_paths)
            elif isinstance(self.doc, PPTXDocument):
                self.logger.debug("PPTX 문서 생성 시작")
                self._create_pptx_document(title, sections, cache_paths)
            else:
                raise ValueError(f"지원하지 않는 문서 유형입니다: {type(self.doc)}")
            
            self.logger.debug("\n=== 문서 생성 완료 ===")

        except Exception as e:
            self.logger.error(f"\n[ERROR] 문서 생성 실패: {str(e)}")
            traceback.print_exc()
            raise

    def _create_pdf_document(self, title: str, sections: Dict, cache_paths: Dict):
            """PDF 문서 생성"""
            try:
                cache_dir = cache_paths['cache_dir']
                
                # 머릿말/꼬릿말 설정
                self.doc.set_header(title, align="left")
                self.doc.set_footer("© 2025 AI.RUN", align="right")
                
                # 문서 제목
                self.doc.add_heading(title, level=1)
                tooltip_text = "이 문서는 AI.RUN 으로 작성된 문서입니다."
                # if self.executive_summary:
                #     tooltip_text += f"\n\n아래의 사용자 요청 프롬프트에 따라 작성되었습니다.\n{self.executive_summary}"
                self.doc.add_tooltip(tooltip_text)
                self.doc.add_spacing(10)
                
                # 섹션 순서대로 처리
                sorted_sections = sorted(self.section_order.items(), key=lambda x: x[1])
                for section_idx, (section_name, section_order) in enumerate(sorted_sections):
                    self.doc.add_heading(f"{section_name}", level=2)
                    
                    # 하위섹션 처리
                    subsection_orders = self.subsection_order.get(section_name, {})
                    sorted_subsections = sorted(subsection_orders.items(), key=lambda x: x[1])
                    
                    for subsection_idx, (subsection_name, subsection_order) in enumerate(sorted_subsections):
                        self.doc.add_heading(f"{subsection_order[1:]}) {subsection_name}", level=3)

                        # 차트 이미지 처리
                        chart_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-chart-*.png"))
                        for chart_file in sorted(chart_files):
                            try:
                                self.doc.add_spacing(5)
                                self.doc.add_image(str(chart_file))
                                self.doc.add_spacing(5)
                            except Exception as e:
                                self.logger.error(f"차트 파일 처리 실패: {str(e)}")

                        # 콘텐츠 파일 처리
                        content_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-content-*.json"))
                        if content_files:
                            try:
                                content_file = sorted(content_files, key=lambda x: int(x.stem.split('-')[-1]))[-1]
                                with open(content_file, 'r', encoding='utf-8') as f:
                                    content_data = json.load(f)
                                    # 템플릿에서 requires_hide 속성 확인
                                    should_hide = False
                                    
                                    # 1. 템플릿 객체에서 requires_hide 확인
                                    template = self.templates.get(section_name, {}).get(subsection_name)
                                    if isinstance(template, SectionConfig) and template.requires_hide:
                                        should_hide = True
                                        self.logger.debug(f"템플릿 설정에 따라 섹션 {section_name}-{subsection_name}의 내용을 숨깁니다.")
                                    
                                    # 2. content_data에서 requires_hide 확인
                                    if 'requires_hide' in content_data and content_data['requires_hide']:
                                        should_hide = True
                                        self.logger.debug(f"콘텐츠 데이터 설정에 따라 섹션 {section_name}-{subsection_name}의 내용을 숨깁니다.")
                                    
                                    if not should_hide:
                                        if 'content' in content_data:
                                            content = content_data['content']
                                            # content가 리스트인 경우 문자열로 변환
                                            if isinstance(content, list):
                                                self.logger.debug(f"콘텐츠가 리스트 형식으로 반환되어 문자열로 변환합니다.")
                                                content = '\n'.join(content)
                                            # 내용을 문단 단위로 분리하여 추가
                                            paragraphs = content.split('\n')
                                            for paragraph in paragraphs:
                                                if paragraph.strip():
                                                    self.doc.add_paragraph(paragraph.strip())
                            except Exception as e:
                                self.logger.error(f"콘텐츠 파일 처리 실패: {str(e)}")

                        # 다이어그램 처리
                        diagram_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-diagram-*.png"))
                        for diagram_file in sorted(diagram_files):
                            try:
                                # 다이어그램 메타데이터 로드
                                diagram_meta_file = next(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-diagramdata-*.json"), None)
                                if diagram_meta_file:
                                    with open(diagram_meta_file, 'r', encoding='utf-8') as f:
                                        diagram_meta = json.load(f)
                                        # 다이어그램 제목 추가
                                        if 'diagram_data' in diagram_meta and 'options' in diagram_meta['diagram_data']:
                                            options = diagram_meta['diagram_data']['options']
                                            if 'title' in options:
                                                self.doc.add_paragraph(f"[그림] {options['title']}", font_size=10, align='center')
                                
                                # 다이어그램 이미지 추가
                                self.doc.add_spacing(5)
                                self.doc.add_image(str(diagram_file))
                                self.doc.add_spacing(5)
                                
                                # 다이어그램 설명 추가 (diagram_meta가 있는 경우에만)
                                if diagram_meta_file:
                                    with open(diagram_meta_file, 'r', encoding='utf-8') as f:
                                        diagram_meta = json.load(f)
                                        if 'diagram_data' in diagram_meta and 'options' in diagram_meta['diagram_data']:
                                            options = diagram_meta['diagram_data']['options']
                                            # if 'description' in options:
                                            #     self.doc.add_paragraph(options['description'])
                            except Exception as e:
                                self.logger.error(f"다이어그램 파일 처리 실패: {str(e)}")
                                
                        # 플로우차트 처리
                        flowchart_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-flowchart-*.png"))
                        for flowchart_file in sorted(flowchart_files):
                            try:
                                # 플로우차트 메타데이터 로드
                                flowchart_meta_file = next(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-flowchartdata-*.json"), None)
                                if flowchart_meta_file:
                                    with open(flowchart_meta_file, 'r', encoding='utf-8') as f:
                                        flowchart_meta = json.load(f)
                                        # 플로우차트 제목 추가
                                        if 'flowchart_data' in flowchart_meta and 'options' in flowchart_meta['flowchart_data']:
                                            options = flowchart_meta['flowchart_data']['options']
                                            if 'title' in options:
                                                self.doc.add_paragraph(f"[그림] {options['title']}", font_size=10, align='center')
                                    
                                # 플로우차트 이미지 추가
                                self.doc.add_spacing(5)
                                self.doc.add_image(str(flowchart_file))
                                self.doc.add_spacing(5)
                            except Exception as e:
                                self.logger.error(f"플로우차트 파일 처리 실패: {str(e)}")

                        # 간트 차트 처리
                        gantt_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-gantt-*.png"))
                        for gantt_file in sorted(gantt_files):
                            try:
                                # 간트 차트 메타데이터 로드
                                gantt_meta_file = next(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-ganttdata-*.json"), None)
                                if gantt_meta_file:
                                    with open(gantt_meta_file, 'r', encoding='utf-8') as f:
                                        gantt_meta = json.load(f)
                                        # 간트 차트 제목 추가
                                        if 'gantt_data' in gantt_meta and 'options' in gantt_meta['gantt_data']:
                                            options = gantt_meta['gantt_data']['options']
                                            if 'title' in options:
                                                self.doc.add_paragraph(f"[그림] {options['title']}", font_size=10, align='center')
                                
                                # 간트 차트 이미지 추가
                                self.doc.add_spacing(5)
                                self.doc.add_image(str(gantt_file))
                                self.doc.add_spacing(5)
                            except Exception as e:
                                self.logger.error(f"간트 차트 파일 처리 실패: {str(e)}")
                                
                        # 시퀀스 다이어그램 처리
                        sequence_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-sequence-*.png"))
                        for sequence_file in sorted(sequence_files):
                            try:
                                # 시퀀스 다이어그램 메타데이터 로드
                                sequence_meta_file = next(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-sequencedata-*.json"), None)
                                if sequence_meta_file:
                                    with open(sequence_meta_file, 'r', encoding='utf-8') as f:
                                        sequence_meta = json.load(f)
                                        # 시퀀스 다이어그램 제목 추가
                                        if 'sequence_data' in sequence_meta and 'options' in sequence_meta['sequence_data']:
                                            options = sequence_meta['sequence_data']['options']
                                            if 'title' in options:
                                                self.doc.add_paragraph(f"[그림] {options['title']}", font_size=10, align='center')
                                
                                # 시퀀스 다이어그램 이미지 추가
                                self.doc.add_spacing(5)
                                self.doc.add_image(str(sequence_file))
                                self.doc.add_spacing(5)
                                
                                # 시퀀스 다이어그램 설명 추가 (sequence_meta가 있는 경우에만)
                                if sequence_meta_file:
                                    with open(sequence_meta_file, 'r', encoding='utf-8') as f:
                                        sequence_meta = json.load(f)
                                        if 'sequence_data' in sequence_meta and 'options' in sequence_meta['sequence_data']:
                                            options = sequence_meta['sequence_data']['options']
                                            # if 'description' in options:
                                            #     self.doc.add_paragraph(options['description'])
                            except Exception as e:
                                self.logger.error(f"시퀀스 다이어그램 파일 처리 실패: {str(e)}")                   
                                                    
                        # 테이블 처리
                        table_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-table-*.csv"))
                        for table_file in sorted(table_files):
                            try:
                                with open(table_file, 'r', encoding='utf-8-sig') as f:
                                    reader = csv.reader(f)
                                    table_data = list(reader)
                                    if table_data:
                                        self.doc.add_table(data=table_data[1:], header=table_data[0])
                            except Exception as e:
                                self.logger.error(f"테이블 파일 처리 실패: {str(e)}")
                        
                        # 마지막 하위섹션이 아닌 경우 페이지 나누기 추가
                        if subsection_idx < len(sorted_subsections) - 1 and self.enable_pagebreak:
                            self.doc.add_page_break()
                    
                    # 마지막 섹션이 아닌 경우 페이지 나누기 추가
                    if section_idx < len(sorted_sections) - 1 and self.enable_pagebreak:
                        self.doc.add_page_break()
                
                self.logger.debug("PDF 문서 생성 완료")
                return True
                
            except Exception as e:
                self.logger.error(f"PDF 문서 생성 실패: {str(e)}")
                traceback.print_exc()
                raise

    def _create_hwp_document(self, title: str, sections: Dict, cache_paths: Dict):
        """HWP 문서 생성"""
        try:
            # 1. 캐시 디렉토리 경로 설정
            cache_dir = cache_paths['cache_dir']
            self.logger.debug(f"캐시 디렉토리: {cache_dir}")
            
            # 2. 메타데이터 파일 읽기
            metadata_path = os.path.join(cache_dir, '000_metadata.json')
            self.logger.debug(f"메타데이터 파일 경로: {metadata_path}")
            with open(metadata_path, 'r', encoding='utf-8') as f:
                metadata = json.load(f)
                self.logger.debug(f"메타데이터 로드 완료: {metadata.get('created_at', '알 수 없음')}")
            
            # 하나의 HWPDocument 인스턴스 사용
            current_doc = self.doc
            current_section = None
            
            # 문서 제목 추가
            self.logger.debug(f"문서 제목 추가: {title}")
            current_doc.add_heading(title, level=1)
            current_doc.add_paragraph("")
            
            tooltip_text = "이 문서는 AI.RUN 으로 작성된 문서입니다."
            current_doc.add_paragraph("")
            current_doc.add_tooltip(tooltip_text)
            current_doc.add_paragraph("")
            
            # 섹션 순서대로 처리
            sorted_sections = sorted(self.section_order.items(), key=lambda x: x[1])
            for section_idx, (section_name, section_order) in enumerate(sorted_sections):
                self.logger.debug(f"섹션 {section_order} ({section_name}) 처리 시작")
                
                # 섹션 제목 추가
                current_doc.add_paragraph("")
                current_doc.add_heading(f"{section_name}", level=2)
                current_section = section_name
                
                # 해당 섹션의 하위섹션 순서대로 처리
                subsection_orders = self.subsection_order.get(section_name, {})
                sorted_subsections = sorted(subsection_orders.items(), key=lambda x: x[1])
                
                for subsection_idx, (subsection_name, subsection_order) in enumerate(sorted_subsections):
                    self.logger.debug(f"하위섹션 {section_order}-{subsection_order} ({subsection_name}) 처리")
                    
                    # 파일 패턴 구성
                    file_pattern = f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}"
                    
                    # 하위섹션 제목 추가
                    current_doc.add_paragraph("")
                    current_doc.add_heading(f"{subsection_order[1:]}) {subsection_name}", level=3)
                    # current_doc.add_heading(f"{subsection_name}", level=3)
                    
                    # 차트 이미지 파일 처리
                    chart_files = list(Path(cache_dir).glob(f"{file_pattern}-chart-*.png"))
                    for chart_file in sorted(chart_files):
                        try:
                            current_doc.add_paragraph("")
                            current_doc.add_image(str(chart_file))
                            current_doc.add_paragraph("")
                        except Exception as e:
                            self.logger.error(f"차트 파일 처리 실패: {str(e)}")
                                                
                    # content.json 파일 처리
                    content_files = list(Path(cache_dir).glob(f"{file_pattern}-content-*.json"))
                    if content_files:
                        content_file = sorted(content_files, key=lambda x: int(x.stem.split('-')[-1]))[-1]
                        try:
                            with open(content_file, 'r', encoding='utf-8') as f:
                                content_data = json.load(f)
                                # 템플릿에서 requires_hide 속성 확인
                                should_hide = False
                                
                                # 1. 템플릿 객체에서 requires_hide 확인
                                template = self.templates.get(section_name, {}).get(subsection_name)
                                if isinstance(template, SectionConfig) and template.requires_hide:
                                    should_hide = True
                                    # self.logger.debug(f"템플릿 설정에 따라 섹션 {section_name}-{subsection_name}의 내용을 숨깁니다.")
                                
                                # 2. content_data에서 requires_hide 확인
                                if content_data.get('requires_hide', False):
                                    should_hide = True
                                    # self.logger.debug(f"콘텐츠 데이터 설정에 따라 섹션 {section_name}-{subsection_name}의 내용을 숨깁니다.")
                                
                                if should_hide:
                                    self.logger.debug(f"섹션 {section_name}-{subsection_name}의 내용은 표시하지 않습니다.")
                                else:
                                    # 내용 추가
                                    if 'content' in content_data:
                                        content = content_data['content']
                                        # content가 리스트인 경우 문자열로 변환
                                        if isinstance(content, list):
                                            self.logger.debug(f"콘텐츠가 리스트 형식으로 반환되어 문자열로 변환합니다.")
                                            content = '\n'.join(content)
                                        paragraphs = content.split('\n')
                                        for para in paragraphs:
                                            if para.strip():
                                                current_doc.add_paragraph(para.strip())                                   
                                current_doc.add_paragraph("")                                        
                        except Exception as e:
                            self.logger.error(f"콘텐츠 파일 처리 실패: {str(e)}")                            

                    # 다이어그램 파일 처리
                    diagram_files = list(Path(cache_dir).glob(f"{file_pattern}-diagram-*.png"))
                    for diagram_file in sorted(diagram_files):
                        try:
                            # 다이어그램 메타데이터 로드
                            diagram_meta_file = next(Path(cache_dir).glob(f"{file_pattern}-diagramdata-*.json"), None)
                            if diagram_meta_file:
                                with open(diagram_meta_file, 'r', encoding='utf-8') as f:
                                    diagram_meta = json.load(f)
                                    # 다이어그램 제목과 설명 추가
                                    if 'diagram_data' in diagram_meta and 'options' in diagram_meta['diagram_data']:
                                        options = diagram_meta['diagram_data']['options']
                                        # if 'title' in options:
                                        #     current_doc.add_paragraph(f"  [그림] {options['title']}")
                            
                            # 다이어그램 이미지 추가
                            current_doc.add_paragraph("")
                            current_doc.add_image(str(diagram_file))
                            current_doc.add_paragraph("")
                            
                            # 다이어그램 설명 추가 (diagram_meta가 있는 경우에만)
                            if diagram_meta_file:
                                with open(diagram_meta_file, 'r', encoding='utf-8') as f:
                                    diagram_meta = json.load(f)
                                    if 'diagram_data' in diagram_meta and 'options' in diagram_meta['diagram_data']:
                                        options = diagram_meta['diagram_data']['options']
                                        # if 'description' in options:
                                        #     current_doc.add_paragraph(options['description'])
                        except Exception as e:
                            self.logger.error(f"다이어그램 파일 처리 실패: {str(e)}")

                    # 플로우차트 처리
                    flowchart_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-flowchart-*.png"))
                    for flowchart_file in sorted(flowchart_files):
                        try:
                            # 플로우차트 메타데이터 로드
                            flowchart_meta_file = next(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-flowchartdata-*.json"), None)
                            if flowchart_meta_file:
                                with open(flowchart_meta_file, 'r', encoding='utf-8') as f:
                                    flowchart_meta = json.load(f)
                                    # 플로우차트 제목 추가
                                    if 'flowchart_data' in flowchart_meta and 'options' in flowchart_meta['flowchart_data']:
                                        options = flowchart_meta['flowchart_data']['options']
                                        # if 'title' in options:
                                        #     self.doc.add_paragraph(f"[그림] {options['title']}")
                            
                                # 플로우차트 이미지 추가
                                current_doc.add_paragraph("")
                                current_doc.add_image(str(flowchart_file))
                                current_doc.add_paragraph("")
                        except Exception as e:
                            self.logger.error(f"플로우차트 파일 처리 실패: {str(e)}")

                    # 간트 차트 처리
                    gantt_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-gantt-*.png"))
                    for gantt_file in sorted(gantt_files):
                        try:
                            # 간트 차트 메타데이터 로드
                            gantt_meta_file = next(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-ganttdata-*.json"), None)
                            if gantt_meta_file:
                                with open(gantt_meta_file, 'r', encoding='utf-8') as f:
                                    gantt_meta = json.load(f)
                                    # 간트 차트 제목 추가
                                    if 'gantt_data' in gantt_meta and 'options' in gantt_meta['gantt_data']:
                                        options = gantt_meta['gantt_data']['options']
                                        # if 'title' in options:
                                            # self.doc.add_paragraph(f"[그림] {options['title']}")
                                
                            # 간트 차트 이미지 추가
                            current_doc.add_paragraph("")
                            current_doc.add_image(str(gantt_file))
                            current_doc.add_paragraph("")
                        except Exception as e:
                            self.logger.error(f"간트 차트 파일 처리 실패: {str(e)}")
                            
                    # 시퀀스 다이어그램 파일 처리
                    sequence_files = list(Path(cache_dir).glob(f"{file_pattern}-sequence-*.png"))
                    for sequence_file in sorted(sequence_files):
                        try:
                            # 시퀀스 다이어그램 메타데이터 로드
                            sequence_meta_file = next(Path(cache_dir).glob(f"{file_pattern}-sequencedata-*.json"), None)
                            if sequence_meta_file:
                                with open(sequence_meta_file, 'r', encoding='utf-8') as f:
                                    sequence_meta = json.load(f)
                                    # 시퀀스 다이어그램 제목과 설명 추가
                                    if 'sequence_data' in sequence_meta and 'options' in sequence_meta['sequence_data']:
                                        options = sequence_meta['sequence_data']['options']
                                        # if 'title' in options:
                                        #     current_doc.add_paragraph(f"  [그림] {options['title']}")
                            
                            # 시퀀스 다이어그램 이미지 추가
                            current_doc.add_paragraph("")
                            current_doc.add_image(str(sequence_file))
                            current_doc.add_paragraph("")
                            
                            # 시퀀스 다이어그램 설명 추가 (sequence_meta가 있는 경우에만)
                            if sequence_meta_file:
                                with open(sequence_meta_file, 'r', encoding='utf-8') as f:
                                    sequence_meta = json.load(f)
                                    if 'sequence_data' in sequence_meta and 'options' in sequence_meta['sequence_data']:
                                        options = sequence_meta['sequence_data']['options']
                                        # if 'description' in options:
                                        #     current_doc.add_paragraph(options['description'])
                        except Exception as e:
                            self.logger.error(f"시퀀스 다이어그램 파일 처리 실패: {str(e)}")
                    
                    # 테이블 파일 처리
                    table_files = list(Path(cache_dir).glob(f"{file_pattern}-table-*.csv"))
                    for table_file in sorted(table_files):
                        try:
                            with open(table_file, 'r', encoding='utf-8') as f:
                                reader = csv.reader(f)
                                table_data = list(reader)
                                if table_data:
                                    current_doc.add_paragraph("")
                                    # current_doc.add_table(data=table_data[1:], header=table_data[0])
                                    current_doc.add_table(data=table_data[1:], header=table_data[0], style=2, header_style=7)
                                    current_doc.add_paragraph("")
                        except Exception as e:
                            self.logger.error(f"테이블 파일 처리 실패: {str(e)}")

                    # 마지막 하위섹션이 아닌 경우 페이지 나누기 추가
                    if subsection_idx < len(sorted_subsections) - 1 and self.enable_pagebreak:
                        current_doc.add_page_break()
                
                # 마지막 섹션이 아닌 경우 페이지 나누기 추가
                if section_idx < len(sorted_sections) - 1 and self.enable_pagebreak:
                    current_doc.add_page_break()
            
            self.logger.debug("HWP 문서 생성 완료")
            return True
            
        except Exception as e:
            self.logger.error(f"HWP 문서 생성 실패: {str(e)}")
            traceback.print_exc()
            raise

    def _create_docx_document(self, title: str, sections: Dict, cache_paths: Dict):
        """DOCX 문서 생성"""
        try:

            self.logger.debug("DOCX 문서 생성 시작")
            
            # 캐시 디렉토리 설정
            cache_dir = cache_paths.get('cache_dir', '')
            if not cache_dir or not os.path.exists(cache_dir):
                raise ValueError(f"Invalid cache directory: {cache_dir}")
            
            # 문서 제목 설정
            self.doc.add_heading(title, level=0)
            
            # 섹션 순서대로 처리
            sorted_sections = sorted(self.section_order.items(), key=lambda x: x[1])
            for section_idx, (section_name, section_order) in enumerate(sorted_sections):
                self.logger.debug(f"섹션 처리 중: {section_name}")
                
                # 섹션 제목 추가
                self.doc.add_heading(f"{section_name}", level=1)
                
                # 하위 섹션 정렬
                subsection_orders = self.subsection_order.get(section_name, {})
                sorted_subsections = sorted(subsection_orders.items(), key=lambda x: x[1])
                
                # 각 하위 섹션 처리
                for subsection_idx, (subsection_name, subsection_order) in enumerate(sorted_subsections):
                    self.logger.debug(f"하위 섹션 처리 중: {subsection_name}")
                    
                    # 하위 섹션 제목 추가
                    self.doc.add_heading(f"{subsection_order[1:]}) {subsection_name}", level=2)
                    
                    # 차트 이미지 처리 (PDF 방식과 유사하게)
                    chart_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-chart-*.png"))
                    for chart_file in sorted(chart_files):
                        try:
                            # 이미지 추가 전 빈 문단 추가
                            self.doc.add_paragraph("")
                            self.doc.add_image(str(chart_file))
                            # 캡션 추가
                            self.doc.add_paragraph("[그림] 차트", style='Caption')
                            # 이미지 추가 후 빈 문단 추가
                            self.doc.add_paragraph("")
                        except Exception as e:
                            self.logger.error(f"차트 파일 처리 실패: {str(e)}")
                    
                    # 콘텐츠 파일 처리
                    content_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-content-*.json"))
                    if content_files:
                        try:
                            content_file = sorted(content_files, key=lambda x: int(x.stem.split('-')[-1]))[-1]
                            with open(content_file, 'r', encoding='utf-8') as f:
                                content_data = json.load(f)
                                # 템플릿에서 requires_hide 속성 확인
                                should_hide = False
                                
                                # 1. 템플릿 객체에서 requires_hide 확인
                                template = self.templates.get(section_name, {}).get(subsection_name)
                                if isinstance(template, SectionConfig) and template.requires_hide:
                                    should_hide = True
                                    self.logger.debug(f"템플릿 설정에 따라 섹션 {section_name}-{subsection_name}의 내용을 숨깁니다.")
                                
                                # 2. content_data에서 requires_hide 확인
                                if 'requires_hide' in content_data and content_data['requires_hide']:
                                    should_hide = True
                                    self.logger.debug(f"콘텐츠 데이터 설정에 따라 섹션 {section_name}-{subsection_name}의 내용을 숨깁니다.")
                                
                                if not should_hide:
                                    if 'content' in content_data:
                                        content = content_data['content']
                                        # content가 리스트인 경우 문자열로 변환
                                        if isinstance(content, list):
                                            self.logger.debug(f"콘텐츠가 리스트 형식으로 반환되어 문자열로 변환합니다.")
                                            content = '\n'.join(content)
                                        # 내용을 문단 단위로 분리하여 추가
                                        paragraphs = content.split('\n')
                                        for paragraph in paragraphs:
                                            if paragraph.strip():
                                                self.doc.add_paragraph(paragraph.strip())
                        except Exception as e:
                            self.logger.error(f"콘텐츠 파일 처리 실패: {str(e)}")
                    
                    # 다이어그램 처리 (PDF 방식과 유사하게)
                    diagram_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-diagram-*.png"))
                    for diagram_file in sorted(diagram_files):
                        try:
                            # 다이어그램 메타데이터 로드
                            diagram_meta_file = next(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-diagramdata-*.json"), None)
                            caption = "[그림] 다이어그램"
                            
                            if diagram_meta_file:
                                with open(diagram_meta_file, 'r', encoding='utf-8') as f:
                                    diagram_meta = json.load(f)
                                    # 다이어그램 제목 추가
                                    if 'diagram_data' in diagram_meta and 'options' in diagram_meta['diagram_data']:
                                        options = diagram_meta['diagram_data']['options']
                                        if 'title' in options:
                                            caption = f"[그림] {options['title']}"
                            
                            # 이미지 추가 전 빈 문단 추가
                            self.doc.add_paragraph("")
                            self.doc.add_image(str(diagram_file))
                            # 캡션 추가
                            self.doc.add_paragraph(caption, style='Caption')
                            # 이미지 추가 후 빈 문단 추가
                            self.doc.add_paragraph("")
                        except Exception as e:
                            self.logger.error(f"다이어그램 파일 처리 실패: {str(e)}")
                    
                    # 플로우차트 처리 (PDF 방식과 유사하게)
                    flowchart_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-flowchart-*.png"))
                    for flowchart_file in sorted(flowchart_files):
                        try:
                            # 플로우차트 메타데이터 로드
                            flowchart_meta_file = next(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-flowchartdata-*.json"), None)
                            caption = "[그림] 플로우차트"
                            
                            if flowchart_meta_file:
                                with open(flowchart_meta_file, 'r', encoding='utf-8') as f:
                                    flowchart_meta = json.load(f)
                                    # 플로우차트 제목 추가
                                    if 'flowchart_data' in flowchart_meta and 'options' in flowchart_meta['flowchart_data']:
                                        options = flowchart_meta['flowchart_data']['options']
                                        if 'title' in options:
                                            caption = f"[그림] {options['title']}"
                            
                            # 이미지 추가 전 빈 문단 추가
                            self.doc.add_paragraph("")
                            self.doc.add_image(str(flowchart_file))
                            # 캡션 추가
                            self.doc.add_paragraph(caption, style='Caption')
                            # 이미지 추가 후 빈 문단 추가
                            self.doc.add_paragraph("")
                        except Exception as e:
                            self.logger.error(f"플로우차트 파일 처리 실패: {str(e)}")
                    
                    # 시퀀스 다이어그램 처리 (PDF 방식과 유사하게)
                    sequence_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-sequence-*.png"))
                    for sequence_file in sorted(sequence_files):
                        try:
                            # 시퀀스 다이어그램 메타데이터 로드
                            sequence_meta_file = next(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-sequencedata-*.json"), None)
                            caption = "[그림] 시퀀스 다이어그램"
                            
                            if sequence_meta_file:
                                with open(sequence_meta_file, 'r', encoding='utf-8') as f:
                                    sequence_meta = json.load(f)
                                    # 시퀀스 다이어그램 제목 추가
                                    if 'sequence_data' in sequence_meta and 'options' in sequence_meta['sequence_data']:
                                        options = sequence_meta['sequence_data']['options']
                                        if 'title' in options:
                                            caption = f"[그림] {options['title']}"
                            
                            # 이미지 추가 전 빈 문단 추가
                            self.doc.add_paragraph("")
                            self.doc.add_image(str(sequence_file))
                            # 캡션 추가
                            self.doc.add_paragraph(caption, style='Caption')
                            # 이미지 추가 후 빈 문단 추가
                            self.doc.add_paragraph("")
                        except Exception as e:
                            self.logger.error(f"시퀀스 다이어그램 파일 처리 실패: {str(e)}")
                    
                    # 테이블 처리 (PDF 방식과 유사하게)
                    table_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-table-*.csv"))
                    for table_file in sorted(table_files):
                        try:
                            with open(table_file, 'r', encoding='utf-8-sig') as f:
                                reader = csv.reader(f)
                                table_data = list(reader)
                                if table_data and len(table_data) > 0:
                                    self.doc.add_table(data=table_data[1:], header=table_data[0])
                        except Exception as e:
                            self.logger.error(f"테이블 파일 처리 실패: {str(e)}")
                    
                    # 마지막 하위섹션이 아닌 경우 페이지 나누기 추가
                    if subsection_idx < len(sorted_subsections) - 1 and self.enable_pagebreak:
                        self.doc.add_page_break()
                
                # 마지막 섹션이 아닌 경우 페이지 나누기 추가
                if section_idx < len(sorted_sections) - 1 and self.enable_pagebreak:
                    self.doc.add_page_break()
            
            self.logger.debug("DOCX 문서 생성 완료")
            return True
            
        except Exception as e:
            self.logger.error(f"DOCX 문서 생성 실패: {str(e)}")
            traceback.print_exc()
            raise

    def _create_pptx_document(self, title: str, sections: Dict, cache_paths: Dict):
        """PPTX 문서 생성"""
        try:

            self.logger.debug("PPTX 문서 생성 시작")
                
            # 캐시 디렉토리 설정
            cache_dir = cache_paths.get('cache_dir', '')
            if not cache_dir or not os.path.exists(cache_dir):
                raise ValueError(f"Invalid cache directory: {cache_dir}")
            
            # 제목 슬라이드 추가
            self.doc.add_title_slide(title)
            
            # 섹션 순서대로 처리
            sorted_sections = sorted(self.section_order.items(), key=lambda x: x[1])
            for section_idx, (section_name, section_order) in enumerate(sorted_sections):
                self.logger.debug(f"섹션 처리 중: {section_name}")
                
                # 섹션 제목 슬라이드 추가 - 'section' 레이아웃 사용
                try:
                    self.doc.add_content_slide(title=section_name, layout='section')
                except Exception as e:
                    self.logger.error(f"섹션 제목 슬라이드 추가 실패: {str(e)}")
                
                # 하위 섹션 정렬
                subsection_orders = self.subsection_order.get(section_name, {})
                sorted_subsections = sorted(subsection_orders.items(), key=lambda x: x[1])
                
                # 각 하위 섹션 처리
                for subsection_idx, (subsection_name, subsection_order) in enumerate(sorted_subsections):
                    self.logger.debug(f"하위 섹션 처리 중: {subsection_name}")
                    
                    # 하위 섹션 제목
                    subsection_title = f"{subsection_order[1:]}) {subsection_name}"
                    
                    # 콘텐츠 파일 처리
                    content = ""
                    content_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-content-*.json"))
                    if content_files:
                        try:
                            content_file = sorted(content_files, key=lambda x: int(x.stem.split('-')[-1]))[-1]
                            with open(content_file, 'r', encoding='utf-8') as f:
                                content_data = json.load(f)
                                # 템플릿에서 requires_hide 속성 확인
                                should_hide = False
                                
                                # 1. 템플릿 객체에서 requires_hide 확인
                                template = self.templates.get(section_name, {}).get(subsection_name)
                                if isinstance(template, SectionConfig) and template.requires_hide:
                                    should_hide = True
                                    self.logger.debug(f"템플릿 설정에 따라 섹션 {section_name}-{subsection_name}의 내용을 숨깁니다.")
                                
                                # 2. content_data에서 requires_hide 확인
                                if 'requires_hide' in content_data and content_data['requires_hide']:
                                    should_hide = True
                                    self.logger.debug(f"콘텐츠 데이터 설정에 따라 섹션 {section_name}-{subsection_name}의 내용을 숨깁니다.")
                                
                                if not should_hide:
                                    if 'content' in content_data:
                                        content = content_data['content']
                                        # content가 리스트인 경우 문자열로 변환
                                        if isinstance(content, list):
                                            self.logger.debug(f"콘텐츠가 리스트 형식으로 반환되어 문자열로 변환합니다.")
                                            content = '\n'.join(content)
                        except Exception as e:
                            self.logger.error(f"콘텐츠 파일 처리 실패: {str(e)}")
                    
                    # 차트 이미지 경로
                    chart_path = None
                    chart_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-chart-*.png"))
                    if chart_files:
                        try:
                            chart_path = str(sorted(chart_files)[0])
                        except Exception as e:
                            self.logger.error(f"차트 파일 처리 실패: {str(e)}")
                    
                    # 다이어그램 이미지 경로
                    diagram_path = None
                    try:
                        diagram_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-diagram-*.png"))
                        if diagram_files:
                            diagram_path = str(sorted(diagram_files)[0])
                    except Exception as e:
                        self.logger.error(f"다이어그램 파일 처리 실패: {str(e)}")
                    
                    # 플로우차트 이미지 경로
                    flowchart_path = None
                    try:
                        flowchart_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-flowchart-*.png"))
                        if flowchart_files:
                            flowchart_path = str(sorted(flowchart_files)[0])
                    except Exception as e:
                        self.logger.error(f"플로우차트 파일 처리 실패: {str(e)}")
                    
                    # 시퀀스 다이어그램 이미지 경로
                    sequence_path = None
                    try:
                        sequence_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-sequence-*.png"))
                        if sequence_files:
                            sequence_path = str(sorted(sequence_files)[0])
                    except Exception as e:
                        self.logger.error(f"시퀀스 다이어그램 파일 처리 실패: {str(e)}")
                    
                    # 테이블 데이터
                    table_data = None
                    try:
                        table_files = list(Path(cache_dir).glob(f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-table-*.csv"))
                        if table_files:
                            with open(sorted(table_files)[0], 'r', encoding='utf-8-sig') as f:
                                reader = csv.reader(f)
                                table_data = list(reader)
                    except Exception as e:
                        self.logger.error(f"테이블 파일 처리 실패: {str(e)}")
                    
                    # 이미지 경로 결정 (우선순위: 차트 > 다이어그램 > 플로우차트 > 시퀀스)
                    image_path = chart_path or diagram_path or flowchart_path or sequence_path
                    
                    # 내용 슬라이드 추가
                    if content:
                        # 내용을 여러 슬라이드로 분할
                        content_chunks = self._split_text_into_chunks(content, chunk_size=200)
                        
                        for i, chunk in enumerate(content_chunks):
                            if i == 0 and image_path:  # 첫 번째 청크에 이미지가 있는 경우
                                try:
                                    # 이미지 슬라이드 추가
                                    self.doc.add_image_slide(
                                        title=subsection_title,
                                        image_path=image_path
                                    )
                                    # 내용 슬라이드 추가
                                    self.doc.add_content_slide(
                                        title=subsection_title,
                                        content=chunk
                                    )
                                except Exception as e:
                                    self.logger.error(f"이미지와 내용 슬라이드 추가 실패: {str(e)}")
                            elif i == 0 and table_data:  # 첫 번째 청크에 테이블이 있는 경우
                                try:
                                    # 테이블 슬라이드 추가
                                    self.doc.add_table_slide(
                                        title=subsection_title,
                                        data=table_data,
                                        header=True
                                    )
                                    # 내용 슬라이드 추가
                                    self.doc.add_content_slide(
                                        title=subsection_title,
                                        content=chunk
                                    )
                                except Exception as e:
                                    self.logger.error(f"테이블과 내용 슬라이드 추가 실패: {str(e)}")
                            else:  # 이미지/테이블이 없거나 두 번째 이후 청크
                                try:
                                    self.doc.add_content_slide(
                                        title=f"{subsection_title}{' (계속)' if i > 0 else ''}",
                                        content=chunk
                                    )
                                except Exception as e:
                                    self.logger.error(f"내용 슬라이드 추가 실패: {str(e)}")
                    else:
                        # 내용이 없는 경우 이미지나 테이블만 표시
                        if image_path:
                            try:
                                self.doc.add_image_slide(
                                    title=subsection_title,
                                    image_path=image_path
                                )
                            except Exception as e:
                                self.logger.error(f"이미지 슬라이드 추가 실패: {str(e)}")
                        elif table_data:
                            try:
                                self.doc.add_table_slide(
                                    title=subsection_title,
                                    data=table_data,
                                    header=True
                                )
                            except Exception as e:
                                self.logger.error(f"테이블 슬라이드 추가 실패: {str(e)}")
            
            self.logger.debug("PPTX 문서 생성 완료")
            return True
            
        except Exception as e:
            self.logger.error(f"PPTX 문서 생성 실패: {str(e)}")
            traceback.print_exc()
            raise

    def _split_text_into_chunks(self, text: str, chunk_size: int = 1000) -> List[str]:
        """긴 텍스트를 적절한 크기의 청크로 나눔"""
        if not text:
            return []
            
        # 문단 단위로 먼저 나누기
        paragraphs = text.split('\n')
        chunks = []
        current_chunk = []
        current_size = 0
        
        for paragraph in paragraphs:
            # 각 문단의 크기 계산
            p_size = len(paragraph)
            
            # 문단이 chunk_size보다 크면 문장 단위로 나누기
            if p_size > chunk_size:
                # 문장 단위로 나누기 (마침표, 물음표, 느낌표 뒤에 공백이 있는 경우)
                sentences = re.split(r'([.!?])\s+', paragraph)
                
                # 문장 단위로 나눈 결과가 홀수 개이면 마지막 문장에 마침표 등이 포함되지 않으므로 처리
                if len(sentences) % 2 == 1:
                    processed_sentences = []
                    for i in range(0, len(sentences) - 1, 2):
                        processed_sentences.append(sentences[i] + sentences[i+1])
                    processed_sentences.append(sentences[-1])
                else:
                    processed_sentences = []
                    for i in range(0, len(sentences), 2):
                        if i+1 < len(sentences):
                            processed_sentences.append(sentences[i] + sentences[i+1])
                        else:
                            processed_sentences.append(sentences[i])
                
                # 문장 단위로 청크 구성
                for sentence in processed_sentences:
                    s_size = len(sentence)
                    
                    # 문장이 너무 길면 강제로 나누기
                    if s_size > chunk_size:
                        # 현재 청크가 있으면 먼저 저장
                        if current_chunk:
                            chunks.append('\n'.join(current_chunk))
                            current_chunk = []
                            current_size = 0
                        
                        # 긴 문장을 chunk_size 단위로 강제 분할
                        for i in range(0, s_size, chunk_size):
                            if i + chunk_size < s_size:
                                chunks.append(sentence[i:i+chunk_size])
                            else:
                                current_chunk.append(sentence[i:])
                                current_size = len(sentence[i:])
                    else:
                        # 현재 청크에 문장을 추가했을 때 크기 초과하면 새 청크 시작
                        if current_size + s_size > chunk_size:
                            chunks.append('\n'.join(current_chunk))
                            current_chunk = [sentence]
                            current_size = s_size
                        else:
                            current_chunk.append(sentence)
                            current_size += s_size
            else:
                # 문단이 chunk_size보다 작으면 그대로 처리
                if current_size + p_size > chunk_size:
                    # 현재 청크가 너무 커지면 새로운 청크 시작
                    if current_chunk:
                        chunks.append('\n'.join(current_chunk))
                        current_chunk = []
                        current_size = 0
                
                current_chunk.append(paragraph)
                current_size += p_size
        
        # 남은 청크 처리
        if current_chunk:
            chunks.append('\n'.join(current_chunk))
        
        return chunks or [text]  # 청크가 없으면 원본 텍스트 반환

    def _get_service_name(self, sections: Dict, business_info: Dict = None) -> Optional[str]:
        """서비스/제품명 추출"""
        try:
            if business_info:
                if 'product_name' in business_info:
                    return business_info['product_name']
                elif 'business_name' in business_info:
                    return business_info['business_name']
            
            if 'executive_summary' in sections and 'business_overview' in sections['executive_summary']:
                overview = sections['executive_summary']['business_overview']
                if isinstance(overview, dict):
                    content = overview.get('content', '')
                else:
                    content = str(overview)
                return content.split('\n')[0].strip()[:20]
                
        except Exception as e:
            self.logger.error(f"서비스명 추출 실패: {str(e)}")
        
        return None

    def create_table_of_contents(self, sections: Dict) -> None:
        """목차를 테이블 형식으로 생성"""
        try:
            self.doc.add_heading("목 차", level=1)
            self.doc.add_paragraph("")
            
            # 목차 테이블 데이터 구성
            table_data = []
            headers = ["", "", ""]
            
            # 섹션 순서대로 처리
            for section_name, section_order in sorted(self.section_order.items(), key=lambda x: x[1]):
                # 대항목 추가 (페이지 번호는 별도의 컨트롤로 추가)
                table_data.append([
                    f"{section_order}. {section_name}", 
                    "············································", 
                    ""
                ])
                
                # 하위항목 추가
                subsection_orders = self.subsection_order.get(section_name, {})
                for subsection_name, subsection_order in sorted(subsection_orders.items(), key=lambda x: x[1]):
                    table_data.append([
                        f"    {subsection_order[1:]}) {subsection_name}", 
                        "", 
                        ""
                    ])

            # 테두리 없는 스타일로 목차 테이블 생성
            no_border_style = {
                "style": {
                    "borderFillIDRef": "1",
                    "cellBorderFillIDRef": "1",
                    "headerFillIDRef": "1"
                }
            }

            self.doc.add_table(
                data=table_data, 
                header=headers, 
                options=no_border_style
            )
            self.doc.add_paragraph("")
            self.doc.add_page_break()
            
        except Exception as e:
            self.logger.error(f"목차 생성 실패: {str(e)}")
            traceback.print_exc()

    def save(self, output_path: str) -> None:
        """문서 저장"""
        try:            
            if not hasattr(self, 'doc') or self.doc is None:
                raise ValueError("문서 객체가 초기화되지 않았습니다")
            
            # 출력 디렉토리 확인 및 생성
            abs_output_path = self._ensure_absolute_path(output_path)
            output_dir = os.path.dirname(abs_output_path)
            BaseServiceClient.safe_makedirs(output_dir)            
            self.logger.debug(f"출력 디렉토리 생성/확인: {output_dir}")
            
            # 파일 확장자 검증
            _, ext = os.path.splitext(output_path)
            if ext.lower() not in ['.hwpx', '.pdf', '.docx', '.pptx']:
                raise ValueError(f"지원하지 않는 파일 형식입니다: {ext}")
            
            # 저장 시도
            self.logger.debug(f"문서 저장 시작: {output_path}")
            
            try:                                   
                # PDF 파일 저장
                if ext.lower() == '.pdf':
                    self.doc.save(output_path, include_toc=False)
                    self.logger.debug("PDF 저장 완료")
                # DOCX 파일 저장
                elif ext.lower() == '.docx':
                    self.doc.save(output_path)
                    self.logger.debug("DOCX 저장 완료")
                # PPTX 파일 저장
                elif ext.lower() == '.pptx':
                    self.doc.save(output_path)
                    self.logger.debug("PPTX 저장 완료")
                # HWPX 파일 저장
                else:
                    self.doc.save(output_path)
                    self.logger.debug("HWPX 저장 완료")
                
                # 파일 생성 확인 및 검증
                if not os.path.exists(output_path):
                    raise FileNotFoundError("저장 후 파일이 존재하지 않습니다")
                
                file_size = os.path.getsize(output_path)
                if file_size == 0:
                    raise ValueError("저장된 파일의 크기가 0입니다")
                
                self.logger.debug(f"파일 저장 완료 (크기: {file_size:,} bytes)")
                
                # HWPX 파일 추가 검증
                if ext.lower() == '.hwpx':
                    import zipfile
                    try:
                        with zipfile.ZipFile(output_path, 'r') as zip_ref:
                            required_files = ['Contents/content.hpf', 'Contents/header.xml', 'META-INF/container.xml']
                            missing_files = [f for f in required_files if f not in zip_ref.namelist()]
                            if missing_files:
                                raise ValueError(f"HWPX 파일에 필수 파일이 누락됨: {', '.join(missing_files)}")
                        self.logger.debug("HWPX 파일 구조 검증 완료")
                    except zipfile.BadZipFile:
                        raise ValueError("생성된 HWPX 파일이 올바른 ZIP 형식이 아닙니다")
                
                return True
                
            except Exception as e:
                # 저장 실패 시 파일 삭제 시도
                if os.path.exists(output_path):
                    try:
                        os.remove(output_path)
                        self.logger.debug("실패한 파일 삭제 완료")
                    except Exception as del_e:
                        self.logger.error(f"실패한 파일 삭제 실패: {str(del_e)}")
                
                raise ValueError(f"문서 저장 중 오류 발생: {str(e)}")
                
        except Exception as e:
            self.logger.error(f"문서 저장 실패: {str(e)}")
            raise

    def _sanitize_filename(self, filename: str) -> str:
        """파일명에서 안전하지 않은 문자 제거"""
        # 1. 공백을 언더스코어로 변경
        # 2. 특수문자 제거 또는 변환
        # 3. 슬래시나 역슬래시 제거
        import re
        
        # 허용할 문자 패턴 정의
        safe_pattern = re.compile(r'[^a-zA-Z0-9가-힣_-]')
        
        # 공백을 언더스코어로 변환
        filename = filename.replace(' ', '_')
        
        # 특수문자를 제거하고 안전한 문자만 남김
        safe_filename = safe_pattern.sub('', filename)
        
        # 결과가 비어있으면 기본값 사용
        if not safe_filename:
            return 'unnamed_section'
        
        return safe_filename

    def _load_section_templates(self) -> Dict[str, Dict[str, SectionConfig]]:
        """섹션 템플릿 로드 및 SectionConfig 객체로 변환"""
        try:
            raw_templates = self.load_prompt_templates(self.executive_summary)
            if not raw_templates:
                raise ValueError("템플릿을 로드할 수 없습니다.")
            
            section_configs = {}
            
            # 섹션 순서는 템플릿의 키 순서대로 자동 생성
            self.section_order = {}
            self.subsection_order = {}
            
            # 메타데이터 키를 제외한 섹션 목록 생성
            metadata_keys = {'name', 'description', 'category', '_metadata'}
            section_list = [key for key in raw_templates.keys() if key not in metadata_keys]
            # self.logger.debug(f"섹션 목록: {section_list}")
            
            # 섹션 순서 설정 (01, 02, ...)
            for section_idx, section_name in enumerate(section_list, 1):
                section_order = f"{section_idx:02d}"
                self.section_order[section_name] = section_order
                # self.logger.debug(f"섹션 순서 설정: {section_name} -> {section_order}")
                
                subsections = raw_templates[section_name]
                if not isinstance(subsections, dict):
                    self.logger.error(f"잘못된 하위 섹션 형식: {section_name}")
                    continue
                
                self.subsection_order[section_name] = {}
                section_configs[section_name] = {}
                
                # 하위 섹션 순서 설정 (01, 02, ...)
                subsection_list = list(subsections.keys())
                # self.logger.debug(f"{section_name} 섹션의 하위 섹션 목록: {subsection_list}")
                
                for subsection_idx, subsection_name in enumerate(subsection_list, 1):
                    subsection_order = f"{subsection_idx:02d}"
                    self.subsection_order[section_name][subsection_name] = subsection_order
                    # self.logger.debug(f"하위 섹션 순서 설정: {section_name}/{subsection_name} -> {section_order}_{subsection_order}")
                    
                    # SectionConfig 객체 생성
                    template_data = subsections[subsection_name]
                    section_configs[section_name][subsection_name] = SectionConfig.from_template(
                        title=subsection_name,
                        template_data=template_data
                    )
            
            # 섹션 순서 정보 검증
            if not self.section_order or not self.subsection_order:
                raise ValueError("섹션 순서 정보가 설정되지 않았습니다.")
            
            # 섹션 순서 정보 로깅
            # self.logger.debug("\n=== 섹션 순서 정보 ===")
            # for section_name, order in self.section_order.items():
            #     self.logger.debug(f"섹션: {section_name} -> {order}")
            #     if section_name in self.subsection_order:
            #         for subsection_name, sub_order in self.subsection_order[section_name].items():
            #             self.logger.debug(f"  └ {subsection_name} -> {order}_{sub_order}")
            
            return section_configs
            
        except Exception as e:
            self.logger.error(f"섹션 템플릿 로드 실패: {str(e)}")
            traceback.print_exc()
            raise  # 오류를 상위로 전파

class BizPlanContentGenerator(BizPlanDocument):
    """보고서 생성 기능 클래스"""
    def __init__(self, output_format: str = 'pdf', logger=None, provider: str = None, model: str = None):
        """초기화"""
        super().__init__(output_format)
        self.usage_tracker = UsageTracker()  # 사용량 추적기 초기화
        self.web = WebSearchService()
        # DocumentProcessor 제거 - 외부 RAG 서비스 사용 (자원 중복 방지)
        self.rag = None  # 외부 클라이언트로 대체
        self.rag_settings = get_rag_settings()
        
        # 진행 상태 추적을 위한 변수 추가
        self.total_sections = 0
        self.completed_sections = 0
        
        # 작업 ID 관련 속성 추가
        self.job_id = None
        self.project_hash = None
        self.timestamp = None
        
        # 사용자 ID 관련 속성 추가
        self.user_id = None
        
        # AI 모델 설정 (사용자가 지정한 경우 우선 사용)
        self.user_provider = provider
        self.user_model = model
        
        # 로거 설정
        self.logger = logger or logging.getLogger("report_generator_default")
        
        # 외부 서비스 클라이언트들 자동 초기화
        self.external_rag_client = RAGServiceClient(logger=self.logger)
        self.external_web_client = WebSearchServiceClient(logger=self.logger)
        
        # 시작 시간 기록
        self._start_time = time.time()
        
        # 시그널 핸들러 초기화
        self._initialize_signal_handlers()
        
        # 출력 버퍼링 비활성화
        sys.stdout.reconfigure(line_buffering=True)
    
    def set_external_clients(self, rag_client=None, web_client=None):
        """외부 서비스 클라이언트들 설정 (자원 중복 방지)"""
        self.external_rag_client = rag_client
        self.external_web_client = web_client
        
        if rag_client:
            self.logger.info("✅ 외부 RAG 클라이언트 설정됨")
        if web_client:
            self.logger.info("✅ 외부 웹검색 클라이언트 설정됨")
    
    def set_user_id(self, user_id: str):
        """사용자 ID 설정 (사용자별 디렉토리 구조 적용)"""
        self.user_id = user_id
        if user_id:
            self.logger.info(f"✅ 사용자 ID 설정됨: {user_id}")
            # 사용자별 캐시 디렉토리 경로 재설정
            self.base_cache_dir = os.path.join(os.path.expanduser('~'), '.airun', 'cache', 'report', user_id)
            BaseServiceClient.safe_makedirs(self.base_cache_dir)
            self.logger.debug(f"사용자별 캐시 디렉토리 설정: {self.base_cache_dir}")

    def _get_ai_params(self):
        """사용자가 지정한 AI 모델 파라미터를 반환"""
        return {
            'provider': self.user_provider,
            'model': self.user_model
        }

    def _generate_job_id(self, executive_summary: str, project_hash: str = None) -> str:
        """작업 ID 생성"""
        if project_hash:
            self.project_hash = project_hash
        else:
            self.project_hash = hashlib.md5(executive_summary.encode('utf-8')).hexdigest()[:8]
        self.timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        self.job_id = f"{self.project_hash}_{self.timestamp}"
        # logger를 새로 만들지 않고, self.logger만 사용!
        return self.job_id

    def _load_system_message(self, message_type: str) -> str:
        """시스템 메시지 파일을 읽어옵니다.
        
        Args:
            message_type (str): 메시지 타입 (예: 'business_info', 'market_analysis', 'section_content', 'competitor_analysis', 'swot_analysis')
            
        Returns:
            str: 시스템 메시지 내용
        """
        try:
            import os
            import re            
            # 현재 스크립트의 디렉토리 경로
            current_dir = os.path.dirname(os.path.abspath(__file__))
            
            # 현재 사용 중인 LLM 제공자 확인
            use_ollama = False
            try:
                # ~/.airun/airun.conf 파일에서 직접 USE_LLM 값 읽기
                config_path = os.path.expanduser('~/.airun/airun.conf')
                if os.path.exists(config_path):
                    with open(config_path, 'r') as f:
                        config_content = f.read()
                    
                    # export USE_LLM="ollama" 형식에서 값 추출
                    llm_match = re.search(r'export\s+USE_LLM="([^"]*)"', config_content)
                    llm_provider = llm_match.group(1) if llm_match else None
                else:
                    llm_provider = None
                
                use_ollama = llm_provider and llm_provider.lower() == 'ollama'
            except:
                # 설정 파일을 읽을 수 없는 경우 기본값 사용
                pass
            
            # 시스템 메시지 파일 경로 (Ollama 사용 시 _ollama 접미사 추가)
            if use_ollama:
                message_file = os.path.join(current_dir, 'config', f'{message_type}_system_message_ollama.txt')
                # Ollama용 파일이 없으면 기본 파일로 폴백
                if not os.path.exists(message_file):
                    message_file = os.path.join(current_dir, 'config', f'{message_type}_system_message.txt')
            else:
                message_file = os.path.join(current_dir, 'config', f'{message_type}_system_message.txt')
            
            # 파일이 존재하는지 확인
            if not os.path.exists(message_file):
                self.logger.error(f"시스템 메시지 파일을 찾을 수 없음: {message_file}")
                return ""
            
            # 파일 읽기
            with open(message_file, 'r', encoding='utf-8') as f:
                message = f.read().strip()

            # 사용자 글로벌 프롬프트가 있으면 시스템 메시지에도 추가
            if hasattr(self, 'executive_summary') and self.executive_summary:
                user_global_prompt = self._extract_user_global_prompt(self.executive_summary)
                if user_global_prompt:
                    message += f"\n\n[중요] 사용자가 지정한 다음 형식 규칙을 반드시 준수하세요:\n{user_global_prompt}"
                
            return message
            
        except Exception as e:
            self.logger.error(f"시스템 메시지 로드 실패: {str(e)}")
            return ""

    def _extract_user_global_prompt(self, executive_summary: str) -> str:
        """사용자가 정의한 글로벌 프롬프트 추출"""
        if not executive_summary:
            return ""
        
        # # prompt: 태그 찾기
        prompt_tag = "# prompt:"
        prompt_index = executive_summary.find(prompt_tag)
        
        if prompt_index == -1:
            return ""  # 태그가 없으면 빈 문자열 반환
        
        # 태그 이후의 내용 추출
        user_prompt = executive_summary[prompt_index + len(prompt_tag):].strip()
        
        # 다음 # 태그가 있으면 그 전까지만 추출
        next_tag_index = user_prompt.find("\n#")
        if next_tag_index != -1:
            user_prompt = user_prompt[:next_tag_index].strip()
        
        return user_prompt

    async def get_rag_results_for_ollama(self, query: str, max_results: int = 3) -> str:
        """
        Ollama 모델에 적합한 형태로 RAG 검색 결과를 가져오는 함수
        
        Args:
            query: 검색 쿼리
            max_results: 반환할 최대 결과 수
            
        Returns:
            str: 요약된 RAG 검색 결과
        """
        try:
            # 외부 RAG 클라이언트 우선 사용 (자원 중복 방지)
            if self.external_rag_client and self.external_rag_client.is_available:
                self.logger.debug("✅ 외부 RAG 클라이언트 사용")
                rag_results = await self.external_rag_client.search(query, max_results=max_results, user_id=self.user_id, rag_search_scope='personal')
            else:
                # 폴백: 내부 DocumentProcessor 사용
                self.logger.debug("⚠️ 폴백: 내부 DocumentProcessor 사용 (외부 클라이언트 불가)")
                if self.rag is None:
                    self.rag = DocumentProcessor()
                rag_results = await self.rag.search(query, user_id=self.user_id, temp_settings=None, rag_search_scope='personal')
            
            if not rag_results:
                self.logger.debug(f"RAG 검색 결과 없음: {query}")
                return ""
                
            # 결과 요약 및 정제
            summarized_results = []
            for i, result in enumerate(rag_results[:max_results], 1):
                # 소스 파일명만 추출
                source = os.path.basename(result.get('source', '알 수 없는 출처'))
                content = result.get('content', '')
                
                # 내용 길이 제한 (Ollama 컨텍스트 길이 고려)
                if len(content) > 500:
                    content = content[:497] + "..."
                    
                summarized_results.append(f"[출처 {i}: {source}]\n{content}\n")
                
            # 결과 조합
            summary = "\n".join(summarized_results)
            self.logger.debug(f"RAG 검색 결과 요약 (길이: {len(summary)}자)")
            return summary
            
        except Exception as e:
            self.logger.error(f"RAG 검색 중 오류: {str(e)}")
            return ""

    async def get_web_search_results_for_ollama(self, query: str, max_results: int = 3) -> str:
        """
        Ollama 모델에 적합한 형태로 웹 검색 결과를 가져오는 함수
        
        Args:
            query: 검색 쿼리
            max_results: 반환할 최대 결과 수
            
        Returns:
            str: 요약된 웹 검색 결과
        """
        try:
            # 외부 웹검색 클라이언트 우선 사용
            if self.external_web_client and self.external_web_client.is_available:
                self.logger.debug("✅ 외부 웹검색 클라이언트 사용")
                search_results = await self.external_web_client.search(query, max_results=max_results)
            else:
                # 내부 웹검색 서비스 사용 (fallback)
                self.logger.debug("⚠️ 내부 웹검색 서비스 사용 (외부 클라이언트 불가)")
                self.web = WebSearchService()        
                
                # 웹 검색 수행
                search_results = await self.web.search(query, max_results=max_results)
            
            if not search_results:
                self.logger.debug(f"웹 검색 결과 없음: {query}")
                return ""
                
            # 결과 요약 및 정제
            summarized_results = []
            for i, result in enumerate(search_results, 1):
                title = result.get('title', '제목 없음')
                snippet = result.get('snippet', result.get('description', ''))
                url = result.get('url', '')
                
                # 내용 길이 제한
                if len(snippet) > 300:
                    snippet = snippet[:297] + "..."
                    
                summarized_results.append(f"[웹 검색 결과 {i}]\n제목: {title}\n내용: {snippet}\n출처: {url}\n")
                
            # 결과 조합
            summary = "\n".join(summarized_results)
            self.logger.debug(f"웹 검색 결과 요약 (길이: {len(summary)}자)")
            return summary
            
        except Exception as e:
            self.logger.error(f"웹 검색 중 오류: {str(e)}")
            return ""

    async def analyze_business_info(self, executive_summary: str, progress_callback=None) -> Dict:
        """사업 정보 분석"""
        # 초기 단계 콜백 (SSE 연결 안정화를 위한 지연)
        if progress_callback:
            await asyncio.sleep(6)  # SSE 연결이 완전히 확립될 시간 제공 (3초 → 6초로 연장)
            await progress_callback("메타데이터 분석 준비 중...", 5, "metadata_init", {
                "section_name": "metadata_init",
                "title": "메타데이터 분석 준비",
                "content": "📊 문서 작성을 위한 설정을 초기화하고 있습니다...",
                "status": "processing"
            })
        
        max_retries = 3  # 최대 재시도 횟수
        enable_rag_search = True  # RAG 검색 활성화 여부
        enable_web_search = False  # 웹 검색 활성화 여부
        guidelines_context = []
        web_search_results = []        
        # 환경 변수에서 AI 설정 가져오기
        config = load_config()
        provider = config.get('USE_LLM', 'openai')
        
        # Ollama 사용 시 검색 기능 비활성화
        if provider == 'ollama': 
            enable_rag_search = True
            enable_web_search = False
            
        search_keywords = [
            "사업계획서 평가기준",
            "사업계획서 작성지침",
            "제안서 평가항목"
        ]        
        
        for attempt in range(max_retries):
            try:
                if enable_web_search:
                    self.logger.debug("-----------------웹 검색 단계 콜백-------------------")
                    # 웹 검색 단계 콜백
                    if progress_callback:
                        await progress_callback("웹 검색 수행 중...", 7, "web_search", {
                            "section_name": "web_search",
                            "title": "웹 검색 수행",
                            "content": "🔍 작성 지침과 평가기준을 검색하고 있습니다...",
                            "status": "processing"
                        })
                    
                    print("  └ 웹 검색 수행 중...")
                    # WebSearchService를 사용한 병렬 검색
                    web_search_results = []
                    for keyword in search_keywords:
                        if provider == 'ollama':
                            results = await self.get_web_search_results_for_ollama(keyword, max_results=2)
                            if results:
                                # 문자열 결과를 리스트 형태로 변환하여 추가
                                web_search_results.append({
                                    'source': 'Web 검색 결과',
                                    'content': results
                                })                    
                        else:
                            results = await self.web.search(keyword, max_results=5)
                            web_search_results.extend(results)
                    
                    if not web_search_results:
                        raise ValueError("검색 결과가 없습니다")
                                    
                if enable_rag_search:
                    self.logger.debug("----------------- RAG 검색 단계 콜백-------------------")
                    # RAG 검색 단계 콜백
                    if progress_callback:
                        await progress_callback("RAG 검색 수행 중...", 8, "rag_search", {
                            "section_name": "rag_search",
                            "title": "지식베이스 검색 수행",
                            "content": "📚 내부 문서에서 작성 지침과 평가기준을 검색하고 있습니다...",
                            "status": "processing"
                        })
                    
                    self.logger.debug("RAG 검색으로 평가기준/작성지침 수집 중...")
                    # 1. 먼저 RAG 검색으로 평가기준과 작성지침 수집
                    guidelines_results = []
                    
                    for keyword in search_keywords:
                        if provider == 'ollama':
                            result_text = await self.get_rag_results_for_ollama(keyword)
                            if result_text:
                                # 문자열 결과를 리스트 형태로 변환하여 추가
                                guidelines_results.append({
                                    'source': '검색 결과',
                                    'content': result_text
                                })
                        else:
                            # 외부 RAG 클라이언트 우선 사용 (자원 중복 방지)
                            if self.external_rag_client and self.external_rag_client.is_available:
                                results = await self.external_rag_client.search(keyword, max_results=5, user_id=self.user_id, rag_search_scope='personal')
                            else:
                                # 폴백: 내부 DocumentProcessor 사용
                                if self.rag is None:
                                    self.rag = DocumentProcessor()
                                results = await self.rag.search(keyword, user_id=self.user_id, temp_settings=None, rag_search_scope='personal')
                            guidelines_results.extend(results)
                    
                    # 검색된 지침들을 하나의 문맥으로 통합
                    guidelines_context = "=== 작성 지침 및 평가기준 ===\n\n"
                    guidelines_context += "\n\n".join([
                        f"=== {result.get('source', '문서')} ===\n{result.get('content', '')}"
                        for result in guidelines_results
                    ])
                    if web_search_results:
                        guidelines_context += "\n\n"
                        guidelines_context += "=== 웹 검색 결과 ===\n\n"
                        guidelines_context += "\n\n".join([
                            f"=== {result.get('source', '문서')} ===\n{result.get('content', '')}"
                            for result in web_search_results
                        ])
                
                # 시스템 메시지 로드
                system_message = self._load_system_message('business_info')
                if not system_message:
                    raise ValueError("사업 정보 분석을 위한 시스템 메시지를 로드할 수 없습니다.")

                # 스키마 정의 수정
                json_schema = {
                    "type": "object",
                    "required": ["product_name", "problem_to_solve", "value_proposition", "key_features", "target_market"],
                    "properties": {
                        "product_name": {
                            "type": "string", 
                            "maxLength": 50,
                            "description": "제품/서비스의 공식 명칭 (50자 이내)",
                            "examples": ["AI 기반 스마트 공정 최적화 시스템", "지능형 물류 관리 플랫폼"]
                        },
                        "problem_to_solve": {
                            "type": "string",
                            "description": "해결하고자 하는 핵심 문제/니즈"
                        },
                        "value_proposition": {
                            "type": "string",
                            "description": "제품/서비스가 제공하는 핵심 가치"
                        },
                        "key_features": {
                            "type": "array",
                            "description": "핵심 기능과 특징 목록",
                            "items": {
                                "type": "object",
                                "required": ["feature", "description"],
                                "properties": {
                                    "feature": {"type": "string"},
                                    "description": {"type": "string"}
                                }
                            }
                        },
                        "target_market": {
                            "type": "object",
                            "required": ["market_characteristics", "market_size", "customer_segments", "region"],
                            "properties": {
                                "market_characteristics": {"type": "string"},
                                "market_size": {"type": "string"},
                                "customer_segments": {
                                    "type": "array",
                                    "items": {"type": "string"}
                                },
                                "region": {"type": "string"}
                            }
                        }
                    }
                }

                # 시스템 메시지에 가이드라인 컨텍스트 추가
                system_message +=f"""{system_message}\n\n{guidelines_context}"""
                
                user_prompt = f"""
다음 사업계획서 내용을 분석하여 구조화된 정보를 추출해주세요.

=== 사업 개요 ===
{executive_summary}
"""

                # AI 분석 단계 콜백
                if progress_callback:
                    await progress_callback("AI 메타데이터 분석 중...", 9, "ai_analysis", {
                        "section_name": "ai_analysis",
                        "title": "AI 사업정보 분석",
                        "content": "🤖 AI가 내용을 분석하여 핵심 정보를 추출하고 있습니다...",
                        "status": "processing"
                    })
                
                self.logger.debug("사업 정보 분석 중...")
                # API 호출 및 응답 처리
                ai_params = self._get_ai_params()
                response = await self.call_ai_api(
                    prompt=user_prompt,
                    require_json=True,
                    json_schema=json_schema,
                    system_message=system_message,
                    **ai_params
                )
                
                self.logger.debug("=== AI 응답 내용 ===")
                self.logger.debug(f"{json.dumps(response, ensure_ascii=False, indent=2)}")
                self.logger.debug("==================")
                
                # 응답 검증
                self.logger.debug("---------------------------응답 검증 시작...")                
                if not isinstance(response, dict):
                    self.logger.error(f"응답 타입 오류: {type(response)}")
                    raise ValueError("응답이 JSON 형식이 아닙니다")

                # 필수 필드 검증
                required_fields = ["product_name", "problem_to_solve", "value_proposition", "key_features", "target_market"]
                missing_fields = [field for field in required_fields if field not in response]
                if missing_fields:
                    self.logger.error(f"필수 필드 누락: {missing_fields}")
                    self.logger.error(f"현재 필드: {list(response.keys())}")
                    raise ValueError(f"필수 필드 누락: {', '.join(missing_fields)}")

                # 서비스/제품명 검증
                self.logger.debug(f"서비스/제품명 검증: {response['product_name']}")
                if not response["product_name"]:
                    self.logger.error("서비스/제품명이 비어있음")
                    raise ValueError("서비스/제품명이 비어있습니다")
                response["product_name"] = response["product_name"][:50]

                # 목표 시장 정보 검증
                self.logger.debug("목표 시장 정보 검증:")
                target_market = response.get("target_market")
                self.logger.debug(f"목표 시장 데이터: {json.dumps(target_market, ensure_ascii=False, indent=2)}")
                
                if not isinstance(target_market, dict):
                    self.logger.error(f"목표 시장 타입 오류: {type(target_market)}")
                    raise ValueError("목표 시장 정보가 올바른 형식이 아닙니다")

                required_market_fields = ["market_characteristics", "market_size", "customer_segments", "region"]
                missing_market_fields = [field for field in required_market_fields if field not in target_market]
                if missing_market_fields:
                    self.logger.error(f"목표 시장 필수 필드 누락: {missing_market_fields}")
                    self.logger.error(f"현재 필드: {list(target_market.keys())}")
                    raise ValueError(f"목표 시장 정보 필수 필드 누락: {', '.join(missing_market_fields)}")

                if not isinstance(target_market["customer_segments"], list):
                    self.logger.error(f"고객 세그먼트 타입 오류: {type(target_market['customer_segments'])}")
                    raise ValueError("고객 세그먼트는 배열 형식이어야 합니다")

                # 주요 기능 및 특징 검증
                self.logger.debug("주요 기능 및 특징 검증:")
                features = response.get("key_features", [])
                self.logger.debug(f"기능 목록: {json.dumps(features, ensure_ascii=False, indent=2)}")
                
                if not isinstance(features, list):
                    self.logger.error(f"기능 목록 타입 오류: {type(features)}")
                    raise ValueError("주요 기능 및 특징은 배열 형식이어야 합니다")
                
                for idx, feature in enumerate(features):
                    if not isinstance(feature, dict) or "feature" not in feature or "description" not in feature:
                        self.logger.error(f"기능 항목 {idx} 형식 오류: {feature}")
                        raise ValueError("주요 기능 및 특징의 형식이 올바르지 않습니다")

                self.logger.debug("모든 검증 통과")

                # 검증 통과 시 캐시 저장
                if hasattr(self, 'cache_paths') and self.cache_paths:
                    self.logger.debug("캐시 저장 중...")
                    try:
                        with open(self.cache_paths['business_info'], 'w', encoding='utf-8') as f:
                            json.dump({
                                'business_info': response,
                                'created_at': datetime.now().isoformat()
                            }, f, ensure_ascii=False, indent=2)
                        self.logger.debug("캐시 저장 완료")
                    except Exception as e:
                        self.logger.error(f"캐시 저장 실패: {str(e)}")
                        self.logger.error(f"사업 정보 캐시 저장 실패: {str(e)}")

                # 완료 단계 콜백
                if progress_callback:
                    await progress_callback("메타데이터 분석 완료", 10, "metadata_complete", {
                        "section_name": "metadata_complete",
                        "title": "메타데이터 분석 완료",
                        "content": f"✅ 기본정보 분석이 완료되었습니다!\n\n📋 주제: {response.get('product_name', '정보없음')}\n🎯 목표시장: {response.get('target_market', {}).get('market_characteristics', '정보없음')}\n💡 핵심가치: {response.get('value_proposition', '정보없음')[:100]}{'...' if len(response.get('value_proposition', '')) > 100 else ''}",
                        "status": "completed"
                    })
                
                self.logger.debug("사업 정보 분석 완료")
                return response

            except Exception as e:
                self.logger.error(f"시도 {attempt + 1}/{max_retries} 실패: {str(e)}")
                if attempt == max_retries - 1:
                    self.logger.error("모든 재시도 실패")
                    # SectionConfig 등 객체가 들어갈 가능성 방지
                    from dataclasses import asdict
                    # executive_summary에서 최소한의 정보 추출 시도 (없으면 빈 값)
                    return {
                        "product_name": "",
                        "problem_to_solve": "",
                        "value_proposition": "",
                        "key_features": [],
                        "target_market": {
                            "market_characteristics": "",
                            "market_size": "",
                            "customer_segments": [],
                            "region": ""
                        },
                        "error": f"사업 정보 분석 실패: {str(e)}"
                    }
                self.logger.debug(f"재시도 중... ({attempt + 2}/{max_retries})")
                await asyncio.sleep(1)  # 재시도 전 잠시 대기

    async def analyze_competitors(self, business_info: Dict) -> Dict:
        """경쟁사 분석"""
        enable_rag_search = False  # RAG 검색 활성화 여부
        enable_web_search = True  # 웹 검색 활성화 여부
        web_search_results = []
        rag_search_results = []
        # 환경 변수에서 AI 설정 가져오기
        config = load_config()
        provider = config.get('USE_LLM', 'openai')        

        # Ollama 사용 시 검색 기능 비활성화
        if provider == 'ollama': 
            enable_rag_search = False
            enable_web_search = True
                
        try:
            print("  └ 경쟁사 정보 수집 중...")
            
            # 캐시된 경쟁사 정보 변환
            if 'competitors' in business_info:
                cached_competitors = business_info['competitors']
                competitors_data = {
                    "competitors": [],
                    "searched_competitors": []  # 프롬프트 치환용
                }
                
                for comp in cached_competitors:
                    competitor = {
                        "company": comp['company'],
                        "main_products": comp.get('main_products', comp.get('description', '정보 없음')),
                        "strengths": comp.get('strengths', ["정보 없음"]),
                        "weaknesses": comp.get('weaknesses', ["정보 없음"]),
                        "market_share": comp.get('market_share', "정보 없음"),
                        "revenue": comp.get('revenue', "정보 없음"),
                        "source": comp.get('url', comp.get('source', '정보 없음'))
                    }
                    competitors_data['competitors'].append(competitor)
                    # 프롬프트 치환용 리스트에 추가
                    competitors_data['searched_competitors'].append(f"{comp['company']}: {competitor['main_products']}")

            # 캐시된 정보가 없는 경우 새로 검색
            service_name = business_info.get('product_name', '')
            if not service_name:
                raise ValueError("서비스/제품명이 없습니다")
            
            # 산업 분야 정보 추출
            industry = business_info.get('industry', '')
            business_type = business_info.get('business_type', '')
            target_market = business_info.get('target_market', {})
            
            if isinstance(target_market, str):
                market_characteristic = target_market
            elif isinstance(target_market, dict):
                market_characteristic = target_market.get('market_characteristics', '')
            else:
                market_characteristic = ''
            
            # 검색 키워드 구성
            search_keywords = []
            
            # 서비스/제품 관련 경쟁사 키워드
            if service_name:
                search_keywords.extend([
                    f"{service_name} 경쟁사",
                    f"{service_name} 유사 서비스",
                    f"{service_name} 대체 서비스"
                ])
            
            # 산업 관련 경쟁사 키워드
            if industry:
                search_keywords.extend([
                    f"{industry} 주요 기업",
                    f"{industry} 시장 점유율",
                    f"{industry} 경쟁 현황"
                ])
            
            # 시장 특성 관련 경쟁사 키워드
            if market_characteristic:
                search_keywords.extend([
                    f"{market_characteristic} 경쟁사 분석",
                    f"{market_characteristic} 시장 경쟁 현황"
                ])
            
            # 사업 유형 관련 경쟁사 키워드
            if business_type:
                search_keywords.extend([
                    f"{business_type} 주요 기업",
                    f"{business_type} 경쟁 현황"
                ])
            
            # 기존 경쟁사 관련 추가 검색
            if 'competitors' in business_info:
                for comp in business_info['competitors']:
                    company_name = comp.get('company', '')
                    if company_name:
                        search_keywords.extend([
                            f"{company_name} 기업 정보",
                            f"{company_name} 사업 현황",
                            f"{company_name} {service_name}"
                        ])
            
            # 중복 제거 및 빈 문자열 제거
            search_keywords = list(set(filter(None, search_keywords)))
            
            if enable_web_search:
                print("  └ 웹 검색 수행 중...")
                # WebSearchService를 사용한 병렬 검색
                web_search_results = []
                for keyword in search_keywords:
                    if provider == 'ollama':
                        results = await self.get_web_search_results_for_ollama(keyword, max_results=2)
                        if results:
                            # 문자열 결과를 리스트 형태로 변환하여 추가
                            web_search_results.append({
                                'source': 'Web 검색 결과',
                                'content': results
                            })                    
                    else:
                        results = await self.web.search(keyword, max_results=5)
                        web_search_results.extend(results)
                
                if not web_search_results:
                    raise ValueError("검색 결과가 없습니다")

            if enable_rag_search:
                self.logger.debug("RAG 검색 중...")
                rag_search_results = []
                
                for keyword in search_keywords:
                    if provider == 'ollama':
                        result_text = await self.get_rag_results_for_ollama(keyword)
                        if result_text:
                            # 문자열 결과를 리스트 형태로 변환하여 추가
                            rag_search_results.append({
                                'source': 'Ollama 검색 결과',
                                'content': result_text
                            })
                    else:
                        # 외부 RAG 클라이언트 우선 사용
                        if self.external_rag_client and self.external_rag_client.is_available:
                            results = await self.external_rag_client.search(keyword, max_results=5, user_id=self.user_id, rag_search_scope='personal')
                        else:
                            # 폴백: 내부 DocumentProcessor 사용
                            if self.rag is None:
                                self.rag = DocumentProcessor()
                            results = await self.rag.search(keyword, user_id=self.user_id, temp_settings=None, rag_search_scope='personal')
                        rag_search_results.extend(results)
                
                # 검색된 지침들을 하나의 문맥으로 통합
                rag_search_results = "=== 작성 지침 및 평가기준 ===\n\n"
                rag_search_results += "\n\n".join([
                    f"=== {result.get('source', '문서')} ===\n{result.get('content', '')}"
                    for result in rag_search_results
                ])
            
            print("  └ 경쟁사 정보 분석 중...")
            
            # 시스템 메시지 로드
            system_message = self._load_system_message('competitors')
            if not system_message:
                raise ValueError("경쟁사 분석을 위한 시스템 메시지를 로드할 수 없습니다.")
            
            # 검색 결과에서 기업 정보 추출
            json_schema_example = {
                "competitors": [
                    {
                        "company": "기업명 (실제 기업명만 사용)",
                        "main_products": "주요 제품/서비스 설명",
                        "strengths": [
                            "강점1 (구체적 수치/사실 포함)",
                            "강점2 (구체적 수치/사실 포함)",
                            "강점3 (구체적 수치/사실 포함)"
                        ],
                        "weaknesses": [
                            "약점1 (구체적 수치/사실 포함)",
                            "약점2 (구체적 수치/사실 포함)",
                            "약점3 (구체적 수치/사실 포함)"
                        ],
                        "market_share": "시장 점유율 (가능한 경우 구체적 수치)",
                        "revenue": "매출액 (최근 연도 기준)",
                        "source": "정보 출처 URL"
                    }
                ]
            }

            prompt = f"""
다음 검색 결과에서 {service_name}와 관련된 실제 경쟁사 정보를 추출해주세요.
가상의 기업명(예: A사, B사 등)은 절대 사용하지 말고, 반드시 실제 기업명을 사용해야 합니다.

검색 결과:
{json.dumps(web_search_results, ensure_ascii=False, indent=2)}

{json.dumps(rag_search_results, ensure_ascii=False, indent=2)}

응답은 반드시 다음 JSON 스키마를 따라야 합니다:
{json.dumps(json_schema_example, ensure_ascii=False, indent=2)}

응답에는 실제로 확인된 경쟁사 정보만 포함하고, 불확실한 정보는 제외하세요.
"""
            # OpenAI API로 경쟁사 정보 추출
            ai_params = self._get_ai_params()
            competitors_data = await self.call_ai_api(
                prompt=prompt,
                require_json=True,
                json_schema={
                    "type": "object",
                    "required": ["competitors"],
                    "properties": {
                        "competitors": {
                            "type": "array",
                            "minItems": 1,
                            "items": {
                                "type": "object",
                                "required": ["company", "main_products", "strengths", "weaknesses", "market_share", "revenue", "source"],
                                "properties": {
                                    "company": {"type": "string", "minLength": 1},
                                    "main_products": {"type": "string", "minLength": 1},
                                    "strengths": {
                                        "type": "array",
                                        "minItems": 1,
                                        "items": {"type": "string", "minLength": 1}
                                    },
                                    "weaknesses": {
                                        "type": "array",
                                        "minItems": 1,
                                        "items": {"type": "string", "minLength": 1}
                                    },
                                    "market_share": {"type": "string"},
                                    "revenue": {"type": "string"},
                                    "source": {"type": "string", "minLength": 1}
                                }
                            }
                        }
                    }
                },
                system_message=system_message
            )

            if not competitors_data or 'competitors' not in competitors_data or not competitors_data['competitors']:
                return {
                    "competitors": [],
                    "error": "검색 결과에서 실제 경쟁사 정보를 찾을 수 없습니다."
                }

            # 유효한 경쟁사 정보만 필터링
            valid_competitors = [
                comp for comp in competitors_data['competitors']
                if comp.get('company') and comp['company'] != "정보 없음" and
                comp.get('main_products') and comp['main_products'] != "정보 없음"
            ]

            if not valid_competitors:
                return {
                    "competitors": [],
                    "error": "유효한 경쟁사 정보를 찾을 수 없습니다."
                }

            print(f"  └ {len(valid_competitors)}개의 경쟁사 정보 수집 완료")
            return {"competitors": valid_competitors}

        except Exception as e:
            self.logger.error("경쟁사 분석 실패: %s", str(e))
            traceback.print_exc()
            # 기존 competitors 정보 반환 (SectionConfig면 dict로 변환)
            competitors = business_info.get('competitors', [])
            # competitors가 SectionConfig 객체 리스트일 수 있음
            from dataclasses import asdict
            if isinstance(competitors, SectionConfig):
                competitors = [asdict(competitors)]
            elif isinstance(competitors, list):
                competitors = [
                    asdict(c) if isinstance(c, SectionConfig) else c
                    for c in competitors
                ]
            return {
                "competitors": competitors,
                "error": f"경쟁사 분석 실패: {str(e)}"
            }

    async def analyze_target_market(self, business_info: Dict) -> Dict:
        """목표 시장 분석"""
        enable_rag_search = True  # RAG 검색 활성화 여부
        enable_web_search = True  # 웹 검색 활성화 여부
        web_search_results = []
        rag_search_results = []
        # 환경 변수에서 AI 설정 가져오기
        config = load_config()
        provider = config.get('USE_LLM', 'openai')

        # Ollama 사용 시 검색 기능 비활성화
        if provider == 'ollama': 
            enable_rag_search = False
            enable_web_search = True
        
        # 검색 키워드 미리 정의
        search_keywords = []
        
        try:
            # 기존 목표 시장 정보 확인
            if 'target_market' in business_info:
                return business_info['target_market']
            
            # 서비스/제품명 확인
            service_name = business_info.get('product_name', '')
            if not service_name:
                raise ValueError("서비스/제품명이 없습니다")
            
            # 목표 시장 정보 추출
            target_market = business_info.get('target_market', {})
            if isinstance(target_market, str):
                market_characteristic = target_market
            elif isinstance(target_market, dict):
                market_characteristic = target_market.get('market_characteristics', '')
            elif isinstance(target_market, list) and target_market:
                market_characteristic = target_market[0]
            else:
                market_characteristic = ''
            
            if not market_characteristic:
                raise ValueError("목표 시장 정보가 없습니다")

            # 산업 분야 정보 추출
            industry = business_info.get('industry', '')
            business_type = business_info.get('business_type', '')
            target_customer = business_info.get('target_customer', '')
                        
            # 검색 키워드 구성
            search_keywords = []
            
            # 시장 규모 관련 키워드
            if service_name:
                search_keywords.extend([
                    f"{service_name} 시장 규모",
                    f"{service_name} 시장 전망",
                    f"{service_name} 산업 동향"
                ])
            
            # 산업/시장 동향 관련 키워드
            if industry:
                search_keywords.extend([
                    f"{industry} 시장 동향",
                    f"{industry} 산업 현황",
                    f"{industry} 시장 성장률"
                ])
            
            # 목표 시장 특성 관련 키워드
            if market_characteristic:
                search_keywords.extend([
                    f"{market_characteristic} 시장 동향",
                    f"{market_characteristic} 시장 현황",
                    f"{market_characteristic} 시장 규모"
                ])
            
            # 사업 유형 관련 키워드
            if business_type:
                search_keywords.extend([
                    f"{business_type} 시장 동향",
                    f"{business_type} 산업 현황"
                ])
            
            # 목표 고객 관련 키워드
            if target_customer:
                search_keywords.extend([
                    f"{target_customer} 소비 트렌드",
                    f"{target_customer} 시장 특성"
                ])
            
            # 중복 제거 및 빈 문자열 제거
            search_keywords = list(set(filter(None, search_keywords)))
            
            # 검색 결과 저장 리스트
            search_results = []
            
            # Ollama 모델 사용 시 검색 결과 제한
            max_search_results = 2 if provider.lower() == 'ollama' else 5
            
            # RAG 검색 수행 (활성화된 경우)
            if enable_rag_search:
                try:
                    print("  └ RAG 검색 수행 중...")
                    rag_search_results = []
                    
                    for keyword in search_keywords:
                        if provider.lower() == 'ollama':
                            result_text = await self.get_rag_results_for_ollama(keyword, max_results=max_search_results)
                            if result_text:
                                rag_search_results.append({
                                    'source': 'RAG 검색 결과',
                                    'content': result_text
                                })
                        else:
                            # 외부 RAG 클라이언트 우선 사용 (자원 중복 방지)
                            if self.external_rag_client and self.external_rag_client.is_available:
                                results = await self.external_rag_client.search(keyword, max_results=max_search_results, user_id=self.user_id, rag_search_scope='personal')
                            else:
                                # 폴백: 내부 DocumentProcessor 사용
                                if self.rag is None:
                                    self.rag = DocumentProcessor()
                                results = await self.rag.search(keyword, user_id=self.user_id, temp_settings=None, rag_search_scope='personal')
                            rag_search_results.extend(results)
                    
                    if rag_search_results:
                        search_results.extend(rag_search_results)
                except Exception as e:
                    self.logger.warning(f"RAG 검색 중 오류 발생: {e}")
            
            # 웹 검색 수행 (활성화된 경우)
            if enable_web_search:
                try:
                    print("  └ 웹 검색 수행 중...")
                    web_search_results = []
                    
                    for keyword in search_keywords:
                        if provider.lower() == 'ollama':
                            results = await self.get_web_search_results_for_ollama(keyword, max_results=max_search_results)
                            if results:
                                web_search_results.append({
                                    'source': 'Web 검색 결과',
                                    'content': results
                                })
                        else:
                            results = await self.web.search(keyword, max_results=max_search_results)
                            web_search_results.extend(results)
                    
                    if web_search_results:
                        search_results.extend(web_search_results)
                except Exception as e:
                    self.logger.warning(f"웹 검색 중 오류 발생: {e}")
            
            # 검색 결과가 없는 경우
            if not search_results:
                self.logger.warning("검색 결과가 없습니다. 기본 목표 시장 정보를 반환합니다.")
                return target_market
            
            # 검색 결과를 문자열로 변환
            search_context = ""
            if provider.lower() == 'ollama':
                # Ollama용 간소화된 문자열 형식
                search_context = "\n\n".join([f"=== {result['source']} ===\n{result['content']}" for result in search_results])
            else:
                # 일반 모델용 상세 형식
                search_context = json.dumps(search_results, ensure_ascii=False, indent=2)
            
            # 시스템 메시지 로드
            system_message = self._load_system_message('target_market')
            if not system_message:
                raise ValueError("목표 시장 분석을 위한 시스템 메시지를 로드할 수 없습니다.")
            
            # JSON 스키마 정의
            json_schema = {
                "type": "object",
                "required": ["market_characteristics", "market_size", "growth_rate", "customer_segments", "region", "competition_intensity", "entry_barriers", "market_trends"],
                "properties": {
                    "market_characteristics": {"type": "string"},
                    "market_size": {"type": "string"},
                    "growth_rate": {"type": "string"},
                    "customer_segments": {
                        "type": "array",
                        "items": {
                            "type": "object",
                            "required": ["segment_name", "characteristics", "size"],
                            "properties": {
                                "segment_name": {"type": "string"},
                                "characteristics": {"type": "string"},
                                "size": {"type": "string"}
                            }
                        }
                    },
                    "region": {"type": "string"},
                    "competition_intensity": {"type": "string"},
                    "entry_barriers": {"type": "string"},
                    "market_trends": {"type": "string"}
                }
            }
            
            # Ollama 사용 시 간소화된 스키마
            if provider.lower() == 'ollama':
                json_schema = {
                    "type": "object",
                    "required": ["market_characteristics", "market_size", "growth_rate", "customer_segments"],
                    "properties": {
                        "market_characteristics": {"type": "string"},
                        "market_size": {"type": "string"},
                        "growth_rate": {"type": "string"},
                        "customer_segments": {
                            "type": "array",
                            "items": {
                                "type": "object",
                                "required": ["segment_name", "characteristics"],
                                "properties": {
                                    "segment_name": {"type": "string"},
                                    "characteristics": {"type": "string"}
                                }
                            }
                        }
                    }
                }
            
            # 프롬프트 생성
            user_prompt = f"""
다음 검색 결과를 바탕으로 {service_name}의 목표 시장에 대한 분석 정보를 추출해주세요.

{search_context}

=== 기존 정보 ===
서비스/제품명: {service_name}
시장특성: {market_characteristic}
"""
            
            # Ollama 사용 시 프롬프트 간소화
            if provider.lower() == 'ollama':
                user_prompt = f"""
{service_name}의 목표 시장에 대한 분석 정보를 추출해주세요.
시장특성: {market_characteristic}

{search_context}
"""
            
            print("  └ 목표 시장 정보 분석 중...")
            # API 호출 및 응답 처리
            response = await self.call_ai_api(
                prompt=user_prompt,
                require_json=True,
                json_schema=json_schema,
                system_message=system_message
            )
            
            # 응답 검증 및 처리
            if isinstance(response, str):
                # 문자열로 반환된 경우 JSON 파싱 시도
                try:
                    response = json.loads(response)
                except json.JSONDecodeError as e:
                    self.logger.error(f"JSON 파싱 실패: {e}")
                    # 파싱 실패 시 기본 딕셔너리 구조 반환
                    self.logger.warning("JSON 파싱 실패로 기본 목표 시장 정보 반환")
                    response = {
                        "market_characteristics": "정보 파싱 오류",
                        "market_size": "분석 필요",
                        "customer_segments": ["일반 고객"],
                        "region": "국내"
                    }
            
            # 응답이 딕셔너리가 아닌 경우 기본값 설정
            if not isinstance(response, dict):
                self.logger.warning(f"예상치 못한 응답 타입: {type(response)}")
                response = {
                    "market_characteristics": "정보 분석 오류",
                    "market_size": "분석 필요", 
                    "customer_segments": ["일반 고객"],
                    "region": "국내"
                }
            
            print(f"  └ 목표 시장 정보 분석 완료")
            return response
            
        except Exception as e:
            self.logger.error(f"목표 시장 분석 실패: {str(e)}")
            traceback.print_exc()
            # 기존 목표 시장 정보 반환 (SectionConfig면 dict로 변환)
            target_market = business_info.get('목표 시장', {})
            if isinstance(target_market, SectionConfig):
                from dataclasses import asdict
                target_market = asdict(target_market)
            return target_market

    def _enforce_formatting(self, content: str, user_global_prompt: str) -> str:
        """사용자 프롬프트에 따라 응답 형식을 강제로 적용"""
        if not user_global_prompt or not content:
            return content
        
        lines = content.split('\n')
        formatted_lines = []
        current_level = 0
        
        for line in lines:
            line = line.strip()
            if not line:
                formatted_lines.append("")
                continue
            
            # 들여쓰기 수준 확인
            indent_level = len(line) - len(line.lstrip())
            line = line.lstrip()
            
            # 이미 글머리 기호가 있는지 확인
            if re.match(r'^[□❍\-]\s', line) or re.match(r'^[가-힣]\.\s', line):
                formatted_lines.append("    " * current_level + line)
                continue
            
            # 문장 시작이 명사구인지 확인
            if re.match(r'^[가-힣]+', line):
                # 들여쓰기 수준에 따라 글머리 기호 적용
                if indent_level == 0:
                    line = f"□ {line}"
                    current_level = 0
                elif indent_level <= 4:
                    line = f"❍ {line}"
                    current_level = 1
                else:
                    line = f"- {line}"
                    current_level = 2
                
                # 문장이 명사로 끝나지 않으면 수정
                if not re.search(r'[가-힣]\s*$', line):
                    line = re.sub(r'[.。]\s*$', '', line)  # 마침표 제거
                    line += "."
            
            formatted_lines.append("    " * current_level + line)
        
        return '\n'.join(formatted_lines)

    async def get_section_content(self, prompt: str, context: Dict) -> str:
        """섹션 내용 생성"""
        try:
            # prompt 매개변수가 SectionConfig 객체인 경우 안전하게 처리
            if isinstance(prompt, SectionConfig):
                prompt = prompt.prompt if prompt.prompt is not None else ""
            elif prompt is None:
                prompt = ""
            
            # prompt가 여전히 문자열이 아닌 경우 강제 변환
            if not isinstance(prompt, str):
                prompt = str(prompt) if prompt is not None else ""
            
            context_prompt = ""
            # 템플릿 속성 가져오기
            template = context.get('template')
            requires_web = False
            requires_rag = False
            requires_table = False
            requires_hide = False
            search_keywords = []
            requires_competitor_analysis = False
            requires_diagram = False
            section_type = "default"  # 기본값 설정

            if isinstance(template, SectionConfig):
                requires_web = template.requires_web
                requires_rag = template.requires_rag
                requires_table = template.requires_table
                requires_hide = template.requires_hide
                search_keywords = template.search_keywords or []
                section_type = template.section_type
            elif isinstance(template, dict):
                requires_web = template.get('requires_web', False)
                requires_rag = template.get('requires_rag', False)
                requires_table = template.get('requires_table', False)
                requires_hide = template.get('requires_hide', False)
                search_keywords = template.get('search_keywords', [])
                requires_competitor_analysis = template.get("requires_competitor_analysis", False)
                requires_diagram = template.get("requires_diagram", False)
                section_type = template.get("section_type", "default")

            # 시스템 메시지 로드
            system_message = self._load_system_message('section')
            if not system_message:
                raise ValueError("섹션 생성을 위한 시스템 메시지를 로드할 수 없습니다.")
            
            # 스타일 메시지 로드
            style_message = self._load_system_message('style')
            if not style_message:
                raise ValueError("섹션 생성을 위한 스타일 메시지를 로드할 수 없습니다.")            
                
            # 검색 결과 수집
            search_results = []
            if search_keywords:
                # 사업 정보에서 필요한 컨텍스트 추출
                business_info = context.get('business_info', {})
                service_name = business_info.get('product_name', '')
                industry = business_info.get('industry', '')
                target_market = business_info.get('target_market', {})
                if isinstance(target_market, str):
                    market_characteristic = target_market
                elif isinstance(target_market, dict):
                    market_characteristic = target_market.get('market_characteristics', '')
                else:
                    market_characteristic = ''

                # 검색 키워드에 사업 컨텍스트 추가
                formatted_keywords = []
                for keyword in search_keywords:
                    # '!' 접두사가 있는 경우 독립 검색어로 처리
                    if keyword.startswith('!'):
                        formatted_keywords.append(keyword[1:])  # '!' 제거하고 그대로 사용
                        continue                    
                    # 키워드 포맷팅 - context에서 SectionConfig 객체를 안전하게 처리
                    safe_context = dict(context)
                    if 'template' in safe_context and isinstance(safe_context['template'], SectionConfig):
                        safe_context['template'] = safe_context['template'].to_dict()
                    formatted_keyword = self._format_prompt(keyword, safe_context)
                    
                    # 사업 관련 키워드 조합
                    if service_name:
                        formatted_keywords.append(f"{service_name} {formatted_keyword}")
                    if industry:
                        formatted_keywords.append(f"{industry} {formatted_keyword}")
                    if market_characteristic:
                        formatted_keywords.append(f"{market_characteristic} {formatted_keyword}")
                        
                    # 기본 키워드도 포함
                    # formatted_keywords.append(formatted_keyword)
                
                # 중복 제거
                formatted_keywords = list(set(formatted_keywords))
                
                # AI가 검색할 키워드를 추출하는 함수추가
                async def extract_intelligent_keywords(prompt_text, context, max_keywords=3):
                    """
                    AI를 사용하여 프롬프트에서 검색에 최적화된 키워드를 자동 추출
                    
                    Args:
                        prompt_text: 섹션 프롬프트 텍스트
                        context: 비즈니스 컨텍스트
                        max_keywords: 최대 키워드 개수
                    
                    Returns:
                        List[str]: 추출된 키워드 목록
                    """
                    try:
                        business_info = context.get('business_info', {})
                        service_name = business_info.get('product_name', '')
                        industry = business_info.get('industry', '')
                        
                        keyword_extraction_prompt = f"""
                    다음 사업계획서 섹션 프롬프트를 분석하여 웹 검색과 문서 검색에 최적화된 키워드를 추출해주세요.

                    **사업 정보:**
                    - 서비스/제품명: {service_name}
                    - 산업분야: {industry}

                    **섹션 프롬프트:**
                    {prompt_text}

                    **추출 기준:**
                    1. 핵심 비즈니스 개념과 용어
                    2. 시장 분석에 필요한 키워드
                    3. 경쟁사 분석 관련 용어
                    4. 기술 및 트렌드 키워드
                    5. 법규 및 제도 관련 용어

                    **요구사항:**
                    - 검색 엔진에서 관련 정보를 찾기 쉬운 구체적인 키워드 생성
                    - 너무 일반적이거나 모호한 키워드는 제외
                    - 최대 {max_keywords}개의 키워드 추출
                    - 한국어와 영어 키워드 모두 포함 가능

                    JSON 형식으로 응답해주세요:
                    {{
                        "web_keywords": ["웹 검색용 키워드1", "웹 검색용 키워드2", ...],
                        "rag_keywords": ["문서 검색용 키워드1", "문서 검색용 키워드2", ...]
                    }}
                    """

                        keyword_schema = {
                            "type": "object",
                            "properties": {
                                "web_keywords": {
                                    "type": "array",
                                    "items": {"type": "string"},
                                    "description": "웹 검색에 최적화된 키워드 목록"
                                },
                                "rag_keywords": {
                                    "type": "array", 
                                    "items": {"type": "string"},
                                    "description": "문서 검색에 최적화된 키워드 목록"
                                }
                            },
                            "required": ["web_keywords", "rag_keywords"]
                        }
                        
                        # AI API 호출
                        response = await self.call_ai_api(
                            prompt=keyword_extraction_prompt,
                            require_json=True,
                            json_schema=keyword_schema,
                            system_message="당신은 사업계획서 작성을 위한 키워드 추출 전문가입니다."
                        )
                        
                        if isinstance(response, dict):
                            web_keywords = response.get('web_keywords', [])
                            rag_keywords = response.get('rag_keywords', [])
                            
                            # 비즈니스 컨텍스트와 조합하여 최종 키워드 생성
                            enhanced_keywords = []
                            
                            for keyword in web_keywords + rag_keywords:
                                # 기본 키워드
                                enhanced_keywords.append(keyword)
                                
                                # 서비스명과 조합
                                if service_name and service_name.strip():
                                    enhanced_keywords.append(f"{service_name} {keyword}")
                                
                                # 산업분야와 조합
                                if industry and industry.strip():
                                    enhanced_keywords.append(f"{industry} {keyword}")
                            
                            # 중복 제거 및 길이 제한
                            unique_keywords = list(set(enhanced_keywords))
                            
                            # 너무 긴 키워드나 특수문자가 많은 키워드 필터링
                            filtered_keywords = []
                            for kw in unique_keywords:
                                if len(kw.strip()) > 2 and len(kw) < 100:
                                    filtered_keywords.append(kw.strip())
                            
                            self.logger.debug(f"AI 추출 키워드 {len(filtered_keywords)}개: {filtered_keywords[:10]}")
                            return filtered_keywords[:max_keywords * 2]  # 여유있게 반환
                            
                        else:
                            self.logger.warning("AI 키워드 추출 응답이 올바르지 않음")
                            return []
                            
                    except Exception as e:
                        self.logger.error(f"AI 키워드 추출 실패: {str(e)}")
                        return []

                # 기존 키워드에 AI 추출 키워드 추가
                # 너무 많은 키워드 검색이 되어서 섹션 작성에 시간이 오래걸리기 때문에 풍부한 검색 결과를 얻는것이 필수인 경우에만 사용
                # if template and hasattr(template, 'prompt'):
                #     ai_keywords = await extract_intelligent_keywords(template.prompt, context)
                #     formatted_keywords.extend(ai_keywords)
                    
                #     # 최종 중복 제거
                #     formatted_keywords = list(set(formatted_keywords))
                #     self.logger.debug(f"총 키워드 개수 (AI 추출 포함): {len(formatted_keywords)}")                
                
                for keyword in formatted_keywords:
                    if requires_web:
                        # 외부 웹검색 클라이언트 우선 사용
                        if self.external_web_client and self.external_web_client.is_available:
                            web_results = await self.external_web_client.search(keyword)
                        else:
                            # 폴백: 내부 웹검색 서비스 사용
                            web_results = await self.web.search(keyword)
                        
                        if web_results:
                            # 웹 검색 결과에 type 필드가 없는 경우 추가
                            for result in web_results:
                                if 'type' not in result:
                                    result['type'] = 'web'
                            search_results.extend(web_results)
                            self.logger.debug(f"[WEB] 웹 검색 결과 {len(web_results)}개 추가됨 (키워드: {keyword})")
                    
                    if requires_rag:
                        # 외부 RAG 클라이언트 우선 사용
                        if self.external_rag_client and self.external_rag_client.is_available:
                            rag_results = await self.external_rag_client.search(keyword, max_results=5, user_id=self.user_id, rag_search_scope='personal')
                        else:
                            # 폴백: 내부 DocumentProcessor 사용
                            if self.rag is None:
                                self.rag = DocumentProcessor()
                            rag_results = await self.rag.search(keyword, user_id=self.user_id, temp_settings=None, rag_search_scope='personal')
                        
                        if rag_results:
                            # RAG 검색 결과에 type 필드가 없는 경우 추가
                            for result in rag_results:
                                if 'type' not in result:
                                    result['type'] = 'rag'
                            search_results.extend(rag_results)
                            self.logger.debug(f"[RAG] 검색 결과 {len(rag_results)}개 추가됨 (키워드: {keyword})")

            # 검색 결과 중복 제거
            seen_contents = set()
            unique_results = []
            for result in search_results:
                # 웹 검색 결과는 URL로 중복 체크
                if result.get('type') == 'web':
                    content_key = result.get('url', '')
                # RAG 검색 결과는 content로 중복 체크
                else:
                    content_key = result.get('content', '')[:200]  # 처음 200자만 비교

                if content_key and content_key not in seen_contents:
                    seen_contents.add(content_key)
                    unique_results.append(result)

            search_results = unique_results

            # 검색 결과 로깅
            self.logger.debug(f"총 수집된 검색 결과: {len(search_results)}개")
            self.logger.debug(f"- 웹 검색 결과: {len([r for r in search_results if r.get('type') == 'web'])}개")
            self.logger.debug(f"- RAG 검색 결과: {len([r for r in search_results if r.get('type') == 'rag'])}개")
           
            # 검색 결과가 있으면 추가
            if search_results:
                context_prompt += "## 관련 검색 결과:\n\n"
                
                # 웹 검색 결과와 RAG 검색 결과 분리
                web_results = [r for r in search_results if r.get('type') == 'web']
                rag_results = [r for r in search_results if r.get('type') == 'rag']
                
                if web_results:
                    context_prompt += "### 웹 검색 결과:\n\n"
                    for result in web_results:
                        if result.get('title'):
                            context_prompt += f"제목: {result['title']}\n"
                        if result.get('url'):
                            context_prompt += f"URL: {result['url']}\n"
                        if result.get('description'):
                            context_prompt += f"설명: {result['description']}\n"
                        if result.get('engine'):
                            context_prompt += f"검색엔진: {result['engine']}\n"
                        context_prompt += "\n"
                
                if rag_results:
                    context_prompt += "### 문서 검색 결과:\n\n"
                    for result in rag_results:
                        if result.get('source'):
                            context_prompt += f"출처: {result['source']}\n"
                        if result.get('content'):
                            context_prompt += f"내용: {result['content']}\n"
                        if result.get('score'):
                            context_prompt += f"관련도: {result['score']:.2f}\n"
                        context_prompt += "\n"

                # 검색 결과 활용 지침 추가
                context_prompt += "### 검색 결과 활용 지침:\n"
                context_prompt += "- 검색 결과의 정보를 우선적으로 활용하세요.\n"
                context_prompt += "- 검색 결과에 없는 정보는 생성하지 말고 '정보 부족'이라고 명시하세요.\n"
                context_prompt += "- 투입인력, 사업추진체계, 업무분장 등은 검색 결과의 실제 정보만 사용하세요.\n\n"

                # 디버그 로깅
                self.logger.debug("=== 검색 결과 프롬프트 ===")
                self.logger.debug(context_prompt)
                self.logger.debug("=========================")

            if context['generated_sections']:
                previous_sections_text = ""
                for section_path, content in context['generated_sections'].items():
                    previous_sections_text += f"=== {section_path} ===\n{content}\n\n"
                
                # 토큰 수 추정 및 요약 처리
                estimated_tokens = len(previous_sections_text) * 2  # 간단한 추정
                if estimated_tokens > BizPlanConfig.SUMMARY_THRESHOLD:
                    try:
                        summarized_context = summarize_content(previous_sections_text)
                        context_prompt = f"## 이전 섹션들의 요약:\n\n{summarized_context}\n\n"
                    except Exception as e:
                        self.logger.warning(f"이전 섹션 요약 실패: {str(e)}")
                        recent_sections = dict(list(context['generated_sections'].items())[-2:])
                        context_prompt = "## 최근 작성된 섹션들:\n\n"
                        for section_path, content in recent_sections.items():
                            context_prompt += f"=== {section_path} ===\n{content}\n\n"
                else:
                    context_prompt = "## 이전 섹션들의 내용:\n\n" + previous_sections_text

            # 현재 시점 정보 추가
            current_time = datetime.now()
            time_context = {
                'current_year': current_time.year,
                'current_month': current_time.month,
                'current_quarter': (current_time.month - 1) // 3 + 1,
                'current_half': 1 if current_time.month <= 6 else 2,
                'next_year': current_time.year + 1,
                'planning_start': current_time.strftime('%Y년 %m월'),
                'planning_period': f"{current_time.year}년 {current_time.month}월 ~ {current_time.year + 5}년 12월"
            }
            
            # 기존 컨텍스트에 시간 정보 추가
            context['time_info'] = time_context

            # 사용자 글로벌 프롬프트 가져오기 (추가된 부분)
            user_global_prompt = context.get('user_global_prompt', '')
                
            # 최종 프롬프트 구성
            formatted_prompt = f"""
# 시간 정보
- 현재 시점: {time_context['planning_start']}
- 계획 기간: {time_context['planning_period']}
- 기준 연도: {time_context['current_year']}년
- 현재 분기: {time_context['current_year']}년 {time_context['current_quarter']}분기
- 모든 계획과 일정은 반드시 {time_context['planning_start']} 이후부터 시작하도록 작성
- 과거 시점이 아닌 현재({time_context['planning_start']}) 시점을 기준으로 작성

# 주어진 필수 정보
{context.get('executive_summary', '')}

# 지금까지 작성한 내용:
{context_prompt}

이전 섹션들의 내용을 참고하여 일관된 문체로 다음 섹션을 작성해주세요.

# 현재 작성할 섹션:
{context['current_section']}/{context['current_subsection']}

{self._format_prompt(prompt, context['business_info'])}

# 작성 지침 :
1. 논리적 완성도를 중시하여 작성해주세요.
   - 주장의 명확성
   - 근거의 충실성
   - 논리 전개의 자연스러움

2. 전문성을 중시하여 작성해주세요.
   - 전문 용어의 적절한 사용
   - 업계 표준 반영
   - 최신 트렌드 반영

3. 실용성을 중시하여 작성해주세요.
   - 실행 가능한 제안
   - 구체적인 방안 제시
   - 현실적인 고려사항 반영
   
# 검색 결과 활용 방법:
- 만약 위에 '## 관련 검색 결과:' 섹션이 있다면, 이 정보는 신뢰할 수 있는 실제 데이터입니다.
- 반드시 이 검색 결과를 우선적으로 활용하여 응답을 작성하세요.
- 검색 결과에 있는 정보는 임의로 변경하거나 무시하지 말고 그대로 사용하세요.
- 검색 결과에 없는 정보는 임의로 생성하지 마세요.
- 특히 투입인력, 사업추진체계, 업무분장 등의 정보는 검색 결과에 있는 실제 정보만 사용하세요.
- 실제 기업이나 기관명이 아닌 가상의 기업명(A사, B사 등) 사용 금지
"""

            # 사용자 글로벌 프롬프트가 있으면 추가 (기존 작성 규칙보다 우선)
            if user_global_prompt:
                formatted_prompt += f"""
# 사용자 지정 작성 규칙
{user_global_prompt}
"""
            else:
                # 기본 작성 규칙 추가
                formatted_prompt += f"""
# 작성 스타일 규칙
{style_message}
"""

            # section_type이 manpower인 경우에 대한 특별 처리
            if context.get("section_type") == "manpower":
                # 사업명을 context에서 가져옴
                project_name = context.get("business_info", {}).get("project_name", "")
                
                # 투입인력 섹션을 위한 추가 지시사항
                additional_instructions = f"""
[인력/조직 구성 작성 필수 지침]
1. 반드시 "{project_name}" 사업의 인력/조직 정보만 사용하십시오.
2. 다른 사업의 인력/조직 정보는 절대 참고하거나 혼용하지 마십시오.
3. 검색 결과에서 "{project_name}" 사업의 관련 정보를 찾을 수 없는 경우:
    - "정보 없음"이라고 명시하십시오.
    - 임의로 다른 사업의 구성을 차용하지 마십시오.
4. 정보 작성 시 다음 순서로 작성하십시오:
    - 총 인력 규모 (제공된 경우)
    - 조직/직무별 구성
    - 세부 역할 및 책임
    - 조직 간 업무분장 (해당되는 경우)

※ 주의사항:
- 검색 결과에서 확인된 실제 정보만 사용하십시오.
- 불확실한 정보는 기재하지 마십시오.
- 추측성 정보는 포함하지 마십시오.               
"""
                
                # 기존 프롬프트에 추가 지시사항을 결합
                formatted_prompt = f"{formatted_prompt}\n\n{additional_instructions}"

            # OpenAI API 호출 시 시스템 메시지 포함
            response = await self.call_ai_api(
                prompt=formatted_prompt,
                require_json=True,
                json_schema={
                    "type": "object",
                    "required": ["content"],
                    "properties": {
                        "content": {"type": "string"}
                    }
                },
                system_message=system_message
            )

            # 응답 처리 및 파싱
            if isinstance(response, dict):
                if 'content' in response:
                    content = response['content']
                    # 사용자 프롬프트에 따라 형식 강제 적용
                    # if not user_global_prompt:
                    #     content = self._enforce_formatting(content, user_global_prompt)
                else:
                    # JSON 구조에서 실제 텍스트 내용 추출
                    def extract_text_content(obj):
                        if isinstance(obj, str):
                            return obj
                        elif isinstance(obj, dict):
                            # 중첩된 딕셔너리에서 텍스트 추출
                            texts = []
                            for key, value in obj.items():
                                if isinstance(value, str):
                                    texts.append(value)
                                elif isinstance(value, (dict, list)):
                                    result = extract_text_content(value)
                                    if result:
                                        texts.append(result)
                            return '\n\n'.join(text for text in texts if text)
                        elif isinstance(obj, list):
                            # 리스트에서 텍스트 추출
                            texts = [extract_text_content(item) for item in obj if item]
                            return '\n\n'.join(text for text in texts if text)
                        return ''

                    content = extract_text_content(response)
                    if not content:
                        raise ValueError("응답에서 텍스트 내용을 추출할 수 없습니다")
            else:
                content = str(response)

            # 헤더 제거
            content = self._remove_section_headers(content, context)
            
            # 불필요한 JSON 형식 문자 제거
            content = re.sub(r'[{}\[\]]', '', content)
            content = re.sub(r'"[^"]+"\s*:', '', content)
            content = re.sub(r'\n\s*\n\s*\n', '\n\n', content)
            
            # 최종 검증
            if not content.strip():
                raise ValueError("생성된 내용이 비어있습니다")
            
            # 자가검증 기능 사용 여부
            if context.get('use_validation', False):
                # 자가 검증 및 개선 수행
                improved_content = await self._validate_and_improve_content(content.strip(), context)
                # self.logger.debug(f"개선된 내용: {improved_content[:200]}...")
                final_content = improved_content
            else:
                # 자가검증 없이 원본 내용 사용
                final_content = content.strip()               

            return final_content

        except Exception as e:
            self.logger.error(f"섹션 내용 생성 실패: {str(e)}")
            traceback.print_exc()
            raise ValueError(f"섹션 내용 생성 실패: {str(e)}")

    async def _validate_and_improve_content(self, content: str, context: Dict) -> str:
        try:
            # 검색 키워드 생성
            keyword_generation_prompt = f"""
다음 내용에서 업데이트가 필요한 데이터를 찾고, 각각에 대한 최적의 검색 키워드를 생성해주세요.

분석할 내용:
{content}

검색이 필요한 정보 유형:
1. 시장/산업 데이터 (시장규모, 성장률, 트렌드)
2. 기업/경쟁사 정보 (매출, 점유율, 성과)
3. 기술/특허 현황
4. 정책/규제 동향
5. 소비자/시장 트렌드

응답 형식:
{{
    "search_items": [
        {{
            "original_text": "원본 텍스트",
            "current_year": "언급된 연도",
            "info_type": "market|company|tech|policy|trend",
            "search_keywords": {{
                "primary": ["핵심 키워드 1", "핵심 키워드 2"],
                "secondary": ["보조 키워드 1", "보조 키워드 2"],
                "english": ["english keyword 1", "english keyword 2"]
            }},
            "priority": 1,
            "context": "해당 정보가 사용된 맥락"
        }}
    ]
}}
            """

            search_items = await self.call_ai_api(
                prompt=keyword_generation_prompt,
                require_json=True
            )

            if not search_items.get("search_items"):
                self.logger.info("업데이트가 필요한 데이터가 발견되지 않았습니다.")
                return content

            # 우선순위에 따라 정렬
            search_items["search_items"].sort(key=lambda x: x["priority"])
            
            updated_content = content
            for item in search_items["search_items"]:
                # 검색 쿼리 구성
                search_queries = []
                
                # 기본 검색 쿼리 생성
                base_query = " ".join([
                    *item["search_keywords"]["primary"],
                    f"{datetime.now().year}",
                    "현황",
                    "통계"
                ])
                search_queries.append(base_query)
                
                # 영문 검색 쿼리 생성
                if item["search_keywords"].get("english"):
                    english_query = " ".join([
                        *item["search_keywords"]["english"],
                        str(datetime.now().year),
                        "statistics",
                        "report"
                    ])
                    search_queries.append(english_query)
                
                # 보조 키워드를 활용한 추가 쿼리
                if item["search_keywords"].get("secondary"):
                    secondary_query = " ".join([
                        *item["search_keywords"]["primary"][:1],
                        *item["search_keywords"]["secondary"],
                        f"{datetime.now().year}",
                    ])
                    search_queries.append(secondary_query)

                latest_data = None
                source_info = None

                # 각 쿼리로 검색 시도
                for query in search_queries:
                    if latest_data:
                        break

                    # RAG 검색 시도
                    try:
                        rag_results = await self.get_rag_results_for_ollama(
                            query=query,
                            max_results=3
                        )
                        if rag_results:
                            latest_data = rag_results
                            source_info = "내부 문서"
                            self.logger.debug(f"RAG 검색 성공: {query}")
                            continue
                    except Exception as e:
                        self.logger.warning(f"RAG 검색 실패: {e}")

                    # 웹 검색 시도
                    try:
                        web_results = await self.get_web_search_results_for_ollama(
                            query=query,
                            max_results=3
                        )
                        if web_results:
                            latest_data = web_results
                            source_info = "웹 검색"
                            self.logger.debug(f"웹 검색 성공: {query}")
                            continue
                    except Exception as e:
                        self.logger.warning(f"웹 검색 실패: {e}")

                if latest_data:
                    # 검색 결과를 바탕으로 내용 업데이트
                    update_prompt = f"""
다음 원본 텍스트를 최신 정보로 업데이트해주세요.

원본 텍스트:
{item["original_text"]}

컨텍스트 (해당 정보가 사용된 맥락):
{item["context"]}

최신 검색 결과:
{latest_data}

정보 유형: {item["info_type"]}

요구사항:
1. 검색된 최신 데이터를 맥락에 맞게 자연스럽게 통합
2. 원본의 논리적 흐름과 스타일 유지
3. 수치는 구체적으로 명시
4. 출처와 시점을 반드시 포함
5. 원본 맥락과의 일관성 유지

응답 형식:
{{
    "updated_text": "업데이트된 텍스트",
    "source": "구체적인 출처 정보",
    "date": "데이터 기준 시점",
    "confidence": "데이터 신뢰도 (0-1)"
}}
                    """

                    try:
                        update_result = await self.call_ai_api(
                            prompt=update_prompt,
                            require_json=True
                        )
                        
                        if update_result and update_result.get("confidence", 0) > 0.7:
                            updated_text = (
                                f"{update_result['updated_text']} "
                                f"(출처: {update_result['source']}, {update_result['date']})"
                            )
                            updated_content = updated_content.replace(
                                item["original_text"],
                                updated_text
                            )
                            self.logger.info(f"데이터 업데이트 완료: {item['info_type']}")
                        else:
                            self.logger.warning(f"신뢰도가 낮은 업데이트 건너뜀: {item['info_type']}")
                    except Exception as e:
                        self.logger.error(f"데이터 업데이트 실패: {e}")
                        continue
                else:
                    self.logger.warning(f"검색 결과를 찾을 수 없음: {item['info_type']}")

            # 최종 검증 및 개선 수행
            validation_prompt = f"""
다음 내용을 검토하고 개선하되, 개선된 최종 본문만 JSON 형식으로 응답해주세요.

검토 기준:
1. 논리적 완결성과 구조
- 주장과 근거가 명확하고 논리적 흐름이 자연스러운가?
- 각 섹션이 적절히 연결되어 있는가?
- 핵심 메시지가 명확히 전달되는가?

2. 데이터와 정보의 신뢰성
- 모든 수치와 통계에 출처가 명시되어 있는가?
- 시장 규모, 성장률 등 주요 수치의 근거가 명확한가?
- 경쟁사 정보가 객관적인가?

3. 전문성과 현실성
- 업계 용어와 최신 트렌드가 적절히 반영되었는가?
- 제안된 전략과 계획이 실현 가능한가?
- 위험 요소와 대응 방안이 구체적인가?

4. 문서 스타일 및 형식
- 전문적이고 객관적인 어조를 유지
- 명확하고 간결한 문장 구조 사용
- 적절한 단락 구분과 논리적 흐름 유지
- 일관된 용어와 표현 사용
           
# [중요] 작성 스타일 규칙
{context.get('style_rules', '')}

검토할 내용:
{updated_content}

응답 형식:
{{
"content": "개선된 본문 내용"
}}
            """

            # API 호출 및 응답 처리
            improved_content = await self.call_ai_api(
                validation_prompt,
                require_json=True,
                json_schema={
                    "type": "object",
                    "properties": {
                        "content": {"type": "string"}
                    },
                    "required": ["content"]
                }
            )
            
            if isinstance(improved_content, dict) and "content" in improved_content:
                result = improved_content["content"].strip()
                self.logger.debug(f"검증 및 개선 결과: {result[:200]}...")
                return result
            else:
                raise ValueError("AI 응답에서 content 필드를 찾을 수 없습니다")
                
        except Exception as e:
            self.logger.error(f"내용 검증 및 개선 실패: {str(e)}")
            return content  # 오류 발생 시 원본 내용 반환

    def _remove_section_headers(self, content: str, context: Dict) -> str:
        """섹션 헤더 제거"""
        # 사업명(서비스/제품명) 가져오기
        business_name = ""
        if 'business_info' in context and 'product_name' in context['business_info']:
            business_name = context['business_info']['product_name']
                    
        header_patterns = [
            rf"###\s*{context['current_section']}/{context['current_subsection']}.*?\n",
            rf"#{1,6}\s*{context['current_section']}/{context['current_subsection']}.*?\n",
            rf"{context['current_section']}/{context['current_subsection']}:.*?\n",
            rf"{context['current_section']}/{context['current_subsection']}\n",
            rf"{context['current_subsection']}:.*?\n",
            rf"{context['current_subsection']}\n"
        ]

        # 사업명(서비스/제품명) 제거 패턴 추가
        if business_name:
            header_patterns.extend([
                rf"#{1,6}\s*{re.escape(business_name)}.*?\n",
                rf"{re.escape(business_name)}:.*?\n",
                rf"{re.escape(business_name)}\n",
                rf"^{re.escape(business_name)}.*?\n",  # 줄 시작 부분에 사업명이 있는 경우
                rf"\n{re.escape(business_name)}.*?\n"  # 줄 중간에 사업명이 있는 경우
            ])
        
        result = content
        for pattern in header_patterns:
            result = re.sub(pattern, '', result, flags=re.IGNORECASE)
        
        return result

    async def get_chart_data(self, section_name: str, content: str) -> Optional[Dict]:
        """차트 데이터 생성"""
        try:
            # self.logger.debug(f"차트 데이터 생성 시작 - 섹션: {section_name}")
            # self.logger.debug(f"입력 내용: {content[:200]}...")

            # 차트 프롬프트 정의
            chart_prompt = f"""
다음은 사업계획서의 {section_name} 섹션 내용입니다. 이 내용을 가장 효과적으로 시각화할 수 있는 차트 데이터를 생성해주세요:

{content}

다음과 같은 형식의 JSON으로 응답해주세요:
{{
    "chart": {{
        "type": "bar",  # bar, line, pie, radar 중 하나
        "data": {{
            "labels": ["2023", "2024", "2025"],  # X축 레이블
            "datasets": [
                {{
                    "label": "시장 규모",
                    "data": [100, 150, 200]  # 실제 데이터 값
                }}
            ]
        }},
        "options": {{
            "title": "시장 규모 추이",
            "yAxisLabel": "금액 (억원)"
        }}
    }}
}}

주의사항:
1. 반드시 숫자 데이터를 포함할 것
2. 레이블과 데이터 개수가 일치해야 함
3. 차트 유형은 데이터 특성에 맞게 선택
4. radar 차트를 선택한 경우:
   - 모든 데이터는 반드시 0~1 사이의 값으로 정규화해야 함
   - 예시: 90% -> 0.9, 85% -> 0.85, 75% -> 0.75            
            """

            # OpenAI API 호출 전 로깅
            # self.logger.debug("OpenAI API 호출 시작")
            # self.logger.debug(f"프롬프트: {chart_prompt}")

            response = await self.call_ai_api(
                prompt=chart_prompt,
                require_json=True,
                json_schema={
                    "type": "object",
                    "required": ["chart"],
                    "properties": {
                        "chart": {
                            "type": "object",
                            "required": ["type", "data", "options"],
                            "properties": {
                                "type": {"type": "string", "enum": ["line", "bar", "pie", "radar"]},
                                "data": {
                                    "type": "object",
                                    "required": ["labels", "datasets"],
                                    "properties": {
                                        "labels": {"type": "array", "items": {"type": "string"}},
                                        "datasets": {
                                            "type": "array",
                                            "items": {
                                                "type": "object",
                                                "required": ["label", "data"],
                                                "properties": {
                                                    "label": {"type": "string"},
                                                    "data": {"type": "array", "items": {"type": "number"}}
                                                }
                                            }
                                        }
                                    }
                                },
                                "options": {
                                    "type": "object",
                                    "required": ["title"],
                                    "properties": {
                                        "title": {"type": "string"},
                                        "yAxisLabel": {"type": "string"}
                                    }
                                }
                            }
                        }
                    }
                }
            )

            # 응답 로깅
            # self.logger.debug(f"OpenAI 응답: {json.dumps(response, ensure_ascii=False, indent=2)}")

            if response and 'chart' in response:
                # self.logger.debug("차트 데이터 생성 성공")
                # self.logger.debug(f"차트 데이터: {json.dumps(response['chart'], ensure_ascii=False, indent=2)}")
                return response
            else:
                self.logger.warning("차트 데이터 없음")
                return None

        except Exception as e:
            self.logger.error(f"차트 데이터 생성 실패: {str(e)}")
            traceback.print_exc()
            return None

    def _build_schema_prompt(self, table_schema: Dict, content: str) -> str:
        """테이블 스키마 기반 프롬프트 생성"""
        try:
            title = table_schema.get('title', '테이블')
            description = table_schema.get('description', '')
            headers = table_schema.get('headers', [])
            validation_rules = table_schema.get('validation_rules', {})
            
            # 헤더 정의 부분
            header_definitions = []
            for header in headers:
                header_def = f"- {header['name']}"
                if header.get('type'):
                    header_def += f" ({header['type']})"
                if header.get('required', True):
                    header_def += " [필수]"
                if header.get('description'):
                    header_def += f": {header['description']}"
                if header.get('options'):
                    header_def += f" (선택지: {', '.join(header['options'])})"
                if header.get('unit'):
                    header_def += f" (단위: {header['unit']})"
                header_definitions.append(header_def)
            
            # 검증 규칙 부분
            validation_text = []
            if validation_rules.get('min_rows'):
                validation_text.append(f"- 최소 {validation_rules['min_rows']}개 행 필요")
            if validation_rules.get('max_rows'):
                validation_text.append(f"- 최대 {validation_rules['max_rows']}개 행까지")
            if validation_rules.get('required_categories'):
                validation_text.append(f"- 필수 포함 카테고리: {', '.join(validation_rules['required_categories'])}")
            
            prompt = f"""
다음 내용을 바탕으로 "{title}" 표를 생성해주세요.

{description}

내용:
{content}

표 구조 정의:
{chr(10).join(header_definitions)}

검증 규칙:
{chr(10).join(validation_text) if validation_text else '- 특별한 검증 규칙 없음'}

반드시 다음 JSON 형식으로 응답해주세요:
{{
    "headers": ["{headers[0]['name'] if headers else '항목'}", ...],
    "rows": [
        [데이터1, 데이터2, ...],
        [데이터3, 데이터4, ...]
    ]
}}

주의사항:
1. 모든 행은 헤더와 동일한 수의 열을 가져야 합니다
2. 필수 필드는 반드시 값이 있어야 합니다
3. 카테고리 필드는 지정된 선택지에서만 선택해주세요
4. 빈 셀은 "-"로 표시해주세요
5. 숫자 필드는 숫자만 입력해주세요 (단위 제외)
6. 백분율 필드는 0-100 사이의 숫자로 입력해주세요
"""
            return prompt
            
        except Exception as e:
            self.logger.error(f"스키마 프롬프트 생성 실패: {str(e)}")
            return f"다음 내용을 표 형식으로 정리해주세요:\n{content}"

    def _build_json_schema_from_table_schema(self, table_schema: Dict) -> Dict:
        """테이블 스키마에서 JSON 스키마 생성"""
        try:
            headers = table_schema.get('headers', [])
            
            json_schema = {
                "type": "object",
                "required": ["headers", "rows"],
                "properties": {
                    "headers": {
                        "type": "array",
                        "items": {"type": "string"}
                    },
                    "rows": {
                        "type": "array",
                        "items": {
                            "type": "array",
                            "items": {"type": "string"}
                        }
                    }
                },
                "additionalProperties": False
            }
            
            # 최소/최대 행 수 제한 추가
            validation_rules = table_schema.get('validation_rules', {})
            if validation_rules.get('min_rows'):
                json_schema["properties"]["rows"]["minItems"] = validation_rules['min_rows']
            if validation_rules.get('max_rows'):
                json_schema["properties"]["rows"]["maxItems"] = validation_rules['max_rows']
            
            return json_schema
            
        except Exception as e:
            self.logger.error(f"JSON 스키마 생성 실패: {str(e)}")
            # 기본 스키마 반환
            return {
                "type": "object",
                "required": ["headers", "rows"],
                "properties": {
                    "headers": {"type": "array", "items": {"type": "string"}},
                    "rows": {"type": "array", "items": {"type": "array", "items": {"type": "string"}}}
                }
            }

    def _validate_table_against_schema(self, table_data: Dict, table_schema: Dict) -> Dict:
        """테이블 데이터를 스키마에 대해 검증"""
        validation_result = {
            'is_valid': True,
            'errors': [],
            'warnings': []
        }
        
        try:
            if not table_data or 'headers' not in table_data or 'rows' not in table_data:
                validation_result['is_valid'] = False
                validation_result['errors'].append("테이블 데이터 형식이 올바르지 않습니다")
                return validation_result
            
            headers = table_data['headers']
            rows = table_data['rows']
            schema_headers = table_schema.get('headers', [])
            validation_rules = table_schema.get('validation_rules', {})
            
            # 헤더 수 검증
            expected_header_count = len(schema_headers)
            if len(headers) != expected_header_count:
                validation_result['warnings'].append(f"헤더 수가 예상과 다릅니다 (예상: {expected_header_count}, 실제: {len(headers)})")
            
            # 행 수 검증
            if validation_rules.get('min_rows') and len(rows) < validation_rules['min_rows']:
                validation_result['errors'].append(f"최소 {validation_rules['min_rows']}개 행이 필요합니다")
                validation_result['is_valid'] = False
                
            if validation_rules.get('max_rows') and len(rows) > validation_rules['max_rows']:
                validation_result['errors'].append(f"최대 {validation_rules['max_rows']}개 행까지 허용됩니다")
                validation_result['is_valid'] = False
            
            # 각 행의 열 수 검증
            for i, row in enumerate(rows):
                if len(row) != len(headers):
                    validation_result['errors'].append(f"{i+1}번째 행의 열 수가 헤더와 일치하지 않습니다")
                    validation_result['is_valid'] = False
            
            # 필수 카테고리 검증
            if validation_rules.get('required_categories'):
                for category in validation_rules['required_categories']:
                    found = False
                    for row in rows:
                        if category in row:
                            found = True
                            break
                    if not found:
                        validation_result['warnings'].append(f"필수 카테고리 '{category}'가 포함되지 않았습니다")
            
        except Exception as e:
            validation_result['is_valid'] = False
            validation_result['errors'].append(f"검증 중 오류 발생: {str(e)}")
        
        return validation_result

    async def get_table_data_with_schema(self, section_name: str, content: str, table_schema: Dict = None) -> Optional[Dict]:
        """스키마 기반 표 데이터 생성"""
        try:
            if table_schema:
                # 스키마 기반 프롬프트 생성
                schema_prompt = self._build_schema_prompt(table_schema, content)
                
                # JSON 스키마 동적 생성
                json_schema = self._build_json_schema_from_table_schema(table_schema)
                
                self.logger.debug(f"스키마 기반 테이블 생성 시작: {table_schema.get('title', '테이블')}")
                
                response = await self.call_ai_api(
                    prompt=schema_prompt,
                    require_json=True,
                    json_schema=json_schema
                )
                
                if response:
                    # 스키마 검증
                    validation_result = self._validate_table_against_schema(response, table_schema)
                    
                    if not validation_result['is_valid']:
                        self.logger.warning(f"테이블 스키마 검증 실패: {validation_result['errors']}")
                        # 재시도 또는 기본 방식으로 폴백
                        self.logger.info("기본 테이블 생성 방식으로 폴백합니다")
                        return await self.get_table_data(section_name, content)
                    
                    if validation_result['warnings']:
                        for warning in validation_result['warnings']:
                            self.logger.warning(f"테이블 검증 경고: {warning}")
                    
                    self.logger.debug(f"스키마 기반 테이블 생성 완료: {len(response.get('headers', []))}개 열, {len(response.get('rows', []))}개 행")
                    return response
                else:
                    self.logger.warning("스키마 기반 테이블 생성 실패, 기본 방식으로 폴백")
                    return await self.get_table_data(section_name, content)
            else:
                # 기존 방식 사용
                return await self.get_table_data(section_name, content)
                
        except Exception as e:
            self.logger.error(f"스키마 기반 테이블 생성 오류: {str(e)}")
            traceback.print_exc()
            # 기본 방식으로 폴백
            return await self.get_table_data(section_name, content)

    async def get_table_data(self, section_name: str, content: str) -> Optional[Dict]:
        """표 데이터 생성"""
        try:
            # OpenAI로 구조화된 데이터 얻기
            response = await self.call_ai_api(
                prompt=f"""
다음 {section_name}의 내용을 표 형식으로 변환해서 JSON으로 응답해주세요.
반드시 모든 행은 헤더와 동일한 수의 열을 가져야 합니다.

{content}

표 작성 규칙:
1. 모든 행은 반드시 첫번째 행(헤더)와 동일한 수의 열을 가져야 합니다
2. 한 셀에 여러 줄의 내용을 넣지 말고, 새로운 행으로 분리하세요
3. 내용이 긴 경우 핵심만 간단히 작성하세요
4. 빈 셀이 있으면 안 됩니다. 내용이 없는 경우 "-" 를 입력하세요
5. 기능점수 및 개발원가산정 경우 SW사업 대가산정 가이드에 따라 표 형식을 작성하세요
6. 종합 산출내역서 경우 종합 산출내역서 경우 표 형식을 작성하세요
7. 총비용, 총 합계 비용은 반드시 표에 제시한 모든 비용이 합계와 일치해야 합니다.
8. 개발 대상 범위 경우 개발 대상 범위 표 형식을 작성하세요.
9. 기능점수 및 개발원가산정 경우 기능점수 및 개발원가산정 표 형식을 작성하세요.
10. 사업추진 체계 경우 사업추진 체계 표 형식을 작성하세요.

반드시 다음 형식의 JSON으로 응답해주세요. 단, 표의 헤더와 데이터는 요청에 따라 적절히 변경될 수 있습니다:
{{
    "headers": ["열1", "열2", "열3", "열4"],
    "rows": [
        ["데이터1", "데이터2", "데이터3", "데이터4"],
        ["데이터5", "데이터6", "데이터7", "데이터8"]
    ]
}}

주의사항:
1. 복잡한 중첩 구조가 아닌 단순한 테이블 형식으로 응답해야 합니다.
2. 모든 행은 반드시 헤더와 동일한 수의 열을 가져야 합니다.
3. 다른 키는 포함하지 마세요. 오직 'headers'와 'rows'만 사용하세요.
4. 내용이 없는 셀은 '-'로 표시하세요.

섹션별 표 형식 예시:

1. 사업추진 체계 표 형식:
{{
    "headers": ["조직", "업무", "투입인력", "비고"],
    "rows": [
        ["주관기관", "담당업무", "담당자", "비상연락처 전화 또는 이메일"],
        ["수행기관", "담당업무", "담당자", "비상연락처 전화 또는 이메일"],
        ["수행기관", "담당업무", "담당자", "비상연락처 전화 또는 이메일"],
    ]
}}

2. 개발 대상 범위 표 형식:
{{
    "headers": ["개발범위", "세부개발범위", "개발범위 설명"],
    "rows": [
        ["개발범위", "세부개발범위", "개발범위 설명"],
        ["개발범위", "세부개발범위", "개발범위 설명"],
        ["개발범위", "세부개발범위", "개발범위 설명"],
    ]
}}

3. 기능 목록 표 형식:
{{
    "headers": ["개발범위", "단위프로세스", "기능명", "기능 설명"],
    "rows": [
        ["개발범위", "단위프로세스", "기능명", "기능 설명"],
        ["개발범위", "단위프로세스", "기능명", "기능 설명"],
        ["개발범위", "단위프로세스", "기능명", "기능 설명"],
    ]
}}

4. 종합 산출내역서 표 형식:
{{
    "headers": ["산출항목", "분류", "이름", "단가", "투입기간", "합계"],
    "rows": [
        ["산출항목", "분류", "이름", "단가", "투입기간", "합계"],
        ["산출항목", "분류", "이름", "단가", "투입기간", "합계"],
        ["산출항목", "분류", "이름", "단가", "투입기간", "합계"],
        ["총비용", "", "", "", "", "총 합계 비용"],
    ]
}}

5. 기능점수 및 개발원가산정 표 형식:
{{
    "headers": ["기능명", "기능점수", "금액", "산출내역"],
    "rows": [
        ["기능명", "기능점수", "금액", "산출내역"],
        ["기능명", "기능점수", "금액", "산출내역"],
        ["기능명", "기능점수", "금액", "산출내역"],
    ]
}}

6. 투입 인력 표 형식:
{{
    "headers": ["업무구분", "이름", "직급", "기술등급", "최종학력", "자격증", "담당업무"],
    "rows": [
        ["업무구분", "투입인력 이름", "투입인력 직급", "투입인력 기술등급", "최종학력", "자격증", "담당업무"],
        ["업무구분", "투입인력 이름", "투입인력 직급", "투입인력 기술등급", "최종학력", "자격증", "담당업무"],
        ["업무구분", "투입인력 이름", "투입인력 직급", "투입인력 기술등급", "최종학력", "자격증", "담당업무"],
    ]
}}

표 작성 규칙:
1. 모든 행은 반드시 첫번째 행(헤더)와 동일한 수의 열을 가져야 합니다
2. 한 셀에 여러 줄의 내용을 넣지 말고, 새로운 행으로 분리하세요
3. 내용이 긴 경우 핵심만 간단히 작성하세요
4. 빈 셀이 있으면 안 됩니다. 내용이 없는 경우 "-" 를 입력하세요
5. 기능점수 및 개발원가산정 경우 SW사업 대가산정 가이드에 따라 표 형식을 작성하세요
6. 종합 산출내역서 경우 종합 산출내역서 경우 표 형식을 작성하세요
7. 총비용, 총 합계 비용은 반드시 표에 제시한 모든 비용이 합계와 일치해야 합니다.
8. 개발 대상 범위 경우 개발 대상 범위 표 형식을 작성하세요.
9. 기능점수 및 개발원가산정 경우 기능점수 및 개발원가산정 표 형식을 작성하세요.
10. 사업추진 체계 경우 사업추진 체계 표 형식을 작성하세요.
""",
                require_json=True,
                json_schema = {
                "type": "object",
                "required": ["headers", "rows"],
                "properties": {
                    "headers": {
                    "type": "array",
                    "items": {"type": "string"}
                    },
                    "rows": {
                    "type": "array",
                    "items": {
                        "type": "array",
                        "items": {"type": "string"}
                    }
                    }
                },
                "additionalProperties": False  # 추가 속성 허용하지 않음
                }
            )

            if not response:
                self.logger.warning("AI 응답 없음")
                return None

            # 중첩 구조 검사 및 디버그 출력 추가
            is_nested = False
            nested_keys = []
            
            # 응답이 중첩 구조인지 확인
            if isinstance(response, dict):
                for key, value in response.items():
                    if isinstance(value, dict):
                        is_nested = True
                        nested_keys.append(key)
                        self.logger.debug(f"중첩 구조 감지: 최상위 키 '{key}'가 중첩된 딕셔너리를 포함")
                        # 중첩 구조의 첫 번째 레벨 출력
                        self.logger.debug(f"'{key}' 내부 키: {list(value.keys())}")
            
            if is_nested:
                self.logger.debug("=== 중첩 구조 감지됨 ===")
                self.logger.debug(f"중첩된 최상위 키: {nested_keys}")
                self.logger.debug("응답 구조:")
                
                # 중첩 구조를 최대 2단계까지 출력
                for key in nested_keys:
                    if key in response:
                        self.logger.debug(f"  {key}:")
                        if isinstance(response[key], dict):
                            for subkey, subvalue in response[key].items():
                                if isinstance(subvalue, dict):
                                    self.logger.debug(f"    {subkey}: {{{len(subvalue)} 개의 키}}")
                                elif isinstance(subvalue, list):
                                    self.logger.debug(f"    {subkey}: [{len(subvalue)} 개의 항목]")
                                else:
                                    self.logger.debug(f"    {subkey}: {type(subvalue).__name__}")
                
                self.logger.debug("=== 중첩 구조 분석 완료 ===")
                self.logger.debug("예상 형식: {'headers': [...], 'rows': [...]}")
                self.logger.debug("실제 응답에는 'headers'와 'rows' 키가 없거나 중첩되어 있습니다.")
                
                # 중첩 구조를 테이블 형식으로 변환 시도
                self.logger.debug("중첩 구조를 테이블 형식으로 변환 시도...")
                converted_response = self._convert_nested_to_table(response)
                if converted_response:
                    self.logger.debug("중첩 구조 변환 성공!")
                    response = converted_response
                else:
                    self.logger.error("중첩 구조를 테이블 형식으로 변환할 수 없습니다.")
                    return None

            # 응답 검증
            if "headers" in response and "rows" in response:
                headers = response["headers"]
                rows = response["rows"]
                
                # 모든 행의 길이가 헤더 길이와 같은지 확인
                header_len = len(headers)
                for i, row in enumerate(rows):
                    if len(row) != header_len:
                        self.logger.error(f"행 길이 불일치 - 헤더: {header_len}, 행 {i}: {len(row)}")
                        return None
                
                return response
            else:
                self.logger.error("응답에 'headers'와 'rows' 키가 없습니다.")
                if is_nested:
                    self.logger.debug(f"응답 키: {list(response.keys())}")
                return None

        except Exception as e:
            self.logger.error(f"표 데이터 생성 실패: {str(e)}")
            traceback.print_exc()
            return None

    def _convert_nested_to_table(self, nested_data: Dict) -> Optional[Dict]:
        """중첩 구조를 테이블 형식으로 변환 - 일반적인 접근 방식"""
        try:
            self.logger.debug("중첩 구조 일반 변환 시작")
            
            # 결과 테이블 초기화
            headers = ["카테고리", "항목", "값"]
            rows = []
            
            # 중첩 구조를 재귀적으로 평면화하는 함수
            def flatten_nested_structure(data, path=None):
                if path is None:
                    path = []
                    
                if isinstance(data, dict):
                    # 딕셔너리인 경우 각 키-값 쌍을 처리
                    for key, value in data.items():
                        current_path = path + [key]
                        
                        if isinstance(value, (dict, list)):
                            # 중첩된 구조는 재귀적으로 처리
                            flatten_nested_structure(value, current_path)
                        else:
                            # 기본 값은 행으로 추가
                            category = path[0] if path else "기본"
                            item = ".".join(current_path[1:]) if len(current_path) > 1 else key
                            rows.append([category, item, str(value)])
                
                elif isinstance(data, list):
                    # 리스트인 경우 각 항목을 처리
                    for i, item in enumerate(data):
                        if isinstance(item, (dict, list)):
                            # 중첩된 구조는 재귀적으로 처리
                            current_path = path + [f"항목{i+1}"]
                            flatten_nested_structure(item, current_path)
                        elif isinstance(item, list):
                            # 리스트의 리스트인 경우 (테이블 행과 유사)
                            category = path[0] if path else "기본"
                            item_name = f"{path[-1] if path else '항목'} {i+1}"
                            
                            # 리스트 항목을 문자열로 변환하여 행으로 추가
                            if len(item) >= 2:
                                rows.append([category, str(item[0]), str(item[1])])
                            elif len(item) == 1:
                                rows.append([category, item_name, str(item[0])])
                            else:
                                rows.append([category, item_name, "-"])
                        else:
                            # 기본 값은 행으로 추가
                            category = path[0] if path else "기본"
                            item_name = f"{path[-1] if path else '항목'} {i+1}"
                            rows.append([category, item_name, str(item)])
            
            # 중첩 구조 평면화 실행
            flatten_nested_structure(nested_data)
            
            # 결과가 없으면 더 단순한 방식으로 시도
            if not rows:
                self.logger.debug("첫 번째 변환 방식으로 행이 생성되지 않음, 대체 방식 시도")
                
                # 최상위 키를 카테고리로 사용하는 단순 변환
                for category, value in nested_data.items():
                    if isinstance(value, dict):
                        for item, subvalue in value.items():
                            if isinstance(subvalue, (str, int, float, bool)):
                                rows.append([category, item, str(subvalue)])
                            elif isinstance(subvalue, list):
                                # 리스트의 각 항목을 별도 행으로 추가
                                for i, list_item in enumerate(subvalue):
                                    if isinstance(list_item, list):
                                        # 리스트의 리스트는 첫 두 항목을 사용
                                        if len(list_item) >= 2:
                                            rows.append([category, str(list_item[0]), str(list_item[1])])
                                        elif len(list_item) == 1:
                                            rows.append([category, f"{item} {i+1}", str(list_item[0])])
                                    else:
                                        rows.append([category, f"{item} {i+1}", str(list_item)])
                            elif isinstance(subvalue, dict):
                                # 중첩된 딕셔너리의 각 항목을 별도 행으로 추가
                                for subitem, subsubvalue in subvalue.items():
                                    rows.append([category, f"{item}.{subitem}", str(subsubvalue)])
                    elif isinstance(value, list):
                        # 최상위 키 아래 리스트가 있는 경우
                        for i, list_item in enumerate(value):
                            if isinstance(list_item, dict):
                                # 딕셔너리의 각 항목을 별도 행으로 추가
                                for key, val in list_item.items():
                                    rows.append([category, key, str(val)])
                            elif isinstance(list_item, list):
                                # 리스트의 리스트는 첫 두 항목을 사용
                                if len(list_item) >= 2:
                                    rows.append([category, str(list_item[0]), str(list_item[1])])
                                elif len(list_item) == 1:
                                    rows.append([category, f"항목 {i+1}", str(list_item[0])])
                            else:
                                rows.append([category, f"항목 {i+1}", str(list_item)])
                    else:
                        # 기본 값은 직접 행으로 추가
                        rows.append([category, "값", str(value)])
            
            # 행이 생성되었는지 확인
            if not rows:
                self.logger.error("중첩 구조에서 테이블 행을 생성할 수 없습니다")
                return None
                
            # 결과 로깅
            self.logger.debug(f"변환된 테이블: {len(rows)}개 행 생성됨")
            self.logger.debug(f"헤더: {headers}")
            if rows:
                self.logger.debug(f"첫 번째 행 예시: {rows[0]}")
                
            return {"headers": headers, "rows": rows}
            
        except Exception as e:
            self.logger.error(f"중첩 구조 변환 실패: {str(e)}")
            traceback.print_exc()
            return None


    async def process_section_template(self, template, input_data, cache_paths):
        """섹션 템플릿 처리"""
        try:
            # 템플릿 설정 확인 및 기본값 초기화
            prompt = ""
            requires_web = False
            requires_rag = False
            requires_table = False
            requires_hide = False
            search_keywords = []
            requires_competitor_analysis = False
            requires_diagram = False
            requires_chart = False
            requires_flow = False
            section_type = "default"  # 기본값으로 초기화

            if isinstance(template, SectionConfig):
                prompt = template.prompt if template.prompt is not None else ""
                requires_web = template.requires_web
                requires_rag = template.requires_rag
                requires_table = template.requires_table
                requires_hide = template.requires_hide
                requires_competitor_analysis = template.requires_competitor_analysis
                requires_diagram = template.requires_diagram
                requires_chart = template.requires_chart
                requires_flow = template.requires_flow
                search_keywords = template.search_keywords or []
                section_type = template.section_type
            elif isinstance(template, dict):
                prompt = template.get("prompt", "")
                requires_web = template.get("requires_web", False)
                requires_rag = template.get("requires_rag", False)
                requires_table = template.get("requires_table", False)
                requires_hide = template.get("requires_hide", False)
                search_keywords = template.get('search_keywords', [])
                requires_competitor_analysis = template.get("requires_competitor_analysis", False)
                requires_diagram = template.get("requires_diagram", False)
                requires_chart = template.get("requires_chart", False)
                requires_flow = template.get("requires_flow", False)
                section_type = template.get("section_type", "default")
            else:
                prompt = str(template) if template is not None else ""
            
            # prompt가 여전히 문자열이 아닌 경우 강제 변환
            if not isinstance(prompt, str):
                prompt = str(prompt) if prompt is not None else ""

            # 타이틀 섹션 특별 처리
            if section_type == "title" and 'business_info' in input_data:
                # business_info에서 서비스/제품명 직접 사용
                business_info = input_data.get('business_info', {})
                service_name = business_info.get('product_name', '')
                
                if service_name:
                    # 서비스/제품명이 있으면 그대로 사용
                    self.logger.debug(f"사업명 섹션에 서비스/제품명 직접 사용: {service_name}")
                    return {
                        "content": service_name,
                        "visualization_paths": []
                    }
                else:
                    # 서비스/제품명이 없으면 기본값 제공
                    self.logger.warning("서비스/제품명을 찾을 수 없음, 기본값 사용")
                    return {
                        "content": "사업명 정보 없음",
                        "visualization_paths": []
                    }
                    
            # 1. 섹션 내용 생성
            content = await self.get_section_content(prompt, input_data)
            
            # content가 리스트인 경우 문자열로 변환
            if isinstance(content, list):
                content = '\n'.join([str(item) for item in content])            
                
            result = {
                "content": content,
                "visualization_paths": []
            }

            # SWOT 분석 섹션인 경우 특별 처리
            if section_type == "swot" or (
                isinstance(input_data.get('current_subsection'), str) and 
                "SWOT" in input_data['current_subsection']
            ):
                try:
                    # SWOT 분석 데이터 추출 및 4분면 매트릭스 생성
                    swot_data = await self._extract_swot_data(content)
                    if swot_data:
                        # DOT 소스 생성
                        dot_source = self._create_swot_matrix_dot(swot_data)
                        diagram_data = {
                            "type": "digraph",
                            "dot_source": dot_source,
                            "options": {
                                "title": "SWOT 분석 매트릭스",
                                "description": "강점, 약점, 기회, 위협 요인 분석"
                            }
                        }
                        result["diagram_data"] = diagram_data
                except Exception as e:
                    self.logger.error(f"SWOT 매트릭스 생성 실패: {str(e)}")

            # 목표 시장 분석이 필요한 섹션인 경우
            if section_type == "market" or (
                isinstance(input_data.get('current_subsection'), str) and 
                any(keyword in input_data['current_subsection'] for keyword in ["시장", "고객", "세그먼트"])
            ):
                try:
                    # 캐시된 목표 시장 정보 로드
                    market_file = os.path.join(cache_paths['cache_dir'], '000_target_market.json')
                    market_data = None
                    
                    if os.path.exists(market_file):
                        with open(market_file, 'r', encoding='utf-8') as f:
                            market_data = json.load(f)
                    
                    if not market_data and 'business_info' in input_data:
                        # 목표 시장 분석 수행
                        market_data = await self.analyze_target_market(input_data['business_info'])
                        # 분석 결과 캐시에 저장
                        with open(market_file, 'w', encoding='utf-8') as f:
                            json.dump(market_data, f, ensure_ascii=False, indent=2)
                    
                    if market_data:
                        # 환경 변수에서 AI 설정 가져오기
                        config = load_config()
                        provider = config.get('USE_LLM', 'openai')
                        
                        # 목표 시장 분석 프롬프트 구성 (키 이름 동적 처리)
                        market_prompt = "# 목표 시장 분석 정보\n\n"
                        
                        # 시장 특성 (영문/한글 키 모두 지원)
                        market_characteristic = market_data.get('market_characteristics', '정보 없음')
                        market_prompt += f"## 시장 특성\n{market_characteristic}\n\n"
                        
                        # 목표 고객 세그먼트 (영문/한글 키 모두 지원)
                        segments = market_data.get('customer_segments', [])
                        if segments:
                            market_prompt += "## 목표 고객 세그먼트\n"
                            
                            # Ollama 모델은 간소화된 형식 사용
                            if provider.lower() == 'ollama':
                                for segment in segments:
                                    segment_name = segment.get('segment_name', '')
                                    description = segment.get('characteristics', '')
                                    market_prompt += f"- {segment_name}: {description}\n"
                            else:
                                market_prompt += f"{json.dumps(segments, ensure_ascii=False, indent=2)}\n\n"
                        
                        # 진출 지역 (영문/한글 키 모두 지원)
                        regions = market_data.get('region', [])
                        if isinstance(regions, list) and regions:
                            market_prompt += f"## 진출 지역\n{', '.join(regions)}\n\n"
                        elif isinstance(regions, str):
                            market_prompt += f"## 진출 지역\n{regions}\n\n"
                        
                        # 진입 장벽 (영문/한글 키 모두 지원)
                        barriers = market_data.get('entry_barriers', [])
                        if barriers:
                            market_prompt += "## 진입 장벽\n"
                            
                            # Ollama 모델은 간소화된 형식 사용
                            if provider.lower() == 'ollama' and isinstance(barriers, list):
                                for barrier in barriers:
                                    market_prompt += f"- {barrier}\n"
                            else:
                                market_prompt += f"{json.dumps(barriers, ensure_ascii=False, indent=2)}\n\n"
                        
                        # 성장률 정보 (있는 경우에만)
                        growth_rate = market_data.get('growth_rate', '')
                        if growth_rate:
                            market_prompt += f"## 시장 성장률\n{growth_rate}\n\n"
                        
                        # 시장 규모 정보 (있는 경우에만)
                        market_size = market_data.get('market_size', '')
                        if market_size:
                            market_prompt += f"## 시장 규모\n{market_size}\n\n"
                        
                        market_prompt += f"위 시장 분석 정보를 바탕으로 다음 내용을 작성해주세요:\n\n{content}"
                        
                        # 기존 content에 시장 분석 정보 추가
                        content = await self.get_section_content(market_prompt, input_data)
                        result["content"] = content

                except Exception as e:
                    self.logger.error(f"목표 시장 정보 처리 중 오류: {str(e)}")

            # 2. 경쟁사 분석이 필요한 섹션인 경우
            if requires_competitor_analysis and 'business_info' in input_data:
                try:
                    # 캐시된 경쟁사 정보 로드
                    competitors_file = os.path.join(cache_paths['cache_dir'], '000_competitors.json')
                    if os.path.exists(competitors_file):
                        with open(competitors_file, 'r', encoding='utf-8') as f:
                            competitors_data = json.load(f)
                            
                        if competitors_data and 'competitors' in competitors_data:
                            # 환경 변수에서 AI 설정 가져오기
                            config = load_config()
                            provider = config.get('USE_LLM', 'openai')
                            
                            # 경쟁사 목록 문자열 생성
                            competitor_names = [comp['company'] for comp in competitors_data['competitors']]
                            competitor_list = ', '.join(competitor_names)

                            # 경쟁사 정보를 구조화된 형식으로 변환
                            competitors_info = []
                            # 검색된 경쟁사 목록 초기화
                            if 'searched_competitors' not in competitors_data:
                                competitors_data['searched_competitors'] = []
                            
                            for comp in competitors_data['competitors']:
                                competitor = {
                                    "company": comp['company'],
                                    "main_products": comp.get('main_products', comp.get('description', '정보 없음')),
                                    "strengths": comp.get('strengths', ["정보 없음"]),
                                    "weaknesses": comp.get('weaknesses', ["정보 없음"]),
                                    "market_share": comp.get('market_share', "정보 없음"),
                                    "revenue": comp.get('revenue', "정보 없음"),
                                    "source": comp.get('url', comp.get('source', '정보 없음'))
                                }
                                competitors_info.append(competitor)
                                # 프롬프트 치환용 리스트에 추가
                                competitors_data['searched_competitors'].append(f"{comp['company']}: {competitor['main_products']}")

                            # Ollama 모델 사용 시 간소화된 프롬프트
                            if provider.lower() == 'ollama':
                                competitors_prompt = f"""
# 경쟁사 분석

분석 대상 경쟁사: {competitor_list}

## 경쟁사 정보
"""
                                # 간소화된 형식으로 경쟁사 정보 추가
                                for comp in competitors_info:
                                    competitors_prompt += f"\n### {comp['company']}\n"
                                    competitors_prompt += f"- 주요 제품/서비스: {comp['main_products']}\n"
                                    
                                    # 강점 목록 (최대 3개만)
                                    competitors_prompt += "- 강점:\n"
                                    for i, strength in enumerate(comp['strengths'][:3]):
                                        competitors_prompt += f"  * {strength}\n"
                                    
                                    # 약점 목록 (최대 3개만)
                                    competitors_prompt += "- 약점:\n"
                                    for i, weakness in enumerate(comp['weaknesses'][:3]):
                                        competitors_prompt += f"  * {weakness}\n"
                                    
                                    # 시장 점유율과 매출 정보가 있는 경우에만 추가
                                    if comp['market_share'] != "정보 없음":
                                        competitors_prompt += f"- 시장 점유율: {comp['market_share']}\n"
                                    if comp['revenue'] != "정보 없음":
                                        competitors_prompt += f"- 매출: {comp['revenue']}\n"
                                
                                competitors_prompt += f"""
## 분석 요구사항
1. 위 경쟁사들의 특징을 비교 분석해주세요.
2. 실제 기업명만 사용하세요.
3. 각 기업의 강점과 약점을 중심으로 분석하세요.

{content}
"""
                            # 일반 모델 사용 시 기존 상세 프롬프트
                            else:
                                competitors_prompt = f"""
# 경쟁사 분석 지침

다음은 실제 시장 조사를 통해 확인된 경쟁사들입니다. 
분석 대상 경쟁사: {competitor_list}

## 경쟁사 상세 정보
{json.dumps(competitors_info, ensure_ascii=False, indent=2)}

## 분석 요구사항

1. 기업명 사용 규칙:
- 위에 명시된 실제 기업명만 사용할 것
- "경쟁사A", "B사" 등 가상의 기업명 사용 시 분석이 무효화됨
- 각 기업의 공식 명칭을 정확히 사용할 것

2. 분석 항목:
□ 개별 기업 분석
- 기업별 핵심 제품/서비스 특징
- 기술력 및 시장 영향력
- 확인된 강점과 약점

□ 경쟁사 간 비교
- 제품/서비스 차별점
- 기술적 특징 비교
- 시장 포지셔닝

□ 시사점 도출
- 각 기업의 전략적 방향
- 시장 내 경쟁 구도
- 우리 기업의 차별화 포인트

3. 정보 표기 규칙:
- 확인된 정보만 포함
- 불확실한 정보는 "정보 없음"으로 표시
- 추측성 내용 배제
- 참고URL은 반드시 출처로 별도의 행에 포함해서 표기할 것
- 개별 기업의 내용을 각각의 행으로 구조화하여 표기할 것

4. 분석 형식:
□ 시장 현황
- 경쟁사 구성: {competitor_list}
- 시장 특성 및 주요 동향

□ 기업별 상세 분석
(각 기업별로 위 분석 항목을 상세히 기술)

□ 종합 분석 및 전략적 시사점
(전체 경쟁 구도와 우리 기업의 차별화 방향)

위 지침에 따라 경쟁사 분석을 작성해주세요.
특히 실제 기업명을 정확히 사용하는 것이 매우 중요합니다.
"""

                            # 기존 content에 경쟁사 분석 추가
                            content = await self.get_section_content(competitors_prompt + "\n\n" + content, input_data)
                            result["content"] = content

                except Exception as e:
                    self.logger.error(f"경쟁사 정보 처리 중 오류: {str(e)}")

            # 3. 차트 생성 시도
            if requires_chart:
                try:
                    chart_data = await self.get_chart_data(input_data['current_section'], content)
                    if chart_data and 'chart' in chart_data:
                        # 차트 데이터 구조 검증
                        if not isinstance(chart_data['chart'], dict):
                            self.logger.error("차트 데이터가 딕셔너리 형식이 아님")
                            raise ValueError("차트 데이터 형식이 올바르지 않습니다")

                        # 필수 필드 검증
                        required_fields = ['type', 'data', 'options']
                        for field in required_fields:
                            if field not in chart_data['chart']:
                                self.logger.error(f"차트 데이터에 필수 필드 '{field}' 누락")
                                raise ValueError(f"차트 데이터에 필수 필드 '{field}'가 없습니다")

                        # 차트 타입 검증
                        valid_chart_types = ['bar', 'line', 'pie', 'radar']
                        chart_type = chart_data['chart']['type']
                        if chart_type not in valid_chart_types:
                            self.logger.error(f"지원하지 않는 차트 타입: {chart_type}")
                            raise ValueError(f"지원하지 않는 차트 타입입니다: {chart_type}")

                        # 데이터셋 검증
                        data = chart_data['chart']['data']
                        if not data.get('labels') or not data.get('datasets'):
                            self.logger.error("차트 데이터에 labels 또는 datasets 누락")
                            raise ValueError("차트 데이터에 labels 또는 datasets가 없습니다")

                        # 차트 타입별 특수 검증
                        if chart_type == 'radar':
                            # 레이더 차트의 경우 모든 값이 0~1 사이여야 함
                            for dataset in data['datasets']:
                                if not all(0 <= value <= 1 for value in dataset['data']):
                                    self.logger.error("레이더 차트 데이터가 0~1 범위를 벗어남")
                                    raise ValueError("레이더 차트의 모든 값은 0과 1 사이여야 합니다")
                        elif chart_type == 'pie':
                            # 파이 차트의 경우 데이터셋이 하나만 있어야 함
                            if len(data['datasets']) != 1:
                                self.logger.error("파이 차트는 하나의 데이터셋만 가능")
                                raise ValueError("파이 차트는 하나의 데이터셋만 가질 수 있습니다")

                        # 라벨 번역
                        translated_chart = chart_data['chart'].copy()
                        translated_chart['data'] = self.translate_chart_labels(data)
                        
                        # 번역된 차트 데이터 저장 (파일 생성은 _save_section_data에서 수행)
                        result["chart_data"] = translated_chart
                        self.logger.debug(f"차트 생성 성공: {chart_type} 타입, {len(data['labels'])} 라벨, {len(data['datasets'])} 데이터셋")

                except Exception as e:
                    self.logger.error(f"차트 생성 오류: {str(e)}")
                    traceback.print_exc()

            # 4. 테이블 생성 시도 (스키마 기반)
            if requires_table and not result.get("table"):
                try:
                    # 템플릿에 table_schema가 있으면 스키마 기반 생성 사용
                    table_schema = None
                    if isinstance(template, SectionConfig) and template.table_schema:
                        table_schema = template.table_schema
                        self.logger.debug(f"테이블 스키마 발견: {table_schema.get('title', '제목없음')}")
                    
                    # 스키마 기반 또는 기존 방식으로 테이블 생성
                    table_data = await self.get_table_data_with_schema(
                        input_data['current_section'], 
                        content, 
                        table_schema
                    )
                    
                    if table_data:
                        # 테이블 데이터 형식 검증
                        if not isinstance(table_data, dict) or 'headers' not in table_data or 'rows' not in table_data:
                            self.logger.error("테이블 데이터 형식 오류")
                            raise ValueError("테이블 데이터 형식이 올바르지 않습니다")
                        
                        # 모든 행의 열 개수가 헤더와 일치하는지 확인
                        headers_count = len(table_data['headers'])
                        for row_idx, row in enumerate(table_data['rows']):
                            if len(row) != headers_count:
                                self.logger.error(f"테이블 행 {row_idx + 1}의 열 개수가 헤더와 일치하지 않습니다")
                                raise ValueError(f"테이블 행 {row_idx + 1}의 열 개수가 헤더와 일치하지 않습니다")
                        
                        # 테이블 데이터만 저장 (파일 생성은 _save_section_data에서 수행)
                        result["table"] = table_data
                        
                        # 스키마 기반인지 기존 방식인지에 따라 로그 메시지 구분
                        schema_info = f" (스키마: {table_schema['title']})" if table_schema else " (기본 방식)"
                        self.logger.debug(f"테이블 생성 성공{schema_info}: {len(table_data['headers'])} 열, {len(table_data['rows'])} 행")
                        
                except Exception as e:
                    self.logger.error(f"테이블 생성 오류: {str(e)}")
                    traceback.print_exc()

            # 다이어그램 생성 시도 (DOT 형식)
            if requires_diagram:
                try:
                    diagram_data = await self.get_diagram_data(input_data['current_section'], content)
                    if diagram_data and 'diagram' in diagram_data:
                        # DOT 형식 다이어그램 데이터 저장
                        result["diagram_data"] = diagram_data['diagram']
                        self.logger.debug(f"다이어그램 생성 성공: {diagram_data['diagram'].get('type', 'digraph')} 타입")
                    else:
                        self.logger.warning("다이어그램 데이터 형식 오류")
                except Exception as e:
                    self.logger.error(f"다이어그램 생성 오류: {str(e)}")
                    traceback.print_exc()

            # 시퀀스/플로우차트/간트 차트 생성 시도 (Mermaid 형식)
            if requires_flow:
                try:
                    # 현재 서브섹션의 section_type 가져오기
                    current_section_type = "default"
                    if isinstance(template, SectionConfig) and template.section_type:
                        current_section_type = template.section_type
                    elif isinstance(template, dict) and 'section_type' in template:
                        current_section_type = template['section_type']
                    
                    self.logger.debug(f"서브섹션 {input_data['current_subsection']}의 section_type: {current_section_type}")
                    
                    # section_type을 직접 전달
                    sequence_data = await self.get_sequence_data(input_data['current_section'], content, section_type)
                    if sequence_data and 'diagram' in sequence_data:
                        diagram_type = sequence_data['diagram'].get('type', 'sequence')
                        
                        # Mermaid 형식 다이어그램 데이터 저장
                        if diagram_type == 'flowchart':
                            result["flowchart_data"] = sequence_data['diagram']
                        elif diagram_type == 'gantt':
                            result["gantt_data"] = sequence_data['diagram']
                        else:  # 기본값은 sequence
                            result["sequence_data"] = sequence_data['diagram']
                            
                        self.logger.debug(f"다이어그램 생성 성공: {diagram_type} 타입")
                    else:
                        self.logger.warning("다이어그램 데이터 형식 오류")

                except Exception as e:
                    self.logger.error(f"다이어그램 생성 오류: {str(e)}")
                    traceback.print_exc()

            # 7. 결과 검증
            if not isinstance(result["content"], str):
                raise ValueError("섹션 내용이 문자열이 아닙니다")

            return result

        except Exception as e:
            self.logger.error(f"섹션 처리 중 오류 발생: {str(e)}")
            traceback.print_exc()
            return {
                "content": f"섹션 처리 실패: {str(e)}",
                "visualization_paths": []
            }

    def create_chart(self, data: Dict, chart_type: str, output_dir: str, timestamp: str, prefix: str) -> Optional[str]:
        """차트 생성"""
        try:
            plt.switch_backend('Agg')
            
            # 출력 디렉토리를 절대 경로로 변환
            abs_output_dir = self._ensure_absolute_path(output_dir)
            os.makedirs(abs_output_dir, exist_ok=True)
            output_path = os.path.join(abs_output_dir, f'{prefix}_{timestamp}.png')
            
            # 이전 차트가 있다면 모두 닫기
            plt.close('all')
            
            # 데이터 구조 표준화
            chart_data = data.get('data', {})
            labels = chart_data.get('labels', [])
            datasets = chart_data.get('datasets', [])
            options = data.get('options', {})
            title = options.get('title', '')
            y_label = options.get('yAxisLabel', '')

            # 데이터 검증
            if not self._validate_chart_data(chart_type, labels, datasets):
                self.logger.error(f"차트 데이터 검증 실패 - 타입: {chart_type}")
                return None

            # matplotlib 설정
            fig, ax, font_prop = create_matplotlib(figsize=(10, 6))
            
            try:
                # 차트 타입별 처리
                if chart_type == 'pie':
                    # 파이 차트 처리...
                    dataset = datasets[0]
                    values = dataset.get('data', [])
                    wedges, texts, autotexts = ax.pie(
                        values,
                        labels=labels,
                        autopct='%1.1f%%',
                        textprops={'fontproperties': font_prop},
                        startangle=90
                    )
                    plt.setp(autotexts, size=8, weight="bold")
                    plt.setp(texts, size=10)
                    ax.legend(
                        wedges,
                        labels,
                        title=dataset.get('label', ''),
                        loc="center left",
                        bbox_to_anchor=(1, 0, 0.5, 1),
                        prop=font_prop
                    )
                    ax.axis('equal')

                elif chart_type == 'radar':
                    # 레이더 차트 처리...
                    ax = fig.add_subplot(111, projection='polar')
                    angles = np.linspace(0, 2*np.pi, len(labels), endpoint=False)
                    angles = np.concatenate((angles, [angles[0]]))
                    
                    for dataset in datasets:
                        values = dataset.get('data', [])
                        label = dataset.get('label', '')
                        values = np.concatenate((values, [values[0]]))
                        ax.plot(angles, values, 'o-', linewidth=2, label=label)
                        ax.fill(angles, values, alpha=0.25)
                    
                    ax.set_xticks(angles[:-1])
                    ax.set_xticklabels(labels, fontproperties=font_prop)
                    ax.legend(loc='upper right', bbox_to_anchor=(0.1, 0.1), prop=font_prop)

                elif chart_type == 'line':
                    # 라인 차트 처리...
                    for dataset in datasets:
                        values = dataset.get('data', [])
                        label = dataset.get('label', '')
                        line = ax.plot(range(len(labels)), values, marker='o', linewidth=2, markersize=8, label=label)
                        
                        for i, value in enumerate(values):
                            value_text = f'{value:,.0f}' if not isinstance(y_label, str) or "개월" not in y_label.lower() else f'{value}개월'
                            ax.text(i, value, value_text, ha='center', va='bottom', fontproperties=font_prop)
                    
                    ax.set_xticks(range(len(labels)))
                    ax.set_xticklabels(labels, fontproperties=font_prop, rotation=45, ha='right')
                    ax.grid(True, linestyle='--', alpha=0.7)
                    if len(datasets) > 1:
                        ax.legend(loc='upper left', bbox_to_anchor=(1, 1), prop=font_prop)

                elif chart_type == 'bar':
                    # 막대 차트 처리...
                    x = np.arange(len(labels))
                    width = 0.35 / len(datasets)
                    
                    for i, dataset in enumerate(datasets):
                        values = dataset.get('data', [])
                        label = dataset.get('label', '')
                        bars = ax.bar(x + i*width, values, width, label=label)
                        
                        # 막대 위에 값 표시
                        for bar in bars:
                            height = bar.get_height()
                            ax.text(bar.get_x() + bar.get_width()/2., height,
                                   f'{height:,.0f}',
                                   ha='center', va='bottom', fontproperties=font_prop)
                    
                    ax.set_xticks(x + width * (len(datasets)-1)/2)
                    ax.set_xticklabels(labels, fontproperties=font_prop, rotation=45, ha='right')
                    ax.legend(prop=font_prop)

                # 차트 스타일링 (레이더와 파이 차트가 아닌 경우에만)
                if chart_type not in ['radar', 'pie']:
                    ax.set_title(title, fontproperties=font_prop, fontsize=14, pad=20)
                    ax.set_xlabel('기간', fontproperties=font_prop, fontsize=12)
                    ax.set_ylabel(y_label, fontproperties=font_prop, fontsize=12)
                    ax.grid(True, linestyle='--', alpha=0.7)
                    ax.yaxis.set_major_formatter(plt.FuncFormatter(lambda x, p: format(int(x), ',')))

                plt.tight_layout()
                
                # 파일 저장
                fig.savefig(output_path, dpi=300, bbox_inches='tight')
                
                # 차트 자원 해제
                plt.close(fig)
                
                self.logger.debug(f"차트 생성 완료: {output_path}")
                return output_path
                
            finally:
                # 예외가 발생하더라도 반드시 차트 자원 해제
                plt.close(fig)
                
        except Exception as e:
            self.logger.error(f"차트 생성 실패: {str(e)}")
            traceback.print_exc()
            return None

    def translate_chart_labels(self, chart_data: Dict) -> Dict:
        """차트 레이블 한국어 변환"""
        if not chart_data:
            return chart_data
        
        # 차트 제목과 축 레이블 변환
        if "options" in chart_data:
            options = chart_data["options"]
            if "title" in options:
                options["title"] = self.CHART_LABEL_MAPPING.get(options["title"], options["title"])
            if "yAxisLabel" in options:
                options["yAxisLabel"] = self.CHART_LABEL_MAPPING.get(options["yAxisLabel"], options["yAxisLabel"])
            
        # 데이터셋 레이블 변환
        if "data" in chart_data and "datasets" in chart_data["data"]:
            for dataset in chart_data["data"]["datasets"]:
                if "label" in dataset:
                    dataset["label"] = self.CHART_LABEL_MAPPING.get(dataset["label"], dataset["label"])
                
        return chart_data

    def _sanitize_for_json(self, obj):
        """SectionConfig 객체를 안전하게 JSON 직렬화 가능한 형태로 변환"""
        if isinstance(obj, SectionConfig):
            return obj.to_dict()
        elif isinstance(obj, dict):
            return {k: self._sanitize_for_json(v) for k, v in obj.items()}
        elif isinstance(obj, list):
            return [self._sanitize_for_json(item) for item in obj]
        elif hasattr(obj, '__dict__'):
            # 기타 객체는 딕셔너리로 변환 시도
            try:
                return self._sanitize_for_json(obj.__dict__)
            except:
                return str(obj)
        else:
            return obj

    def _format_prompt(self, template: str, context: Dict) -> str:
        """프롬프트 템플릿 포맷팅"""
        try:
            # 템플릿의 플레이스홀더를 context 값으로 대체
            formatted = template
            for key, value in context.items():
                try:
                    if isinstance(value, SectionConfig):
                        # SectionConfig 객체는 딕셔너리로 변환 후 JSON 문자열로 변환
                        value = json.dumps(value.to_dict(), ensure_ascii=False, indent=2)
                    elif isinstance(value, (dict, list)):
                        # 딕셔너리나 리스트는 안전하게 정리한 후 JSON 문자열로 변환
                        sanitized_value = self._sanitize_for_json(value)
                        value = json.dumps(sanitized_value, ensure_ascii=False, indent=2)
                    
                    # 안전한 문자열 변환
                    if not isinstance(value, str):
                        value = str(value) if value is not None else ""
                    
                    placeholder = "{" + key + "}"
                    formatted = formatted.replace(placeholder, value)
                except Exception as e:
                    self.logger.warning(f"컨텍스트 키 '{key}' 처리 실패: {str(e)}")
                    # 실패한 경우 빈 문자열로 대체
                    placeholder = "{" + key + "}"
                    formatted = formatted.replace(placeholder, "")
            return formatted
            
        except Exception as e:
            self.logger.error(f"프롬프트 포맷팅 실패: {str(e)}")
            self.logger.error(f"문제가 된 context 키들: {list(context.keys())}")
            # 각 키별로 타입 확인
            for key, value in context.items():
                self.logger.error(f"  - {key}: {type(value)}")
            return template

    def _validate_section_files(self, section_data: Dict, cache_paths: Dict, section_name: str, subsection_name: str) -> bool:
        """섹션 파일 검증"""
        try:
            self.logger.debug(f"섹션 파일 검증 시작: {section_name}/{subsection_name}")
            
            # 1. 섹션 순서 정보 가져오기
            section_order = self.section_order.get(section_name)
            subsection_order = self.subsection_order.get(section_name, {}).get(subsection_name)
            
            if not section_order or not subsection_order:
                self.logger.error(f"섹션 순서 정보 없음: {section_name}/{subsection_name}")
                return False

            # 2. 파일 기본 정보 구성
            cache_dir = cache_paths['cache_dir']
            safe_section = self._sanitize_filename(section_name)
            safe_subsection = self._sanitize_filename(subsection_name)
            
            # 3. content.json 파일 검증
            content_pattern = f"{section_order}-{subsection_order}-{safe_section}-{safe_subsection}-content-*.json"
            content_files = list(Path(cache_dir).glob(content_pattern))
            
            if not content_files:
                self.logger.error(f"content.json 파일을 찾을 수 없음: {content_pattern}")
                return False

            # 가장 최근 파일 사용
            content_file = sorted(content_files, key=lambda x: int(x.name.split('-')[-1].split('.')[0]))[-1]
            
            # 캐시 유효성 검사 (24시간)
            if time.time() - os.path.getmtime(content_file) > 24 * 60 * 60:
                self.logger.debug(f"캐시 만료: {content_file}")
                return False
            
            try:
                with open(content_file, 'r', encoding='utf-8') as f:
                    content_data = json.load(f)
                    if not content_data.get('content'):
                        self.logger.error("content.json에 내용이 없음")
                        return False
                    
                    # 캐시된 내용을 section_data에 복원
                    section_data['content'] = content_data['content']
                    
            except Exception as e:
                self.logger.error(f"content.json 파일 읽기 실패: {str(e)}")
                return False

            # 4. 차트 파일 검증 - 템플릿에 requires_chart 설정이 있는 경우에만 체크
            template = self.templates.get(section_name, {}).get(subsection_name)
            requires_chart = False

            if isinstance(template, SectionConfig) and template.requires_chart:
                requires_chart = True

            if requires_chart:
                chart_pattern = f"{section_order}-{subsection_order}-{safe_section}-{safe_subsection}-chart-*.png"
                chart_files = list(Path(cache_dir).glob(chart_pattern))

                # 차트 파일이 없거나 차트 메타데이터가 없으면 재생성 필요
                chartdata_pattern = f"{section_order}-{subsection_order}-{safe_section}-{safe_subsection}-chartdata-*.json"
                chartdata_files = list(Path(cache_dir).glob(chartdata_pattern))

                if not chart_files or not chartdata_files:
                    # 차트 파일이나 메타데이터가 없으면 차트 데이터를 삭제하여 재생성 유도
                    if 'chart_data' in section_data:
                        self.logger.debug(f"차트 파일 또는 메타데이터 없음: 재생성 필요")
                        del section_data['chart_data']
                elif chartdata_files:
                    # 차트 메타데이터 로드
                    chartdata_file = sorted(chartdata_files, key=lambda x: int(x.name.split('-')[-1].split('.')[0]))[-1]
                    try:
                        with open(chartdata_file, 'r', encoding='utf-8') as f:
                            chart_meta = json.load(f)
                            section_data['chart_data'] = chart_meta.get('chart_data')
                    except Exception as e:
                        self.logger.error(f"차트 메타데이터 로드 실패: {str(e)}")
                        if 'chart_data' in section_data:
                            del section_data['chart_data']

            # 5. 테이블 파일 검증 - 템플릿에 requires_table 설정이 있는 경우에만 체크
            requires_table = False

            if isinstance(template, SectionConfig) and template.requires_table:
                requires_table = True

            if requires_table:
                table_pattern = f"{section_order}-{subsection_order}-{safe_section}-{safe_subsection}-table-*.csv"
                table_files = list(Path(cache_dir).glob(table_pattern))

                # 테이블 파일이 없거나 테이블 메타데이터가 없으면 재생성 필요
                tabledata_pattern = f"{section_order}-{subsection_order}-{safe_section}-{safe_subsection}-tabledata-*.json"
                tabledata_files = list(Path(cache_dir).glob(tabledata_pattern))

                if not table_files or not tabledata_files:
                    # 테이블 파일이나 메타데이터가 없으면 테이블 데이터를 삭제하여 재생성 유도
                    if 'table' in section_data:
                        self.logger.debug(f"테이블 파일 또는 메타데이터 없음: 재생성 필요")
                        del section_data['table']
                elif tabledata_files:
                    # 테이블 메타데이터 로드
                    tabledata_file = sorted(tabledata_files, key=lambda x: int(x.name.split('-')[-1].split('.')[0]))[-1]
                    try:
                        with open(tabledata_file, 'r', encoding='utf-8') as f:
                            table_data = json.load(f)
                            section_data['table'] = table_data.get('table')
                    except Exception as e:
                        self.logger.error(f"테이블 데이터 로드 실패: {str(e)}")
                        if 'table' in section_data:
                            del section_data['table']

            # 6. 다이어그램 파일 검증 - 템플릿에 requires_diagram 또는 requires_flow 설정이 있는 경우에만 체크
            requires_diagram = False
            requires_flow = False

            if isinstance(template, SectionConfig):
                requires_diagram = template.requires_diagram
                requires_flow = template.requires_flow

            if requires_diagram or requires_flow:
                diagram_pattern = f"{section_order}-{subsection_order}-{safe_section}-{safe_subsection}-diagram-*.png"
                diagram_files = list(Path(cache_dir).glob(diagram_pattern))

                # 다이어그램 파일이 없거나 다이어그램 메타데이터가 없으면 재생성 필요
                diagramdata_pattern = f"{section_order}-{subsection_order}-{safe_section}-{safe_subsection}-diagramdata-*.json"
                diagramdata_files = list(Path(cache_dir).glob(diagramdata_pattern))

                if not diagram_files or not diagramdata_files:
                    # 다이어그램 파일이나 메타데이터가 없으면 다이어그램 데이터를 삭제하여 재생성 유도
                    if 'diagram_data' in section_data:
                        self.logger.debug(f"다이어그램 파일 또는 메타데이터 없음: 재생성 필요")
                        del section_data['diagram_data']
                    if 'sequence_data' in section_data:
                        self.logger.debug(f"시퀀스 다이어그램 파일 또는 메타데이터 없음: 재생성 필요")
                        del section_data['sequence_data']
                    if 'flowchart_data' in section_data:
                        self.logger.debug(f"플로우차트 다이어그램 파일 또는 메타데이터 없음: 재생성 필요")
                        del section_data['flowchart_data']
                    if 'gantt_data' in section_data:
                        self.logger.debug(f"간트 다이어그램 파일 또는 메타데이터 없음: 재생성 필요")
                        del section_data['gantt_data']
                elif diagramdata_files:
                    # 다이어그램 메타데이터 로드
                    diagramdata_file = sorted(diagramdata_files, key=lambda x: int(x.name.split('-')[-1].split('.')[0]))[-1]
                    try:
                        with open(diagramdata_file, 'r', encoding='utf-8') as f:
                            diagram_meta = json.load(f)
                            section_data['diagram_data'] = diagram_meta.get('diagram_data')
                    except Exception as e:
                        self.logger.error(f"다이어그램 메타데이터 로드 실패: {str(e)}")
                        if 'diagram_data' in section_data:
                            del section_data['diagram_data']
                        if 'sequence_data' in section_data:
                            del section_data['sequence_data']
                        if 'flowchart_data' in section_data:
                            del section_data['flowchart_data']
                        if 'gantt_data' in section_data:
                            del section_data['gantt_data']

            self.logger.debug(f"섹션 파일 검증 성공: {section_name}/{subsection_name}")
            return True

        except Exception as e:
            self.logger.error(f"섹션 파일 검증 중 오류 발생: {str(e)}")
            traceback.print_exc()
            return False
        
    async def create_business_plan(self, executive_summary: str, output_dir: str, output_format: str = 'pdf', timestamp: str = None, use_cache: bool = True, project_hash: str = None, template: str = None, progress_callback=None, modification_mode: str = 'full', existing_document_path: str = None, existing_document_content: str = None, sections_to_modify: List[str] = None, preserve_formatting: bool = True) -> Dict:
        web_search_results = []
        rag_search_results = []
        # 환경 변수에서 AI 설정 가져오기
        config = load_config()
        provider = config.get('USE_LLM', 'openai')        
        try:           
            start_time = time.time()
            
            # output_path 변수 초기화 (finally 블록에서 사용되므로 미리 정의)
            output_path = None
            
            # 작업 ID 생성
            self.project_hash = project_hash or hashlib.md5(executive_summary.encode('utf-8')).hexdigest()[:8]  # 먼저 project_hash 설정
            self.timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            self.job_id = f"{self.project_hash}_{self.timestamp}"  # job_id는 로깅용으로만 사용
            
            # 사용자별 캐시 경로에 로그 디렉토리 생성
            if self.user_id:
                # 사용자별 캐시 경로: ~/.airun/cache/report/{username}/{project_hash}/logs/
                log_dir = os.path.join(self.base_cache_dir, self.project_hash, 'logs')
            else:
                # 사용자 ID가 없으면 기본 경로 사용
                log_dir = os.path.expanduser('~/.airun/logs')
            
            os.makedirs(log_dir, exist_ok=True)
            
            # 로거를 새로운 로그 파일 경로로 재설정
            log_filename = f"report_generator_{self.project_hash}.log"
            log_file_path = os.path.join(log_dir, log_filename)
            self.logger = _setup_logger(job_id=self.job_id, project_hash=self.project_hash, log_file=log_file_path)
            
            self.logger.info(f"🚀 문서 생성을 시작합니다... (작업 ID: {self.job_id})")
            self.logger.info(f"📁 로그 파일 경로: {log_file_path}")
            
            # 진행률 콜백 호출
            if progress_callback:
                progress_callback("문서 생성 초기화 중...", 5)

            # 사용자 글로벌 프롬프트 추출 (추가된 부분)
            user_global_prompt = self._extract_user_global_prompt(executive_summary)
            if user_global_prompt:
                self.logger.debug(f"사용자 글로벌 프롬프트 추출됨: {user_global_prompt[:100]}...")
                self.logger.info(f"✅ 사용자 지정 작성 규칙이 적용됩니다.")
                            
            # executive_summary 설정
            self.executive_summary = executive_summary
            
            # 부분 수정 모드 처리
            existing_sections = {}
            if modification_mode == 'partial':
                self.logger.info("📝 부분 수정 모드가 활성화되었습니다.")
                
                # 기존 문서 파싱
                if existing_document_path and os.path.exists(existing_document_path):
                    parser = DocumentParser(logger=self.logger)
                    existing_sections = parser.parse_document(existing_document_path)
                    self.logger.info(f"📄 기존 문서 파싱 완료: {len(existing_sections)}개 섹션")
                elif existing_document_content:
                    parser = DocumentParser(logger=self.logger)
                    existing_sections = parser.parse_text_content(existing_document_content)
                    self.logger.info(f"📄 기존 문서 내용 파싱 완료: {len(existing_sections)}개 섹션")
                else:
                    self.logger.warning("⚠️ 부분 수정 모드이지만 기존 문서가 제공되지 않았습니다. 전체 재생성으로 진행합니다.")
                    modification_mode = 'full'
                
                # 수정할 섹션 검증
                if sections_to_modify:
                    # 사용자가 지정한 섹션들이 실제로 존재하는지 확인
                    available_sections = list(existing_sections.keys())
                    valid_sections = []
                    for section in sections_to_modify:
                        if section in available_sections:
                            valid_sections.append(section)
                        else:
                            # 유사한 섹션명 찾기
                            similar = [s for s in available_sections if section.lower() in s.lower() or s.lower() in section.lower()]
                            if similar:
                                self.logger.warning(f"⚠️ 섹션 '{section}'을 찾을 수 없습니다. 유사한 섹션: {similar}")
                            else:
                                self.logger.warning(f"⚠️ 섹션 '{section}'을 찾을 수 없습니다.")
                    
                    sections_to_modify = valid_sections
                    self.logger.info(f"🔧 수정 대상 섹션: {sections_to_modify}")
                else:
                    self.logger.warning("⚠️ 수정할 섹션이 지정되지 않았습니다. 전체 재생성으로 진행합니다.")
                    modification_mode = 'full'
            
            # 템플릿 로드
            raw_templates = self.load_prompt_templates(executive_summary, template=template)
            if not raw_templates:
                self.logger.error("템플릿 로드 실패")
                raise ValueError("템플릿 로드 실패")
                
            # 섹션 템플릿 초기화
            self.templates = {}
            self.section_order = {}
            self.subsection_order = {}
            
            # sections 변수 명시적 초기화
            sections = {}
            
            # 메타데이터 키를 제외한 섹션 목록 생성
            metadata_keys = {'name', 'description', 'category', '_metadata'}
            section_list = [key for key in raw_templates.keys() if key not in metadata_keys]
            
            # 섹션 순서 설정
            for section_idx, section_name in enumerate(section_list, 1):
                section_order = f"{section_idx:02d}"
                self.section_order[section_name] = section_order
                
                subsections = raw_templates[section_name]
                if not isinstance(subsections, dict):
                    self.logger.error(f"잘못된 하위 섹션 형식: {section_name}")
                    continue
                
                self.subsection_order[section_name] = {}
                self.templates[section_name] = {}
                
                # 하위 섹션 순서 설정
                subsection_list = list(subsections.keys())
                for subsection_idx, subsection_name in enumerate(subsection_list, 1):
                    subsection_order = f"{subsection_idx:02d}"
                    self.subsection_order[section_name][subsection_name] = subsection_order
                    
                    # SectionConfig 객체 생성
                    template_data = subsections[subsection_name]
                    self.templates[section_name][subsection_name] = SectionConfig.from_template(
                        title=subsection_name,
                        template_data=template_data
                    )
            
            # 캐시 상태 확인 로깅 추가
            cache_dir = os.path.join(self.base_cache_dir, self.project_hash)
            self.logger.debug(f"작성 중인 문서 데이터 저장 경로: {cache_dir}")
            os.makedirs(cache_dir, exist_ok=True)
            
            # 캐시 디렉토리 초기화
            cache_paths = self._initialize_cache_directories(cache_dir)
            
            # 캐시된 데이터 로드 및 상태 출력
            cached_data = {}
            if use_cache:
                cached_data = self._load_cached_data(cache_dir)
                print("\n💾 캐시 데이터 상태:")
                print(f"  ├ 사업 정보: {'있음' if 'business_info' in cached_data else '없음'}")
                print(f"  ├ 경쟁사 정보: {'있음' if 'competitors' in cached_data else '없음'}")
                print(f"  ├ 목표 시장 분석: {'있음' if 'target_market_analysis' in cached_data else '없음'}")
                if 'sections' in cached_data:
                    print(f"  ├ 섹션 데이터: {len(cached_data['sections'])}개 섹션")
                    for section_name, subsections in cached_data['sections'].items():
                        print(f"  │  └ {section_name}: {len(subsections)}개 하위섹션")
                else:
                    print("  └ 섹션 데이터: 없음")
            else:
                print("\n⚠️ 캐시 사용하지 않음")

            if os.path.exists(cache_dir):
                cache_files = sorted(list(Path(cache_dir).glob('*')), key=lambda x: (
                    # 000_ 파일들을 먼저 정렬
                    not x.name.startswith('000_'),
                    # 섹션 순서로 정렬
                    int(x.name.split('-')[0] if not x.name.startswith('000_') and '-' in x.name else '0'),
                    # 하위섹션 순서로 정렬
                    int(x.name.split('-')[1] if not x.name.startswith('000_') and '-' in x.name else '0'),
                    # content 파일을 먼저, 그 다음 chart/table 파일, 마지막으로 메타데이터 파일 정렬
                    0 if x.name.endswith('content.json') else 
                    1 if x.name.endswith(('.png', '.csv')) else 
                    2 if x.name.endswith(('chartdata.json', 'tabledata.json')) else 3,
                    # 마지막으로 파일명으로 정렬
                    x.name
                ))
                print(f"  ├ 캐시 폴더 존재: {len(cache_files)}개 파일 발견")
                for file in cache_files:
                    self.logger.debug(f"  │  └ {file.name} ({datetime.fromtimestamp(file.stat().st_mtime).strftime('%Y-%m-%d %H:%M:%S')})")
            else:
                self.logger.debug("  └ 캐시 폴더 없음: 새로 생성됨")

            # 전체 섹션 수 계산
            self.total_sections = 0
            section_counts = {}
            for section_name, subsections in self.templates.items():
                if section_name != '_metadata' and isinstance(subsections, dict):
                    section_order = self.section_order.get(section_name, '00')
                    section_counts[section_order] = len(subsections)
                    self.total_sections += len(subsections)

            self.logger.debug(f"총 섹션 수: {self.total_sections}")
            self.logger.debug("섹션별 하위섹션 수:")
            for section_order, count in sorted(section_counts.items()):
                self.logger.debug(f"  섹션 {section_order}: {count}개 하위섹션")
            self.completed_sections = 0

            # 출력 디렉토리 설정
            self.output_dir = output_dir
            os.makedirs(self.output_dir, exist_ok=True)
            
            # 캐시 디렉토리 설정
            cache_paths = self._initialize_cache_directories(cache_dir)
            
            # 캐시된 데이터 로드 및 상태 출력
            cached_data = {}
            if use_cache:
                cached_data = self._load_cached_data(cache_dir)
                print("\n💾 캐시 데이터 상태:")
                print(f"  ├ 사업 정보: {'있음' if 'business_info' in cached_data else '없음'}")
                print(f"  ├ 경쟁사 정보: {'있음' if 'competitors' in cached_data else '없음'}")
                print(f"  ├ 목표 시장 분석: {'있음' if 'target_market_analysis' in cached_data else '없음'}")
                if 'sections' in cached_data:
                    print(f"  ├ 섹션 데이터: {len(cached_data['sections'])}개 섹션")
                    for section_name, subsections in cached_data['sections'].items():
                        print(f"  │  └ {section_name}: {len(subsections)}개 하위섹션")
                else:
                    print("  └ 섹션 데이터: 없음")
            else:
                print("\n⚠️ 캐시 사용하지 않음")

            if os.path.exists(cache_dir):
                cache_files = sorted(list(Path(cache_dir).glob('*')), key=lambda x: (
                    # 000_ 파일들을 먼저 정렬
                    not x.name.startswith('000_'),
                    # 섹션 순서로 정렬
                    int(x.name.split('-')[0] if not x.name.startswith('000_') and '-' in x.name else '0'),
                    # 하위섹션 순서로 정렬
                    int(x.name.split('-')[1] if not x.name.startswith('000_') and '-' in x.name else '0'),
                    # content 파일을 먼저, 그 다음 chart/table 파일, 마지막으로 메타데이터 파일 정렬
                    0 if x.name.endswith('content.json') else 
                    1 if x.name.endswith(('.png', '.csv')) else 
                    2 if x.name.endswith(('chartdata.json', 'tabledata.json')) else 3,
                    # 마지막으로 파일명으로 정렬
                    x.name
                ))
                print(f"  ├ 캐시 폴더 존재: {len(cache_files)}개 파일 발견")
                for file in cache_files:
                    self.logger.debug(f"  │  └ {file.name} ({datetime.fromtimestamp(file.stat().st_mtime).strftime('%Y-%m-%d %H:%M:%S')})")
            else:
                self.logger.debug("  └ 캐시 폴더 없음: 새로 생성됨")

            # 메타데이터 처리
            metadata_file = cache_paths['metadata']
            if os.path.exists(metadata_file):
                try:
                    with open(metadata_file, 'r', encoding='utf-8') as f:
                        metadata = json.load(f)
                        last_updated = metadata.get('last_updated', '알 수 없음')
                        if last_updated != '알 수 없음':
                            try:
                                # ISO 형식의 시간을 한국 시간으로 변환
                                last_updated_dt = datetime.fromisoformat(last_updated)
                                last_updated_str = last_updated_dt.strftime('%Y-%m-%d %H:%M:%S')
                                self.logger.debug(f"기존 프로젝트 메타데이터 로드: {last_updated_str}")
                            except ValueError:
                                self.logger.debug(f"기존 프로젝트 메타데이터 로드: {last_updated}")
                        else:
                            self.logger.debug("기존 프로젝트 메타데이터 로드: 알 수 없음")
                except Exception as e:
                    self.logger.error(f"메타데이터 로드 실패: {str(e)}")
                    metadata = {}
            else:
                metadata = {}

            # 메타데이터 업데이트 - 작업 ID 및 사용자 정보 포함
            metadata.update({
                'job_id': self.job_id,
                'project_hash': self.project_hash,
                'timestamp': self.timestamp,
                'executive_summary': executive_summary,
                'last_updated': datetime.now().isoformat(),
                'outputs': metadata.get('outputs', [])
            })
            
            # 사용자 정보가 있는 경우 메타데이터에 추가
            if hasattr(self, 'user_id') and self.user_id:
                metadata['user_id'] = self.user_id

            # 메타데이터 저장
            try:
                with open(metadata_file, 'w', encoding='utf-8') as f:
                    json.dump(metadata, f, ensure_ascii=False, indent=2)
            except Exception as e:
                self.logger.error(f"메타데이터 저장 실패: {str(e)}")

            # 1. 사업 정보 분석
            self.logger.info("📊 기본 정보 분석 중...")
            if progress_callback:
                progress_callback("기본 정보 분석 중...", 10)
                
            business_info = cached_data.get('business_info')
            if not business_info:
                # 외부 RAG 클라이언트 사용 (자원 중복 방지)
                if self.external_rag_client:
                    self.logger.info("✅ 외부 RAG 클라이언트를 사용하여 사업 정보 분석")
                
                business_info = await self.analyze_business_info(executive_summary, progress_callback)
                self.logger.info("✅ 기본 정보 분석 완료")
                with open(cache_paths['business_info'], 'w', encoding='utf-8') as f:
                    json.dump({'business_info': business_info}, f, ensure_ascii=False, indent=2)
            else:
                self.logger.info("✅ 캐시된 사업 정보 사용")
                
            if progress_callback:
                progress_callback("기본 정보 분석 완료", 15)

            # 2. 경쟁사 분석
            self.logger.info("🔍 경쟁사 분석 중...")
            if progress_callback:
                progress_callback("경쟁사 분석 중...", 20)
                
            competitors = cached_data.get('competitors')
            if not competitors:
                # 외부 웹검색 클라이언트 사용 (자원 중복 방지)
                if self.external_web_client:
                    self.logger.info("✅ 외부 웹검색 클라이언트를 사용하여 경쟁사 분석")
                
                competitors = await self.analyze_competitors(business_info)
                print("✅ 경쟁사 분석 완료")
                with open(cache_paths['competitors'], 'w', encoding='utf-8') as f:
                    json.dump(competitors, f, ensure_ascii=False, indent=2)
            else:
                print("✅ 캐시된 경쟁사 정보 사용")
            
            business_info['competitors'] = competitors
            
            if progress_callback:
                progress_callback("경쟁사 분석 완료", 25)

            # 3. 섹션 생성
            print("\n📝 섹션 생성 시작...")
            if progress_callback:
                progress_callback("섹션 생성 시작...", 30)
                
            # sections 변수를 명시적으로 초기화
            sections = {}
            # 캐시된 섹션 데이터가 있으면 로드
            if use_cache and 'sections' in cached_data:
                sections = cached_data['sections']
                print(f"  ├ 캐시된 섹션 데이터 로드됨: {len(sections)}개 섹션")
                for section_name, subsections in sections.items():
                    print(f"  │  └ {section_name}: {len(subsections)}개 하위섹션")
            else:
                print("  └ 새로운 섹션 데이터 생성")
            
            # context 생성 시 SectionConfig 객체 안전 처리
            executive_summary_value = metadata.get('executive_summary', executive_summary)
            if isinstance(executive_summary_value, SectionConfig):
                executive_summary_value = executive_summary_value.prompt
            elif executive_summary_value is None:
                executive_summary_value = executive_summary
            
            context = {
                'business_info': business_info,
                'executive_summary': executive_summary_value,
                'generated_sections': {},
                'user_global_prompt': user_global_prompt
            }

            self.logger.debug("Context dictionary created for prompt formatting. Checking types:")
            for k, v in context.items():
                self.logger.debug(f"  - context['{k}']: type={type(v)}")
                if isinstance(v, dict):
                    for k2, v2 in v.items():
                        if isinstance(v2, dict):
                            self.logger.debug(f"    - context['{k}']['{k2}']: type={type(v2)} contains dict")
                        else:
                            self.logger.debug(f"    - context['{k}']['{k2}']: type={type(v2)}")

            for section_name, subsections in self.templates.items():
                if section_name == '_metadata':
                    continue

                print(f"\n📑 {section_name} 작성 중...")
                self.logger.debug(f"섹션 처리 시작: {section_name}, 타입: {type(subsections)}")
                
                # 부분 수정 모드에서 섹션 스킵 여부 확인
                if modification_mode == 'partial':
                    # 현재 섹션이 수정 대상이 아닌 경우, 기존 내용 유지
                    section_needs_modification = False
                    
                    # sections_to_modify가 지정된 경우, 해당 섹션만 수정
                    if sections_to_modify:
                        for target_section in sections_to_modify:
                            if target_section.lower() in section_name.lower() or section_name.lower() in target_section.lower():
                                section_needs_modification = True
                                break
                    
                    if not section_needs_modification:
                        # 기존 섹션 내용을 찾아서 유지
                        existing_content = None
                        for existing_section_name, existing_content_text in existing_sections.items():
                            if existing_section_name.lower() in section_name.lower() or section_name.lower() in existing_section_name.lower():
                                existing_content = existing_content_text
                                break
                        
                        if existing_content:
                            print(f"  ⏭️ {section_name} - 기존 내용 유지 (수정 대상 아님)")
                            # 기존 내용을 섹션 데이터로 변환
                            if section_name not in sections:
                                sections[section_name] = {}
                            # 임시로 단일 하위섹션으로 처리 (향후 더 정교한 파싱 가능)
                            subsection_name = list(subsections.keys())[0] if subsections else "content"
                            sections[section_name][subsection_name] = {
                                'content': existing_content,
                                'title': section_name
                            }
                            continue
                        else:
                            print(f"  ⚠️ {section_name} - 기존 내용을 찾을 수 없어 새로 생성합니다.")
                
                # 섹션이 없는 경우에만 초기화
                if section_name not in sections:
                    sections[section_name] = {}

                if isinstance(subsections, dict):
                    for subsection_name, template in subsections.items():
                        try:
                            self.logger.debug(f"하위섹션 처리 시작: {section_name}/{subsection_name}, 템플릿 타입: {type(template)}")
                            
                            # 템플릿 객체 상세 디버깅
                            if isinstance(template, SectionConfig):
                                self.logger.debug(f"SectionConfig 객체 - title: {template.title}, prompt 길이: {len(template.prompt)}")
                            else:
                                self.logger.debug(f"템플릿 내용: {str(template)[:100]}...")
                            
                            # 진행률 계산 (30-85% 구간에서 섹션 처리)
                            base_progress = 30
                            section_progress_range = 55  # 85 - 30 = 55
                            section_progress = (self.completed_sections / self.total_sections) * section_progress_range
                            current_progress = base_progress + section_progress
                            
                            context['current_section'] = section_name
                            context['current_subsection'] = subsection_name
                            # SectionConfig 객체를 안전하게 처리
                            if isinstance(template, SectionConfig):
                                context['template'] = template  # SectionConfig 객체는 그대로 전달 (process_section_template에서 처리)
                                self.logger.debug(f"context에 SectionConfig 객체 설정: {template.title}")
                            else:
                                context['template'] = template
                                self.logger.debug(f"context에 일반 템플릿 설정: {type(template)}")
                            
                            # 진행률 콜백 호출
                            if progress_callback:
                                await progress_callback(f"{section_name} - {subsection_name} 처리 중...", int(current_progress))

                            # 캐시 검증 - 파일 존재 여부와 내용 검증
                            cache_key = f"{section_name}_{subsection_name}"
                            section_data = sections.get(section_name, {}).get(subsection_name, {})
                            
                            # 캐시 파일 패턴 구성
                            section_order = self.section_order.get(section_name, '00')
                            subsection_order = self.subsection_order.get(section_name, {}).get(subsection_name, '00')
                            if section_order and subsection_order:
                                # 새로운 파일명 규칙에 맞게 패턴 수정
                                file_pattern = f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-*"
                                cache_files = list(Path(cache_paths['cache_dir']).glob(file_pattern))
                                
                                # 캐시 파일이 있고 유효하면 사용
                                if use_cache and cache_files:
                                    # content.json 파일 찾기
                                    content_file = next((f for f in cache_files if 'content' in f.name), None)
                                    if content_file and content_file.exists():
                                        try:
                                            with open(content_file, 'r', encoding='utf-8') as f:
                                                cached_content = json.load(f)
                                                if cached_content and 'content' in cached_content:
                                                    # 템플릿 요구사항 확인
                                                    requires_chart = False
                                                    requires_table = False
                                                    requires_diagram = False
                                                    requires_flow = False
                                                    
                                                    if isinstance(template, SectionConfig):
                                                        requires_chart = template.requires_chart
                                                        requires_table = template.requires_table
                                                        requires_diagram = template.requires_diagram
                                                        requires_flow = template.requires_flow
                                                    
                                                    # 차트 파일 확인
                                                    chart_missing = False
                                                    if requires_chart:
                                                        chart_pattern = f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-chart-*.png"
                                                        chart_files = list(Path(cache_paths['cache_dir']).glob(chart_pattern))
                                                        
                                                        # 차트 파일이 없는 경우에만 재생성 필요 (메타데이터는 체크하지 않음)
                                                        if len(chart_files) == 0:
                                                            chart_missing = True
                                                            self.logger.debug(f"차트 파일 없음: 재생성 필요 - {section_name}/{subsection_name}")

                                                    # 테이블 파일 확인
                                                    table_missing = False
                                                    if requires_table:
                                                        table_pattern = f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-table-*.csv"
                                                        table_files = list(Path(cache_paths['cache_dir']).glob(table_pattern))
                                                        
                                                        # 테이블 파일이 없는 경우에만 재생성 필요 (메타데이터는 체크하지 않음)
                                                        if len(table_files) == 0:
                                                            table_missing = True
                                                            self.logger.debug(f"테이블 파일 없음: 재생성 필요 - {section_name}/{subsection_name}")

                                                    # 다이어그램 파일 확인
                                                    diagram_missing = False
                                                    if requires_diagram or requires_flow:
                                                        # 다양한 다이어그램 타입 패턴 검색
                                                        diagram_patterns = [
                                                            f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-diagram-*.png",
                                                            f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-flowchart-*.png",
                                                            f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-gantt-*.png",
                                                            f"{section_order}-{subsection_order}-{self._sanitize_filename(section_name)}-{self._sanitize_filename(subsection_name)}-sequence-*.png"
                                                        ]
                                                        
                                                        # 모든 패턴에 대해 파일 검색
                                                        diagram_files = []
                                                        for pattern in diagram_patterns:
                                                            diagram_files.extend(list(Path(cache_paths['cache_dir']).glob(pattern)))

                                                        # 파일이 하나도 없는 경우에만 재생성 필요
                                                        if len(diagram_files) == 0:
                                                            diagram_missing = True
                                                            self.logger.debug(f"다이어그램 파일 없음: 재생성 필요 - {section_name}/{subsection_name}")
                                                    
                                                    # 필요한 파일이 모두 있으면 캐시 사용, 아니면 재생성
                                                    if not chart_missing and not table_missing and not diagram_missing:
                                                        print(f"  └ {subsection_name} - 캐시 사용 ({current_progress:.1f}%)")
                                                        sections[section_name][subsection_name] = cached_content
                                                        context['generated_sections'][cache_key] = cached_content['content']
                                                        self.completed_sections += 1
                                                        
                                                        # 캐시 사용 시에도 콜백 호출 (섹션 내용 포함)
                                                        if progress_callback:
                                                            try:
                                                                # 캐시된 섹션의 전체 데이터를 구조화하여 전달
                                                                section_data = {
                                                                    'content': cached_content.get('content', ''),
                                                                    'has_chart': 'chart_data' in cached_content,
                                                                    'has_table': 'table' in cached_content,
                                                                    'has_diagram': 'diagram_data' in cached_content,
                                                                    'has_flowchart': 'flowchart_data' in cached_content,
                                                                    'has_gantt': 'gantt_data' in cached_content,
                                                                    'has_sequence': 'sequence_data' in cached_content,
                                                                    'visualization_paths': cached_content.get('visualization_paths', [])
                                                                }
                                                                
                                                                # 차트 데이터가 있으면 추가
                                                                if 'chart_data' in cached_content:
                                                                    section_data['chart_data'] = cached_content['chart_data']
                                                                
                                                                # 테이블 데이터가 있으면 추가
                                                                if 'table' in cached_content:
                                                                    section_data['table_data'] = cached_content['table']
                                                                
                                                                # 다이어그램 데이터가 있으면 추가
                                                                if 'diagram_data' in cached_content:
                                                                    section_data['diagram_data'] = cached_content['diagram_data']
                                                                
                                                                # 플로우차트 데이터가 있으면 추가
                                                                if 'flowchart_data' in cached_content:
                                                                    section_data['flowchart_data'] = cached_content['flowchart_data']
                                                                
                                                                # 간트차트 데이터가 있으면 추가
                                                                if 'gantt_data' in cached_content:
                                                                    section_data['gantt_data'] = cached_content['gantt_data']
                                                                
                                                                # 시퀀스 다이어그램 데이터가 있으면 추가
                                                                if 'sequence_data' in cached_content:
                                                                    section_data['sequence_data'] = cached_content['sequence_data']
                                                                
                                                                # 비동기 함수인지 확인하고 적절하게 호출
                                                                if asyncio.iscoroutinefunction(progress_callback):
                                                                    await progress_callback(f"{section_name} - {subsection_name} 완료 (캐시)", int(current_progress), f"{section_name}_{subsection_name}", section_data)
                                                                else:
                                                                    progress_callback(f"{section_name} - {subsection_name} 완료 (캐시)", int(current_progress), f"{section_name}_{subsection_name}", section_data)
                                                            except Exception as callback_error:
                                                                self.logger.warning(f"progress_callback 호출 중 오류: {str(callback_error)}")
                                                        
                                                        continue
                                                    else:
                                                        print(f"  └ {subsection_name} - 시각화 파일 재생성 ({current_progress:.1f}%)")
                                        except Exception as e:
                                            self.logger.error(f"캐시 파일 읽기 실패: {str(e)}")

                                # 캐시가 없거나 유효하지 않은 경우 새로 생성
                                print(f"  └ {subsection_name} 작성 중... ({current_progress:.1f}%)")
                                
                                # process_section_template 호출 전 디버깅
                                self.logger.debug(f"process_section_template 호출 준비:")
                                self.logger.debug(f"  - template 타입: {type(template)}")
                                self.logger.debug(f"  - context 키들: {list(context.keys())}")
                                self.logger.debug(f"  - cache_paths 타입: {type(cache_paths)}")
                                
                                try:
                                    result = await self.process_section_template(template, context, cache_paths)
                                    self.logger.debug(f"process_section_template 성공, 결과 타입: {type(result)}")
                                except Exception as section_error:
                                    self.logger.error(f"process_section_template 실패:")
                                    self.logger.error(f"  - 오류: {str(section_error)}")
                                    self.logger.error(f"  - 오류 타입: {type(section_error)}")
                                    import traceback
                                    self.logger.error(f"  - 스택 트레이스:\n{traceback.format_exc()}")
                                    raise section_error
                                
                                sections[section_name][subsection_name] = result
                                
                                if not self._save_section_data(result, section_name, subsection_name, cache_paths):
                                    raise ValueError("섹션 데이터 저장 실패")
                                
                                context['generated_sections'][cache_key] = result.get('content', '')
                                self.completed_sections += 1
                                
                                # 섹션 완료 후 콜백 호출 (섹션 내용 포함)
                                if progress_callback:
                                    try:
                                        # 섹션의 전체 데이터를 구조화하여 전달
                                        section_data = {
                                            'content': result.get('content', ''),
                                            'has_chart': 'chart_data' in result,
                                            'has_table': 'table' in result,
                                            'has_diagram': 'diagram_data' in result,
                                            'has_flowchart': 'flowchart_data' in result,
                                            'has_gantt': 'gantt_data' in result,
                                            'has_sequence': 'sequence_data' in result,
                                            'visualization_paths': result.get('visualization_paths', [])
                                        }
                                        
                                        # 차트 데이터가 있으면 추가
                                        if 'chart_data' in result:
                                            section_data['chart_data'] = result['chart_data']
                                        
                                        # 테이블 데이터가 있으면 추가
                                        if 'table' in result:
                                            section_data['table_data'] = result['table']
                                        
                                        # 다이어그램 데이터가 있으면 추가
                                        if 'diagram_data' in result:
                                            section_data['diagram_data'] = result['diagram_data']
                                        
                                        # 플로우차트 데이터가 있으면 추가
                                        if 'flowchart_data' in result:
                                            section_data['flowchart_data'] = result['flowchart_data']
                                        
                                        # 간트차트 데이터가 있으면 추가
                                        if 'gantt_data' in result:
                                            section_data['gantt_data'] = result['gantt_data']
                                        
                                        # 시퀀스 다이어그램 데이터가 있으면 추가
                                        if 'sequence_data' in result:
                                            section_data['sequence_data'] = result['sequence_data']
                                        
                                        # 비동기 함수인지 확인하고 적절하게 호출
                                        if asyncio.iscoroutinefunction(progress_callback):
                                            await progress_callback(f"{section_name} - {subsection_name} 완료", int(current_progress), f"{section_name}_{subsection_name}", section_data)
                                        else:
                                            progress_callback(f"{section_name} - {subsection_name} 완료", int(current_progress), f"{section_name}_{subsection_name}", section_data)
                                    except Exception as callback_error:
                                        self.logger.warning(f"progress_callback 호출 중 오류: {str(callback_error)}")

                        except Exception as e:
                            self.logger.error(f"섹션 처리 중 오류 발생:")
                            self.logger.error(f"  - 섹션: {section_name}/{subsection_name}")
                            self.logger.error(f"  - 오류: {str(e)}")
                            self.logger.error(f"  - 오류 타입: {type(e)}")
                            import traceback
                            self.logger.error(f"  - 스택 트레이스:\n{traceback.format_exc()}")
                            
                            print("    ❌ 오류: %s", str(e))
                            sections[section_name][subsection_name] = {
                                "content": f"섹션 생성 실패: {str(e)}",
                                "visualization_paths": []
                            }
                            self.completed_sections += 1

            # 4. 문서 생성 및 저장
            print("\n📄 최종 문서 생성 중...")
            if progress_callback:
                progress_callback("최종 문서 생성 중...", 90)
                
            self.create_document(sections, business_info, cache_paths)
            
            if progress_callback:
                progress_callback("문서 저장 중...", 95)
                
            # product_name을 기반으로 최종 파일명 재구성
            product_name = business_info.get('product_name', '')
            if product_name:
                # 파일명에 안전한 문자로 변환
                safe_product_name = self._sanitize_filename(product_name[:30])  # 길이 제한
                
                # 타임스탬프 사용 (전달받은 timestamp 또는 현재 시간)
                if timestamp:
                    file_timestamp = timestamp
                else:
                    file_timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
                
                # 새 파일명 생성
                final_filename = f"{safe_product_name}_{file_timestamp}.{output_format}"
                final_output_path = os.path.join(output_dir, final_filename)
                
                self.logger.info(f"파일명 결정: product_name='{product_name}' -> '{final_filename}'")
                
                # 새 경로로 저장
                self.save(final_output_path)
                print(f"✅ 문서 저장 완료: {final_filename}")
                
                # 최종 output_path 설정 (반환값에 사용)
                output_path = final_output_path
            else:
                # product_name이 없으면 기본 파일명 사용
                if timestamp:
                    file_timestamp = timestamp
                else:
                    file_timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
                    
                default_filename = f"사업계획서_{file_timestamp}.{output_format}"
                default_output_path = os.path.join(output_dir, default_filename)
                
                self.save(default_output_path)
                print(f"✅ 문서 저장 완료: {default_filename}")
                
                # 최종 output_path 설정
                output_path = default_output_path
            
            if progress_callback:
                progress_callback("보고서 생성 완료!", 100)

            # 메타데이터의 outputs 배열 업데이트
            output_info = {
                "file_path": output_path,
                "format": self.output_format,
                "created_at": datetime.now().isoformat(),
                "file_size": os.path.getsize(output_path) if os.path.exists(output_path) else 0,
                "sections_count": len(sections),
                "total_subsections": sum(len(subsections) for subsections in sections.values())
            }
            
            if 'outputs' not in metadata:
                metadata['outputs'] = []
            metadata['outputs'].append(output_info)

            # 메타데이터 파일 업데이트
            try:
                with open(metadata_file, 'w', encoding='utf-8') as f:
                    json.dump(metadata, f, ensure_ascii=False, indent=2)
                self.logger.debug(f"메타데이터 업데이트 완료 (outputs 추가)")
            except Exception as e:
                self.logger.error(f"메타데이터 저장 실패: {str(e)}")

        except Exception as e:
            self.logger.error("오류 발생: %s", str(e))
            self.logger.error("문서 생성 실패: %s", str(e))
            traceback.print_exc()
            return {
                "success": False,
                "error": str(e)
            }
            
        finally:
            # 작업 완료 후 통계
            end_time = time.time()
            total_duration = end_time - start_time
            usage_stats = self.usage_tracker.get_usage_stats()

            self.logger.info("==============================\n")
            self.logger.info(f"총 소요 시간: {total_duration:.1f}초")
            self.logger.info(f"AI 호출 횟수: {usage_stats['api_calls']}회")
            self.logger.info(f"총 토큰 사용량: {usage_stats['total_tokens']:,}개")
            self.logger.info(f"- 프롬프트 토큰: {usage_stats['prompt_tokens']:,}개")
            self.logger.info(f"- 응답 토큰: {usage_stats['completion_tokens']:,}개")
            self.logger.info(f"")
            self.logger.info(f"총 비용: USD ${usage_stats['total_cost']:.2f} (KRW ₩ {usage_stats['total_cost_krw']:,.0f} 원)")
            # self.logger.info(f"적용 환율: $1 = ₩{usage_stats['exchange_rate']:,.2f}")
            self.logger.info("==============================\n")

            # output_path가 정의되지 않은 경우 기본값 반환
            if output_path is None:
                return {
                    "success": False,
                    "error": "파일 생성에 실패했습니다."
                }

            return {
                "success": True,
                "file_path": output_path,
                "format": self.output_format,
                "status": "completed",
                "sections": sections,
                "cache_dir": cache_dir,
                "project_hash": self.project_hash,
                "statistics": {
                    "duration": total_duration,
                    "usage": usage_stats
                }
            }
            
    def _validate_chart_data(self, chart_type: str, labels: List, datasets: List) -> bool:
        """차트 데이터 검증"""
        try:
            if not labels or not datasets:
                self.logger.error("레이블 또는 데이터셋이 비어있음")
                return False

            for dataset in datasets:
                if 'data' not in dataset or 'label' not in dataset:
                    self.logger.error("데이터셋 형식 오류")
                    return False

                data = dataset['data']
                
                # 데이터 개수와 레이블 개수 일치 확인
                if len(data) != len(labels):
                    self.logger.error(f"데이터({len(data)})와 레이블({len(labels)}) 개수 불일치")
                    return False

                # 차트 타입별 검증
                if chart_type == 'pie':
                    # 파이 차트는 음수 값 불가
                    if any(not isinstance(x, (int, float)) or x < 0 for x in data):
                        self.logger.error("파이 차트에 부적절한 데이터 포함")
                        return False

                elif chart_type in ['bar', 'line']:
                    # 숫자 데이터 확인
                    if not all(isinstance(x, (int, float)) for x in data):
                        self.logger.error("숫자가 아닌 데이터 포함")
                        return False

                elif chart_type == 'radar':
                    # 레이더 차트는 숫자 데이터만 허용
                    if not all(isinstance(x, (int, float)) for x in data):
                        self.logger.error("레이더 차트에 숫자가 아닌 데이터 포함")
                        return False
                    
                    # 음수 값은 허용하지 않음
                    if any(x < 0 for x in data):
                        self.logger.error("레이더 차트에 음수 데이터 포함")
                        return False

            return True

        except Exception as e:
            self.logger.error("데이터 검증 중 오류 발생: %s", str(e))
            return False

    def _generate_filename(self, section_name: str, subsection_name: str, file_type: str, sequence: str) -> str:
        """새로운 규칙에 따른 파일명 생성
        Args:
            section_name: 섹션명
            subsection_name: 하위섹션명
            file_type: 파일 유형 (content, chart, chartdata, table, tabledata)
            sequence: 시퀀스 번호 (문자열)
        """
        section_order = self.section_order.get(section_name, '00')
        subsection_order = self.subsection_order.get(section_name, {}).get(subsection_name, '00')
        
        safe_section = self._sanitize_filename(section_name)
        safe_subsection = self._sanitize_filename(subsection_name)
        
        return f"{section_order}-{subsection_order}-{safe_section}-{safe_subsection}-{file_type}-{sequence}"

    async def get_diagram_data(self, section_name: str, content: str) -> Optional[Dict]:
        """다이어그램 데이터 생성"""
        try:
            # 다이어그램 기본 스타일 정의
            default_styles = {
                "graph": {
                    "size": "8.5,11!",
                    "ratio": "fill",
                    "fontname": "Pretendard",
                    "fontsize": "14",
                    "rankdir": "TB",
                    "splines": "ortho",
                    "nodesep": "0.8",
                    "ranksep": "1.0",
                    "pad": "0.5"
                },
                "node": {
                    "fontname": "Pretendard",
                    "fontsize": "12",
                    "shape": "box",
                    "style": "rounded,filled",
                    "fillcolor": "#f8f9fa",
                    "color": "#495057",
                    "margin": "0.3,0.2",
                    "height": "0.6"
                },
                "edge": {
                    "fontname": "Pretendard",
                    "fontsize": "10",
                    "color": "#495057",
                    "arrowsize": "0.8",
                    "penwidth": "1.0"
                }
            }

            # 섹션별 특화 스타일 정의
            section_styles = {
                "조직 구성": {
                    "node": {
                        "shape": "box",
                        "style": "rounded,filled",
                        "fillcolor": "#e7f5ff",
                        "color": "#1971c2"
                    },
                    "edge": {
                        "color": "#1971c2",
                        "arrowhead": "normal"
                    }
                },
                "추진일정": {
                    "node": {
                        "shape": "box",
                        "style": "rounded,filled",
                        "fillcolor": "#fff9db",
                        "color": "#e67700"
                    },
                    "edge": {
                        "color": "#e67700",
                        "arrowhead": "vee"
                    }
                },
                "컨소시엄 구성": {
                    "node": {
                        "shape": "box",
                        "style": "rounded,filled",
                        "fillcolor": "#ebfbee",
                        "color": "#2b8a3e"
                    },
                    "edge": {
                        "color": "#2b8a3e",
                        "arrowhead": "diamond"
                    }
                },
                "운영 전략": {
                    "node": {
                        "shape": "box",
                        "style": "rounded,filled",
                        "fillcolor": "#f3f0ff",
                        "color": "#5f3dc4"
                    },
                    "edge": {
                        "color": "#5f3dc4",
                        "arrowhead": "normal"
                    }
                },
                "시스템 구성도": {
                    "node": {
                        "shape": "box",
                        "style": "rounded,filled",
                        "fillcolor": "#e7f5ff",
                        "color": "#1971c2",
                        "margin": "0.3,0.2"
                    },
                    "database": {
                        "shape": "cylinder",
                        "style": "filled",
                        "fillcolor": "#ebfbee",
                        "color": "#2b8a3e"
                    },
                    "external": {
                        "shape": "hexagon",
                        "style": "filled",
                        "fillcolor": "#ffe3e3",
                        "color": "#e03131"
                    },
                    "security": {
                        "shape": "shield",
                        "style": "filled",
                        "fillcolor": "#f3f0ff",
                        "color": "#5f3dc4"
                    },
                    "network": {
                        "shape": "diamond",
                        "style": "filled",
                        "fillcolor": "#e6f4ff",
                        "color": "#1c7ed6"
                    },
                    "edge": {
                        "color": "#868e96",
                        "arrowhead": "normal",
                        "fontname": "Pretendard",
                        "fontsize": "10"
                    }
                },
                "시스템 아키텍처": {
                    "node": {
                        "shape": "box",
                        "style": "rounded,filled",
                        "fillcolor": "#e7f5ff",
                        "color": "#1971c2",
                        "margin": "0.3,0.2"
                    },
                    "database": {
                        "shape": "cylinder",
                        "style": "filled",
                        "fillcolor": "#ebfbee",
                        "color": "#2b8a3e"
                    },
                    "external": {
                        "shape": "hexagon",
                        "style": "filled",
                        "fillcolor": "#ffe3e3",
                        "color": "#e03131"
                    },
                    "security": {
                        "shape": "shield",
                        "style": "filled",
                        "fillcolor": "#f3f0ff",
                        "color": "#5f3dc4"
                    },
                    "network": {
                        "shape": "diamond",
                        "style": "filled",
                        "fillcolor": "#e6f4ff",
                        "color": "#1c7ed6"
                    },
                    "edge": {
                        "color": "#868e96",
                        "arrowhead": "normal",
                        "fontname": "Pretendard",
                        "fontsize": "10"
                    }
                }
            }

            # 현재 섹션에 맞는 스타일 선택
            current_styles = section_styles.get(section_name, {})
            
            # 스타일 문자열 생성
            style_str = """
    // 기본 그래프 스타일
    graph [
        size="8.5,11!",
        ratio=fill,        
        fontname="Pretendard",
        fontsize=14,
        rankdir=TB,
        splines=ortho,
        nodesep=0.8,
        ranksep=1.0,
        pad=0.5
    ];

    // 기본 노드 스타일
    node [
        fontname="Pretendard",
        fontsize=12,
        shape=box,
        style="rounded,filled",
        fillcolor="{node_fillcolor}",
        color="{node_color}",
        margin="0.3,0.2"
    ];

    // 기본 엣지 스타일
    edge [
        fontname="Pretendard",
        fontsize=10,
        color="{edge_color}",
        arrowhead="{edge_arrowhead}",
        arrowsize=0.8,
        penwidth=1.0
    ];
""".format(
    node_fillcolor=current_styles.get('node', {}).get('fillcolor', default_styles['node']['fillcolor']),
    node_color=current_styles.get('node', {}).get('color', default_styles['node']['color']),
    edge_color=current_styles.get('edge', {}).get('color', default_styles['edge']['color']),
    edge_arrowhead=current_styles.get('edge', {}).get('arrowhead', 'normal')
)

            # 시스템 메시지 로드
            system_message = self._load_system_message('diagram')
            if not system_message:
                raise ValueError("다이어그램 생성을 위한 시스템 메시지를 로드할 수 없습니다.")
            
            # 다이어그램 프롬프트 정의
            diagram_prompt = f"""
다음은 사업계획서의 {section_name} 섹션 내용입니다. 이 내용을 가장 효과적으로 시각화할 수 있는 다이어그램의 DOT 언어 소스코드를 지침에 따라 생성해주세요.
특히 다음 사항을 준수해주세요:
- 계층 구조 (사용자/프레젠테이션/비즈니스/데이터)를 명확히 표현
- 각 컴포넌트의 역할에 맞는 shape 사용
- 통신 방식에 따른 적절한 화살표 스타일 적용
- 보안 요소 및 확장성 요소 포함
- DOT 문법을 정확히 준수 (색상 속성은 color="#123456" 형식으로)

# {section_name} 섹션 내용
{content}

# 기본 스타일
{style_str}

# 응답 형식
다음과 같은 형식의 JSON으로 응답해주세요:
{{
    "diagram": {{
        "type": "digraph",
        "dot_source": "완전한 DOT 소스코드를 여기에 작성",
        "options": {{
            "title": "다이어그램 제목",
            "description": "다이어그램 설명"
        }}
    }}
}}
"""

            response = await self.call_ai_api(
                prompt=diagram_prompt,
                require_json=True,
                json_schema={
                    "type": "object",
                    "required": ["diagram"],
                    "properties": {
                        "diagram": {
                            "type": "object",
                            "required": ["type", "dot_source", "options"],
                            "properties": {
                                "type": {"type": "string", "enum": ["digraph", "graph"]},
                                "dot_source": {"type": "string", "minLength": 1},
                                "options": {
                                    "type": "object",
                                    "required": ["title", "description"],
                                    "properties": {
                                        "title": {"type": "string"},
                                        "description": {"type": "string"}
                                    }
                                }
                            }
                        }
                    }
                },
                system_message=system_message
            )

            # 응답 검증 및 반환
            if response and 'diagram' in response:
                # 다이어그램 타입 검증
                diagram_type = response['diagram'].get('type', 'digraph')
                if diagram_type not in ['digraph', 'graph']:
                    self.logger.warning(f"지원하지 않는 다이어그램 타입: {diagram_type}")
                    return None

                return response
            else:
                self.logger.warning("다이어그램 데이터 없음")
                return None

        except Exception as e:
            self.logger.error(f"다이어그램 데이터 생성 실패: {str(e)}")
            traceback.print_exc()
            return None

    def create_diagram(self, data: Dict, output_dir: str, timestamp: str, prefix: str) -> Optional[str]:
        """다이어그램 생성
        
        Args:
            data: 다이어그램 데이터
            output_dir: 출력 디렉토리
            timestamp: 타임스탬프
            prefix: 파일명 접두사
            
        Returns:
            생성된 PNG 파일 경로 또는 None
        """
        try:
            # 데이터 구조 검증 및 로깅
            self.logger.debug(f"다이어그램 데이터: {json.dumps(data, ensure_ascii=False, indent=2)}")
            
            if not isinstance(data, dict):
                self.logger.error(f"잘못된 데이터 형식: {type(data)}")
                return None

            # 데이터 구조 변경: data['diagram'] -> data
            dot_source = data.get('dot_source', '')
            
            if not dot_source:
                self.logger.error("DOT 소스코드가 비어있음")
                return None
            
            # 파일 경로 설정
            abs_output_dir = self._ensure_absolute_path(output_dir)
            os.makedirs(abs_output_dir, exist_ok=True)
            dot_file = os.path.join(abs_output_dir, f'{prefix}_{timestamp}.dot')
            png_file = os.path.join(abs_output_dir, f'{prefix}_{timestamp}.png')
            svg_file = os.path.join(abs_output_dir, f'{prefix}_{timestamp}.svg')            
            
            try:
                # DOT 파일 작성
                with open(dot_file, 'w', encoding='utf-8') as f:
                    f.write(dot_source)
                
                # PNG 생성
                result_png = subprocess.run(
                    ['dot', '-Tpng', dot_file, '-o', png_file],
                    capture_output=True,
                    text=True,
                    check=True
                )
                
                # SVG 생성
                result_svg = subprocess.run(
                    ['dot', '-Tsvg', dot_file, '-o', svg_file],
                    capture_output=True,
                    text=True,
                    check=True
                )
                
                if os.path.exists(png_file) and os.path.exists(svg_file):
                    self.logger.debug(f"다이어그램 생성 완료")
                    return png_file
                else:
                    self.logger.error("다이어그램 파일이 생성되지 않음")
                    return None
                    
            except subprocess.CalledProcessError as e:
                self.logger.error(f"graphviz 실행 실패: {e.stderr}")
                # 오류 발생 시에도 DOT 파일은 보존
                return None

        except Exception as e:
            self.logger.error(f"다이어그램 생성 실패: {str(e)}")
            traceback.print_exc()
            return None

    def _ensure_absolute_path(self, path: str) -> str:
        """상대 경로를 절대 경로로 변환"""
        if os.path.isabs(path):
            return path
        return os.path.abspath(os.path.join(os.getcwd(), path))

    async def _extract_swot_data(self, content: str) -> Dict:
        """SWOT 분석 내용에서 각 요소 추출"""
        try:
            # 시스템 메시지 로드
            system_message = self._load_system_message('swot')
            if not system_message:
                raise ValueError("SWOT 분석을 위한 시스템 메시지를 로드할 수 없습니다.")
            
            # OpenAI API를 사용하여 SWOT 요소 추출
            prompt = f"""
다음 SWOT 분석 내용에서 강점(Strengths), 약점(Weaknesses), 기회(Opportunities), 위협(Threats) 요소를 
짧은 명사형으로 3개씩 추출하고, 이를 조합한 전략(SO/WO/ST/WT)을 40자 이내의 명사형으로 각각 최소 3개 이상 제시해주세요.

분석 내용:
{content}

응답은 반드시 다음 JSON 스키마를 따라야 합니다:
{{
    "strengths": ["강점1", "강점2", "강점3"],
    "weaknesses": ["약점1", "약점2", "약점3"],
    "opportunities": ["기회1", "기회2", "기회3"],
    "threats": ["위협1", "위협2", "위협3"],
    "strategies": {{
        "SO": ["전략1", "전략2", "전략3", "전략4"],
        "WO": ["전략1", "전략2", "전략3", "전략4"],
        "ST": ["전략1", "전략2", "전략3", "전략4"],
        "WT": ["전략1", "전략2", "전략3", "전략4"]
    }}
}}
"""
            response = await self.call_ai_api(
                prompt=prompt,
                require_json=True,
                json_schema={
                    "type": "object",
                    "required": ["strengths", "weaknesses", "opportunities", "threats", "strategies"],
                    "properties": {
                        "strengths": {
                            "type": "array",
                            "minItems": 3,
                            "maxItems": 3,
                            "items": {"type": "string", "minLength": 1}
                        },
                        "weaknesses": {
                            "type": "array",
                            "minItems": 3,
                            "maxItems": 3,
                            "items": {"type": "string", "minLength": 1}
                        },
                        "opportunities": {
                            "type": "array",
                            "minItems": 3,
                            "maxItems": 3,
                            "items": {"type": "string", "minLength": 1}
                        },
                        "threats": {
                            "type": "array",
                            "minItems": 3,
                            "maxItems": 3,
                            "items": {"type": "string", "minLength": 1}
                        },
                        "strategies": {
                            "type": "object",
                            "required": ["SO", "WO", "ST", "WT"],
                            "properties": {
                                "SO": {
                                    "type": "array",
                                    "minItems": 3,
                                    "items": {"type": "string", "maxLength": 40}
                                },
                                "WO": {
                                    "type": "array",
                                    "minItems": 3,
                                    "items": {"type": "string", "maxLength": 40}
                                },
                                "ST": {
                                    "type": "array",
                                    "minItems": 3,
                                    "items": {"type": "string", "maxLength": 40}
                                },
                                "WT": {
                                    "type": "array",
                                    "minItems": 3,
                                    "items": {"type": "string", "maxLength": 40}
                                }
                            }
                        }
                    }
                },
                system_message=system_message
            )
            return response
        except Exception as e:
            self.logger.error(f"SWOT 데이터 추출 실패: {str(e)}")
            return None

    def _create_swot_matrix_dot(self, swot_data: Dict) -> str:
        """SWOT 매트릭스 DOT 소스 생성"""
        try:
            # 각 항목을 글머리 기호로 변환
            def format_items(items):
                if isinstance(items, list):
                    return '<br align="left"/><br align="left"/>'.join(f'- {item.strip()}'.replace('&', '&amp;') for item in items)
                elif isinstance(items, str):
                    return '<br align="left"/><br align="left"/>'.join(f'- {item.strip()}'.replace('&', '&amp;') for item in items.split('\n'))
                return ''

            strengths = format_items(swot_data['strengths'])
            weaknesses = format_items(swot_data['weaknesses'])
            opportunities = format_items(swot_data['opportunities'])
            threats = format_items(swot_data['threats'])

            # 전략 텍스트
            strategies = swot_data.get('strategies', {})
            so_strategy = format_items(strategies.get('SO', ''))
            wo_strategy = format_items(strategies.get('WO', ''))
            st_strategy = format_items(strategies.get('ST', ''))
            wt_strategy = format_items(strategies.get('WT', ''))

            dot_source = f"""
digraph G {{
    // 기본 설정
    graph [
        rankdir=TB,
        splines=none,  // 화살표/선 제거
        nodesep=0.8,   // 노드 간격 증가
        ranksep=0.8,   // 계층 간격 증가
        pad=0.5        // 전체 여백 증가
    ];
    
    // 노드 기본 스타일
    node [
        shape=none,
        fontname="Pretendard",
        fontsize=28,   // 기본 폰트 크기 2배 증가
        margin="0.2,0.2"  // 노드 여백 증가
    ];

    // SWOT 매트릭스
    swot [label=<
        <table border="0" cellspacing="0" cellpadding="20" cellborder="0" align="left">
            <tr>
                <td width="200"></td>
                <td width="350" bgcolor="#ffffff" align="left">
                    <font face="NotoSans" point-size="32"><br align="left"/> <b>강점 (Strengths)</b></font><br align="left"/><br align="left"/>   
                    <font face="NotoSans" point-size="28"><br align="left"/>{strengths}</font><br align="left"/>
                </td>
                <td width="350" bgcolor="#ffffff" align="left">
                    <font face="NotoSans" point-size="32"><br align="left"/> <b>약점 (Weaknesses)</b></font><br align="left"/><br align="left"/>
                    <font face="NotoSans" point-size="28"><br align="left"/>{weaknesses}</font><br align="left"/>
                </td>
            </tr>
            <tr>
                <td bgcolor="#ffffff" align="left" valign="top">
                    <font face="NotoSans" point-size="32"><br align="left"/> <b>기회 (Opportunities)</b></font><br align="left"/><br align="left"/>
                    <font face="NotoSans" point-size="28"><br align="left"/>{opportunities}</font><br align="left"/>
                </td>
                <td bgcolor="#e7f5ff" align="left" valign="top" cellspacing="1">
                    <font face="NotoSans" point-size="32"><br align="left"/> <b>SO 전략</b></font><br align="left"/><br align="left"/>
                    <font face="NotoSans" point-size="28"><br align="left"/>{so_strategy}</font><br align="left"/>
                </td>
                <td bgcolor="#fff3bf" align="left" valign="top" cellspacing="1">
                    <font face="NotoSans" point-size="32"><br align="left"/> <b>WO 전략</b></font><br align="left"/><br align="left"/>
                    <font face="NotoSans" point-size="28"><br align="left"/>{wo_strategy}</font><br align="left"/>
                </td>
            </tr>
            <tr>
                <td bgcolor="#ffffff" align="left" valign="top">
                    <font face="NotoSans" point-size="32"><br align="left"/> <b>위협 (Threats)</b></font><br align="left"/><br align="left"/>
                    <font face="NotoSans" point-size="28"><br align="left"/>{threats}</font><br align="left"/>
                </td>
                <td bgcolor="#ebfbee" align="left" valign="top" cellspacing="1">
                    <font face="NotoSans" point-size="32"><br align="left"/> <b>ST 전략</b></font><br align="left"/><br align="left"/>
                    <font face="NotoSans" point-size="28"><br align="left"/>{st_strategy}</font><br align="left"/>
                </td>
                <td bgcolor="#ffe3e3" align="left" valign="top" cellspacing="1">
                    <font face="NotoSans" point-size="32"><br align="left"/> <b>WT 전략</b></font><br align="left"/><br align="left"/>
                    <font face="NotoSans" point-size="28"><br align="left"/>{wt_strategy}</font><br align="left"/>
                </td>
            </tr>
        </table>
    >];
}}
"""
            return dot_source
        except Exception as e:
            self.logger.error(f"SWOT 매트릭스 DOT 소스 생성 실패: {str(e)}")
            return None

    def create_mermaid_diagram(self, mermaid_code: str, output_path: str, diagram_type: str = 'sequence') -> Optional[str]:
        """Mermaid 다이어그램을 이미지로 생성
        
        Args:
            mermaid_code: Mermaid 문법으로 작성된 다이어그램 코드
            output_path: 출력할 이미지 파일 경로
            diagram_type: 다이어그램 타입 ('sequence', 'flowchart', 'gantt')
            
        Returns:
            생성된 이미지 파일 경로 또는 None
        """
        try:
            # mmdc 명령어 존재 여부 확인 (여러 경로에서 찾기)
            mmdc_path = shutil.which('mmdc')
            
            # 기본 경로에서 찾을 수 없으면 일반적인 npm 글로벌 경로들 확인
            if not mmdc_path:
                possible_paths = [
                    '/home/chaeya/.nvm/versions/node/v18.20.7/bin/mmdc',
                    '/usr/local/bin/mmdc',
                    '/opt/nodejs/bin/mmdc',
                    '/usr/bin/mmdc'
                ]
                
                for path in possible_paths:
                    if os.path.exists(path) and os.path.isfile(path):
                        mmdc_path = path
                        self.logger.debug(f"mmdc 명령어를 다음 경로에서 찾음: {mmdc_path}")
                        break
            
            if not mmdc_path:
                self.logger.error("mermaid-cli가 설치되지 않았습니다. 'npm install -g @mermaid-js/mermaid-cli' 명령으로 설치하세요.")
                return None

            # 출력 디렉토리 생성
            os.makedirs(os.path.dirname(output_path), exist_ok=True)

            # 소스 파일 경로 생성 (.mmd 확장자)
            source_path = output_path.rsplit('.', 1)[0] + '.mmd'

            # 다이어그램 타입에 따른 추가 설정
            config_options = []
            
            # 다이어그램 타입별 특수 처리
            if diagram_type == 'flowchart':
                # 플로우차트는 더 넓은 너비 설정
                config_options.extend(['--width', '1200'])
            elif diagram_type == 'gantt':
                # 간트 차트는 더 넓은 너비와 높이 설정
                config_options.extend(['--width', '1200', '--height', '800'])
            
            # 다이어그램 타입 로깅
            self.logger.debug(f"다이어그램 생성 시작: {diagram_type} 타입")
            self.logger.debug(f"출력 경로: {output_path}")

            # Mermaid 코드를 소스 파일에 저장
            with open(source_path, 'w', encoding='utf-8') as f:
                f.write(mermaid_code)

            # Mermaid CLI 명령 구성
            mmdc_command = [mmdc_path, '-i', source_path, '-o', output_path]
            mmdc_command.extend(config_options)
            
            # Mermaid CLI로 이미지 생성
            result = subprocess.run(
                mmdc_command,
                capture_output=True,
                text=True,
                check=True
            )

            if result.returncode != 0:
                self.logger.error(f"Mermaid 다이어그램 생성 실패: {result.stderr}")
                return None

            if os.path.exists(output_path):
                self.logger.debug(f"Mermaid 다이어그램 생성 완료: {output_path}")
                return output_path
            else:
                self.logger.error("다이어그램 파일이 생성되지 않았습니다")
                return None

        except Exception as e:
            self.logger.error(f"Mermaid 다이어그램 생성 중 오류: {str(e)}")
            traceback.print_exc()
            return None
        
    async def get_sequence_data(self, section_name: str, content: str, section_type: str = "default") -> Optional[Dict]:
        """시퀀스 다이어그램 또는 특수 다이어그램 데이터 생성"""
        try:
            # content가 SectionConfig, 리스트나 딕셔너리인 경우 문자열로 변환
            if isinstance(content, SectionConfig):
                content = content.prompt if content.prompt is not None else ""  # SectionConfig 객체는 prompt 속성을 사용
            elif isinstance(content, (list, dict)):
                content = json.dumps(content, ensure_ascii=False, indent=2)
            
            # content가 여전히 문자열이 아닌 경우 강제 변환
            if not isinstance(content, str):
                content = str(content) if content is not None else ""
                        
            # 시스템 메시지 로드
            system_message = self._load_system_message('sequence')
            if not system_message:
                raise ValueError("다이어그램 생성을 위한 시스템 메시지를 로드할 수 없습니다.")
            
            # 전달받은 section_type 사용 (템플릿에서 가져오는 부분 제거)
            self.logger.debug(f"전달받은 섹션 타입: {section_type}")
   
            # 기본 다이어그램 프롬프트 초기화
            diagram_prompt = f"""
다음은 사업계획서의 {section_name} 섹션 내용입니다. 이 내용에서 프로세스 흐름을 추출하여
Mermaid 시퀀스 다이어그램 코드를 생성해주세요.

내용:
{content}
"""            
            # 섹션 타입에 따라 다른 프롬프트 구성
            if section_type == "organization":
                # 조직도 프롬프트
                diagram_prompt = f"""
다음은 사업계획서의 {section_name} 섹션 내용입니다. 이 내용에서 조직 구조를 추출하여
Mermaid flowchart 다이어그램 코드를 생성해주세요.

내용:
{content}

다음과 같은 형식의 JSON으로 응답해주세요:
{{
    "diagram": {{
        "type": "flowchart",
        "mermaid_code": "flowchart TD\\n    A[\\\"조직1\\\"] ---> B[\\\"조직2\\\"]\\n    A ---> C[\\\"조직3\\\"]\\n    classDef default fill:#FFFFFF,stroke:#000000,stroke-width:0.5px,color:#000000;",
        "options": {{
            "title": "추진 체계",
            "description": "추진 체계 설명"
        }}
    }}
}}

주의사항:
1. 반드시 flowchart TD 형식을 사용하세요 (Top-Down 방향)
2. 조직 간 계층 구조를 명확히 표현하세요
3. 공공문서에 적합한 흑백 스타일을 사용하세요
4. 박스는 흰색 배경, 검은색 테두리로 표현하세요
5. 조직명은 실제 이름을 사용하세요
6. 화살표는 ---> 형식을 사용하여 직각 연결선으로 표현하세요
7. 반드시 다음 스타일 클래스를 포함하세요: classDef default fill:#FFFFFF,stroke:#000000,stroke-width:0.5px,color:#000000;
8. 다음 초기화 설정을 반드시 포함하세요:
%%{{init: {{ 
    'theme': 'base',
    'themeVariables': {{
        'fontFamily': 'Pretendard',
        'fontSize': '16px',
        'primaryColor': '#000000',
        'primaryTextColor': '#000000',
        'primaryBorderColor': '#000000',
        'lineColor': '#000000',
        'secondaryColor': '#FFFFFF',
        'tertiaryColor': '#FFFFFF'
    }}
}}%%
"""
            elif section_type == "timeline":
                # 일정표/간트차트 프롬프트
                diagram_prompt = f"""
다음은 사업계획서의 {section_name} 섹션 내용입니다. 이 내용에서 일정이나 계획 정보를 추출하여 
Mermaid 간트 차트 다이어그램 코드를 생성해주세요.
- 코드 생성시 반드시 최대 3개의 section으로 이내로 작성 하세요.
- 프로젝트 기간의 경우 다음 설정을 반드시 포함하세요:
    dateFormat YYYY-MM-DD
    axisFormat %Y-%m
    tickInterval 1month

내용:
{content}

다음과 같은 형식의 JSON으로 응답해주세요:
{{
    "diagram": {{
        "type": "gantt",
        "mermaid_code": "gantt\\n    dateFormat YYYY-MM-DD\\n    section 개발\\n    요구사항 분석 :a1, 2023-10-30, 7d\\n    마일스톤1 : milestone, 2023-11-05, 0d\\n    설계 단계 :a2, 2023-11-06, 7d",
        "options": {{
            "title": "프로젝트 일정표",
            "description": "프로젝트 일정 설명"
        }}
    }}
}}

주의사항:
1. 간트 차트의 마일스톤은 '마일스톤명 : milestone, 날짜, 0d' 형식으로 작성해야 합니다.
2. 'milestone' 키워드는 태스크 유형으로만 사용하고, 텍스트 앞에 붙이지 마세요.
3. 날짜 형식은 YYYY-MM-DD를 사용하세요.
   - 프로젝트 기간의 경우 다음 설정을 반드시 포함하세요:
    dateFormat YYYY-MM-DD
    axisFormat %Y-%m
    tickInterval 1month
4. 각 태스크에는 고유한 ID와 기간을 지정하세요.
5. 'title' 행을 제외하고 코드를 생성하세요. 제목은 별도로 처리됩니다.
6. 가독성을 위해 최대 8개의 태스크만 포함하세요.
7. 각 태스크명은 15자를 넘지 않도록 간단명료하게 작성하세요.
8. 주요 단계별로 section을 나누어 구성하되, 최대 3개의 section으로 제한하세요.
9. 다음 초기화 설정을 반드시 포함하세요:
%%{{init: {{ 
    'theme': 'base',
    'themeVariables': {{
        'fontFamily': 'Pretendard',
        'fontSize': '16px',
        'messageFontFamily': 'Pretendard',
        'noteFontFamily': 'Pretendard',
        'actorFontFamily': 'Pretendard',
        'actorBackground': '#FFFFFF',
        'actorBorder': '#000000',
        'activationBackground': '#F5F5F5',
        'activationBorderColor': '#000000',
        'messageFontColor': '#000000',
        'noteBkgColor': '#FFFFFF',
        'noteBorderColor': '#000000',
        'primaryColor': '#000000',
        'primaryBorderColor': '#000000',    
        'primaryTextColor': '#000000',       
        'secondaryColor': '#FFFFFF',         
        'tertiaryColor': '#FFFFFF',         
        'taskBorderColor': '#000000',       
        'taskBkgColor': '#FFFFFF',          
        'taskTextColor': '#000000',         
        'sectionBkgColor': '#FFFFFF',       
        'sectionBkgColor2': '#FFFFFF',     
        'gridColor': '#000000',            
        'doneTaskBkgColor': '#FFFFFF',     
        'doneTaskBorderColor': '#000000',   
        'activeTaskBorderColor': '#000000', 
        'activeTaskBkgColor': '#FFFFFF',    
        'todayLineColor': '#000000'        
    }}
}}%%
"""
            else:
                # 기본 시퀀스 다이어그램 프롬프트
                diagram_prompt = f"""
다음은 사업계획서의 {section_name} 섹션 내용입니다. 이 내용에서 프로세스나 단계별 흐름을 추출하여 
Mermaid 시퀀스 다이어그램 코드를 생성해주세요.

내용:
{content}

다음과 같은 형식의 JSON으로 응답해주세요:
{{
    "diagram": {{
        "type": "sequence",
        "mermaid_code": "sequenceDiagram\\n    participant A\\n    participant B\\n    A->>B: Message\\n",
        "options": {{
            "title": "프로세스 흐름도",
            "description": "프로세스 설명"
        }}
    }}
}}

주의사항:
1. 시퀀스 다이어그램은 시간 순서나 단계별 흐름을 보여주어야 합니다
2. 참여자(participant)는 실제 주체나 시스템을 반영해야 합니다
3. 화살표는 상호작용의 성격에 맞게 사용하세요
4. 필요한 경우 activate/deactivate로 활성화 상태를 표시하세요
5. loop, alt, opt 등의 제어 구조를 적절히 활용하세요
6. 다음 초기화 설정을 반드시 포함하세요:
%%{{init: {{ 
    'theme': 'base',
    'themeVariables': {{
        'fontFamily': 'Pretendard',
        'fontSize': '16px',
        'messageFontFamily': 'Pretendard',
        'noteFontFamily': 'Pretendard',
        'actorFontFamily': 'Pretendard',
        'actorBackground': '#FFFFFF',
        'actorBorder': '#000000',
        'activationBackground': '#F5F5F5',
        'activationBorderColor': '#000000',
        'messageFontColor': '#000000',
        'noteBkgColor': '#FFFFFF',
        'noteBorderColor': '#000000',
        'primaryColor': '#FFFFFF',
        'primaryBorderColor': '#000000',    
        'primaryTextColor': '#000000',       
        'secondaryColor': '#FFFFFF',         
        'tertiaryColor': '#FFFFFF',         
        'taskBorderColor': '#000000',       
        'taskBkgColor': '#FFFFFF',          
        'taskTextColor': '#000000',         
        'sectionBkgColor': '#FFFFFF',       
        'sectionBkgColor2': '#FFFFFF',     
        'gridColor': '#000000',            
        'doneTaskBkgColor': '#FFFFFF',     
        'doneTaskBorderColor': '#000000',   
        'activeTaskBorderColor': '#000000', 
        'activeTaskBkgColor': '#FFFFFF',    
        'todayLineColor': '#000000'  
    }}
}}%%
"""

            # OpenAI API 호출
            response = await self.call_ai_api(
                prompt=diagram_prompt,
                require_json=True,
                json_schema={
                    "type": "object",
                    "required": ["diagram"],
                    "properties": {
                        "diagram": {
                            "type": "object",
                            "required": ["type", "mermaid_code", "options"],
                            "properties": {
                                "type": {"type": "string", "enum": ["sequence", "flowchart", "gantt"]},
                                "mermaid_code": {"type": "string", "minLength": 1},
                                "options": {
                                    "type": "object",
                                    "required": ["title", "description"],
                                    "properties": {
                                        "title": {"type": "string"},
                                        "description": {"type": "string"}
                                    }
                                }
                            }
                        }
                    }
                },
                system_message=system_message
            )

            if response and 'diagram' in response:
                # 다이어그램 타입 로깅
                self.logger.debug(f"생성된 다이어그램 타입: {response['diagram'].get('type', 'unknown')}")
                return response
            else:
                self.logger.warning("다이어그램 데이터 없음")
                return None

        except Exception as e:
            self.logger.error(f"다이어그램 데이터 생성 실패: {str(e)}")
            traceback.print_exc()
            return None

    def _initialize_signal_handlers(self):
        """시그널 핸들러 초기화"""
        import signal
        
        def signal_handler(signum, frame):
            """시그널 처리 함수"""
            if signum == signal.SIGINT:
                print("\n\n⚠️ Ctrl+C가 감지되었습니다. 작업을 안전하게 중단합니다...")
                self.logger.warning("사용자가 Ctrl+C로 작업 중단을 요청했습니다")
                
                # 현재까지의 통계 출력
                end_time = time.time()
                total_duration = end_time - self._start_time
                usage_stats = self.usage_tracker.get_usage_stats()
                
                self.logger.info("\n=== 작업 중단 시점의 통계 ===")
                progress = (self.completed_sections / self.total_sections) * 100 if self.total_sections > 0 else 0
                self.logger.info(f"진행률: {progress:.1f}%")
                self.logger.info(f"소요 시간: {total_duration:.1f}초")
                self.logger.info(f"AI 호출 횟수: {usage_stats['api_calls']}회")
                self.logger.info(f"총 토큰 사용량: {usage_stats['total_tokens']:,}개")
                self.logger.info(f"총 비용: USD ${usage_stats['total_cost']:.2f}")
                self.logger.info(f"       KRW ₩{usage_stats['total_cost_krw']:,.0f}")
                self.logger.info("===============================")
                
                # 프로그램 종료
                sys.exit(1)
        
        # SIGINT(Ctrl+C) 시그널 핸들러 등록
        signal.signal(signal.SIGINT, signal_handler)

class UsageTracker:
    """사용량 추적 클래스"""
    def __init__(self, logger=None):
        self.logger = logger or logging.getLogger("report_generator_default")
        self.usage_data = {
            'total_tokens': 0,
            'prompt_tokens': 0,
            'completion_tokens': 0,
            'api_calls': 0,
            'total_cost': 0.0,
            'usage_by_model': {}  # 모델별 사용량 추적 추가
        }
        
        # price.json 파일에서 가격 정보 로드
        self.price_data = self._load_price_data()        
        # 환율 정보 초기화
        self.exchange_rate = self._get_exchange_rate()
        
    def _load_price_data(self) -> Dict:
        """가격 정보 로드"""
        try:
            
            # 설정 파일에서 AIRUN_PATH 가져오기
            config = self._load_config()
            airun_path = config.get('AIRUN_PATH')
            if airun_path:
                price_file = os.path.join(airun_path, 'pages', 'price.json')
            else:
                # 현재 실행 파일 위치 기준으로 설정
                current_dir = os.path.dirname(os.path.abspath(__file__))
                price_file = os.path.join(current_dir, '..', '..', 'pages', 'price.json')
                
            self.logger.debug("가격 정보 파일 경로: %s", price_file)
                
            
            with open(price_file, 'r', encoding='utf-8') as f:
                return json.load(f)
        except Exception as e:
            self.logger.error("가격 정보 로드 실패: %s", str(e))
            # 기본 가격 정보 반환 - gpt-4와 gpt-4-mini만 포함
            return {
                'openai': {
                    'gpt-4': {
                        'prompt': 0.03,
                        'completion': 0.06
                    },
                    'gpt-4-mini': {
                        'prompt': 0.01,
                        'completion': 0.03
                    }
                }
            }           
            
    def _load_config(self) -> dict:
        """~/.airun/airun.conf 파일에서 설정을 읽어옵니다."""
        config = {}
        config_path = os.path.expanduser("~/.airun/airun.conf")
        
        try:
            with open(config_path, 'r', encoding='utf-8') as f:
                for line in f:
                    line = line.strip()
                    if line and line.startswith('export '):
                        # 'export KEY="VALUE"' 형식 파싱
                        line = line.replace('export ', '')
                        key, value = line.split('=', 1)
                        key = key.strip()
                        value = value.strip().strip('"').strip("'")
                        config[key] = value
            return config
        except Exception as e:
            self.logger.error(f"설정 파일 로드 실패: {str(e)}")
            return {}  # 오류 발생 시 빈 딕셔너리 반환
                    
    def _get_exchange_rate(self) -> float:
        """환율 정보를 가져옵니다. (USD to KRW)"""
        try:
            # 1. 먼저 설정 파일에서 환율 확인
            config = self._load_config()
            if 'EXCHANGE_RATE_USD_KRW' in config:
                return float(config['EXCHANGE_RATE_USD_KRW'])
            
            # 2. 환율 API 호출 시도
            try:
                import requests
                url = "https://api.exchangerate-api.com/v4/latest/USD"
                response = requests.get(url, timeout=5)
                if response.status_code == 200:
                    data = response.json()
                    if 'rates' in data and 'KRW' in data['rates']:
                        return float(data['rates']['KRW'])
            except:
                pass
                
            # 3. 기본값 반환 (실패 시)
            return 1300.0
            
        except Exception as e:
            self.logger.error(f"환율 정보 조회 실패: {str(e)}")
            return 1300.0  # 기본값
                    
    def calculate_cost(self, provider: str, model: str, prompt_tokens: int, completion_tokens: int) -> float:
        """토큰 사용량에 따른 비용 계산"""
        try:
            if not self.price_data:
                self.logger.warning("가격 정보가 로드되지 않았습니다.")
                return 0.0

            # ollama 제공자는 무료로 처리
            if provider.lower() == 'ollama':
                # self.logger.debug(f"Ollama 모델 '{model}'은(는) 무료.")
                return 0.0

            if provider not in self.price_data:
                self.logger.warning(f"제공자 '{provider}'에 대한 가격 정보가 없습니다.")
                return 0.0

            if model not in self.price_data[provider]:
                self.logger.warning(f"모델 '{model}'에 대한 가격 정보가 없습니다.")
                return 0.0

            price_info = self.price_data[provider][model]
            prompt_cost = (prompt_tokens / 1000.0) * price_info['prompt']
            completion_cost = (completion_tokens / 1000.0) * price_info['completion']
            
            total_cost = prompt_cost + completion_cost
            # self.logger.debug(f"비용 계산: {provider}/{model} -> ${total_cost:.4f} (prompt: ${prompt_cost:.4f}, completion: ${completion_cost:.4f})")
            
            return total_cost
            
        except Exception as e:
            self.logger.error(f"비용 계산 중 오류 발생: {str(e)}")
            return 0.0


    def track_usage(self, provider: str, model: str, prompt_tokens: int, completion_tokens: int):
        """사용량 추적"""
        try:
            # 기본 통계 업데이트
            self.usage_data['prompt_tokens'] += prompt_tokens
            self.usage_data['completion_tokens'] += completion_tokens
            self.usage_data['total_tokens'] += prompt_tokens + completion_tokens
            self.usage_data['api_calls'] += 1
            
            # 모델별 사용량 추적
            model_key = f"{provider}/{model}"
            if model_key not in self.usage_data['usage_by_model']:
                self.usage_data['usage_by_model'][model_key] = {
                    'prompt_tokens': 0,
                    'completion_tokens': 0,
                    'total_tokens': 0,
                    'api_calls': 0,
                    'cost': 0.0
                }
            
            model_stats = self.usage_data['usage_by_model'][model_key]
            model_stats['prompt_tokens'] += prompt_tokens
            model_stats['completion_tokens'] += completion_tokens
            model_stats['total_tokens'] += prompt_tokens + completion_tokens
            model_stats['api_calls'] += 1
            
            # 비용 계산
            cost = self.calculate_cost(provider, model, prompt_tokens, completion_tokens)
            self.usage_data['total_cost'] += cost
            model_stats['cost'] += cost
            
            # 로그에 사용량 기록
            # self.logger.debug(f"API 호출: {provider}/{model}")
            # self.logger.debug(f"토큰 사용: prompt={prompt_tokens}, completion={completion_tokens}")
            # self.logger.debug(f"비용: ${cost:.4f}")

        except Exception as e:
            self.logger.error(f"사용량 추적 중 오류 발생: {str(e)}")
            traceback.print_exc()

    def get_usage_stats(self) -> Dict:
        """사용량 통계 반환"""
        try:
            stats = self.usage_data.copy()
            
            # 원화 금액 계산
            krw_cost = stats['total_cost'] * self.exchange_rate
            stats['total_cost_krw'] = krw_cost
            stats['exchange_rate'] = self.exchange_rate
            
            # 모델별 통계에 원화 금액 추가
            for model_key, model_stats in stats['usage_by_model'].items():
                model_stats['cost_krw'] = model_stats['cost'] * self.exchange_rate
            
            return stats
        except Exception as e:
            self.logger.error(f"통계 데이터 생성 중 오류 발생: {str(e)}")
            return {
                'total_tokens': 0,
                'prompt_tokens': 0,
                'completion_tokens': 0,
                'api_calls': 0,
                'total_cost': 0.0,
                'total_cost_krw': 0.0,
                'exchange_rate': self.exchange_rate,
                'usage_by_model': {}
            }

    def print_current_stats(self, prefix: str = "현재까지"):
        """현재까지의 통계 출력"""
        try:
            stats = self.get_usage_stats()
            
            self.logger.info("\n=== %s 사용량 통계 ===", prefix)
            self.logger.info("AI 호출 횟수: %d회", stats['api_calls'])
            self.logger.info("총 토큰 사용량: %d개", stats['total_tokens'])
            self.logger.info("- 프롬프트 토큰: %d개", stats['prompt_tokens'])
            self.logger.info("- 응답 토큰: %d개", stats['completion_tokens'])
            self.logger.info("")
            
            # 모델별 통계 출력
            if stats['usage_by_model']:
                self.logger.info("모델별 사용량:")
                for model_key, model_stats in sorted(stats['usage_by_model'].items()):
                    self.logger.info(f"  {model_key}:")
                    self.logger.info(f"    호출 횟수: {model_stats['api_calls']}회")
                    self.logger.info(f"    총 토큰: {model_stats['total_tokens']}개")
                    self.logger.info(f"    비용: ${model_stats['cost']:.4f} (₩{model_stats['cost_krw']:,.0f})")
            
            self.logger.info("")
            self.logger.info("총 비용: $%.4f (₩%d)", stats['total_cost'], stats['total_cost_krw'])
            self.logger.info("적용 환율: $1 = ₩%.2f", stats['exchange_rate'])
            self.logger.info("========================\n")
            sys.stdout.flush()
        except Exception as e:
            self.logger.error(f"통계 출력 중 오류 발생: {str(e)}")
            traceback.print_exc()






