import TaskStatus from '../models/TaskStatus.js';
import ExecutionManager from './ExecutionManager.js';
import { oraStart, oraSucceed, oraFail } from '../../../oraManager.js';
import chalk from 'chalk';

class TaskManager {
    constructor() {
        this.steps = new Map();
        this.overallStatus = TaskStatus.NOT_STARTED;
        this.startedAt = null;
        this.completedAt = null;
        this.apiCalls = new Map();
        this.modelCosts = new Map();
        this.tokenCounts = new Map();
        this.executionManager = new ExecutionManager();
        this.taskContext = null;  // 전체 작업 컨텍스트 저장
    }

    async addSteps(steps, taskAnalysis = null) {
        // 작업 계획만 표시하고 원본 응답은 표시하지 않음
        this.executionManager.progress.showTaskPlan(taskAnalysis, steps);
        
        // 구조화된 작업 컨텍스트 초기화
        this.taskContext = {
            originalRequest: steps[0]?.context?.originalRequest || '',
            paths: steps[0]?.context?.paths || { inputs: [], output: '' },
            taskSteps: steps[0]?.context?.taskSteps?.map(s => ({
                ...s,
                status: 'pending',
                result: null,
                startTime: null,
                endTime: null,
                error: null
            })) || []
        };
        
        // 단계 정보는 내부적으로만 저장
        for (const step of steps) {
            // 각 단계에 taskContext 연결
            step.context = {
                ...step.context,
                ...this.taskContext
            };
            this.steps.set(step.id, step);
        }
    }

    async getNextExecutableStep() {
        for (const step of this.steps.values()) {
            if (step.status === TaskStatus.NOT_STARTED && this.areDependenciesMet(step)) {
                return step;
            }
        }
        return null;
    }

    areDependenciesMet(step) {
        return step.dependencies.every(depId => {
            const depStep = this.steps.get(depId);
            return depStep && depStep.status === TaskStatus.COMPLETED;
        });
    }

    async executeTask() {
        this.startedAt = new Date();
        this.overallStatus = TaskStatus.IN_PROGRESS;
        
        try {
            this.executionManager.progress.showExecutionStart();
            
            let lastResult = null;
            let executedSteps = 0;
            const totalSteps = this.steps.size;

            while (executedSteps < totalSteps) {
                const step = await this.getNextExecutableStep();
                if (!step) {
                    // 실행 가능한 단계가 없지만 아직 완료되지 않은 단계가 있는 경우
                    const remainingSteps = Array.from(this.steps.values())
                        .filter(s => s.status !== TaskStatus.COMPLETED)
                        .map(s => s.id);
                    
                    if (remainingSteps.length > 0) {
                        throw new Error(`다음 단계들을 실행할 수 없습니다: ${remainingSteps.join(', ')}`);
                    }
                    break;
                }

                step.status = TaskStatus.IN_PROGRESS;
                
                // taskSteps 상태 업데이트
                const taskStep = this.taskContext.taskSteps.find(s => s.id === step.id);
                if (taskStep) {
                    taskStep.status = 'running';
                    taskStep.startTime = new Date().toISOString();
                }

                try {
                    // 이전 단계의 결과를 컨텍스트에 추가
                    step.context = {
                        ...step.context,
                        lastExecutionResult: lastResult?.output || '',
                        lastExecutionPath: lastResult?.path || null,
                        lastExecutionCode: lastResult?.script || null,
                        previousSteps: Object.fromEntries(
                            Array.from(this.steps.entries())
                                .filter(([_, s]) => s.status === TaskStatus.COMPLETED)
                                .map(([id, s]) => [id, s.result])
                        )
                    };

                    const result = await this.executionManager.executeStep(step);
                    step.result = result;
                    step.status = TaskStatus.COMPLETED;
                    step.completedAt = new Date();
                    lastResult = result;
                    executedSteps++;
                    
                    // taskSteps 상태 업데이트
                    if (taskStep) {
                        taskStep.status = 'completed';
                        taskStep.endTime = new Date().toISOString();
                        taskStep.result = {
                            output: result.result?.output || '',
                            files: result.result?.files || [],
                            data: result.result?.data || null
                        };
                    }
                } catch (error) {
                    step.status = TaskStatus.FAILED;
                    step.result = error.message;
                    this.overallStatus = TaskStatus.FAILED;
                    
                    // taskSteps 상태 업데이트
                    if (taskStep) {
                        taskStep.status = 'failed';
                        taskStep.endTime = new Date().toISOString();
                        taskStep.error = error.message;
                    }
                    
                    throw error;
                }
            }

            this.completedAt = new Date();
            this.overallStatus = TaskStatus.COMPLETED;
            
            // 실행 결과 표시
            this.executionManager.progress.showExecutionResult(this.executionManager.executionResults);
        } catch (error) {
            this.overallStatus = TaskStatus.FAILED;
            // 실행 결과 표시 (실패한 경우에도)
            this.executionManager.progress.showExecutionResult(this.executionManager.executionResults);
            throw error;
        }
    }

    printSummary() {
        console.log('\n=== 작업 실행 결과 ===');
        
        // taskSteps 기반으로 결과 출력
        if (this.taskContext?.taskSteps) {
            for (const step of this.taskContext.taskSteps) {
                const status = {
                    'completed': chalk.green('✓'),
                    'failed': chalk.red('✗'),
                    'running': chalk.blue('⋯'),
                    'pending': chalk.gray('○')
                }[step.status];

                console.log(`${status} ${step.number}. [${step.id}] ${step.description}`);
                
                if (step.error) {
                    console.log(chalk.red(`   ⚠️ 오류: ${step.error}`));
                }
                
                if (step.result) {
                    const resultStr = typeof step.result === 'string' ? 
                        step.result : JSON.stringify(step.result, null, 2);
                    console.log(chalk.gray(`   📝 결과: ${resultStr}`));
                }
                
                if (step.startTime && step.endTime) {
                    const duration = (new Date(step.endTime) - new Date(step.startTime)) / 1000;
                    console.log(chalk.gray(`   ⏱️ 실행 시간: ${duration.toFixed(2)}초`));
                }
            }
        }

        if (this.startedAt && this.completedAt) {
            const duration = (this.completedAt - this.startedAt) / 1000;
            console.log(`\n총 실행 시간: ${duration.toFixed(2)}초`);
        }
    }
}

export default TaskManager; 