/**
 * @swagger
 * tags:
 *   - name: Chat
 *     description: 채팅 관련 API
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     ChatRequest:
 *       type: object
 *       required:
 *         - prompt
 *       properties:
 *         prompt:
 *           type: string
 *           description: 사용자의 입력 텍스트
 *         options:
 *           type: object
 *           properties:
 *             temperature:
 *               type: number
 *               description: 응답의 창의성 정도 (0.0 ~ 1.0)
 *             sessionId:
 *               type: string
 *               description: |
 *                 세션 ID - 대화 문맥을 유지하기 위한 고유 식별자입니다.
 *                 - 제공하지 않으면 서버에서 자동으로 생성합니다.
 *                 - 연속된 대화에서 같은 세션 ID를 사용하면 문맥이 유지됩니다.
 *                 - 이전 대화 내용을 참조할 수 있게 합니다.
 *             username:
 *               type: string
 *               description: |
 *                 사용자명 (권장 방식).
 *                 - **명시적으로 제공할 경우**: 해당 사용자의 문서만 검색합니다.
 *                 - **제공하지 않을 경우**: 로그인한 API 요청자의 사용자명이 자동으로 사용됩니다.
 *                 - **`all`로 지정할 경우**: 모든 공용 문서를 대상으로 검색합니다.
 *             userId:
 *               type: integer
 *               description: |
 *                 사용자 ID (정수형, 하위 호환성용).
 *                 - username을 제공하지 않은 경우에만 사용됩니다.
 *                 - 새로운 개발에서는 username 사용을 권장합니다.
 *             rag:
 *               type: boolean
 *               description: RAG 기능 사용 여부
 *             web:
 *               type: boolean
 *               description: 웹 검색 기능 사용 여부
 *             image:
 *               type: string
 *               description: |
 *                 이미지 데이터 또는 경로.
 *                 - base64 인코딩된 이미지 데이터 (data:image/... 형식)
 *                 - 이미지 파일 경로 (절대 경로 또는 상대 경로)
 *             settings:
 *               type: object
 *               properties:
 *                 USE_LLM:
 *                   type: string
 *                   enum: [openai, anthropic, gemini, ollama]
 *                   description: 사용할 LLM 프로바이더
 *     ChatResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *         data:
 *           type: object
 *           properties:
 *             id:
 *               type: string
 *             response:
 *               type: string
 *             timestamp:
 *               type: string
 *               format: date-time
 */

/**
 * @swagger
 * /api/v1/chat:
 *   post:
 *     summary: 채팅 메시지 전송
 *     description: |
 *       AI와의 대화를 위한 메시지를 전송합니다.
 *       텍스트 메시지와 함께 이미지도 전송할 수 있으며, 이미지에 대한 설명이나 분석을 요청할 수 있습니다.
 *       이미지는 base64 인코딩된 데이터나 파일 경로로 전달할 수 있습니다.
 *     tags: [Chat]
 *     security:
 *       - apiKeyAuth: []
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ChatRequest'
 *           examples:
 *             text-only:
 *               summary: 텍스트만 있는 요청
 *               value:
 *                 prompt: "안녕하세요"
 *                 options:
 *                   temperature: 0.7
 *                   rag: false
 *                   web: false
 *                   username: "admin"
 *             with-image-path:
 *               summary: 이미지 경로가 포함된 요청
 *               value:
 *                 prompt: "이 이미지에 대해 설명해주세요"
 *                 options:
 *                   temperature: 0.7
 *                   image: "./images/example.jpg"
 *                   rag: false
 *                   web: false
 *                   username: "admin"
 *             with-image-base64:
 *               summary: Base64 이미지가 포함된 요청
 *               value:
 *                 prompt: "이 이미지에 대해 설명해주세요"
 *                 options:
 *                   temperature: 0.7
 *                   image: "data:image/jpeg;base64,/9j/4AAQSkZJRg..."
 *                   rag: false
 *                   web: false
 *                   username: "admin"
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 jobId:
 *                   type: string
 *                   description: 작업 ID (상태 확인에 사용)
 *                 status:
 *                   type: string
 *                   enum: [queued, processing, completed]
 *                   description: 작업 상태
 *                 message:
 *                   type: string
 *                   description: 상태 메시지
 *             examples:
 *               text-response:
 *                 summary: 작업 큐 응답
 *                 value:
 *                   success: true
 *                   jobId: "12345"
 *                   status: "queued"
 *                   message: "작업이 큐에 추가되었습니다."
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       400:
 *         description: |
 *           잘못된 요청
 *           - 이미지 파일을 찾을 수 없는 경우
 *           - 지원하지 않는 이미지 형식
 *           - 파일 읽기 권한이 없는 경우
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/chat/status/{jobId}:
 *   get:
 *     operationId: getChatStatus
 *     summary: 채팅 작업 상태 확인
 *     description: 채팅 요청의 처리 상태를 확인합니다.
 *     tags: [Chat]
 *     parameters:
 *       - in: path
 *         name: jobId
 *         required: true
 *         schema:
 *           type: string
 *         description: 작업 ID
 *     responses:
 *       200:
 *         description: 작업 상태 정보
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     jobId:
 *                       type: string
 *                     state:
 *                       type: string
 *                       enum: [completed, failed, active, waiting, delayed]
 *                     message:
 *                       type: string
 *                     timestamp:
 *                       type: string
 *                       format: date-time
 */

/**
 * @swagger
 * /api/v1/chat/sync:
 *   post:
 *     summary: 채팅 메시지 전송 (동기식)
 *     description: |
 *       AI와의 대화를 위한 메시지를 전송하고 응답을 즉시 받습니다.
 *       텍스트 메시지와 함께 이미지도 전송할 수 있으며, 이미지에 대한 설명이나 분석을 요청할 수 있습니다.
 *       이미지는 base64 인코딩된 데이터나 파일 경로로 전달할 수 있습니다.
 *       
 *       세션 ID를 제공하면 이전 대화 컨텍스트가 유지됩니다:
 *       - 첫 요청 시 sessionId를 제공하지 않으면 자동으로 생성되며 응답에 포함됩니다.
 *       - 후속 요청에서 같은 sessionId를 사용하면 이전 대화 내용이 참조됩니다.
 *       - 서로 다른 대화에는 다른 sessionId를 사용해야 합니다.
 *     tags: [Chat]
 *     security:
 *       - apiKeyAuth: []
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ChatRequest'
 *           examples:
 *             text-only:
 *               summary: 텍스트만 있는 요청
 *               value:
 *                 prompt: "안녕하세요"
 *                 options:
 *                   temperature: 0.7
 *                   rag: false
 *                   web: false
 *                   username: "admin"
 *             with-session:
 *               summary: 세션 ID를 포함한 요청
 *               value:
 *                 prompt: "안녕하세요"
 *                 options:
 *                   temperature: 0.7
 *                   sessionId: "user-session-123456"
 *                   rag: false
 *                   web: false
 *                   username: "admin"
 *             with-rag-username:
 *               summary: 특정 사용자에 대한 RAG 검색 요청
 *               value:
 *                 prompt: "프로젝트 A의 진행 상황 알려줘"
 *                 options:
 *                   rag: true
 *                   username: "project_a_user"
 *             with-image-path:
 *               summary: 이미지 경로가 포함된 요청
 *               value:
 *                 prompt: "이 이미지에 대해 설명해주세요"
 *                 options:
 *                   temperature: 0.7
 *                   image: "./images/example.jpg"
 *                   rag: false
 *                   web: false
 *                   username: "admin"
 *             with-image-base64:
 *               summary: Base64 이미지가 포함된 요청
 *               value:
 *                 prompt: "이 이미지에 대해 설명해주세요"
 *                 options:
 *                   temperature: 0.7
 *                   image: "data:image/jpeg;base64,/9j/4AAQSkZJRg..."
 *                   rag: false
 *                   web: false
 *                   username: "admin"
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     response:
 *                       type: string
 *                       description: AI의 응답 텍스트 (이미지 설명 포함)
 *                     usage:
 *                       type: object
 *                       nullable: true
 *                       description: 토큰 사용량 정보
 *                     sessionId:
 *                       type: string
 *                       description: 대화 세션 ID (후속 요청에 사용)
 *                     isNewSession:
 *                       type: boolean
 *                       description: 새로 생성된 세션인지 여부
 *                     timestamp:
 *                       type: string
 *                       format: date-time
 *             examples:
 *               text-response:
 *                 summary: 텍스트 응답
 *                 value:
 *                   success: true
 *                   data:
 *                     response: "안녕하세요! 무엇을 도와드릴까요?"
 *                     usage: null
 *                     sessionId: "session-1234567890"
 *                     isNewSession: true
 *                     timestamp: "2025-04-22T02:45:12.345Z"
 *               with-session:
 *                 summary: 기존 세션 응답
 *                 value:
 *                   success: true
 *                   data:
 *                     response: "안녕하세요! 이전 대화를 이어서 진행할 수 있어요."
 *                     usage: {
 *                       prompt_tokens: 150,
 *                       completion_tokens: 50,
 *                       total_tokens: 200
 *                     }
 *                     sessionId: "user-session-123456"
 *                     isNewSession: false
 *                     timestamp: "2025-04-22T02:46:15.345Z"
 *               image-response:
 *                 summary: 이미지 설명 응답
 *                 value:
 *                   success: true
 *                   data:
 *                     response: "이 이미지는 푸른 하늘 아래 있는 산과 호수를 보여줍니다..."
 *                     usage: {
 *                       prompt_tokens: 100,
 *                       completion_tokens: 150,
 *                       total_tokens: 250
 *                     }
 *                     sessionId: "image-session-5678"
 *                     isNewSession: true
 *                     timestamp: "2025-04-22T02:45:12.345Z"
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       400:
 *         description: |
 *           잘못된 요청
 *           - 이미지 파일을 찾을 수 없는 경우
 *           - 지원하지 않는 이미지 형식
 *           - 파일 읽기 권한이 없는 경우
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

export const chatSchemas = {
    ChatRequest: {
        type: 'object',
        required: ['prompt'],
        properties: {
            prompt: {
                type: 'string',
                description: '사용자의 입력 텍스트'
            },
            options: {
                type: 'object',
                properties: {
                    temperature: {
                        type: 'number',
                        description: '응답의 창의성 정도 (0.0 ~ 1.0)'
                    },
                    sessionId: {
                        type: 'string',
                        description: '세션 ID - 대화 문맥 유지를 위해 사용. 제공하지 않으면 자동 생성됨'
                    },
                    username: {
                        type: 'string',
                        description: "사용자명 (권장 방식). RAG 검색을 수행할 사용자명. 제공하지 않으면 로그인한 사용자의 사용자명이 자동으로 사용됩니다."
                    },
                    userId: {
                        type: 'integer',
                        description: "사용자 ID (정수형, 하위 호환성용). username을 제공하지 않은 경우에만 사용됩니다."
                    },
                    rag: {
                        type: 'boolean',
                        description: 'RAG 기능 사용 여부'
                    },
                    web: {
                        type: 'boolean',
                        description: '웹 검색 기능 사용 여부'
                    },
                    image: {
                        type: 'string',
                        description: '이미지 데이터(base64) 또는 이미지 파일 경로'
                    },
                    settings: {
                        type: 'object',
                        properties: {
                            USE_LLM: {
                                type: 'string',
                                enum: ['openai', 'anthropic', 'gemini', 'ollama'],
                                description: '사용할 LLM 프로바이더'
                            }
                        }
                    }
                }
            }
        }
    }
}; 