// 환자 데이터 자동 처리 시스템
// 특정 파일명 패턴의 엑셀 파일을 자동으로 PostgreSQL 테이블로 변환

import XLSX from 'xlsx';

/**
 * 환자 데이터 파일 패턴 정의
 * 다음 파일명들이 업로드되면 자동으로 환자 데이터 테이블로 변환:
 * - patient_data.xlsx
 * - patients.xlsx  
 * - health_checkup.xlsx
 * - 건강검진데이터.xlsx
 * - 환자데이터.xlsx
 */
const PATIENT_DATA_FILE_PATTERNS = [
    /^patient_?data\.xlsx?$/i,
    /^patients\.xlsx?$/i,
    /^health_?checkup\.xlsx?$/i,
    /^건강검진데이터\.xlsx?$/i,
    /^환자데이터\.xlsx?$/i,
    /^sample_patients_\d+\.xlsx?$/i, // sample_patients_50.xlsx 등
    /^jp-data\.xlsx?$/i // 기존 일본 데이터 파일명
];

/**
 * 파일명이 환자 데이터 패턴과 일치하는지 확인
 * @param {string} filename - 체크할 파일명
 * @returns {boolean} 환자 데이터 파일 여부
 */
function isPatientDataFile(filename) {
    return PATIENT_DATA_FILE_PATTERNS.some(pattern => pattern.test(filename));
}

/**
 * 요단백 코드 변환 함수
 * @param {string} value - 원본 값
 * @returns {string} 변환된 값
 */
function convertUrineProtein(value) {
    if (!value || value === 'ー') return 'ー';
    if (value === '±') return '±';
    if (value === '＋' || value === '＋(1＋)') return '＋';
    if (value === '＋＋' || value === '＋＋（2＋）') return '＋＋';
    if (value === '＋＋＋' || value === '＋＋＋（3＋）') return '＋＋＋';
    return value;
}

/**
 * 날짜 파싱 함수
 * @param {*} dateValue - 날짜 값
 * @returns {Date|null} 파싱된 날짜
 */
function parseDate(dateValue) {
    if (!dateValue) return null;
    if (dateValue instanceof Date) return dateValue;
    
    // Excel 시리얼 번호인 경우
    if (typeof dateValue === 'number') {
        const date = new Date((dateValue - 25569) * 86400 * 1000);
        return date;
    }
    
    // 문자열인 경우
    if (typeof dateValue === 'string') {
        // 2025-04-01 형식
        const match = dateValue.match(/(\\d{4})-(\\d{2})-(\\d{2})/);
        if (match) {
            return new Date(parseInt(match[1]), parseInt(match[2]) - 1, parseInt(match[3]));
        }
    }
    
    return null;
}

/**
 * 숫자 파싱 함수
 * @param {*} value - 값
 * @returns {number|null} 파싱된 숫자
 */
function parseNumber(value) {
    if (!value || value === '' || value === '-') return null;
    const num = parseFloat(value);
    return isNaN(num) ? null : num;
}

/**
 * 엑셀 파일을 PostgreSQL 환자 테이블로 변환
 * @param {string} filePath - 엑셀 파일 경로
 * @param {string} filename - 파일명
 * @returns {Object} 처리 결과
 */
async function processPatientDataFile(filePath, filename) {
    const logger = console;
    
    try {
        logger.log(`[PatientDataProcessor] 환자 데이터 파일 처리 시작: ${filename}`);
        
        // DatabaseManager 연결
        const dbModule = await import('../services/database/index.js');
        const dbManager = dbModule.default;
        
        // 환자 데이터 테이블이 존재하는지 확인하고 없으면 생성
        await ensurePatientHealthDataTable(dbManager);
        
        // 엑셀 파일 읽기
        const workbook = XLSX.readFile(filePath);
        const sheetName = workbook.SheetNames[0];
        const worksheet = workbook.Sheets[sheetName];
        
        // JSON으로 변환 (헤더 포함)
        const data = XLSX.utils.sheet_to_json(worksheet, { header: 1 });
        
        logger.log(`[PatientDataProcessor] 총 ${data.length}개 행 발견`);
        
        // 헤더 행은 인덱스 0, 데이터는 인덱스 1부터 시작
        const dataStartIndex = 1;
        
        if (dataStartIndex === -1) {
            logger.log('[PatientDataProcessor] 환자 데이터를 찾을 수 없습니다.');
            // DatabaseManager는 자동으로 연결 관리
            return { success: false, error: '환자 데이터를 찾을 수 없음' };
        }
        
        logger.log(`[PatientDataProcessor] 데이터 시작 인덱스: ${dataStartIndex}`);
        
        // 기존 데이터 삭제
        await dbManager.query('DELETE FROM patient_health_data');
        logger.log('[PatientDataProcessor] 기존 데이터 삭제 완료');
        
        let insertCount = 0;
        
        // 데이터 행 처리
        for (let i = dataStartIndex; i < data.length; i++) {
            const row = data[i];
            
            // 빈 행이거나 환자명이 없는 행 건너뛰기
            if (!row || row.length < 20 || !row[2]) {
                continue;
            }
            
            // 환자명이 있는 실제 데이터 행인지 확인 (한국어 이름)
            const patientName = row[2];
            if (typeof patientName !== 'string' || patientName.length < 2) {
                continue;
            }
            
            try {
                const insertQuery = `
                    INSERT INTO patient_health_data (
                        patient_name, gender, birth_date, checkup_date,
                        systolic_bp_1, systolic_bp_2, systolic_bp_avg,
                        diastolic_bp_1, diastolic_bp_2, diastolic_bp_avg,
                        triglycerides, hdl, ldl,
                        blood_sugar, fasting_blood_sugar, random_blood_sugar, hba1c,
                        bmi, waist_circumference, urine_protein, source_file
                    ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18, $19, $20, $21)
                `;
                
                const values = [
                    row[2] || null, // patient_name (성명)
                    parseNumber(row[4]) || null, // gender (성별)
                    parseDate(row[5]), // birth_date (생년월일)
                    parseDate(row[6]), // checkup_date (정기건진수진일)
                    parseNumber(row[7]), // systolic_bp_1 (수축기혈압1회째)
                    parseNumber(row[8]), // systolic_bp_2 (수축기혈압2회째)
                    parseNumber(row[9]), // systolic_bp_avg (수축기혈압평균)
                    parseNumber(row[10]), // diastolic_bp_1 (확장기혈압1회째)
                    parseNumber(row[11]), // diastolic_bp_2 (확장기혈압2회째)
                    parseNumber(row[12]), // diastolic_bp_avg (확장기혈압평균)
                    parseNumber(row[13]), // triglycerides (중성지방)
                    parseNumber(row[14]), // hdl (HDL)
                    parseNumber(row[15]), // ldl (LDL)
                    parseNumber(row[16]), // blood_sugar (혈당)
                    parseNumber(row[17]), // fasting_blood_sugar (공복시혈당)
                    parseNumber(row[18]), // random_blood_sugar (수시혈당)
                    parseNumber(row[19]), // hba1c (HbA1c)
                    parseNumber(row[20]), // bmi (BMI)
                    parseNumber(row[21]), // waist_circumference (복위)
                    convertUrineProtein(row[22]), // urine_protein (요단백)
                    filename // source_file (원본 파일명)
                ];
                
                await dbManager.query(insertQuery, values);
                insertCount++;
                
                if (insertCount <= 5) {
                    logger.log(`[PatientDataProcessor] 환자 ${insertCount}: ${patientName} 삽입 완료`);
                }
                
            } catch (error) {
                logger.error(`[PatientDataProcessor] 행 ${i} 처리 중 오류:`, error.message);
            }
        }
        
        // DatabaseManager는 자동으로 연결을 관리하므로 별도의 close가 필요 없음
        
        logger.log(`[PatientDataProcessor] 총 ${insertCount}명의 환자 데이터 삽입 완료!`);
        
        return {
            success: true,
            insertCount,
            message: `${insertCount}명의 환자 데이터가 성공적으로 임포트되었습니다.`,
            filename: filename
        };
        
    } catch (error) {
        logger.error('[PatientDataProcessor] 처리 오류:', error);
        return {
            success: false,
            error: error.message,
            filename: filename
        };
    }
}

/**
 * 환자 건강 데이터 테이블이 존재하는지 확인하고 없으면 생성
 * @param {Object} dbManager - 데이터베이스 매니저
 */
async function ensurePatientHealthDataTable(dbManager) {
    const logger = console;
    
    try {
        // 테이블 존재 여부 확인
        const result = await dbManager.query(`
            SELECT EXISTS (
                SELECT FROM information_schema.tables 
                WHERE table_schema = 'public' 
                AND table_name = 'patient_health_data'
            );
        `);
        
        const tableExists = result.rows[0].exists;
        
        if (!tableExists) {
            logger.log('[PatientDataProcessor] patient_health_data 테이블이 없음. 생성 중...');
            
            // 환자 건강 데이터 테이블 생성
            await dbManager.query(`
                CREATE TABLE patient_health_data (
                    id SERIAL PRIMARY KEY,
                    patient_name VARCHAR(100) NOT NULL,
                    gender INTEGER,
                    birth_date DATE,
                    checkup_date DATE,
                    systolic_bp_1 NUMERIC(5,1),
                    systolic_bp_2 NUMERIC(5,1), 
                    systolic_bp_avg NUMERIC(5,1),
                    diastolic_bp_1 NUMERIC(5,1),
                    diastolic_bp_2 NUMERIC(5,1),
                    diastolic_bp_avg NUMERIC(5,1),
                    triglycerides NUMERIC(7,1),
                    hdl NUMERIC(5,1),
                    ldl NUMERIC(5,1),
                    blood_sugar NUMERIC(5,1),
                    fasting_blood_sugar NUMERIC(5,1),
                    random_blood_sugar NUMERIC(5,1),
                    hba1c NUMERIC(4,2),
                    bmi NUMERIC(4,1),
                    waist_circumference NUMERIC(5,1),
                    urine_protein VARCHAR(10),
                    source_file VARCHAR(255) DEFAULT '건강검진데이터.xlsx',
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                );
            `);
            
            // 인덱스 생성
            await dbManager.query(`
                CREATE INDEX IF NOT EXISTS idx_patient_health_data_name ON patient_health_data(patient_name);
                CREATE INDEX IF NOT EXISTS idx_patient_health_data_checkup_date ON patient_health_data(checkup_date);
                CREATE INDEX IF NOT EXISTS idx_patient_health_data_created_at ON patient_health_data(created_at);
            `);
            
            logger.log('[PatientDataProcessor] patient_health_data 테이블 생성 완료');
        } else {
            logger.log('[PatientDataProcessor] patient_health_data 테이블이 이미 존재함');
        }
        
    } catch (error) {
        logger.error('[PatientDataProcessor] 테이블 생성 오류:', error);
        throw error;
    }
}

export {
    isPatientDataFile,
    processPatientDataFile,
    ensurePatientHealthDataTable,
    PATIENT_DATA_FILE_PATTERNS
};