import { NextRequest, NextResponse } from 'next/server';
import * as XLSX from 'xlsx';
import { v4 as uuidv4 } from 'uuid';
import pg from 'pg';
import { getDatabaseConfig } from '@/config/serverConfig';

const { Pool } = pg;

// 데이터베이스 연결
function getDbConnection() {
  const config = getDatabaseConfig();
  return new Pool({
    host: config.host,
    port: config.port,
    user: config.user,
    password: config.password,
    database: config.name,
    max: 5
  });
}

export async function POST(request: NextRequest) {
  try {
    // 임시로 인증 건너뛰기 (기존 방식 유지)
    const userId = 'admin';

    const formData = await request.formData();
    const file = formData.get('file') as File;

    if (!file) {
      return NextResponse.json({
        success: false,
        error: '파일이 업로드되지 않았습니다.'
      }, { status: 400 });
    }

    // 파일 확장자 검증
    if (!file.name.match(/\.(xlsx|xls)$/i)) {
      return NextResponse.json({
        success: false,
        error: 'Excel 파일(.xlsx, .xls)만 업로드 가능합니다.'
      }, { status: 400 });
    }

    // 파일 크기 검증 (10MB)
    if (file.size > 10 * 1024 * 1024) {
      return NextResponse.json({
        success: false,
        error: '파일 크기는 10MB 이하여야 합니다.'
      }, { status: 400 });
    }

    // Excel 파일 읽기
    const buffer = await file.arrayBuffer();
    const workbook = XLSX.read(buffer, { type: 'buffer' });
    
    if (workbook.SheetNames.length === 0) {
      return NextResponse.json({
        success: false,
        error: 'Excel 파일에 시트가 없습니다.'
      }, { status: 400 });
    }

    // 첫 번째 시트 읽기 (또는 특정 시트명 지정)
    const firstSheetName = workbook.SheetNames[0];
    const worksheet = workbook.Sheets[firstSheetName];
    const jsonData = XLSX.utils.sheet_to_json(worksheet);

    if (jsonData.length === 0) {
      return NextResponse.json({
        success: false,
        error: 'Excel 파일에 데이터가 없습니다.'
      }, { status: 400 });
    }

    // 데이터베이스 연결
    const dbManager = getDbConnection();

    // 새로운 배치 작업 생성
    const jobId = uuidv4();
    
    await dbManager.query(`
      INSERT INTO health_batch_jobs (id, user_id, filename, status, total_records)
      VALUES ($1, $2, $3, 'pending', $4)
    `, [jobId, userId, file.name, jsonData.length]); // 실제 데이터 행 수
    
    // 환자 데이터 파싱 및 데이터베이스 저장
    let insertCount = 0;
    const patients = [];

    for (let i = 0; i < jsonData.length; i++) {
      const row = jsonData[i] as any;

      // 환자명 추출 (다양한 컬럼명 대응)
      let name = row['환자명'] || row['이름'] || row['성명'] ||
                 row['patient_name'] || row['name'] ||
                 row['氏名'] || row['患者名'];

      // 환자명이 없는 경우 기본 이름 생성
      if (!name || typeof name !== 'string' || name.trim() === '') {
        name = `환자_${i + 1}`;  // 행번호 기반 기본 이름
        console.log(`행 ${i + 1}: 환자명이 없어서 기본 이름 사용: ${name}, 사용 가능한 키:`, Object.keys(row));
      }

      // 원시 데이터를 JSONB로 저장
      const rawData = row;

      try {
        // 환자 데이터를 데이터베이스에 저장
        const result = await dbManager.query(`
          INSERT INTO health_batch_patients (job_id, patient_name, row_number, raw_data)
          VALUES ($1, $2, $3, $4)
          RETURNING id
        `, [jobId, name.trim(), i + 1, rawData]);

        insertCount++;

        // 미리보기용 데이터 (모든 환자 데이터 포함)
        patients.push({
          id: result.rows[0].id,
          name: name.trim(),
          rawData: row
        });

      } catch (dbError) {
        console.error(`환자 ${name} 저장 중 오류:`, dbError);
      }
    }

    if (insertCount === 0) {
      // 빈 작업 삭제
      await dbManager.query('DELETE FROM health_batch_jobs WHERE id = $1', [jobId]);
      
      return NextResponse.json({
        success: false,
        error: '유효한 환자 데이터를 찾을 수 없습니다. 환자명 컬럼이 포함된 Excel 파일인지 확인해주세요.'
      }, { status: 400 });
    }

    // 작업 정보 업데이트
    await dbManager.query(`
      UPDATE health_batch_jobs 
      SET total_records = $1, status = 'pending'
      WHERE id = $2
    `, [insertCount, jobId]);

    return NextResponse.json({
      success: true,
      data: {
        jobId,
        patients,
        totalCount: insertCount,
        filename: file.name,
        message: `${insertCount}명의 환자 데이터가 데이터베이스에 저장되었습니다.`
      }
    });

  } catch (error) {
    console.error('Excel 파일 파싱 오류:', error);
    
    return NextResponse.json({
      success: false,
      error: error instanceof Error ? error.message : 'Excel 파일 파싱에 실패했습니다.'
    }, { status: 500 });
  }
}