import { NextRequest, NextResponse } from 'next/server';
import { getLLMFinetuneServerUrl } from '@/config/serverConfig';

const LLM_FINETUNE_API_BASE_URL = getLLMFinetuneServerUrl();

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const modelId = searchParams.get('modelId');
    const evalId = searchParams.get('evalId');
    
    let url = `${LLM_FINETUNE_API_BASE_URL}/evaluation`;
    if (evalId) {
      url += `/${evalId}/results`;
    } else if (modelId) {
      url += `?modelId=${modelId}`;
    }
    
    console.log('[LLM Evaluation API] 요청 URL:', url);
    
    const response = await fetch(url);
    console.log('[LLM Evaluation API] 응답 상태:', response.status, response.statusText);
    
    if (!response.ok) {
      console.error('[LLM Evaluation API] HTTP 오류:', response.status, response.statusText);
      const errorText = await response.text();
      console.error('[LLM Evaluation API] 오류 내용:', errorText);
      
      return NextResponse.json(
        { 
          success: false, 
          error: `LLM 평가 서버 오류: ${response.status} ${response.statusText}`,
          details: errorText
        },
        { status: response.status }
      );
    }
    
    const data = await response.json();
    console.log('[LLM Evaluation API] 성공적으로 데이터 받음');
    
    return NextResponse.json(data);
  } catch (error) {
    console.error('[LLM Evaluation API] 평가 조회 실패:', error);
    
    const isNetworkError = error instanceof Error && (
      error.message.includes('ECONNREFUSED') ||
      error.message.includes('ENOTFOUND') ||
      error.message.includes('ETIMEDOUT') ||
      error.message.includes('fetch failed')
    );
    
    return NextResponse.json(
      { 
        success: false, 
        error: isNetworkError 
          ? `LLM 파인튜닝 서버(${LLM_FINETUNE_API_BASE_URL})에 연결할 수 없습니다. 서버가 실행 중인지 확인하세요.`
          : '평가 결과를 조회할 수 없습니다.',
        details: error instanceof Error ? error.message : String(error)
      },
      { status: isNetworkError ? 503 : 500 }
    );
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    
    console.log('[LLM Evaluation API] 평가 시작 요청:', body);
    
    // 필수 필드 검증
    const { modelId, evaluationType } = body;
    if (!modelId || !evaluationType) {
      return NextResponse.json(
        { success: false, error: '모델 ID와 평가 타입은 필수입니다.' },
        { status: 400 }
      );
    }
    
    // 평가 타입 검증
    const validEvaluationTypes = ['perplexity', 'bleu', 'rouge', 'custom'];
    if (!validEvaluationTypes.includes(evaluationType)) {
      return NextResponse.json(
        { success: false, error: `지원하지 않는 평가 타입입니다. 가능한 타입: ${validEvaluationTypes.join(', ')}` },
        { status: 400 }
      );
    }
    
    const response = await fetch(`${LLM_FINETUNE_API_BASE_URL}/evaluation/start`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        modelId,
        evaluationType,
        testDatasetId: body.testDatasetId,
        
        // 평가 설정
        metrics: body.metrics || ['perplexity', 'accuracy'],
        sampleSize: body.sampleSize || 100,
        maxLength: body.maxLength || 512,
        
        // Custom 평가 설정
        customPrompts: body.customPrompts || [],
        expectedAnswers: body.expectedAnswers || [],
        
        // 추가 설정
        temperature: body.temperature || 0.7,
        topP: body.topP || 0.9,
        topK: body.topK || 50
      })
    });
    
    const data = await response.json();
    console.log('[LLM Evaluation API] 평가 시작 응답:', data);
    
    return NextResponse.json(data);
    
  } catch (error) {
    console.error('[LLM Evaluation API] 평가 시작 실패:', error);
    return NextResponse.json(
      { success: false, error: 'LLM 평가를 시작할 수 없습니다.' },
      { status: 500 }
    );
  }
}

export async function DELETE(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const evalId = searchParams.get('evalId');
    
    if (!evalId) {
      return NextResponse.json(
        { success: false, error: '평가 ID가 필요합니다.' },
        { status: 400 }
      );
    }
    
    const response = await fetch(`${LLM_FINETUNE_API_BASE_URL}/evaluation/${evalId}/stop`, {
      method: 'POST'
    });
    
    const data = await response.json();
    
    return NextResponse.json(data);
    
  } catch (error) {
    console.error('[LLM Evaluation API] 평가 중단 실패:', error);
    return NextResponse.json(
      { success: false, error: '평가를 중단할 수 없습니다.' },
      { status: 500 }
    );
  }
}