import { NextRequest, NextResponse } from 'next/server';

const RAG_API_BASE_URL = process.env.NEXT_PUBLIC_RAG_API_SERVER || 'http://localhost:5600';
const API_SERVER_BASE_URL = process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500/api/v1';

export async function GET() {
  try {
    console.log('[RAG API] 모델 목록 요청 시작');
    console.log('[RAG API] RAG_API_BASE_URL:', RAG_API_BASE_URL);
    console.log('[RAG API] 요청 URL:', `${RAG_API_BASE_URL}/models/list`);
    
    const response = await fetch(`${RAG_API_BASE_URL}/models/list`);
    console.log('[RAG API] 응답 상태:', response.status, response.statusText);
    
    if (!response.ok) {
      console.error('[RAG API] HTTP 오류:', response.status, response.statusText);
      const errorText = await response.text();
      console.error('[RAG API] 오류 내용:', errorText);
      
      return NextResponse.json(
        { 
          success: false, 
          error: `RAG 서버 오류: ${response.status} ${response.statusText}`,
          details: errorText
        },
        { status: response.status }
      );
    }
    
    const data = await response.json();
    console.log('[RAG API] 성공적으로 데이터 받음:', Object.keys(data));
    
    return NextResponse.json(data);
  } catch (error) {
    console.error('[RAG API] 모델 목록 호출 실패:', error);
    console.error('[RAG API] 오류 타입:', error instanceof Error ? error.name : typeof error);
    console.error('[RAG API] 오류 메시지:', error instanceof Error ? error.message : String(error));
    
    // 네트워크 연결 오류 체크
    const isNetworkError = error instanceof Error && (
      error.message.includes('ECONNREFUSED') ||
      error.message.includes('ENOTFOUND') ||
      error.message.includes('ETIMEDOUT') ||
      error.message.includes('fetch failed')
    );
    
    return NextResponse.json(
      { 
        success: false, 
        error: isNetworkError 
          ? `RAG 서버(${RAG_API_BASE_URL})에 연결할 수 없습니다. 서버가 실행 중인지 확인하세요.`
          : '모델 목록을 불러올 수 없습니다.',
        details: error instanceof Error ? error.message : String(error)
      },
      { status: isNetworkError ? 503 : 500 }
    );
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { action, ...params } = body;
    
    let endpoint = '';
    let method = 'POST';
    let useMainApiServer = false;
    
    switch (action) {
      case 'finetune':
        endpoint = '/models/finetune';
        break;
      case 'analyze-resources':
        endpoint = '/models/analyze-resources';
        break;
      case 'get-config':
        endpoint = '/config';
        method = 'GET';
        break;
      case 'set-active-model':
        // 기본 모델인지 확인
        if (params.modelId === 'nlpai-lab/KURE-v1') {
          // 기본 모델은 RAG 서버의 설정 API 직접 호출
          endpoint = '/config/set-active-model';
          params.model_name = params.modelId;
          delete params.modelId;
        } else {
          // 파인튜닝된 모델은 활성화 엔드포인트 사용
          const encodedModelId = encodeURIComponent(params.modelId);
          endpoint = `/models/${encodedModelId}/activate`;
          delete params.modelId;
        }
        break;
      case 'evaluate-model':
        // 모델 평가 액션 - 새로 추가된 통합 API 사용
        endpoint = '/models';
        // action 필드를 params에 다시 추가 (RAG 서버에서 필요)
        params.action = action;
        console.log('DEBUG: evaluate-model 요청 params:', JSON.stringify(params, null, 2));
        break;
      case 'get-evaluation-results':
        // 평가 결과 조회 액션 - 새로 추가된 통합 API 사용
        endpoint = '/models';
        // action 필드를 params에 다시 추가 (RAG 서버에서 필요)
        params.action = action;
        break;
      default:
        return NextResponse.json(
          { success: false, error: '지원하지 않는 액션입니다.' },
          { status: 400 }
        );
    }
    
    // 요청할 서버 결정
    const baseUrl = useMainApiServer ? API_SERVER_BASE_URL : RAG_API_BASE_URL;
    
    // GET 요청인 경우 body 없이 호출
    const requestOptions: RequestInit = {
      method,
      headers: { 'Content-Type': 'application/json' }
    };
    
    if (method !== 'GET') {
      requestOptions.body = JSON.stringify(params);
    }
    
    const response = await fetch(`${baseUrl}${endpoint}`, requestOptions);
    
    if (!response.ok) {
      // RAG 서버 연결 실패 또는 HTTP 오류 처리
      const errorMessage = response.status === 0 || !response.status
        ? 'RAG 서버에 연결할 수 없습니다. 서버가 실행 중인지 확인하세요.'
        : `RAG API 호출 실패: ${response.status} ${response.statusText}`;
      
      return NextResponse.json(
        { success: false, error: errorMessage },
        { status: response.status || 503 }
      );
    }
    
    const data = await response.json();
    
    // RAG 서버 응답 형식을 프론트엔드 형식으로 변환
    if (action === 'analyze-resources') {
      // RAG 서버는 status: 'success'를 사용하므로 success: true로 변환
      if (data.status === 'success' && data.system_resources) {
        return NextResponse.json({
          success: true,
          resources: {
            total_memory_gb: data.system_resources.total_memory_gb,
            available_memory_gb: data.system_resources.available_memory_gb,
            cpu_cores: data.system_resources.cpu_cores,
            gpu_available: data.system_resources.gpu_available,
            gpu_memory_gb: data.system_resources.gpu_memory_gb,
            gpu_available_memory_gb: data.system_resources.gpu_available_memory_gb,
            recommendations: {
              gpu_mode: data.optimized_settings?.gpu_mode || {
                batch_size: 4,
                num_workers: 2,
                use_amp: true
              },
              cpu_mode: data.optimized_settings?.cpu_mode || {
                batch_size: 2,
                num_workers: 4
              }
            }
          }
        });
      } else {
        return NextResponse.json({
          success: false,
          error: data.error || '시스템 자원 분석에 실패했습니다.'
        });
      }
    }
    
    // get-config 액션 처리
    if (action === 'get-config') {
      return NextResponse.json({
        success: true,
        config: data.config || data
      });
    }
    
    // set-active-model 액션 처리  
    if (action === 'set-active-model') {
      return NextResponse.json({
        success: data.success || data.status === 'success',
        message: data.message || '모델이 성공적으로 변경되었습니다.',
        error: data.error
      });
    }
    
    // evaluate-model 액션 처리
    if (action === 'evaluate-model') {
      return NextResponse.json({
        success: data.success || false,
        metrics: data.metrics || {},
        model_id: data.model_id,
        test_samples: data.test_samples || 0,
        evaluation_time: data.evaluation_time || 0,
        error: data.error
      });
    }
    
    // get-evaluation-results 액션 처리
    if (action === 'get-evaluation-results') {
      return NextResponse.json({
        success: data.success || false,
        evaluation_results: data.evaluation_results || [],
        error: data.error
      });
    }
    
    // 다른 액션들의 경우 기존 로직 유지
    // RAG 서버에서 반환한 데이터에 error 필드가 없으면 기본 메시지 추가
    if (!data.success && !data.error) {
      data.error = 'RAG 서버에서 알 수 없는 오류가 발생했습니다.';
    }
    
    return NextResponse.json(data);
  } catch (error) {
    console.error('RAG API 모델 요청 실패:', error);
    
    // 연결 오류인지 확인
    const isConnectionError = error instanceof Error && 
      (error.message.includes('ECONNREFUSED') || 
       error.message.includes('fetch failed') ||
       (error.cause && typeof error.cause === 'object' && 'code' in error.cause && error.cause.code === 'ECONNREFUSED'));
    
    const errorMessage = isConnectionError
      ? 'RAG 서버에 연결할 수 없습니다. 서버가 실행 중인지 확인하세요.'
      : '모델 API 요청을 처리할 수 없습니다.';
    
    return NextResponse.json(
      { success: false, error: errorMessage },
      { status: isConnectionError ? 503 : 500 }
    );
  }
}

export async function DELETE(request: NextRequest) {
  try {
    const body = await request.json();
    const { modelId } = body;
    
    if (!modelId) {
      return NextResponse.json(
        { success: false, error: '모델 ID가 필요합니다.' },
        { status: 400 }
      );
    }
    
    // 기본 모델은 삭제할 수 없음
    if (modelId === 'nlpai-lab/KURE-v1' || modelId.includes('base')) {
      return NextResponse.json(
        { success: false, error: '기본 모델은 삭제할 수 없습니다.' },
        { status: 400 }
      );
    }
    
    // RAG 서버의 모델 삭제 API 호출
    const encodedModelId = encodeURIComponent(modelId);
    const response = await fetch(`${RAG_API_BASE_URL}/models/${encodedModelId}`, {
      method: 'DELETE'
    });
    
    if (!response.ok) {
      const errorMessage = response.status === 0 || !response.status
        ? 'RAG 서버에 연결할 수 없습니다. 서버가 실행 중인지 확인하세요.'
        : `모델 삭제 실패: ${response.status} ${response.statusText}`;
      
      return NextResponse.json(
        { success: false, error: errorMessage },
        { status: response.status || 503 }
      );
    }
    
    const data = await response.json();
    
    return NextResponse.json({
      success: data.success || data.status === 'success',
      message: data.message || '모델이 성공적으로 삭제되었습니다.',
      error: data.error
    });
    
  } catch (error) {
    console.error('모델 삭제 요청 실패:', error);
    
    const isConnectionError = error instanceof Error && 
      (error.message.includes('ECONNREFUSED') || 
       error.message.includes('fetch failed') ||
       (error.cause && typeof error.cause === 'object' && 'code' in error.cause && error.cause.code === 'ECONNREFUSED'));
    
    const errorMessage = isConnectionError
      ? 'RAG 서버에 연결할 수 없습니다. 서버가 실행 중인지 확인하세요.'
      : '모델 삭제 요청을 처리할 수 없습니다.';
    
    return NextResponse.json(
      { success: false, error: errorMessage },
      { status: isConnectionError ? 503 : 500 }
    );
  }
}