import { NextRequest, NextResponse } from 'next/server';

export async function GET(request: NextRequest) {
  try {

    // URL 파라미터에서 파일명과 사용자 ID 가져오기
    const searchParams = request.nextUrl.searchParams;
    const filename = searchParams.get('filename');
    const userId = searchParams.get('userId');

    if (!filename || !userId) {
      return NextResponse.json(
        { success: false, error: { code: 'BAD_REQUEST', message: '파일명과 사용자 ID가 필요합니다.' } },
        { status: 400 }
      );
    }

    // RAG 서버에 문서 내용 요청
    const ragServerUrl = process.env.NEXT_PUBLIC_RAG_API_SERVER || 'http://localhost:5600';
    const response = await fetch(`${ragServerUrl}/document/preview?filename=${encodeURIComponent(filename)}&user_id=${encodeURIComponent(userId)}`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
      },
    });

    if (!response.ok) {
      // RAG 서버 오류 처리
      const errorData = await response.json().catch(() => ({ error: 'RAG 서버 오류' }));
      console.error('RAG 서버 문서 미리보기 오류:', errorData);
      
      return NextResponse.json(
        { 
          success: false, 
          error: { 
            code: 'RAG_SERVER_ERROR', 
            message: errorData.error || '문서를 불러올 수 없습니다.' 
          } 
        },
        { status: response.status }
      );
    }

    const data = await response.json();

    // 문서 내용 반환
    return NextResponse.json({
      success: true,
      data: {
        content: data.content || '',
        metadata: data.metadata ? {
          filename: data.metadata.filename || filename,
          size: data.metadata.size,
          modified: data.metadata.modified,
          chunks: data.metadata.chunks,
          type: data.metadata.type
        } : null
      }
    });

  } catch (error) {
    console.error('문서 미리보기 API 오류:', error);
    return NextResponse.json(
      { 
        success: false, 
        error: { 
          code: 'INTERNAL_SERVER_ERROR', 
          message: '서버 오류가 발생했습니다.' 
        } 
      },
      { status: 500 }
    );
  }
}