import { NextRequest, NextResponse } from "next/server";
import { pg } from "@/app/api/db";

/**
 * GET issues with pagination and search.
 */
export async function GET(req: NextRequest) {
  const { searchParams } = new URL(req.url);
  const userName = searchParams.get("userName");
  const userRole = searchParams.get("userRole");
  const searchTerm = searchParams.get("searchTerm") || "";
  const page = parseInt(searchParams.get("page") || "1", 10);
  const limit = parseInt(searchParams.get("limit") || "10", 10);
  const offset = (page - 1) * limit;

  // Log received parameters for debugging
  console.log("[ISSUES_GET_PARAMS]", { userName, userRole, searchTerm });

  // 사용자 정보 처리 (guest 사용자 허용)
  const effectiveUserName = userName || 'guest';
  const effectiveUserRole = userRole || 'guest';

  try {
    const params: any[] = [];
    let whereClauses: string[] = [];

    // 관리자가 아닌 경우의 처리
    if (effectiveUserRole !== "admin" && effectiveUserRole !== "manager") {
      if (effectiveUserRole === 'guest') {
        // guest 사용자는 모든 티켓 조회 가능 (읽기 전용)
        // whereClauses에 조건 추가하지 않음
      } else {
        // 일반 로그인 사용자는 자신의 티켓만 조회
        whereClauses.push(`LOWER(author_name) = LOWER($${params.push(effectiveUserName)})`);
      }
    }

    if (searchTerm) {
      whereClauses.push(`(title ILIKE $${params.push(`%${searchTerm}%`)} OR question ILIKE $${params.push(`%${searchTerm}%`)})`);
    }

    const whereString = whereClauses.length > 0 ? `WHERE ${whereClauses.join(" AND ")}` : "";

    // Get total count for pagination
    const countQuery = `SELECT COUNT(*) FROM support_tickets ${whereString}`;
    const countResult = await pg.query(countQuery, params);
    const totalIssues = parseInt(countResult.rows[0].count, 10);

    // Get issues for the current page
    const issuesQuery = `SELECT * FROM support_tickets ${whereString} ORDER BY created_at DESC LIMIT $${params.length + 1} OFFSET $${params.length + 2}`;
    const issuesResult = await pg.query(issuesQuery, [...params, limit, offset]);
    const issues = issuesResult.rows;
    
    const transformedIssues = issues.map(issue => ({
      ...issue,
      createdAt: issue.created_at,
      author: { name: issue.author_name, role: issue.author_role },
      attachment_image: issue.attachment_image ? issue.attachment_image.toString('base64') : null,
      chat_history: issue.chat_history || [],
      session_id: issue.session_id || issue.id,
    }));
    
    return NextResponse.json({ issues: transformedIssues, totalCount: totalIssues });
  } catch (error) {
    console.error("[ISSUES_GET_ERROR]", error);
    return NextResponse.json({ error: "Internal Server Error" }, { status: 500 });
  }
}

/**
 * POST a new issue.
 */
export async function POST(req: NextRequest) {
  try {
    const formData = await req.formData();
    const authorString = formData.get('author') as string;
    const title = formData.get('title') as string;
    const question = formData.get('question') as string;
    const imageFile = formData.get('image') as File | null;

    if (!authorString || !title || !question) {
      return NextResponse.json({ error: "Missing required fields" }, { status: 400 });
    }
    
    const author = JSON.parse(authorString);
    if (!author || !author.name || !author.role) {
      return NextResponse.json({ error: "Missing author fields" }, { status: 400 });
    }

    let imageBuffer: Buffer | null = null;
    let imageMimeType: string | null = null;
    if (imageFile) {
      const bytes = await imageFile.arrayBuffer();
      imageBuffer = Buffer.from(bytes);
      imageMimeType = imageFile.type;
    }

    const result = await pg.query(
      "INSERT INTO support_tickets (title, question, author_name, author_role, attachment_image, attachment_image_mimetype) VALUES ($1, $2, $3, $4, $5, $6) RETURNING *",
      [title, question, author.name, author.role, imageBuffer, imageMimeType]
    );
    
    const newIssueRaw = result.rows[0];
    // Convert Buffer to a format that can be sent via JSON (e.g., base64)
    if (newIssueRaw.attachment_image) {
        newIssueRaw.attachment_image = newIssueRaw.attachment_image.toString('base64');
    }

    const newIssue = {
      ...newIssueRaw,
      createdAt: newIssueRaw.created_at,
      author: { name: newIssueRaw.author_name, role: newIssueRaw.author_role },
    }

    return NextResponse.json(newIssue, { status: 201 });
  } catch (error) {
    console.error("[ISSUES_POST_ERROR]", error);
    return NextResponse.json({ error: "Internal Server Error" }, { status: 500 });
  }
} 