import { NextRequest, NextResponse } from 'next/server';
import OpenAI from 'openai';
import Anthropic from '@anthropic-ai/sdk';

import * as fs from 'fs';
import * as path from 'path';
import * as os from 'os';

// airun.conf에서 API 키 가져오기
const getApiKeys = () => {
  const config: any = {};
  
  // 1. process.env에서 환경변수 우선 로드
  for (const [key, value] of Object.entries(process.env)) {
    if (value) {
      config[key] = value;
    }
  }
  
  // 2. airun.conf 파일에서 추가 설정 로드
  const configPath = path.join(os.homedir(), '.airun', 'airun.conf');
  if (fs.existsSync(configPath)) {
    try {
      const content = fs.readFileSync(configPath, 'utf8');
      for (const line of content.split('\n')) {
        const trimmed = line.trim();
        if (trimmed.startsWith('export ')) {
          const match = trimmed.match(/export\s+([A-Z_]+)=["']?([^"']+)["']?/);
          if (match) {
            // process.env에 있는 값을 덮어쓰지 않음 (process.env 우선)
            if (!config[match[1]]) {
              config[match[1]] = match[2];
            }
          }
        }
      }
    } catch (error) {
      console.error('airun.conf 읽기 실패:', error);
    }
  }
  
  console.log('🔧 API 키 로드 상태:', {
    OPENAI_API_KEY: config.OPENAI_API_KEY ? `✅ ${config.OPENAI_API_KEY.substring(0, 10)}...` : '❌ 없음',
    ANTHROPIC_API_KEY: config.ANTHROPIC_API_KEY ? `✅ ${config.ANTHROPIC_API_KEY.substring(0, 10)}...` : '❌ 없음'
  });
  
  return {
    openai: config.OPENAI_API_KEY,
    anthropic: config.ANTHROPIC_API_KEY
  };
};

export async function POST(request: NextRequest) {
  console.log('🧪 이미지 API 테스트 시작');
  
  try {
    const body = await request.json();
    const { provider, message, imageBase64 } = body;
    
    console.log(`🎯 프로바이더: ${provider}`);
    console.log(`📝 메시지: ${message}`);
    console.log(`🖼️ 이미지 Base64 길이: ${imageBase64?.length || 0}`);
    
    const apiKeys = getApiKeys();
    
    if (provider === 'openai') {
      if (!apiKeys.openai) {
        throw new Error('OpenAI API 키가 설정되지 않았습니다');
      }
      
      console.log('🤖 OpenAI API 호출 시작');
      const openai = new OpenAI({
        apiKey: apiKeys.openai
      });
      
      const response = await openai.chat.completions.create({
        model: 'gpt-4o-mini', // 이미지를 지원하는 모델
        messages: [
          {
            role: 'user',
            content: [
              {
                type: 'text',
                text: message
              },
              {
                type: 'image_url',
                image_url: {
                  url: `data:image/jpeg;base64,${imageBase64}`
                }
              }
            ]
          }
        ],
        max_tokens: 1000
      });
      
      console.log('✅ OpenAI 응답 성공');
      return NextResponse.json({
        success: true,
        provider: 'openai',
        response: response.choices[0].message.content,
        usage: response.usage
      });
      
    } else if (provider === 'anthropic') {
      if (!apiKeys.anthropic) {
        throw new Error('Anthropic API 키가 설정되지 않았습니다');
      }
      
      console.log('🤖 Anthropic API 호출 시작');
      const anthropic = new Anthropic({
        apiKey: apiKeys.anthropic
      });
      
      const response = await anthropic.messages.create({
        model: 'claude-3-sonnet-20240229',
        max_tokens: 1000,
        messages: [
          {
            role: 'user',
            content: [
              {
                type: 'text',
                text: message
              },
              {
                type: 'image',
                source: {
                  type: 'base64',
                  media_type: 'image/jpeg',
                  data: imageBase64
                }
              }
            ]
          }
        ]
      });
      
      console.log('✅ Anthropic 응답 성공');
      return NextResponse.json({
        success: true,
        provider: 'anthropic',
        response: response.content[0].type === 'text' ? response.content[0].text : '',
        usage: response.usage
      });
      
    } else {
      throw new Error(`지원하지 않는 프로바이더: ${provider}`);
    }
    
  } catch (error) {
    console.error('❌ 이미지 API 테스트 실패:', error);
    return NextResponse.json({
      success: false,
      error: error instanceof Error ? error.message : '알 수 없는 오류',
      stack: error instanceof Error ? error.stack : undefined
    }, { status: 500 });
  }
} 