'use client';

import React, { useState, useEffect } from 'react';
import { XMarkIcon } from '@heroicons/react/24/outline';
import GraphRAGVisualization from './GraphRAGVisualization';
import { getRagServerUrl } from '@/config/serverConfig';

interface GraphData {
  nodes: any[];
  links: any[];
  communities: any[];
  stats: {
    total_nodes: number;
    total_links: number;
    total_communities: number;
    entity_types: string[];
    relationship_types: string[];
  };
}

interface GraphRAGModalProps {
  isOpen: boolean;
  onClose: () => void;
  query?: string;
  userId?: string;
  sessionId?: string;
}

const GraphRAGModal: React.FC<GraphRAGModalProps> = ({
  isOpen,
  onClose,
  query,
  userId,
  sessionId
}) => {
  const [graphData, setGraphData] = useState<GraphData | null>(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [viewMode, setViewMode] = useState<'overview' | 'query-focused' | 'entity-focused'>('overview');
  const [selectedEntity, setSelectedEntity] = useState<string | null>(null);

  // GraphRAG 데이터 로드
  const loadGraphData = async () => {
    setLoading(true);
    setError(null);
    
    try {
      const ragServerUrl = getRagServerUrl();

      // 1. admin RAG 페이지: 청크 정보가 JSON으로 전달된 경우 확인
      let chunkInfo = null;
      try {
        const parsed = JSON.parse(query || '{}');
        if (parsed.type === 'chunk' && parsed.chunk_index !== undefined) {
          chunkInfo = { ...parsed, method: 'database' };
        } else if (parsed.type === 'chunk_content' && parsed.chunk_content) {
          chunkInfo = { ...parsed, method: 'content' };
        }
      } catch {
        // JSON 파싱 실패시 채팅 페이지 쿼리로 처리
      }

      if (chunkInfo) {
        console.log('청크 정보 GraphRAG 데이터 로드:', chunkInfo);

        if (chunkInfo.method === 'content') {
          // 청크 내용을 직접 전달받은 경우: 내용 기반 엔티티 검색
          const queryResponse = await fetch(`${ragServerUrl}/graphrag/query-entities`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json'
            },
            body: JSON.stringify({
              query: chunkInfo.chunk_content, // 청크 내용만 쿼리로 사용
              user_id: chunkInfo.user_id || userId || 'admin',
              mode: 'graph',
              view_mode: viewMode,
              top_k: 15 // 청크 내용 기반이므로 적당한 수
            })
          });

          if (queryResponse.ok) {
            const result = await queryResponse.json();
            console.log('청크 내용 기반 엔티티 추출 결과:', result);

            if (result.success) {
              const graphData = transformDatabaseGraphData(result, chunkInfo.chunk_content); // chunk_content를 query로 사용
              setGraphData(graphData);
              return;
            }
          }
        } else {
          // 기존 방식: 데이터베이스에서 청크 검색
          const chunkResponse = await fetch(`${ragServerUrl}/graphrag/chunk-entities`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json'
            },
            body: JSON.stringify({
              filename: chunkInfo.filename,
              chunk_index: chunkInfo.chunk_index,
              user_id: chunkInfo.user_id || userId || 'admin'
            })
          });

          if (chunkResponse.ok) {
            const result = await chunkResponse.json();
            console.log('청크 엔티티 추출 결과:', result);

            if (result.success) {
              const graphData = transformDatabaseGraphData(result, query);
              setGraphData(graphData);
              return;
            }
          }
        }
      }

      // 2. 일반 채팅 페이지: 질의 기반 검색
      const response = await fetch(`${ragServerUrl}/graphrag/query-entities`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          query: query,
          user_id: userId || 'admin',
          mode: 'graph',
          view_mode: viewMode,
          top_k: 10
        })
      });

      if (response.ok) {
        const result = await response.json();
        console.log('GraphRAG 데이터 로드 결과:', result);

        if (result.success) {
          const graphData = transformDatabaseGraphData(result, query);
          setGraphData(graphData);
        } else {
          setError(result.error || 'GraphRAG 데이터를 불러오는데 실패했습니다.');
        }
      } else {
        setError('GraphRAG 서비스에 연결할 수 없습니다.');
      }
    } catch (error) {
      console.error('GraphRAG 데이터 로드 오류:', error);
      setError('GraphRAG 데이터를 불러오는 중 오류가 발생했습니다.');
    } finally {
      setLoading(false);
    }
  };;

  // 데이터베이스 graph_entities 데이터를 시각화 데이터로 변환
  const transformDatabaseGraphData = (result: any, searchQuery?: string): GraphData => {
    const nodes: any[] = [];
    const links: any[] = [];
    const entityMap = new Map();
    const entityIdSet = new Set(); // 중복 방지

    console.log('데이터베이스 그래프 데이터 변환:', result);

    if (result.success && result.data) {
      // 1단계: 직접 매칭된 entities 처리
      if (result.data.entities && Array.isArray(result.data.entities)) {
        result.data.entities.forEach((entity: any) => {
          const node = {
            id: entity.id.toString(),
            label: entity.name,
            type: entity.type,
            description: entity.description || '',
            importance: 1.0, // 직접 매칭된 엔티티는 최고 중요도
            community: 0,
            documents: entity.source_documents || [],
            highlighted: searchQuery ? 
              entity.name.toLowerCase().includes(searchQuery.toLowerCase()) : false
          };
          
          nodes.push(node);
          entityMap.set(entity.id, node.id);
          entityIdSet.add(entity.id);
        });
      }

      // 2단계: relationships에서 참조되는 모든 엔티티 ID 수집
      const relationshipEntityIds = new Set();
      if (result.data.relationships && Array.isArray(result.data.relationships)) {
        result.data.relationships.forEach((rel: any) => {
          relationshipEntityIds.add(rel.source_entity_id);
          relationshipEntityIds.add(rel.target_entity_id);
        });
      }

      // 3단계: 관계에 포함된 엔티티들 중 아직 추가되지 않은 것들을 데이터베이스에서 조회하여 추가
      const missingEntityIds = Array.from(relationshipEntityIds).filter(id => !entityIdSet.has(id));
      console.log('관계에서 참조되지만 누락된 엔티티 ID들:', missingEntityIds);

      // 누락된 엔티티들을 백엔드의 관계 데이터에서 이름을 가져와서 노드 생성
      if (missingEntityIds.length > 0) {
        // 관계 데이터에서 누락된 엔티티들의 이름을 찾아서 노드 생성
        const relationshipEntities = new Map();
        
        if (result.data.relationships) {
          result.data.relationships.forEach((rel: any) => {
            if (rel.source_name && rel.source_entity_id) {
              relationshipEntities.set(rel.source_entity_id, {
                name: rel.source_name,
                id: rel.source_entity_id
              });
            }
            if (rel.target_name && rel.target_entity_id) {
              relationshipEntities.set(rel.target_entity_id, {
                name: rel.target_name,
                id: rel.target_entity_id
              });
            }
          });
        }
        
        missingEntityIds.forEach(entityId => {
          if (!entityIdSet.has(entityId)) {
            const entityInfo = relationshipEntities.get(entityId);
            const node = {
              id: entityId.toString(),
              label: entityInfo ? entityInfo.name : `Entity ${entityId}`, // 실제 이름 사용
              type: 'related', // 관계를 통해 찾은 엔티티
              description: entityInfo ? `관련 엔티티: ${entityInfo.name}` : '관련 엔티티',
              importance: 0.6, // 간접 엔티티는 중간 중요도
              community: 0,
              documents: [],
              highlighted: false
            };
            
            nodes.push(node);
            entityMap.set(entityId, node.id);
            entityIdSet.add(entityId);
          }
        });
      }

      // 4단계: relationships 처리 (이제 모든 엔티티가 존재함)
      if (result.data.relationships && Array.isArray(result.data.relationships)) {
        result.data.relationships.forEach((rel: any) => {
          // 백엔드에서 source_name, target_name을 제공하므로 이를 사용
          const sourceId = entityMap.get(rel.source_entity_id);
          const targetId = entityMap.get(rel.target_entity_id);
          
          if (sourceId && targetId) {
            links.push({
              source: sourceId,
              target: targetId,
              type: rel.relationship_type,
              weight: rel.weight || 0.5,
              strength: rel.weight || 0.5,
              description: rel.description || '',
              // 엔티티 이름도 추가로 저장
              source_name: rel.source_name || '',
              target_name: rel.target_name || ''
            });
          } else {
            // entityMap에서 찾지 못한 경우, 백엔드의 엔티티 이름 사용
            console.log(`관계에서 누락된 엔티티: ${rel.source_name} (${rel.source_entity_id}) -> ${rel.target_name} (${rel.target_entity_id})`);
          }
        });
      }

      // communities 처리
      if (result.data.communities && Array.isArray(result.data.communities)) {
        const communities = result.data.communities.map((comm: any) => ({
          id: comm.id,
          label: comm.title || `Community ${comm.id}`,
          nodes: comm.entities || [],
          size: comm.entities?.length || 0
        }));

        // 커뮤니티 정보로 노드 업데이트
        communities.forEach(community => {
          community.nodes.forEach((entityId: any) => {
            const node = nodes.find(n => n.id === entityId.toString());
            if (node) {
              node.community = community.id;
            }
          });
        });

        return {
          nodes,
          links,
          communities,
          stats: {
            total_nodes: nodes.length,
            total_links: links.length,
            total_communities: communities.length,
            entity_types: [...new Set(nodes.map(n => n.type))],
            relationship_types: [...new Set(links.map(l => l.type))]
          }
        };
      }
    }

    // 기본 커뮤니티 생성 (communities 데이터가 없는 경우)
    const communities: any[] = [];
    const communityMap = new Map();
    
    nodes.forEach(node => {
      const commId = node.community;
      if (!communityMap.has(commId)) {
        const community = {
          id: commId,
          label: `Community ${commId}`,
          nodes: [],
          size: 0
        };
        communityMap.set(commId, community);
        communities.push(community);
      }
      communityMap.get(commId).nodes.push(node.id);
      communityMap.get(commId).size++;
    });

    return {
      nodes,
      links,
      communities,
      stats: {
        total_nodes: nodes.length,
        total_links: links.length,
        total_communities: communities.length,
        entity_types: [...new Set(nodes.map(n => n.type))],
        relationship_types: [...new Set(links.map(l => l.type))]
      }
    };
  };

  // 질의 기반 데이터 생성 (API 실패 시 폴백)
  const generateQueryBasedData = (searchQuery?: string): GraphData => {
    if (!searchQuery) {
      return generateSampleData(searchQuery);
    }

    // 조사 제거를 위한 키워드 추출 (백엔드와 동일한 방식)
    const particles = ['은', '는', '이', '가', '을', '를', '의', '에', '에게', '께', '에서', '으로', '로', '과', '와'];
    const stopWords = ['는', '은', '이', '가', '을', '를', '의', '에', '와', '과', '에서', '그리고', '그런데', '하지만'];

    const rawWords = searchQuery.toLowerCase().split(/[\s,\?\.!]+/).filter(word => word.length > 1);
    const keywords = rawWords.map(word => {
      // 조사 제거
      for (const particle of particles) {
        if (word.endsWith(particle)) {
          const baseWord = word.slice(0, -particle.length);
          if (baseWord.length >= 2) {
            return baseWord;
          }
        }
      }
      return word;
    }).filter(word => !stopWords.includes(word) && word.length >= 2);

    const nodes: any[] = [];
    const links: any[] = [];

    // 중심 질의 노드
    const queryNode = {
      id: 'query',
      label: searchQuery.length > 15 ? searchQuery.substring(0, 15) + '...' : searchQuery,
      type: 'query',
      description: `사용자 질의: ${searchQuery}`,
      importance: 1.0,
      community: 0,
      documents: [],
      highlighted: true
    };
    nodes.push(queryNode);

    // 키워드 기반 엔티티 노드 생성
    keywords.slice(0, 6).forEach((keyword, index) => {
      const node = {
        id: `keyword_${index}`,
        label: keyword,
        type: keyword.match(/\d/) ? 'number' : 'concept',
        description: `키워드: ${keyword}`,
        importance: 0.7 - (index * 0.1),
        community: Math.floor(index / 2) + 1,
        documents: [`관련문서_${keyword}.pdf`],
        highlighted: false
      };
      nodes.push(node);

      // 질의 노드와 연결
      links.push({
        source: 'query',
        target: `keyword_${index}`,
        type: 'contains',
        weight: 0.8 - (index * 0.1),
        strength: 0.7
      });
    });

    // 키워드 간 연결 생성
    for (let i = 0; i < Math.min(keywords.length - 1, 5); i++) {
      if (i + 1 < keywords.length) {
        links.push({
          source: `keyword_${i}`,
          target: `keyword_${i + 1}`,
          type: 'related',
          weight: 0.5,
          strength: 0.4
        });
      }
    }

    return {
      nodes,
      links,
      communities: [
        { id: 0, label: 'Query', nodes: ['query'], size: 1 },
        { id: 1, label: 'Keywords 1', nodes: keywords.slice(0, 2).map((_, i) => `keyword_${i}`), size: 2 },
        { id: 2, label: 'Keywords 2', nodes: keywords.slice(2, 4).map((_, i) => `keyword_${i + 2}`), size: 2 },
        { id: 3, label: 'Keywords 3', nodes: keywords.slice(4, 6).map((_, i) => `keyword_${i + 4}`), size: 2 }
      ].filter(c => c.size > 0),
      stats: {
        total_nodes: nodes.length,
        total_links: links.length,
        total_communities: Math.ceil(keywords.length / 2) + 1,
        entity_types: ['query', 'concept', 'number'],
        relationship_types: ['contains', 'related']
      }
    };
  };

  // 샘플 데이터 생성
  const generateSampleData = (searchQuery?: string): GraphData => {
    const sampleNodes = [
      { id: 'ai', label: 'Artificial Intelligence', type: 'technology', importance: 1.0, community: 0 },
      { id: 'ml', label: 'Machine Learning', type: 'technology', importance: 0.9, community: 0 },
      { id: 'python', label: 'Python', type: 'technology', importance: 0.7, community: 1 },
      { id: 'data', label: 'Data Science', type: 'concept', importance: 0.8, community: 0 },
      { id: 'algorithm', label: 'Algorithms', type: 'concept', importance: 0.6, community: 1 },
      { id: 'neural', label: 'Neural Networks', type: 'technology', importance: 0.85, community: 0 }
    ].map(node => ({
      ...node,
      description: `Sample description for ${node.label}`,
      highlighted: searchQuery ? node.label.toLowerCase().includes(searchQuery.toLowerCase()) : false,
      documents: [`document_${node.id}.pdf`]
    }));

    const sampleLinks = [
      { source: 'ai', target: 'ml', type: 'includes', weight: 0.9, strength: 0.8 },
      { source: 'ai', target: 'neural', type: 'uses', weight: 0.8, strength: 0.7 },
      { source: 'ml', target: 'python', type: 'implemented_in', weight: 0.7, strength: 0.6 },
      { source: 'ml', target: 'data', type: 'processes', weight: 0.85, strength: 0.75 },
      { source: 'python', target: 'algorithm', type: 'implements', weight: 0.6, strength: 0.5 },
      { source: 'neural', target: 'algorithm', type: 'based_on', weight: 0.7, strength: 0.6 }
    ];

    return {
      nodes: sampleNodes,
      links: sampleLinks,
      communities: [
        { id: 0, label: 'AI & ML', nodes: ['ai', 'ml', 'data', 'neural'], size: 4 },
        { id: 1, label: 'Programming', nodes: ['python', 'algorithm'], size: 2 }
      ],
      stats: {
        total_nodes: sampleNodes.length,
        total_links: sampleLinks.length,
        total_communities: 2,
        entity_types: ['technology', 'concept'],
        relationship_types: ['includes', 'uses', 'implemented_in', 'processes', 'implements', 'based_on']
      }
    };
  };

  // 모달이 열릴 때 데이터 로드
  useEffect(() => {
    if (isOpen) {
      loadGraphData();
    }
  }, [isOpen, query, userId, viewMode]);

  // ESC 키로 모달 닫기
  useEffect(() => {
    const handleKeyDown = (event: KeyboardEvent) => {
      if (event.key === 'Escape' && isOpen) {
        onClose();
      }
    };

    document.addEventListener('keydown', handleKeyDown);
    return () => document.removeEventListener('keydown', handleKeyDown);
  }, [isOpen, onClose]);

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 z-50 overflow-hidden">
      {/* 배경 오버레이 */}
      <div 
        className="absolute inset-0 bg-black bg-opacity-50 dark:bg-black dark:bg-opacity-70 transition-opacity"
        onClick={onClose}
      />
      
      {/* 모달 컨텐츠 */}
      <div className="relative z-10 flex min-h-full items-center justify-center p-4">
        <div 
          className="relative w-full max-w-7xl rounded-lg shadow-xl"
          style={{
            backgroundColor: 'var(--card-bg)',
            border: '1px solid var(--card-border)'
          }}
        >
          {/* 헤더 */}
          <div 
            className="flex items-center justify-between p-6 border-b"
            style={{ borderBottomColor: 'var(--card-border)' }}
          >
            <div>
              <h2 
                className="text-xl font-semibold"
                style={{ color: 'var(--text-primary)' }}
              >
                지식 그래프
              </h2>
              {query && (
                <p 
                  className="text-sm mt-1"
                  style={{ color: 'var(--text-secondary)' }}
                >
                  {(() => {
                    // JSON 청크 정보인지 확인 (문자열이 {로 시작하고 JSON 파싱이 가능한 경우만)
                    if (query && query.trim().startsWith('{')) {
                      try {
                        const parsed = JSON.parse(query);
                        if (parsed.type === 'chunk') {
                          return `문서 청크: "${parsed.filename}" (${parsed.chunk_index + 1}번째)`;
                        } else if (parsed.type === 'chunk_content' && parsed.chunk_content) {
                          const truncatedContent = parsed.chunk_content.length > 50 
                            ? parsed.chunk_content.substring(0, 50) + '...' 
                            : parsed.chunk_content;
                          return `청크 내용: "${truncatedContent}"`;
                        }
                      } catch {
                        // JSON 파싱 실패시 일반 쿼리로 처리 (아래로 이동)
                      }
                    }

                    // 일반 사용자 쿼리 처리 (채팅 페이지에서 오는 경우)
                    const truncatedQuery = query && query.length > 50 
                      ? query.substring(0, 50) + '...' 
                      : query;
                    return `질의: "${truncatedQuery}"`;
                  })()}
                </p>
              )}
            </div>
            
            {/* 뷰 모드 선택 */}
            <div className="flex items-center gap-4">
              <div 
                className="flex rounded-lg border"
                style={{ borderColor: 'var(--card-border)' }}
              >
                <button
                  onClick={() => setViewMode('overview')}
                  className="px-3 py-1 text-sm rounded-l-lg transition-colors"
                  style={{
                    backgroundColor: viewMode === 'overview' ? 'var(--btn-primary-bg)' : 'var(--card-bg)',
                    color: viewMode === 'overview' ? 'var(--btn-primary-fg)' : 'var(--text-primary)',
                  }}
                  onMouseEnter={(e) => {
                    if (viewMode !== 'overview') {
                      e.currentTarget.style.backgroundColor = 'var(--card-hover)';
                    }
                  }}
                  onMouseLeave={(e) => {
                    if (viewMode !== 'overview') {
                      e.currentTarget.style.backgroundColor = 'var(--card-bg)';
                    }
                  }}
                >
                  전체
                </button>
                <button
                  onClick={() => setViewMode('query-focused')}
                  className="px-3 py-1 text-sm transition-colors"
                  style={{
                    backgroundColor: viewMode === 'query-focused' ? 'var(--btn-primary-bg)' : 'var(--card-bg)',
                    color: viewMode === 'query-focused' ? 'var(--btn-primary-fg)' : 'var(--text-primary)',
                  }}
                  onMouseEnter={(e) => {
                    if (viewMode !== 'query-focused') {
                      e.currentTarget.style.backgroundColor = 'var(--card-hover)';
                    }
                  }}
                  onMouseLeave={(e) => {
                    if (viewMode !== 'query-focused') {
                      e.currentTarget.style.backgroundColor = 'var(--card-bg)';
                    }
                  }}
                >
                  질의 중심
                </button>
                <button
                  onClick={() => setViewMode('entity-focused')}
                  className="px-3 py-1 text-sm rounded-r-lg transition-colors"
                  style={{
                    backgroundColor: viewMode === 'entity-focused' ? 'var(--btn-primary-bg)' : 'var(--card-bg)',
                    color: viewMode === 'entity-focused' ? 'var(--btn-primary-fg)' : 'var(--text-primary)',
                  }}
                  onMouseEnter={(e) => {
                    if (viewMode !== 'entity-focused') {
                      e.currentTarget.style.backgroundColor = 'var(--card-hover)';
                    }
                  }}
                  onMouseLeave={(e) => {
                    if (viewMode !== 'entity-focused') {
                      e.currentTarget.style.backgroundColor = 'var(--card-bg)';
                    }
                  }}
                >
                  엔티티 중심
                </button>
              </div>

              <button
                onClick={onClose}
                className="p-1 transition-colors"
                style={{ color: 'var(--text-muted)' }}
                onMouseEnter={(e) => {
                  e.currentTarget.style.color = 'var(--text-secondary)';
                }}
                onMouseLeave={(e) => {
                  e.currentTarget.style.color = 'var(--text-muted)';
                }}
              >
                <XMarkIcon className="h-6 w-6" />
              </button>
            </div>
          </div>

          {/* 뷰 모드 설명 */}
          <div
            className="px-6 py-3 border-b text-sm"
            style={{
              backgroundColor: 'var(--header-bg)',
              borderBottomColor: 'var(--card-border)',
              color: 'var(--text-secondary)'
            }}
          >
            {viewMode === 'overview' && (
              <div className="flex items-center gap-2">
                <span className="font-medium" style={{ color: 'var(--text-primary)' }}>🌐 전체 보기:</span>
                <span>각 커뮤니티의 핵심 엔티티들을 균형있게 표시하여 전체 지식 구조를 파악합니다</span>
              </div>
            )}
            {viewMode === 'query-focused' && (
              <div className="flex items-center gap-2">
                <span className="font-medium" style={{ color: 'var(--text-primary)' }}>🎯 질의 중심:</span>
                <span>사용자 질문과 관련된 커뮤니티만 선별하여 연결된 지식 그래프를 표시합니다</span>
              </div>
            )}
            {viewMode === 'entity-focused' && (
              <div className="flex items-center gap-2">
                <span className="font-medium" style={{ color: 'var(--text-primary)' }}>⭐ 엔티티 중심:</span>
                <span>중요도가 높은 핵심 엔티티들과 그들 간의 강한 연결 관계를 중심으로 표시합니다</span>
              </div>
            )}
          </div>

          {/* 컨텐츠 */}
          <div className="p-6">
            {loading && (
              <div className="flex items-center justify-center h-96">
                <div className="text-center">
                  <div 
                    className="animate-spin rounded-full h-12 w-12 border-b-2 mx-auto mb-4"
                    style={{ borderBottomColor: 'var(--btn-primary-bg)' }}
                  ></div>
                  <div style={{ color: 'var(--text-secondary)' }}>GraphRAG 데이터 로딩 중...</div>
                </div>
              </div>
            )}

            {error && (
              <div 
                className="border-l-4 p-4 mb-4"
                style={{ 
                  backgroundColor: 'var(--warning-bg)', 
                  borderLeftColor: 'var(--warning-border)' 
                }}
              >
                <div 
                  className="text-sm"
                  style={{ color: 'var(--warning)' }}
                >
                  {error}
                </div>
              </div>
            )}

            {graphData && !loading && (
              <div className="relative">
                <GraphRAGVisualization
                  data={graphData}
                  width={1000}
                  height={600}
                  userId={userId}
                  query={query}
                  viewMode={viewMode}
                  onNodeClick={(node) => {
                    console.log('노드 클릭:', node);
                    setSelectedEntity(node.id);
                  }}
                  onLinkClick={(link) => {
                    console.log('링크 클릭:', link);
                  }}
                  className="rounded-lg"
                  style={{ border: '1px solid var(--card-border)' }}
                />
              </div>
            )}
          </div>

          {/* 푸터 */}
          <div 
            className="flex items-center justify-between p-6 border-t"
            style={{ 
              borderTopColor: 'var(--card-border)',
              backgroundColor: 'var(--header-bg)' 
            }}
          >
            <div 
              className="text-sm"
              style={{ color: 'var(--text-secondary)' }}
            >
              Graph RAG Visualization
            </div>
            <div className="flex gap-2">
              <button
                onClick={loadGraphData}
                disabled={loading}
                className="px-4 py-2 text-sm rounded-lg transition-colors disabled:opacity-50"
                style={{
                  backgroundColor: 'var(--btn-primary-bg)',
                  color: 'var(--btn-primary-fg)'
                }}
                onMouseEnter={(e) => {
                  if (!loading) {
                    e.currentTarget.style.backgroundColor = 'var(--btn-primary-hover)';
                  }
                }}
                onMouseLeave={(e) => {
                  if (!loading) {
                    e.currentTarget.style.backgroundColor = 'var(--btn-primary-bg)';
                  }
                }}
              >
                새로고침
              </button>
              <button
                onClick={onClose}
                className="px-4 py-2 text-sm rounded-lg transition-colors"
                style={{
                  backgroundColor: 'var(--text-muted)',
                  color: 'white'
                }}
                onMouseEnter={(e) => {
                  e.currentTarget.style.backgroundColor = 'var(--text-secondary)';
                }}
                onMouseLeave={(e) => {
                  e.currentTarget.style.backgroundColor = 'var(--text-muted)';
                }}
              >
                닫기
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default GraphRAGModal;
