'use client';

import React, { useEffect, useRef, useState, useCallback, useMemo } from 'react';
import * as d3 from 'd3';

interface GraphNode {
  id: string;
  label: string;
  type: string;
  description?: string;
  importance?: number;
  community?: number;
  documents?: string[];
  highlighted?: boolean;
  x?: number;
  y?: number;
  fx?: number | null;
  fy?: number | null;
}

interface GraphLink {
  source: string | GraphNode;
  target: string | GraphNode;
  type: string;
  weight?: number;
  description?: string;
  strength?: number;
}

interface Community {
  id: number;
  label: string;
  nodes: string[];
  size: number;
}

interface GraphData {
  nodes: GraphNode[];
  links: GraphLink[];
  communities: Community[];
  stats: {
    total_nodes: number;
    total_links: number;
    total_communities: number;
    entity_types: string[];
    relationship_types: string[];
  };
}

interface GraphRAGVisualizationProps {
  data?: GraphData;
  width?: number;
  height?: number;
  userId?: string;
  documentName?: string;
  query?: string;
  viewMode?: 'overview' | 'query-focused' | 'entity-focused';
  onNodeClick?: (node: GraphNode) => void;
  onLinkClick?: (link: GraphLink) => void;
  className?: string;
}

const GraphRAGVisualization: React.FC<GraphRAGVisualizationProps> = ({
  data,
  width = 800,
  height = 600,
  userId,
  documentName,
  query,
  viewMode = 'overview',
  onNodeClick,
  onLinkClick,
  className = ''
}) => {
  const svgRef = useRef<SVGSVGElement>(null);
  const [selectedNode, setSelectedNode] = useState<GraphNode | null>(null);
  const [hoveredNode, setHoveredNode] = useState<GraphNode | null>(null);
  const [zoomLevel, setZoomLevel] = useState(1);

  // CSS 변수를 읽어오는 유틸리티 함수
  const getCSSVariable = useCallback((variableName: string): string => {
    if (typeof window !== 'undefined') {
      return getComputedStyle(document.documentElement).getPropertyValue(variableName).trim();
    }
    return '';
  }, []);

  // 다크모드 감지
  const [isDarkMode, setIsDarkMode] = useState(false);
  
  useEffect(() => {
    const checkDarkMode = () => {
      setIsDarkMode(document.documentElement.classList.contains('dark'));
    };
    
    checkDarkMode();
    
    // 다크모드 변경 감지
    const observer = new MutationObserver(checkDarkMode);
    observer.observe(document.documentElement, {
      attributes: true,
      attributeFilter: ['class']
    });
    
    return () => observer.disconnect();
  }, []);

  // Microsoft GraphRAG 표준 엔티티 타입별 색상 매핑 (CSS 변수 기반)
  const getNodeColor = useCallback((type: string, community?: number) => {
    // 다크모드/라이트모드에 따른 기본 색상 팔레트
    const getTypeColor = (colorVar: string, fallback: string) => {
      const color = getCSSVariable(colorVar);
      return color || fallback;
    };

    const typeColors: { [key: string]: string } = {
      // CSS 변수를 사용한 색상 매핑
      'person': getTypeColor('--error-500', '#ff6b6b'),           // 인물
      'organization': getTypeColor('--info-500', '#4ecdc4'),      // 조직  
      'location': getTypeColor('--primary-500', '#45b7d1'),       // 위치
      'geo': getTypeColor('--primary-500', '#45b7d1'),           // 지리적 위치
      'event': getTypeColor('--warning-500', '#ff6348'),         // 이벤트
      'technology': getTypeColor('--success-500', '#96ceb4'),     // 기술
      'product': getTypeColor('--warning-400', '#feca57'),       // 제품
      'concept': getTypeColor('--secondary-500', '#ff9ff3'),     // 개념
      'claim': getTypeColor('--secondary-600', '#9b59b6'),       // 주장/클레임
      'document': getTypeColor('--warning-600', '#f39c12'),      // 문서
      'unknown': getCSSVariable('--text-muted') || '#95a5a6'     // 미분류
    };

    const baseColor = typeColors[type.toLowerCase()] || typeColors['unknown'];

    // Leiden 알고리즘 기반 커뮤니티별 색상 조정 (Microsoft GraphRAG 표준)
    if (community !== undefined) {
      // 커뮤니티별로 색조 회전하되 채도와 명도는 다크모드에 따라 조정
      const baseHsl = d3.hsl(baseColor);
      const hue = (baseHsl.h + (community * 45)) % 360;
      const saturation = isDarkMode ? 0.6 : 0.8;
      const lightness = isDarkMode ? 0.7 : 0.65;
      return d3.hsl(hue, saturation, lightness).toString();
    }

    return baseColor;
  }, [getCSSVariable, isDarkMode]);

  // Microsoft GraphRAG 표준 관계 타입별 색상 매핑 (CSS 변수 기반)
  const getLinkColor = useCallback((type: string, weight?: number) => {
    const getLinkTypeColor = (colorVar: string, fallback: string) => {
      const color = getCSSVariable(colorVar);
      return color || fallback;
    };

    const linkColors: { [key: string]: string } = {
      // 표준 관계 타입 - CSS 변수 기반
      'related': getLinkTypeColor('--text-muted', '#bdc3c7'),           // 일반적 관련성
      'works_for': getLinkTypeColor('--info-500', '#3498db'),           // 근무 관계
      'located_in': getLinkTypeColor('--error-500', '#e74c3c'),         // 위치 관계
      'part_of': getLinkTypeColor('--warning-500', '#f39c12'),          // 소속 관계
      'owns': getLinkTypeColor('--purple-500', '#9b59b6'),              // 소유 관계
      'created': getLinkTypeColor('--success-500', '#1abc9c'),          // 생성 관계
      'uses': getLinkTypeColor('--text-secondary', '#34495e'),          // 사용 관계
      'mentions': getLinkTypeColor('--text-muted', '#95a5a6'),          // 언급 관계
      'collaborates_with': getLinkTypeColor('--success-600', '#27ae60'), // 협력 관계
      'connected_to': getLinkTypeColor('--purple-600', '#8e44ad'),      // 연결 관계
      'supports': getLinkTypeColor('--info-600', '#16a085'),            // 지원 관계
      'opposes': getLinkTypeColor('--error-600', '#c0392b')             // 반대 관계
    };

    const baseColor = linkColors[type.toLowerCase()] || linkColors['related'];

    // 관계 강도(weight)에 따른 투명도 조정
    if (weight !== undefined) {
      return d3.color(baseColor)?.copy({ opacity: Math.max(0.3, weight) }).toString() || baseColor;
    }

    return baseColor;
  }, [getCSSVariable]);

  // 노드 크기 계산 (Microsoft GraphRAG 표준: 관계 수 기반)
  const getNodeSize = useCallback((importance: number = 1, relationshipCount?: number) => {
    // 기본적으로 중요도 기반 크기 계산
    let baseSize = Math.max(8, Math.min(24, 10 + importance * 12));

    // 관계 수가 제공된 경우 이를 반영 (Microsoft GraphRAG 표준)
    if (relationshipCount !== undefined) {
      const relationshipFactor = Math.min(2, Math.log10(relationshipCount + 1) / 2);
      baseSize = Math.max(8, Math.min(30, baseSize * (1 + relationshipFactor)));
    }

    return baseSize;
  }, []);

  // 뷰 모드에 따른 데이터 필터링
  const filteredData = useMemo(() => {
    if (!data) return null;

    let filteredNodes = [...data.nodes];
    let filteredLinks = [...data.links];

    switch (viewMode) {
      case 'query-focused':
        if (query) {
          // Microsoft GraphRAG 공식 방식: 커뮤니티 기반 필터링
          const particles = [
            '은', '는', '이', '가', '을', '를', '의', '에', '에서', '로', '으로',
            '와', '과', '도', '만', '부터', '까지', '에게', '한테', '께', '더러',
            '라', '이라', '라서', '니까', '하고', '랑'
          ];

          const queryWords = query.toLowerCase().split(/\s+/);
          const cleanedQueryWords = queryWords.map(word => {
            for (const particle of particles) {
              if (word.endsWith(particle)) {
                const baseWord = word.slice(0, -particle.length);
                if (baseWord.length >= 2) return baseWord;
              }
            }
            return word;
          }).filter(word => word.length >= 2);

          // 1단계: 질의 관련 노드 찾기 (커뮤니티 중심)
          let relevantNodes = data.nodes.filter(node => {
            const keywordMatch = [...queryWords, ...cleanedQueryWords].some(word =>
              node.label.toLowerCase().includes(word) ||
              node.description?.toLowerCase().includes(word) ||
              node.type.toLowerCase().includes(word)
            );
            return keywordMatch || node.highlighted;
          });

          // 2단계: 관련 노드들의 커뮤니티 추출
          const relevantCommunities = new Set(relevantNodes.map(n => n.community).filter(c => c !== undefined));

          if (relevantCommunities.size > 0) {
            // Microsoft GraphRAG 방식: 커뮤니티 전체를 포함
            const communityNodes = data.nodes.filter(node =>
              relevantCommunities.has(node.community)
            );

            // 3단계: 커뮤니티 내 연결된 컴포넌트 분석
            const communityNodeIds = new Set(communityNodes.map(n => n.id));
            const communityLinks = data.links.filter(link => {
              const sourceId = typeof link.source === 'string' ? link.source : link.source.id;
              const targetId = typeof link.target === 'string' ? link.target : link.target.id;
              return communityNodeIds.has(sourceId) && communityNodeIds.has(targetId);
            });

            // Union-Find로 가장 큰 연결 컴포넌트 찾기
            const parent = new Map<string, string>();
            communityNodes.forEach(node => parent.set(node.id, node.id));

            function find(x: string): string {
              if (parent.get(x) !== x) {
                parent.set(x, find(parent.get(x)!));
              }
              return parent.get(x)!;
            }

            function union(x: string, y: string): void {
              const rootX = find(x);
              const rootY = find(y);
              if (rootX !== rootY) {
                parent.set(rootY, rootX);
              }
            }

            communityLinks.forEach(link => {
              const sourceId = typeof link.source === 'string' ? link.source : link.source.id;
              const targetId = typeof link.target === 'string' ? link.target : link.target.id;
              union(sourceId, targetId);
            });

            // 가장 큰 연결 컴포넌트 선택
            const componentSizes = new Map<string, number>();
            communityNodes.forEach(node => {
              const root = find(node.id);
              componentSizes.set(root, (componentSizes.get(root) || 0) + 1);
            });

            let largestComponentRoot = '';
            let maxSize = 0;
            componentSizes.forEach((size, root) => {
              if (size > maxSize) {
                maxSize = size;
                largestComponentRoot = root;
              }
            });

            // 4단계: 최종 노드 선택 (Microsoft GraphRAG 방식)
            if (largestComponentRoot && maxSize >= 3) {
              // 가장 큰 연결 컴포넌트 선택
              filteredNodes = communityNodes.filter(node => find(node.id) === largestComponentRoot);
            } else if (relevantCommunities.size === 1) {
              // 단일 커뮤니티: 중요도 상위 노드 선택
              filteredNodes = communityNodes
                .sort((a, b) => (b.importance || 0) - (a.importance || 0))
                .slice(0, Math.min(15, communityNodes.length));
            } else {
              // 다중 커뮤니티: 각 커뮤니티에서 상위 3개씩 선택
              const nodesByCommunity = new Map<number, GraphNode[]>();
              communityNodes.forEach(node => {
                if (node.community !== undefined) {
                  if (!nodesByCommunity.has(node.community)) {
                    nodesByCommunity.set(node.community, []);
                  }
                  nodesByCommunity.get(node.community)!.push(node);
                }
              });

              filteredNodes = [];
              nodesByCommunity.forEach(nodes => {
                const topNodes = nodes
                  .sort((a, b) => (b.importance || 0) - (a.importance || 0))
                  .slice(0, 3);
                filteredNodes.push(...topNodes);
              });
            }
          } else {
            // 커뮤니티 정보가 없는 경우: 중요도 기반 선택
            filteredNodes = relevantNodes
              .sort((a, b) => (b.importance || 0) - (a.importance || 0))
              .slice(0, Math.min(10, relevantNodes.length));
          }

          // 최종 링크 필터링
          const finalNodeIds = new Set(filteredNodes.map(n => n.id));
          filteredLinks = data.links.filter(link => {
            const sourceId = typeof link.source === 'string' ? link.source : link.source.id;
            const targetId = typeof link.target === 'string' ? link.target : link.target.id;
            return finalNodeIds.has(sourceId) && finalNodeIds.has(targetId);
          });
        }
        break;
        
      case 'entity-focused':
        // Microsoft GraphRAG 방식: 연결성과 중요도 모두 고려
        // 1단계: 높은 중요도 노드들 선별 (상위 50%)
        const sortedByImportance = [...data.nodes].sort((a, b) => (b.importance || 0) - (a.importance || 0));
        const importantNodes = sortedByImportance.slice(0, Math.max(10, Math.floor(sortedByImportance.length * 0.5)));

        // 2단계: 중요 노드들과 연결된 링크 찾기
        const importantNodeIds = new Set(importantNodes.map(n => n.id));
        const importantLinks = data.links.filter(link => {
          const sourceId = typeof link.source === 'string' ? link.source : link.source.id;
          const targetId = typeof link.target === 'string' ? link.target : link.target.id;
          return importantNodeIds.has(sourceId) && importantNodeIds.has(targetId);
        });

        // 3단계: 가장 큰 연결 컴포넌트 찾기
        if (importantLinks.length > 0) {
          const parent = new Map<string, string>();
          importantNodes.forEach(node => parent.set(node.id, node.id));

          function find(x: string): string {
            if (parent.get(x) !== x) {
              parent.set(x, find(parent.get(x)!));
            }
            return parent.get(x)!;
          }

          function union(x: string, y: string): void {
            const rootX = find(x);
            const rootY = find(y);
            if (rootX !== rootY) {
              parent.set(rootY, rootX);
            }
          }

          importantLinks.forEach(link => {
            const sourceId = typeof link.source === 'string' ? link.source : link.source.id;
            const targetId = typeof link.target === 'string' ? link.target : link.target.id;
            union(sourceId, targetId);
          });

          const componentSizes = new Map<string, number>();
          importantNodes.forEach(node => {
            const root = find(node.id);
            componentSizes.set(root, (componentSizes.get(root) || 0) + 1);
          });

          let largestComponentRoot = '';
          let maxSize = 0;
          componentSizes.forEach((size, root) => {
            if (size > maxSize) {
              maxSize = size;
              largestComponentRoot = root;
            }
          });

          // 가장 큰 연결 컴포넌트 선택
          if (largestComponentRoot && maxSize >= 5) {
            filteredNodes = importantNodes.filter(node => find(node.id) === largestComponentRoot);
          } else {
            // 연결된 컴포넌트가 작으면 상위 10개 노드만
            filteredNodes = importantNodes.slice(0, 10);
          }
        } else {
          // 링크가 없으면 상위 10개 노드만
          filteredNodes = importantNodes.slice(0, 10);
        }

        // 최종 링크 필터링
        const entityNodeIds = new Set(filteredNodes.map(n => n.id));
        filteredLinks = data.links.filter(link => {
          const sourceId = typeof link.source === 'string' ? link.source : link.source.id;
          const targetId = typeof link.target === 'string' ? link.target : link.target.id;
          return entityNodeIds.has(sourceId) && entityNodeIds.has(targetId);
        });
        break;

      case 'overview':
      default:
        // Microsoft GraphRAG 방식: 커뮤니티 기반 전체 보기
        // 커뮤니티별로 상위 노드들 선택하여 최대 50개로 제한
        const communitiesMap = new Map<number, GraphNode[]>();
        data.nodes.forEach(node => {
          const community = node.community ?? 0;
          if (!communitiesMap.has(community)) {
            communitiesMap.set(community, []);
          }
          communitiesMap.get(community)!.push(node);
        });

        // 각 커뮤니티에서 상위 노드들 선택
        filteredNodes = [];
        const maxNodesPerCommunity = Math.max(3, Math.floor(50 / communitiesMap.size));

        communitiesMap.forEach(nodes => {
          const topCommunityNodes = nodes
            .sort((a, b) => (b.importance || 0) - (a.importance || 0))
            .slice(0, maxNodesPerCommunity);
          filteredNodes.push(...topCommunityNodes);
        });

        // 전체 노드 수 제한 (50개)
        if (filteredNodes.length > 50) {
          filteredNodes = filteredNodes
            .sort((a, b) => (b.importance || 0) - (a.importance || 0))
            .slice(0, 50);
        }

        // 링크 필터링
        const overviewNodeIds = new Set(filteredNodes.map(n => n.id));
        filteredLinks = data.links.filter(link => {
          const sourceId = typeof link.source === 'string' ? link.source : link.source.id;
          const targetId = typeof link.target === 'string' ? link.target : link.target.id;
          return overviewNodeIds.has(sourceId) && overviewNodeIds.has(targetId);
        });
        break;
    }

    return {
      ...data,
      nodes: filteredNodes,
      links: filteredLinks,
      stats: {
        ...data.stats,
        total_nodes: filteredNodes.length,
        total_links: filteredLinks.length
      }
    };
  }, [data, viewMode, query]);

  // D3.js 시각화 초기화 및 업데이트
  useEffect(() => {
    if (!filteredData || !svgRef.current) return;

    const svg = d3.select(svgRef.current);
    svg.selectAll("*").remove(); // 기존 요소 제거

    // SVG 크기 설정
    svg.attr("width", width).attr("height", height);

    // 줌 동작 설정
    const zoom = d3.zoom<SVGSVGElement, unknown>()
      .scaleExtent([0.1, 4])
      .on("zoom", (event) => {
        container.attr("transform", event.transform);
        setZoomLevel(event.transform.k);
      });

    svg.call(zoom);

    // 컨테이너 그룹 생성
    const container = svg.append("g").attr("class", "graph-container");

    // 화살표 마커 정의
    const defs = svg.append("defs");
    
    defs.append("marker")
      .attr("id", "arrowhead")
      .attr("viewBox", "0 -5 10 10")
      .attr("refX", 20)
      .attr("refY", 0)
      .attr("markerWidth", 6)
      .attr("markerHeight", 6)
      .attr("orient", "auto")
      .append("path")
      .attr("d", "M0,-5L10,0L0,5")
      .attr("fill", getCSSVariable('--text-muted') || '#999');

    // 포스 시뮬레이션 설정
    const simulation = d3.forceSimulation<GraphNode>(filteredData.nodes)
      .force("link", d3.forceLink<GraphNode, GraphLink>(filteredData.links)
        .id(d => d.id)
        .distance(d => 100 / (d.strength || 0.5))
        .strength(d => d.strength || 0.5)
      )
      .force("charge", d3.forceManyBody().strength(-300))
      .force("center", d3.forceCenter(width / 2, height / 2))
      .force("collision", d3.forceCollide().radius(d => {
        const relationshipCount = filteredData.links.filter(link =>
          (typeof link.source === 'string' ? link.source : link.source.id) === d.id ||
          (typeof link.target === 'string' ? link.target : link.target.id) === d.id
        ).length;
        return getNodeSize(d.importance, relationshipCount) + 8;
      }));

    // 링크 생성 (Microsoft GraphRAG 표준 스타일링)
    const link = container.append("g")
      .attr("class", "links")
      .selectAll("line")
      .data(filteredData.links)
      .enter().append("line")
      .attr("stroke", d => getLinkColor(d.type, d.weight))
      .attr("stroke-width", d => Math.max(1, Math.sqrt(d.weight || 1) * 2.5))
      .attr("stroke-opacity", d => Math.max(0.3, (d.weight || 0.5) * 0.8))
      .attr("marker-end", "url(#arrowhead)")
      .style("cursor", "pointer")
      .on("click", (event, d) => {
        event.stopPropagation();
        onLinkClick?.(d);
      });

    // 노드 그룹 생성
    const nodeGroup = container.append("g")
      .attr("class", "nodes")
      .selectAll("g")
      .data(filteredData.nodes)
      .enter().append("g")
      .attr("class", "node-group")
      .style("cursor", "pointer")
      .call(d3.drag<SVGGElement, GraphNode>()
        .on("start", (event, d) => {
          if (!event.active) simulation.alphaTarget(0.3).restart();
          d.fx = d.x;
          d.fy = d.y;
        })
        .on("drag", (event, d) => {
          d.fx = event.x;
          d.fy = event.y;
        })
        .on("end", (event, d) => {
          if (!event.active) simulation.alphaTarget(0);
          d.fx = null;
          d.fy = null;
        })
      );

    // 노드 원 생성 (Microsoft GraphRAG 표준: 관계 수 반영)
    const circles = nodeGroup.append("circle")
      .attr("r", d => {
        // 관계 수 계산 (연결된 링크 수)
        const relationshipCount = filteredData.links.filter(link =>
          (typeof link.source === 'string' ? link.source : link.source.id) === d.id ||
          (typeof link.target === 'string' ? link.target : link.target.id) === d.id
        ).length;
        return getNodeSize(d.importance, relationshipCount);
      })
      .attr("fill", d => getNodeColor(d.type, d.community))
      .attr("stroke", d => d.highlighted ? (getCSSVariable('--error-500') || '#ff6b6b') : (getCSSVariable('--card-bg') || '#fff'))
      .attr("stroke-width", d => d.highlighted ? 3 : 2)
      .attr("opacity", 0.9);

    // 노드 레이블 생성 (Microsoft GraphRAG 표준)
    const labels = nodeGroup.append("text")
      .text(d => d.label.length > 12 ? d.label.substring(0, 12) + "..." : d.label)
      .attr("font-size", d => Math.max(9, Math.min(12, getNodeSize(d.importance) * 0.6)))
      .attr("font-family", "system-ui, -apple-system, sans-serif")
      .attr("text-anchor", "middle")
      .attr("dy", d => {
        const relationshipCount = filteredData.links.filter(link =>
          (typeof link.source === 'string' ? link.source : link.source.id) === d.id ||
          (typeof link.target === 'string' ? link.target : link.target.id) === d.id
        ).length;
        return getNodeSize(d.importance, relationshipCount) + 16;
      })
      .attr("fill", getCSSVariable('--text-primary') || '#2c3e50')
      .attr("font-weight", "500")
      .attr("pointer-events", "none");

    // 마우스 이벤트 처리
    nodeGroup
      .on("mouseover", (event, d) => {
        setHoveredNode(d);
        
        // 연결된 노드와 링크 하이라이트
        const connectedNodes = new Set<string>();
        connectedNodes.add(d.id);
        
        link.attr("stroke-opacity", l => {
          const isConnected = (l.source as GraphNode).id === d.id || (l.target as GraphNode).id === d.id;
          if (isConnected) {
            connectedNodes.add(typeof l.source === 'string' ? l.source : l.source.id);
            connectedNodes.add(typeof l.target === 'string' ? l.target : l.target.id);
            return 1;
          }
          return 0.1;
        });
        
        circles.attr("opacity", node => connectedNodes.has(node.id) ? 1 : 0.2);
        labels.attr("opacity", node => connectedNodes.has(node.id) ? 1 : 0.3);
      })
      .on("mouseout", () => {
        setHoveredNode(null);
        
        // 하이라이트 해제
        link.attr("stroke-opacity", 0.6);
        circles.attr("opacity", 0.8);
        labels.attr("opacity", 1);
      })
      .on("click", (event, d) => {
        event.stopPropagation();
        setSelectedNode(d);
        onNodeClick?.(d);
      });

    // 시뮬레이션 업데이트
    simulation.on("tick", () => {
      link
        .attr("x1", d => (d.source as GraphNode).x || 0)
        .attr("y1", d => (d.source as GraphNode).y || 0)
        .attr("x2", d => (d.target as GraphNode).x || 0)
        .attr("y2", d => (d.target as GraphNode).y || 0);

      nodeGroup
        .attr("transform", d => `translate(${d.x || 0},${d.y || 0})`);
    });

    // 컴포넌트 언마운트 시 시뮬레이션 중지
    return () => {
      simulation.stop();
    };

  }, [filteredData, width, height, getNodeColor, getLinkColor, getNodeSize, onNodeClick, onLinkClick, viewMode, query]);

  return (
    <div className={`graphrag-visualization ${className}`}>
      {/* 컨트롤 패널 */}
      <div 
        className="absolute top-4 left-4 p-3 rounded-lg shadow-lg z-10"
        style={{
          backgroundColor: getCSSVariable('--body-bg') || 'rgba(255, 255, 255, 0.9)',
          border: `1px solid ${getCSSVariable('--card-border') || '#e5e7eb'}`,
          color: getCSSVariable('--text-primary') || '#1f2937'
        }}
      >
        <div className="text-sm font-semibold mb-2">지식 그래프</div>
        {filteredData && (
          <div className="text-sm space-y-1">
            <div>노드: {filteredData.stats.total_nodes}개</div>
            <div>관계: {filteredData.stats.total_links}개</div>
            <div>커뮤니티: {filteredData.stats.total_communities}개</div>
            <div>줌: {(zoomLevel * 100).toFixed(0)}%</div>
            {viewMode !== 'overview' && (
              <div className="text-sm mt-1 opacity-75">
                모드: {viewMode === 'query-focused' ? '질의 중심' : '엔티티 중심'}
              </div>
            )}
          </div>
        )}
        
        {/* Microsoft GraphRAG 표준 범례 */}
        <div 
          className="mt-3 pt-2 border-t"
          style={{ borderTopColor: getCSSVariable('--card-border') || '#e5e7eb' }}
        >
          <div className="text-sm font-medium mb-2">엔티티 타입</div>
          <div className="grid grid-cols-1 gap-1 text-sm mb-3">
            {filteredData?.stats.entity_types.slice(0, 5).map(type => (
              <div key={type} className="flex items-center gap-2">
                <div
                  className="w-3 h-3 rounded-full border border-gray-300"
                  style={{ backgroundColor: getNodeColor(type) }}
                />
                <span className="truncate capitalize">{type}</span>
              </div>
            ))}
          </div>

          <div className="text-sm font-medium mb-1">관계 타입</div>
          <div className="grid grid-cols-1 gap-1 text-sm">
            {filteredData?.stats.relationship_types.slice(0, 4).map(relType => (
              <div key={relType} className="flex items-center gap-2">
                <div
                  className="w-4 h-0.5 rounded"
                  style={{ backgroundColor: getLinkColor(relType) }}
                />
                <span className="truncate text-sm">{relType.replace(/_/g, ' ')}</span>
              </div>
            ))}
          </div>

          <div 
            className="mt-2 pt-2 border-t text-sm"
            style={{ 
              borderTopColor: getCSSVariable('--card-border') || '#e5e7eb',
              color: getCSSVariable('--text-muted') || '#6b7280'
            }}
          >
            <div>• 노드 크기: 관계 수 반영</div>
            <div>• 색상: 커뮤니티 구분</div>
          </div>
        </div>
      </div>

      {/* 선택된 노드 정보 패널 */}
      {selectedNode && (
        <div 
          className="absolute top-4 right-4 p-4 rounded-lg shadow-lg z-10 max-w-sm"
          style={{
            backgroundColor: getCSSVariable('--body-bg') || 'rgba(255, 255, 255, 0.95)',
            border: `1px solid ${getCSSVariable('--card-border') || '#e5e7eb'}`,
            color: getCSSVariable('--text-primary') || '#1f2937'
          }}
        >
          <div className="flex justify-between items-start mb-2">
            <h3 className="font-semibold text-sm">{selectedNode.label}</h3>
            <button 
              onClick={() => setSelectedNode(null)}
              className="transition-colors"
              style={{ 
                color: getCSSVariable('--text-muted') || '#9ca3af',
              }}
              onMouseEnter={(e) => {
                e.currentTarget.style.color = getCSSVariable('--text-secondary') || '#6b7280';
              }}
              onMouseLeave={(e) => {
                e.currentTarget.style.color = getCSSVariable('--text-muted') || '#9ca3af';
              }}
            >
              ✕
            </button>
          </div>
          <div className="text-sm space-y-1">
            <div><span className="font-medium">타입:</span> {selectedNode.type}</div>
            <div><span className="font-medium">중요도:</span> {selectedNode.importance?.toFixed(2) || 'N/A'}</div>
            <div><span className="font-medium">커뮤니티:</span> {selectedNode.community || 0}</div>
            {selectedNode.description && (
              <div><span className="font-medium">설명:</span> {selectedNode.description}</div>
            )}

          </div>
        </div>
      )}

      {/* Hover 정보 툴팁 */}
      {hoveredNode && (
        <div 
          className="absolute bottom-4 left-4 p-2 rounded text-sm z-10 pointer-events-none"
          style={{
            backgroundColor: getCSSVariable('--tooltip-bg') || 
              (document.documentElement.classList.contains('dark') ? 'rgba(0, 0, 0, 0.85)' : 'rgba(255, 255, 255, 0.95)'),
            color: getCSSVariable('--tooltip-fg') || 
              (document.documentElement.classList.contains('dark') ? '#ffffff' : '#1a1a1a'),
            border: document.documentElement.classList.contains('dark') ? '1px solid rgba(255, 255, 255, 0.1)' : '1px solid rgba(0, 0, 0, 0.1)',
            boxShadow: document.documentElement.classList.contains('dark') ? 
              '0 4px 12px rgba(0, 0, 0, 0.3)' : '0 4px 12px rgba(0, 0, 0, 0.15)'
          }}
        >
          <div className="font-semibold">{hoveredNode.label}</div>
          <div>{hoveredNode.type}</div>
          {hoveredNode.description && (
            <div className="max-w-xs truncate">{hoveredNode.description}</div>
          )}
        </div>
      )}

      {/* SVG 캔버스 */}
      <svg
        ref={svgRef}
        className="w-full h-full rounded-lg"
        style={{ 
          minHeight: '400px',
          backgroundColor: getCSSVariable('--card-bg') || '#ffffff',
          border: `1px solid ${getCSSVariable('--card-border') || '#e5e7eb'}`
        }}
      />
      
      {/* 로딩 상태 */}
      {!data && (
        <div 
          className="absolute inset-0 flex items-center justify-center rounded-lg"
          style={{ backgroundColor: getCSSVariable('--header-bg') || '#f9fafb' }}
        >
          <div className="text-center">
            <div 
              className="animate-spin rounded-full h-8 w-8 border-b-2 mx-auto mb-2"
              style={{ borderBottomColor: getCSSVariable('--primary-500') || '#2563eb' }}
            ></div>
            <div 
              className="text-sm"
              style={{ color: getCSSVariable('--text-secondary') || '#6b7280' }}
            >
              GraphRAG 데이터 로딩 중...
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default GraphRAGVisualization;