'use client';

import { useEffect, useRef } from 'react';
import { FC } from 'react';

// 필요한 인터페이스 정의
interface Message {
  id: string;
  content: string;
  role: 'user' | 'assistant' | 'system';
  createdAt: string;
  imageUrl?: string;
  isStreaming?: boolean;
  files?: Array<{
    name: string;
    size: number;
    type?: string;
    url?: string;
  }>;
  error?: boolean;
  isLoading?: boolean;
}

// 간단한 마크다운 컴포넌트
const Markdown = ({ content }: { content: string }) => {
  if (!content) return null;
  // 실제로는 ReactMarkdown 같은 라이브러리를 사용하는 것이 좋습니다.
  return <div className="whitespace-pre-wrap">{content}</div>;
};

// 스켈레톤 로딩 컴포넌트
const Skeleton = ({ className = '' }: { className?: string }) => {
  return (
    <div
      className={`bg-gray-200 dark:bg-gray-700 animate-pulse rounded ${className}`}
    />
  );
};

// 알림 컴포넌트들
const Alert = ({ 
  variant = 'default',
  className = '',
  children 
}: { 
  variant?: 'default' | 'destructive', 
  className?: string, 
  children: React.ReactNode 
}) => {
  return (
    <div 
      className={`p-4 rounded-md ${
        variant === 'destructive' 
          ? 'bg-red-100 dark:bg-red-900 text-red-800 dark:text-red-100' 
          : 'bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-100'
      } ${className}`}
    >
      {children}
    </div>
  );
};

const AlertTitle = ({ children }: { children: React.ReactNode }) => {
  return <h5 className="font-medium mb-1">{children}</h5>;
};

const AlertDescription = ({ children }: { children: React.ReactNode }) => {
  return <div className="text-sm">{children}</div>;
};

// AlertCircle 아이콘 컴포넌트 직접 구현
const AlertCircle = ({ className = "h-6 w-6" }: { className?: string }) => (
  <svg
    xmlns="http://www.w3.org/2000/svg"
    width="24"
    height="24"
    viewBox="0 0 24 24"
    fill="none"
    stroke="currentColor"
    strokeWidth="2"
    strokeLinecap="round"
    strokeLinejoin="round"
    className={className}
  >
    <circle cx="12" cy="12" r="10"></circle>
    <line x1="12" y1="8" x2="12" y2="12"></line>
    <line x1="12" y1="16" x2="12.01" y2="16"></line>
  </svg>
);

interface ChatMessagesProps {
  messages: Message[];
  isLoading?: boolean;
  error?: string | null;
  isStreaming?: boolean;
}

const ChatMessages: FC<ChatMessagesProps> = ({ messages, isLoading, error, isStreaming }) => {
  const messagesEndRef = useRef<HTMLDivElement>(null);

  // 메시지가 추가될 때마다 스크롤을 맨 아래로 이동
  useEffect(() => {
    if (messagesEndRef.current) {
      messagesEndRef.current.scrollIntoView({ behavior: 'smooth' });
    }
  }, [messages, isStreaming]);

  // 로딩 중이고 메시지가 없는 경우
  if (isLoading && messages.length === 0) {
    return (
      <div className="space-y-4">
        <div className="flex items-start gap-4 animate-pulse">
          <div className="h-10 w-10 rounded-full bg-gray-200 dark:bg-gray-700 shrink-0"></div>
          <div className="space-y-2 flex-1">
            <Skeleton className="h-4 w-[200px]" />
            <Skeleton className="h-20 w-full" />
          </div>
        </div>
      </div>
    );
  }

  // 에러가 있는 경우
  if (error) {
    return (
      <Alert variant="destructive" className="mb-4">
        <AlertCircle className="h-4 w-4" />
        <AlertTitle>오류</AlertTitle>
        <AlertDescription>{error}</AlertDescription>
      </Alert>
    );
  }

  // 메시지가 없는 경우
  if (messages.length === 0) {
    return (
      <div className="flex items-center justify-center h-full">
        <div className="text-center text-gray-500 dark:text-gray-400">
          <p>이 대화에는 아직 메시지가 없습니다.</p>
          <p>첫 메시지를 보내 대화를 시작하세요!</p>
        </div>
      </div>
    );
  }

  // 화면에 메시지 표시
  return (
    <div className="flex flex-col space-y-4">
      {messages.map((message) => (
        <div
          key={message.id}
          className={`flex ${
            message.role === 'user' ? 'justify-end' : 'justify-start'
          }`}
        >
          <div
            className={`max-w-[95%] p-4 ${
              message.role === 'user'
                ? 'bg-blue-700 dark:bg-blue-600 text-white rounded-2xl rounded-tr-sm shadow-lg border border-blue-800 dark:border-blue-500'
                : message.error
                ? 'bg-red-100 text-red-900 rounded-2xl rounded-tl-sm'
                : 'bg-gray-100 dark:bg-gray-800 text-gray-900 dark:text-gray-100 rounded-2xl rounded-tl-sm'
            }`}
          >
            {message.isLoading ? (
              <div className="flex items-center space-x-2">
                <div className="animate-spin rounded-full h-4 w-4 border-2 border-gray-300 border-t-blue-600"></div>
                <span className="text-sm text-gray-500 dark:text-gray-400">{message.content}</span>
              </div>
            ) : (
              <div className="whitespace-pre-wrap">{message.content}</div>
            )}
          </div>
        </div>
      ))}
      {isLoading && messages.length === 0 && (
        <div className="flex justify-center">
          <div className="animate-spin rounded-full h-8 w-8 border-4 border-gray-300 border-t-blue-600"></div>
        </div>
      )}
      {error && (
        <div className="flex justify-center">
          <div className="bg-red-100 text-red-900 rounded-lg p-4">{error}</div>
        </div>
      )}
    </div>
  );
};

export default ChatMessages; 