import { Pool } from 'pg';
import { getApiServerUrl } from '../config/serverConfig';

interface DatabaseConfig {
  host: string;
  port: number;
  user: string;
  password: string;
  database: string;
}

let cachedConfig: DatabaseConfig | null = null;
let configPool: Pool | null = null;

/**
 * 중앙화된 API 서버에서 데이터베이스 설정을 가져옵니다
 */
async function fetchDatabaseConfig(): Promise<DatabaseConfig> {
  if (cachedConfig) {
    return cachedConfig;
  }

  try {
    const apiUrl = getApiServerUrl();
    const response = await fetch(`${apiUrl}/config`, {
      headers: {
        'Content-Type': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch config: ${response.status}`);
    }

    const result = await response.json();
    
    if (!result.success || !result.data) {
      throw new Error('Invalid config response format');
    }
    // API 서버에서 받은 설정을 사용
    const config = result.data.configs;
    
    cachedConfig = {
      host: config.DB_HOST || 'localhost',
      port: parseInt(config.DB_PORT || '5433'),
      user: config.DB_USER || 'ivs', 
      password: config.DB_PASSWORD || '1234',
      database: config.DB_NAME || 'airun',
    };

    return cachedConfig;
  } catch (error) {
    console.warn('Failed to fetch centralized config, using fallback:', error);
    
    // API 서버가 응답하지 않는 경우 환경변수 fallback 사용
    cachedConfig = {
      host: process.env.DB_HOST || process.env.POSTGRES_HOST || 'localhost',
      port: parseInt(process.env.DB_PORT || process.env.POSTGRES_PORT || '5433'),
      user: process.env.DB_USER || process.env.POSTGRES_USER || 'ivs',
      password: process.env.DB_PASSWORD || process.env.POSTGRES_PASSWORD || '1234',
      database: process.env.DB_NAME || process.env.POSTGRES_DB || 'airun',
    };

    return cachedConfig;
  }
}

/**
 * 중앙화된 설정을 사용하는 데이터베이스 Pool을 가져옵니다
 */
export async function getDatabasePool(): Promise<Pool> {
  if (configPool) {
    return configPool;
  }

  const config = await fetchDatabaseConfig();
  
  configPool = new Pool({
    host: config.host,
    port: config.port,
    database: config.database,
    user: config.user,
    password: config.password,
    max: 20, // 최대 연결 수
    idleTimeoutMillis: 30000,
    connectionTimeoutMillis: 2000,
  });

  return configPool;
}

/**
 * 캐시된 설정을 초기화합니다 (설정 변경 시 사용)
 */
export function clearConfigCache(): void {
  cachedConfig = null;
  if (configPool) {
    configPool.end();
    configPool = null;
  }
}

/**
 * 현재 데이터베이스 설정을 반환합니다 (디버깅용)
 */
export async function getCurrentDatabaseConfig(): Promise<DatabaseConfig> {
  return await fetchDatabaseConfig();
}