import { logger } from '../../utils/logger.js';

// 메모리 성능 모니터링 및 통계
export class MemoryPerformanceMonitor {
    constructor() {
        this.stats = {
            totalQueries: 0,
            successfulMatches: 0,
            averageRelevanceScore: 0,
            memoryUsageByCategory: {},
            searchPerformance: [],
            qualityScores: []
        };
        this.sessionStats = new Map(); // 세션별 통계
    }

    // 메모리 검색 성능 기록
    recordSearchPerformance(userId, query, results, searchTime) {
        this.stats.totalQueries++;
        
        const relevantResults = results.filter(r => r.relevanceScore > 0.5);
        if (relevantResults.length > 0) {
            this.stats.successfulMatches++;
        }
        
        // 평균 관련성 점수 업데이트
        const avgRelevance = results.reduce((sum, r) => sum + (r.relevanceScore || 0), 0) / results.length;
        this.stats.averageRelevanceScore = (
            (this.stats.averageRelevanceScore * (this.stats.totalQueries - 1) + avgRelevance) / 
            this.stats.totalQueries
        );
        
        // 검색 성능 기록
        this.stats.searchPerformance.push({
            timestamp: new Date(),
            userId,
            queryLength: query.length,
            resultCount: results.length,
            relevantCount: relevantResults.length,
            searchTime,
            avgRelevance
        });
        
        // 최근 100개만 유지
        if (this.stats.searchPerformance.length > 100) {
            this.stats.searchPerformance.shift();
        }
        
        // 세션별 통계 업데이트
        if (!this.sessionStats.has(userId)) {
            this.sessionStats.set(userId, {
                queries: 0,
                successRate: 0,
                avgRelevance: 0,
                categories: {}
            });
        }
        
        const userStats = this.sessionStats.get(userId);
        userStats.queries++;
        userStats.successRate = (userStats.successRate * (userStats.queries - 1) + (relevantResults.length > 0 ? 1 : 0)) / userStats.queries;
        userStats.avgRelevance = (userStats.avgRelevance * (userStats.queries - 1) + avgRelevance) / userStats.queries;
    }

    // 메모리 사용 패턴 기록
    recordMemoryUsage(memories) {
        for (const memory of memories) {
            const category = memory.category || 'unknown';
            if (!this.stats.memoryUsageByCategory[category]) {
                this.stats.memoryUsageByCategory[category] = {
                    count: 0,
                    totalRelevance: 0,
                    avgImportance: 0,
                    lastUsed: null
                };
            }
            
            const categoryStats = this.stats.memoryUsageByCategory[category];
            categoryStats.count++;
            categoryStats.totalRelevance += memory.relevanceScore || 0;
            categoryStats.avgImportance = (categoryStats.avgImportance * (categoryStats.count - 1) + memory.importance_level) / categoryStats.count;
            categoryStats.lastUsed = new Date();
        }
    }

    // 메모리 품질 점수 기록
    recordQualityScore(memoryId, qualityScore) {
        this.stats.qualityScores.push({
            timestamp: new Date(),
            memoryId,
            score: qualityScore
        });
        
        // 최근 1000개만 유지
        if (this.stats.qualityScores.length > 1000) {
            this.stats.qualityScores.shift();
        }
    }

    // 성능 리포트 생성
    generatePerformanceReport() {
        const recentPerformance = this.stats.searchPerformance.slice(-50); // 최근 50개
        const avgSearchTime = recentPerformance.length > 0 
            ? recentPerformance.reduce((sum, p) => sum + p.searchTime, 0) / recentPerformance.length 
            : 0;
        const successRate = this.stats.totalQueries > 0 
            ? (this.stats.successfulMatches / this.stats.totalQueries) * 100 
            : 0;
        
        const categoryUsage = Object.entries(this.stats.memoryUsageByCategory)
            .map(([category, stats]) => ({
                category,
                usage: stats.count,
                avgRelevance: stats.totalRelevance / stats.count,
                avgImportance: stats.avgImportance,
                lastUsed: stats.lastUsed
            }))
            .sort((a, b) => b.usage - a.usage);
        
        const recentQuality = this.stats.qualityScores.slice(-100);
        const avgQuality = recentQuality.length > 0 
            ? recentQuality.reduce((sum, q) => sum + q.score, 0) / recentQuality.length 
            : 0;
        
        return {
            overview: {
                totalQueries: this.stats.totalQueries,
                successRate: successRate.toFixed(1) + '%',
                avgRelevanceScore: this.stats.averageRelevanceScore.toFixed(3),
                avgSearchTime: avgSearchTime.toFixed(2) + 'ms',
                avgQualityScore: avgQuality.toFixed(3)
            },
            categoryUsage,
            recentPerformance: recentPerformance.slice(-10),
            recommendations: this.generateRecommendations(successRate, avgQuality, categoryUsage)
        };
    }

    // 최적화 권장사항 생성
    generateRecommendations(successRate, avgQuality, categoryUsage) {
        const recommendations = [];
        
        if (successRate < 70) {
            recommendations.push({
                type: 'search_optimization',
                priority: 'high',
                message: '검색 성공률이 낮습니다. 키워드 추출 알고리즘을 개선하거나 메모리 카테고리를 재조정하세요.'
            });
        }
        
        if (avgQuality < 0.6) {
            recommendations.push({
                type: 'quality_improvement',
                priority: 'medium',
                message: '메모리 품질이 낮습니다. 자동 저장 기준을 강화하거나 품질 관리 주기를 단축하세요.'
            });
        }
        
        const underusedCategories = categoryUsage.filter(c => c.usage < 5 && c.avgRelevance < 0.3);
        if (underusedCategories.length > 0) {
            recommendations.push({
                type: 'category_cleanup',
                priority: 'low',
                message: `사용률이 낮은 카테고리가 있습니다: ${underusedCategories.map(c => c.category).join(', ')}`
            });
        }
        
        return recommendations;
    }

    // 통계 초기화
    reset() {
        this.stats = {
            totalQueries: 0,
            successfulMatches: 0,
            averageRelevanceScore: 0,
            memoryUsageByCategory: {},
            searchPerformance: [],
            qualityScores: []
        };
        this.sessionStats.clear();
    }

    // 사용자별 통계 조회
    getUserStats(userId) {
        return this.sessionStats.get(userId) || {
            queries: 0,
            successRate: 0,
            avgRelevance: 0,
            categories: {}
        };
    }
}

// 메모리 시스템 진단 함수
export async function diagnoseMemorySystem(userId) {
    try {
        const startTime = Date.now();
        
        // aiFeatures.js에서 필요한 함수들 가져오기
        const { retrieveUserMemories, evaluateMemoryQuality, calculateMemoryRelevance } = await import('../../aiFeatures.js');
        
        // 1. 메모리 개수 및 분포 확인
        const allMemories = await retrieveUserMemories(userId, '', { limit: 1000 });
        const memories = allMemories.success ? allMemories.memories : [];
        
        const categoryDistribution = memories.reduce((acc, memory) => {
            const category = memory.category || 'unknown';
            acc[category] = (acc[category] || 0) + 1;
            return acc;
        }, {});
        
        // 2. 메모리 품질 평가
        let totalQuality = 0;
        let qualityCount = 0;
        const qualityIssues = [];
        
        for (const memory of memories.slice(0, 50)) { // 최대 50개만 평가
            const quality = await evaluateMemoryQuality(memory);
            totalQuality += quality.score;
            qualityCount++;
            
            if (quality.score < 0.4) {
                qualityIssues.push({
                    id: memory.id,
                    title: memory.title.substring(0, 50) + '...',
                    score: quality.score,
                    issues: Object.entries(quality.details)
                        .filter(([key, value]) => value < 0.5)
                        .map(([key]) => key)
                });
            }
        }
        
        // 3. 검색 성능 테스트
        const testQueries = [
            '코딩 스타일 선호도',
            '프로젝트 설정',
            '사용자 선호사항',
            '기술적 요구사항',
            '작업 방식'
        ];
        
        const searchResults = [];
        for (const query of testQueries) {
            const searchStart = Date.now();
            const results = await retrieveUserMemories(userId, query, { limit: 10 });
            const searchTime = Date.now() - searchStart;
            
            searchResults.push({
                query,
                resultCount: results.success ? results.memories.length : 0,
                searchTime,
                hasRelevant: results.success && results.memories.some(m => 
                    calculateMemoryRelevance(query, m) > 0.5
                )
            });
        }
        
        // 4. 성능 모니터에서 통계 가져오기
        const performanceReport = memoryMonitor.generatePerformanceReport();
        
        const diagnosticTime = Date.now() - startTime;
        
        return {
            success: true,
            timestamp: new Date().toISOString(),
            diagnosticTime: diagnosticTime + 'ms',
            memoryCount: memories.length,
            categoryDistribution,
            qualityMetrics: {
                averageQuality: qualityCount > 0 ? (totalQuality / qualityCount).toFixed(3) : 'N/A',
                lowQualityCount: qualityIssues.length,
                qualityIssues: qualityIssues.slice(0, 10) // 최대 10개만 표시
            },
            searchPerformance: {
                averageSearchTime: (searchResults.reduce((sum, r) => sum + r.searchTime, 0) / searchResults.length).toFixed(2) + 'ms',
                successfulQueries: searchResults.filter(r => r.hasRelevant).length,
                totalQueries: searchResults.length,
                detailedResults: searchResults
            },
            overallPerformance: performanceReport,
            recommendations: [
                ...performanceReport.recommendations,
                ...(memories.length > 500 ? [{
                    type: 'memory_cleanup',
                    priority: 'medium',
                    message: '메모리 개수가 많습니다. 정기적인 정리를 권장합니다.'
                }] : []),
                ...(qualityIssues.length > memories.length * 0.2 ? [{
                    type: 'quality_improvement',
                    priority: 'high',
                    message: '품질이 낮은 메모리가 20% 이상입니다. 자동 저장 기준을 재검토하세요.'
                }] : [])
            ]
        };
        
    } catch (error) {
        console.error('메모리 시스템 진단 실패:', error);
        return {
            success: false,
            error: error.message,
            timestamp: new Date().toISOString()
        };
    }
}

// 메모리 최적화 실행
export async function optimizeMemorySystem(userId, options = {}) {
    try {
        const { refreshMemoryRelevance, resolveMemoryConflicts } = await import('../../aiFeatures.js');
        
        const results = {
            cleaned: 0,
            optimized: 0,
            errors: []
        };
        
        // 1. 품질 관리 실행
        if (options.cleanupQuality !== false) {
            const qualityResult = await refreshMemoryRelevance(userId);
            if (qualityResult.success) {
                results.cleaned += qualityResult.deactivated;
                results.optimized += qualityResult.upgraded;
            } else {
                results.errors.push('품질 관리 실패: ' + qualityResult.error);
            }
        }
        
        // 2. 충돌 해결 실행
        if (options.resolveConflicts !== false) {
            const conflictResult = await resolveMemoryConflicts(userId);
            if (conflictResult.success) {
                results.optimized += conflictResult.resolved;
            } else {
                results.errors.push('충돌 해결 실패: ' + conflictResult.error);
            }
        }
        
        // 3. 카테고리 재분류 (향후 구현 예정)
        if (options.recategorize === true) {
            results.errors.push('카테고리 재분류는 아직 구현되지 않았습니다.');
        }
        
        logger.info(`🔧 메모리 시스템 최적화 완료: 정리 ${results.cleaned}개, 최적화 ${results.optimized}개`);
        
        return {
            success: results.errors.length === 0,
            results,
            timestamp: new Date().toISOString()
        };
        
    } catch (error) {
        console.error('메모리 시스템 최적화 실패:', error);
        return {
            success: false,
            error: error.message,
            timestamp: new Date().toISOString()
        };
    }
}

// 전역 성능 모니터 인스턴스
export const memoryMonitor = new MemoryPerformanceMonitor();

// 메모리 성능 추적 래퍼 함수
export function trackMemorySearch(userId, query, searchFunction) {
    return async (...args) => {
        const startTime = Date.now();
        try {
            const results = await searchFunction(...args);
            const searchTime = Date.now() - startTime;
            
            // 결과가 배열 형태인지 확인하고 성능 기록
            if (results && Array.isArray(results.memories)) {
                memoryMonitor.recordSearchPerformance(userId, query, results.memories, searchTime);
                memoryMonitor.recordMemoryUsage(results.memories);
            }
            
            return results;
        } catch (error) {
            const searchTime = Date.now() - startTime;
            memoryMonitor.recordSearchPerformance(userId, query, [], searchTime);
            throw error;
        }
    };
}

// 주기적 성능 리포트 생성
export function schedulePerformanceReport(intervalMinutes = 60) {
    setInterval(() => {
        const report = memoryMonitor.generatePerformanceReport();
        if (report.overview.totalQueries > 0) {
            logger.info('📊 메모리 성능 리포트:', {
                queries: report.overview.totalQueries,
                successRate: report.overview.successRate,
                avgRelevance: report.overview.avgRelevanceScore,
                recommendations: report.recommendations.length
            });
            
            // 권장사항이 있으면 경고 로그
            if (report.recommendations.length > 0) {
                const highPriorityRecs = report.recommendations.filter(r => r.priority === 'high');
                if (highPriorityRecs.length > 0) {
                    logger.warn('⚠️ 메모리 시스템 최적화 필요:', highPriorityRecs.map(r => r.message));
                }
            }
        }
    }, intervalMinutes * 60 * 1000);
} 