/**
 * @swagger
 * tags:
 *   - name: Session
 *     description: 세션 관리 관련 API
 */

/**
 * @swagger
 * /api/v1/sessions:
 *   post:
 *     summary: 세션 생성
 *     description: 새로운 채팅 세션을 생성합니다.
 *     tags: [Session]
 *     security:
 *       - apiKeyAuth: []
 *       - bearerAuth: [] 
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/SessionCreateRequest'
 *           example:
 *             provider: "openai"
 *             model: "gpt-4"
 *             parameters: {
 *               temperature: 0.7
 *             }
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   $ref: '#/components/schemas/Session'
 *             example:
 *               success: true
 *               data: {
 *                 id: "chat_123",
 *                 created_at: "2025-04-22T05:30:08.520Z",
 *                 last_active: "2025-04-22T05:30:08.520Z",
 *                 provider: "openai",
 *                 model: "gpt-4",
 *                 parameters: {
 *                   temperature: 0.7
 *                 }
 *               }
 *       401:
 *         $ref: '#/components/responses/UnauthorizedError'
 *       400:
 *         $ref: '#/components/responses/BadRequestError'
 *   get:
 *     summary: 세션 목록 조회
 *     description: 모든 활성 세션의 목록을 조회합니다.
 *     tags: [Session]
 *     security:
 *       - apiKeyAuth: []
 *       - bearerAuth: [] 
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: array
 *                   items:
 *                     $ref: '#/components/schemas/Session'
 *             example:
 *               success: true
 *               data: [{
 *                 id: "chat_123",
 *                 created_at: "2025-04-22T05:30:08.520Z",
 *                 last_active: "2025-04-22T05:30:08.520Z",
 *                 provider: "openai",
 *                 model: "gpt-4",
 *                 parameters: {
 *                   temperature: 0.7
 *                 }
 *               }]
 *       401:
 *         $ref: '#/components/responses/UnauthorizedError'
 *   delete:
 *     summary: 모든 세션 삭제
 *     description: 모든 활성 세션을 삭제합니다.
 *     tags: [Session]
 *     security:
 *       - apiKeyAuth: []
 *       - bearerAuth: [] 
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     message:
 *                       type: string
 *                       description: 삭제 결과 메시지
 *                     deletedCount:
 *                       type: integer
 *                       description: 삭제된 세션 수
 *             example:
 *               success: true
 *               data: {
 *                 message: "모든 세션이 성공적으로 삭제되었습니다.",
 *                 deletedCount: 5
 *               }
 *       401:
 *         $ref: '#/components/responses/UnauthorizedError'
 */

/**
 * @swagger
 * /api/v1/sessions/{sessionId}:
 *   get:
 *     summary: 특정 세션 조회
 *     description: sessionId로 특정 세션의 상세 정보를 조회합니다.
 *     tags: [Session]
 *     security:
 *       - apiKeyAuth: []
 *       - bearerAuth: [] 
 *     parameters:
 *       - in: path
 *         name: sessionId
 *         required: true
 *         schema:
 *           type: string
 *         description: 세션 ID
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   $ref: '#/components/schemas/Session'
 *             example:
 *               success: true
 *               data: {
 *                 id: "chat_123",
 *                 created_at: "2025-04-22T05:30:08.520Z",
 *                 last_active: "2025-04-22T05:30:08.520Z",
 *                 provider: "openai",
 *                 model: "gpt-4",
 *                 parameters: {
 *                   temperature: 0.7
 *                 }
 *               }
 *       401:
 *         $ref: '#/components/responses/UnauthorizedError'
 *       404:
 *         $ref: '#/components/responses/NotFoundError'
 *   delete:
 *     summary: 세션 삭제
 *     description: 특정 세션을 삭제합니다.
 *     tags: [Session]
 *     security:
 *       - apiKeyAuth: []
 *       - bearerAuth: [] 
 *     parameters:
 *       - in: path
 *         name: sessionId
 *         required: true
 *         schema:
 *           type: string
 *         description: 세션 ID
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *             example:
 *               success: true
 *       401:
 *         $ref: '#/components/responses/UnauthorizedError'
 *       404:
 *         $ref: '#/components/responses/NotFoundError'
 *   put:
 *     summary: 세션 업데이트
 *     description: 특정 세션의 정보를 업데이트합니다.
 *     tags: [Session]
 *     security:
 *       - apiKeyAuth: []
 *       - bearerAuth: [] 
 *     parameters:
 *       - in: path
 *         name: sessionId
 *         required: true
 *         schema:
 *           type: string
 *         description: 세션 ID
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/SessionUpdateRequest'
 *           example:
 *             parameters: {
 *               temperature: 0.8
 *             }
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   $ref: '#/components/schemas/Session'
 *             example:
 *               success: true
 *               data: {
 *                 id: "chat_123",
 *                 created_at: "2025-04-22T05:30:08.520Z",
 *                 last_active: "2025-04-22T05:35:08.520Z",
 *                 provider: "openai",
 *                 model: "gpt-4",
 *                 parameters: {
 *                   temperature: 0.8
 *                 }
 *               }
 *       401:
 *         $ref: '#/components/responses/UnauthorizedError'
 *       404:
 *         $ref: '#/components/responses/NotFoundError'
 */

export const sessionSchemas = {
    Session: {
        type: 'object',
        properties: {
            id: {
                type: 'string',
                description: '세션 ID'
            },
            created_at: {
                type: 'string',
                format: 'date-time',
                description: '세션 생성 시간'
            },
            last_active: {
                type: 'string',
                format: 'date-time',
                description: '마지막 활성 시간'
            },
            provider: {
                type: 'string',
                description: 'AI 제공자'
            },
            model: {
                type: 'string',
                description: 'AI 모델'
            },
            parameters: {
                type: 'object',
                description: '세션 파라미터'
            }
        }
    },
    SessionCreateRequest: {
        type: 'object',
        required: ['provider'],
        properties: {
            provider: {
                type: 'string',
                description: 'AI 제공자'
            },
            model: {
                type: 'string',
                description: 'AI 모델'
            },
            parameters: {
                type: 'object',
                description: '세션 파라미터'
            }
        }
    },
    SessionUpdateRequest: {
        type: 'object',
        properties: {
            parameters: {
                type: 'object',
                description: '업데이트할 세션 파라미터'
            }
        }
    }
}; 